/* Doors for Linux.
   Copyright (C) 1998 Jason Lango

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2, or (at
   your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

/*
**  door_ioctl.h - Interface structures and defines for doors driver.
**  Jason Lango <jal@rampant.org>
*/

#ifndef __DOOR_IOCTL_H__
#define __DOOR_IOCTL_H__

#include "door.h"

#define DOOR_ARGBUF_SIZE	(16 * 1024)

/* /dev/door ioctl() numbers. */
#define DOOR_CREATE		1 /* _IOR('d', 1, struct door_create_args) */
#define DOOR_CALL		2 /* _IOR('d', 2, struct door_call_args) */
#define DOOR_FATTACH		3 /* _IOR('d', 3, struct door_fattach_args) */
#define DOOR_FDETACH		4 /* _IOR('d', 4, struct door_fdetach_args) */
#define DOOR_RETURN		5 /* _IOR('d', 5, struct door_return_args) */
#define DOOR_WAIT_SVR_CREATE	6 /* _IOR('d', 6, struct door_wait_svr_create_args) */
#define DOOR_INFO		7 /* _IOR('d', 7, struct door_info_args) */

/*
 * DOOR_PTR()/DOOR_UINTPTR() used to bless pointers passed to/from the
 * kernel s.t.  gcc doesn't issue a warning about sizes and we don't
 * risk sign extension problems.  Casts to ABI-appropriate pointer
 * size integer.
 */
#define DOOR_UINTPTR(ptr)	((unsigned long)(ptr))
#define DOOR_PTR(ptr)		((door_ptr_t)DOOR_UINTPTR(ptr))

/*
 * door_karg_t - pointer-size neutral version of door_arg_t, for
 * user/kernel argument passing on multiple ABI machines.
 */
typedef struct door_karg {
	door_ptr_t	data_ptr;
	np_door_size_t	data_size;
	door_ptr_t	desc_ptr;
	np_door_size_t	desc_num;
	door_ptr_t	rbuf;
	np_door_size_t	rsize;
} door_karg_t;

struct door_create_args {
	door_ptr_t	server_procedure;
	door_ptr_t	cookie;
	door_attr_t	attributes;
};

struct door_fattach_args {
	int		fd;
	door_ptr_t	name;
};

struct door_fdetach_args {
	door_ptr_t	name;
};

struct door_call_args {
	int		fd;
	door_karg_t	params;
	door_ptr_t	rparams;	/* points to door_karg_t */
};

struct door_return_outargs {
	/* server_proc arguments */
	door_ptr_t		proc;
	door_ptr_t		cookie;
	door_ptr_t		data_ptr;
	np_door_size_t		data_size;
	door_ptr_t		desc_ptr;
	np_door_size_t		desc_num;

	door_ptr_t		munmap_addr;
	np_door_size_t		munmap_len;

	/* door_cred values */
	door_cred_t		cred;
};

struct door_return_args {
	door_ptr_t		data_ptr;
	np_door_size_t		data_size;
	door_ptr_t		desc_ptr;
	np_door_size_t		num_desc;

	door_ptr_t		argstash;
	np_door_size_t		argstash_size;

	door_ptr_t		munmap_addr;
	np_door_size_t		munmap_len;

	door_ptr_t		outargs; /* points to door_return_outargs */
};

struct door_wait_svr_create_args {
	door_info_t info;
};

struct door_info_args {
	int			fd;
	door_ptr_t		info;
};

#endif /* __DOOR_IOCTL_H__ */
