/* Doors for Linux.
   Copyright (C) 1998 Jason Lango

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2, or (at
   your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

/*
** door_create.c - door_create() and server functions.
** Jason Lango <jal@rampant.org>
*/

#include "door.h"
#include "door_ioctl.h"
#include "libdoor.h"
#include <errno.h>
#include <pthread.h>
#include <sys/ioctl.h>
#include <stdlib.h>
#include <string.h>
#include <bits/libc-lock.h>

#ifdef weak_extern
weak_extern (pthread_create)
#else
# pragma weak pthread_create
#endif

static void __door_default_server_create(door_info_t *info);

np_door_create_proc_t	__door_server_create_proc
	= __door_default_server_create;

int			__door_svr_create_thr_started = 0;
static pthread_mutex_t	__door_svr_create_mut = PTHREAD_MUTEX_INITIALIZER;
static pthread_t	__door_svr_create_thr;
static pthread_key_t	__door_priv_key;

static void *__door_wait_svr_create(void *arg)
{
	door_info_t info;

	for (;;) {
		if (ioctl(__door_dev_fd, DOOR_WAIT_SVR_CREATE, &info) < 0) {
			__door_perror("ioctl(WAIT_SVR_CREATE)");
			break;
		}

		/* XXX: private server pools not implemented */
		(*__door_server_create_proc)(NULL);
	}

	return NULL;
}

static void *__door_default_server(void *arg)
{
	pthread_setcancelstate(PTHREAD_CANCEL_DISABLE, NULL);
	if (__door_return(NULL, 0, NULL, 0) < 0)
		__door_perror("door_return()");
	__door_error("should not have returned from door_return()");
	return NULL;
}

static void __door_default_server_create(door_info_t *info)
{
	pthread_t thr;

	if (pthread_create(&thr, NULL, __door_default_server, NULL) < 0)
		__door_perror("pthread_create");
}

struct door_thread_priv *__door_get_priv(void)
{
	struct door_thread_priv *p;

	p = __pthread_getspecific(__door_priv_key);

	if (!p) {
		p = (struct door_thread_priv*)
			malloc(sizeof(struct door_thread_priv));
		if (!p)
			return NULL;

		p->dp_toplevel_initialized = 0;
		p->dp_argstash = malloc(ARGSTASH_SIZE);
		p->dp_serve_args.munmap_addr = 0;

		if (!p->dp_argstash) {
			free(p);
			return NULL;
		}

		if (__pthread_setspecific(__door_priv_key, p) < 0)
			__door_perror("pthread_setspecific");
	}

	return p;
}

static int __door_server_init(void)
{

	if (__pthread_key_create(&__door_priv_key, NULL) < 0) {
		__door_perror("pthread_key_create");
		__set_errno(ENOSYS);
		return -1;
	}

	if (pthread_create(&__door_svr_create_thr, NULL,
		__door_wait_svr_create, NULL) < 0) {
		__door_perror("pthread_create");
		__set_errno(ENOSYS);
		return -1;
	}

	__door_svr_create_thr_started = 1;
	return 0;
}

int __door_create(np_door_server_proc_t server_procedure,
	void *cookie, u_int attributes)
{
	struct door_create_args args;

	DOOR_INIT();

	/* We only need a server creation thread in a doors server
	 * process.
	 */

	if (!__door_svr_create_thr_started) {
		int error;

		if (pthread_create == NULL) {
		       __door_error("This program is not linked against the POSIX Thread Library!");
		       abort ();
		}

		if (__pthread_mutex_lock(&__door_svr_create_mut) < 0)
			__door_perror("server create mutex lock");

		error = 0;
		if (!__door_svr_create_thr_started)
			error = __door_server_init();

		if (__pthread_mutex_unlock(&__door_svr_create_mut) < 0)
			__door_perror("server create mutex unlock");

		if (error)
			return -1;
	}

	args.server_procedure = DOOR_PTR(server_procedure);
	args.cookie = DOOR_PTR(cookie);
	args.attributes = attributes;

	return ioctl(__door_dev_fd, DOOR_CREATE, &args);
}

weak_alias(__door_create, door_create)
