/* Doors for Linux.
   Copyright (C) 1998 Jason Lango

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2, or (at
   your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

/*
** door_return.c
** Jason Lango <jal@rampant.org>
*/

#include "door.h"
#include "door_ioctl.h"
#include "libdoor.h"
#include <sys/ioctl.h>
#include <errno.h>

int __door_return(void *data_ptr, size_t data_size,
	door_desc_t *desc_ptr, size_t num_desc)
{
	struct door_return_args args;
	struct door_thread_priv * volatile priv;

	DOOR_INIT();

	priv = __door_get_priv();
	if (priv == NULL) {
		__set_errno(EINVAL);
		return -1;
	}

	if (!priv->dp_toplevel_initialized) {
		if (setjmp(priv->dp_toplevel)) {
			/* At this point, we're in the stack frame of the
			   top level door_return and we've just received
			   a new message from a client process.
			*/
			struct door_return_outargs *sa = &priv->dp_serve_args;
			np_door_server_proc_t proc;

			proc = (np_door_server_proc_t)DOOR_UINTPTR(sa->proc);
			/*
			 * Since this process created the door, the
			 * pointer sizes involved are correct for the
			 * current ABI (since this might involve
			 * casting the 64 bit door_ptr_t down to a 32
			 * bit pointer value).
			 */
			(*proc)((void*)DOOR_UINTPTR(sa->cookie),
				(void*)DOOR_UINTPTR(sa->data_ptr),
				sa->data_size,
				(door_desc_t*)DOOR_UINTPTR(sa->desc_ptr),
				sa->desc_num);

			__door_error("server procedure should not return");
		}

		priv->dp_toplevel_initialized = 1;
	}

	args.data_ptr = DOOR_PTR(data_ptr);
	args.data_size = data_size;
	args.desc_ptr = DOOR_PTR(desc_ptr);
	args.num_desc = num_desc;
	args.argstash = DOOR_PTR(priv->dp_argstash);
	args.argstash_size = ARGSTASH_SIZE;
	args.munmap_addr = priv->dp_serve_args.munmap_addr;
	args.munmap_len = priv->dp_serve_args.munmap_len;
	args.outargs = DOOR_PTR(&priv->dp_serve_args);

	if (ioctl(__door_dev_fd, DOOR_RETURN, &args) < 0)
		return -1;

	/* Unroll the stack to the top level door_return.
	*/
	longjmp(priv->dp_toplevel, 1);

	__door_error("door_return shouldn't return");
	abort();
	return 0;
}

weak_alias(__door_return, door_return)
