/* Doors for Linux.
   Copyright (C) 1998 Jason Lango

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2, or (at
   your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

/*
**  libdoor.h - Doors library internal header file.
**  Jason Lango <jal@rampant.org>
*/

#ifndef __LIBDOOR_H__
#define __LIBDOOR_H__

#include "door_ioctl.h"
#include <setjmp.h>
#include <pthread.h>

/* Library constants
 * -----------------
 */

/* ARGSTASH_SIZE controls the size of the per-server thread buffer
   for copying and holding input arguments to a door_call.
   The kernel expects it to be twice the maximum buffer size
   (to hold inargs + indoors).
*/
#define ARGSTASH_SIZE (2 * DOOR_ARGBUF_SIZE)


/* Library data types
 * ------------------
 */

/* Each server thread has a door_thread_priv structure dynamically
 * allocated and stored as thread private data.
 */

struct door_thread_priv {
	int				dp_toplevel_initialized;
	jmp_buf				dp_toplevel;
	char				*dp_argstash;
	struct door_return_outargs	dp_serve_args;
};


/* Library global functions
 * ------------------------
 */

/* DOOR_INIT() is called by all doors external interfaces callable by
 * a doors client or server process to initialize the doors device.
 * Only functions which need to contact the doors driver need to use
 * this function.
 */
#define DOOR_INIT()						\
	do {							\
		if (__door_dev_fd < 0)				\
			__door_init();				\
		if (__door_dev_fd < 0) {			\
			errno = ENOSYS;				\
			return -1;				\
		}						\
	} while (0)

extern void __door_init(void);
extern void __door_error(const char *fmt, ...);

#define __door_perror(msg) __door_error("%s: %s", (msg), strerror(errno))

extern struct door_thread_priv *__door_get_priv(void);


/* Library global variables
 * ------------------------
 */

extern np_door_create_proc_t	__door_server_create_proc;
extern int			__door_dev_fd;
extern int			__door_svr_create_thr_started;

#endif /* __LIBDOOR_H__ */
