#include <stdio.h>
#include <string.h>
#include <stdint.h>
#include <stdlib.h>

#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <endian.h>
#include <byteswap.h>

#define TESTNORMALLVM 0

#if BYTE_ORDER == LITTLE_ENDIAN
#  define xlate16(x) (x)
#  define xlate32(x) (x)
#  define xlate64(x) (x)
#  define xlate16_be(x) bswap_16(x)
#  define xlate32_be(x) bswap_32(x)
#  define xlate64_be(x) bswap_64(x)
#elif BYTE_ORDER == BIG_ENDIAN
#  define xlate16(x) bswap_16(x)
#  define xlate32(x) bswap_32(x)
#  define xlate64(x) bswap_64(x)
#  define xlate16_be(x) (x)
#  define xlate32_be(x) (x)
#  define xlate64_be(x) (x)
#else
#  include <asm/byteorder.h>
#  define xlate16(x) __cpu_to_le16((x))
#  define xlate32(x) __cpu_to_le32((x))
#  define xlate64(x) __cpu_to_le64((x))
#  define xlate16_be(x) __cpu_to_be16((x))
#  define xlate32_be(x) __cpu_to_be32((x))
#  define xlate64_be(x) __cpu_to_be64((x))
#endif

#define SECTOR_SIZE 512
#define ID_LEN 32
#define FMTT_MAGIC "\040\114\126\115\062\040\170\133\065\101\045\162\060\116\052\076"

#define READ_FIRST_SECTORS 32

/* On disk - 32 bytes */
struct label_header {
    int8_t id[8];       /* LABELONE */
    uint64_t sector_xl; /* Sector number of this label */
    uint32_t crc_xl;    /* From next field to end of sector */
    uint32_t offset_xl; /* Offset from start of struct to contents */
    int8_t type[8];     /* LVM2 001 */
} __attribute__ ((packed));


/* On disk */
struct disk_locn {
	uint64_t offset;    /* Offset in bytes to start sector */
	uint64_t size;      /* Bytes */
} __attribute__ ((packed));

/* On disk */
struct pv_header {
	int8_t pv_uuid[ID_LEN];

	/* This size can be overridden if PV belongs to a VG */
	uint64_t device_size_xl;    /* Bytes */

	/* NULL-terminated list of data areas followed by */
	/* NULL-terminated list of metadata area headers */
	struct disk_locn disk_areas_xl[0];  /* Two lists */
} __attribute__ ((packed));

/* On disk */
struct raw_locn {
	uint64_t offset;    /* Offset in bytes to start sector */
	uint64_t size;      /* Bytes */
	uint32_t checksum;
	uint32_t filler;
} __attribute__ ((packed));

/* On disk */
/* Structure size limited to one sector */
struct mda_header {
	uint32_t checksum_xl;   /* Checksum of rest of mda_header */
	int8_t magic[16];   /* To aid scans for metadata */
	uint32_t version;
	uint64_t start;     /* Absolute start byte of mda_header */
	uint64_t size;      /* Size of metadata area */

	struct raw_locn raw_locns[0];   /* NULL-terminated list */
} __attribute__ ((packed));


void* first_n_sectors = NULL;

void save_uuid(const char* uuid)
{
	FILE* pvuuidf = fopen("pvuuid", "w");
	if (pvuuidf == NULL) {
		perror("fopen pvuuid:");
		exit(3);
	}
	fprintf(pvuuidf, "%s", uuid);
	fclose(pvuuidf);
}

void save_metadata(const char* metadata)
{
	FILE* mdf = fopen("metadata", "w");
	if (mdf == NULL) {
		perror("fopen metadata:");
		exit(3);
	}
	fprintf(mdf, "%s", metadata);
	fclose(mdf);
}

void* read_device(int fd, void* mem, off_t offset, ssize_t size)
{
	int r;
	ssize_t sz;
	r = lseek(fd, offset, SEEK_SET);
	if (r == (off_t)-1) {
		perror("lseek failed:");
		exit(3);
	}
	sz = read(fd, mem, size);
	if (sz == -1) {
		perror("read failed:");
		exit(3);
	}
	return mem;
}

void* read_data_from_device(int fd, unsigned long offset, int size)
{
	void* mem;

	mem = malloc(size);
	if (mem == NULL) {
		perror("malloc failed:");
		exit(3);
	}

	mem = read_device(fd, mem, offset, size);

	return mem;
}

int adjust_read_size(int fd, void ** first_n_sectors, int current_size, int need_size)
{
	while (current_size < need_size) {
		current_size = current_size * 2;
	}

	*first_n_sectors = realloc(*first_n_sectors, current_size);
	if (*first_n_sectors == NULL) {
		perror("realloc failed:");
		exit(3);
	}

	*first_n_sectors = read_device(fd, *first_n_sectors, 0, current_size);

	return current_size;
}


struct label_header* get_label_header(void* start, int index)
{
	return (struct label_header*)(start + index * SECTOR_SIZE);
}

struct pv_header* get_pv_header(struct label_header *lh)
{
	return (struct pv_header*)(((void*)lh)+sizeof(struct label_header));
}

struct mda_header* get_mda_header(int fd, struct  pv_header* ph, int metadata_area_index)
{
	unsigned long offset = xlate64(ph->disk_areas_xl[metadata_area_index].offset);
	unsigned long size = xlate64(ph->disk_areas_xl[metadata_area_index].size);
#if TESTNORMALLVM
	return (struct mda_header*)read_data_from_device(fd, offset, size);
#else
	printf("adjust 0x200 offset in the offset of metadata header\n");
	return (struct mda_header*)read_data_from_device(fd, offset + 0x200, size);
#endif
}

const char* get_metadata_str(struct mda_header *mdh)
{
	ssize_t mdh_offset;
	mdh_offset = mdh->raw_locns[0].offset;
//	printf("metadata_str size: %ld\n", mdh->raw_locns[0].size);
	return (char*)((void*)mdh+mdh_offset);
}

static void search_label(int dev_fd)
{
	int label_index;
	int disk_area_index;
	int current_read = 1;
	struct label_header *lh;
	struct pv_header *ph;
	struct mda_header *mdh;
	const char* metadata_str;
	int found = 0;

	current_read = adjust_read_size(dev_fd, &first_n_sectors, current_read, READ_FIRST_SECTORS * SECTOR_SIZE);

	/*FIXME: How many sectors need to be scaned */
	for (label_index = 0; label_index < READ_FIRST_SECTORS; label_index++) {
		lh = get_label_header(first_n_sectors, label_index);

		if (strncmp(lh->id, "LABELONE", 8) == 0) {
			printf("label found at sector %d\n", label_index);
			if ( ((long)(xlate64(lh->sector_xl))) == label_index - 1) {
				printf("label_index: %d, sector: %ld, offset: %d\n", label_index, (long)(xlate64(lh->sector_xl)), (int)(xlate32(lh->offset_xl)));
				found = 1;
				break;
			} else {
#if TESTNORMALLVM
				found = 1;
				break;
#else
				printf("But seems to be not the one inside CSM. Skip it.\n");
#endif
			}
		}
	}

	if (found == 1) {
		ph = get_pv_header(lh);

		ph->pv_uuid[ID_LEN] = 0;
		printf("pv uuid: %s\n", ph->pv_uuid);
		save_uuid((char*)ph->pv_uuid);
		printf("pv size: %lu\n", (unsigned long)xlate64(ph->device_size_xl));
		printf("data area\n");
		for (disk_area_index =0; disk_area_index < 100; disk_area_index++) {
			if ((xlate64(ph->disk_areas_xl[disk_area_index].offset)) == 0 && (xlate64(ph->disk_areas_xl[disk_area_index].size)) == 0 ) {
				disk_area_index++;
				break;
			}
			printf("disk_area_index: %d, offset: %lu, size: %ld\n", disk_area_index, 
					(unsigned long)(xlate64(ph->disk_areas_xl[disk_area_index].offset)), 
					(long)(xlate64(ph->disk_areas_xl[disk_area_index].size)));
		}
		printf("metadata area\n");
		for (; disk_area_index< 100; disk_area_index++) {
			if ((xlate64(ph->disk_areas_xl[disk_area_index].offset)) == 0 && (xlate64(ph->disk_areas_xl[disk_area_index].size)) ==0 ) {
				disk_area_index--;
				break;
			}
			printf("disk_area_index: %d, offset: %lu, size: %ld\n", disk_area_index, 
					(unsigned long)(xlate64(ph->disk_areas_xl[disk_area_index].offset)), 
					(long)(xlate64(ph->disk_areas_xl[disk_area_index].size)));
		}

		mdh = get_mda_header(dev_fd, ph, disk_area_index);

		if (strncmp(mdh->magic, FMTT_MAGIC, 16) != 0) {
			fprintf(stderr, "Magic number of mda header is wrong\n");
			exit(3);
		}

		metadata_str = get_metadata_str(mdh);

		save_metadata(metadata_str);
		printf("metadata: \n=====================\n%s\n=====================\n", metadata_str);
		printf("metadata len: %d\n", strlen(metadata_str));
	} else {
		printf("no LVM2 label found\n");
		exit(1);
	}
}

int main(int argc, char* argv[])
{
	int dev_fd;
	dev_fd = open(argv[1], O_RDWR);
	if (dev_fd == -1) {
		perror("open failed:");
		exit(3);
	}
	search_label(dev_fd);
	exit(0);
}
