//////////////////////////////////////////////////////////////////////////////
// atomset.h 

#ifndef ATOMSET_H
#define ATOMSET_H

#include <algorithm>
#include <set>

//////////////////////////////////////////////////////////////////////////////
class ATOMSET : private set<ATOM>
//////////////////////////////////////////////////////////////////////////////
    {
public:
    typedef set<ATOM>::const_iterator const_iterator;
    typedef set<ATOM>::iterator       iterator;

    ATOMSET()
        {
        }

    ATOMSET(const ATOMSET &atomset2) : set<ATOM>()
        {
        add(atomset2);
        }

    void operator= (const ATOMSET &atomset2)
        {
        assert( 0 );

        if( this != &atomset2 )
            {
            clear();
            add(atomset2);
            }
        }


    iterator begin()
        {
        return set<ATOM>::begin();
        }

    const_iterator begin() const
        {
        return set<ATOM>::begin();
        }

    iterator end()
        {
        return set<ATOM>::end();
        }

    const_iterator end() const
        {
        return set<ATOM>::end();
        }

    void add(const ATOM& atom)
        {
        insert(atom);
        }

    void add(const ATOMSET& atomset)
        {
        insert(atomset.begin(),atomset.end());
        }

    bool empty() const
        {
        return set<ATOM>::empty();
        }

    bool contains(const ATOM& atom) const
        {
        return ( set<ATOM>::find(atom) != set<ATOM>::end() );
        }

    size_t size() const
        {
        return set<ATOM>::size();
        }


    void erase(const ATOM& atom)
        {
        set<ATOM>::erase(atom);
        }
               
    void clear()
        {
        set<ATOM>::erase(begin(),end());
        }

    template <class F>
    void foreach(F fn) const
        {
        for( const_iterator j=begin(); j != end(); j++)
            fn(*j);
        }

    // "continuing" indicates whether this ATOMSET should be printed
    // as a continuation of another ATOMSET, which is not true by
    // default.
    // returns true if anything was printed
    bool print(ostream& out, bool continuing=false) const
        {
        if(begin() == end())
            return false;
        for( ATOMSET::const_iterator i = begin();
            i != end();
            i++ )
            {
            if ( !(*i).getPredItem().isCensored() )
                {
                // Do not print a comma before the first atom
                // except if we are in continuing mode.
                if( i != begin() || continuing)
                    out << ", ";
                out << (*i);
                }
            }

        return true;
        }

    /** filters and then prints atoms given to it.
     */
    class printFilter
	{
	ostream &out;
	bool continuing;

    public:
        bool didPrint;

        printFilter()
            : out(cout)
            {
            assert( 0 );
            }

        printFilter(const printFilter& p)
            : out(p.out), continuing(p.continuing), didPrint(p.didPrint)
            {
            }

        printFilter(ostream& out2, bool continuing2)
            : out(out2), continuing(continuing2), didPrint(false)
            {
            }

        void operator=(const printFilter&)
            {
            assert( 0 );
            }

	void operator()(ATOM const &a)
	    {
	    if( ! a.getPredItem().isCensored() )
		{
		if (continuing)
		    out << ", ";
		else
		    continuing=true;
                if( ObjectsExist )
                    a.printWithoutObjectID(out);
                else
                    out << a;
		didPrint=true;
		}
	    }
	};

    /** prints a censored version of this ATOMSET to the given output stream.
     * Uses globals PredFilter and PredPFilter that specify the wanted
     * predicates. If both are empty, no filtering will take place, and the
     * printout is verbatim.
     * @param out the destination output stream.
     * @param continuing whether the output continues preceding output.
     * @return whether something was printed.
     */
    bool printCensored(ostream& out, bool continuing) const
        {
        if ( !PredFilter.empty() || !PredPFilter.empty() || ProjectionsExist
             || ObjectsExist )
	    {
            printFilter filter = for_each(begin(),end(),
                                          printFilter(out,continuing));
            return filter.didPrint;
            }
        else
            return print(out,continuing);
        }

    //////////////////////////////////////////////////////////////////////////
    class FIND_RESULT
    //////////////////////////////////////////////////////////////////////////
        {
        const NAMES_ITEM       &pattern;
        ATOMSET::const_iterator i, end;
#ifndef NDEBUG
        const ATOMSET*          assertAtomset;
        size_t                  assertSize;
#endif
    public:
        FIND_RESULT()
            : pattern(pattern) // This is a cute trick to avoid warnings
                               // about pattern being uninitialized.
            {
            memset(this,0,sizeof(FIND_RESULT));
            }

        FIND_RESULT(const FIND_RESULT &f)
            : pattern(f.pattern), i(f.i), end(f.end)
            {
#ifndef NDEBUG
            assertAtomset=f.assertAtomset;
            assertSize=f.assertSize;
#endif
            }

        FIND_RESULT(const NAMES_ITEM &pattern2, const ATOMSET &atoms)
            : pattern(pattern2)
            {
            end=atoms.end(); 

            for(i=atoms.begin(); i != end; i++)
                if( (*i).getPredItem() == pattern )
                    break;

#ifndef NDEBUG
            assertAtomset=&atoms;
            assertSize=atoms.size();
#endif
            }

        void operator=(const FIND_RESULT &f)
            {
            memcpy(this,&f,sizeof(FIND_RESULT));
            }


        inline bool atEnd() const
            {
            return i == end;
            }

        inline void operator++()
            {
            assert( i != end );
            assert( assertSize == assertAtomset->size() );

            i++;

            // Have we exhausted all ATOMs in this ATOMSET with the
            // required predicate name?
            if( i != end  &&  (*i).getPredItem() != pattern )
                i=end;
            }

        inline void operator++(int)
            {
            ++(*this);
            }

        inline const ATOM& operator*() const
            {
            assert( i != end );
            assert( assertSize == assertAtomset->size() );

            return *i;
            }

        inline const ATOM* operator->() const
            {
            return &(operator*());
            }
        };

    // Note: It is the responsibility of the caller to deallocate the memory
    // consumed by the return value.
    inline void find(const ATOM &pattern, FIND_RESULT &f) const
        {
        FIND_RESULT result(pattern.getPredItem(),*this);
        f = result;
        } 
    };

//////////////////////////////////////////////////////////////////////////////
inline ostream& operator<< (
//
    ostream&       out,
    const ATOMSET& atomset )
    {
    out << "{";
    atomset.print(out);
    out << "}";
 
    return out;
    }

#endif

// Local Variables:
// mode: c++
// c-file-style: "dl"
// End:
