//////////////////////////////////////////////////////////////////////////////
// grounding.h

#ifndef GROUNDING_H
#define GROUNDING_H

typedef vector<bool> BOOLS;

enum INTERPRETATIONS_RANGE {rI,rDeltaNF,rI_DeltaNF};
enum TYPECLAUSE { rule, strongConstraint, weakConstraint };

extern const char *RangeName[];	// the above ranges as text

class OrderedBody;

/** cell is a class which stores information about a literal's arguments.
 * Arguments are bound as a consequence of the grounding of previous
 * literals with commune arguments. The class cell stores also
 * information about the ground atoms which are allowed to ground the
 * literal, in case their matching is successful.
 */
class cell
    {
    enum LOOKING_IN {PositiveI,UndefinedI,PositiveDeltaNF,UndefinedDeltaNF};

    typedef ATOMSET::FIND_RESULT TABLE;

    /// Status stores whether arguments are unbound.
    /// Status[i] is true, iff the i-th argument of the literal is unbound.
    BOOLS Status;

    /// The predicate type of the current literal. 
    PREDICATE_NAMES::TYPE type;

    /// pInstances points to the current ground atom named like this literal.
    /// For EDB literals, pInstances points to EDB.
    /// For literals having interpretations range rI, pInstances points to I 
    /// (positive or undefined part). 
    /// For literals having interpretations range rDeltaNF, pInstances points 
    /// to deltaNF (positive or undefined part). 
    /// Finally, for literals having interpretations range rI_DeltaNF, pInstances 
    /// points either to I or deltaNF (positive or undefined parts). 
    TABLE *pInstances;  
    
    /// count1 and count2 provide counters for built-in predicates.
    unsigned count1;
    unsigned count2;

    /// Interpretations range determines which ground atoms are matched with 
    /// the literal.
    /// A value of range rI allows matching with atoms generated in all but
    /// the last iteration (atoms in I).
    /// A value of range rDeltaNF allows matching with atoms generated in the 
    /// last iteration (atoms in deltaNF).
    /// A value of range rI_DeltaNF allows matching with atoms generated in 
    /// all iterations (atoms in I or atoms in deltaNF).
    /// Non-recursive literals always have range rI.
    INTERPRETATIONS_RANGE interpretationsRange;

    /// Variable lookingIn is used in order to remember which set of ground 
    /// atoms we are examining (atoms in the positive or undefined part of I 
    /// or atoms in the positive or undefined part of deltaNF).
    LOOKING_IN lookingIn;

    void initializeStatus(const LITERAL&, const TERM[]);
public:
    cell() 
        : pInstances(0), count1(0), count2(0)
        {
        assert(0);
        }

    cell(const cell&);

    cell(
        const LITERAL&, 
        const TERM[],
        INTERPRET&, 
        INTERPRET*, 
        unsigned*, 
        unsigned*, 
        INTERPRETATIONS_RANGE);
    ~cell();

    void operator=(const cell&);

    BOOLS &get_Status() { return Status; }
    bool atEnd();
    TABLE *getSubstAtom();
    void goToNextSubstAtom(const LITERAL&, INTERPRET&, INTERPRET*);

    /** Provide access to counter 1.
     * The counters are used to enumerate over integer ranges in arithmetics.
     * @return the current value of counter 1.
     */
    unsigned counter1() { return count1; }

    /** Provide access to counter 2.
     * The counters are used to enumerate over integer ranges in arithmetics.
     * @return the current value of counter 2.
     */
    unsigned counter2() { return count2; }

    /** Move counter 1 to the next value.
     */
    void advance1() { count1++; }

    /** Move counter 2 to the next value.
     */
    void advance2() { count2++; }

    /** Load counter 1 with zero.
     * The enumeration relying on this counter will start anew.
     */
    void reset1() { count1=0; }

    /** Load counter 2 with zero.
     * The enumeration relying on this counter will start anew.
     */
    void reset2() { count2=0; }
    };

class BACKJUMPING
    {
    // bodySize stores the number of literals in the body.
    int bodySize;

    // A vector of n_par elements containing for each variable 
    // the index of the literal binding it.
    vector<int> binder;

    // A vector of bodySize elements containing, for each literal Li in 
    // the body, the index of the closest literal (among L1,...,Li-1)  
    // binding one of the variables of Li. 
    vector<int> closestBinderVars;

    // A vector of bodySize elements containing, for each positive 
    // regular literal Li in the body, the index of the closest literal 
    // (among L1,...,Li-1) binding one of the variables of the 
    // dependencies set of Li.
    // For aggregate literals and negative literals the vector does not 
    // make sense.
    vector<int> closestBinderDS;

    // A vector of bodySize elements containing, for each positive regular 
    // literal Li in the body, the index of the closest literal (among 
    // L1,...,Li-1) binding one of the output variables (outputVars).
    // For aggregate literals and negative literals the vector does not 
    // make sense.
    vector<int> closestBinderOV;

    // A matrix of bodySize rows and n_par columns storing,
    // for each positive regular literal Li in the body and for each 
    // variable Vi, whether Vi belongs to the dependencies set of Li, i.e.
    // depSet[i][j] is true, iff the j-th variable belongs to the 
    // dependencies set of the i-th literal in the body.
    // Intuitively, the dependencies set of a literal L in OrdB
    // is the set of variables depending on the instantiation of
    // the literal L. For further details see NMR04 paper....!!!!!!!.
    vector< vector<bool> > depSet;

    // A vector of n_par elements storing output variables.
    // Variables appearing in weak constraints or in the head of a rule
    // are all output vars. 
    // Furthermore, variables appearing in the body of a rule (or in 
    // a constraint) are output vars if they occur in an unsolved literal
    // or in a literal whose dependencies set contains some other output 
    // variables.  
    vector<bool> outputVars;

    // A matrix having bodySize rows and n_par columns and storing for
    // each aggregate its global variables.
    // globalVarsInAggr[i][j] is true, iff the j-th variable 
    // is a global variable for the i-th literal in the body.
    // rows corresponding to regular literal are left empty.  
    vector< vector<bool> > globalVarsInAggr;

    // The index of the last regular positive (non propositional) literal 
    // in the body.   
    int lastRegularAtom;

    // The index of the last literal in the body binding an output var.
    int greatestBinderOV;    

    // Functions computing structures needed by the BJ algorithm. 
    void computeBinders(const OrderedBody&, const int&, vector<bool>&);
    void updateClosestBinder(vector<int>&, const int&, const unsigned&);
    void computeClosestBinderVars(const OrderedBody&, const int&);
    void computeDepSet(const OrderedBody&, const int&);
    void computeOutputVars(const OrderedBody&, const int&, const bool*); 

    // Auxiliary functions. 
    void addVarsTo(vector<bool>&, const int&, const OrderedBody&);
    void computeGlobalVarsInAggregates(const OrderedBody&);
    bool intersectionR_A(const LITERAL&, const unsigned&);
    bool intersectionR_R(const LITERAL&, const LITERAL&);
    bool intersectionA_A(const unsigned&, const unsigned&);
    bool intersection(const OrderedBody&, const unsigned&, const unsigned&);
     
public:
   
    BACKJUMPING()
        {
        assert(0);
        }

    BACKJUMPING(const BACKJUMPING&)
        {
        assert(0);
        }
 
    BACKJUMPING(const OrderedBody&, bool*);

    void operator=(const BACKJUMPING&)
        {
        assert(0);
        }

    ~BACKJUMPING()
        {
        }

    void computeStructures(
        const OrderedBody&, 
        const bool*, 
        const TYPECLAUSE&,
        const bool&);
    inline int getBinder(const int&) const ;
    inline int getClosestBinderVars(
        const OrderedBody&, 
        CONSTRAINT::const_iterator) const; 
    inline int getClosestBinderDS(
        const OrderedBody&, 
        CONSTRAINT::const_iterator) const;
    inline int getClosestBinderOV(const unsigned&) const;
    inline bool isOutputVar(const int&) const;
    int computeGreatestBinderOV(const OrderedBody &) const;
    int getGreatestBinderOV() const;  
    void resetStructures(bool*);
    };

typedef vector<cell> CELLS;

class GROUNDING
    {
    /** Map variables to their bindings.
     * When the i-th variable is unbound, vTOc[i] is bound to a NullConst.
     * Otherwise, vTOc[i] contains the TERM of type constant (symbolic or
     * numerical) that is the current binding of the i-th variable.
     */
    TERM vTOc[n_par]; 

    /** Store whether predicates are completely defined.
     * solved[i] is true, iff the i-th predicate has no undefined ground
     * instances.
     * If the i-th predicate is not IDB, solved[i] is not used.
     */
    bool *solved;
    
    /** Store whether predicates were already considered in grounding.
     * done[i] is true, iff the i-th predicate is part of a preceeding
     * component of the non-ground dependency graph, which has already been
     * processed.
     * If the i-th predicate is not IDB, done[i] is not used.
     */
    bool *done;

    /** Store the number of ground instances of predicates.
     * size[i] contains the number of ground instances of the i-th predicate.
     */
    unsigned *size;

    INTERPRET *deltaNF;
    
    INTERPRET *newNF;

    /** Store the number of occurances of predicates in deltaNF.
     * deltaSize[i] contains the number of ground instances of the i-th
     * predicate found in deltaNF.
     */
    unsigned *deltaSize;

    /** Store the number of occurances of predicates in newNF.
     * newSize[i] contains the number of ground instances of the i-th
     * predicate found in newNF.
     */
    unsigned *newSize;
 
    /** RecursiveAggrRule stores information related to recursive aggregates,
     * so that their instantiation can be postponed to the end of the 
     * grounding process.
     */
    struct RecursiveAggrRule
        {
        GRULE rule;

        // Recursive aggregates in "rule".
        vector<LITERAL> recursiveAggregates;

        TERM *v2c;

        RecursiveAggrRule()
             {
             assert(0);
             }

        RecursiveAggrRule( const RecursiveAggrRule &r )
            : rule(r.rule), 
              recursiveAggregates(r.recursiveAggregates)
            {
            if( r.v2c )
                {
                v2c = new TERM[n_par];
                memcpy( v2c, r.v2c, n_par * sizeof(TERM) );
                }
            }
        
        RecursiveAggrRule(
            const GRULE &r,
            const vector<LITERAL> &a,
            const TERM v2c1[] )
            :  rule(r), 
               recursiveAggregates(a)
            {
            if( v2c1 )
                {
                v2c = new TERM[n_par];
                memcpy( v2c, v2c1, n_par * sizeof(TERM) );
                }
            }

        void operator=( const RecursiveAggrRule & )
            {
            assert(0);
            }

        ~RecursiveAggrRule()
            {
            if( v2c )
                delete[] v2c;  
            }
        };

    vector<RecursiveAggrRule> recursiveAggrRules;

    DEPGRAPH<ATOM,CONSTRAINTS,PROGRAM> &NonGroundDepGraph;
    
    bool printInstantiation;
    enum STATUS { OK, ConstraintAlwaysViolated, AggregateInvalidUse };
    STATUS status;

    template<class T>
    bool failedInstantiation(const T&) const;

    bool trueHead(
        const DISJUNCTION&, 
        const INTERPRET&, 
        const INTERPRET&,
        const TERM []) const;
    bool isAlreadyInstantiated(const RULE&, unsigned&) const;
    bool isAlwaysFalse(const RULE&) const;
    bool negativeLiteralIsFalse(const LITERAL&, const INTERPRET*) const;
    bool ignoreConstraint(const CONSTRAINT&) const;
    bool checkViolatedAggregate(const LITERAL&, TERM []);
    bool hasUnstratifiedAggregates(const RULE&, unsigned int) const;
    bool isRecursiveAggregate(const ATOM&, unsigned) const;
    bool findSingleDefinition(
        const PROGRAM &,
        const ATOM &,
        const RULE **) const;

    void init();
   
    bool Match(const LITERAL&, cell&, TERM[n_par]);
    bool MatchBuiltin(const LITERAL&, cell&);
    template<class T>
    bool MatchAggregate( const T&, TERM[n_par] );
 
    void AddToGroundIDB(
        const RULE&, 
        bool&, 
        INTERPRET*,
        INTERPRET*, 
        unsigned*,
        const unsigned);
    void AddToGroundCONSTRAINTS(
        const CONSTRAINT&, 
        bool&, 
        INTERPRET*, 
        INTERPRET*, 
        unsigned*,
        const unsigned);
    void AddToGroundWEAKCONSTRAINTS(
        const WEAKCONSTRAINT&, 
        bool&, 
        INTERPRET*, 
        INTERPRET*, 
        unsigned*,
        const unsigned);

    // begin BACKJUMPING functions 
    void computeHeadVars(const RULE&, bool*);

    void backToLit(
        OrderedBody&, 
        CONSTRAINT::const_iterator&,
        const int&, 
        vector<cell>&, 
        TERM [n_par]);

    void updateCSB( 
        const bool&,
        const int&,
        int&,
        const int&,
        const BACKJUMPING& );

    void backFromSolutionFound(
        OrderedBody&,
        CONSTRAINT::const_iterator&,
        BACKJUMPING&, 
        int&, bool&,
        vector<cell>& );
// end  BACKJUMPING functions

    void InstantiateRule(
        const RULE&, 
        bool, 
        bool&, 
        INTERPRET*, 
        INTERPRET*, 
        unsigned*, 
        unsigned*, 
        unsigned,
        const unsigned);
    void InstantiateConstraint(const CONSTRAINT&, unsigned);
    void InstantiateWeakConstraint(const WEAKCONSTRAINT&, unsigned);

    template<class TYPE>
    void InstantiateBody(
        OrderedBody&, 
        const TYPECLAUSE&,
        bool&, 
        const TYPE&, 
        bool&, 
        INTERPRET*, 
        INTERPRET*, 
        unsigned*, 
        unsigned*,
        void(GROUNDING::*AddFunction)(
            const TYPE&, 
            bool&, 
            INTERPRET*, 
            INTERPRET*, 
            unsigned*,
            unsigned),
        const unsigned,
        bool*);

    void RestoreAggregate(AGGREGATEATOM &);
    void Restore(const LITERAL&, const BOOLS&, TERM[n_par]);
    
    void UpdateI(const INTERPRET&, unsigned*);
        
    void SimplifyGroundIDB();

    void fillGConjunction(
        GCONJUNCTION&, 
        const CONJUNCTION&,
        const TERM [],
        const INTERPRET*,
        const unsigned,
        vector<LITERAL>* );
            
public:

    GROUNDING()
        : NonGroundDepGraph(NonGroundDepGraph)
        {
        assert(0);
        }
    
    GROUNDING(
        DEPGRAPH<ATOM,CONSTRAINTS,PROGRAM> &dg,
        const bool printInst)
        : NonGroundDepGraph(dg), printInstantiation(printInst)
        {
        init();
        status = OK;
        }
    
    GROUNDING(const GROUNDING &g)
        : NonGroundDepGraph(g.NonGroundDepGraph)
        {
        assert(0);
	}

    void operator=(const GROUNDING&)
        {
        assert(0);
	}
        
    void cleanMemory() const;
     
    bool Ground(); 
 
    };

    class initializeSize : public unary_function<ATOM&,void>
        {
        unsigned *size;  

    public:

        initializeSize(unsigned *size2)
            :size(size2)
            {
            assert( size );
            }

        void operator() (const ATOM &a) const
            {
            assert( ! a.isBuiltin() );

            size[a.getIndex()]++;
            }
        };

    class UpdateIpositive : public unary_function<ATOM&,void>
        {
        unsigned *deltaSize;
        unsigned *size;

    public:

        UpdateIpositive(unsigned *deltaSize2, unsigned *size2)
            : deltaSize(deltaSize2), size(size2)
            {
            assert( size && deltaSize );
            }

        void operator() (const ATOM &a) const
            {
            assert( ! a.isBuiltin() );
            
            size[a.getIndex()]++;
            deltaSize[a.getIndex()]=0;
            
            I.addPositive(a,false);
            }
        };

    class UpdateIundefined : public unary_function<ATOM&,void>
        {
        unsigned *deltaSize;
        unsigned *size;

    public:
        UpdateIundefined(unsigned *deltaSize2, unsigned *size2)
            : deltaSize(deltaSize2), size(size2)
            {
            assert( size && deltaSize );
            }

        void operator() (const ATOM &a) const
            {
            assert( ! a.isBuiltin() );
            
            size[a.getIndex()]++;
            deltaSize[a.getIndex()]=0;
            
            I.addUndefined(a);
            }
        };

#endif

// Local Variables:
// mode: c++
// c-file-style: "dl"
// End:
