%{
//////////////////////////////////////////////////////////////////////////////
// parser.y

#include "builtin.h"

#define STRINGIFY(X) STRINGIFY1(X)
#define STRINGIFY1(X) #X

LOCAL_VARIABLES LocalVariables;

NEG_ATOMS NegativeAtoms;

#define INVALID_GUARD "#invalid_guard#" 

//////////////////////////////////////////////////////////////////////////////

/** builds a new TERM with the given name.
 * A name starting with a capital letter means that this should be a variable.
 * It is generated as a constant otherwise.
 * @param id the name.
 * @return the new TERM.
 */
TERM *newTERM(const char* id)
    {
    if (isupper(id[0]))
	return new TERM(LocalVariables.record(id));
    else
	return new TERM(id);
    }

/** builds a new TERMS list with the given TERMs.
 * @param new1 the first member.
 * @param new2 (optional) the second member.
 * @param new3 (optional) the third member.
 * @return the new TERMS.
 */
TERMS* newTERMS(
    const TERM* new1,
    const TERM* new2,
    const TERM* new3 )
    {
    TERMS* x=new TERMS;

    x->push_back(*new1);
    if (new2)
        x->push_back(*new2);
    if (new3)
        x->push_back(*new3);

    return x;
    }

/** builds a new built-in ATOM.
 * @param type specifies exact built-in.
 * @param terms the list of parameters.
 * @return the new ATOM.
 */
ATOM* newBuiltin(BUILTIN type, const TERMS* terms)
    {
    return new ATOM(type,terms);
    }

/** builds a new negative ATOM with the given name.
 * @param name the ATOM's intended name.
 * @param terms the list of parameters.
 * @return the new ATOM.
 */
ATOM* newNegativeATOM(const char *name, const TERMS *terms)
    {
    char *negname=new char[strlen(name)+2];

    negname[0] = '-';
    strcpy(negname+1,name);

    ATOM *atom = new ATOM(negname,terms,PREDICATE_NAMES::typeUndef);

    delete[] negname;

    NegativeAtoms.insert(atom->getIndex());

    return atom;
    }

enum WCType { WCfree, WCnone, WClayer, WCweight, WCfull };

/** determines whether a given weak-constraint-type is compatible with
 * previous ones.
 *
 * An error is issued if the given type is incompatible.
 *
 * @param requested the type of the current constraint.
 * @return whether the check suceeded.
 */
static bool checkWC(WCType requested)
    {
    static WCType state=WCfree;

    if (state==WCfree)
	state=requested;
    else if (state!=requested)
	{
	yyerror("Weak constraint has different weight-format than others");
        parser_errors++;

        return false;
	}

    return true;
    }

static unsigned newObjectID(const char* id, OIDindices* oind)
    {
    ObjectIDs.push_back(OID(id,oind));
    CurrentOID = ObjectIDs.size() - 1;

    return ObjectIDs.size()-1;
    }

OIDindices* newOIDindex(OIDindices* oind, const char* id)
    {
    assert(oind);

    if( (TERM::Strings.find(NAMES_ITEM(id))).second )
        {
        OIDs::const_iterator i = ObjectIDs.begin();

        for( ; i != ObjectIDs.end() && (*i) != id; i++ )
            {
            }

        if( i != ObjectIDs.end() )
            {
            (*oind).push_back(i - ObjectIDs.begin());
            return oind;
            }
        }

    yyerror("Undefined object identifier");
    parser_errors++;
    return oind;
    }


class DefeasibilitySetter : public unary_function<ATOM&,void>
    {
    const unsigned oi;
    const ATOM &pred;
public:
    DefeasibilitySetter(const unsigned oi2, const ATOM &pred2)
        : oi(oi2), pred(pred2)
        {
        }

    void operator() (const ATOM &a) const
        {
        if( a == pred )
             {
             if( TraceLevel >= 3)
                 cdebug << "Defeasible atom: " << pred << " in "
                        << ObjectIDs[oi] << endl;
             ObjectIDs[oi].addDefeasible(pred);
             }
        }
    };

static void set_defeasible(const ATOM &pred, const OIDindices *oind)
    {
    if( ! oind )
        return;

    for( OIDindices::const_iterator i = oind->begin();
         i != oind->end();
         i++ )
        {
        if( ObjectIDs[*i].isDefeasible(pred) )
            {
            if( TraceLevel >= 3 )
                cdebug << pred << " is already defeasible in " 
                       << ObjectIDs[*i]
                       << ". Aborting defeasibility check." << endl;
            return;
            }

        // check rules in ObjectIDs[*i]
        ObjectIDs[*i].getEDB()->foreach( DefeasibilitySetter(*i,pred) );

        for( RULES::const_iterator j = (ObjectIDs[*i].getIDB())->begin();
             j != (ObjectIDs[*i].getIDB())->end();
             j++ )
            {
            for( DISJUNCTION::const_iterator k = j->getHead().begin();
                 k != j->getHead().end();
                 k++ )
                if( *k == pred )
                    {
                    if( TraceLevel >= 3 )
                        cdebug << "Defeasible atom: " << pred << " in " 
                               << ObjectIDs[*i] << endl;
                    ObjectIDs[*i].addDefeasible(pred);
                    }
            }

        set_defeasible(pred,ObjectIDs[*i].getParentOIDindices());
        }
    }

/////////////////////////////////////////////////////////////////////////////
/**Substitute v with the constant c when occurring in atom a as variable.
 */
static void substitute( ATOM &a,
    const TERM &c,
    const TERM &v )
    {
    assert( a.isRegularAtom() );
    assert( c.getType() == TERM::String 
            || c.getType() == TERM::Integer );
    assert( v.getType() == TERM::Variable );

    if( a.getParams() )
        replace( a.getParamsForModification()->begin(), 
                 a.getParamsForModification()->end(),
                 v, c );
    }

/////////////////////////////////////////////////////////////////////////////
/**Substitute v with c in each atom of disjunction d. 
 */
static void substitute( DISJUNCTION &d,
    const TERM &c,
    const TERM &v )
    {
    assert( c.getType() == TERM::String 
            || c.getType() == TERM::Integer );
    assert( v.getType() == TERM::Variable );

    for( DISJUNCTION::iterator i = d.begin();
         i != d.end();
         i++ )
        substitute( (*i),c,v );
    }

/////////////////////////////////////////////////////////////////////////////
/**Substitute in each positive literal appearing in c the occurrences of 
 * v with the constant c.
 */
static void substitute( CONJUNCTION &con,
    const TERM &c,
    const TERM &v )
    {
    assert( c.getType()==TERM::String 
            || c.getType() == TERM::Integer );
    assert( v.getType() == TERM::Variable );

    for( CONJUNCTION::iterator l = con.pos_begin();
         l != con.pos_end();
         l++ )
        {
        if( (*l).isAggregate() )
            {
            // Substitute v by c in the auxiliary atom.
            substitute( (*l).getAggregate()
                        .getAuxAtomForModification(),c,v );
            // If guards are equal to the var term substitute
            // them with the constant value. 
            if( (*l).getAggregate().getUpperGuardAsVar() == v )
                (*l).getAggregate().setUpperGuard(c);
            if( (*l).getAggregate().getLowerGuardAsVar() == v )
                (*l).getAggregate().setLowerGuard(c);
            }
        else
            substitute( (*l),c,v );
        }
    }

/////////////////////////////////////////////////////////////////////////////
/**Substitute in each atom occurring in c, and d all of the occurrences
 * of v with c.
 */
inline static void substitute( 
    DISJUNCTION       *dis,
    CONJUNCTION       &con,
    const TERM        &c,
    const TERM        &v )
    {
    if( dis )
        substitute(*dis,c,v);
    substitute(con,c,v);
    }

/////////////////////////////////////////////////////////////////////////////
/* For each equality builtin in conjunction c in which a constant is
 * assigned to a variable, substitute each occurrence of the variable 
 * by the constant in all the literals appearing in c, and all the
 * atoms appearing in d.
 */
static void bindTermOfBuiltinEquality( DISJUNCTION *d, CONJUNCTION &c )
    {
    for( CONJUNCTION::iterator l = c.begin();
         l != c.end();
         l++ )
        {
        // Check whether the literal is an equality builtin.
        if( (*l).isBuiltinEquality() )
            {
            // Note: don't nitialize t1, antd t2 as reference
            // because of side effects tested upon procedure
            // substitute(ATOM &, const TERM &, const TERM &).
            const TERM t1 = *( (*l).getParams()->begin() );
            const TERM t2 = *( (*l).getParams()->begin()+1 );

            // Check whether the first argument is a variable and 
            // the second argument is a constant.
            if( t1.getType() == TERM::Variable 
                && ( t2.getType() == TERM::String 
                     || t2.getType() == TERM::Integer ) )
                {
                substitute(d,c,t2,t1);
                }
            // Check whether the first argument is a constant and
            // the second argument is a variable.
            else if( ( t1.getType() == TERM::String
                       || t1.getType() == TERM::Integer )
                     && t2.getType() == TERM::Variable )
                {
                substitute(d,c,t1,t2);
                }
            }
        }
    }

static void newRule(const RULE &rule)
    {
    if( rule.getBody() )
        bindTermOfBuiltinEquality( &( rule.getHeadForModification() ),
                      *( rule.getBodyForModification() ) );

    if( ! rule.isSafe() )
        {
        yyerror("Rule is not safe");
        parser_errors++;
        }
    else if( LocalVariables.getNumber() > n_par )
        {
        yyerror("Rule contains too many variables");
        parser_errors++;
        }
    else 
        {
        assert( rule.hasHead() );

        const DISJUNCTION &head = rule.getHead();

        // Is this rule in fact a fact?
        if( ! rule.isDisjunctive()  &&  ! rule.getBody() )
            {
            const ATOM &a=*(head.begin());

            if( a.refineType(PREDICATE_NAMES::typeEDB) )
                {
                assert( CurrentOID < ObjectIDs.size() );
                ObjectIDs[CurrentOID].addEDBFact(a);

                OIDindices* oind = ObjectIDs[CurrentOID].getParentOIDindices();

                if(oind)
                    {
                    if( TraceLevel >= 3 )
                        cdebug << "Defeasibility checking " << a 
                               << " in " << ObjectIDs[CurrentOID] 
                               << "." << endl;

                    if( a.occursComplementary() )
                        {
                        ATOM complement(a);
                        a.getComplementary(a.getParams(),complement);
                        set_defeasible(complement, oind);
                        }
                    }
                }
            else
                {
                // Using an IDB predicate as EDB.
                a.refineType(PREDICATE_NAMES::typeIDB);
                IDB.push_back(rule);
                }
            }
        else
            // We are dealing with a "real" rule.
            {
            for( DISJUNCTION::const_iterator i=head.begin();
                 i != head.end();
                 i++ )
                {
                if( ! (*i).refineType(PREDICATE_NAMES::typeIDB) )
                    {
                    // This predicate previously was known as EDB, so now
                    // we have to move over all instances to the IDB...
                    ATOMSET::FIND_RESULT f; 
                    EDB.find(*i, f);

                    for( ; ! f.atEnd(); f++)
                        {
                        DISJUNCTION d;
                        d.add(*f);
                        RULE r(&d,0);

                        IDB.push_back(r);

                        if( PTraceLevel >= 1 )
                            cdebug << "Moving " << *f
                                   << " from EDB to IDB as "
                                   << r
                                   << endl;
                        }

                    EDB.erasePred(*i);
                    }
                }
                    
            if( LocalVariables.getNumber() )
                IsIDBGround=false;

            assert( CurrentOID < ObjectIDs.size() );
            ObjectIDs[CurrentOID].addIDBRule(rule);

            OIDindices *oind = ObjectIDs[CurrentOID].getParentOIDindices();

            if( oind )
                {
                for( DISJUNCTION::const_iterator i=head.begin();
                     i != head.end();
                     i++ )
                    {
                    if( TraceLevel >= 3 )
                        cdebug << "Defeasibility checking " << (*i)
                               << " in " << ObjectIDs[CurrentOID] 
                               << "." << endl;

                    if( (*i).occursComplementary() )
                        {
                        ATOM complement(*i);
                        (*i).getComplementary((*i).getParams(),complement);
                        set_defeasible(complement, oind);
                        }
                    }
                }
            }
        }
    LocalVariables.clear();
    }

static void newStrictRule(const RULE &rule)
    {
    unsigned CurrentOIDStacked = CurrentOID;
    // Strict rules simply go into the default object.
    CurrentOID = 0;
    newRule(rule);
    CurrentOID = CurrentOIDStacked;
    StrictRulesExist = true;
    }

static void newConstraint(CONJUNCTION &constraint)
    {
    bindTermOfBuiltinEquality(0,constraint);

    if ( ! constraint.isSafe())
        {
        yyerror("Constraint is not safe");
        parser_errors++;
        }
    else if ( LocalVariables.getNumber() > n_par )
        {
        yyerror("Constraint contains too many variables");
        parser_errors++;
        }
    else
        {
        if( LocalVariables.getNumber() )
            IsConstraintsGround=false;

        CONSTRAINT c(constraint,false);
        Constraints.push_back(c);
        }

    LocalVariables.clear();
    }

static void newWeakConstraint(
    CONJUNCTION   *wconstraint,
    const WEIGHTS *weights )
    {
    bindTermOfBuiltinEquality(0,*wconstraint);

    if (weights)
        {
        if ( ! ( wconstraint->isSafe() && safeWeights(*weights,*wconstraint)))
            {
            yyerror("Weak constraint is not safe");
            parser_errors++;
            }
        else if( LocalVariables.getNumber() > n_par )
            {
            yyerror("Weak constraint contains too many variables");
            parser_errors++;
            }
        else
            {
            if( LocalVariables.getNumber() )
                IsWConstraintsGround=false;

            if( ! (weights->first).isVar() && ! (weights->first).isInt() )
                {
                yyerror("A weight should be a number or a variable");
                parser_errors++;
                }
            else if( ! (weights->second).isVar() && ! (weights->second).isInt() )
                {
                yyerror("A level should be a number or a variable");
                parser_errors++;
                }
            else if( (weights->first).isInt() && (weights->first).getInt() < 0 )
                {
                yyerror("Weight must be a non-negative integer.");
                parser_errors++;
                }
            else if( (weights->second).isInt() && (weights->second).getInt() <= 0 )
                {
                yyerror("Level must be a positive integer.");
                parser_errors++;
                }
            else
                {
                WEAKCONSTRAINT c(*(wconstraint),(*weights).first,(*weights).second);
                WConstraints.push_back(c);
                }
            }
        }

    LocalVariables.clear();
    }

//////////////////////////////////////////////////////////////////////////////
// Parse and standardize the lower (upper) guard of an aggregate.
// Possible parse errors are properly flagged, and a dummy guard
// is returned in order to allow to proceed in a safe and sound way.
//
// @param inclusive true if the guard is inclusive (>=), false
// else (>).
// @param isLower true if processing lower guard, false else.

static GUARD* standardizeGuard(
    TERM         *term,
    const bool    inclusive,
    const bool    isLower)
    {
    assert(term);

    bool guardOperator = true;

    // Proceed with a parsing check of the guard.
    if( ! ( term->isVar() || term->isInt() ) )
        {
        yyerror("A guard should be a number or a variable");
        parser_errors++;
        }
    else if( isLower && term->isInt() && ( term->getInt() < 0 ) )
        {
        yyerror("Lower guard should be greater-or-equal than 0");
        parser_errors++;
        }
    else if( !isLower && term->isInt() && ( term->getInt() < 0 ) )
        {
        yyerror("Upper guard should be greater-or-equal than 0");
        parser_errors++;
        }
    else if( !isLower && !inclusive && term->isInt() && 
        ( term->getInt() == 0 ) )
        {
        yyerror("Upper guard should be greater than 0");
        parser_errors++;
        }
    else if( isLower && !inclusive && term->isInt() && 
        ( term->getInt() == INT_MAX ) )
        {
        yyerror("Lower guard should be less than MAXINT");
        parser_errors++;
        }
    else
        {
        // Perform the standardization of the guard.
        assert( term );
        TERM ident;

        if( term->isInt() )
            {
            if(!inclusive)
                if( isLower ) 
                    {
                    assert( (term->getInt()) < INT_MAX );
                    ident = TERM( term->getInt() +1, 0);
                    }
                else
                    {
                    assert( term->getInt() > 0 );
                    ident = TERM( term->getInt() -1, 0);
                    }
            else
                ident = TERM(term->getInt(), 0);
            }
        else
            {
            assert( term->isVar() );
            ident = TERM(*term);
            guardOperator = inclusive;
            }

        return new GUARD( ident, guardOperator ); 
        }

    // Parse error(s) have been found, return a dummy guard.
    return new GUARD( TERM(INVALID_GUARD), guardOperator );
    }

//////////////////////////////////////////////////////////////////////////////
//
// Build an aggregate atom.
// Set the assignment flag iff the aggregate is of the form 
// X <= aggregate( Vars : Conjunction ) <= X.
// Note: to allow standardization of aggregate's conjunction
// w.r.t. the body under construction conj is processed when
// StandardizeBody is called.
// Side effects: aggrPred, lower, upper are freed upon return.
//
static ATOM* newAggregate(
    GUARD         *lower,
    GUARD         *upper,
    AGGREGATEPRED *aggrPred )
    {
    assert( lower && upper );
    assert( aggrPred );
 
    if( PTraceLevel >= 1 )
        cdebug << (*lower).first
               << ( (*lower).second ? " <= " : " < " )
               << (*aggrPred).getAggregateType().getName()
               << "( " << (*aggrPred).getVars() << " :"
               << (*aggrPred).getConjunction() << " )"
               << ( (*upper).second ? " <= " : " < " )
               << (*upper).first 
               << endl;
             
    // Build the aggregate to return.
    AGGREGATEATOM aggregate(*lower,*upper,*aggrPred);

    // Set the assignment flag if this is a assigning aggregate.
    if( lower->first.isVar() && lower->first == upper->first 
        && lower->second == upper->second )
        aggregate.setAssignment(true);

    // aggrPred,lower,upper are freed upon return.
    if(aggrPred) 
        delete aggrPred;

    assert( lower != upper );
    if(lower)
        delete lower;
    if(upper)
        delete upper;

    return new ATOM(aggregate);
    }        

/**Check whether all of the variables in vars
 * occur in any of the literals in conjunction. 
 */
static bool varsOccurInConjunction(
    const TERMS &vars,
    const CONJUNCTION &conjunction)
// @param vars is a vector of variables.
// @param conjunction is the aggregate conjunction.
    {
    assert(conjunction.size() > 0);
    assert(vars.size() > 0 );
 
    bool found = false;
    for( TERMS::const_iterator t = vars.begin();
        t != vars.end();
        t++)
        {
        for( CONJUNCTION::const_iterator lit = conjunction.begin();
            lit != conjunction.end() && !found;
            lit++)
            {
            assert( (*lit).isRegularAtom() );
            if( (*lit).getParams() && contains(*((*lit).getParams()),*t) )
                found=true;
            }
 
        if(!found)
            // Vars to be aggregated over should appear in Free.
            return false;
        }
 
    return true;
    } 

/**Return true if the variables to be aggregated over are free w.r.t.
 * the conjunction, in which the aggregate is contained, false else.
 * A variable is free if it appears in a literal of the aggregate
 * conjunction, and neither appears in any regular atom of the
 * conjunction in which the aggregate occurs, nor in any guard of an
 * aggregate atom in this conjunction.
 * Assumption: conj contains aggr.
 */
static bool freeVarsToBeAggregatedOver(
    const AGGREGATEATOM &aggr,
    const CONJUNCTION& conj )
// @param aggr is an aggregate atom.
// @param conj is the body aggr occurs in.
    {
    // Check sanity of the aggregate atom.
    assert( aggr.getFreeVars().size() > 0 );
    assert( aggr.getConjunction().size() > 0 );

    // We cannot check whether aggr is in conj, as the equality
    // operator for aggregate atoms is guaranteed to work only after
    // standardization.

    const TERMS &vars = aggr.getFreeVars();

    // First check whether all variables to be aggregated over occur
    // in the aggregate conjunction.
    if ( varsOccurInConjunction(vars,aggr.getConjunction()) )
        {
        // Now make sure that the variables to be aggregated over do
        // not occur in the rest of the conjunction in which the
        // aggregate occurs. For regular atoms, all variables are
        // considered, for aggregate atoms the guards. Note that aggr
        // is assumed to be in conj.
        for(CONJUNCTION::const_iterator lit = conj.begin();
            lit != conj.end();
            lit++)
            {
            // For each atom in conj:
            if( (*lit).isRegularAtom() )
                {
                // For a regular atom, check whether any of its variables
                // is among the variables to be aggregated over. If so,
                // we immediately return false.
                if( (*lit).getParams() )
                    {
                    for(TERMS::const_iterator var = vars.begin();
                        var != vars.end();
                        var++)
                        {
                        if( contains(*((*lit).getParams()),*var))
                            {
                            // A free variable occurs in the remainder of the body.
                            return false;
                            }
                        }
                    }
                }
            else
                {
                // For aggregate atoms, make sure that the guards are
                // not variables to be aggregated over.
                assert( (*lit).isAggregate() );
                if( contains(vars,(*lit).getAggregate().getLowerGuardAsVar())
                    || contains(vars,(*lit).getAggregate().getUpperGuardAsVar()))
                    return false;
                }
            }
        }
    else
        {
        return false;
        }
    return true;
    }

/////////////////////////////////////////////////////////////////////////////
/**Create a string representing a TERM as follows:
 * o if the TERM represents a variable build the string "_"localIndex,
 *   where localIndex is an index obtained using the class LOCAL_VARIABLES.
 *   The use of a local index its necessary to obtain for a variable an 
 *   association with a value non depending from the order of the literals 
 *   in the conjunction in which the variable appears.
 * o if the TERM represents an integer build the string "#"intValue;
 * o if the TERM represents a constant build the string "$"constantIndex;
 */
static void identToToken( TERMS::const_iterator j,
    char *str, 
    LOCAL_VARIABLES &vars)
    {
    switch( (*j).getType() )
        {
        case TERM::Variable:
            {
            // Convert the index of a variable into a string. 
            sprintf( str,"%u",(*j).getVar() );
            // Associate a local index to a string.
            unsigned localIndex = vars.record(str);
            // Convert the local index into a string.
            sprintf(str,"_%u",localIndex);
            break;
            }
        case TERM::String:
            sprintf( str,"$%lu",static_cast<unsigned long>((*j).getIndex()) );
            break;
        case TERM::Integer:
            sprintf( str,"#%u",(*j).getInt() );
            break;
        default:
            assert( 0 );
        }
    }

/////////////////////////////////////////////////////////////////////////////
/**Build the name of the auxiliary atom as follows:
 *   o the string "aux#", followed by
 *   o the strings associated with the free variables of the aggregate 
 *     function (where, for every variable, we print a local index, obtained 
 *     by using an instance of the class LOCAL_VARIABLES, and the 
 *     entries are separated by "_"), followed by
 *   o the strings associated with the predicate names and the parameters
 *     contained into the body of the auxiliary rule, and finally 
 *     followed by
 *   o the arity of the auxiliary atom.
 *
 * - Example 1 -
 * Given the following constraint
 *     :- #count{Y:b(Y,a), c(X), not a(Y,1,X)} > 1.
 * the auxiliary atom the name is:
 *     aux#_0~10_0$1~11_1!~12_0#1_1|1
 * - Example 2 -
 * Given the following constraint
 *     :- p(X), #count{Y:b(Y,a), c(X), not a(Y,1,X)} > 1.
 * the auxiliary atom the name is:
 *     aux#_0~10_1~11_0$1~12_1!~13_0#1_1|2
 * - Example 3 -
 * Given the program with th following constraints
 *     :- #count{Y:a(Y),b(Y,X)} > 5.
 *     :- p(X), #count{Y:a(Y),b(Y,X)} > 1.
 * for the first aggregate the auxiliary name is
 *     aux#_0~11_0~12_0_1!|1
 * and for the second aggregate is
 *     aux#_0~11_0~12_0_1!|2
 * - Example 4 - 
 * Given the program with the following constraints
 *     :- #count{Y: not a(Y),b(Y,X)} > 5.
 *     :- #count{Y: b(Y,X),not a(Y)} > 5.
 * for the two aggregates we obtain the same auxiliary atom name
 *     aux#_0~11_0_1!~10_0|1
 */
string createAuxAtomName( const AGGREGATEATOM &aggr,
    const CONJUNCTION &auxConj,
    unsigned arity )
    { 
    LOCAL_VARIABLES vars;
    
    // Initialize auxAtomName as "aux#".
    string auxAtomName(AUXNAME);
    
    // Initialize the string to associate to a TERM.
    char str[21];
    
    for( TERMS::const_iterator i = aggr.getFreeVars().begin(); 
        i != aggr.getFreeVars().end();  
        i++ )
        {
        assert( (*i).isVar() );
        identToToken(i,str,vars);
        auxAtomName += str;
        }

    for( CONJUNCTION::const_iterator i = auxConj.begin();
        i != auxConj.end(); 
        i++ )
        {
        if ( (*i).isNegative() )
            auxAtomName += "!";

        // Build the string to associate to the name of the current 
        // literal and append it to auxAtomName.  
        sprintf ( str,"~%u",(*i).getIndex() );
        auxAtomName += str;

        if ( (*i).getParams() )
            for( TERMS::const_iterator j = (*i).getParams()->begin(); 
                j != (*i).getParams()->end();  
                j++ )
                {
                identToToken(j,str,vars);
                auxAtomName += str;
                }
        }

    // Finally, append the arity of the auxiliary atom to auxAtomName.
    sprintf ( str,"|%u",arity );
    auxAtomName += str;

    return auxAtomName;
    }

/////////////////////////////////////////////////////////////////////////////
/**For each aggregate aggr in CONJUNCTION c of the form
 * #aggr_type( Vars : Conj(Free,Bound) )
 * is built a new auxiliary atom of the form
 * aux#conjunction2string(Vars,Bound).
 * which is the head of the auxiliary rule.
 *
 * Free is the set of variables appearing only in aggr.
 * Bound is the set of variables appearing in c and aggr conjunction.
 * Vars should be included into Free otherwise it is raised an error.
 */
static ATOM buildAuxAtom(const AGGREGATEATOM &aggr,
    const CONJUNCTION &conj,
    const CONJUNCTION &auxConj)
//
// @param aggr    is the aggregateatom.
// @param conj    is the body of the rule to be standardized.
// @param auxConj is the body of the auxiliary rule.
// returns the built auxAtom.
//
    {
    assert( aggr.getConjunction().size() > 0 );
    assert( aggr.getFreeVars().size() > 0 );
 
    TERMS auxparams(aggr.getFreeVars());
    CONJUNCTION aggrConj(aggr.getConjunction());

    for(CONJUNCTION::const_iterator aggLit = aggrConj.begin();
        aggLit != aggrConj.end();
        aggLit++)
        {
        assert( (*aggLit).isRegularAtom() );
        if( (*aggLit).getParams() )
            {
            for(TERMS::const_iterator t = (*aggLit).getParams()->begin();
                t != (*aggLit).getParams()->end();
                t++)
                {
                // If param is not a variable to be aggregated over,
                // check whether it is contained in some literals of
                // c, i.e, whether it is bound.
                if((*t).isVar())
                    {
                    for(CONJUNCTION::const_iterator lit = conj.begin();
                        lit != conj.end();
                        lit++)
                        {
                        if( (*lit).isRegularAtom() && (*lit).getParams()
                            && contains(*((*lit).getParams()),*t))
                            {
                            // Check whether t is a variable to be
                            // aggregated over. This is an error and
                            // should have been discovered earlier by
                            // freeVarsToBeAggregatedOver().
                            assert( ! contains(aggr.getFreeVars(),*t) );
                            if(!contains(auxparams,*t))
                                {
                                // t is Bound,add unique in auxparams.
                                auxparams.push_back(*t);
                                }
                            }
                        }
                    }//if((*t).isVar())
                }//for(TERMS::const_iterator t
            }//if( (*aggLit).getParams() )
        }//for(CONJUNCTION::const_iterator aggLit

    // There should always be variables to be aggregated over.
    assert( auxparams.size() > 0 );

    // Build a new name for auxAtom.
    string auxname = createAuxAtomName(aggr,auxConj,auxparams.size());   

    // Build the atom and return.
    return ATOM(auxname.c_str(),&auxparams);
    }

/////////////////////////////////////////////////////////////////////////////
/**Return true if at least one variable appearing in lit1 occurs in lit2.
 */
static bool isPartiallyBoundBy(const LITERAL &lit1, const LITERAL &lit2)
    {
    assert( lit1.isRegularAtom() && lit2.isRegularAtom() );

    if( lit1.getParams() && lit2.getParams() )
        {
        for(TERMS::const_iterator t = lit1.getParams()->begin();
            t != lit1.getParams()->end();t++)
            if( contains( *( lit2.getParams() ), *t) )
                return true;
        }

    return false; 
    }

/////////////////////////////////////////////////////////////////////////////
static bool checkAndRemoveQuotes(char *s)
// 
// Remove enclosing quotes from a quoted string.
// Return true if the string was properly quoted, false otherwise.
//
    {
    size_t l=strlen(s);

    if( s[0] != '\"'  ||  s[l-1] != '\"' )
        return false;

    memmove(s,s+1,l-2);
    s[l-2]='\0';

    return true;
    }

/////////////////////////////////////////////////////////////////////////////
/**Return true if each variable appearing in lit1 occurs in
 * a saviour literal of either c1 or c2, for which isRegularAtom() holds.
 */
static bool checkBinding(
    const LITERAL &lit1,
    const CONJUNCTION &c1,
    const CONJUNCTION &c2 )
    {
    assert( lit1.isRegularAtom() && lit1.getParams() );

    bool binding = true;
    for(TERMS::const_iterator t = lit1.getParams()->begin();
       t != lit1.getParams()->end();
       t++)
       {
       if( (*t).isVar() )
           {
           // We don't have a binding for this variable yet.
           binding = false;

           // Search a binding for t in conjunction c1.
           for(CONJUNCTION::const_iterator lit2 = c1.begin();
               !binding && lit2 != c1.end(); lit2++)
               {
               if( (*lit2).isSaviour()
                   && (*lit2).isRegularAtom()
                   && (*lit2).getParams()
                   && contains(*((*lit2).getParams()),*t) )
                   // Binding found.
                   binding = true;
               }

           // Search a binding for t in conjunction c2.
           for(CONJUNCTION::const_iterator lit2 = c2.begin();
               !binding && lit2 != c2.end(); lit2++)
               {
               if( (*lit2).isSaviour()
                   && (*lit2).isRegularAtom()
                   && (*lit2).getParams()
                   && contains(*((*lit2).getParams()),*t) )
                   // Binding found.
                   binding = true;
               }

           if(! binding)
               // There is at least a variable
               // of lit1 which is unbound.
               return false;
           }
       }
    assert(binding);

    return true;
    }

/////////////////////////////////////////////////////////////////////////////
/**Build a safe conjunction as the union of the literals appearing in c1
 * and c2, representing the body of an auxiliary rule, without considering 
 * aggregates and propositional literals appearing in c1. 
 * A literal lit1 of c1 is added iff, either it is saved by one (or more)
 * literal(s) of c1, or it saves one (or more) literal(s) appearing in c1,
 * or it is safe and all of its variables occur in one or more literals of 
 * c2.
 * If lit1 is not safe an error is returned.
 * A literal lit2 of c2 is added iff it is safe, otherwise an error is 
 * returned.
 */
static pair <CONJUNCTION,bool> buildAuxRuleBody(const CONJUNCTION &c1,
    const CONJUNCTION &c2)
//
// @param c1 the body of the rule, constraint where aggregates appear.
// @param c2 the internal conjunction of a aggregate.
// return a pair<CONJUNCTION,bool>, the second item indicates whether the
// body has been built successfully; the first item contains the body.
//
    {
    CONJUNCTION auxbody;

    for(CONJUNCTION::const_iterator lit2 = c2.begin();
        lit2 != c2.end();
        lit2++)
        {
        assert( (*lit2).isRegularAtom() );

        if( (*lit2).getParams() )
            {
            if( !( (*lit2).isSaviour() || checkBinding(*lit2,c1,c2) ) )
                {
                yyerror("Aggregate conjunction is not safe");
                parser_errors++;
                return pair<CONJUNCTION,bool>(CONJUNCTION(),false);
                }
            }

        auxbody.addOrderedUnique(*lit2);
        }

    assert ( auxbody.begin() != auxbody.end() );

    for(CONJUNCTION::const_iterator lit1 = c1.begin();
        lit1 != c1.end();
        lit1++)
        {
        if( (*lit1).isRegularAtom() )
            {
            if( (*lit1).getParams() )
                {
                if(! (*lit1).isSaviour() )
                    {
                    // Add lit1 iff all of its variables are bound.
                    if( checkBinding(*lit1,c1,c2) )
                        auxbody.addOrderedUnique(*lit1);
                    else if(! c1.isSafe() )
                        {
                        // A safety error has occurred, anyway the
                        // error message is printed out later, when
                        // the safety check is performed upon the
                        // body of the rule/constraint containing lit1.
                        if(PTraceLevel >=1)
                            cdebug << "Safety violation; "
                                      "cannot build the body of aux rule"
                                   << endl;

                        parser_errors++;
                        return pair<CONJUNCTION,bool>(CONJUNCTION(),false);
                        }
                    }
                else
                    {
                    bool found = false;
 
                    // Add the literal either if there exists
                    // a literal of c1 to be saved by lit1...
                    for(CONJUNCTION::const_iterator i = c1.begin();
                        i != c1.end() && !found;
                        i++)
                        {
                        if( (*i).isRegularAtom()
                            && ! (*i).isSaviour()
                            && isPartiallyBoundBy(*i,*lit1) )
                            {
                            found = true;
                            auxbody.addOrderedUnique(*lit1);
                            }
                        }
 
                    // ...or if there is a binding with a literal of 
                    // the inner conjunction of the aggregate.
                    for(CONJUNCTION::const_iterator i = c2.begin();
                        i != c2.end() && !found;
                        i++)
                        {
                        // In the inner conjunction of the aggregate
                        // only regular atoms are allowed.
                        assert((*i).isRegularAtom());

                        if( isPartiallyBoundBy(*i,*lit1) )
                            {
                            found = true;
                            auxbody.addOrderedUnique(*lit1);
                            }
                        }
                    }
                }
            }
        }

    return pair<CONJUNCTION,bool>(auxbody,true);
    }

/////////////////////////////////////////////////////////////////////////////
/**For each aggregate into CONJUNCTION c of the form
 * #aggr_type( Vars : Conj(Free,Bound) )
 * is built a new auxiliary rule of the form
 * aux#conjunction2string(Vars,Bound) :- Body(Free,Bound,Const).
 *
 * The head of the auxiliary rule, i.e. auxAtom(Vars,Bound), replaces
 * the aggregate's conjunction into the current aggregate
 * #aggr_type( Vars : aux#conjunction2string(Vars,Bound) )
 *
 * Bound is the set of variables appearing in both c and the aggregate's 
 * conj.
 * Free is the set of variables appearing in the aggregate only.
 * Vars must be included in Free, this is checked when building auxAtom.
 *
 * The body of the auxiliary rule Body(Free,Bound,Const) is the union of the
 * literals (but aggregates) contained in c and the aggregate's conjunction.
 */
static void standardizeBody(CONJUNCTION &c)
//
// @param c is the body of the rule to be standardized.
// Side effects: if building auxiliary rule fails auxAtom doesn't replace
// the inner conjunction of the current aggregate. 
// Note: in the special case that the current aggregate conjunction contains
// only one atom to be aggregated over, and the body of the corresponding 
// auxiliary rule would only contain that literal in its body, no auxiliary
// rule is needed as the aggregate already is standardized.
//
    {
    if( PTraceLevel >= 1 )
        cdebug << "Body standardization" << endl;

    for(CONJUNCTION::const_iterator i = c.begin();
        i != c.end();
        i++)
        {
        if( (*i).isAggregate() )
            {
            // First check admissibility (variables must be "free").
            if( ! freeVarsToBeAggregatedOver((*i).getAggregate(),c) )
                {
                yyerror("Variables to be aggregated over must be free");
                parser_errors++;
                // Do not process this aggregate. Potentially there
                // are others, process them (even if the parsing will
                // eventually fail) to discover more errors.
                continue;
                }

            // Build the body of the auxiliary rule.
            CONJUNCTION aggrConj((*i).getAggregate().getConjunction());
            pair<CONJUNCTION,bool> result1 
                 = buildAuxRuleBody(c,aggrConj);

            // Upon success build the auxiliary rule and insert
            // the auxiliary atom into the aggregate.   
            if(result1.second)
                {
                bool buildAuxRule = true;

                // Don't build the auxiliary rule, if its body
                // consists of only one atom, the variables of which
                // are precisely the variables aggregated over.
                if( result1.first.size() == 1 )
                    {
                    // We will not build the aux. rule, unless projected
                    // variables are found.
                    buildAuxRule = false;

                    // There is no front() in CONJUNCTION.
                    const TERMS* auxTerms =
                        (*(result1.first.begin())).getParams();

                    if( auxTerms )
                        for( TERMS::const_iterator t = (*auxTerms).begin();
                             t != (*auxTerms).end();
                             t++ )
                            if( (*t).isVar() 
                                && !contains((*i).getAggregate().getFreeVars(),
                                         *t) )
                                buildAuxRule = true;
                    }

                if( ! buildAuxRule )
                    {
                    // We don't need to build the aux rule.
                    if( PTraceLevel >= 1 )
                        cdebug << "  Aggregate is already standardized, "
                                  "no need to build aux rule."
                               << endl;
                    }
                else
                    {
                    // Build auxAtom(Vars,Bound). 
                    ATOM auxAtom = 
                    buildAuxAtom( (*i).getAggregate(),c,result1.first );

                    // Build the auxiliary rule, add it to the IDB and
                    // insert the aux atom into the aggregate.

                    // Build the head of the auxiliary rule.
                    DISJUNCTION auxhead;
                    auxhead.add(auxAtom);

                    if( PTraceLevel >= 1 )      
                        cdebug << "  Adding aux rule "
                               << RULE(&auxhead,&result1.first)
                               << endl;

                    newRule( RULE(&auxhead,&result1.first) );


                    // Insert auxAtom into aggregate.
                    (*i).getAggregate().setAuxAtom(auxAtom);
                    }
                }
            else
                {
                if( PTraceLevel >= 1 )
                    cdebug <<"  Cannot build body of aux rule."
                           << endl ;
                }
            }
        }
    }

//////////////////////////////////////////////////////////////////////////////
// This helper function throws a parse Error whenever MaxInteger is not 
// specified:
void checkMaxintDefined()
    {
    if( MaxInteger == -1 )
        {
        yyerror("arithmetics is only supported if "
                "#maxint has been set");
        parser_errors++;
        }
    }
 
//////////////////////////////////////////////////////////////////////////////

%}


%union {
    char*          string;
    BUILTIN        builtin;
    TERMS*         terms;
    TERM*          term;
    ATOM*          atom;
    DISJUNCTION*   head;
    LITERAL*       literal;
    CONJUNCTION*   conjunction;
    AGGREGATE_TYPE*aggregate_type;
    AGGREGATEPRED* aggregate_pred;
    RULE*          rule;
    unsigned       integer;
    WEIGHTS*       weights;
    OIDindices*    oids;
    }

%token <string>         ID NUM
%type  <string>         ident object_id
%type  <string>         importTypes importConversions exportReplace
                        username password sql_select
%type  <term>           term number_param term_sans_maxint MAXINTEGER_EQUALS_term ident_or_fixedpt ident_or_number
%type  <builtin>        binop tertop
%type  <terms>          terms terms1 terms2 terms3
%type  <atom>           user_pred builtin_pred propositional_atom aggregate
%type  <literal>        literal
%type  <head>           head
%type  <conjunction>    body 
%type  <aggregate_type> aggregate_type
%type  <aggregate_pred> aggregate_pred
%type  <rule>           rule strictrule rule0
%type  <weights>        weights
%type  <integer>        number fixedpt
%type  <oids>           object_ids inherit

%token ERROR

%token NEWLINE
%token COMMA DOT COLON SEMICOLON EXCLAMATION PLUS ASTERISK EQUALS VERTICAL_BAR

%token ID_OR_VEL CONS WEAK_CONS ANON_VAR
%token PARAM_OPEN PARAM_CLOSE
%token BRACKET_OPEN BRACKET_CLOSE
%token CURLY_OPEN CURLY_CLOSE
%token PRED_EQUAL PRED_UNEQUAL PRED_LESS PRED_GREATER
%token PRED_LESS_OR_EQ PRED_GREATER_OR_EQ
%token PRED_INT PRED_SUCC
%token PRED_ANY PRED_AVG PRED_COUNT PRED_MAX PRED_MIN PRED_SUM PRED_TIMES
%token NOT TRUE_NOT
%token MAXINTEGER

%token BOOL_QUERY
%token META_IMPORT META_EXPORT META_QUERY_EXPORT

%start input
%%

HEAD_SEPARATOR  : ID_OR_VEL | VERTICAL_BAR | SEMICOLON;

TAIL_SEPARATOR  : COMMA;

PARAM_SEPARATOR : COMMA;

OBJECTID_SEPARATOR : COMMA;


input     :
          | input line
          | error { parser_errors++; }
          ;

line      : meta_rule NEWLINE
          | object
          | clause
          | query
          | MAXINTEGER_EQUALS_term DOT
                              {
                                TERM* ident=$1;

                                if( MaxInteger == -1 )
                                    {
                                    if( ident->getType() == TERM::Integer )
                                        {
                                        MaxInteger=ident->getInt();

                                        if( MaxInteger < MaxIntegerSeen )
                                            {
                                            yyerror("#maxint must not be less "
                                                    "than the maximum integer "
                                                    "occurring in the input");
                                            parser_errors++;
                                            }
                                        }
                                    else
                                        {
                                        yyerror("parse error in #maxint statement");
                                        parser_errors++;
                                        }

                                    delete ident;
                                    }
                                else
                                    yyerror("Cannot redefine #maxint. "
                                            "Ignoring this definition");
                              }
          ;

clauses   :
          | clauses clause
          ;

object    : object_id
            inherit
            { newObjectID($1,$2); }
            CURLY_OPEN
            clauses
            CURLY_CLOSE
            { CurrentOID = 0; }
          ;

inherit   : { $$ = 0; }
          | COLON object_ids { $$ = $2; }
          ;

object_id  : ident               { $$=$1; }
          ;

object_ids : object_id          { $$ = newOIDindex(new OIDindices,$1); }
          | object_ids OBJECTID_SEPARATOR object_id { $$ = newOIDindex($1,$3); }
          ;

clause    : rule              { newRule(*($1)); delete $1; }
          | strictrule        { newStrictRule(*($1)); delete $1; }
          | constraint        { } 
          | wconstraint       { } 
          ;

MAXINTEGER_EQUALS_term : MAXINTEGER EQUALS term
                              {
                              $$=$3;
                              }
                        ;

meta_rule : import
          | export
          ;

import    : META_IMPORT PARAM_OPEN ID COMMA username COMMA password
            COMMA sql_select COMMA ID importTypes PARAM_CLOSE DOT {
                                // Import EDB atoms from a relational database.
                                // This is the five-argument version of #import;
                                // The first argument contains a valid DNS, the
                                // second contains the name of the user, the
                                // third the password of the user, the
                                // fourth contains a select string and the fifth
                                // contains a predicate name, the sixth last contains
                                // the conversion.
                                // note: to pass a select string, the user has
                                // to enclose it in " ".

                                if( PTraceLevel >=1 )
                                    cdebug << "#import " << $3 << " " << $5
                                           << " " << " " << $7 << " "
                                           << $9 <<  " " << $11 << endl;

                                odbcAddImport( $3, $5, $7,
                                               $9, $11, $12 );

                                delete[] $3;
                                delete[] $5;
                                delete[] $7;
                                delete[] $9;
                                delete[] $11;
                                delete[] $12;
                              }
          ;

importTypes: /*empty*/        {
                                $$ = new char[2];
                                strcpy($$,"");
                              }
          | COMMA ID COLON importConversions
                              {
                                if ( strcmp($2,"type") )
                                    {
                                    yyerror("type definition must be start with keyword 'type'");
                                    parser_errors++;
                                    }

                                $$ = $4;
                              }
          ;

importConversions: ID         {
                                $$ = $1;
                              }
          | importConversions COMMA ID {
                                $$ = new char[strlen($1)+strlen($3)+3];
                                sprintf($$, "%s, %s", $1, $3);
                                delete[] $1;
                                delete[] $3;
                              }
          ;

export    : META_EXPORT PARAM_OPEN ID COMMA username COMMA password 
            COMMA ID COMMA ID exportReplace PARAM_CLOSE DOT {

                                // This is the six-argument version of #export,
                                // the arguments being: a valid DNS, the name
                                // of the user, the password, the name of the
                                // predicate to export, the name of the table,
                                // and a delete condition.

                                if( PTraceLevel >=1 )
                                    cdebug << "#export " << $3 << " " << $5
                                           << " " << " "<< $7 << " " << $9 <<  " "
                                           << $11 << " " << $12 << endl;
                                ODBC_Exports.push_back(
                                     ODBC_EXPORT(  $3, $5, $7,
                                                   $11, $9, $12 ) );
                                delete[] $3;
                                delete[] $5;
                                delete[] $7;
                                delete[] $9;
                                delete[] $11;
                                delete[] $12;
                              }
          | META_QUERY_EXPORT PARAM_OPEN ID COMMA username COMMA password
            COMMA ID exportReplace PARAM_CLOSE DOT {

                                // This is the five-argument version of #export,
                                // to export query results,
                                // the arguments being: a valid DNS, the name
                                // of the user, the password, 
                                // the name of the table,
                                // and a delete condition.

                                if( PTraceLevel >=1 )
                                    cdebug << "#exportQueryResult " << $3 << " " << $5
                                           << " " << " "<< $7 << " " << $9 <<  " "
                                           << $10 << endl;
                                ODBC_Exports.push_back(
                                     ODBC_EXPORT(  $3, $5, $7,
                                                   $9, "", $10 ) );
                                delete[] $3;
                                delete[] $5;
                                delete[] $7;
                                delete[] $9;
                                delete[] $10;
                              }

          ;

username  : ID                {
                                if( ! checkAndRemoveQuotes($1) )
                                    {
                                    yyerror("user string must be enclosed by double quotes");
                                    parser_errors++;
                                    }
                                $$ = $1;
                              }
          ;

password  : ID                {
                                if( ! checkAndRemoveQuotes($1) )
                                    {
                                    yyerror("password string must be enclosed by double quotes");
                                    parser_errors++;
                                    }
                                $$ = $1;
                              }
          ;

sql_select: ID               {
                                if ( ! checkAndRemoveQuotes($1) )
                                    {
                                    yyerror("select string must be enclosed by double quotes");
                                    parser_errors++;
                                    }
                                $$ = $1;
                              }
          ;

exportReplace :               { $$ = new char[2]; strcpy($$,""); }
          | COMMA ID          {
                              if( !checkAndRemoveQuotes($2) )
                                  {
                                  yyerror("delete condition must be enclosed by double quotes");
                                  parser_errors++;
                                  }
                              $$ = $2; 
                              }
          ;

rule      : rule0 DOT
          ;

strictrule: rule0 EXCLAMATION
          ;

rule0     : head              { $$=new RULE($1, 0); delete $1; } 
          | head CONS         { $$=new RULE($1, 0); delete $1; }
          | head CONS body    { standardizeBody(*$3);
                                $$=new RULE($1,$3); 
                                delete $1;
                                delete $3;
                              }
          | ident PARAM_OPEN number_param DOT DOT number_param PARAM_CLOSE
                              {
                                const TERM var(0U);
                                TERMS params;
                                params.push_back(var);

                                const ATOM headatom(($1),&params);
                                DISJUNCTION head;
                                head.add(headatom);
 
                                const ATOM *b1=newBuiltin(BuiltinInt,&params);
 
                                const TERMS *p2=newTERMS(&var,($3));
                                const ATOM *b2=newBuiltin(BuiltinGreaterEqual,p2);
                                delete p2;
 
                                const TERMS *p3=newTERMS(&var,($6));
                                const ATOM *b3=newBuiltin(BuiltinLessEqual,p3);
                                delete p3;
 
                                CONJUNCTION body;
                                body.add(*b1);
                                body.add(*b2);
                                body.add(*b3);
 
                                delete b1;
                                delete b2;
                                delete b3;
 
                                $$=new RULE(&head, &body);
 
                                delete[] $1;
                                delete $3;
                                delete $6;
                              }
          ;

constraint: CONS body DOT     { standardizeBody(*$2);
                                newConstraint(*($2)); 
                                delete $2; 
                              }
          ;

wconstraint: WEAK_CONS body DOT weights {
                                standardizeBody(*$2);
                                newWeakConstraint($2,$4);
                                delete $2;
                                delete $4;
                              }
          ;

weights   :		      {
                                if( checkWC(WCnone) )
                                    $$ = new WEIGHTS(TERM(SCALEFACT,0), TERM(1,0));
                                else
                                    $$ = 0;
			      }
	  | BRACKET_OPEN COLON ident_or_number BRACKET_CLOSE {
                                if( checkWC(WClayer) )
                                    $$ = new WEIGHTS(TERM(SCALEFACT,0),*($3));
                                else
                                    $$ = 0;

                                delete $3;
                              }
	  | BRACKET_OPEN ident_or_fixedpt COLON BRACKET_CLOSE {
                                if( checkWC(WCweight) )
                                    $$ = new WEIGHTS(*($2), TERM(1,0));
                                else
                                    $$ = 0;

                                delete $2;
                              }
	  | BRACKET_OPEN ident_or_fixedpt COLON ident_or_number BRACKET_CLOSE {
                                if( checkWC(WCfull) )
                                    $$ = new WEIGHTS(*($2),*($4));
                                else
                                    $$ = 0;

                                delete $2;
                                delete $4;
                              }
	  ;

ident_or_fixedpt : ident      {
                                $$ = newTERM($1);
                                delete[] $1;
                              }
          | fixedpt           {
                                $$ = new TERM($1,0);
                              }
          ;

ident_or_number : ident       {
                                $$ = newTERM($1);
                                delete[] $1;
                              }
          | number            {
                                $$ = new TERM($1,0);
                              }
          ;

fixedpt	  : number	      { $$=($1)*SCALEFACT; }
	  | number DOT NUM    {
				char *err;
				static char buf[1024];
				unsigned s=strlen($3);
				if (s>SCALE)
				    {
				    yyerror("Number has more precision "
					    "than supported. Truncated");
				    strncpy(buf,($3),SCALE);
				    }
				else
				    {
				    strcpy(buf,($3));
				    memset(buf+s,'0',SCALE-s);
				    }
				buf[SCALE]='\0';
				$$=(($1)*SCALEFACT)+strtoul(buf,&err,10);
				assert( *err == '\0' );

				delete[] $3;
			      }
	  ;

number	  : NUM		      {
				char *err;
				$$=strtoul(($1),&err,10);
				assert( *err == '\0' );

				delete[] $1;
			      }
	  ;

query     : body BOOL_QUERY   { 
                                if( Query )
                                    {
                                    cout << "Query " << *($1) << " replaces " 
                                         << *Query << "." << endl;
                                    delete Query; 
                                    }

                                if( ! $1->isSafe() )
                                    {
                                    yyerror("Query is not safe");
                                    parser_errors++;
                                    }
                                else if( LocalVariables.getNumber() > n_par )
                                    {
                                    yyerror("Query contains too many variables");
                                    parser_errors++;
                                    }

                                Query=$1;

                                LocalVariables.clear();
                              }
          ;


head      : user_pred         {
                                $$=new DISJUNCTION;
                                $$->add(*($1)); 
                                delete $1;
                              }
          | head HEAD_SEPARATOR user_pred
                              {
                                $1->add(*($3)); 
                                delete $3;
                                $$=$1;
                              }
          ;


body      : literal           {
                                $$=new CONJUNCTION;
                                $$->add(*($1)); 
                                delete $1;
                              }
          | body TAIL_SEPARATOR literal
                              {
                                $1->addUnique(*($3));
                                delete $3;
                                $$=$1;
                              }
          ;



literal   : user_pred         {
                                $$=new LITERAL(false,*($1));
                                delete $1;
                              }
          | NOT user_pred     { 
                                $$=new LITERAL(true,*($2));
                                delete $2;
                              }

          | builtin_pred      { 
                                $$=new LITERAL(false,*($1));
                                delete $1;
                              }

          | aggregate         {
                                $$=new LITERAL(false,*($1));
                                delete $1;
                              }
          | NOT aggregate     { 
                                $$=new LITERAL(true,*($2));
                                delete $2;
                              }
          ;


user_pred : propositional_atom
          | ident PARAM_OPEN terms PARAM_CLOSE
                              {
                                $$ = new ATOM($1,$3,PREDICATE_NAMES::typeUndef);
			        delete[] $1;
                                delete $3;
                              }
          | TRUE_NOT ident PARAM_OPEN terms PARAM_CLOSE
                              {
                                $$=newNegativeATOM($2,$4);
				delete[] $2;
                                delete $4;
                              }
          ;

propositional_atom      : ident             {
                                $$ = new ATOM($1,0,PREDICATE_NAMES::typeUndef);
				delete[] $1;
                              }
          | TRUE_NOT ident    {
                                $$=newNegativeATOM($2,0);
				delete[] $2;
                              }
          ;

terms    : term             {
                                $$=new TERMS;

                                $$->push_back(*($1));
				delete $1;
                              }
         | terms PARAM_SEPARATOR term
                              {
                                $$=$1;

                                $$->push_back(*($3));
				delete $3;
                              }
         ;


builtin_pred : binop terms2
                              {
                                $$=newBuiltin($1,$2); 
                                delete $2;
                              }
          | EQUALS terms2    
                              {
                                $$=newBuiltin(BuiltinEquality,$2);
                                delete $2;
                              }
          | term binop term
                              {
                                TERMS* terms=newTERMS($1,$3);

                                $$=newBuiltin($2,terms);
                                delete terms;
				delete $1;
				delete $3;
                              }
          | term_sans_maxint EQUALS term
                              {
                                TERMS *terms=newTERMS($1,$3);

                                $$=newBuiltin(BuiltinEquality,terms);
                                delete terms;
                                delete $1;
                                delete $3;
                              }   
          | PRED_INT terms1
			      {
                                checkMaxintDefined();
                                $$=newBuiltin(BuiltinInt,$2);
                                delete $2;
			      }
          | PRED_SUCC terms2
			      {
                                checkMaxintDefined();
			        $$=newBuiltin(BuiltinSucc,$2);
				delete $2;
			      }
          | tertop terms3
                              {
			        $$=newBuiltin($1,$2);
				delete $2;
			      }
          | term_sans_maxint EQUALS term tertop term
                              {
                                TERMS* terms=newTERMS($3,$5,$1);

                                $$=newBuiltin($4,terms);
                                delete terms;
                                delete $1;
                                delete $3;
                                delete $5;
                              }
          | MAXINTEGER_EQUALS_term tertop term
                              {
				if( MaxInteger == -1 )
				    {
				    yyerror("#maxint used but no upper "
					    "integer bound (option -N) given");
                                    parser_errors++;
				    }

                                TERM ident(MaxInteger,0);
                                TERMS *terms=newTERMS($1,$3,&ident);
                                $$=newBuiltin($2,terms);
                                delete terms;
				delete $1;
				delete $3;
			      }
	  ;

aggregate : term PRED_LESS aggregate_pred {

                                $$=newAggregate(
                                       standardizeGuard($1,false,true),
                                       standardizeGuard(new TERM(INT_MAX,0), 
                                           true,false),
                                       $3 );
                              }
          | term PRED_LESS_OR_EQ aggregate_pred {
                                $$=newAggregate(
                                       standardizeGuard($1,true,true),
                                       standardizeGuard(new TERM(INT_MAX,0),
                                           true,false),
                                       $3 );
                              }

          | term_sans_maxint EQUALS aggregate_pred {
                                $$=newAggregate(
                                       standardizeGuard($1,true,true),
                                       standardizeGuard($1,true,false),
                                       $3 );
                              }
          | term PRED_GREATER_OR_EQ aggregate_pred {
                                $$=newAggregate(
                                    standardizeGuard(new TERM(0,0),
                                        true,true),
                                    standardizeGuard($1,true,false),
                                    $3 );
                              }
          | term PRED_GREATER aggregate_pred {
                                $$=newAggregate(
                                    standardizeGuard(new TERM(0,0),
                                        true,true),
                                    standardizeGuard($1,false,false),
                                    $3 );
                              }

          /* AGGREGATE OPERATOR BOUND */ 
          | aggregate_pred PRED_LESS term {
                                $$=newAggregate(
                                    standardizeGuard(new TERM(0,0),
                                        true,true),
                                    standardizeGuard($3,false,false),
                                    $1 );
                              }
          | aggregate_pred PRED_LESS_OR_EQ term {
                                $$=newAggregate(
                                    standardizeGuard(new TERM(0,0),
                                        true,true),
                                    standardizeGuard($3,true,false),
                                    $1 );
                              }
          | aggregate_pred EQUALS term {
                                $$=newAggregate(
                                    standardizeGuard($3,true,true),
                                    standardizeGuard($3,true,false),
                                    $1 );
                              }
          | aggregate_pred PRED_GREATER_OR_EQ term {
                                $$=newAggregate(
                                    standardizeGuard($3,true,true),
                                    standardizeGuard(new TERM(INT_MAX,0),
                                        true,false),
                                    $1 );
                              }
          | aggregate_pred PRED_GREATER term {
                                $$=newAggregate(
                                    standardizeGuard($3,false,true),
                                    standardizeGuard(new TERM(INT_MAX,0),
                                        true,false),
                                    $1 );
                              }

          /* BOUND OPERATOR AGGREGATE OPERATOR BOUND */ 
          | term PRED_LESS aggregate_pred PRED_LESS term {
                                $$=newAggregate(
                                    standardizeGuard($1,false,true),
                                    standardizeGuard($5,false,false),
                                    $3 );
                              }
          | term PRED_LESS aggregate_pred PRED_LESS_OR_EQ term {
                                $$=newAggregate(
                                    standardizeGuard($1,false,true),
                                    standardizeGuard($5,true,false),
                                    $3 );
                              }
          | term PRED_LESS_OR_EQ aggregate_pred PRED_LESS term {
                                $$=newAggregate(
                                    standardizeGuard($1,true,true),
                                    standardizeGuard($5,false,false),
                                    $3 );
                              }
          | term PRED_LESS_OR_EQ aggregate_pred PRED_LESS_OR_EQ term {
                                $$=newAggregate(
                                    standardizeGuard($1,true,true),
                                    standardizeGuard($5,true,false),
                                    $3 );
                              }
          ;

binop    : PRED_EQUAL         {
				$$=BuiltinEquality;
                              } 
         | PRED_UNEQUAL       {
				$$=BuiltinInequality;
                              }
         | PRED_LESS          {
				$$=BuiltinLess;
                              }
         | PRED_GREATER       {
				$$=BuiltinGreater;
                              }
	 | PRED_LESS_OR_EQ    {
			        $$=BuiltinLessEqual;
			      }
	 | PRED_GREATER_OR_EQ {
			        $$=BuiltinGreaterEqual;
			      }
         ;

tertop   : PLUS               {
                                checkMaxintDefined();
			        $$=BuiltinAddition;
                              } 
         | ASTERISK	      {
                                checkMaxintDefined();
				$$=BuiltinMultiplication;
                              }
	 ;

aggregate_pred : aggregate_type CURLY_OPEN terms COLON body CURLY_CLOSE
                              { 
                                $$ = new AGGREGATEPRED(*$1,*$3,*$5);
                                delete $1;
                                delete $3;
                                delete $5;
                              } 
         ;

aggregate_type: PRED_ANY
                              {
                                $$=new AGGREGATE_TYPE(AGGREGATE_TYPE::Any);
                              }
         | PRED_AVG
                              {
                                $$=new AGGREGATE_TYPE(AGGREGATE_TYPE::Avg);
                              }
         | PRED_COUNT
                              {
                                $$=new AGGREGATE_TYPE(AGGREGATE_TYPE::Count);
                              }
         | PRED_MAX
                              {
                                $$=new AGGREGATE_TYPE(AGGREGATE_TYPE::Max);
                              }
         | PRED_MIN
                              {
                                $$=new AGGREGATE_TYPE(AGGREGATE_TYPE::Min);
                              }
         | PRED_SUM
                              {
                                $$=new AGGREGATE_TYPE(AGGREGATE_TYPE::Sum);
                              }
         | PRED_TIMES
                              {
                                $$=new AGGREGATE_TYPE(AGGREGATE_TYPE::Times);
                              }
         ;
	  
terms1   : PARAM_OPEN term PARAM_CLOSE
                              {
                                $$=newTERMS($2);
				delete $2;
                              }
         ;

terms2   : PARAM_OPEN term PARAM_SEPARATOR term PARAM_CLOSE
                              {
                                $$=newTERMS($2,$4);
				delete $2;
				delete $4;
                              }
         ;

terms3   : PARAM_OPEN term PARAM_SEPARATOR term PARAM_SEPARATOR term PARAM_CLOSE
                              {
                                $$=newTERMS($2,$4,$6);
				delete $2;
				delete $4;
				delete $6;
                              }
         ;

term     : ident              {
				$$=newTERM($1);
                                delete[] $1;
			      }
         | number_param       {
                                $$=$1;
                              }
         | ANON_VAR	      {
			        $$=new TERM(LocalVariables.anonymous());
			      }
         | maxinteger         {
                                $$=new TERM(MaxInteger,0);
                              }
         | TRUE_NOT term      {
                                yyerror("Cannot use true negation for terms");
                                parser_errors++;

                                $$=$2;
                              }
         ;

term_sans_maxint : ident      {
                                $$=newTERM($1);
                                delete[] $1;
                              }
         | number_param       {
                                $$=$1;
                              }
         | ANON_VAR           {
                                $$=new TERM(LocalVariables.anonymous());
                              }
         ;

ident    : ID		      {
				$$=$1;
				if (strlen($$) > IDENT_LENGTH)
				    {
				    $$[IDENT_LENGTH]=0;
				    yyerror("identifier too long, truncated "
					    "to " STRINGIFY(IDENT_LENGTH)
					    " characters");
                                    parser_errors++;
				    } 
			      }
         | ID_OR_VEL	      {
				$$=new char[2];
				$$[0]='v';
				$$[1]=0;
			      }
         ;

number_param: number          {
                                if( MaxInteger != -1  
                                    && static_cast<int>($1) > MaxInteger )
                                    {
                                    yyerror("number outside of given "
                                            "integer range");
                                    parser_errors++;
                                    }

                                if( static_cast<int>($1) > MaxIntegerSeen )
                                    MaxIntegerSeen=$1;

                                $$=new TERM($1,0);
                              }
         ;

maxinteger : MAXINTEGER       {
                                if( MaxInteger == -1 )
                                    {
                                    yyerror("#maxint used but no upper "
                                            "integer bound (option -N) given");
                                    parser_errors++;

                                    MaxInteger=0;
                                    }
                              }
         ;

%%

// Local Variables:
// mode: c++
// c-file-style: "dl"
// End:
