//////////////////////////////////////////////////////////////////////////////
// rewrite-magic.C

#define WITH_GLOBAL_VARIABLES

#include "dl.h"
#include <stack>


// return true if a predicate is adorned, false otherwise
// a predicate is adorned if its name contains the character '#'
bool isAdorned (const NAMES_ITEM &nameAtom)
    {
    return ( strchr(nameAtom.getName(),'#') != 0 );
    }


// get the adornment from the name of an adorned atom
const char* getAdornment(const NAMES_ITEM &nameAtom)
    {
    assert( isAdorned(nameAtom) );
    char *adornment = new char[strlen(strchr(nameAtom.getName(),'#'))];
    strcpy(adornment,strchr(nameAtom.getName(),'#')+1);
    return adornment;
    }


// receive the name of an adorned atom and get the value of the
// adornment corresponding to a particular position pos
char getAdornment(const NAMES_ITEM &nameAtom, unsigned pos)
    {
    assert( isAdorned(nameAtom) );
    const char *adornment = getAdornment(nameAtom);
    assert( pos < strlen(adornment) );
    char binding = adornment[pos];
    delete[] adornment;
    return binding;
    }

// receive the name of an adorned atom and get the name of the atom
// without adornment
char* getUnadornedName(const NAMES_ITEM &nameAdAtom)
    {
    const char *adornment = getAdornment(nameAdAtom);
    unsigned len=strlen(nameAdAtom.getName()) - 
        strlen(adornment)-1;
    delete[] adornment;
    char *name= new char[len];
    strncpy(name,nameAdAtom.getName(),len);
    name[len]='\0';
    return name;
    }

// receive an atom and an adornment and builds a new adorned atom
ATOM buildAdornedAtom(const ATOM &a,const char *adornment)
    {
    // it is possible to adorn an atom only if it is IDB
    assert( a.isIDB() && !a.isPropositional() );
    char *name = new char[strlen(a.getName())+a.getArity()+2];
    strcpy(name,a.getName()); 
    strcat(strcat(name,"#"),adornment);
    ATOM aa(name,a.getParams(),a.getType()); 
    delete[] name;
    return aa;
    }

// receive an atom and an adorned name and build a new adorned atom
ATOM buildAdornedAtom(const ATOM &a,const NAMES_ITEM &adornedName)
    {
    // it is possible to adorn an atom only if it is IDB and not
    // propositional
    assert( a.isIDB() && !a.isPropositional() );
    return ATOM(adornedName.getName(),a.getParams(),a.getType());
    }


// receive a single query atom and build the adorned version
ATOM adornQueryAtom(const ATOM &query_atom)
    {
    const TERMS *params = query_atom.getParams();
    char *adornment= new char[query_atom.getArity()+1];

    // initialize the string
    strcpy(adornment,"");
    assert( params != 0 );
    for( TERMS::const_iterator i=params->begin();
         i != params->end();
         i++ )
        if( (*i).isVar() )
            strcat(adornment,"f");
        else
            // FIXME: We would need a method isConstant() here.
            {
            assert ( !(*i).isNull() );
            strcat(adornment,"b");
            }
    ATOM a(buildAdornedAtom(query_atom,adornment));
    delete[] adornment;
    return a;
    } 


// store the set of bound variables for each rule
typedef set<TERM> SetofBoundVars;


// get the set of bound variables of one adorned atom and insert
// them into the set of bound variables of the rule
// @param adorned_atom, the atom whose bound variables must be added
//        to the set of bound variables BoundVars
// @param BoundVars, the set of bound variables at the current
//        position of the considered rule
void getBoundVars(const ATOM &adorned_atom,
                  SetofBoundVars &BoundVars)
    {
    assert( adorned_atom.isIDB() );
    if ( !adorned_atom.isPropositional() ) 
        {
        const TERMS *params = adorned_atom.getParams();
        assert( params != 0 );
        const char *adornment =
            getAdornment(adorned_atom.getPredItem());
        const char *adornmentpos = adornment;

        assert( strlen(adornment)==(params->size()) );

        for( TERMS ::const_iterator i=params->begin();
             i != params->end() ;
             i++,adornmentpos++ )
            {
            assert( !(*i).isNull() );
            if( (*i).isVar() && (*adornmentpos == 'b') ) 
                BoundVars.insert(*i);
            }

        delete[] adornment;
        }
    }


// adorn a literal according to the set of bound variables collected
// so far
// @param L, the literal to be adorned
// @param BoundVars, the set of bound variables at the current
//        position of the considered rule
LITERAL adornLiteral(const LITERAL &L,const SetofBoundVars &BoundVars)
    {
    assert( L.isIDB() && !L.isPropositional() );
    char *adornment= new char[L.getArity()+1];
    strcpy(adornment,"");
    const TERMS *params = L.getParams();
    assert( params != 0 );
    for( TERMS::const_iterator i=params->begin();
         i != params->end();
         i++ )
        if( (*i).isVar() )
            {
            // if it is contained into the SetofBoundVars put it to
            // bound,else put it to false
            if( BoundVars.find(*i) != BoundVars.end() )
                strcat(adornment,"b");
            else
                strcat(adornment,"f");
            }
        else
            // the term is not a variable: it is a constant
            // FIXME: We would need a method isConstant() here.
            {
            assert( !(*i).isNull() );
            strcat(adornment,"b");
            }
    LITERAL L1(L.isNegative(),
               buildAdornedAtom(L.getAtom(),adornment));
    delete[] adornment;
    return L1;
    }


// check if an atom has at least one bound term w.r.t.
//  the current set of bound variables
// @param a, the atom to be checked if it has bound terms 
// @param BV, the set of bound variables at the current position of
//        the considered rule
bool hasBoundTerms(const ATOM &a,const SetofBoundVars &BV)
    { 
    assert( !a.isPropositional() );
    //check if the literal contains at least a bound term
    const TERMS *params = a.getParams();
    assert( params != 0 );
    for( TERMS::const_iterator k=params->begin();
         k != params->end();
         k++ )
        if( ((*k).isVar() && BV.find(*k) != BV.end())
            // FIXME: We would need a method isConstant() here.
            || !(*k).isVar() )
            return true;
    return false;
    }


// receive an adorned IDB atom and return true if it contains at least
// a bound term
// @param a, the adorned IDB atom to be checked if it has bound terms
bool hasBoundTermsIDB(const ATOM &ad)
    {
    assert( ad.isIDB() );
    if ( ad.isPropositional() ) 
        return false;
    const char *adornment = getAdornment(ad.getPredItem());
    bool bound = ( strchr(adornment,'b') != 0 );
    delete[] adornment;
    return bound;
    }


// calculate the number of bound terms in atom a w.r.t. the
// current set of bound variables
// @param a, the atom whose number of bound terms must be calculated
// @param BV, the set of bound variables at the current position of
//        the considered rule
size_t getNumberBoundTerms(const ATOM &a,const SetofBoundVars &BV)
    {
    size_t count=0;
    const TERMS *params = a.getParams();
    assert( params != 0 );
    for( TERMS::const_iterator i=params->begin();
         i != params->end();
         i++ )
        if( ( (*i).isVar() && BV.find(*i) != BV.end() )
            // FIXME: We would need a method isConstant() here.
            || !(*i).isVar() )
            count++;
    return count;    
    }


// update the set of bound variables with all the variables of a
// literal
// @param BV, the set of bound variables at the current position of
//        the considered rule
// @param L, the literal whose variables must be added to the set of
//        bound variables BV
void updateBoundVars(SetofBoundVars &BV,const LITERAL &L)
    {
    assert( !L.isPropositional() );
    const TERMS *params = L.getParams();
    assert( params != 0 );
    for( TERMS::const_iterator i=params->begin();
         i != params->end();
         i++ )
        if( (*i).isVar() && BV.find(*i) == BV.end() )
            BV.insert(*i);
    }


//////////////////////////////////////ADORN///////////////////////

// return true if a literal is a comparison built-in (except the
// equality built-in)
// FIXME: this function should probably be relocated
// @param L, the builtin literal to be checked if it a comparison
//        builtin, (except the equality built-in)
bool isComparison(const LITERAL &L)
    {
    assert( L.isBuiltin() );
    switch( L.getBuiltinID() )
        {
        case BuiltinInequality:
        case BuiltinLess:
        case BuiltinGreater:
        case BuiltinLessEqual:
        case BuiltinGreaterEqual:
            return true;       
        default: return false;
        }
    }


// return true if the variables of a literal are all bound w.r.t.
// the current set of bound variables
// @param L, the literal to be checked
// @param BV, the set of bound variables at the current position of
//        the considered rule
bool fullyBound(const LITERAL &L,const SetofBoundVars &BV)
    {
    assert( !L.isPropositional() );
    const TERMS *params = L.getParams();
    assert( params != 0 );
    for( TERMS ::const_iterator i=params->begin();
         i != params->end();
         i++ )
        if( (*i).isVar() && BV.find(*i) == BV.end() )
            return false;
    return true;
    }


// return true if a literal is permissible w.r.t. the current set of
// bound variables
// the condition of permissibility depends on the type
// of the literal and on the number of bound terms
// intuitively, a literal is permissible, if the number of tuples it
// represents is considerably restricted
// @param L, the literal to be checked
// @param BV, the set of bound variables at the current position of
//        the considered rule
bool isPermissible(const LITERAL &L,const SetofBoundVars &BV)
    {
    assert( L.isRegularAtom() );
    /// a propositional literal is permissible
    if ( L.isPropositional() )
        return true;
    if( ( L.isNegative() || (L.isBuiltin() && isComparison(L)) )
        && fullyBound(L,BV) )
        return true;
    if( L.isBuiltin() && ( L.getBuiltinID()==BuiltinEquality
                           && hasBoundTerms(L,BV) ) )
        return true;
    // FIXME: to decide how many variables are requested to be bound
    // for math operators
    if ( L.isBuiltin() && ( L.getBuiltinID()==BuiltinAddition || 
                            L.getBuiltinID()==BuiltinMultiplication )
         && hasBoundTerms(L,BV) )
        return true;
    if ( !L.isNegative() && !L.isBuiltin() )
        return true;
    return false;  
    }


// return true if the first literal is preferable compared to the
// second literal w.r.t. the current set of bound variables
// the condition of preferability depends on the ability of a literal
// to propagate the binding
// @param L1, the literal to be compared
// @param L2, the literal to be compared
// @param BV, the set of bound variables at the current position of
//        the considered rule
bool isPreferable(const LITERAL &L1,
                  const LITERAL &L2,
                  const SetofBoundVars &BV)
    {
    // an EDB propositional literal is preferable
    if( L1.isEDB() && L1.isPropositional() )
        return true;
    if( L2.isEDB() && L2.isPropositional() )
        return false;
    // an IDB propositional literal is not preferable
    if( L2.isIDB() && L2.isPropositional() )
        return true;
    if( L1.isIDB() && L1.isPropositional() )
        return false;
    // if the first literal has bound terms and the second doesn't, it
    // is preferable; if the dual condition holds, it is dealt with by
    // the final return false
    if( hasBoundTerms(L1,BV) && !hasBoundTerms(L2,BV) )
	return true;
    // if both literals have bound terms...
    if( hasBoundTerms(L1,BV) && hasBoundTerms(L2,BV) )
        {
        // ... an EDB and a builtin are preferable
        if( L1.isEDB() || L1.isBuiltin() )
	    return true;
        else // L1 is IDB
            if( L2.isIDB() )
                // ... and if both literals are IDB, the one with the
                // greater number of bound terms is preferable
                return getNumberBoundTerms(L1,BV)
                    >= getNumberBoundTerms(L2,BV);
        }

    // if both literals are completely free, choose the one which is
    // EDB or a builtin; if even this does not decide, try to keep the
    // original ordering
    if( !hasBoundTerms(L1,BV) && !hasBoundTerms(L2,BV)
        && ( L1.isEDB() || L1.isBuiltin() || L2.isIDB() ) )
	return true;
    return false;
    }


// if a literal is IDB and not propositional adorn it and add it to
// the body of the rule to be adorned, otherwise simply add it
//
// @param L, the literal to be added
// @param body, the conjunction to which the literal is to be added
// @param AdornedPredicates, the set of predicate names of the already
//        adorned predicates
// @param StackOfAdornedPredicates, the stack of adorned predicate
//        names to be propagated
// @param BV, the set of bound variables at the current position of
//        the considered rule
//
void adornAndAddLiteralToBody(const LITERAL &L,
                              CONJUNCTION &body,
                              set<NAMES_ITEM> &AdornedPredicates,
                              stack<NAMES_ITEM>
                              &StackOfAdornedPredicates,
                              SetofBoundVars &BV)
    {
    if( L.isIDB() )
        {
        if( !L.isPropositional() )
            {
            LITERAL aL(adornLiteral(L,BV));
            if ( AdornedPredicates.find(aL.getPredItem())
                == AdornedPredicates.end() )
                {
                // push the adorned IDB atom a on the stack of adorned
                // atoms and insert it in the set of adorned atoms
                AdornedPredicates.insert(AdornedPredicates.end(),
                                         aL.getPredItem());
                StackOfAdornedPredicates.push(aL.getPredItem());
                }
            // add the chosen adorned literal to the body of the new
            // adorned rule
            // FIXME: The separation between positive and negative
            // literals is explicitly ignored in order to ensure the
            // intended ordering of literals.
            body.insert(body.neg_end(),aL);
            }
        else
            {
            if( AdornedPredicates.find(L.getPredItem())
                == AdornedPredicates.end() )
                {
                //push the propositional IDB atom a on the stack of
                //adorned atoms and insert it in the set of adorned
                //atoms
                AdornedPredicates.insert(AdornedPredicates.end(),
                                         L.getPredItem());
                StackOfAdornedPredicates.push(L.getPredItem());
                }
            //add the chosen literal to the body of the new adorned
            //rule
            // FIXME: The separation between positive and negative
            // literals is explicitly ignored in order to ensure the
            // intended ordering of literals.
            body.insert(body.neg_end(),L);
            }
        }
    else
        // the EDB literals must be simply added to the body of the
        // new adorned rule
        // FIXME: The separation between positive and negative
        // literals is explicitly ignored in order to ensure the
        // intended ordering of literals.
        body.insert(body.neg_end(),L);
    }


// receive an adorned predicate aa and a rule r with the unadorned
// version of aa in the head and return the rule r1, which is the
// adorned version of r w.r.t. the adornment of aa
// @param r, the rule to be adorned
// @param aa, the predicate name for which the binding is propagated
// @param AdornedPredicates, the set of predicate names of the already
//        adorned predicates
// @param StackOfAdornedPredicates, the stack of adorned predicate
//        names to be propagated
RULE adornRule(const RULE &r,
               const NAMES_ITEM &aa,
               stack<NAMES_ITEM> &StackOfAdornedPredicates,
               set<NAMES_ITEM> &AdornedPredicates)

    {
    assert( r.hasHead()  &&  ! r.isDisjunctive() );

    DISJUNCTION newRuleHead;
    const ATOM &head = *(r.getHead().begin());

    if( TraceLevel >= 1 )
        {
        // Print this at most once for each rule.
        cdebug << "Adorning   " << r << endl;
        }

    // the rules which are IDB facts, e.g. p(1,2)., matching with the
    // unadorned version of aa, must be adorned w.r.t. the adornment
    // of aa.
    // for instance if the adornment of aa is '#bb' the new
    // adorned IDB fact will be p#bb(1,2).
    if( !r.hasBody() )
        {
        if( !head.isPropositional() )
            {
#ifndef NDEBUG
            const char *name = getUnadornedName(aa);
            assert( strcmp( head.getName(), name ) == 0 );
            delete[] name;
#endif
            newRuleHead.add(buildAdornedAtom(head,aa));
            return RULE(&newRuleHead,0);
            }
        else
            return RULE(r);
        }
    else 
        // the rule to be adorned has a body
        {
        if( !head.isPropositional() )
            {
#ifndef NDEBUG
            const char *name = getUnadornedName(aa);
            assert( strcmp( head.getName(), name ) == 0 );
            delete[] name;
#endif
            // the new rule must have in the head a copy of the
            // literal aa with the same adornment
            newRuleHead.add(buildAdornedAtom(head,aa));
            }
        else 
            newRuleHead.add(head);  

        CONJUNCTION newRuleBody;

        //initialize BV with the set of bound variables of the adorned head
        SetofBoundVars BV;
        assert( newRuleHead.size() == 1 );
        getBoundVars(*(newRuleHead.begin()),BV);

        // store all the body literals of r in notYetConsideredLiterals
        vector<LITERAL> notYetConsideredLiterals;
        notYetConsideredLiterals.insert(notYetConsideredLiterals.begin(),
                                        r.getBody()->begin(),
                                        r.getBody()->end());

        // add the permissible literals of the original body to the
        // new body according to the order defined by the preferential
        // criterion.
        bool foundPermissible = true;
        while( !notYetConsideredLiterals.empty() && foundPermissible )
            {
            foundPermissible=false;
            vector<LITERAL>::iterator i =
                notYetConsideredLiterals.begin();

            // selection of the first permissible LITERAL to propagate
            // bindings
            while( i != notYetConsideredLiterals.end() 
                  && !foundPermissible )
                {
                if( isPermissible(*i,BV) )
                    // first permissible literal found
                    foundPermissible=true;
                else
                    // not yet found a permissible literal
                    i++;
                }

            if( i ==notYetConsideredLiterals.end() )
                // No permissible literal found.
                foundPermissible=false;
            else
                {
                // selection of the best permissible LITERAL to
                // propagate bindings, through the comparison of the
                // chosen permissible body literal with the remaining
                // permissible body literals
                if( i != notYetConsideredLiterals.end()-1 )
                    {
                    vector<LITERAL>::iterator j = i+1;
                    while( i != notYetConsideredLiterals.end()-1
                           && j != notYetConsideredLiterals.end() )
                        {
                        while( j!=notYetConsideredLiterals.end()
                               && !isPermissible(*j,BV) )
                            j++;
                        if( j!= notYetConsideredLiterals.end() )
                            {
                            // another permissible literal found;
                            // choose the preferable one
                            i = (isPreferable(*i,*j,BV))?i:j;
                            j++;
                            }
                        }
                    }
	        // If a literal is IDB and not propositional, adorn it
	        // and add it to the body of the rule to be adorned;
	        // otherwise simply add it to the body of the rule.
                adornAndAddLiteralToBody(*i,
                                         newRuleBody,
                                         AdornedPredicates,
                                         StackOfAdornedPredicates,
                                         BV);
                // update the set of bound variables of the current
                // rule adding all of the variables of the just chosen
                // literal
                if( !(*i).isPropositional() && hasBoundTerms(*i,BV) )
                    updateBoundVars(BV,*i);

                // we have considered the literal *i
                notYetConsideredLiterals.erase(i);
                }
            }
	
	// if there are remaining literals it means that they are not
	// permissible and we do not have to propagate the adornment
	// through them, but they must be added to the body of the new
	// rule in any case
        for( vector<LITERAL>::iterator k=
                 notYetConsideredLiterals.begin();
             k!= notYetConsideredLiterals.end();
             k++ )
            adornAndAddLiteralToBody(*k,
                                     newRuleBody,
                                     AdornedPredicates,
                                     StackOfAdornedPredicates,
                                     BV);

        // finally create the adorned rule        
        RULE r1(&newRuleHead,&newRuleBody);

        if( TraceLevel >= 1 )
            {
            // Print this at most once for each rule.
            cdebug << "Adorned rule   " << r1 << endl;
            }

        return r1;
        }

    // This point should not be reached in any execution path, but some
    // compilers do not recognise this.
    assert(0);
    return RULE(0,0);
    }


// adorn the rules of the program (stored in the global variable IDB)
// w.r.t. the adorned predicate aa
//
// @param aa, the predicate name for which the binding is propagated
// @param StackOfAdornedPredicates, the stack of adorned predicate
//        names to be propagated
// @param AdornedRules, the adorned rules 
// @param AdornedPredicates, the set of predicate names of the already
//        adorned predicates
void adornSubprogram(const NAMES_ITEM &aa,
                     stack<NAMES_ITEM> &StackOfAdornedPredicates,
                     RULES &AdornedRules,
                     set<NAMES_ITEM> &AdornedPredicates)

    {
    for( RULES::const_iterator i=IDB.begin();
         i != IDB.end();
         i++ )
        {
        // Check if the head predicate matches with the not adorned
        // version of aa.
        assert( i->hasHead()  &&  ! i->isDisjunctive() );

        const ATOM &aa1 = *( i->getHead().begin() );
        if ( isAdorned(aa) )
            {
            // check if the head predicate matches with the unadorned
            // version of aa
            const char *name = getUnadornedName(aa);
            if( strcmp( aa1.getName(), name ) == 0 )
                AdornedRules.push_back(adornRule( *i,
                                                  aa,
                                                  StackOfAdornedPredicates,
                                                  AdornedPredicates));
            delete[] name;
            }
        else
            {
            if( strcmp( aa1.getName(), aa.getName()) == 0 ) 
                AdornedRules.push_back(adornRule( *i,
                                                  aa,
                                                  StackOfAdornedPredicates,
                                                  AdornedPredicates));
            }
        }
    }


// adorn the original query (given by the global variable Query) and
// return the adorned query
// @param AdornedPredicates, the set of predicate names of the already
//        adorned predicates
// @param StackOfAdornedPredicates, the stack of adorned predicate
//        names to be propagated
CONJUNCTION adornQuery(set<NAMES_ITEM> &AdornedPredicates,
                        stack<NAMES_ITEM> &StackOfAdornedPredicates)
    {
    CONJUNCTION aq;
    for( CONJUNCTION::const_iterator i = (*Query).begin();
         i != (*Query).end();
         i++ )
        if ( (*i).isIDB() && !(*i).isPropositional() )
            {
            const ATOM &aa=adornQueryAtom(*i);
            aq.add(aa);
            if( AdornedPredicates.find(aa.getName())
                == AdornedPredicates.end() )
                {        
                AdornedPredicates.insert(AdornedPredicates.end(),
                                         aa.getName());
                StackOfAdornedPredicates.push(aa.getName());
                }
            }
        else 
            if( (*i).isIDB() )
                aq.add(*i);

    return aq;
    }


// adorn the query (given by the global variable Query) and propagate
// the binding to the rules whose heads match with one query
// predicate, and from the heads of these rules to their body
// literals, creating new adorned predicates; for each new adorned
// predicate repeat the adornment and propagate the binding to the
// rules.
// At the end return the adorned rules.
// @param adornedQuery, a query structure which should hold the
//        adorned query
RULES adorn(CONJUNCTION &adornedQuery)
    {
    RULES AdornedRules;
    set<NAMES_ITEM> AdornedPredicates;
    stack<NAMES_ITEM> StackOfAdornedPredicates;
    // adorn the query
    adornedQuery = adornQuery(AdornedPredicates,
                              StackOfAdornedPredicates);
    // adorn the rules
    while( !StackOfAdornedPredicates.empty() )
        {
        NAMES_ITEM a(StackOfAdornedPredicates.top());
        StackOfAdornedPredicates.pop();
        adornSubprogram(a,
                        StackOfAdornedPredicates,
                        AdornedRules,
                        AdornedPredicates);
        }
    return AdornedRules;
    }


////////////////////////////////GENERATE///////////////////////


// receive an adorned atom aa and return a magic atom with the same
// predicate name beginning with underscore and with only the bound
// variables of aa as parameters
// @param aa, the atom to transform
ATOM magic(const ATOM &aa)
    {
    assert( aa.isIDB() && !aa.isPropositional() );
    TERMS newParams;
    const TERMS *params = aa.getParams();
    assert( params != 0 );
    const char *adornment = getAdornment(aa.getPredItem());
    const char *adornmentpos = adornment;
    for( TERMS ::const_iterator i=params->begin();
         i != params->end();
         i++,adornmentpos++ )
        if( *adornmentpos == 'b' )
            newParams.push_back(*i);
    delete[] adornment;
    char *magicName = new char[strlen(aa.getName())+2];
    strcpy(magicName,"_");
    strcat(magicName,aa.getName());
    ATOM m(magicName,&newParams,aa.getType());
    delete[] magicName;
    return m;
    }


// Receive the adorned rules and return the generated rules.
// For each adorned rule, build a new magic rule for each adorned IDB
// literal L in the body which has at least one bound term.
// The head of the magic rule consists of the magic version of L while
// the body consists of the magic version of the original head and all
// the literals preceding L in the adorned rule.
//
// @param AdornedRules, the adorned rules to be considered
RULES generate(RULES &AdornedRules)
    {
    RULES MagicRules;

    for( RULES::const_iterator k=AdornedRules.begin();
         k != AdornedRules.end();
         k++ )
        {
        assert( k->hasHead()  &&  ! k->isDisjunctive() );

	const ATOM &head = *(k->getHead().begin());

        // create magic rules only for predicates having at least one
        // bound argument
        if( hasBoundTermsIDB(head) )
            {
            SetofBoundVars BV;
            getBoundVars(head,BV);
            if( (*k).hasBody() )
                {
                // for each adorned predicate of the body with bound
                // arguments build a magic rule
                for( CONJUNCTION::const_iterator i=
                         (*k).getBody()->begin();
                     i != (*k).getBody()->end();
                     i++ )
                    {
                    if( isAdorned((*i).getName())
                        && hasBoundTermsIDB((*i).getAtom())
                        // to avoid generating a useless "reflexive"
                        // magic rule of the form
                        // "_p#bf(V):- a(Y,V),_p#bf(V)", it is
                        // necessary to check if the literal of the
                        // body to be "switched" with the head of the
                        // adorned rule is not equal.
                        && head != *i )
                        {
                        DISJUNCTION newMagicRuleHead;
                        CONJUNCTION newMagicRuleBody;
                        // the new rule must have the magic version of
                        // the considered body literal in the head...
                        newMagicRuleHead.add(magic(*i));
                        // ...and the magic version of the original
                        // head in the body
                        newMagicRuleBody.add(LITERAL(magic(head)));
                        // add to the new magic rule all the literals
                        // preceding the considered body literal
                        for( CONJUNCTION::const_iterator j=
                                 (*k).getBody()->begin();
                             j != (*k).getBody()->end() 
                                 && LITERAL(*i) != (*j);
                             j++ )
                            newMagicRuleBody.add(*j);

                        // build the new magic rule
                        RULE mRule(&newMagicRuleHead,
                                   &newMagicRuleBody);
                        MagicRules.push_back(mRule);
                        if( TraceLevel >= 1 )
                            {
                            // Print this at most once for each rule.
                            cdebug << "Generated magic rule   "
                                   << mRule << endl;
                            }                        
                        }
                    }
                }
            }
        }
    return MagicRules;
    }

//////////////////////////MODIFY////////////////////////////////////

// receive the adorned rules and return the modified rules, obtained
// by adding to the body of each adorned rule the magic version of the
// head predicate, if it has at least one bound term
// @param AdornedRules, the rules to modify
RULES modify(RULES &AdornedRules)
    {
    RULES ModifiedRules;
    for( RULES::iterator i=AdornedRules.begin();
         i!=AdornedRules.end();
         i++ )
        {  
        if( TraceLevel >= 1 )
            cdebug << "Modifying rule   " << *i << endl;

        assert( i->hasHead()  &&  ! i->isDisjunctive() );

        const ATOM &head = *(i->getHead().begin());

        // copy the current rule
        RULE r(*i);

        if( hasBoundTermsIDB(head) )
            {
            LITERAL mL(magic(head));
            if( r.hasBody() )
                (*(r.getBodyForModification())).
                    insert((*(r.getBodyForModification()))
                           .pos_begin(),mL);
            else // it is a fact
                {
                assert( ! r.getBody() );
                CONJUNCTION body;
                body.insert(body.pos_begin(),mL);
                r.setBody(&body);
                }
            }

        ModifiedRules.push_back(r);

        if( TraceLevel >= 1 )
            {
            cdebug << "Modified rule   " << r << endl;
            }
        }
    return ModifiedRules;
    }


//////////////////////////////QUERY REWRITING/////////////////////

// for each adorned query atom build new rules with the original (not
// adorned) atom in the head and the adorned atom in the body.
// For instance, if we have the adorned query atom 'q#bb(2,5)',
// the new  rule will be: q(2,5):- q#bb(2,5). 
// This is because it is necessary to derive the original query atoms
// from the rewritten rules where the IDB atoms are magic and adorned.
// @param adornedQuery, the adorned query to be processed
RULES buildQueryRules(const CONJUNCTION &adornedQuery)
    {
    RULES queryRules;
    for( CONJUNCTION::const_iterator j=adornedQuery.begin();
         j != adornedQuery.end();
         j++ )
        {
        if( isAdorned((*j).getName()))
            {
            DISJUNCTION headQueryRule;
            const char *name = getUnadornedName((*j).getName());
            headQueryRule.add(ATOM(name,(*j).getParams(),(*j).getType()));
            delete[] name;
            CONJUNCTION bodyQueryRule;
            bodyQueryRule.add(*j);
            const RULE &qr = RULE(&headQueryRule,&bodyQueryRule);
            queryRules.push_back(qr);
            if( TraceLevel >= 1 )
                cdebug << "query rule   " << qr << endl;
            }
        }

    return queryRules;
    }


// for each adorned atom of the adorned query build a magic fact for
// instance, if we have the adorned query atom 'q#bf(2,X)', the new
// magic fact will be: _q#bf(2).
// This is because it is a relevant fact for answering the original
// query.
// @param adornedQuery, the adorned query to be processed
RULES buildMagicQueryFacts(const CONJUNCTION &AdornedQuery)
    {
    RULES magicQueryFacts;
    for( CONJUNCTION::const_iterator i=AdornedQuery.begin();
         i!= AdornedQuery.end();
         i++ )
        if( !(*i).isPropositional() && (*i).isIDB() )
            {
            DISJUNCTION headMagicQueryFact;
            headMagicQueryFact.add(magic(*i));
            const RULE &qf = RULE(&headMagicQueryFact,0);
            magicQueryFacts.push_back(qf);
            if( TraceLevel >= 1 )
                cdebug << "magic query fact   " << qf << endl;
            }
    return magicQueryFacts;
    }


// add to the global IDB the modified rules and at the same time
// implicitly [via the functionality of add()] reorder the literals of
// the body of each rule w.r.t. positive and negative literals, as
// required by the grounding step, since in the adornRule procedure
// they were created without respecting the separation between
// positive and negative literals
void addReorderedRules(const RULES &rules)
    {
    for( RULES::const_iterator i=rules.begin(); 
         i!=rules.end();
         i++)
        {
        RULE r(*i);
        if( (*i).hasBody() )
            {
            assert( r.getBody() );
            (*(r.getBodyForModification())).clear();
            for( CONJUNCTION::const_iterator j =
                     (*i).getBody()->begin();
                 j != (*i).getBody()->end();
                 j++ )
                (*(r.getBodyForModification())).add(*j);
            }
        IDB.push_back(r);
        }
    }


// at the moment the Magic Sets technique cannot be applied to
// disjunctive programs, check this condition in the global IDB
// FIXME: This should be a temporary measure.
// FIXME: A method like this should perhaps be provided inside(?) RULES.
bool isDisjunctive()
    {
    for( RULES::const_iterator i=IDB.begin();
         i != IDB.end();
         i++ )
        if( (*i).isDisjunctive() )
            return true;
    return false;
    }


// Check if the query contains at least a literal which is IDB and not
// propositional.
// FIXME: once we support unbound queries we will need to check if at least
// one query literal contains at least one bound term.
bool queryIsIDB()
    {
    for( CONJUNCTION::const_iterator i=Query->begin();
         i != Query->end();
         i++ )
        if((*i).isIDB() && !(*i).isPropositional())
            return true;
    return false;
    }

///////////////////////////////////////////////////////////
//MAGIC REWRITING
///////////////////////////////////////////////////////////

// perform the Magic Sets Optimization Technique
void MagicRewriting()
    {
    bool ok = true;
    if( !queryIsIDB() )
        {
        cerr << "The query must have at least one not "
                "propositional IDB atom."<< endl;
        ok = false;
        }
    if( Constraints.begin() !=  Constraints.end() )
        {
        cerr << "Magic Sets for constraints are not yet implemented."
             << endl;
        ok = false;
        }
    if( isDisjunctive() )
        {
        cerr << "Magic Sets for disjunctive programs are not yet implemented."
             << endl;
        ok = false;
        }

    if( !ok )
        cerr << "It is not possible to apply the Magic Sets optimization."
             << endl;
    else
        {
        RULES AdornedRules,MagicRules,ModifiedRules;

        // ADORN STEP
        CONJUNCTION adornedQuery;
        AdornedRules=adorn(adornedQuery);

        // GENERATE STEP
        MagicRules=generate(AdornedRules);
        RULES magicQueryFacts(buildMagicQueryFacts(adornedQuery));
        MagicRules.insert(MagicRules.end(),magicQueryFacts.begin(),
                          magicQueryFacts.end());

        //MODIFY STEP
        ModifiedRules=modify(AdornedRules);

        // substitute the rewritten program for the original one
        IDB.clear();
        addReorderedRules(ModifiedRules);
        IDB.insert(IDB.begin(),MagicRules.begin(),MagicRules.end());
        RULES queryRules(buildQueryRules(adornedQuery));
        IDB.insert(IDB.begin(),queryRules.begin(),queryRules.end());

        }
    }
