//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: Union.hpp,v 1.3 2004/02/08 18:44:19 delpinux Exp $


// This class provides a toolkit to manipulate Unions. It inherits
// from the Shape class.

#ifndef UNION_HPP
#define UNION_HPP

#include <Object.hpp>
#include <list>

/*!
  \class Union

  This is the class which defines a Virtual Reality set operation: the \a union.

  \author Stphane Del Pino.
*/
class Union
  : public Shape
{
private:
  typedef std::list<ReferenceCounting<Object> > ObjectList;

public:
  typedef std::list<ReferenceCounting<Object> >::iterator iterator;
  typedef std::list<ReferenceCounting<Object> >::const_iterator const_iterator;

private:
  //! The list of Shapes contained in the union.
  ObjectList __objects;

  //! Returns \p true if the point \a p is inside the Union.
  inline bool inShape (const TinyVector<3, real_t>& p) const;

  //! Prints the Union informations using the stream \a s.
  std::ostream& put(std::ostream& s) const;

public:

  const const_iterator begin() const
  {
    return __objects.begin();
  }

  iterator begin()
  {
    return __objects.begin();
  }


  const const_iterator end() const
  {
    return __objects.end();
  }

  iterator end()
  {
    return __objects.end();
  }

  //! Adds the Object \a O inside the Union.
  inline void push_back(ReferenceCounting<Object> O)
  {
    __objects.push_back(O);
  }

  //! Default constructor.
  Union();

  //! Copy constructor.
  Union(const Union& U);

  //! Destructor.
  ~Union()
  {
    ;
  }

};

//! Returns \p true if the point \a p is inside the Union.
inline bool Union::inShape (const TinyVector<3, real_t>& p) const
{
  for (Union::const_iterator i=__objects.begin();
       i != __objects.end(); ++i) {
    if((*(*i)).inside(p)) {
      return true;
    }
  }

  return false;
}

#endif // UNION_HPP

