//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: BoundaryExpression.cpp,v 1.8 2006/04/14 09:53:02 delpinux Exp $

#include <MeshExpression.hpp>

#include <Mesh.hpp>

#include <MeshOfHexahedra.hpp>
#include <MeshOfTetrahedra.hpp>
#include <Structured3DMesh.hpp>

#include <BoundaryExpression.hpp>

#include <Boundary.hpp>

#include <BoundaryPOVRay.hpp>
#include <BoundarySurfaceMesh.hpp>
#include <BoundaryReferences.hpp>
#include <BoundaryList.hpp>

ReferenceCounting<Boundary>
BoundaryExpression::boundary()
{
  return __boundary;
}

BoundaryExpression::BoundaryExpression(const BoundaryExpression& e)
  : Expression(Expression::boundary),
    __boundaryType(e.__boundaryType)
{
  ;
}

BoundaryExpression::BoundaryExpression(const BoundaryExpression::BoundaryType& t)
  : Expression(Expression::boundary),
    __boundaryType(t)
{
  ;
}

BoundaryExpression::~BoundaryExpression()
{
  ;
}

std::ostream& BoundaryExpressionSurfaceMesh::put(std::ostream& os) const
{
  os << "Surface Mesh";
  return os;
}

void BoundaryExpressionSurfaceMesh::execute()
{
  (*__surfaceMeshExpression).execute();

  Mesh* m = (*__surfaceMeshExpression).mesh();

  switch (m->family()) {
  case Mesh::volume: {
    switch (m->type()) {
    case Mesh::cartesianHexahedraMesh: {
      const Structured3DMesh& mesh = dynamic_cast<const Structured3DMesh&>(*m);
      __boundary = new BoundarySurfaceMesh(mesh.surfaceMesh());
      break;
    }
    case Mesh::tetrahedraMesh: {
      const MeshOfTetrahedra& mesh = dynamic_cast<const MeshOfTetrahedra&>(*m);
      __boundary = new BoundarySurfaceMesh(mesh.surfaceMesh());
      break;
    }
    case Mesh::hexahedraMesh: {
      const MeshOfHexahedra& mesh = dynamic_cast<const MeshOfHexahedra&>(*m);
      __boundary = new BoundarySurfaceMesh(mesh.surfaceMesh());
      break;
    }
    default: {
      throw ErrorHandler(__BASE_FILE__,__LINE__,
			 "Unexpected volume mesh type",
			 ErrorHandler::unexpected);
    }
    }
    break;
  }
  case Mesh::surface: {
    assert((*m).type() == Mesh::surfaceMeshTriangles);
    __boundary
      = new BoundarySurfaceMesh(dynamic_cast<const SurfaceMeshOfTriangles&>(*m));
    break;
  }
  default: {
    throw ErrorHandler(__BASE_FILE__,__LINE__,
		       "Unexpected mesh family",
		       ErrorHandler::unexpected);
  }
  }
}

BoundaryExpressionSurfaceMesh::
BoundaryExpressionSurfaceMesh(ReferenceCounting<MeshExpression> m)
  : BoundaryExpression(BoundaryExpression::surfaceMesh),
    __surfaceMeshExpression(m)
{
  ;
}

BoundaryExpressionSurfaceMesh::
BoundaryExpressionSurfaceMesh(const BoundaryExpressionSurfaceMesh& m)
  : BoundaryExpression(m),
    __surfaceMeshExpression(m.__surfaceMeshExpression)
{
  ;
}

BoundaryExpressionSurfaceMesh::
~BoundaryExpressionSurfaceMesh()
{
  ;
}

std::ostream& BoundaryExpressionPOVRay::put(std::ostream& os) const
{
  os << (*__povrayReference);
  return os;
}

void BoundaryExpressionPOVRay::execute()
{
  (*__povrayReference).execute();

  BoundaryPOVRay* b = new BoundaryPOVRay();

  TinyVector<3> ref;
  for (size_t i=0; i<3; ++i)
    ref[i] = (*__povrayReference).value(i);

  (*b).addPOVRef(ref);
  __boundary = b;
}


void BoundaryExpressionReferences::execute()
{
  for (BoundaryExpressionReferences::ReferencesSet::iterator i
	 = __references.begin();
       i != __references.end(); ++i)
    {
      ReferenceCounting<RealExpression> ref = *i;
      (*ref).execute();
    }

  BoundaryReferences* b= new BoundaryReferences();

  for (BoundaryExpressionReferences::ReferencesSet::iterator i
	 = __references.begin();
       i != __references.end(); ++i) {
    ReferenceCounting<RealExpression> ref = *i;
    const double value = (*ref).realValue();
    if ((value<0) or (value != int(value))) {
      throw ErrorHandler(__FILE__,__LINE__,
			 stringify(value)
			 +" is not a correct boundary reference\n"
			 +"Boundary references must be positive integers",
			 ErrorHandler::normal);
    }
    b->add(static_cast<size_t>(value));
  }
  __boundary = b;
}

std::ostream& BoundaryExpressionReferences::put(std::ostream& os) const
{
  for (BoundaryExpressionReferences::ReferencesSet::const_iterator i
	 = __references.begin();
       i != __references.end(); ++i) {
    if (i != __references.begin()) {
      os  << ',';
    }
    os << (**i);
  }
  return os;
}

void
BoundaryExpressionList::execute()
{
  BoundaryList* b = new BoundaryList();

  for(List::iterator i = __boundaries.begin();
      i != __boundaries.end(); ++i) {
    (**i).execute();
    b->add((**i).boundary());
  }
  __boundary = b;
}

std::ostream&
BoundaryExpressionList::put(std::ostream& os) const
{
  for (List::const_iterator i = __boundaries.begin();
       i != __boundaries.end(); ++i) {
    if (i != __boundaries.begin()) {
      os << ',';
    }
    os << (**i);
  }
  return os;
}
