//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: Instruction.cpp,v 1.17 2005/11/03 21:12:50 delpinux Exp $

#include <config.h>

#include <cstdlib>
#include <fstream>

#include <Mesh.hpp>
#include <MeshWriter.hpp>

#include <Structured3DMesh.hpp>

#include <Instruction.hpp>
#include <FunctionExpression.hpp>
#include <StringExpression.hpp>

#include <Scene.hpp>

#include <MeshExpression.hpp>
#include <Variable.hpp>

#include <Information.hpp>

#include <VTKDriver.hpp>

#include <FEMFunction.hpp>
#include <ErrorHandler.hpp>

void InstructionExec::execute()
{
  (*__command).execute();
  std::string command = (*__command).value();
  ffout(2) << "executing: " << command << '\n';
#ifdef ALLOW_EXEC
  int result = system(command.c_str());
  if (result == -1) {
    throw ErrorHandler(__FILE__,__LINE__,
		       "could not execute: "+command,
		       ErrorHandler::normal);
  }
#else // ALLOW_EXEC
  fferr(2) << "warning: this version was not compiled with 'exec' support.\n";
#endif // ALLOW_EXEC
}

InstructionExec::InstructionExec(ReferenceCounting<StringExpression> command)
  : Instruction(Instruction::exec),
    __command(command)
{
  ;
}

InstructionExec::InstructionExec(const InstructionExec& I)
  : Instruction(I),
    __command(I.__command)
{
  ;
}

InstructionExec:: ~InstructionExec()
{
  ;
}


InstructionSave::
InstructionSave(ReferenceCounting<FileDescriptor> descriptor,
		ReferenceCounting<StringExpression> fileName,
		ReferenceCounting<MeshExpression> mesh)
  : Instruction(Instruction::save),
    __fileDescriptor(descriptor),
    __fileName(fileName),
    __mesh(mesh)
{
  ;
}

InstructionSave::InstructionSave(const InstructionSave& I)
  : Instruction(I),
    __fileDescriptor(I.__fileDescriptor),
    __fileName(I.__fileName),
    __mesh(I.__mesh)
{
  ;
}

InstructionSave::~InstructionSave()
{
  ;
}

InstructionSaveMesh::
InstructionSaveMesh(ReferenceCounting<FileDescriptor> descriptor,
		    ReferenceCounting<StringExpression> fileName,
		    ReferenceCounting<MeshExpression> mesh)
  : InstructionSave(descriptor, fileName, mesh)
{
  ;
}

InstructionSaveMesh::
InstructionSaveMesh(const InstructionSaveMesh& I)
  : InstructionSave(I)
{
  ;
}

InstructionSaveMesh::
~InstructionSaveMesh()
{
  ;
}

void InstructionSaveMesh::execute()
{
  (*__mesh).execute();
  (*__fileName).execute();
  Information::instance().setMesh(__mesh);

  const std::string CR = (*__fileDescriptor).cr();

  switch((*__fileDescriptor).format()) {
  case FileDescriptor::raw: {
    std::string msh = (*__fileName).value();
    throw ErrorHandler(__FILE__,__LINE__,
		       "cannot use raw format to save a mesh!",
		       ErrorHandler::normal);
    break;
  }
  case FileDescriptor::medit: {
    //! plots the Mesh
    std::string msh = (*__fileName).value();

    msh += ".mesh";
    
    std::ofstream fmsh(msh.c_str());
    if (not(fmsh)) {
      throw ErrorHandler(__FILE__,__LINE__,
			 "cannot open file '"+msh+"'",
			 ErrorHandler::normal);
    }

    MeshWriter w(fmsh,(*__mesh).mesh(),CR);

    break;
  }
  default: {
    throw ErrorHandler(__FILE__,__LINE__,
		       "unexpected file type",
		       ErrorHandler::normal);
  }
  }

  //! The mesh is nomore used in this region.
  Information::instance().unsetMesh();
}

InstructionSaveFunction::
InstructionSaveFunction(ReferenceCounting<FileDescriptor> descriptor,
			ReferenceCounting<StringExpression> fileName,
			ReferenceCounting<FunctionExpression> function,
			ReferenceCounting<MeshExpression> mesh)
  : InstructionSave(descriptor, fileName, mesh),
    __function(function)
{
  ;
}

InstructionSaveFunction::
InstructionSaveFunction(const InstructionSaveFunction& I)
  : InstructionSave(I),
    __function(I.__function)
{
  ;
}

InstructionSaveFunction::
~InstructionSaveFunction()
{
  ;
}

void InstructionSaveFunction::execute()
{
  (*__mesh).execute();
  (*__function).execute();
  (*__fileName).execute();
  Information::instance().setMesh(__mesh);

  const std::string CR = (*__fileDescriptor).cr();

  const Mesh& mesh = static_cast<const Mesh&>(*(*__mesh).mesh());

  switch((*__fileDescriptor).format()) {
  case FileDescriptor::raw: {
    std::ofstream fsol((*__fileName).value());
    if (fsol.bad()) {
      throw ErrorHandler(__FILE__,__LINE__,
			 "cannot open file '"
			 +stringify((*__fileName).value())+"'",
			 ErrorHandler::normal);
    }

    switch ((*(*__function).value()).type()) {
    case FunctionExpression::fem: {
      FunctionExpressionFEM& f
 	= static_cast<FunctionExpressionFEM&>(*(*__function).value());
      if ((*f.mesh()).mesh() == ((*__mesh).mesh())) {
	(f.femFunction()).dump(fsol, CR);
      } // if not continues the standard method
      break;
    }
    default: {
      FunctionExpression& f = (*__function);
      for (size_t i=0; i<mesh.numberOfVertices(); ++i) {
	const TinyVector<3>& X = mesh.vertex(i);
	fsol << f.value(X[0], X[1], X[2]) << CR;
      }
    }
    }
    break;
  }
  case FileDescriptor::medit: {
    //! plots the solution
    std::string sol = (*__fileName).value();
    sol += ".bb";
    std::ofstream fsol(sol.c_str());
    if (not(fsol)) {
      throw ErrorHandler(__FILE__,__LINE__,
			 "cannot open file '"+stringify(sol)+"'",
			 ErrorHandler::normal);
    }

    fsol << "3 1 " << mesh.numberOfVertices() << " 2" << CR;

    switch ((*(*__function).value()).type()) {
    case FunctionExpression::fem: {
      FunctionExpressionFEM& f
 	= static_cast<FunctionExpressionFEM&>(*(*__function).value());
      if ((*f.mesh()).mesh() == ((*__mesh).mesh())) {
	(f.femFunction()).dump(fsol, CR);
	break;
      } // if not continues the standard method
    }
    default: {
      FunctionExpression& f = (*__function);
      for (size_t i=0; i<mesh.numberOfVertices(); ++i) {
	const TinyVector<3>& X = mesh.vertex(i);
	fsol << f.value(X[0], X[1], X[2]) << CR;
      }
    }
    }
    break;
  }
  default: {
    throw ErrorHandler(__FILE__,__LINE__,
		       "unexpected file type",
		       ErrorHandler::unexpected);
  }
  }

  //! The mesh is nomore used in this region.
  Information::instance().unsetMesh();
}

void
InstructionAffectation<FunctionExpressionFEM, FunctionVariable>::
execute()
{
  (*__expression).execute();

  ReferenceCounting<FunctionExpression> f = (*__variable).expression();

  FunctionExpressionFEM& femFunction = static_cast<FunctionExpressionFEM&>(*f);
  const UserFunctionLanguage u(__expression);
  femFunction.femFunction() = u;
}


InstructionSaveField::
InstructionSaveField(ReferenceCounting<FileDescriptor> descriptor,
		     ReferenceCounting<StringExpression> fileName,
		     ReferenceCounting<FieldExpression> field,
		     ReferenceCounting<MeshExpression> mesh)
  : InstructionSave(descriptor,fileName,mesh),
    __field(field)
{
  ;
}

InstructionSaveField::
InstructionSaveField(const InstructionSaveField& I)
  : InstructionSave(I),
    __field(I.__field)
{
  ;
}

InstructionSaveField::
~InstructionSaveField()
{
  ;
}

void InstructionSaveField::execute()
{
  (*__mesh).execute();
  (*__field).execute();
  (*__fileName).execute();
  Information::instance().setMesh(__mesh);

  const std::string CR = (*__fileDescriptor).cr();

  const Mesh& mesh = static_cast<const Mesh&>(*(*__mesh).mesh());

  switch((*__fileDescriptor).format()) {
  case FileDescriptor::raw: {
    throw ErrorHandler(__FILE__,__LINE__,
		       "not implemented",
		       ErrorHandler::normal);
  }
  case FileDescriptor::medit: {
    //! plots the solution
    std::string sol = (*__fileName).value();
    sol += ".bb";
    std::ofstream fsol(sol.c_str());
    if (not(fsol)) {
      throw ErrorHandler(__FILE__,__LINE__,
			 "error: cannot open file '"+sol+"'",
			 ErrorHandler::normal);
    }

    fsol << "3 3 " << mesh.numberOfVertices() << " 2" << CR;

    bool f1IsFEMOnThatMesh = false;
    bool f2IsFEMOnThatMesh = false;
    bool f3IsFEMOnThatMesh = false;

    const Vector<real_t>* v1 = 0;
    const Vector<real_t>* v2 = 0;
    const Vector<real_t>* v3 = 0;

    if ((*((*__field).f1()).value()).type() == FunctionExpression::fem) {
      FunctionExpressionFEM& f
	= static_cast<FunctionExpressionFEM&>(*((*__field).f1()).value());
      if ((*f.mesh()).mesh() == ((*__mesh).mesh())) {
	f1IsFEMOnThatMesh = true;
	v1 = &(f.femFunction().values());
      }
    }

    if ((*((*__field).f2()).value()).type() == FunctionExpression::fem) {
      FunctionExpressionFEM& f
	= static_cast<FunctionExpressionFEM&>(*((*__field).f2()).value());
      if ((*f.mesh()).mesh() == ((*__mesh).mesh())) {
	f2IsFEMOnThatMesh = true;
	v2 = &(f.femFunction().values());
      }
    }

    if ((*((*__field).f3()).value()).type() == FunctionExpression::fem) {
      FunctionExpressionFEM& f
	= static_cast<FunctionExpressionFEM&>(*((*__field).f3()).value());
      if ((*f.mesh()).mesh() == ((*__mesh).mesh())) {
	f3IsFEMOnThatMesh = true;
	v3 = &(f.femFunction().values());
      }
    }

    for (size_t i=0; i<mesh.numberOfVertices(); ++i) {
      const TinyVector<3>& X = mesh.vertex(i);
      if (f1IsFEMOnThatMesh) {
	fsol << (*v1)[i] << ' ';
      } else {
	fsol << (*__field).f1().value(X[0], X[1], X[2]) << ' ';
      }
      if (f2IsFEMOnThatMesh) {
	fsol << (*v2)[i] << ' ';
      } else {
	fsol << (*__field).f2().value(X[0], X[1], X[2]) << ' ';
      }
      if (f3IsFEMOnThatMesh) {
	fsol << (*v3)[i] << CR;
      } else {
	fsol << (*__field).f3().value(X[0], X[1], X[2]) << CR;
      }
    }
    break;
  }
  default: {
    throw ErrorHandler(__FILE__,__LINE__,
		       "unknown file type",
		       ErrorHandler::unexpected);
  }
  }

  //! The mesh is nomore used in this region.
  Information::instance().unsetMesh();
}

void InstructionPlot::execute()
{
  (*__mesh).execute();

  Mesh& M = (*(*__mesh).mesh());
  VTKDriver d;

  if (__f == 0) {
    d.plot(M);
  } else {
    (*__f).execute();
    ReferenceCounting<UserFunction> u
      = new UserFunctionLanguage((*__f).value());
    d.plot(M, u);
  }
}

InstructionPlot::InstructionPlot(ReferenceCounting<MeshExpression> m,
				 ReferenceCounting<FunctionExpression> f)
  : Instruction(Instruction::plot),
    __mesh(m), 
    __f(f)
{
  ;
}

InstructionPlot::InstructionPlot(const InstructionPlot& I)
  : Instruction(I),
    __mesh(I.__mesh)
{
  ;
}

InstructionPlot::~InstructionPlot()
{
  ;
}

void InstructionUsingScene::execute()
{
  (*__sceneExpression).execute();
  Information::instance().setScene((*__sceneExpression).scene());
}

InstructionUsingScene::InstructionUsingScene(ReferenceCounting<SceneExpression> e)
  : Instruction(Instruction::Using),
    __sceneExpression(e)
{
  ;
}

InstructionUsingScene::InstructionUsingScene(const InstructionUsingScene& I)
  : Instruction(I),
    __sceneExpression(I.__sceneExpression)
{
  ;
}

InstructionUsingScene::~InstructionUsingScene()
{
  ;
}

void InstructionCoarseMesh::execute()
{
  Information::instance().setCoarseMesh(__coarseMesh);
}

