//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: BoundaryConditionExpression.hpp,v 1.6 2006/03/12 20:17:23 delpinux Exp $

#ifndef BOUNDARYCONDITION_EXPRESSION_HPP
#define BOUNDARYCONDITION_EXPRESSION_HPP

#include <Expression.hpp>
#include <Variable.hpp>

#include <BoundaryExpression.hpp>

/**
 * @file   BoundaryConditionExpression.hpp
 * @author Stephane Del Pino
 * @date   Wed Jan  5 20:09:56 2005
 * 
 * @brief  This class defines the base class of BoundaryCondition expressions
 * 
 */
class BoundaryCondition;
class BoundaryConditionExpression
  : public Expression
{
protected:
  ReferenceCounting<BoundaryCondition> __boundaryCondition;

  ReferenceCounting<BoundaryExpression> __boundary;

  ReferenceCounting<FunctionVariable> __unknown;

public:
  enum BoundaryConditionType {
    dirichlet,
    neumann,
    fourrier // also known as Robin
  };

  ReferenceCounting<FunctionVariable> unknown() const
  {
    return __unknown;
  }

private:
  BoundaryConditionExpression::BoundaryConditionType __boundaryConditionType;

public:
  bool hasPOVBoundary() const
  {
    return (*__boundary).hasPOVBoundary();
  }

  ReferenceCounting<BoundaryCondition> boundaryCondition();

  const BoundaryConditionExpression::BoundaryConditionType& boundaryConditionType() const
  {
    return __boundaryConditionType;
  }

  BoundaryConditionExpression(const BoundaryConditionExpression& e);

  BoundaryConditionExpression(ReferenceCounting<BoundaryExpression> boundary,
			      ReferenceCounting<FunctionVariable> unknown,
			      const BoundaryConditionExpression::BoundaryConditionType& t);

  virtual ~BoundaryConditionExpression();
};

/**
 * @class BoundaryConditionExpressionDirichlet
 * @author Stephane Del Pino
 * @date   Wed Jan  5 20:09:24 2005
 * 
 * @brief This class defines the class of dirichlet BoundaryCondition
 * expressions
 * 
 */

class BoundaryConditionExpressionDirichlet
  : public BoundaryConditionExpression
{
private:
  ReferenceCounting<FunctionExpression> __g;

  std::ostream& put(std::ostream& os) const;

public:
  void execute();

  BoundaryConditionExpressionDirichlet(ReferenceCounting<FunctionVariable> unknown,
				       ReferenceCounting<FunctionExpression> g,
				       ReferenceCounting<BoundaryExpression> boundary);

  BoundaryConditionExpressionDirichlet(const BoundaryConditionExpressionDirichlet& d);

  ~BoundaryConditionExpressionDirichlet();
};


/**
 * @class BoundaryConditionExpressionNeumann
 * @author Stephane Del Pino
 * @date   Wed Jan  5 20:08:47 2005
 * 
 * @brief This class defines the class of neumann BoundaryCondition
 * expressions.
 * 
 */

class BoundaryConditionExpressionNeumann
  : public BoundaryConditionExpression
{
private:
  ReferenceCounting<FunctionExpression> __g;

  std::ostream& put(std::ostream& os) const;

public:
  void execute();

  BoundaryConditionExpressionNeumann(ReferenceCounting<FunctionVariable> unknown,
				     ReferenceCounting<FunctionExpression> g,
				     ReferenceCounting<BoundaryExpression> boundary);

  BoundaryConditionExpressionNeumann(const BoundaryConditionExpressionNeumann& n);

  ~BoundaryConditionExpressionNeumann();
};

/**
 * @class BoundaryConditionExpressionFourrier
 * @author Stephane Del Pino
 * @date   Wed Jan  5 20:08:06 2005
 * 
 * @brief This class defines the class of fourrier BoundaryCondition
 * expressions
 * 
 */

class BoundaryCondition;
class BoundaryConditionExpressionFourrier
  : public BoundaryConditionExpression
{
private:
  ReferenceCounting<FunctionExpression> __Alpha;
  ReferenceCounting<FunctionExpression> __g;

  std::ostream& put(std::ostream& os) const;

public:
  void execute();

  BoundaryConditionExpressionFourrier(ReferenceCounting<FunctionVariable> unknown,
				      ReferenceCounting<FunctionExpression> alpha,
				      ReferenceCounting<FunctionExpression> g,
				      ReferenceCounting<BoundaryExpression> boundary);

  BoundaryConditionExpressionFourrier(const BoundaryConditionExpressionFourrier& f);

  ~BoundaryConditionExpressionFourrier();
};

#endif // BOUNDARYCONDITION_EXPRESSION_HPP
