//////////////////////////////////////////////////////////////////////////////
// dltypes.h

#ifndef DLTYPES_H
#define DLTYPES_H

#include <cstdio>
#include <map>
#include <vector>

#if HAVE_GCC_VERSION(3,1)
    #include <ext/hash_map>
    using namespace __gnu_cxx;
    #define HAVE_HASHMAP
#elif HAVE_GCC_VERSION(2,95)
    #include <hash_map>
    #define HAVE_HASHMAP
#else
    #error "I do not know about this compiler yet!"
    #include <map>
#endif

#ifndef assert
# include <assert.h>
#endif

#include "names.h"
#include "builtin.h"


#define IDENT_LENGTH   1001     // maximum ID length, not counting trailing '\0' 

const unsigned SCALE=3;	// number of digits allowed after the decimal point
const unsigned SCALEFACT=1000;	// 10^SCALE

//////////////////////////////////////////////////////////////////////////////
// print_list

template <class T_ITERATOR>
void print_list(
    ostream&         out,
    const T_ITERATOR begin,
    const T_ITERATOR end,
    const char*      seperator )
    {
    for(T_ITERATOR i=begin; i != end; i++)
        {
        if( i != begin )
            out << seperator;
        out << *i;
        }
    }

template <class T>
void print_list(
    ostream&    out,
    const T&    x,
    const char* seperator)
    {
    print_list(out,x.begin(),x.end(),seperator);
    }


//////////////////////////////////////////////////////////////////////////////
class TERM
//
    {
public:
    enum TYPE { Variable, String, Integer, NullConst };
    typedef STRING_CONSTANTS::index_t index_t;

private:
    TYPE type;

    union
        {
	index_t    c_index;
        unsigned   var;
        } data;

    const NAMES_ITEM& getName() const
        {
        assert( type==String );

        return Strings.getItem(data.c_index);
        }

public:
    static STRING_CONSTANTS Strings;
    static INTEGER_CONSTANTS Integers;

    TYPE getType() const
        {
        return type;
        } 

    // FIXME: Actually, this is redundant now.
    bool isVar() const
        {
        return type==Variable;
        }

    bool isInt() const
        {
        return type==Integer;
        }

    index_t getIndex() const
	{
        assert( type==String || type==Integer );

	return data.c_index;
	}

    const char* getNameAsString() const
        {
        assert( type==String );

        return( Strings.getItem(data.c_index).getName() );
        }

    unsigned getVar() const
        {
        assert( type==Variable );

        return data.var;
        }

    int getInt() const
	{
	assert( type==Integer );

        return( Integers.getItem(data.c_index) );
	}

    bool isNull() const
        {
        return type == NullConst;
        }

public:
    TERM()
        {
        type = NullConst;
        data.c_index = 0;
        }

    TERM(const TERM &term) : type(term.type), data(term.data)
        {
        }

    TERM &operator= (const TERM &term)
        {
        if( this != &term )
            {
	    type=term.type;
            data=term.data;
            }

        return *this;
        }
 
    TERM(const char *name2)
        {
        type=String;
        data.c_index = (Strings.add(NAMES_ITEM(name2))).first;
        }

    TERM(const unsigned var2)
        {
	type=Variable;
        data.var=var2;
        }

    // The second argument is only there to differentiate this constructor
    // from the previous one.
    TERM(const int number, const int)
	{
	type=Integer;
        data.c_index = (Integers.add(number)).first;
	}

    ~TERM()
        {
        }

    TERM* clone() const
        {
        assert(this);

        return new TERM(*this);
        }
 
    // Simplified version of the operator!= to be called in grounding 
    // to compare only TERM of type String and Integer. 
    int diffConstants(const TERM &term2) const
        {
        assert( type==String || type==Integer );

        if( type != term2.type )
            return (int)type - (int)term2.type;
        return (data.c_index - term2.data.c_index);
        } 
    
    int operator!= (const TERM &term2) const
        {
        if( type != term2.type )
            return (int)type - (int)term2.type;

	switch (type)
	    {
	    case Variable:
		return getVar() - term2.getVar();
            case String:
                // Note: This does *not* compare the actual strings, but still
                //       imposes a semi-ordering on the set of all TERMS.
                return (data.c_index - term2.data.c_index);
	    case Integer:
                return getInt() - term2.getInt();
	    case NullConst:
                // All NullConsts are equal.
                return false;
	    default:
		assert(0);
                return 0;
	    }
        }

    int lexicographic_compare(const TERM &term2) const
        {
        if( type == String  &&  term2.type == String )
            return strcmp( Strings.getItem(data.c_index).getName(),
                           Strings.getItem(term2.data.c_index).getName() );
        else
            return *this != term2;
        }

    bool operator== (const TERM &term2) const
        {
        return ( *this != term2 ) == 0;
        }

    bool operator< (const TERM &term2) const
        {
        return ( *this != term2 ) < 0;
        }

    bool operator<= (const TERM &term2) const
        {
        return ( *this != term2 ) <= 0;
        }

    bool operator> (const TERM &term2) const
        {
	return ( *this != term2 ) > 0;
        }

    bool operator>= (const TERM &term2) const
        {
        return ( *this != term2 ) >= 0;
        }

    ostream& printTo(ostream& out) const
	{
	switch (type)
	    {
	    case Variable:
		out << '_' << getVar();
		break;
	    case String:
		out << getNameAsString();
		break;
	    case Integer:
                if( PTraceLevel >= 1 )
                    out << getInt() << "/#";
                else
                    out << getInt();
		break;
	    default:
		assert(0);
	    }
	return out;
	}
    };


inline ostream& operator<< (ostream &out, const TERM &term)
    {
    return term.printTo(out);
    }


//////////////////////////////////////////////////////////////////////////////
typedef vector<TERM> TERMS;
//

inline ostream& operator<< (ostream &out, const TERMS &terms)
    {
    print_list(out,terms,",");
    return out;
    }

inline int compare (const TERMS *a, const TERMS *b)
    {
    size_t a_size = a ? a->size() : 0;
    size_t b_size = b ? b->size() : 0;
    if( a_size != b_size )
        return a_size - b_size;

    for(unsigned i=0; i < a_size; i++)
        {
        int diff = ( (*a)[i] != (*b)[i] );

        if( diff )
            return diff;
        }

    // No difference found, so the to TERMS are equal.
    return 0;
    }

/** determines whether v contains pattern.
 * @return true if pattern is found.
 */
inline bool contains(const TERMS &v, const TERM &pattern)
    {
    return find(v.begin(),v.end(),pattern) != v.end();
    }

class AGGREGATEFUNCTION;
class AGGREGATEATOM;

//////////////////////////////////////////////////////////////////////////////
class ATOM
//
    {
    typedef PREDICATE_NAMES::index_t index_t;

    index_t    p_index;  
    union{
        TERMS           *params;
        AGGREGATEATOM   *aggregate;
    } data;

public:
    inline void setParams(const TERMS *params2);
    inline void setAggregate(const AGGREGATEATOM *aggregate2);


    static PREDICATE_NAMES Predicates;

    /** Each predicate is associated with a unique non-negative integer
     *  (starting at 0); these integers can be used to access arrays
     *  and the like.
     */
    unsigned getIndex() const
        {
        return p_index;
        }

    const NAMES_ITEM& getPredItem() const
        {
        return Predicates.getItem(p_index);
        }
  
    const TERMS *getParams() const
        {
        assert( isRegularAtom() );
        return data.params;
        }

    TERMS *getParamsForModification() const
        {
        assert( isRegularAtom() );
        return data.params;
        }

    AGGREGATEATOM &getAggregate() const
        {
        assert( isAggregate() );
        assert( data.aggregate );

        return *data.aggregate;
        }

    ATOM() 
        {
        assert( 0 );
        }

    ATOM(const ATOM &atom2) 
	: p_index(atom2.p_index)
        {
        if( atom2.isAggregate() )
            {
            data.aggregate=0;
            setAggregate(atom2.data.aggregate);
            }
        else
            {
            data.params=0;
            setParams(atom2.data.params);
            }
        }

    ATOM(const ATOM &atom2, const TERMS *params2)
        : p_index(atom2.p_index)
        {
        // We cannot use member initialization here.
        assert( atom2.isRegularAtom() );
        data.params=0;
        setParams(params2);
        }

    inline ATOM(const ATOM &atom2, const TERM v2c[]);

    ATOM(const char *name,
         const TERMS *params2,
         PREDICATE_NAMES::TYPE type = PREDICATE_NAMES::typeIDB)
        {
	unsigned arity = ( params2 ? params2->size() : 0 );

	pair<index_t,bool> res = Predicates.add(name,arity,type);

        // Handle arity mismatches and similar problems.  We can safely ignore
        // these for Query predicates.
	if( ! res.second  &&  type != PREDICATE_NAMES::typeQuery )
	    {

	    if( Predicates.getArity(res.first) != arity )
		{
		cerr << name << ", first used with arity " 
		     << Predicates.getArity(res.first) 
		     << ", now seen with arity " << arity << "." << endl;
                exit(EXIT_FAILURE);
		}
            else
                // Other mismatches should never happen: the parser should
                // catch them, and tightly coupled frontends should take care. 
                InternalError("ATOM constructor");
            }

        p_index = res.first;
        data.params=0;
        setParams(params2);
        }

    inline ATOM(const AGGREGATEATOM &aggregate2);

    ATOM(BUILTIN type, const TERMS *params2)
        : p_index(type)              // indices and builtin numbers are equal
        {
        assert( (params2 ? params2->size() : 0)
                == Predicates.getArity(p_index) );
        // We cannot use member initialization here.
        data.params=0;
        setParams(params2);
        }

    ATOM(const PREDICATE_NAMES::index_t index, const TERMS *params2 )
        : p_index(index)
        {
        assert( (params2 ? params2->size() : 0)
                == Predicates.getArity(p_index) );
        // We cannot use member initialization here. 
        data.params=0;
        setParams(params2);
        }

    // Like the previous constructor, but with a dummy parameter that
    // indicates that we do not want consistency checks.
    ATOM(const PREDICATE_NAMES::index_t index,
        const TERMS *params2,
        const bool )
        : p_index(index)
        {
        // We cannot use member initialization here.
        data.params=0; 
        setParams(params2);
        }

    inline ATOM& operator= (const ATOM &atom2);

    inline ~ATOM();

    friend int operator!= (const ATOM&, const ATOM&);

    const char* getPredName() const
        {
        return getPredItem().getName();
        }

    // The following is needed by the depgraph but should not be used
    // elsewhere.
    const char* getName() const
        {
        return getPredName();
        }

    bool isPropositional() const
        {
        // Aggregates are not propositional by definition; for other
        // atoms we check whether they have any parameters.
        if( isAggregate() )
            return false;
        return ( data.params == 0 || data.params->size() == 0 );
        }

    bool isGround() const
        {
	if(data.params)
	    {
	    for( TERMS::const_iterator i=data.params->begin();
		 i != data.params->end();
		 i++ )
		{
		if( (*i).isVar() )
		    return false;
		}
	    }

        return true;
        }

    bool isEDB() const
        {
        return Predicates.isEDB(p_index);
        }

    bool isIDB() const
        {
        return Predicates.isIDB(p_index);
        }

    bool isUndef() const
        {
        return Predicates.isUndef(p_index);
        }

    /** determines if this is a built-in predicate.
     * @return whether this predicate is built-in.
     */
    bool isBuiltin() const
	{
	return Predicates.isBuiltin(p_index);
	}

    /** Determine if this is a built-in equality predicate.
     * @return true if this built-in is a equality.
     */
    bool isBuiltinEquality() const
        {
        return (isBuiltin() && getBuiltinID() == BuiltinEquality);
        }

    /** determines if this is a aggregate predicate.
     * @return whether this predicate is aggregate.
     */
    bool isAggregate() const
        {
        return Predicates.isAggregate(p_index);
        }

    // Regular atoms are those, which (possibly) have a regular parameter
    // list.  For now this holds for all non-aggregates, but this may well
    // change in the future.
    bool isRegularAtom() const
        {
        return ! isAggregate();
        }

    inline int compare(const AGGREGATEATOM &a) const;

    /** for a built-in predicate, obtain its ID.
     * Must not be called on user predicates, only on built-in predicates.
     * @returns the BUILTIN ID of this predicate.
     */
    BUILTIN getBuiltinID() const
        {
        assert( isBuiltin() );

	// Builtins come first.
        return static_cast<BUILTIN>(p_index);
        }

    /** determines if this is a complex built-in predicate.
     * Must not be called on user predicates, only on built-in predicates.
     * @return whether this predicate is really a built-in function.
     */
    bool isComplexBuiltin() const
        {
        return getBuiltinID() >= NSimpleBuiltins;
        }

    bool refineType(PREDICATE_NAMES::TYPE type) const
        {
	return Predicates.refineType(p_index,type);
	}

    static bool getAtomByPredName(
        const char *pname,
        const TERMS *params2,
        ATOM &result )
        {
        pair<index_t,bool> r = Predicates.find(NAMES_ITEM(pname));

        if( r.second )
            {
            // An existing predicate name.
            assert( params2 ? Predicates.getArity(r.first) == params2->size()
                            : Predicates.getArity(r.first) == 0 );
            result = ATOM(r.first,params2);
            return true;
            }
        else
            {
            // Nonexisting predicate name.
            return false;
            }
        }

    bool getPositive(const TERMS *params2, ATOM &result ) const
        {
        assert(getPredItem().isNegative());

        return getAtomByPredName(getPredItem().getPositiveName(),
                                 params2, 
                                 result );
        }

    bool getNegative(const TERMS *params2, ATOM &result ) const
        {
        assert(getPredItem().isPositive());

        const char *temp = getPredItem().getNegativeName();
        bool success = getAtomByPredName(temp,
                                         params2, 
                                         result );
        delete[] temp;
        return success;
        }

    bool getComplementary(const TERMS *params2, ATOM &result ) const
        {
        if( getPredItem().isPositive() )
            {
            const char* temp = getPredItem().getNegativeName();
            bool success = getAtomByPredName(temp, params2, result );
            delete[] temp;
            return success;
            }
        else
            return getAtomByPredName(getPredItem().getPositiveName(),
                                     params2, result );
        }


    unsigned getArity() const
        {
        return Predicates.getArity(p_index);
        }

    PREDICATE_NAMES::TYPE getType() const
        {
        return Predicates.getType(p_index);
        }

    /////////////////////////////////////////////////////////////////////////
    // occursComplementary returns true if a complementary atom exists.
    // This is done by checking whether the complementary predicate name
    // exists.
    //
    // @return true if a complementary atom exists
    //
    bool occursComplementary() const
        {
        const char* temp;
        bool success;

        if( getPredItem().isPositive() )
            {
            temp = getPredItem().getNegativeName();
            success = Predicates.find(temp).second;
            delete[] temp;
            }
        else
            {
            temp = getPredItem().getPositiveName();
            success = Predicates.find(temp).second;
            }

        return success;
        }

    void printWithoutObjectID(ostream &out) const
        {
        out << getPredName();

        if( ! isPropositional() )
            {
            assert( isRegularAtom() );
            if( occursComplementary() )
                {
                assert( getParams() );
                assert( getParams()->begin() != getParams()->end() );
                TERMS i(getParams()->begin()+1,getParams()->end());
                if( i.size() )
                    out << "(" << i << ")";
                }
            else
                out << "(" << *getParams() << ")";
            }
        }

    void setNameAndArity(const char *name, unsigned arity)
        {
        // FIXME: This is probably not the best way.
        pair<index_t,bool> res = Predicates.add(name,arity,getType());

        p_index = res.first;
        }

    void setArity(unsigned arity)
        {
        Predicates.refineArity(p_index,arity);
        }

    };

inline ostream &operator<< (ostream &out, const AGGREGATEATOM &a);

inline ostream& operator<< (ostream &out, const ATOM &atom)
    {
    if(atom.isRegularAtom())
        { 
        ATOM::Predicates.print(atom.getIndex(),out);

        if( ! atom.isPropositional() )
            {
            out << "(" << *(atom.getParams()) << ")";
            }
        }
    else
        {
        assert( atom.isAggregate() );
        out << atom.getAggregate();
        }

    return out;
    }


inline int operator!= (const ATOM &a, const ATOM &b)
    {
    int diff = ( a.getIndex() - b.getIndex() );
    if( diff )
        return diff;
    if( a.isRegularAtom() && b.isRegularAtom() )
        return compare(a.getParams(),b.getParams());

    assert(a.isAggregate() && b.isAggregate());
    return a.compare(b.getAggregate());
    }
 
inline bool operator== (const ATOM &a, const ATOM &b)
    {
    return ( a != b ) == 0;
    }
 
inline bool operator< (const ATOM &a, const ATOM &b)
    {
    return ( a != b ) < 0;
    }

//////////////////////////////////////////////////////////////////////////////

#ifdef HAVE_HASHMAP

namespace
#if HAVE_GCC_VERSION(3,1)
    __gnu_cxx
#else
    std
#endif
{

template<>
struct hash<ATOM>
    {
    // FIXME: We probably should reconsider this hash function carefully.
    size_t operator()(const ATOM& x) const
        {
        hash<NAMES_ITEM> namesHasher;
        size_t h = namesHasher(x.getPredItem());
         
        if( x.isRegularAtom() )
            {
            if( const TERMS *i = x.getParams() )
                {
                for( TERMS::const_iterator p=i->begin();
                     p != i->end();
                     p++ )
                    {
                    switch( (*p).getType() )
                        {
			case TERM::String:
                            h = h*5+(*p).getIndex();
                            break;
                        case TERM::Integer:
                            h = h*5+(*p).getInt();
                            break;
                        default:
                            assert( 0 );
                        }
                    }
                }
            } 

        return h;
        }
    };

}

#endif

//////////////////////////////////////////////////////////////////////////////
typedef vector<ATOM> ATOMS;
//

#include "literal.h"

class GATOM;

typedef TLITERAL<ATOM>            LITERAL;
typedef TLITERAL<GATOM>           GLITERAL;

#include "rule.h"
typedef TRULE< GATOM,GLITERAL >   GRULE;

typedef vector<LITERAL>           LITERALS;

typedef TDISJUNCTION< ATOM  >     DISJUNCTION;
typedef TDISJUNCTION< GATOM >     GDISJUNCTION; 

typedef TCONJUNCTION< LITERAL >   CONJUNCTION;
typedef TCONJUNCTION< GLITERAL >  GCONJUNCTION;

typedef TCONSTRAINT< LITERAL >    CONSTRAINT;
typedef TCONSTRAINT< GLITERAL >   GCONSTRAINT;

typedef TWEAKCONSTRAINT< LITERAL >    WEAKCONSTRAINT;
typedef TWEAKCONSTRAINT< GLITERAL >   GWEAKCONSTRAINT;

typedef TRULE< ATOM,LITERAL >     RULE;

typedef vector< GCONJUNCTION::const_iterator > GLITERALPTRS;

typedef vector< GRULE >::iterator GRULEPTR;

typedef vector< RULE >            RULES;
typedef vector< GRULE >           GRULES;

typedef vector< CONSTRAINT >      CONSTRAINTS;
typedef vector< GCONSTRAINT >     GCONSTRAINTS;

typedef vector< WEAKCONSTRAINT >      WEAKCONSTRAINTS;
typedef vector< GWEAKCONSTRAINT >    GWEAKCONSTRAINTS;

typedef set<PREDICATE_NAMES::index_t>
				  NEG_ATOMS;

typedef vector< GCONSTRAINT >::iterator
                                  GCONSTRAINTPTR;
typedef vector< GCONSTRAINTPTR >  GCONSTRAINTPTRS;

typedef vector< GWEAKCONSTRAINT >::iterator GWEAKCONSTRAINTPTR;
typedef vector< GWEAKCONSTRAINTPTR >  GWEAKCONSTRAINTPTRS;

typedef vector< unsigned > OIDindices;

typedef vector< unsigned > CriteriaSequence;

#include "interpret.h"

//////////////////////////////////////////////////////////////////////////////
// OID represents an object for the inheritance language extension.
// 
// It consists of:
// oid: an object identifier
// oind: the parent objects' indices (w.r.t. the global structure ObjectIDs)
// edb, idb: the program associated to the objects
// defeasibles: the defeasibe atoms in this object
//
class OID
    {
public:

    typedef set<ATOM> DEFEASIBLES;

private:

    TERM        oid;
    OIDindices  *oind;

    VATOMSET    *edb;
    RULES       *idb;

    DEFEASIBLES *defeasibles;

public:

    // Default constructor, not used.
    OID()
        {
        assert(0);
        }

    // Constructor for regular objects: the structures for the program
    // and the defeasible atoms are created and filled later; the
    // object identifier and parent objects must be fixed upon object
    // creation.
    OID(const char *oid1, OIDindices *oind1)
        {
        assert(oid1);
        assert(strlen(oid1));

        oid = TERM(oid1);
        oind = oind1;
        edb = new VATOMSET;
        idb = new RULES;

        defeasibles = new DEFEASIBLES;
        }

    // Constructor for the default object: It uses global structures
    // for storing its program, pointers to which should be passed as
    // arguments. The default object is isolated in the inheritance
    // hierarchy, therefore no defeasible atoms can exist.
    OID(const char *oid1, OIDindices *oind1, VATOMSET *edb1, RULES *idb1)
        {
        assert(oid1);
        assert(strlen(oid1));

        oid = TERM(oid1);
        oind = oind1;
        edb = edb1;
        idb = idb1;

        defeasibles = NULL;
        }

    // Copy constructor: edb, idb, and defeasibles are not copied
    // physically; multiple copies share one structure of each.
    OID(const OID &oid2)
        {
        oid = oid2.oid;
        oind = oid2.oind;
        edb = oid2.edb;
        idb = oid2.idb;
        defeasibles = oid2.defeasibles;
        }

    // Assignment: edb, idb, and defeasibles are not copied
    // physically; multiple copies share one structure of each.
    OID& operator= (const OID &oid2)
        {
        if( &oid2 != this )
            {
            oid = oid2.oid;
            oind = oid2.oind;
            edb = oid2.edb;
            idb = oid2.idb;
            defeasibles = oid2.defeasibles;
            }
        return *this;
        }

    // Destructor:
    // FIXME: edb, idb, defeasibles are never freed if they were
    //        created in the constructor.
    ~OID()
        {
        }

    void addIDBRule(const RULE &rule)
        {
        idb->push_back(rule);
        }

    void addEDBFact(const ATOM &a)
        {
        edb->add(a);
        }

    RULES* getIDB() const
        {
        return idb;
        }

    VATOMSET *getEDB() const
        {
        return edb;
        }

    void setIDB(const RULES &idb2)
        {
        idb = new RULES(idb2);
        }

    void setEDB(const VATOMSET &edb2)
        {
        edb = new VATOMSET(edb2);
        }

    OIDindices *getParentOIDindices() const
        {
        return oind;
        }

    const TERM &getName() const
        {
        return oid;
        }

    const char *getString() const
        {
        return oid.getNameAsString();
        }

    bool isDefeasible(const ATOM &a) const
        {
        return ( (defeasibles != NULL) 
                 && (defeasibles->find(a) != defeasibles->end()) );
        }

    void addDefeasible(const ATOM &a)
        {
        assert(defeasibles);
        defeasibles->insert(a);
        }

    DEFEASIBLES* getDefeasibles() const
        {
        return defeasibles;
        }
    };


// The object name is used to test inequality (equality, not
// equivalence is tested).
inline bool operator!= (const OID &oid, const char *oidstring)
    {
    return strcmp(oid.getString(),oidstring);
    }

inline bool operator== (const OID &oid, const char *oidstring)
    {
    return ! (oid != oidstring);
    }

inline bool operator!= (const OID &oid, const OID &oid2)
    {
    return oid != oid2;
    }

inline bool operator== (const OID &oid, const OID &oid2)
    {
    return ! (oid != oid2);
    }

inline ostream& operator<< (ostream &out, const OID &oid)
    {
    out << oid.getString();
    return out;
    }

typedef vector< OID > OIDs;


//////////////////////////////////////////////////////////////////////////////
/** GATOM is a representation of a ground atom.
 * 
 * A GATOM is represented internally by an index.
 *
 * The ATOMs corresponding to all GATOMs are stored in the global
 * i2hTable, where we can efficiently perform searches using an ATOM
 * as a key and immediately obtain that ATOM's index.
 * On the other hand, by means of h2iTable we can immediately access
 * the ATOM corresponding to each GATOM.  
 */
class GATOM
//
    {
    friend class GSUBSET;
    friend int main(int argc, char* argv[]);

    unsigned index;

#ifdef HAVE_HASHMAP
    typedef hash_map<ATOM,unsigned> GATOMHASH;
#else
    typedef map<ATOM,unsigned> GATOMHASH;
#endif

    static GATOMHASH h2iTable;
    static vector<GATOMHASH::const_iterator> i2hTable;

    // Indicate whether the tables above are frozen, i.e., whether it is
    // possible to add new GATOMs. As GATOMs are managed on a global base,
    // this is quite useful for catching bugs.
    static bool frozen;

public:
    explicit GATOM(const unsigned i)
        : index(i)
        { }

    explicit GATOM(const ATOM &atom2);

    GATOM()
        {
        assert( 0 );
        }

    GATOM(const GATOM& atom2)
        : index(atom2.index)
        {
        }

    GATOM& operator= (const GATOM& atom2)
        {
        if( this != &atom2 )
            {
            index=atom2.index;
            }

        return *this;
        }

    ~GATOM()
        {
        }
    
    static void freeze()
        {
        frozen=true;
        }

    static bool isfrozen()
        {
        return frozen;
        }

    /** Each GATOM is associated with a unique non-negative integer
     *  (starting at 0); these integers can be used to access arrays
     *  and the like.
     */
    unsigned getIndex() const
        {
        return index;
        }

    // Dummy for depgraph.h only:
    int* getName() const
        {
        assert( 0 );
        return 0;
        }

    // Return whether this is an aggregate.
    bool isAggregate() const
        {
        return getTheRealOne().isAggregate();
        }

    bool isRegularAtom() const
        {
        return ! isAggregate();
        }

    // Return the aggregate stored in vector aggregates.     
    AGGREGATEATOM &getAggregate() const
        {
        return getTheRealOne().getAggregate();
        }

    const bool hasToBeCheckedForSupportedness() const
        {
        return ! isAggregate();
        }

    const ATOM &getTheRealOne() const
        {
        return (*i2hTable[index]).first;
        }

    // Are these two GATOMS complements?
    bool isComplementary(const GATOM& gatom2) const
        {
        bool success = false;
        if( getTheRealOne().getPredItem().isPositive() ==
            gatom2.getTheRealOne().getPredItem().isNegative() &&
            ( !strcmp(getTheRealOne().getPredItem().getPositiveName(),
                      gatom2.getTheRealOne().getName())  || 
            !strcmp(getTheRealOne().getName(),
                    gatom2.getTheRealOne().getPredItem().getPositiveName()) ) )
            {
            if ( *(getTheRealOne().getParams()) == 
                 *(gatom2.getTheRealOne().getParams()) )
                {
                success = true;
                }
            }
        return success;
        }

    int  operator!= (const GATOM&) const;
    bool operator== (const GATOM&) const;
    bool operator<  (const GATOM&) const;
    };

class AGGREGATE_TYPE
    {
public:
    enum TYPE { Any, Avg, Count, Max, Min, Sum, Times};

private:
    TYPE type;
    
public:
    AGGREGATE_TYPE()
        {
        assert(0);
        }

    AGGREGATE_TYPE( const TYPE &type2)
        :type(type2)
        {
        }
 
    AGGREGATE_TYPE( const AGGREGATE_TYPE &a)
        : type(a.type)
        {
        }

    AGGREGATE_TYPE &operator= (const AGGREGATE_TYPE a)
        {
        if(this != &a)
            type = a.type;

        return *this;
        }

    TYPE getType() const
        {
        return type;
        }

    const char *getName() const
        {
        switch ( type )
            {
            case Any:
                return "#any";
            case Avg:
                return "#avg";
            case Count:
                return "#count";
            case Max:
                return "#max";
            case Min:
                return "#min";
            case Sum:
                return "#sum";
            case Times:
                return "#times";
            default:
                assert(0);
                return 0;
            }

        }

    TYPE operator()() const
        {
        return type;
        }
    };

/////////////////////////////////////////////////////////////////////////////
class AGGREGATEPRED
//
    {
    AGGREGATE_TYPE aggr_type;
    TERMS       var;
    CONJUNCTION con;
public:
    AGGREGATEPRED()
        {
        assert(0);
        }

    AGGREGATEPRED( AGGREGATE_TYPE aggr_type2,
        const TERMS &var2,
        const CONJUNCTION &con2 )
        : aggr_type(aggr_type2),
          var(var2), 
          con(con2)
        {
        }
   
    AGGREGATEPRED(const AGGREGATEPRED&)
        {
        assert(0);
        }
 
    void operator=(const AGGREGATEPRED&)
        {
        assert(0);
        }
 
    ~AGGREGATEPRED()
        {
        }

    AGGREGATE_TYPE getAggregateType() const
        {
        return aggr_type;
        }

    const TERMS &getVars() const
        {
        return var;
        }
    const CONJUNCTION& getConjunction() const
        {
        return con;
        }
    };

typedef pair<TERM,bool> GUARD;

/**Return  1 if g1 >  g2,
 * return -1 if g1 <  g2,
 * return  0 if g1 == g2.
 */
inline int operator!=(const GUARD &g1, const GUARD &g2)
    {
    int i = g1.second ? 1 : 0;
    int j = g2.second ? 1 : 0;
 
    if (i != j)
        return i - j;
 
    return (g1.first != g2.first);
    }

inline bool operator==(const GUARD &g1, const GUARD &g2)
    {
    return (g1 != g2) == 0;
    }

inline bool operator<=(const GUARD &g1, const GUARD &g2)
    {
    return (g1 != g2) <= 0;
    }

inline bool operator<(const GUARD &g1, const GUARD &g2)
    {
    return (g1 != g2) < 0;
    }

inline void printGuard(const GUARD &g, ostream &out)
    {
    if(g.first.isVar())
        out << g.first;
    else if(g.first.isInt())
        out << g.first.getInt();
    else
        assert(0);
    }

typedef pair<TERMS,GATOM> AGGREGATESETVALUE;

inline int operator!=(const AGGREGATESETVALUE &a1,
    const AGGREGATESETVALUE &a2)
    {
    int diff = ( a1.first != a2.first );
    if (diff)
        return diff;

    return ( a1.second != a2.second );
    }

inline ostream &operator<< (ostream &out, const AGGREGATESETVALUE &a)
    {
    out << "<";
    if( a.first.size() > 1 )
        out << "(";
    out << a.first;
    if( a.first.size() > 1 )
        out << ")";
    out << "," << a.second.getTheRealOne() << ">";

    return out;
    }

typedef set<AGGREGATESETVALUE> AGGREGATESET;

inline int operator!=(const AGGREGATESET &s1, const AGGREGATESET &s2)
    {
    int diff = s1.size() - s2.size();
 
    if (diff)
        return diff;
 
    AGGREGATESET::iterator i=s1.begin(); 
    AGGREGATESET::iterator j=s2.begin();

    while(i != s1.end() && j != s2.end())
        {
        diff = *i != *j;
        if(diff)
            return diff;
        i++; j++;
        }
 
    return 0;
    }

inline ostream &operator<< (ostream &out, const AGGREGATESET &a)
    {
    out << "{";
    print_list(out,a.begin(),a.end(),","); 
    out << "}";
 
    return out;
    }

//////////////////////////////////////////////////////////////////////////////

#ifdef HAVE_HASHMAP

namespace
#if HAVE_GCC_VERSION(3,1)
    __gnu_cxx
#else
    std
#endif
{

template<>
struct hash<AGGREGATESET>
    {
    // FIXME: To get a smaller value for the hash function we should 
    // skip the predicate names from all atoms but the first.
    size_t operator()(const AGGREGATESET& x) const
        {
        size_t h = 0;
        hash<ATOM> atomHasher;
        for( AGGREGATESET::const_iterator i = x.begin();
            i != x.end();
            i++ )
            h += atomHasher( (*i).second.getTheRealOne() );

        return h;       
        }
    };

}

#endif

/////////////////////////////////////////////////////////////////////////////
/**Each instance of the class AGGREGATEFUNCTION contains a pointer to an
 * aggregateSet. During instantiation of the GATOMs these AGGREGATESETs
 * corresponding to all AGGREGATEFUNCTIONs are stored in the global
 * i2hAggregateSetTable, where we can efficiently lookup the index of an 
 * AGGREGATESET using that AGGREGATESET as a key and immediately obtain the 
 * AGGREGATESET's index. On the other hand, by means of h2iTable we can 
 * immediately access the AGGREGATESET corresponding to each 
 * AGGREGATEFUNCTION.
 *
 * The field aggregateSet is initialized to an empty set by the constructors
 * of the class. This way, we can distinguish between three different
 * situations:
 *   1. aggregateSet is not instantiated, hence it is still represented by
 *                var: conj
 *      This situation can be checked by the condition
 *                aggregateSet.size() == 0
 *   2. aggregateSet has been instantiated. In this case we check for  
 *                aggregateSet.size() != 0
 *   3. aggregateSet has been stored in h2iAggregateSetTable and aggregateSet
 *      has been set to the null pointer. In this case we check for 
 *                aggregateSet == NULL
 */
class AGGREGATEFUNCTION
//
    {
protected:
#ifdef HAVE_HASHMAP
    typedef hash_map<AGGREGATESET,unsigned> AGGREGATESETHASH;
#else
    typedef map<AGGREGATESET,unsigned> AGGREGATESETHASH;
#endif
   
    static AGGREGATESETHASH h2iAggregateSetTable;
    static vector<AGGREGATESETHASH::const_iterator> i2hAggregateSetTable;
    
    TERMS          freeVars;
    CONJUNCTION    auxAtom;
    unsigned       aggregateSetIndex;
    AGGREGATESET  *aggregateSet;

    // Minimum value of the aggregate function. It's updated by the
    // values of true and undefined GATOMs in the aggregateSet.
    int lowerBound;
    // Maximum value of the aggregate function. It's updated by the
    // values of true and undefined GATOMs in the aggregateSet.
    int upperBound;

    // The initial values for lowerBound and upperBound are the following:
    // - count [0,0];
    // - max   [INT_MIN,INT_MIN];
    // - min   [INT_MIN,INT_MIN];
    // - sum   [0,0];
    // - times [0,1].

    AGGREGATE_TYPE aggregate_type;

    const AGGREGATESET &getAggregateSetFromHTable() const
        {
        assert( ! aggregateSet );
        return i2hAggregateSetTable[aggregateSetIndex]->first;
        }

    void verifyIsInt( const TERM &term ) const
        {
        if ( ! term.isInt() )
            {
            cerr << "Error: the free variable inside the aggregate can be "
                 << endl << "instantiated only with integers values."
                 << endl;
            exit(1);
            }
        }

    virtual void initBounds() {}

public:  
    AGGREGATEFUNCTION()
        : aggregateSet(0)
        {
        assert(0);
        }

    AGGREGATEFUNCTION(const AGGREGATEPRED &aggrPred)
        : freeVars(aggrPred.getVars()),
          auxAtom(aggrPred.getConjunction()),
          aggregateSetIndex(0),
          aggregateSet(0),
          lowerBound(0),            
          upperBound(0),
          aggregate_type(aggrPred.getAggregateType())   
        {
        assert( ! aggregateSet );
        aggregateSet = new AGGREGATESET();
        }
  
    AGGREGATEFUNCTION(const AGGREGATEFUNCTION &a2)
        : freeVars(a2.freeVars),
          auxAtom(a2.auxAtom),
          aggregateSetIndex(a2.aggregateSetIndex),
          aggregateSet(0),
          lowerBound(a2.lowerBound),
          upperBound(a2.upperBound),
          aggregate_type(a2.aggregate_type)
        {
        if ( a2.aggregateSet )
            aggregateSet = new AGGREGATESET( *a2.aggregateSet );
        }

    AGGREGATEFUNCTION(const AGGREGATEFUNCTION &a2,
        const TERM v2c[])
        : freeVars(a2.freeVars),
          aggregateSetIndex(a2.aggregateSetIndex),
          aggregateSet(0),
          lowerBound(a2.lowerBound),
          upperBound(a2.upperBound),
          aggregate_type(a2.aggregate_type)
        {
        const ATOM &a = *a2.auxAtom.begin();
        auxAtom.add( LITERAL( ATOM(a,v2c) ) );

        if ( a2.aggregateSet )
            aggregateSet = new AGGREGATESET( *a2.aggregateSet );
        }

    AGGREGATEFUNCTION& operator=(const AGGREGATEFUNCTION &a2)
        {
        if( this != &a2 )
            {
            freeVars = a2.freeVars;
            auxAtom = a2.auxAtom;
            aggregateSetIndex = a2.aggregateSetIndex;
            lowerBound = a2.lowerBound;
            upperBound = a2.upperBound;
            aggregate_type = a2.aggregate_type;
            if( aggregateSet )
                delete aggregateSet;
            if ( a2.aggregateSet ) 
                aggregateSet = new AGGREGATESET( *a2.aggregateSet ); 
            else 
                aggregateSet = 0;
            } 
        return *this;
        }

    inline AGGREGATEFUNCTION *newAggregateFunction() const;
    inline AGGREGATEFUNCTION *newAggregateFunction(const TERM v2c[]) const;
    inline AGGREGATEFUNCTION *newAggregateFunction(
        const AGGREGATEPRED  &) const;

    /** Insert each new instance of aggregateSet into a static hash 
     *  table, save its index in this table in aggregateSetIndex, 
     *  and set the pointer to aggregateSet to 0. 
     */
    void moveAggregateSetToHash()
        {
        if( aggregateSet )
            {
            pair<AGGREGATESETHASH::iterator,bool> p =
                h2iAggregateSetTable.insert( AGGREGATESETHASH::value_type
                ( *aggregateSet, h2iAggregateSetTable.size() ) );

            aggregateSetIndex = (*p.first).second;

            // Add this to the hash table only if it is a new AGGREGATESET.
            if( p.second )
                i2hAggregateSetTable.push_back(p.first);

            delete aggregateSet;
            aggregateSet = 0;
            }
        }

    AGGREGATE_TYPE getAggregateType() const
        {
        return aggregate_type;
        }

    unsigned getAggregateSetIndex() const
        {
        return aggregateSetIndex;
        }
 
    const ATOM& getAuxAtom() const
        {
        assert(auxAtom.size()==1);
        return *(auxAtom.begin());
        }

    ATOM &getAuxAtomForModification()
        {
        assert(auxAtom.size()==1);
        return *(auxAtom.begin());
        }

    void setAuxAtom(const ATOM &atom2)
        {
        auxAtom.clear();
        auxAtom.add(atom2);
        }

    const TERMS &getFreeVars() const
        {
        return freeVars; 
        }

    const CONJUNCTION& getConjunction() const
        {
        assert (auxAtom.size()>=1);
        return auxAtom;
        }
    
    const AGGREGATESET &getAggregateSet() const
        {
        if ( aggregateSet )
            return *aggregateSet;
        else
            return getAggregateSetFromHTable();
        }    

    const AGGREGATESET *getAggregateSetPtr() const
        {
        if( aggregateSet )
            return aggregateSet;
        else
            return &( getAggregateSetFromHTable() );
        }
 
    inline void addToAggregateSet(const AGGREGATESETVALUE &g); 
    
    void reset() 
        {
        initBounds();
        if ( aggregateSet )
            aggregateSet->clear();
        }
 
    virtual ~AGGREGATEFUNCTION() 
        {
        if(aggregateSet)
            delete aggregateSet;   
        }

    /** Return the number of ground auxAtoms appearing in the aggregate 
     *  function and already true w.r.t. the current interpretation.
     */      
    int getLowerBound() const
        {
        return lowerBound;
        }

    /** Return the number of ground auxAtoms appearing in the aggregate 
     *  function which can become true w.r.t. the current interpretation.
     */    
    int getUpperBound() const
        {
        return upperBound;    
        }

    /** Return the number of ground auxAtoms appearing in the aggregate
     *  function and already true w.r.t. the current interpretation.
     */
    int &getLowerBoundAsReference()
        {
        return lowerBound;
        }
 
    /** Return the number of ground auxAtoms appearing in the aggregate
     *  function which can become true w.r.t. the current interpretation.
     */
    int &getUpperBoundAsReference()
        {
        return upperBound;
        }

    virtual void updateBounds(const TERM &) {}
    virtual void updateBounds(const TERM &, const bool) {}
    virtual void finishBounds() {}

    // The following two methods are implemented only in the subclass
    // MAXFUNCTION. For the other subclasses any returned value 
    // is fine, except INT_MIN. 
    virtual int getMaxUndef() const {return 0;}
    virtual int getMaxTrue() const {return 0;}

    // The following two methods are implemented only in the subclass
    // MINFUNCTION. For the other subclasses any returned value 
    // is fine, except INT_MIN. 
    virtual int getMinUndef() const {return 0;}
    virtual int getMinTrue() const {return 0;}

    virtual void setMaxUndef(int) {}
    virtual void setMaxTrue(int) {}
    virtual void setMinUndef(int) {}
    virtual void setMinTrue(int) {}
    };

inline int operator!=( const AGGREGATEFUNCTION &a, 
    const AGGREGATEFUNCTION &b )
    {
    int diff = ( a.getFreeVars() != b.getFreeVars() );
    if ( diff )
        return diff;

    diff = ( a.getAuxAtom() != b.getAuxAtom() );
    if ( diff )
        return diff;

    int i = a.getAggregateSetPtr() ? 1 : 0;
    int j = b.getAggregateSetPtr() ? 1 : 0;
    if ( i != j )
        return i - j;
    if ( a.getAggregateSetPtr() )
        {
        assert( b.getAggregateSetPtr() );
        diff = ( a.getAggregateSet() != b.getAggregateSet() );
        if ( diff )
            return diff;
        }

    // If we compare two aggregate functions with all of the above equal,
    // we still cannot be sure about their equality.  It may happen that
    // both sets are empty (and thus equal), but for different reasons: an
    // aggregate set can be empty if only true GATOMs have been found in the
    // set or if no GATOMs have been found at all.  In order to distinguish
    // these two cases, we also need to compare  the lower bounds (which are
    // updated in the first case only).
    diff =  a.getLowerBound() != b.getLowerBound();
    if( diff )
        return diff;

    return 0;
    }

inline bool operator==( const AGGREGATEFUNCTION &a, 
    const AGGREGATEFUNCTION &b )
    {
    return ( a != b ) == 0;
    }

inline ostream &operator<< (ostream &out, const AGGREGATEFUNCTION &a)
    {
    if (GTraceLevel >= 1  || TraceLevel >= 1)
        out << " [" << a.getLowerBound() << "] ";

    out << a.getAggregateType().getName();

    if( a.getAggregateSet().size() )
        out << a.getAggregateSet();
    else
        out << "{" << a.getFreeVars() << " : " << a.getAuxAtom() << "}";

    if (GTraceLevel >= 1  || TraceLevel >= 1)
        out << " [" << a.getUpperBound() << "] ";

    return out;
    }

class COUNTFUNCTION : public AGGREGATEFUNCTION
    {
    void initBounds()
        {
        lowerBound=0;
        upperBound=0;
        }

public:
    COUNTFUNCTION()
        {
        assert(0);
        }

    COUNTFUNCTION(const AGGREGATEPRED &aggrPred)
        : AGGREGATEFUNCTION(aggrPred)
        {
        }

    COUNTFUNCTION(const AGGREGATEFUNCTION &aggrFunct)
        : AGGREGATEFUNCTION(aggrFunct)
        {
        }

    COUNTFUNCTION(const AGGREGATEFUNCTION &aggrFunct,
        const TERM v2c[])
        : AGGREGATEFUNCTION(aggrFunct,v2c)
        {
        }

    void operator=(const COUNTFUNCTION &)
        {
        assert(0);
        }

    /**Increment the upper bound if a new ground instance of the 
     * auxiliary atom is found.
     */
    void updateBounds(const TERM &)
        {
        upperBound++;
        }

    /**Increment the lower bound if a true instance of the auxiliary 
     * atom is found; decrement the upper bound if a false instance
     * of the auxiliary atom is found (one of false and true). 
     */
    void updateBounds(const TERM &, const bool truthValue)
        {
        if( truthValue )
            lowerBound++;
        else
            upperBound--;
        }
    };

class MAXFUNCTION : public AGGREGATEFUNCTION
    {
    // Maximum value among the projected terms of undefined auxiliary atoms.
    int maxUndef;
    // Maximum value among the projected terms of true auxiliary atoms.
    int maxTrue;

    void init(const AGGREGATEFUNCTION &aggrFunct)
        {
        assert( aggrFunct.getAggregateType().getType()
                == AGGREGATE_TYPE::Max );

        const MAXFUNCTION *m = static_cast<const MAXFUNCTION*>(&aggrFunct);
        maxUndef = m->maxUndef;
        maxTrue = m->maxTrue;
        }

    void initBounds()
        {
        // Initialize lower and upper bounds such that the guards are 
        // violated whenever the aggregateSet is empty, regardless whether 
        // the instantiation has not derived any GATOM in the set as 
        // undefined or true, or all GATOMs in the set became false during 
        // propagation.
        lowerBound = INT_MIN;
        upperBound = INT_MIN;
        maxUndef = INT_MIN;
        maxTrue = INT_MIN;
        }

public:
    MAXFUNCTION()
        {
        assert(0);
        }

    MAXFUNCTION(const AGGREGATEPRED &aggrPred)
        : AGGREGATEFUNCTION(aggrPred) 
        {
        initBounds();
        }

    MAXFUNCTION(const AGGREGATEFUNCTION &aggrFunct)
        : AGGREGATEFUNCTION(aggrFunct)
        {
        init(aggrFunct);
        }

    MAXFUNCTION(const AGGREGATEFUNCTION &aggrFunct,
        const TERM v2c[])
        : AGGREGATEFUNCTION(aggrFunct,v2c)
        {
        init(aggrFunct);
        }

    void operator=(const MAXFUNCTION &)
        {
        assert(0);
        }

    int getMaxUndef() const  { return maxUndef; }

    int getMaxTrue() const  { return maxTrue; }

    int &getMaxUndefAsReference()
        {
        return maxUndef;
        }

    int &getMaxTrueAsReference()
        {
        return maxTrue;
        }

    void setMaxUndef(int max)
        {
        maxUndef = max;
        }

    void setMaxTrue(int max)
        {
        maxTrue = max;
        }

    /**Update the lower and upper bounds as follows:
     *   - lowerBound = maxTrue
     *   - upperBound = max{ maxUndef, maxTrue }
     */     
    void adjustBounds()
        {
        lowerBound = maxTrue;
        upperBound = max(maxUndef,maxTrue);
        }

    void updateBounds(const TERM &term)
        {
        verifyIsInt(term);
        int i = term.getInt();

        if( maxUndef < i )
            maxUndef = i;

        // Update lower and upper bounds.
        adjustBounds();
        }

    void updateBounds(const TERM &term, const bool isTrue)
        {
        if( isTrue )
            {
            // Update maxTrue.
            verifyIsInt(term);

            if( maxTrue < term.getInt() )
                maxTrue = term.getInt();
            }

        // Update lower and upper bounds.
        adjustBounds();
        } 
    };

class MINFUNCTION : public AGGREGATEFUNCTION
    {       
    // Minimum value among the projected terms of undefined GATOMs in the
    // aggregateSet.
    int minUndef;
    // Minimum value among the projected terms of true GATOMs in the
    // aggregateSet.
    int minTrue;

    void init(const AGGREGATEFUNCTION &aggrFunct)
        {
        assert( aggrFunct.getAggregateType().getType()
                == AGGREGATE_TYPE::Min );

        const MINFUNCTION *m = static_cast<const MINFUNCTION*>(&aggrFunct);
        minUndef = m->minUndef;
        minTrue = m->minTrue;
        }

    void initBounds()
        {       
        // Initialize lower and upper bounds such that the guards are
        // violated whenever the aggregateSet is empty, regardless whether
        // the instantiation has not derived any GATOM in the set as
        // undefined or true, or all GATOMs in the set became false during
        // propagation.
        lowerBound = INT_MIN;
        upperBound = INT_MIN;
        minUndef = INT_MIN;
        // Initialize minTrue to INT_MAX, so that its value can be correctly 
        // used to update the upper bound value, also in the case in which
        // no True GATOMs have been found in the aggregateSet.
        minTrue = INT_MAX;
        }

public:
    MINFUNCTION()
        {
        assert(0);
        }

    MINFUNCTION(const AGGREGATEPRED &aggrPred)
        : AGGREGATEFUNCTION(aggrPred)
        {
        initBounds();
        }

    MINFUNCTION(const AGGREGATEFUNCTION &aggrFunct)
        : AGGREGATEFUNCTION(aggrFunct)
        {
        init(aggrFunct);
        }

    MINFUNCTION(const AGGREGATEFUNCTION &aggrFunct,
        const TERM v2c[])
        : AGGREGATEFUNCTION(aggrFunct,v2c)
        {
        init(aggrFunct);
        }

    void operator=(const MINFUNCTION &)
        {
        assert(0);
        }
			     
    int getMinUndef() const  { return minUndef; }

    int getMinTrue() const  { return minTrue; }

    int &getMinUndefAsReference()
        {
        return minUndef;
        }

    int &getMinTrueAsReference()
        {
        return minTrue;
        }

    void setMinUndef(int min)
        {
        minUndef = min;
        }

    void setMinTrue(int min)
        {
        minTrue = min;
        }

    /**Update the lower and upper bounds as follows:
     *   - lowerBound = min{ minUndef, minTrue }
     *   - upperBound = minTrue
     */     
    void adjustBounds()
        {
        lowerBound = min(minUndef,minTrue);
        upperBound = minTrue;
        }

    void updateBounds(const TERM &term)
        {
        verifyIsInt(term);
        int i = term.getInt();

        if( minUndef == INT_MIN  ||  minUndef > i )
            minUndef = i;

        // Update lower and upper bounds.
        adjustBounds();
        }

    void updateBounds(const TERM &term, const bool isTrue)
        {
        if( isTrue )
            {
            // Update minTrue.
            verifyIsInt(term);

            if( minTrue == INT_MAX  ||  minTrue > term.getInt() )
                minTrue = term.getInt();
            }

        // Update lower and upper bounds.
        adjustBounds();
        } 
    };


class SUMFUNCTION : public AGGREGATEFUNCTION
    {
    void initBounds()
        {
        lowerBound=0;
        upperBound=0;
        }

public:
    SUMFUNCTION()
        {
        assert(0);
        }

    SUMFUNCTION(const AGGREGATEPRED &aggrPred)
        : AGGREGATEFUNCTION(aggrPred)
        {
        }

    SUMFUNCTION(const AGGREGATEFUNCTION &aggrFunct)
        : AGGREGATEFUNCTION(aggrFunct)
        {
        }

    SUMFUNCTION(const AGGREGATEFUNCTION &aggrFunct,
        const TERM v2c[])
        : AGGREGATEFUNCTION(aggrFunct,v2c)
        {
        }

    void operator=(const SUMFUNCTION &)
        {
        assert(0);
        }

    /**Add term to the upper bound if a new ground instance of the 
     * auxiliary atom is found.
     */    
    void updateBounds(const TERM &term)
        {
        verifyIsInt(term);
        upperBound += term.getInt();
        }

    /**Add term to the lower bound if a true instance of the auxiliary
     * atom is found; subtract term to the upper bound if a false 
     * instance of the auxiliary atom is found (one of false and true).
     */
    void updateBounds(const TERM &term, const bool truthValue)
        {
        verifyIsInt(term);

        if( truthValue )
            lowerBound += term.getInt(); 
        else
            upperBound -= term.getInt();
        }
    };

class TIMESFUNCTION : public AGGREGATEFUNCTION
    {
    // Current value of the aggregate function.
    int currentValue;

    void init(const AGGREGATEFUNCTION &aggrFunct)
        {
        currentValue = ( static_cast<const TIMESFUNCTION*>(&aggrFunct) )
                       ->currentValue;
        }

    void initBounds()
        {
        currentValue = 1;
        upperBound = 1;
        lowerBound = 0;
        }

public:
    TIMESFUNCTION()
        {
        assert(0);
        }

    TIMESFUNCTION(const AGGREGATEPRED &aggrPred)
        : AGGREGATEFUNCTION(aggrPred)
        {
        initBounds();
        }

    TIMESFUNCTION(const AGGREGATEFUNCTION &aggrFunct)
        : AGGREGATEFUNCTION(aggrFunct)
        {
        init(aggrFunct);
        }

    TIMESFUNCTION(const AGGREGATEFUNCTION &aggrFunct,
        const TERM v2c[])
        : AGGREGATEFUNCTION(aggrFunct,v2c)
        {
        init(aggrFunct);
        }

    void operator=(const TIMESFUNCTION &)
        {
        assert(0);
        }

    int &getCurrentValueAsReference()
        {
        return currentValue;
        }

    void incrLowerBound(const TERM &term)
        {
        // Check whether the aggregate value can't change any longer.
        if( lowerBound == 0 && upperBound == 0 && currentValue == 0 )
            return;

        if( lowerBound == 0 )
            currentValue *= term.getInt();
        else
            lowerBound *= term.getInt();

        if( currentValue == 0 )
            upperBound = 0;
        }

    void decrUpperBound(const TERM &term)
        {
        // Check whether the aggregate value can't change any longer.
        if( lowerBound == 0 && upperBound == 0 && currentValue == 0 )
            return;

        if( term.getInt() == 0 )
            lowerBound = currentValue;
        else
            upperBound /= term.getInt();
        }

    /**Multiply term by the upper bound if it is greater than 0 
     * if a new ground instance of the auxiliary atom is found.
     */
    void updateBounds(const TERM &term)
        {
        verifyIsInt(term);

        if( term.getInt() > 0 )
            upperBound *= term.getInt();
        }

    /**If a true instance of the auxiliary atom is found, multiply lowerBound 
     * by term if lowerBound has already been initialized to a positive value;
     * multiply currentValue by term, otherwise. If currentValue becomes 0,
     * set also the upperBound to 0.
     * If a false instance of the auxiliary atom is found, divide upperBound 
     * by term if it is greater than 0; assign currentValue to the lower 
     * bound, otherwise (one of false and true).
     */
    void updateBounds(const TERM &term, const bool truthValue)
        {
        verifyIsInt(term);

        // Check whether the aggregate value can't change any longer.
        if( lowerBound == 0 && upperBound == 0 && currentValue == 0 )
            return;

        if( truthValue )
            incrLowerBound(term);
        else
            decrUpperBound(term);
        }
    
    /* Check whether the aggregateSet is empty or the value 0 doesn't
     * occur in the aggregateSet. In this case assign the current value 
     * to the lower bound.
     */
    void finishBounds()
        {
        const TERMS &terms = ( *getAggregateSet().begin() ).first;
        if( getAggregateSet().empty() 
            || ( ( *terms.begin() ).getInt() != 0 ) )
            lowerBound = currentValue;

        if( currentValue == 0 )
            upperBound = currentValue;
        }
    };

// This struct has been defined to allow the construction of a hash table 
// of pointers to AGGREGATEFUNCTIONs; if we build the hash table by 
// inserting AGGREGATEFUNCTIONPTRs instead of pointers to AGGREGATEFUNCTIONs,
// before a new insertion, the AGGREGATEFUNCTIONs will be properly compared 
// and not their pointers. 
struct AGGREGATEFUNCTIONPTR
    {
    AGGREGATEFUNCTION *aggregateFunction;

    AGGREGATEFUNCTIONPTR()
        : aggregateFunction(0)
        {
        assert(0);
        }

    AGGREGATEFUNCTIONPTR(const AGGREGATEFUNCTION &a)
        : aggregateFunction(0)
        {
        assert( ! aggregateFunction );
        aggregateFunction = a.newAggregateFunction();
        }

    AGGREGATEFUNCTIONPTR(const AGGREGATEFUNCTIONPTR &a)
        : aggregateFunction(0)
        {
        assert( ! aggregateFunction );
        if( a.aggregateFunction )
            aggregateFunction = a.aggregateFunction->newAggregateFunction();      
        }
 
    AGGREGATEFUNCTIONPTR& operator=(const AGGREGATEFUNCTIONPTR &a2)
        {
        if ( this != & a2 )  
            {
            if( aggregateFunction )
                delete aggregateFunction;
            if ( a2.aggregateFunction )
                aggregateFunction = a2. aggregateFunction;
            else
                aggregateFunction = 0;
            }

        return *this;
        }

    ~AGGREGATEFUNCTIONPTR()
        {
        if( aggregateFunction )
            delete aggregateFunction;
        }
    };

inline int operator!=( const AGGREGATEFUNCTIONPTR &a,
    const AGGREGATEFUNCTIONPTR &b)
        {
        return ( a.aggregateFunction->getAggregateSetIndex() - 
            b.aggregateFunction->getAggregateSetIndex() );
        }

inline bool operator==( const AGGREGATEFUNCTIONPTR &a,
    const AGGREGATEFUNCTIONPTR &b)
        {
        return ( a != b ) == 0; 
        }

//////////////////////////////////////////////////////////////////////////////
#ifdef HAVE_HASHMAP

namespace
#if HAVE_GCC_VERSION(3,1)
    __gnu_cxx
#else
    std
#endif
{

template<>
struct hash<AGGREGATEFUNCTIONPTR>
    {
    size_t operator()(const AGGREGATEFUNCTIONPTR x) const
        {
        // FIXME: We have to consider also the type of the aggregate
        // function to calculate correctly the index h.
        size_t h = x.aggregateFunction->getAggregateSetIndex();

        return h;
        }
    };

}

#endif

/////////////////////////////////////////////////////////////////////////////
class AGGREGATEATOM 
//
    {
protected:
#ifdef HAVE_HASHMAP
    typedef hash_map<AGGREGATEFUNCTIONPTR,unsigned> AGGREGATEFUNCTIONHASH;
#else
    typedef map<AGGREGATEFUNCTIONPTR,unsigned> AGGREGATEFUNCTIONHASH;
#endif

    static AGGREGATEFUNCTIONHASH h2iAggregateFunctionTable;
    static vector<AGGREGATEFUNCTIONHASH::const_iterator> 
                                 i2hAggregateFunctionTable;
    static unsigned aggregateFunctionsCount;

    AGGREGATEFUNCTION *aggregateFunction;
    unsigned           aggregateFunctionIndex;
    GUARD              lowerGuard;
    GUARD              upperGuard;
    bool               assignment;

    /** Determine whether guards are violated.
     *  @return true if one of the following happens:
     *  - lowerGuard > upperGuard;
     *
     *  - lowerGuard is inclusive
     *         lowerGuard > upperBound;
     *         upperGuard is not inclusive, and lowerGuard > upperGuard -1;
     *  - upperGuard is inclusive
     *         upperGuard < lowerBound;
     *         lowerGuard is not inclusive, and lowerGuard +1 > upperGuard;
     *  - lowerGuard is not inclusive
     *         lowerGuard >= upperBound;
     *  - upperGuard is not inclusive, i.e. AGGREGATEPRED < upperGuard,
     *         upperGuard <= lowerBound.
     */
    inline bool checkForViolation(const int &lg,
        const int &ug) const
        {
        int lg1 = (isLowerGuardInclusive() ? lg : lg +1);
        int ug1 = (isUpperGuardInclusive() ? ug : ug -1);

        return lg1 > ug1 || lg1 > getUpperBound() || ug1 < getLowerBound();
        }

    /** Determine whether guards can never be violated.
     *  @return true if one of the following happens:
     *  - lowerGuard is inclusive, i.e. lowerGuard <= AGGREGATEPRED,
     *    and lowerGuard <= lowerBound
     *  - lowerGuard is not inclusive, i.e. lowerGuard < AGGREGATEPRED,
     *    and lowerGuard < lowerBound;
     *  - upperGuard is inclusive, i.e. AGGREGATEPRED <= upperGuard,
     *    and upperGuard >= upperBound
     *  - upperGuard is not inclusive, i.e. AGGREGATEPRED < upperGuard,
     *    and  upperGuard > upperBound.
     */
    bool checkForSatisfaction(const int &lg, 
        const int &ug) const
        {
        return((lg < getLowerBound()
                || (isLowerGuardInclusive() && lg==getLowerBound()))
               && (ug > getUpperBound()
                || (isUpperGuardInclusive() && ug==getUpperBound())));
        }

    const AGGREGATEFUNCTION &getAggregateFunctionFromHTable() const
        {
        assert( ! aggregateFunction );
        return *(((*i2hAggregateFunctionTable[aggregateFunctionIndex]).first).
            aggregateFunction);
        }

    AGGREGATEFUNCTION* getAggregateFunctionFromHTablePtr() const
        {
        assert( ! aggregateFunction );
        return ( *i2hAggregateFunctionTable[aggregateFunctionIndex] ).first.
            aggregateFunction;
        }
    
public:  
    AGGREGATEATOM()
        : aggregateFunction(0)
        {
        assert(0);
        }
 
    AGGREGATEATOM(const GUARD &lowerGuard2,
                  const GUARD &upperGuard2,
                  const AGGREGATEPRED &aggrPred)
        : aggregateFunction(0),
          aggregateFunctionIndex(0),
          lowerGuard(lowerGuard2),
          upperGuard(upperGuard2),
          assignment(false)
        {
        assert( ! aggregateFunction );
        aggregateFunction= ( AGGREGATEFUNCTION(aggrPred) ).
            newAggregateFunction(aggrPred);
        }

    AGGREGATEATOM(const AGGREGATEATOM& a2)
        : aggregateFunction(0),  
          aggregateFunctionIndex(a2.aggregateFunctionIndex),
          lowerGuard(a2.lowerGuard),
          upperGuard(a2.upperGuard),
          assignment(a2.assignment)
        {
        if ( a2.aggregateFunction )
            aggregateFunction = a2.aggregateFunction->newAggregateFunction();
        }
    
    AGGREGATEATOM( const AGGREGATEATOM &a2, const TERM v2c[] )
        {
        aggregateFunctionIndex = 0;

        // If the lower guard is inclusive...
        if( a2.lowerGuard.second )
            {
            // ...assign the ground value.
            if( a2.lowerGuard.first.isVar() )
                lowerGuard.first
                    = TERM(v2c[a2.lowerGuard.first.getVar()].getInt(),0);
            else
                lowerGuard.first = a2.lowerGuard.first;

            lowerGuard.second = a2.lowerGuard.second;
            }
        else
            {
            // If the lower guard is exclusive, make it inclusive and
            // assign the updated ground value.
            // l < X becomes l+1 <= X.
            if( a2.lowerGuard.first.isVar() )
                lowerGuard.first
                    = TERM(v2c[a2.lowerGuard.first.getVar()].getInt()+1,0);
            else
                lowerGuard.first = TERM(a2.lowerGuard.first.getInt()+1,0);

            lowerGuard.second = true;
            }

        // If the upper guard is inclusive...
        if( a2.upperGuard.second )
            {
            // ...assign the ground value.
            if( a2.upperGuard.first.isVar() )
                upperGuard.first
                    = TERM(v2c[a2.upperGuard.first.getVar()].getInt(),0);
            else
                upperGuard.first = a2.upperGuard.first;

            upperGuard.second = a2.upperGuard.second;
            }
        else
            {
            // If the upper guard is exclusive, make it inclusive and
            // assign the updated ground value.
            // X < u becomes X <= u-1.
            if( a2.upperGuard.first.isVar() )
                upperGuard.first
                    = TERM(v2c[a2.upperGuard.first.getVar()].getInt()-1,0);
            else
                upperGuard.first = TERM(a2.upperGuard.first.getInt()-1,0);

            upperGuard.second = true;
            }

        assignment = a2.assignment;

        assert( a2.aggregateFunction );
        aggregateFunction = a2.aggregateFunction->newAggregateFunction(v2c);
        }
 
    AGGREGATEATOM& operator=(const AGGREGATEATOM& a2)
        {
        if( this != &a2 )
            {
            aggregateFunctionIndex = a2.aggregateFunctionIndex;
            lowerGuard = a2.lowerGuard;
            upperGuard = a2.upperGuard;
            assignment = a2.assignment;
            if ( aggregateFunction )
                delete aggregateFunction;
            if ( a2.aggregateFunction )
                aggregateFunction = a2.aggregateFunction
                    ->newAggregateFunction();
            else
                aggregateFunction = 0;
            } 
        return *this;
        }

    AGGREGATE_TYPE getAggregateType() const
        {
        return getAggregateFunctionPtr()->getAggregateType();
        }

    static const unsigned getAggregateFunctionsCount()
        {
        return aggregateFunctionsCount;
        }

    void moveAggregateSetToHash()
        {
        getAggregateFunctionPtr()->moveAggregateSetToHash();
        }

    /** Insert each new instance of aggregateFunction into a static hash
     *  table, save its index in this table in aggregateFunctionIndex,
     *  and set the pointer to aggregateFunction to 0.
     */
    void moveAggregateFunctionToHash()
        {
        if( aggregateFunction )
            {
            pair<AGGREGATEFUNCTIONHASH::iterator,bool> p =
                h2iAggregateFunctionTable.insert(
                    AGGREGATEFUNCTIONHASH::value_type
                      (AGGREGATEFUNCTIONPTR(*aggregateFunction),
                       h2iAggregateFunctionTable.size()) );

            aggregateFunctionIndex = (*p.first).second;

            // Add this to the hash table only if it is a new 
            //AGGREGATEFUNCTION.
            if( p.second )
                {
                i2hAggregateFunctionTable.push_back(p.first);
                aggregateFunctionsCount++; 
                }

            delete aggregateFunction;
            aggregateFunction = 0;
            }
        }

    /** Move the aggregateSet and the aggregateFunction into the
     *  related hash tables.
     */ 
    void moveToHashes()
        {
        moveAggregateSetToHash();
        moveAggregateFunctionToHash();	 
        }

    const ATOM& getAuxAtom() const
        {
	return aggregateFunction->getAuxAtom();
        }

    ATOM &getAuxAtomForModification() const
        {
        return aggregateFunction->getAuxAtomForModification();
        }

    void setAuxAtom(const ATOM &atom2)
        {
        aggregateFunction->setAuxAtom(atom2);
        }

    const TERMS &getFreeVars() const
        {
        return aggregateFunction->getFreeVars(); 
        }

    const CONJUNCTION& getConjunction() const
        {
	return aggregateFunction->getConjunction();
        }

    unsigned getAggregateFunctionIndex() const
        {
        return aggregateFunctionIndex;
        }

    AGGREGATEFUNCTION* getAggregateFunctionPtr() const
        {
        if ( aggregateFunction )
            return aggregateFunction;
        else
            return getAggregateFunctionFromHTablePtr();
        }
    
    const AGGREGATESET &getAggregateSet() const
        {
        return getAggregateFunctionPtr()->getAggregateSet();
        }    

    void addToAggregateSet(const AGGREGATESETVALUE &g) 
        {   
        aggregateFunction->addToAggregateSet(g);       
        }
    
    void reset() 
        {
	aggregateFunction->reset();
        }
 
    ~AGGREGATEATOM()
        {
        if(aggregateFunction)
            delete aggregateFunction;
        }

    bool isAssignment() const
        {
        return assignment;
        }

    int operator!= (const AGGREGATEATOM &a)     
        {
        int diff = (lowerGuard != a.lowerGuard);
        if (diff)
            return diff;
        diff = (upperGuard != a.upperGuard);
        if (diff)
            return diff;
        diff = isAssignment() - a.isAssignment();
        if (diff)
            return diff;
        diff = (*getAggregateFunctionPtr() != *a.getAggregateFunctionPtr());
        if (diff)
	    return diff;

        return 0; 
        }

    void setAssignment(const bool &assignment2) 
        {
        assignment = assignment2; 
        }
    
    const GUARD& getLowerGuard() const
        {
        return lowerGuard;
        } 
    
    const GUARD& getUpperGuard() const
        {
        return upperGuard;
        }
 
    bool isLowerGuardInclusive() const
        {
        return lowerGuard.second;
        }  

    bool isUpperGuardInclusive() const
        {
        return upperGuard.second;
        }
    
    /** Return the number of ground auxAtoms,
     *  appearing in the aggregate function, already
     *  true w.r.t. the current interpretation.
     */      
    int getLowerBound() const
        {
        return getAggregateFunctionPtr()->getLowerBound();
        }

    /** Return the number of ground auxAtoms,
     *  appearing in the aggregate function,
     *  which can become true 
     *  w.r.t. the current interpretation.
     */    
    int getUpperBound() const
        {
        return getAggregateFunctionPtr()->getUpperBound();    
        }

    int getMaxUndef() const
        {
        return getAggregateFunctionPtr()->getMaxUndef();
        }

    int getMaxTrue() const
        {
        return getAggregateFunctionPtr()->getMaxTrue();
        }

    void setMaxUndef(int max)
        {
        getAggregateFunctionPtr()->setMaxUndef(max);
        }

    void setMaxTrue(int max)
        {
        getAggregateFunctionPtr()->setMaxTrue(max);
        }

    int getMinUndef() const
        {
        return getAggregateFunctionPtr()->getMinUndef();
        }

    int getMinTrue() const
        {
        return getAggregateFunctionPtr()->getMinTrue();
        }

    void setMinUndef(int min)
        {
        getAggregateFunctionPtr()->setMinUndef(min);
        }

    void setMinTrue(int min)
        {
        getAggregateFunctionPtr()->setMinTrue(min);
        }

    void updateBounds(const TERM &term)
        {
        getAggregateFunctionPtr()->updateBounds(term);
        }

    void updateBounds(const TERM &t, const bool trueGatom)
        {
        getAggregateFunctionPtr()->updateBounds(t,trueGatom);
        }

    void finishBounds()
        {
        getAggregateFunctionPtr()->finishBounds();
        }

    /** Determine whether guards are violated.
     *  This function is used during propagation.
     */
    bool isViolated() const
        {
        assert( getLowerGuardAsVar().isInt() );
        assert( getUpperGuardAsVar().isInt() );

        int lg = getLowerGuardAsVar().getInt();
        int ug = getUpperGuardAsVar().getInt();

        return checkForViolation(lg,ug);
        }

    /** Determine whether guards are violated.
     *  This function is used during grounding.
     */
    bool isViolated(const TERM v2c[]) const
       {
       assert( ! getLowerGuardAsVar().isNull() );
       assert( ! getUpperGuardAsVar().isNull() );

       int lg;
       int ug;
       if( getLowerGuardAsVar().isInt() )
           lg = getLowerGuardAsVar().getInt();
       else
           lg = v2c[getLowerGuardAsVar().getVar()].getInt();

       if( getUpperGuard().first.isInt() )
           ug = getUpperGuardAsVar().getInt();
       else
           ug = v2c[getUpperGuardAsVar().getVar()].getInt();

       return checkForViolation(lg,ug);
       }

    /** Determine whether guards can never be violated.
     *  This function is used during propagation.
     */
    bool isSatisfied() const
        {
        assert( getLowerGuardAsVar().isInt() );
        assert( getUpperGuardAsVar().isInt() );

        int lg = getLowerGuardAsVar().getInt();
        int ug = getUpperGuardAsVar().getInt();

        return checkForSatisfaction(lg,ug);
        }

    /** Determine whether guards can never be violated.
     *  This function is used during grounding.
     */
    bool isSatisfied(const TERM v2c[]) const
        {
        int lg;
        int ug;
        if( getLowerGuard().first.isInt() )
            lg = getLowerGuardAsVar().getInt();
        else
            lg = v2c[getLowerGuardAsVar().getVar()].getInt();

        if( getUpperGuard().first.isInt() )
            ug = getUpperGuardAsVar().getInt();
        else
            ug = v2c[getUpperGuardAsVar().getVar()].getInt();

        return checkForSatisfaction(lg,ug);
        }

    int getLowerGuardAsInt() const
        {
        assert(lowerGuard.first.isInt());
        return lowerGuard.first.getInt();
        }

    int getUpperGuardAsInt() const
        {
        assert(upperGuard.first.isInt());
        return upperGuard.first.getInt();
        }
 
    TERM getLowerGuardAsVar() const
        {
        return lowerGuard.first; 
        }   
   
    TERM getUpperGuardAsVar() const 
        {
        return upperGuard.first;
        }   

    void setLowerGuard(const TERM &term)
        {
        lowerGuard.first = term;
        }

     void setUpperGuard(const TERM &term)
        {
        upperGuard.first = term;
        }
    };

inline ostream &operator<< (ostream &out, const GATOM &atom)
    {
    if( OptionPrintSATlike  &&  atom.isRegularAtom() )
        out << atom.getIndex();
    else
        out << atom.getTheRealOne();

    return out;
    }

inline int GATOM::operator!= (const GATOM& b) const
    {
    return b.index - index;
    }

inline bool GATOM::operator== (const GATOM& b) const
    {
    return ( *this != b ) == 0;
    }

inline bool GATOM::operator< (const GATOM& b) const
    {
    return ( *this != b ) < 0;
    }

inline GATOM::GATOM(const ATOM &atom2)
    { 
    pair<GATOMHASH::iterator,bool> p=
        h2iTable.insert(GATOMHASH::value_type(atom2,h2iTable.size()));

    index=(*p.first).second;
 
    // Is this a new GATOM?
    if( p.second )
        {
        i2hTable.push_back(p.first);

        if( frozen )
            InternalError("freeze violation");
        }
    }

inline void ATOM::setParams(const TERMS *params2)
    {
    if( params2 )
        {
        assert( ! data.params );

        data.params=new TERMS(*params2);
        }
    else
        data.params=0;

    }

inline void ATOM::setAggregate(const AGGREGATEATOM *aggregate2)
    {
    if( aggregate2 )
        {
        assert( ! data.aggregate );

        data.aggregate=new AGGREGATEATOM(*aggregate2);
        }
    else
        data.aggregate=0;
    }

inline ATOM::ATOM(const ATOM &atom2, const TERM v2c[])
    : p_index(atom2.p_index)
    {
    if( ! atom2.isAggregate() )
        {
        // We cannot use member initialization here.
        assert( isRegularAtom() );
        data.params=0;
        if( atom2.getParams() )
            {
            data.params=new TERMS;
            for( TERMS::const_iterator j=atom2.getParams()->begin();
                j != atom2.getParams()->end();
                j++ )
                {
                if ((*j).isVar() && ! (v2c[(*j).getVar()]).isNull() )
                    data.params->push_back(v2c[(*j).getVar()]);
                else
                    data.params->push_back(*j);
                }
            }
        }
    else
        {
        data.aggregate=0;
        data.aggregate=new AGGREGATEATOM(atom2.getAggregate(),v2c);
        }
    }

inline ATOM::ATOM(const AGGREGATEATOM &aggregate2)
    {
    pair<index_t,bool> res = Predicates.add(aggregate2.
        getAggregateType().getName(),
        0,
        PREDICATE_NAMES::typeAggregate);

    p_index = res.first;
    data.aggregate=0;
    setAggregate( &aggregate2 );
    }

inline ATOM& ATOM::operator= (const ATOM &atom2)
    {
    if( this != &atom2 )
        {
        if( isRegularAtom() )
            {
            if( data.params )
                delete data.params;
            }
        else
            {
            assert( isAggregate() );
            if ( data.aggregate )
                delete data.aggregate;
            }

        // Assign p_index now to avoid side effects
        // upon methods accessing the field "Predicates"
        // by using p_index itself as an entry.
        p_index=atom2.p_index;
 
        if( atom2.isRegularAtom() )
            {
            data.params=0;
            setParams(atom2.data.params);
            }
        else
            {
            assert( atom2.isAggregate() );
            data.aggregate=0;
            setAggregate(atom2.data.aggregate);
            }
        }
 
    return *this;
    }

inline ostream &operator<< (ostream &out, const AGGREGATEATOM &a)
    {
    // Print the lower guard.
    printGuard(a.getLowerGuard(),out);
    out << (a.isLowerGuardInclusive() ? " <= " : " < ");

    // Print the aggregate function.
    assert(a.getAggregateFunctionPtr());
    out << *a.getAggregateFunctionPtr();

    // Print the upper guard.
    out << (a.isUpperGuardInclusive() ? " <= " : " < ");
    printGuard(a.getUpperGuard(),out);

    return out;
    }

AGGREGATEFUNCTION *AGGREGATEFUNCTION::newAggregateFunction() const
    {
    switch ( aggregate_type.getType() )
        {
        case AGGREGATE_TYPE::Count:
            return new COUNTFUNCTION(*this);
        case AGGREGATE_TYPE::Max:
            return new MAXFUNCTION(*this);
        case AGGREGATE_TYPE::Min:
            return new MINFUNCTION(*this);
        case AGGREGATE_TYPE::Sum:
            return new SUMFUNCTION(*this);
        case AGGREGATE_TYPE::Times:
            return new TIMESFUNCTION(*this);
        default:
            assert(0);
            return 0;
        }
    }

AGGREGATEFUNCTION *AGGREGATEFUNCTION::newAggregateFunction(
    const TERM v2c[]) const
    {
    switch ( aggregate_type.getType() )
        {
        case AGGREGATE_TYPE::Count:
            return new COUNTFUNCTION(*this,v2c);
        case AGGREGATE_TYPE::Max:
            return new MAXFUNCTION(*this,v2c);
        case AGGREGATE_TYPE::Min:
            return new MINFUNCTION(*this,v2c);
        case AGGREGATE_TYPE::Sum:
            return new SUMFUNCTION(*this,v2c);
        case AGGREGATE_TYPE::Times:
            return new TIMESFUNCTION(*this,v2c);
        default:
            assert(0);
            return 0;
        }
    }

AGGREGATEFUNCTION *AGGREGATEFUNCTION::newAggregateFunction(
    const AGGREGATEPRED &aggrPred) const
    {
    switch ( aggregate_type.getType() )
        {
        case AGGREGATE_TYPE::Count:
            return new COUNTFUNCTION(aggrPred);
        case AGGREGATE_TYPE::Max:
            return new MAXFUNCTION(aggrPred);
        case AGGREGATE_TYPE::Min:
            return new MINFUNCTION(aggrPred);
        case AGGREGATE_TYPE::Sum:
            return new SUMFUNCTION(aggrPred);
        case AGGREGATE_TYPE::Times:
            return new TIMESFUNCTION(aggrPred);
        default:
            assert(0);
            return 0;
        }
    }

inline void AGGREGATEFUNCTION::addToAggregateSet(
    const AGGREGATESETVALUE &g) 
    {   
    if ( ! aggregateSet )
        aggregateSet = new AGGREGATESET();
    (void) aggregateSet->insert(g); 
    }
 
inline ATOM::~ATOM()
    {
    if( isAggregate() )
        {
        assert( data.aggregate );
        delete data.aggregate;
        }
    else
        {
        if( data.params )
            delete data.params;
        }
    }

inline int ATOM::compare(const AGGREGATEATOM &a) const
    {
    if( getAggregate() != a ) 
        return 1;
    
    return 0;
    } 

//////////////////////////////////////////////////////////////////////////////
typedef vector<GATOM> GATOMS;
//

#include "grule.h"

#include "program.h"

typedef TPROGRAM<RULE>            PROGRAM;

// Usually GRULES is used instead of GPROGRAM. However, some generic
// classes (DEPGRAPH, for example) rely on the addExit() and
// addRecursive() methods, which are provided by GPROGRAM, but not by
// GRULES, so we'll have to keep it for compatibility.

typedef TPROGRAM<GRULE>           GPROGRAM;

#endif

// Local Variables:
// mode: c++
// c-file-style: "dl"
// End:
