/////////////////////////////////////////////////////////////////////////////
// inheritance.C
//
// In this file most of the routines for the inheritance "frontend"
// are located.

// The global variables EDB, IDB, Constraints, RuleID.
#define WITH_GLOBAL_VARIABLES

#define OBJECT_PREC_NAME "__prec"

#include "dl.h"
#include <stdio.h>

/////////////////////////////////////////////////////////////////////////////
// compl_predname generates the complementary predicate name by just
// adding or deleting '-' at the beginning of the string.
//
// Note: The result should be deallocated when it is no longer used.
//
// @param   predname, the predicate name of which the complementary name
//          should be generated
// @return  the complementary predicate name
//
static const char* compl_predname(const char *predname)
    {
    char* complpred;

    // The predicate name should exist and be of non-zero length.
    assert(predname);
    assert(strlen(predname));

    if( predname[0] == '-' )
        {
        // If the first character is '-', the name is that of a negative
        // literal.
        // In this case, the result is obtained by just omitting the
        // first character.
        complpred = new char[strlen(predname)];
        strcpy(complpred,predname+1);
        }
    else
        {
        // If the first character is not '-', the name is that of a
        // positive literal.
        // In this case, add '-' to the front of the resultstring.
        complpred = new char[strlen(predname)+2];
        complpred[0] = '-';
        strcpy(complpred+1,predname);
        }

    return complpred;
    }

/////////////////////////////////////////////////////////////////////////////
// objectProjectionName() creates the name for a predicate where the
// object ID is projected out, given some predicate name.
//
// @param predname, the source name
//
// @return the generated name
//
static const char *objectProjectionName(const char *predname)
    {
    char *projname = new char[strlen(predname)+2];
    strcpy(projname,"@");
    strcat(projname,predname);

    return projname;
    }

/////////////////////////////////////////////////////////////////////////////
// ruleOverrideName() gives a name for representing that a rule is
// overridden.  To this end, we use the global variable RuleID to get
// a new number representing the current rule, which is coded into the
// name (could also do this by giving the id as an argument).
//
// @return the generated name
//
static const char *ruleOverrideName()
    {
    // FIXME: 64bit integers can have a maximum length of 20
    // Assuming this here.
    char *predname = new char[5+20];
    strcpy(predname,"__ovr");
    sprintf(predname+5,"%d",RuleID++);

    return predname;
    }

/////////////////////////////////////////////////////////////////////////////
// atomOverrideName() gives a name for representing that an atom is
// overridden or defeated, given the atom's predicate name.
//
// @return the generated name
//
static const char *atomOverrideName(const char *predname)
    {
    char *ovrname = new char[strlen(predname)+2];
    strcpy(ovrname,"^");
    strcat(ovrname,predname);

    return ovrname;
    }

/////////////////////////////////////////////////////////////////////////////
// Generate the precedence relation wrt to an object ID as facts.
// 
// @param o          the object ID
// @param oind       indices of object IDs which precede o
// @param ObjectIDs  all object IDs
//
static void generateOIDPrecedence(
    const OID &o,
    OIDindices *oind, 
    const OIDs &ObjectIDs )
    {
    if(oind)
        {

        // If preceding objects exist, generate a fact for each of
        // them defining the preference relation between o and the
        // object in question. Then the procedure is called
        // recursively with the direct preceders of each object in
        // oind. Termination is guaranteed because the precedence
        // relation must not be acyclic (forbidden by syntax).
        for( OIDindices::const_iterator i = oind->begin();
             i != oind->end();
             i++ )
            {
            // Create the parameterlist (current OID followed by
            // preceding OID).
            TERMS *id = new TERMS;
            (*id).push_back(o.getName());
            (*id).push_back(ObjectIDs[*i].getName());

            // Add the fact to EDB.
            EDB.add(ATOM(OBJECT_PREC_NAME,id,PREDICATE_NAMES::typeEDB));

            if( TraceLevel >= 3 )
                cdebug << ATOM(OBJECT_PREC_NAME,id,PREDICATE_NAMES::typeEDB)
                       << " added to EDB." << endl;

            delete id;

            // Call the procedure recursively on objects which precede
            // the current object, which already precedes o. This
            // expands the transitive reduction of the precedence
            // relation.
            generateOIDPrecedence(o, ObjectIDs[*i].getParentOIDindices(),
                                  ObjectIDs);
            }
        }
    }

/////////////////////////////////////////////////////////////////////////////
// Generate overriding rules for defeasible literals in an object ID.
// 
// @param o  the object ID in question
//
static void generateDefeasibility(const OID &o)
    {
    OID::DEFEASIBLES* d = o.getDefeasibles();

    // If o does not contain any defeasible literal, exit immediately.
    if( ! d )
        return;

    // Otherwise, create a rule for each defeasible literal.
    for( OID::DEFEASIBLES::const_iterator i = d->begin();
         i != d->end();
         i++ )
        {
        unsigned j = 0;
        DISJUNCTION head;
        unsigned thisArity = (*i).getParams() ? (*i).getParams()->size() : 0;

        // We precede a predicate name by '^' to express that the
        // predicate is defeated in the object specified in the first
        // argument.

        // The head of the rule defining this defeasibility predicate
        // for some predicate p is ^p(oid, _0, ..., _<arity>)

        const char *predname = atomOverrideName((*i).getPredName());

        // Construct the parameterlist: First add the object ID,
        // lateron add as many distinct variables as there were
        // original arguments.
        TERMS id;
        id.push_back(o.getName());
        for( j = 0; j < thisArity; j++ )
            id.push_back(TERM(j));

        head.add(ATOM(predname,&id,PREDICATE_NAMES::typeIDB));
        delete[] predname;

        // The body of the rule expresses: The complimentary literal
        // has been derived in an object of lower precedence.  We use
        // a new variable for representing the object of lower
        // precedence, which is the first argument of the
        // complementary literal. The rest of the parameters of this
        // literal is equal to the corresponding parameters of the
        // head atom.

        CONJUNCTION body;

        TERMS id2;
        assert( j == thisArity );
        id2.push_back(TERM(j));
        for( j = 0; j < thisArity; j++ )
            id2.push_back(TERM(j));

        // The complementary literal might need arity adjustment.
        pair<PREDICATE_NAMES::index_t,bool> result 
            = ATOM::Predicates.find(compl_predname((*i).getPredName()));

        // The complementary predicate must exist, otherwise it cannot
        // be defeasible.
        assert(result.second);

        // Adjust the arity, if necessary.
        if( id2.size() > ATOM::Predicates.getArity(result.first) )
            ATOM::Predicates.refineArity(result.first, id2.size());

        // Now we may safely generate the atom and add it to the body.

        body.add(ATOM(compl_predname((*i).getPredName()), &id2,
                      PREDICATE_NAMES::typeUndef));

        // The second atom in the body expresses that the object in
        // which the first has been derived is of lower precedence
        // than o.
        TERMS id3;
        assert( j == thisArity );
        id3.push_back(TERM(j));
        id3.push_back(o.getName());

        body.add(ATOM(OBJECT_PREC_NAME,&id3,PREDICATE_NAMES::typeUndef));

        IDB.push_back( RULE(&head,&body) );

        if( TraceLevel >= 3 )
            {
            cdebug << IDB.back() << " added to IDB." << endl;
            }
        }
    }

/////////////////////////////////////////////////////////////////////////////
// isDefeasible() decides whether a rule is defeasible in a certain object.
//
// @param o, an object
// @param r, a rule
//
// @result true if r is defeasible in o.
//
static bool isDefeasible(const OID &o, const RULE &r)
    {
    // A rule is defeasible in an object if all of its head atoms are
    // defeasible in that object.

    assert( r.hasHead() );

    const DISJUNCTION &head = r.getHead();

    for( DISJUNCTION::const_iterator i = head.begin();
         i != head.end();
         i++ )
        if( ! o.isDefeasible(*i) )
            return false;

    return true;
    }

/////////////////////////////////////////////////////////////////////////////
// addHeadParamsWithoutOIDs() appends all (non object id) parameters
// of the given head to the parameterlist given in the first argument.
//
// @param id, the parameterlist to be expanded
// @param head, the head from which the parameters should be extracted
//
static void addHeadParamsWithoutOIDs(TERMS *id, const DISJUNCTION &head)
    {
    for( DISJUNCTION::const_iterator i = head.begin();
         i != head.end();
         i++ )
        {
        if( (*i).getParams() )
            {
            if( (*i).occursComplementary() )
                {
                // Omit the first parameter (the object ID).
                id->insert(id->end(), (*i).getParams()->begin()+1, 
                           (*i).getParams()->end());
                }
            else
                {
                id->insert(id->end(), (*i).getParams()->begin(), 
                           (*i).getParams()->end());
                }
            }
        }
    }

/////////////////////////////////////////////////////////////////////////////
// addOIDParam() adds a given identifier at the place of the object ID.
//
// @param atom, the atom to be modified
// @param id, the identifier to be inserted
//
static void addOIDParam(ATOM &atom, const TERM &id)
    {
    TERMS* params = const_cast<TERMS *>(atom.getParams());

    if( ! params )
        {
        // If there were no parameters, create them for insertion.
        params = new TERMS;
        atom.setParams(params);
        delete params;
        // setParams copies the parameterlist.
        params = const_cast<TERMS *>(atom.getParams());
        }

    // Adjust the arity if necessary.
    if( params->size() == atom.getArity() )
        {
        atom.setArity(atom.getArity()+1);
        }

    // Insert the identifier at the front.
    params->insert(params->begin(),id);
    }

/////////////////////////////////////////////////////////////////////////////
// addOIDParamInHead() adds a given object id to those head atoms
// which also occur complementary.
//
// @param head, the head to be modified
// @param o, the object identifier to be inserted
//
static void addOIDParamInHead(DISJUNCTION *head, const OID &o)
    {
    assert(head);
    for( DISJUNCTION::iterator i=head->begin();
         i != head->end();
         i++ )
        {
        if( (*i).occursComplementary() )
            {
            addOIDParam(*i, o.getName());
            }
        }
    }

/////////////////////////////////////////////////////////////////////////////
// generateObjectProjectionRule() creates the rule which defines the
// atom where the object id is projected out.
//
// @param atom, the atom for which the rule should be generated
//
static void generateObjectProjectionRule(ATOM &atom)
    {
    const char *predname = objectProjectionName(atom.getPredName());

    // Create the head atom. Use the concrete atom's
    // arity (getParams()->size()) and not its global one
    // (getArity), as it is not known at this point
    // whether the global one has already been adjusted or
    // not (complementary occuring atoms' arities are
    // increased by one when their first occurrence is
    // encountered).
    TERMS* params;
    unsigned thisArity = atom.getParams() ? atom.getParams()->size() : 0;
    unsigned v = 0;

    if( thisArity )
        {
        params = new TERMS;
        for( ; v < thisArity; v++ )
            params->push_back(TERM(v));
        }
    else
        params = 0;

    DISJUNCTION headProj;
    headProj.add(ATOM(predname,params,PREDICATE_NAMES::typeIDB));

    delete[] predname;

    // For the body atom's parameters add a new variable
    // to the front of params..

    // Allocate the parameterlist if it was 0 before
    if( ! params )
        params = new TERMS;

    // Increase the predicate's arity if it has not been
    // adjusted yet. Use this atoms arity and compare it
    // with the global arity. If they are the same, we
    // have to increase the global arity by one.
    if( thisArity == atom.getArity() )
        {
        atom.setArity(atom.getArity()+1);
        }

    // Add the new variable at the beginning of the
    // parameterlist.
    params->insert(params->begin(),TERM(v++));

    CONJUNCTION bodyProj;

    bodyProj.add(ATOM(atom.getPredName(),params,PREDICATE_NAMES::typeUndef));

    delete params;

    IDB.push_back(RULE(&headProj,&bodyProj));

    if( TraceLevel >= 3 )
        cdebug << IDB.back() << " added to IDB." << endl;
    }

/////////////////////////////////////////////////////////////////////////////
// addOIDParamInBody() adds new variables (meaning derived in any
// object) to positive body literals which occur complementary,
// respectively replaces negative body literals occurring also
// complementary by a version which projects out the object identifier
// (otherwise negation would be unsafe).
//
// @param body, the body to be modified
// @param next_variable_number, the next available variable number in
//        this body.
//
static void addOIDParamInBody(
    CONJUNCTION* body,
    unsigned next_variable_number )
    {
    if( ! body )
        return;

    unsigned variable_counter = next_variable_number;

    // For positive body literals occurring complementary, simply add
    // a new variable.
    for( CONJUNCTION::iterator i=body->pos_begin();
         i != body->pos_end();
         i++ )
        {
        if( (*i).occursComplementary() )
            {
            addOIDParam(*i, TERM(variable_counter++));
            }
        }

    // For negative body literals occurring complementary, change the
    // name of the predicate and add a defining rule if it doesn't
    // exist yet.
    for( CONJUNCTION::iterator i=body->neg_begin();
         i != body->neg_end();
         i++ )
        {
        if( (*i).occursComplementary() )
            {
            // In this case, we must project out the object identifier
            // (the first argument) which is added to each literal
            // which occurs also complementary. We need a separate
            // predicate for that purpose.
            const char *predname = objectProjectionName((*i).getPredName());

            if( ! ATOM::Predicates.find(predname).second )
                {
                // If the projection predicate does not occur yet, add
                // its defining rule
                generateObjectProjectionRule(*i);
                }

            // Set the name of this literal to the one representing
            // the projection and set its arity to the global arity
            // (which has been increased by one for sure at this
            // point) of this literal minus one.
            (*i).setNameAndArity(predname, (*i).getArity() - 1);

            delete predname;
            }
        }
    }

/////////////////////////////////////////////////////////////////////////////
// addOIDParamInRule() adds a given object id to the head atoms
// and variables to the positive body atoms and rewrites the negative
//  body literals.
//
// @param o, the object identifier to be inserted
// @param rule, the rule to be modified
//
static void addOIDParamInRule(const OID &o, RULE &r)
    {
    assert( r.hasHead() );

    DISJUNCTION &head = r.getHeadForModification();
    CONJUNCTION *body = r.getBodyForModification();

    addOIDParamInHead(&head,o);

    if( body )
        {
        addOIDParamInBody(body,r.get_next_variable_number());
        }

    if( TraceLevel >= 3 )
        cdebug << "Modified rule: " << r << endl;
    }


/////////////////////////////////////////////////////////////////////////////
// addOverrideToDefeasibleRule() adds a literal meaning "unless the rule is
// overridden in o" to the body.
//
// @param o, the current object
// @param r, the rule to be modified
// @param predname, the predicate name of the literal to be added
//
static void addOverrideToDefeasibleRule(
    const OID &o,
    RULE &r,
    const char *predname )
    {
    CONJUNCTION *body = r.getBodyForModification();


    TERMS *id = new TERMS;
    id->push_back(o.getName());
    addHeadParamsWithoutOIDs(id,r.getHead());

    if( ! body )
        {
        body = new CONJUNCTION();
        r.setBody(body);
        delete body;
        body = r.getBodyForModification();        
        }

    body->add(LITERAL(true,ATOM(predname,id,PREDICATE_NAMES::typeUndef)));

    if( TraceLevel >= 3 )
        cdebug << "Rewritten rule: " << r << endl;
    }

/////////////////////////////////////////////////////////////////////////////
// addOverrideRule() adds a rule defining when a rule is overridden.
//
// @param o, the current object
// @param r, the rule for which the rule should be added
// @param predname, the predicate name of the literal to be added
//
static void addOverrideRule(
    const OID &o,
    RULE &r,
    const char *predname )
    {
    // Create a rule defining a predicate which represents the fact
    // that the given rule is overridden in an object.

    // A rule is overridden in object o, if all of its head atoms are
    // overridden in o.

    // We have to include all parameters (without the potentially
    // already added object IDs) of head atoms into the head atom of
    // this rule.

    TERMS *id = new TERMS;
    id->push_back(o.getName());
    addHeadParamsWithoutOIDs(id,r.getHead());

    // Create the head.

    DISJUNCTION head;
    head.add(ATOM(predname,id,PREDICATE_NAMES::typeIDB));
    
    delete id;

    // For the body we add for each head atom of the original rule an
    // atom which represents the fact that the original atom is
    // overridden in o.

    CONJUNCTION body;

    assert( r.hasHead() );

    for( DISJUNCTION::const_iterator i = r.getHead().begin();
         i != r.getHead().end();
         i++ )
        {
        const char *ovrname = atomOverrideName((*i).getPredName());
        body.add(ATOM(ovrname,(*i).getParams(),PREDICATE_NAMES::typeUndef));
        delete[] ovrname;
        }

    assert( body.begin() != body.end() );

    // Add the rule.

    IDB.push_back(RULE(&head,&body));

    if( TraceLevel >= 3 )
        {
        cdebug << IDB.back() << " added to IDB." << endl;
        }
    }

/////////////////////////////////////////////////////////////////////////////
// InheritanceRewritingRules() rewrites all rules in the given object
//
// @param o, the object in which rules should be rewritten
//
void InheritanceRewritingRules(const OID &o)
    {
    VATOMSET *edb = o.getEDB();
    RULES *idb = o.getIDB();

    // Rewrite and (potentially) relocate EDB facts.
    if( edb )
        {
        for( vector<ATOMSET>::iterator v = edb->v.begin();
             v != edb->v.end();
             v++ )
            for( ATOMSET::iterator i = v->begin();
                 i != v->end();
                 i++ )
                {
                if( o.isDefeasible(*i) )
                    {
                    // If the atom is defeasible we would have to make an
                    // IDB rule out of it (adding "not overridden in o" to
                    // the body). This is currently not possible.
                    cerr << "An EDB fact (" << *i << ") would have to be "
                            "rewritten. Currently, we do not handle this "
                            "automatically. Please rewrite your program, "
                            "such that " << *i << " belongs to the IDB." 
                         << endl;
                    }
                else
                    {
                    // If it is not defeasible, simply add the object ID
                    // if it occurs complementary and do the relocation
                    // (if necessary).
                    if( (*i).occursComplementary() )
                        {
                        addOIDParam(const_cast<ATOM&>(*i),o.getName());
                        }
                    if( o.getEDB() != &EDB )
                        {
                        // Just relocate if the EDB is not equal to the
                        // global one.
                        EDB.add(*i);
                        if( TraceLevel >= 3 )
                            {
                            cdebug << "Relocated " << *i << " from " << o 
                                   << "'s EDB to the global EDB." << endl;
                            }
                        }
                    }
                }

        }

    // Rewrite and (potentially) relocate the IDB rules.
    if( idb )
        {
        for( RULES::iterator i = idb->begin();
             i != idb->end();
             i++ )
            {
            // NOTE: isDefeasible() must be called before any OIDs are
            // added, since atoms are compared and the comparison
            // would fail because of the additional OID which was not
            // there before.
            if( isDefeasible(o,*i) )
                {
                // First we add OIDs resp. variables.
                addOIDParamInRule(o,*i);

                // Then we add a rule for determining when this rule
                // is overridden and a literal to this rule for
                // specifying that is only applicable if it is not
                // ovverridden.
                const char *predname = ruleOverrideName();
                addOverrideToDefeasibleRule(o, *i, predname);
                addOverrideRule(o, *i, predname);
                }
            else
                addOIDParamInRule(o,*i);

            if( o.getIDB() != &IDB )
                {
                // Relocate the rule only if the object's IDB is not
                // equal to the global one.
                IDB.push_back(*i);
                if( TraceLevel >= 3 )
                    {
                    cdebug << "Relocated " << *i << " from " << o 
                           << "'s IDB to the global IDB." << endl;
                    }
                }
            }
        }
    }

/////////////////////////////////////////////////////////////////////////////
// InheritanceRewritingConstraints() adds an argument to all positive
// literals which occur complementary resp. replaces negative literals
// which occur complementary by literals where the additional object
// id is projected out.
//
static void InheritanceRewritingConstraints()
    {
    for( CONSTRAINTS::iterator i = Constraints.begin();
         i != Constraints.end();
         i++ )
        {
        addOIDParamInBody(&(*i),(*i).get_next_variable_number());
        if( TraceLevel >= 3 )
            {
            cdebug << "Rewritten constraint: " << *i << endl;
            }
        }
    }

/////////////////////////////////////////////////////////////////////////////
// InheritanceRewriting() performs all of the rewriting required for the
// inheritance frontend.
//
// @params ObjectIDs, the existing objects
//
void InheritanceRewriting(const OIDs &ObjectIDs)
    {
    if( TraceLevel >= 3 )
        cdebug << "Inheritance Rewriting." << endl;

    for( OIDs::const_iterator i = ObjectIDs.begin();
         i != ObjectIDs.end();
         i++ )
        {
        // Generate the object precedence.
        generateOIDPrecedence(*i,(*i).getParentOIDindices(), ObjectIDs);

        // Generate the rules and constraints for the defeasible literals.
        generateDefeasibility(*i);

        // Rewrite the rules in the current object and add them to the
        // global rules.
        InheritanceRewritingRules(*i);
        }

    // Rewrite the constraints (independent from objects).
    InheritanceRewritingConstraints();

    if( TraceLevel >= 3 )
        cdebug << "Inheritance Rewriting finished." << endl;
    }

// Local Variables:
// c-file-style: "dl"
// End:
