//////////////////////////////////////////////////////////////////////////////
// rule.h

#ifndef RULE_H
#define RULE_H

//////////////////////////////////////////////////////////////////////////////
template <class T>
class TDISJUNCTION
//////////////////////////////////////////////////////////////////////////////
    {
    vector<T> items;

public:
    typedef typename vector<T>::iterator iterator;
    typedef typename vector<T>::const_iterator const_iterator;

    const_iterator begin() const { return items.begin(); }
    iterator       begin()       { return items.begin(); }
    const_iterator end()   const { return items.end();   }
    iterator       end()         { return items.end();   }

    TDISJUNCTION() : items()
        {
        }

    TDISJUNCTION(const TDISJUNCTION& disj)
        : items(disj.begin(),disj.end())
        {
        }

    TDISJUNCTION(const_iterator begin2, const_iterator end2)
        : items(begin2,end2)
        {
        }

    void operator=(const TDISJUNCTION&)
        {
        assert( 0 );
        }


    size_t size() const
        {
        return end()-begin();
        }


    void add(const T& item)
        {
        items.push_back(item);
        }

    const_iterator find(const T& item) const
        {
        return std::find(begin(),end(),item);
        }

    void addUnique(const T& item)
        {
        if( find(item) == end() )
            add(item);
        }

    iterator remove(iterator i)
        {
        return items.erase(i);
        }
    };


//////////////////////////////////////////////////////////////////////////////
template <class T>
class TCONJUNCTION
//////////////////////////////////////////////////////////////////////////////
    {
    typedef vector<T> LITERALS;
    
    LITERALS literals;
    typename LITERALS::iterator separator;

    void compute_separator()
        {
//cdebug << "Hello world!" << endl;
        for(separator=literals.begin(); separator != literals.end(); separator++)
            if( (*separator).isNegative() )
                break;
        }

public:
    typedef typename LITERALS::iterator iterator;
    typedef typename LITERALS::const_iterator const_iterator;

    const_iterator begin()     const { return literals.begin(); }
    iterator       begin()           { return literals.begin(); }
    const_iterator end()       const { return literals.end();   }
    iterator       end()             { return literals.end();   }
    const_iterator pos_begin() const { return literals.begin(); }
    iterator       pos_begin()       { return literals.begin(); }
    const_iterator pos_end()   const { return separator; }
    iterator       pos_end()         { return separator; }
    const_iterator neg_begin() const { return separator; }
    iterator       neg_begin()       { return separator; }
    const_iterator neg_end()   const { return literals.end(); }
    iterator       neg_end()         { return literals.end(); }

    TCONJUNCTION() : literals()
        {
        separator=literals.end();
        }

    TCONJUNCTION(const TCONJUNCTION& c)
        : literals(c.literals)
        {
        compute_separator();
        }

    TCONJUNCTION(const_iterator begin2, const_iterator end2)
        : literals(begin2,end2)
        {
        compute_separator();
        }

    TCONJUNCTION& operator=(const TCONJUNCTION& c)
        {
        if( this != &c )
            {
            this->literals=c.literals;
            compute_separator();
            }

        return *this;
        }

    // This method is *extremely* expensive, and its
    // use should be avoided whereever is possible.
    int compare(const TCONJUNCTION &c) const
        {
        int diff = size() - c.size();
        if (diff != 0)
            return diff;
 
        // Sort the literals to make sure that they are equal.
        LITERALS lits1 = literals;
        size_t i = pos_end() - pos_begin(); 

        sort(lits1.begin(),lits1.begin() +i);
        sort(lits1.begin() +i,lits1.begin() +size());

        LITERALS lits2 = c.literals;
        i = c.pos_end() - c.pos_begin();

        // Sort the literals to make sure that they are equal.
        sort(lits2.begin(),lits2.begin() +i);
        sort(lits2.begin() +i,lits2.begin() +c.size());

        for(size_t j=0; j<size(); j++)
             {
             if (lits1[j] != lits2[j])
                 return (lits1[j] != lits2[j]);
             }
 
        return 0;
        }

    size_t size() const
        {
        return end()-begin();
        }

    ////////////////////////////////////////////////////////////
    // FIXME: currently used in Magic Sets rewriting because there it
    // is necessary to generate a particular ordering of the literals
    // which would violate the positive/negative body separation. It
    // might be useful to have a specific class for purposes like this
    // in the future.
    void insert(iterator pos, const T &literal)
        {
        literals.insert(pos,literal);
        }
        
    void add(const T& literal)
        {
        assert( literals.begin() <= separator  &&  separator <= literals.end() );

        size_t old_size=literals.capacity();

        if( literal.isNegative() )
            literals.push_back(literal);
        else
            literals.insert(separator,literal);

        if( literals.capacity() != old_size )
            compute_separator();
        else if( ! literal.isNegative() )
            separator++;

        assert( literals.begin() <= separator  &&  separator <= literals.end() );
        }

    const_iterator find(const T &literal) const
        {
        const_iterator found;

        if( literal.isNegative() )
            {
            found = std::find(neg_begin(),neg_end(),literal);
            }
        else
            {
            // If nothing is found within the positive literals return end() 
            // to follow the normal conventions of find().
            found = std::find(pos_begin(),pos_end(),literal);
            if( found == pos_end() )
                found = end();
            }

        return found;
        }

    void addUnique(const T& literal)
        {
        if( find(literal) == end() )
            {
            add(literal);
            }        
        }

    // Add a unique literal in ascending order.
    void addOrderedUnique(const T &literal)
        {
        if(find(literal) == end())
            {
            assert(literals.begin() <= separator
                && separator <= literals.end());
 
            size_t old_size = literals.capacity();
            iterator i;
 
            // Perform a binary search to find the right entry for literal.
            if(literal.isNegative())
                i = lower_bound(neg_begin(),neg_end(),literal);
            else
                i = lower_bound(pos_begin(),pos_end(),literal);
 
            // Insert literal before i.
            literals.insert(i,literal);
 
            if( literals.capacity() != old_size )
                compute_separator();
            else if( ! literal.isNegative() )
                separator++;
 
            assert(literals.begin() <= separator 
                && separator <= literals.end());
            }// if(find == end())
        }

    iterator remove(iterator i)
        {
        assert( literals.begin() <= separator  &&  separator <= literals.end() );
        
        if( ! (*i).isNegative() )
            separator--;
            
        iterator return_value=literals.erase(i);
        
        assert( literals.begin() <= separator  &&  separator <= literals.end() );

        return return_value;
        }

    void clear()
        {
        literals.clear();
        compute_separator();
        }

    bool isSafe() const;	


    const unsigned get_next_variable_number() const
        {
        unsigned next_var = 0;

        for( typename TCONJUNCTION<T>::const_iterator i = begin();
             i != end();
             i++ )
            {
            if( (*i).getParams() )
                for( TERMS::const_iterator j = (*i).getParams()->begin();
                     j != (*i).getParams()->end();
                     j++ )
                    if( (*j).isVar() && (*j).getVar() + 1 > next_var )
                        next_var = (*j).getVar() + 1;
            }

        return next_var;
        }

    };

#if 0
template <class T>
    void TCONJUNCTION<T>::compute_separator()
        {
//cdebug << "Hello world!" << endl;
        for(separator=literals.begin(); separator != literals.end(); separator++)
            if( (*separator).isNegative() )
                break;
        }
#endif

//////////////////////////////////////////////////////////////////////////////
template <class T>
inline ostream& operator<< (ostream& out, const TCONJUNCTION<T>& conj)
//
    {
    print_list(out,conj,", ");

    return out;
    }


//////////////////////////////////////////////////////////////////////////////
template <class T1, class T2>
class TRULE;

//////////////////////////////////////////////////////////////////////////////
template <>
class TRULE<ATOM, LITERAL>
    {
    TDISJUNCTION<ATOM> *head;
    TCONJUNCTION<LITERAL> *body;

    void init(const TDISJUNCTION<ATOM> *h,
	      const TCONJUNCTION<LITERAL> *b )
        {
        if( h )
            head=new TDISJUNCTION<ATOM>(*h);
        else
            head=0;

        if( b )
            body=new TCONJUNCTION<LITERAL>(*b);
        else
            body=0;
        }

    void erase()
        {
        if( head )
            delete head;
        if( body )
            delete body;
        }

public:
    TRULE()
        {
        assert( 0 );
        }

    TRULE(const TRULE &rule2)
        {
        init(rule2.head,rule2.body);
        }

    TRULE(const TDISJUNCTION<ATOM> *h, const TCONJUNCTION<LITERAL> *b) 
        {
        init(h,b);
        }

    ~TRULE()
        {
        erase();
        }

    TRULE& operator= (const TRULE& rule2)
        {
        if( this != &rule2 )
            {
            erase();
            init(rule2.head,rule2.body);
            }
        
        return *this;
        }

    void removeBody()
        {
        if( body )
            {
            delete body;
            body=0;
            }
        }

    void setBody(TCONJUNCTION<LITERAL> *b)
        {
        if( b )
            body = new TCONJUNCTION<LITERAL>(*b);
        else
            body = 0;
        }

    bool hasHead() const
        {
        return ( head != 0 );
        }

    bool hasBody() const
        {
        return ( body != 0 );
        }

    const TDISJUNCTION<ATOM> &getHead() const
        {
        assert( head );
        return *head;
        }

    TDISJUNCTION<ATOM> &getHeadForModification() const
        {
        assert( head );
        return *head;
        }

    const TCONJUNCTION<LITERAL>* getBody() const
        {
        return body;
        }

    TCONJUNCTION<LITERAL> *getBodyForModification()
        {
        return body;
        }
//FIXME: duplicate!
    TCONJUNCTION<LITERAL> *getBodyForModification() const
        {
        return body;
        }

    bool isDisjunctive() const
        { 
        return head->size() > 1;
        }

    bool isSafe() const;

    const unsigned get_next_variable_number() const
        {
        unsigned next_var = 0;

#ifndef NDEBUG
        // This code is used for asserting safety,
        // i.e. that no variable occurs in 
        // the head which does not occur in the body.
        unsigned next_var_head = 0;
        assert( head);
        for( TDISJUNCTION<ATOM>::const_iterator i = head->begin();
             i != head->end();
             i++ )
            {
            if( (*i).getParams() )
                {
                for( TERMS::const_iterator j = (*i).getParams()->begin();
                     j != (*i).getParams()->end();
                     j++ )
                    // If var+1 is bigger than next_var, then next_var
                    // already exists and is set to the next available
                    // value (var+1).
                    if( (*j).isVar() && (*j).getVar() + 1 > next_var )
                        next_var_head = (*j).getVar() + 1;
                }
            }
#endif

        if( body )
            next_var = body->get_next_variable_number();

#ifndef NDEBUG
        assert(next_var >= next_var_head );
#endif
        return next_var;
        }
    };


//////////////////////////////////////////////////////////////////////////////
template <class T1, class T2>
inline ostream& operator<< (ostream& out, const TRULE<T1,T2>& rule)
//
    {
    if( rule.hasHead() )
        {
        print_list(out,rule.getHead()," v ");
        }

    if( rule.hasBody() )
        {
        out << " :- "
            << *(rule.getBody());
        }

    out << ".";

    return out;
    }


class INTERPRET;        // Required by one of the constructors of TCONSTRAINT.

//////////////////////////////////////////////////////////////////////////////
template <class T>
class TCONSTRAINT : public TCONJUNCTION<T>
//////////////////////////////////////////////////////////////////////////////
    {

private:
    bool internal;

public:

    TCONSTRAINT(bool i=false) 
	: TCONJUNCTION<T>(),
          internal(i)
	{
	}
        
    TCONSTRAINT( typename TCONJUNCTION<T>::const_iterator b,
                 typename TCONJUNCTION<T>::const_iterator e,
                 bool intern ) 
	: TCONJUNCTION<T>(b,e),
        internal(intern)
	{
	}

    TCONSTRAINT(const TCONSTRAINT<LITERAL>& c, 
		const TERM* j, 
		INTERPRET* I,
		bool intern)
	: TCONJUNCTION<T>(c, j, I),
          internal(intern)
	{
	}

    TCONSTRAINT(const TCONJUNCTION<T>& c, bool intern)
	: TCONJUNCTION<T>(c),
          internal(intern)
	{
	}

 TCONSTRAINT(const TCONSTRAINT<LITERAL>& c)
        : TCONJUNCTION<T>(c), 
          internal(c.internal)
	{
	}

// Use default copy constructor (calls base class copy constructor and
// copies memberwise).
    
// Use default assignment operator (call base class assignment
// operator and call assignment operator for each member)
	    
    bool isInternal() const
	{
	return internal;
	}

    };

template <class T>
inline ostream& operator<< (ostream& out, const TCONSTRAINT<T>& constr)
//
    {
    out << ":- ";
    print_list(out,constr,", ");
    out << ".";

    return out;
    }

typedef pair<TERM, TERM> WEIGHTS;

//////////////////////////////////////////////////////////////////////////////
template <class T>
class TWEAKCONSTRAINT : public TCONSTRAINT<T>
//////////////////////////////////////////////////////////////////////////////
    {

private:
    WEIGHTS weights;
public:
    // TERM(1,0) is used as a default value for both weight and level.

    TWEAKCONSTRAINT(TERM weight = TERM(1,0), TERM level = TERM(1,0))
        :TCONSTRAINT<T>(), weights(weight, level)
	{
        }
        
    TWEAKCONSTRAINT(typename TCONJUNCTION<T>::const_iterator b,
                    typename TCONJUNCTION<T>::const_iterator e, 
                    TERM weight = TERM(1,0), 
                    TERM level = TERM(1,0)) 
	: TCONSTRAINT<T>(b, e, false),weights(weight, level)
	{
	}

    TWEAKCONSTRAINT(const TCONJUNCTION<T>& c,
                    TERM weight = TERM(1,0), 
                    TERM level = TERM(1,0))
	: TCONSTRAINT<T>(c,false),weights(weight, level)
	{
	}

    void setWeights(const TERM &w1, const TERM &w2)
        {
        assert( w1.isInt() );
        assert( w2.isInt() );
        weights.first = w1;
        weights.second = w2;
        }

    // Get the information of weight and level.
    const WEIGHTS& getWeights() const
        {
	return weights;
	}

    // Get the information of weight.
    const unsigned int getWeight() const
        {
	return weights.first.getInt();
	}

    // Get the information of level.
    const unsigned int getLevel() const
        {
	return weights.second.getInt();
	}

    inline ostream& printWeights(ostream&) const;

    bool isSafe() const;
    };

template <class T>
inline ostream& TWEAKCONSTRAINT<T>::printWeights(ostream& out) const
    {
    // Print weight.
    out << "[";
    if (getWeights().first.isVar())
        out << getWeights().first;
    else if (getWeights().first.isInt())
        if( PTraceLevel >= 1 )
            out << (getWeights().first.getInt()/SCALEFACT)
                << "/#";
        else
            out << (getWeights().first.getInt()/SCALEFACT);
    else
        assert(0);

    out << ":";

    // Print level.
    if (getWeights().second.isVar())
	out << getWeights().second;
    else if (getWeights().second.isInt())
        if( PTraceLevel >= 1 )
            out << (getWeights().second.getInt())
                << "/#";
        else
            out << (getWeights().second.getInt());
    else
        assert(0);

    out << "]";

    return out;
    }

template <class T>
inline ostream& operator<< (ostream& out, const TWEAKCONSTRAINT<T>& wconstr)
//
    {
    out << ":~ ";
    print_list(out,wconstr,", ");
    out << ". ";
    wconstr.printWeights(out);
    return out;
    }

bool safeWeights(const WEIGHTS&, const TCONJUNCTION<LITERAL>&);
#endif

// Local Variables:
// mode: c++
// c-file-style: "dl"
// End:
