/*================================================================
 * sequencer device handler
 *
 * Copyright (C) 1996-1998 Takashi Iwai
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *================================================================*/

#include <stdio.h>
#include <math.h>
#include <unistd.h>
#include <string.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <math.h>
#include <fcntl.h>
#ifdef linux
#include <sys/soundcard.h>
#include <linux/awe_voice.h>
#elif defined(__FreeBSD__)
#include <machine/soundcard.h>
#include <awe_voice.h>
#endif
#include "config.h"
#include "seq.h"
#include "midievent.h"
#include "channel.h"
#include "controls.h"

/*----------------------------------------------------------------
 * common variables
 *----------------------------------------------------------------*/

int seqfd;	/* sequencer file descpriptor */
int awe_dev;	/* awedrv device number */

#define SEQBUF_LEN	2048
#define SEQUENCER_DEV	"/dev/sequencer"

/* sequencer event buffer;
 * allocate enough size than the defined size to avoid dropping events
 */
unsigned char _seqbuf[SEQBUF_LEN * 2];
int _seqbuflen = SEQBUF_LEN;
int _seqbufptr = 0;

static int echo_sync = 0;	/* use echo back for timer sync */
static int writeptr = 0;	/* pending position */
static int seqblock = 0;	/* block if seqbuf is full */
static int pending = 0;		/* events are pending? */
static int opened = 0;		/* seq device is opened? */
static int ticks;		/* timer resolution in Hz */


/* check the seq device is writable now */
static int seq_writable(int wait)
{
	fd_set fds;
	struct timeval tv;
	int rc;

	FD_ZERO(&fds);
	FD_SET(seqfd, &fds);
	if (wait < 0) {
		/* no timeout */
		rc = select(seqfd + 1, NULL, &fds, NULL, NULL);
	} else {
		tv.tv_sec = 0;
		tv.tv_usec = wait;
		rc = select(seqfd + 1, NULL, &fds, NULL, &tv);
	}
	if (rc > 0 && FD_ISSET(seqfd, &fds))
		return 1;

	return 0;
}

static void do_write(int do_anyway)
{
	int i, size;
	size = _seqbufptr - writeptr;
	if ((i = write(seqfd, _seqbuf + writeptr, size)) < 0) {
		if (! do_anyway) {
			fprintf(stderr, "can't write to sequencer.");
			perror("errno");
			exit(1);
		}
	} else if (i < size) {
		writeptr += i;
		pending = 1;
	} else {
		_seqbufptr = 0;
		writeptr = 0;
		pending = 0;
	}
}

/* dump the buffer to seq device.
 * if the device is blocked, set pending flag and returns.
 * if writable, dump as much as possible
 */
void seqbuf_dump()
{
	if (_seqbufptr) {
		if (seqblock)
			do_write(0);
		else {
			if (_seqbufptr >= SEQBUF_LEN*2 - 16) {
				/* the reserved buffer is almost full;
				   write all data as emergency */
				while (pending)
					do_write(1);
			} else if (! seq_writable(0)) {
				pending = 1;
			} else
				do_write(0);
		}
	}
}

/* set blocking mode */
void seq_blocking_mode(int mode)
{
	seqblock = mode;
}

/* the buffered events are pending? */
int seq_pending(void)
{
	if (pending)
		seqbuf_dump();
	return pending;
}

/* sequencer device is already opened? */
int seq_opened(void)
{
	return opened;
}

/* initialize sequencer device */
int seq_init(char *device, int index, int do_echoback)
{
	int i;
	int nrsynths;
	struct synth_info card_info;

	if (opened)
		return 0;
	if (device == NULL)
		device = SEQUENCER_DEV;

	echo_sync = 0;
	/* use echo back synchronization mechanism */
	if (do_echoback && (seqfd = open(device, O_RDWR)) >= 0)
		echo_sync = 1;
	else if ((seqfd = open(device, O_WRONLY)) < 0) {
		if (ctl == NULL) {
			perror("sequencer device");
			exit(1);
		}
		return 1;
	}

	/* check AWE driver */
	if (ioctl(seqfd, SNDCTL_SEQ_NRSYNTHS, &nrsynths) == -1) {
		fprintf(stderr, "there is no soundcard\n");
		if (ctl) ctl->close();
		exit(1);
	}
	if (index >= 0) {
		card_info.device = index;
		if (ioctl(seqfd, SNDCTL_SYNTH_INFO, &card_info) < 0 ||
		    card_info.synth_type != SYNTH_TYPE_SAMPLE ||
		    card_info.synth_subtype != SAMPLE_TYPE_AWE32) {
			fprintf(stderr, "invalid soundcard (device = %s, index = %d)\n", device, index);
			if (ctl) ctl->close();
			exit(1);
		}
		awe_dev = index;
	} else {
		/* auto probe */
		awe_dev = -1;
		for (i = 0; i < nrsynths; i++) {
			card_info.device = i;
			if (ioctl(seqfd, SNDCTL_SYNTH_INFO, &card_info) == -1) {
				fprintf(stderr, "cannot get info on soundcard\n");
				perror(SEQUENCER_DEV);
				if (ctl) ctl->close();
				exit(1);
			}
			if (card_info.synth_type == SYNTH_TYPE_SAMPLE
			    && card_info.synth_subtype == SAMPLE_TYPE_AWE32)
				awe_dev = i;
		}
		if (awe_dev < 0) {
			fprintf(stderr, "No AWE synth device is found\n");
			if (ctl) ctl->close();
			exit(1);
		}
	}

	ticks = 0; /* get the default ticks */
	if (ioctl(seqfd, SNDCTL_SEQ_CTRLRATE, &ticks) < 0) {
		fprintf(stderr, "can't get ticks from sequencer\n");
		if (ctl) ctl->close();
		exit(1);
	}
	if (ticks < 100) {
		fprintf(stderr, "invalid ticks value\n");
		if (ctl) ctl->close();
		exit(1);
	}

	/* set channel mode on */
	AWE_SET_CHANNEL_MODE(awe_dev, AWE_PLAY_MULTI);
	/*_TIMER_EVENT(TMR_WAIT_REL, 1);*/
	seqbuf_dump();

	opened = 1;
	return 0;
}

/* close the seq device */
void seq_end()
{
	_seqbufptr = writeptr = 0;
	pending = 0;
	if (opened)
		close(seqfd);
	opened = 0;
}

/* return system ticks */
int seq_system_ticks(void)
{
	return ticks;
}

/* get the rest memory size */
int seq_mem_avail(void)
{
	int mem_avail = awe_dev;
	ioctl(seqfd, SNDCTL_SYNTH_MEMAVL, &mem_avail);
	return mem_avail;
}

/*----------------------------------------------------------------
 * timer events
 *----------------------------------------------------------------*/

static int start_csec;			/* base offset of timer */
static int cur_csec;			/* current time */
static int cur_amount;			/* current time */
static struct timeval start_time;	/* timeval at start_csec */
static int timer_started;		/* echo back is called? */

/* set timer offset */
static void seq_set_timer(int csec)
{
	gettimeofday(&start_time, NULL);
	start_csec = csec;
}

int seq_timer_started(void)
{
	return timer_started;
}

/* send timer clear event */
void seq_clear(int csec)
{
	SEQ_START_TIMER();
	seq_echo(csec);
	timer_started = !echo_sync;
	seq_set_timer(csec);
	cur_csec = csec;
	cur_amount = 0;
}

/* get the current time from timer */
int seq_curtime()
{
	int d1, d2;
	struct timeval now_time;

	gettimeofday(&now_time, NULL);
	d1 = now_time.tv_sec - start_time.tv_sec;
	d2 = now_time.tv_usec - start_time.tv_usec;
	while (d2 < 0) {
		d2 += 1000000;
		d1--;
	}
	d2 /= 10000;
	return (d2 + d1 * 100 + start_csec);
}

/* send wait event for time centi seconds; not actually sleeping */
void seq_wait(int time)
{
	cur_csec += time;
	if (ticks != 100) {
		time *= ticks;
		cur_amount += time % 100;
		time /= 100;
		if (cur_amount > 100) {
			time += cur_amount / 100;
			cur_amount %= 100;
		}
	}
	SEQ_DELTA_TIME(time);
}

/* get the last time on seq buffer */
int seq_lasttime()
{
	return cur_csec;
}
	

/* sleep for csec, or get an echo back from seq device */
int seq_sleep(int csec)
{
	if (echo_sync) {
		/* check echo back from seq device */
		int rc;
		fd_set fds;
		struct timeval tv;

		tv.tv_sec = 0;
		tv.tv_usec = csec * 10000;
		FD_ZERO(&fds);
		FD_SET(seqfd, &fds);
		rc = select(seqfd + 1, &fds, NULL, NULL, &tv);
		if (rc > 0 && FD_ISSET(seqfd, &fds)) {
			int msg;
			if (read(seqfd, &msg, 4) != 4 ||
			    (msg & 0xff) != SEQ_ECHO)
				return seq_curtime();
			msg >>= 8;
			timer_started = 1;
			seq_set_timer(msg);
			/*fprintf(stderr, "catch %d\n", msg);*/
			return msg;
		}
	} else {
		/* just sleep for a while */
		usleep(csec * 10000);
	}

	return seq_curtime();
}


/* send echo back time to the seq device */
void seq_echo(int csec)
{
	if (echo_sync) {
		SEQ_ECHO_BACK(csec);
	}
}


/* sync to the last event */
int seq_wait_time(int endcs)
{
	seqbuf_dump();
	ioctl(seqfd, SNDCTL_SEQ_SYNC);
	if (!seqblock && timer_started) {
		int csec = seq_curtime();
		while (csec < endcs)
			csec = seq_sleep(endcs - csec);
	}
	return endcs;
}

/*----------------------------------------------------------------
 * terminate sounds immediately via ioctl.
 * all events in the seq device and buffer are cleared.
 *----------------------------------------------------------------*/

void seq_terminate_all(void)
{
	ioctl(seqfd, SNDCTL_SEQ_RESET);
	_seqbufptr = writeptr = 0;
	pending = 0;
}	

/*----------------------------------------------------------------
 * normal play event handlers; put on seq buffer
 *----------------------------------------------------------------*/

void seq_set_bank(int v, int bank)
{
	SEQ_CONTROL(awe_dev, v, CTL_BANK_SELECT, bank);
}

void seq_note_off_all(void)
{
	/*SEQ_CONTROL(awe_dev, 0, 123, 0);*/
	AWE_NOTEOFF_ALL(awe_dev);
}

void seq_sound_off_all(void)
{
	/*SEQ_CONTROL(awe_dev, 0, 120, 0);*/
	AWE_RELEASE_ALL(awe_dev);
}

void seq_control(int v, int type, int val)
{
	SEQ_CONTROL(awe_dev, v, type, val);
}

void seq_set_program(int v, int pgm)
{
	SEQ_SET_PATCH(awe_dev, v, pgm);
}

void seq_start_note(int v, int note, int vel)
{
	SEQ_START_NOTE(awe_dev, v, note, vel);
}

void seq_stop_note(int v, int note, int vel)
{
	SEQ_STOP_NOTE(awe_dev, v, note, vel);
}

/* val : 128 = 1 semitone */
void seq_pitchsense(int v, int val)
{
	val = val * 100 / 128;
	SEQ_BENDER_RANGE(awe_dev, v, val);
}

void seq_pitchbend(int v, int val)
{
	SEQ_BENDER(awe_dev, v, val);
}

void seq_aftertouch(int v, int note, int vel)
{
	AWE_KEY_PRESSURE(awe_dev, v, note, vel);
}

void seq_chan_pressure(int v, int val)
{
	AWE_CHN_PRESSURE(awe_dev, v, val);
}

/* tuning:
 *   coarse = -8192 to 8192 (100 cent per 128)
 *   fine = -8192 to 8192 (max=100cent)
 */
void seq_detune(int v, int coarse, int fine)
{
	/* 4096 = 1200 cents in AWE parameter */
	int val;
	val = coarse * 4096 / (12 * 128);
	val += fine / 24;
	if (val) {
		AWE_SEND_EFFECT(awe_dev, v, AWE_FX_INIT_PITCH, val);
	} else {
		AWE_UNSET_EFFECT(awe_dev, v, AWE_FX_INIT_PITCH);
	}
}

/* set effect */
void seq_send_effect(int v, int type, int val)
{
	AWE_SEND_EFFECT(awe_dev, v, type, val);
}

/* send effect */
void seq_add_effect(int v, int type, int val)
{
	AWE_ADD_EFFECT(awe_dev, v, type, val);
}

/* send soft pedal on/off: emulation by effect control */
void seq_soft_pedal(int v, int val)
{
	if (val == 127) {
		AWE_ADD_EFFECT(awe_dev, v, AWE_FX_CUTOFF, -160);
	} else {
		AWE_UNSET_EFFECT(awe_dev, v, AWE_FX_CUTOFF);
	}
}

/*----------------------------------------------------------------
 * send the control events;  put on seq buffer
 *----------------------------------------------------------------*/

void seq_reset_channel(int v)
{
	AWE_RESET_CHANNEL(awe_dev, v);
}

void seq_reset_control(int v)
{
	AWE_RESET_CONTROL(awe_dev, v);
}

void seq_set_realtime_pan(int mode)
{
	AWE_REALTIME_PAN(awe_dev, mode);
}

void seq_set_def_drum(int val)
{
	AWE_MISC_MODE(awe_dev, AWE_MD_DEF_DRUM, val);
}

void seq_set_drumchannels(int channels)
{
	AWE_DRUM_CHANNELS(awe_dev, channels);
}

void seq_channel_priority(int mode)
{
	AWE_MISC_MODE(awe_dev, AWE_MD_CHN_PRIOR, mode);
}

void seq_new_volume_mode(int mode)
{
	AWE_MISC_MODE(awe_dev, AWE_MD_NEW_VOLUME_CALC, mode);
}

/*----------------------------------------------------------------
 * immediately change the values; sent direcly to the seq device
 *----------------------------------------------------------------*/

void seq_clear_samples(void)
{
	ioctl(seqfd, SNDCTL_SEQ_RESETSAMPLES, &awe_dev);
}

void seq_remove_samples(void)
{
	AWE_REMOVE_LAST_SAMPLES(seqfd, awe_dev);
}

#ifdef USE_OSSSEQ
/* toggle queue handling mode */
#define SEQ_QUEUE_REALTIME()		_TIMER_EVENT(TMR_WAIT_ABS, 0)
#define SEQ_QUEUE_SCHEDULED()		_TIMER_EVENT(TMR_WAIT_REL, 0)
#define AWE_DO_CMD(fd,dev,v,cmd,p1,p2)	{SEQ_QUEUE_REALTIME(); _AWE_CMD(dev, v, cmd, p1, p2); SEQ_QUEUE_SCHEDULED(); seqbuf_dump();}
#else
#define AWE_DO_CMD(fd,dev,v,cmd,p1,p2)	_AWE_CMD_NOW(fd, dev, v, cmd, p1, p2)
#endif

void seq_set_reverb(int mode)
{
	AWE_DO_CMD(seqfd, awe_dev, 0, _AWE_REVERB_MODE, mode, 0)
}

void seq_set_chorus(int mode)
{

	AWE_DO_CMD(seqfd, awe_dev, 0, _AWE_CHORUS_MODE, mode, 0)
}

void seq_equalizer(int bass, int treble)
{
	AWE_DO_CMD(seqfd, awe_dev, 0, _AWE_EQUALIZER, bass, treble)
}

/* change volume in percent (0-100) */
void seq_change_volume(int vol)
{
	int atten;
	if (vol <= 0)
		atten = 0xff;
	else
		atten = (int)(-log10((double)vol / 100.0) * 20 * 8 / 3);
	AWE_DO_CMD(seqfd, awe_dev, 0, _AWE_INITIAL_VOLUME, atten, 1)
}
