/*================================================================
 * Tk volume bar window
 *
 * Copyright (C) 1996-1998 Takashi Iwai
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *================================================================*/

#include <stdio.h>
#include "trace.h"
#include "config.h"
#include "util.h"
#include "channel.h"
#include "ext_c.h"

static void trace_volume(int ch, int val, int immediate);
static void trace_panning(int ch, int val);
static void trace_prog_init(int ch);
static void trace_prog(int ch, int val);
static void trace_bank(int ch, int val);
static void trace_sustain(int ch, int val);

/*----------------------------------------------------------------
 * functions for easy programming
 *----------------------------------------------------------------*/

extern Tcl_Interp *my_interp;

/* evaluate Tcl script */
char *v_eval(char *fmt, ...)
{
	char buf[256];
	va_list ap;
	va_start(ap, fmt);
	vsprintf(buf, fmt, ap);
	Tcl_Eval(my_interp, buf);
	va_end(ap);
	return my_interp->result;
}

/* read Tcl array variable */
char *v_get2(char *v1, char *v2)
{
	return Tcl_GetVar2(my_interp, v1, v2, 0);
}


/*----------------------------------------------------------------
 * update Tcl timer / trace window
 *----------------------------------------------------------------*/

static int prev_vel[MAX_MIDI_CHANNELS];
static ChannelStat prev_stat[MAX_MIDI_CHANNELS];
static int multi_part;
static int last_time = -1;


#define FRAME_WIN	".body.trace"
#define CANVAS_WIN	FRAME_WIN ".c"

#define BAR_WID 20
#define BAR_HGT 130
#define WIN_WID (BAR_WID * 16)
#define WIN_HGT (BAR_HGT + 11 + 17)
#define BAR_HALF_HGT (WIN_HGT / 2 - 11 - 17)


/* create a velocity bar window, and set the timer callback */
int TraceCreate(ClientData clientData, Tcl_Interp *interp,
		int argc, char *argv[])
{
	int i;
	v_eval("frame %s -bg black", FRAME_WIN);
	v_eval("canvas %s -width %d -height %d -bd 0 -bg black "
	       "-highlightthickness 0",
	       CANVAS_WIN, WIN_WID, WIN_HGT);
	v_eval("pack %s -side top -fill x", CANVAS_WIN);
	for (i = 0; i < 32; i++) {
		char *color;
		v_eval("%s create text 0 0 -anchor n -fill white -text 00 "
		       "-tags prog%d", CANVAS_WIN, i);
		v_eval("%s create poly 0 0 0 0 0 0 -fill yellow -tags pos%d",
		       CANVAS_WIN, i);
		color = (i == 9 || i == 25) ? "red" : "green";
		v_eval("%s create rect 0 0 0 0 -fill %s -tags bar%d "
		       "-outline \"\"", CANVAS_WIN, color, i);
	}
	v_eval("set Stat(TimerId) -1");

	/* force to reset */
	prev_stat[i].preset = -1;
	prev_stat[i].bank = -1;
	prev_stat[i].controls[CTL_PAN] = -1;
	prev_stat[i].controls[CTL_SUSTAIN] = -1;

	return TCL_OK;
}

/* bank number; draw drum chennls in red bar */
static void trace_bank(int ch, int val)
{
	if (prev_stat[ch].bank == val)
		return;
	v_eval("%s itemconfigure bar%d -fill %s",
	       CANVAS_WIN, ch,
	       (val == 128 ? "red" : "green"));
	prev_stat[ch].bank = val;
}

/* write program number */
static void trace_prog(int ch, int val)
{
	if (prev_stat[ch].preset == val)
		return;
	v_eval("%s itemconfigure prog%d -text %02X",
	       CANVAS_WIN, ch, val);
	prev_stat[ch].preset = val;
}

/* sustained; draw program number in green */
static void trace_sustain(int ch, int val)
{
	if (prev_stat[ch].controls[CTL_SUSTAIN] == val)
		return;
	v_eval("%s itemconfigure prog%d -fill %s",
	       CANVAS_WIN, ch,
	       (val == 127 ? "green" : "white"));
	prev_stat[ch].controls[CTL_SUSTAIN] = val;
}

/* initialize window; check whether multi part mode or not */
static void trace_prog_init(int ch)
{
	int item, yofs, bar, x, y;

	item = ch;
	yofs = 0;
	bar = multi_part ? BAR_HALF_HGT : BAR_HGT;
	if (ch >= 16) {
		ch -= 16;
		if (!multi_part)
			yofs = -500;
	} else if (multi_part)
		yofs = WIN_HGT / 2;
	x = ch * BAR_WID + BAR_WID/2;
	y = bar + 11 + yofs;
	v_eval("%s coords prog%d %d %d", CANVAS_WIN, item, x, y);
}

#define DELTA_VEL	16

/* write volume bar */
static void trace_volume(int ch, int val, int immediate)
{
	int item, bar, yofs, x1, y1, x2, y2;

	if (immediate)
		prev_vel[ch] = val;
	else {
		if (prev_vel[ch] == val)
			return;
		else if (prev_vel[ch] > val)
			prev_vel[ch] -= DELTA_VEL;
		if (prev_vel[ch] < val)
			prev_vel[ch] = val;
	}

	item = ch;
	yofs = 0;
	bar = multi_part ? BAR_HALF_HGT : BAR_HGT;
	if (ch >= 16) {
		ch -= 16;
		if (!multi_part)
			yofs = -500;
	} else if (multi_part)
		yofs = WIN_HGT / 2;
	x1 = ch * BAR_WID;
	y1 = bar - 1 + yofs;
	x2 = x1 + BAR_WID - 1;
	y2 = y1 - bar * prev_vel[item] / 127;
	v_eval("%s coords bar%d %d %d %d %d", CANVAS_WIN,
	       item, x1, y1, x2, y2);
}

/* write panning position mark */
static void trace_panning(int ch, int val)
{
	int item, bar, yofs;
	int x, ap, bp;

	if (val < 0) {
		v_eval("%s coords pos%d -1 0 -1 0 -1 0", CANVAS_WIN, ch);
		return;
	}

	if (prev_stat[ch].controls[CTL_PAN] == val)
		return;
	item = ch;
	yofs = 0;
	bar = multi_part ? BAR_HALF_HGT : BAR_HGT;
	if (ch >= 16) {
		ch -= 16;
		if (!multi_part)
			yofs = -500;
	} else if (multi_part)
		yofs = WIN_HGT / 2;

	x = BAR_WID * ch;
	ap = BAR_WID * val / 127;
	bp = BAR_WID - ap - 1;
	v_eval("%s coords pos%d %d %d %d %d %d %d", CANVAS_WIN, item,
	       ap + x, bar + 5 + yofs,
	       bp + x, bar + 1 + yofs,
	       bp + x, bar + 9 + yofs);
	prev_stat[ch].controls[CTL_PAN] = val;
}

/* reset all status */
int TraceReset(ClientData clientData, Tcl_Interp *interp,
	       int argc, char *argv[])
{
	int i;

	multi_part = extpanel->multi_part;
	last_time = -1;
	for (i = 0; i < 32; i++) {
		trace_volume(i, 0, TRUE);
		trace_panning(i, -1);
		trace_prog_init(i);
		trace_prog(i, 0);
		trace_sustain(i, 0);
	}
	extpanel->reset_panel = 0;

	return TCL_OK;
}



/* update status */
static void update_notes(void)
{
	int i;
	for (i = 0; i < 32; i++) {
		trace_volume(i, extpanel->maxvel[i], FALSE);
		extpanel->maxvel[i] = extpanel->vel[i];
		trace_panning(i, extpanel->channels[i].controls[CTL_PAN]);
		trace_bank(i, extpanel->channels[i].bank);
		trace_prog(i, extpanel->channels[i].preset);
		trace_sustain(i, extpanel->channels[i].controls[CTL_SUSTAIN]);
	}
}

/* check status change, and reset the timer callback */
int TraceUpdate(ClientData clientData, Tcl_Interp *interp,
		int argc, char *argv[])
{
	char *playing = v_get2("Stat", "Playing");
	if (playing && *playing != '0') {
		int csec = extpanel->curcs / 100;
		if (extpanel->reset_panel || extpanel->multi_part != multi_part) {
			v_eval("TraceReset");
		}
		if (last_time != csec) {
			v_eval("SetTime %d", csec);
			last_time = csec;
		}
		update_notes();
	}
	v_eval("set Stat(TimerId) [after 50 TraceUpdate]");
	return TCL_OK;
}


