/*
 * ALC885 Mac Pro preset module
 *
 * Copyright (C) 2007 Takashi Iwai
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 */

#include "config.h"
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include "hda/cmds.h"
#include "hda/io.h"
#include "hda/mixer.h"
#include "hda/patch.h"
#include "hda/codecs-helper.h"


#define ALC882_DIGOUT_NID	0x06
#define ALC882_DIGIN_NID	0x0a

static hda_nid_t alc885_dac_nids[4] = {
	/* front, rear, clfe, rear_surr */
	0x02, 0x03, 0x04, 0x05
};

/* Mac Pro test */
static struct hda_std_mixer alc885_macpro_mixer[] = {
	HDA_CODEC_VOLUME("Front Playback Volume", 0x0c, 0x0, HDA_OUTPUT),
	HDA_BIND_MUTE("Front Playback Switch", 0x0c, HDA_INPUT),
	HDA_CODEC_MUTE("Headphone Playback Switch", 0x18, 0x0, HDA_OUTPUT),
	HDA_CODEC_VOLUME("Line Playback Volume", 0x0b, 0x01, HDA_INPUT),
	HDA_CODEC_MUTE("Line Playback Switch", 0x0b, 0x01, HDA_INPUT),
	HDA_CODEC_VOLUME("PC Speaker Playback Volume", 0x0b, 0x02, HDA_INPUT),
	HDA_CODEC_MUTE("PC Speaker Playback Switch", 0x0b, 0x02, HDA_INPUT),
	{ } /* end */
};

static struct hda_verb alc885_macpro_init_verbs[] = {
	/* Front mixer: unmute input/output amp left and right (volume = 0) */
	{0x0c, AC_VERB_SET_AMP_GAIN_MUTE, AMP_OUT_ZERO},
	{0x0c, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_MUTE(0)},
	{0x0c, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_MUTE(1)},
	/* Front Pin: output 0 (0x0c) */
	{0x15, AC_VERB_SET_PIN_WIDGET_CONTROL, PIN_OUT},
	{0x15, AC_VERB_SET_AMP_GAIN_MUTE, AMP_OUT_UNMUTE},
	{0x15, AC_VERB_SET_CONNECT_SEL, 0x00},
	/* Front Mic pin: input vref at 80% */
	{0x19, AC_VERB_SET_PIN_WIDGET_CONTROL, PIN_VREF80},
	{0x19, AC_VERB_SET_AMP_GAIN_MUTE, AMP_OUT_MUTE},
	/* Speaker:  output */
	{0x1a, AC_VERB_SET_PIN_WIDGET_CONTROL, PIN_OUT},
	{0x1a, AC_VERB_SET_AMP_GAIN_MUTE, AMP_OUT_UNMUTE},
	{0x1a, AC_VERB_SET_CONNECT_SEL, 0x04},
	/* Headphone output (output 0 - 0x0c) */
	{0x18, AC_VERB_SET_PIN_WIDGET_CONTROL, PIN_HP},
	{0x18, AC_VERB_SET_AMP_GAIN_MUTE, AMP_OUT_UNMUTE},
	{0x18, AC_VERB_SET_CONNECT_SEL, 0x00},

	/* FIXME: use matrix-type input source selection */
	/* Mixer elements: 0x18, 19, 1a, 1b, 1c, 1d, 14, 15, 16, 17, 0b */
	/* Input mixer1: unmute Mic, F-Mic, Line, CD inputs */
	{0x24, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_UNMUTE(0)},
	{0x24, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_MUTE(3)},
	{0x24, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_MUTE(2)},
	{0x24, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_MUTE(4)},
	/* Input mixer2 */
	{0x23, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_UNMUTE(0)},
	{0x23, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_MUTE(3)},
	{0x23, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_MUTE(2)},
	{0x23, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_MUTE(4)},
	/* Input mixer3 */
	{0x22, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_UNMUTE(0)},
	{0x22, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_MUTE(3)},
	{0x22, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_MUTE(2)},
	{0x22, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_MUTE(4)},
	/* ADC1: mute amp left and right */
	{0x07, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_MUTE(0)},
	{0x07, AC_VERB_SET_CONNECT_SEL, 0x00},
	/* ADC2: mute amp left and right */
	{0x08, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_MUTE(0)},
	{0x08, AC_VERB_SET_CONNECT_SEL, 0x00},
	/* ADC3: mute amp left and right */
	{0x09, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_MUTE(0)},
	{0x09, AC_VERB_SET_CONNECT_SEL, 0x00},

	{ }
};

static int alc885_gpio_mute(int pin, int muted)
{
	unsigned int gpiostate, gpiomask, gpiodir;
	struct hda_verb verbs1[3] = {
		{ codec_info.afg, AC_VERB_SET_GPIO_MASK, 0 },
		{ codec_info.afg, AC_VERB_SET_GPIO_DIRECTION, 0 },
		{ }
	};
	struct hda_verb verbs2[2] = {
		{ codec_info.afg, AC_VERB_SET_GPIO_DATA, 0 },
		{ }
	};
	struct hda_cmds cmds[] = {
		STD_VERBS(verbs1),
		STD_DELAY(1),
		STD_VERBS(verbs2),
		STD_NULL
	};

	gpiostate = hda_codec_read(codec_info.afg, AC_VERB_GET_GPIO_DATA, 0);
	if (!muted)
		gpiostate |= (1 << pin);
	else
		gpiostate &= ~(1 << pin);
	verbs2[0].param = gpiostate;

	gpiomask = hda_codec_read(codec_info.afg, AC_VERB_GET_GPIO_MASK, 0);
	gpiomask |= (1 << pin);
	verbs1[0].param = gpiomask;

	gpiodir = hda_codec_read(codec_info.afg, AC_VERB_GET_GPIO_DIRECTION, 0);
	gpiodir |= (1 << pin);
	verbs1[1].param = gpiodir;

	return hda_register_init_cmds(cmds);
}

static struct codec_config_preset macpro_preset = {
	.init_verbs = { alc885_macpro_init_verbs },
	.mixers = { alc885_macpro_mixer },
	.num_dacs = 1,
	.dac_nids = alc885_dac_nids,
	.dig_out_nid = ALC882_DIGOUT_NID,
	.dig_in_nid = ALC882_DIGIN_NID,
};

static int patch_alc885_macpro(const struct hda_codec_table *tbl,
			       int board_config, const char **args)
{
	int err;

	err = alc885_gpio_mute(0, 0);
	if (err < 0)
		return err;
	err = alc885_gpio_mute(1, 0);
	if (err < 0)
		return err;

	return codec_build_preset(tbl, &macpro_preset);
}

/*
 */

static struct hda_codec_table macpro_table[] = {
	{ .id = 0x10ec0885, .vendor_name = "Realtek", .name = "ALC885",
	  .patch = patch_alc885_macpro },
	{ }
};

const struct hda_codec_table *patch_descriptor(void)
{
	return macpro_table;
}
