/*
 * AD1981 preset module
 *
 * Copyright (C) 2007 Takashi Iwai
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * AD1981 HD specific
 */

#include "config.h"
#include <stdio.h>
#include <string.h>
#include "hda/codecs-helper.h"


#define AD1981_SPDIF_OUT	0x02
#define AD1981_DAC		0x03
#define AD1981_ADC		0x04
#define AD1981_CAPSRC		0x15

/* 0x0c, 0x09, 0x0e, 0x0f, 0x19, 0x05, 0x18, 0x17 */
static struct hda_input_mux ad1981_capture_source = {
	.num_items = 7,
	.items = {
		{ "Front Mic", 0x0 },
		{ "Line", 0x1 },
		{ "Mix", 0x2 },
		{ "Mix Mono", 0x3 },
		{ "CD", 0x4 },
		{ "Mic", 0x6 },
		{ "Aux", 0x7 },
	},
};
static hda_nid_t ad1981_capture_pins[] = {
	0x08, 0x09, 0, 0, 0x19, 0x18, 0x17,
};

/*
 * SPDIF playback route
 */
static struct hda_verb spdif_src_pcm[] = {
	{ AD1981_SPDIF_OUT, AC_VERB_SET_CONNECT_SEL, 0 },
	{ }
};
static struct hda_verb spdif_src_adc[] = {
	{ AD1981_SPDIF_OUT, AC_VERB_SET_CONNECT_SEL, 1 },
	{ }
};

static struct hda_cmds spdif_playback_src[] = {
	STD_LABEL("PCM"), STD_VERBS(spdif_src_pcm),
	STD_LABEL("ADC"), STD_VERBS(spdif_src_adc),
	STD_NULL
};

static struct hda_std_mixer ad1981_mixers[] = {
	HDA_CODEC_VOLUME("Front Playback Volume", 0x05, 0x0, HDA_OUTPUT),
	HDA_CODEC_MUTE("Front Playback Switch", 0x05, 0x0, HDA_OUTPUT),
	HDA_CODEC_VOLUME("Headphone Playback Volume", 0x06, 0x0, HDA_OUTPUT),
	HDA_CODEC_MUTE("Headphone Playback Switch", 0x06, 0x0, HDA_OUTPUT),
	HDA_CODEC_VOLUME_MONO("Mono Playback Volume", 0x07, 1, 0x0, HDA_OUTPUT),
	HDA_CODEC_MUTE_MONO("Mono Playback Switch", 0x07, 1, 0x0, HDA_OUTPUT),
	HDA_CODEC_VOLUME("PCM Playback Volume", 0x11, 0x0, HDA_OUTPUT),
	HDA_CODEC_MUTE("PCM Playback Switch", 0x11, 0x0, HDA_OUTPUT),
	HDA_CODEC_VOLUME("Front Mic Playback Volume", 0x12, 0x0, HDA_OUTPUT),
	HDA_CODEC_MUTE("Front Mic Playback Switch", 0x12, 0x0, HDA_OUTPUT),
	HDA_CODEC_VOLUME("Line Playback Volume", 0x13, 0x0, HDA_OUTPUT),
	HDA_CODEC_MUTE("Line Playback Switch", 0x13, 0x0, HDA_OUTPUT),
	HDA_CODEC_VOLUME("Aux Playback Volume", 0x1b, 0x0, HDA_OUTPUT),
	HDA_CODEC_MUTE("Aux Playback Switch", 0x1b, 0x0, HDA_OUTPUT),
	HDA_CODEC_VOLUME("Mic Playback Volume", 0x1c, 0x0, HDA_OUTPUT),
	HDA_CODEC_MUTE("Mic Playback Switch", 0x1c, 0x0, HDA_OUTPUT),
	HDA_CODEC_VOLUME("CD Playback Volume", 0x1d, 0x0, HDA_OUTPUT),
	HDA_CODEC_MUTE("CD Playback Switch", 0x1d, 0x0, HDA_OUTPUT),
	HDA_CODEC_VOLUME_MONO("PC Speaker Playback Volume", 0x0d, 1, 0x0, HDA_OUTPUT),
	HDA_CODEC_MUTE_MONO("PC Speaker Playback Switch", 0x0d, 1, 0x0, HDA_OUTPUT),
	HDA_CODEC_VOLUME("Front Mic Boost Volume", 0x08, 0x0, HDA_INPUT),
	HDA_CODEC_VOLUME("Mic Boost Volume", 0x18, 0x0, HDA_INPUT),
	HDA_CODEC_VOLUME("Capture Volume", 0x15, 0x0, HDA_OUTPUT),
	HDA_CODEC_MUTE("Capture Switch", 0x15, 0x0, HDA_OUTPUT),
	HDA_MIXER_ENUM("IEC958 Playback Source", spdif_playback_src),
	{ } /* end */
};

static struct hda_verb ad1981_init_verbs[] = {
	/* Front, HP, Mono; mute as default */
	{0x05, AC_VERB_SET_AMP_GAIN_MUTE, 0xb080},
	{0x06, AC_VERB_SET_AMP_GAIN_MUTE, 0xb080},
	{0x07, AC_VERB_SET_AMP_GAIN_MUTE, 0xb080},
	/* Beep, PCM, Front Mic, Line, Rear Mic, Aux, CD-In: mute */
	{0x0d, AC_VERB_SET_AMP_GAIN_MUTE, 0xb080},
	{0x11, AC_VERB_SET_AMP_GAIN_MUTE, 0xb080},
	{0x12, AC_VERB_SET_AMP_GAIN_MUTE, 0xb080},
	{0x13, AC_VERB_SET_AMP_GAIN_MUTE, 0xb080},
	{0x1b, AC_VERB_SET_AMP_GAIN_MUTE, 0xb080},
	{0x1c, AC_VERB_SET_AMP_GAIN_MUTE, 0xb080},
	{0x1d, AC_VERB_SET_AMP_GAIN_MUTE, 0xb080},
	/* Front, HP selectors; from Mix */
	{0x05, AC_VERB_SET_CONNECT_SEL, 0x01},
	{0x06, AC_VERB_SET_CONNECT_SEL, 0x01},
	/* Mono selector; from Mix */
	{0x0b, AC_VERB_SET_CONNECT_SEL, 0x03},
	/* Mic Mixer; select Front Mic */
	{0x1e, AC_VERB_SET_AMP_GAIN_MUTE, 0xb000},
	{0x1f, AC_VERB_SET_AMP_GAIN_MUTE, 0xb080},
	/* Mic boost: 0dB */
	{0x08, AC_VERB_SET_AMP_GAIN_MUTE, 0xb000},
	{0x18, AC_VERB_SET_AMP_GAIN_MUTE, 0xb000},
	/* Record selector: Front mic */
	{0x15, AC_VERB_SET_CONNECT_SEL, 0x0},
	{0x15, AC_VERB_SET_AMP_GAIN_MUTE, 0xb080},
	/* SPDIF route: PCM */
	{0x02, AC_VERB_SET_CONNECT_SEL, 0x0},
	/* Front Pin */
	{0x05, AC_VERB_SET_PIN_WIDGET_CONTROL, 0x40 },
	/* HP Pin */
	{0x06, AC_VERB_SET_PIN_WIDGET_CONTROL, 0xc0 },
	/* Mono Pin */
	{0x07, AC_VERB_SET_PIN_WIDGET_CONTROL, 0x40 },
	/* Front & Rear Mic Pins */
	{0x08, AC_VERB_SET_PIN_WIDGET_CONTROL, 0x24 },
	{0x18, AC_VERB_SET_PIN_WIDGET_CONTROL, 0x24 },
	/* Line Pin */
	{0x09, AC_VERB_SET_PIN_WIDGET_CONTROL, 0x20 },
	/* Digital Beep */
	{0x0d, AC_VERB_SET_CONNECT_SEL, 0x00},
	/* Line-Out as Input: disabled */
	{0x1a, AC_VERB_SET_AMP_GAIN_MUTE, 0xb080},
	{ } /* end */
};

/*
 * Patch for HP nx6320
 *
 * nx6320 uses EAPD in the reverse way - EAPD-on means the internal
 * speaker output enabled _and_ mute-LED off.
 */

#define AD1981_HP_EVENT		0x37
#define AD1981_MIC_EVENT	0x38

static struct hda_verb ad1981_hp_init_verbs[] = {
	{0x05, AC_VERB_SET_EAPD_BTLENABLE, 0x00 }, /* default off */
	{}
};

/* turn on/off EAPD (+ mute HP) as a master switch */
static struct hda_verb eapd_on[] = {
	{ 0x12, AC_VERB_SET_EAPD_BTLENABLE, 0x02 },
	{ }
};
static struct hda_verb eapd_off[] = {
	{ 0x12, AC_VERB_SET_EAPD_BTLENABLE, 0x00 },
	{ }
};

static struct hda_cmds ad1981_hp_master_on_cmds[] = {
	STD_VERBS(eapd_on),
	STD_STEREO_UNMUTE(0x06, HDA_AC_OUT),
	STD_NULL
};
static struct hda_cmds ad1981_hp_master_off_cmds[] = {
	STD_VERBS(eapd_off),
	STD_STEREO_MUTE(0x06, HDA_AC_OUT),
	STD_NULL
};

static struct hda_cmds ad1981_hp_master_sws[] = {
	STD_CMDS(ad1981_hp_master_on_cmds),
	STD_CMDS(ad1981_hp_master_off_cmds),
	STD_NULL
};

/* bind volumes of both NID 0x05 and 0x06 */
static u32 ad1981_hp_master_vols[] = {
	HDA_COMPOSE_AMP_VAL(0x05, 3, 0, HDA_OUTPUT),
	HDA_COMPOSE_AMP_VAL(0x06, 3, 0, HDA_OUTPUT),
	0
};

/* mute internal speaker if HP is plugged */
static struct codec_config_unsol hp_automute_unsol = {
	.nid = 0x06,
	.tag = AD1981_HP_EVENT,
	.cmds = STD_JACK_SENSE(0x06, STD_STEREO_MUTE(0x05, HDA_AC_OUT),
			       STD_STEREO_UNMUTE(0x05, HDA_AC_OUT)),
};

/* toggle input of built-in and mic jack appropriately */
static struct hda_verb mic_jack_on[] = {
	{0x1f, AC_VERB_SET_AMP_GAIN_MUTE, 0xb080},
	{0x1e, AC_VERB_SET_AMP_GAIN_MUTE, 0xb000},
	{}
};
static struct hda_verb mic_jack_off[] = {
	{0x1e, AC_VERB_SET_AMP_GAIN_MUTE, 0xb080},
	{0x1f, AC_VERB_SET_AMP_GAIN_MUTE, 0xb000},
	{}
};
static struct codec_config_unsol hp_automic_unsol = {
	.nid = 0x08,
	.tag = AD1981_MIC_EVENT,
	.cmds = STD_JACK_SENSE(0x08, STD_VERBS(mic_jack_on),
			       STD_VERBS(mic_jack_off)),
};

static struct hda_input_mux ad1981_hp_capture_source = {
	.num_items = 3,
	.items = {
		{ "Mic", 0x0 },
		{ "Docking-Station", 0x1 },
		{ "Mix", 0x2 },
	},
};

static struct hda_std_mixer ad1981_hp_mixers[] = {
	HDA_MIXER_BIND_VOLUME("Master Playback Volume", ad1981_hp_master_vols),
	HDA_MIXER_BOOL("Master Playback Switch", ad1981_hp_master_sws),
	HDA_CODEC_VOLUME("PCM Playback Volume", 0x11, 0x0, HDA_OUTPUT),
	HDA_CODEC_MUTE("PCM Playback Switch", 0x11, 0x0, HDA_OUTPUT),
#if 0
	/* FIXME: analog mic/line loopback doesn't work with my tests...
	 *        (although recording is OK)
	 */
	HDA_CODEC_VOLUME("Mic Playback Volume", 0x12, 0x0, HDA_OUTPUT),
	HDA_CODEC_MUTE("Mic Playback Switch", 0x12, 0x0, HDA_OUTPUT),
	HDA_CODEC_VOLUME("Docking-Station Playback Volume", 0x13, 0x0, HDA_OUTPUT),
	HDA_CODEC_MUTE("Docking-Station Playback Switch", 0x13, 0x0, HDA_OUTPUT),
	HDA_CODEC_VOLUME("Internal Mic Playback Volume", 0x1c, 0x0, HDA_OUTPUT),
	HDA_CODEC_MUTE("Internal Mic Playback Switch", 0x1c, 0x0, HDA_OUTPUT),
	/* FIXME: does this laptop have analog CD connection? */
	HDA_CODEC_VOLUME("CD Playback Volume", 0x1d, 0x0, HDA_OUTPUT),
	HDA_CODEC_MUTE("CD Playback Switch", 0x1d, 0x0, HDA_OUTPUT),
#endif
	HDA_CODEC_VOLUME("Mic Boost Volume", 0x08, 0x0, HDA_INPUT),
	HDA_CODEC_VOLUME("Internal Mic Boost Volume", 0x18, 0x0, HDA_INPUT),
	HDA_CODEC_VOLUME("Capture Volume", 0x15, 0x0, HDA_OUTPUT),
	HDA_CODEC_MUTE("Capture Switch", 0x15, 0x0, HDA_OUTPUT),
	{ } /* end */
};

/* configuration for Toshiba Laptops */
static struct hda_verb ad1981_toshiba_init_verbs[] = {
	{0x05, AC_VERB_SET_EAPD_BTLENABLE, 0x01 }, /* default on */
	{}
};

static struct hda_std_mixer ad1981_toshiba_mixers[] = {
	HDA_CODEC_VOLUME("Amp Volume", 0x1a, 0x0, HDA_OUTPUT),
	HDA_CODEC_MUTE("Amp Switch", 0x1a, 0x0, HDA_OUTPUT),
	{ }
};

/* configuration for Lenovo Thinkpad T60 */
static struct hda_std_mixer ad1981_thinkpad_mixers[] = {
	HDA_CODEC_VOLUME("Master Playback Volume", 0x05, 0x0, HDA_OUTPUT),
	HDA_CODEC_MUTE("Master Playback Switch", 0x05, 0x0, HDA_OUTPUT),
	HDA_CODEC_VOLUME("PCM Playback Volume", 0x11, 0x0, HDA_OUTPUT),
	HDA_CODEC_MUTE("PCM Playback Switch", 0x11, 0x0, HDA_OUTPUT),
	HDA_CODEC_VOLUME("Mic Playback Volume", 0x12, 0x0, HDA_OUTPUT),
	HDA_CODEC_MUTE("Mic Playback Switch", 0x12, 0x0, HDA_OUTPUT),
	HDA_CODEC_VOLUME("CD Playback Volume", 0x1d, 0x0, HDA_OUTPUT),
	HDA_CODEC_MUTE("CD Playback Switch", 0x1d, 0x0, HDA_OUTPUT),
	HDA_CODEC_VOLUME("Mic Boost Volume", 0x08, 0x0, HDA_INPUT),
	HDA_CODEC_VOLUME("Capture Volume", 0x15, 0x0, HDA_OUTPUT),
	HDA_CODEC_MUTE("Capture Switch", 0x15, 0x0, HDA_OUTPUT),
	HDA_MIXER_ENUM("IEC958 Playback Source", spdif_playback_src),
	{ } /* end */
};

static struct hda_input_mux ad1981_thinkpad_capture_source = {
	.num_items = 3,
	.items = {
		{ "Mic", 0x0 },
		{ "Mix", 0x2 },
		{ "CD", 0x4 },
	},
};


/* models */
enum {
	AD1981_BASIC,
	AD1981_HP,
	AD1981_THINKPAD,
	AD1981_TOSHIBA,
	AD1981_MODELS
};

static const char *ad1981_models[AD1981_MODELS] = {
	[AD1981_BASIC]		= "basic",
	[AD1981_HP]		= "hp",
	[AD1981_THINKPAD]	= "thinkpad",
	[AD1981_TOSHIBA]	= "toshiba"
};

static hda_nid_t ad1981_dac_nids[1] = { AD1981_DAC };

static struct codec_config_preset ad1981_presets[] = {
	[AD1981_BASIC] = {
		.init_verbs = {
			ad1981_init_verbs,
		},
		.mixers = {
			ad1981_mixers,
		},
		.num_dacs = 1,
		.dac_nids = ad1981_dac_nids,
		.adc_nid = AD1981_ADC,
		.input_mux = &ad1981_capture_source,
		.input_mux_nid = AD1981_CAPSRC,
		.input_mux_pins = ad1981_capture_pins,
		.dig_out_nid = AD1981_SPDIF_OUT,
	},
	[AD1981_HP] = {
		.init_verbs = {
			ad1981_init_verbs,
			ad1981_hp_init_verbs,
		},
		.mixers = {
			ad1981_hp_mixers,
		},
		.num_dacs = 1,
		.dac_nids = ad1981_dac_nids,
		.adc_nid = AD1981_ADC,
		.input_mux = &ad1981_hp_capture_source,
		.input_mux_nid = AD1981_CAPSRC,
		.unsols = {
			 &hp_automute_unsol,
			 &hp_automic_unsol,
		 },
	},
	[AD1981_THINKPAD] = {
		.init_verbs = {
			ad1981_init_verbs,
		},
		.mixers = {
			ad1981_thinkpad_mixers,
		},
		.num_dacs = 1,
		.dac_nids = ad1981_dac_nids,
		.adc_nid = AD1981_ADC,
		.input_mux = &ad1981_thinkpad_capture_source,
		.input_mux_nid = AD1981_CAPSRC,
		.dig_out_nid = AD1981_SPDIF_OUT,
	},
	[AD1981_TOSHIBA] = {
		.init_verbs = {
			ad1981_init_verbs,
			ad1981_toshiba_init_verbs,
		},
		.mixers = {
			ad1981_hp_mixers,
			ad1981_toshiba_mixers,
		},
		.num_dacs = 1,
		.dac_nids = ad1981_dac_nids,
		.adc_nid = AD1981_ADC,
		.input_mux = &ad1981_hp_capture_source,
		.input_mux_nid = AD1981_CAPSRC,
		.unsols = {
			 &hp_automute_unsol,
			 &hp_automic_unsol,
		 },
	},
};

static int patch_ad1981(const struct hda_codec_table *tbl,
			int board_config, const char **args)
{
	if (board_config < 0 || board_config >= AD1981_MODELS)
		board_config = 0;
	return codec_build_preset(tbl, &ad1981_presets[board_config]);
}

/*
 */

static struct hda_codec_table ad1981_table[] = {
	{ .id = 0x11d41981, .vendor_name = "Analog Devices", .name = "AD1981HD",
	  .num_presets = AD1981_MODELS, .presets = ad1981_models,
	  .patch = patch_ad1981 },
	{ }
};

const struct hda_codec_table *patch_descriptor(void)
{
	return ad1981_table;
}
