/*
 * Generic widget tree parser module
 *
 * Copyright (c) 2007 Takashi Iwai <tiwai@suse.de>
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include "hda/cmds.h"
#include "hda/io.h"
#include "hda/mixer.h"
#include "hda/log.h"
#include "hda/pincfg.h"
#include "hda/patch.h"

/* widget node for parsing */
struct hda_gnode {
	hda_nid_t nid;		/* NID of this widget */
	unsigned short nconns;	/* number of input connections */
	hda_nid_t *conn_list;
	unsigned int wid_caps;	/* widget capabilities */
	unsigned char type;	/* widget type */
	unsigned char pin_ctl;	/* pin controls */
	unsigned char checked;	/* the flag indicates that the node is already parsed */
	unsigned int pin_caps;	/* pin widget capabilities */
	unsigned int def_cfg;	/* default configuration */
	unsigned int amp_out_caps;	/* AMP out capabilities */
	unsigned int amp_in_caps;	/* AMP in capabilities */
	struct hda_gnode *next;
};

/* patch-specific record */

#define MAX_PCM_VOLS	2
struct pcm_vol {
	struct hda_gnode *node;	/* Node for PCM volume */
	unsigned int index;	/* connection of PCM volume */
};


static struct hda_gnode *dac_node[2];	/* DAC node */
static struct hda_gnode *out_pin_node[2];	/* Output pin (Line-Out) node */
static struct pcm_vol pcm_vol[MAX_PCM_VOLS];	/* PCM volumes */
static unsigned int pcm_vol_nodes;	/* number of PCM volumes */

static struct hda_gnode *adc_node;	/* ADC node */
static struct hda_input_mux input_mux;
static char cap_labels[HDA_MAX_NUM_INPUTS][16];

static unsigned int def_amp_in_caps;
static unsigned int def_amp_out_caps;

static struct hda_gnode *nid_list, *nid_last;

static struct hda_verb_array auto_init;

/*
 * retrieve the default device type from the default config value
 */
#define defcfg_type(node) (((node)->def_cfg & AC_DEFCFG_DEVICE) >> \
			   AC_DEFCFG_DEVICE_SHIFT)
#define defcfg_location(node) (((node)->def_cfg & AC_DEFCFG_LOCATION) >> \
			       AC_DEFCFG_LOCATION_SHIFT)
#define defcfg_port_conn(node) (((node)->def_cfg & AC_DEFCFG_PORT_CONN) >> \
				AC_DEFCFG_PORT_CONN_SHIFT)

/*
 * destructor
 */
static void patch_generic_free(void)
{
	struct hda_gnode *node, *next;

	/* free all widgets */
	for (node = nid_list; node; node = next) {
		next = node->next;
		free(node->conn_list);
		free(node);
	}
	free(auto_init.verbs);
}


/*
 * add a new widget node and read its attributes
 */
static int add_new_node(hda_nid_t nid)
{
	struct hda_gnode *node;
	int nconns;
	hda_nid_t conn_list[HDA_MAX_CONNECTIONS];

	node = malloc(sizeof(*node));
	if (!node) {
		log_error("generic: cannot malloc\n");
		return -ENOMEM;
	}
	memset(node, 0, sizeof(*node));
	node->nid = nid;
	nconns = hda_get_connections(nid, conn_list, HDA_MAX_CONNECTIONS);
	if (nconns < 0) {
		log_error("Cannot get connections for NID 0x%x\n", nid);
		free(node);
		return nconns;
	}
	node->conn_list = malloc(sizeof(hda_nid_t) * nconns);
	if (!node->conn_list) {
		log_error("generic: cannot malloc\n");
		free(node);
		return -ENOMEM;
	}
	memcpy(node->conn_list, conn_list, nconns * sizeof(hda_nid_t));
	node->nconns = nconns;
	node->wid_caps = hda_get_wcaps(nid);
	node->type = (node->wid_caps & AC_WCAP_TYPE) >> AC_WCAP_TYPE_SHIFT;

	if (node->type == AC_WID_PIN) {
		node->pin_caps = hda_param_read(node->nid, AC_PAR_PIN_CAP);
		node->pin_ctl = hda_codec_read(node->nid,
					       AC_VERB_GET_PIN_WIDGET_CONTROL,
					       0);
		node->def_cfg = hda_codec_read(node->nid,
					       AC_VERB_GET_CONFIG_DEFAULT, 0);
	}

	if (node->wid_caps & AC_WCAP_OUT_AMP) {
		if (node->wid_caps & AC_WCAP_AMP_OVRD)
			node->amp_out_caps =
				hda_param_read(node->nid, AC_PAR_AMP_OUT_CAP);
		if (!node->amp_out_caps)
			node->amp_out_caps = def_amp_out_caps;
	}
	if (node->wid_caps & AC_WCAP_IN_AMP) {
		if (node->wid_caps & AC_WCAP_AMP_OVRD)
			node->amp_in_caps =
				hda_param_read(node->nid, AC_PAR_AMP_IN_CAP);
		if (! node->amp_in_caps)
			node->amp_in_caps = def_amp_in_caps;
	}
	if (nid_last)
		nid_last->next = node;
	else
		nid_list = node;
	node->next = NULL;
	return 0;
}

/*
 * build the AFG subtree
 */
static int build_afg_tree(void)
{
	int i, nodes, err;
	hda_nid_t afg, nid;

	afg = codec_info.afg;
	def_amp_out_caps = hda_param_read(afg, AC_PAR_AMP_OUT_CAP);
	def_amp_in_caps = hda_param_read(afg, AC_PAR_AMP_IN_CAP);

	nodes = hda_get_sub_nodes(afg, &nid);
	if (!nid || nodes < 0) {
		log_error("Invalid AFG subtree\n");
		return -EINVAL;
	}

	/* parse all nodes belonging to the AFG */
	for (i = 0; i < nodes; i++, nid++) {
		err = add_new_node(nid);
		if (err < 0)
			return err;
	}

	return 0;
}


/*
 * look for the node record for the given NID
 */
/* FIXME: should avoid the braindead linear search */
static struct hda_gnode *hda_get_node(hda_nid_t nid)
{
	struct hda_gnode *node;

	for (node = nid_list; node; node = node->next) {
		if (node->nid == nid)
			return node;
	}
	return NULL;
}

/*
 * unmute (and set max vol) the output amplifier
 */
static int unmute_output(struct hda_gnode *node)
{
	unsigned int val, ofs;

	log_verbose("UNMUTE OUT: NID=0x%x\n", node->nid);
	val = (node->amp_out_caps & AC_AMPCAP_NUM_STEPS) >>
		AC_AMPCAP_NUM_STEPS_SHIFT;
	ofs = (node->amp_out_caps & AC_AMPCAP_OFFSET) >>
		AC_AMPCAP_OFFSET_SHIFT;
	if (val >= ofs)
		val -= ofs;
	val |= AC_AMP_SET_LEFT | AC_AMP_SET_RIGHT;
	val |= AC_AMP_SET_OUTPUT;
	hda_codec_write(node->nid, AC_VERB_SET_AMP_GAIN_MUTE, val);
	return hda_append_verb(&auto_init,
			       node->nid, AC_VERB_SET_AMP_GAIN_MUTE, val);
}

/*
 * unmute (and set max vol) the input amplifier
 */
static int unmute_input(struct hda_gnode *node, unsigned int index)
{
	unsigned int val, ofs;

	log_verbose("UNMUTE IN: NID=0x%x IDX=0x%x\n", node->nid, index);
	val = (node->amp_in_caps & AC_AMPCAP_NUM_STEPS) >>
		AC_AMPCAP_NUM_STEPS_SHIFT;
	ofs = (node->amp_in_caps & AC_AMPCAP_OFFSET) >>
		AC_AMPCAP_OFFSET_SHIFT;
	if (val >= ofs)
		val -= ofs;
	val |= AC_AMP_SET_LEFT | AC_AMP_SET_RIGHT;
	val |= AC_AMP_SET_INPUT;
	// awk added - fixed to allow unmuting of indexed amps
	val |= index << AC_AMP_SET_INDEX_SHIFT;
	hda_codec_write(node->nid, AC_VERB_SET_AMP_GAIN_MUTE, val);
	return hda_append_verb(&auto_init,
			       node->nid, AC_VERB_SET_AMP_GAIN_MUTE, val);
}

/*
 * select the input connection of the given node.
 */
static int select_input_connection(struct hda_gnode *node, unsigned int index)
{
	log_verbose("CONNECT: NID=0x%x IDX=0x%x\n", node->nid, index);
	hda_codec_write(node->nid, AC_VERB_SET_CONNECT_SEL, index);
	return hda_append_verb(&auto_init,
			       node->nid, AC_VERB_SET_CONNECT_SEL, index);
}

/*
 * clear checked flag of each node in the node list
 */
static void clear_check_flags(void)
{
	struct hda_gnode *node;

	for (node = nid_list; node; node = node->next)
		node->checked = 0;
}

/*
 * parse the output path recursively until reach to an audio output widget
 *
 * returns 0 if not found, 1 if found, or a negative error code.
 */
static int parse_output_path(struct hda_gnode *node, int dac_idx)
{
	int i, err;
	struct hda_gnode *child;

	if (node->checked)
		return 0;

	node->checked = 1;
	if (node->type == AC_WID_AUD_OUT) {
		if (node->wid_caps & AC_WCAP_DIGITAL) {
			log_verbose("Skip Digital OUT node %x\n", node->nid);
			return 0;
		}
		log_verbose("AUD_OUT found %x\n", node->nid);
		if (dac_node[dac_idx]) {
			/* already DAC node is assigned,
			 * just unmute & connect
			 */
			return node == dac_node[dac_idx];
		}
		dac_node[dac_idx] = node;
		if ((node->wid_caps & AC_WCAP_OUT_AMP) &&
		    pcm_vol_nodes < MAX_PCM_VOLS) {
			pcm_vol[pcm_vol_nodes].node = node;
			pcm_vol[pcm_vol_nodes].index = 0;
			pcm_vol_nodes++;
		}
		return 1; /* found */
	}

	for (i = 0; i < node->nconns; i++) {
		child = hda_get_node(node->conn_list[i]);
		if (!child)
			continue;
		err = parse_output_path(child, dac_idx);
		if (err < 0)
			return err;
		else if (err > 0) {
			/* found one,
			 * select the path, unmute both input and output
			 */
			if (node->nconns > 1)
				select_input_connection(node, i);
			unmute_input(node, i);
			unmute_output(node);
			if (dac_node[dac_idx] &&
			    pcm_vol_nodes < MAX_PCM_VOLS &&
			    !(dac_node[dac_idx]->wid_caps & AC_WCAP_OUT_AMP)) {
				if ((node->wid_caps & AC_WCAP_IN_AMP) ||
				    (node->wid_caps & AC_WCAP_OUT_AMP)) {
					int n = pcm_vol_nodes;
					pcm_vol[n].node = node;
					pcm_vol[n].index = i;
					pcm_vol_nodes++;
				}
			}
			return 1;
		}
	}
	return 0;
}

/*
 * Look for the output PIN widget with the given jack type
 * and parse the output path to that PIN.
 *
 * Returns the PIN node when the path to DAC is established.
 */
static struct hda_gnode *parse_output_jack(int jack_type)
{
	struct hda_gnode *node;
	int err;

	for (node = nid_list; node; node = node->next) {
		if (node->type != AC_WID_PIN)
			continue;
		/* output capable? */
		if (!(node->pin_caps & AC_PINCAP_OUT))
			continue;
		if (defcfg_port_conn(node) == AC_JACK_PORT_NONE)
			continue; /* unconnected */
		if (jack_type >= 0) {
			if (jack_type != defcfg_type(node))
				continue;
			if (node->wid_caps & AC_WCAP_DIGITAL)
				continue; /* skip SPDIF */
		} else {
			/* output as default? */
			if (!(node->pin_ctl & AC_PINCTL_OUT_EN))
				continue;
		}
		clear_check_flags();
		err = parse_output_path(node, 0);
		if (err < 0)
			return NULL;
		if (!err && out_pin_node[0]) {
			err = parse_output_path(node, 1);
			if (err < 0)
				return NULL;
		}
		if (err > 0) {
			/* unmute the PIN output */
			unmute_output(node);
			/* set PIN-Out enable */
			hda_codec_write(node->nid,
					AC_VERB_SET_PIN_WIDGET_CONTROL,
					AC_PINCTL_OUT_EN |
					((node->pin_caps & AC_PINCAP_HP_DRV) ?
					 AC_PINCTL_HP_EN : 0));
			hda_append_verb(&auto_init, node->nid,
					AC_VERB_SET_PIN_WIDGET_CONTROL,
					AC_PINCTL_OUT_EN |
					((node->pin_caps & AC_PINCAP_HP_DRV) ?
					 AC_PINCTL_HP_EN : 0));
			return node;
		}
	}
	return NULL;
}


/*
 * parse outputs
 */
static int parse_output(void)
{
	struct hda_gnode *node;

	/*
	 * Look for the output PIN widget
	 */
	/* first, look for the line-out pin */
	node = parse_output_jack(AC_JACK_LINE_OUT);
	if (node) /* found, remember the PIN node */
		out_pin_node[0] = node;
	else {
		/* if no line-out is found, try speaker out */
		node = parse_output_jack(AC_JACK_SPEAKER);
		if (node)
			out_pin_node[0] = node;
	}
	/* look for the HP-out pin */
	node = parse_output_jack(AC_JACK_HP_OUT);
	if (node) {
		if (!out_pin_node[0])
			out_pin_node[0] = node;
		else
			out_pin_node[1] = node;
	}

	if (!out_pin_node[0]) {
		/* no line-out or HP pins found,
		 * then choose for the first output pin
		 */
		out_pin_node[0] = parse_output_jack(-1);
		if (!out_pin_node[0])
			log_info("generic: no proper output path found\n");
	}

	return 0;
}

/*
 * return the string name of the given input PIN widget
 */
static const char *get_input_type(struct hda_gnode *node, unsigned int *pinctl)
{
	unsigned int location = defcfg_location(node);
	switch (defcfg_type(node)) {
	case AC_JACK_LINE_IN:
		if ((location & 0x0f) == AC_JACK_LOC_FRONT)
			return "Front Line";
		return "Line";
	case AC_JACK_CD:
#if 0
		if (pinctl)
			*pinctl |= AC_PINCTL_VREF_GRD;
#endif
		return "CD";
	case AC_JACK_AUX:
		if ((location & 0x0f) == AC_JACK_LOC_FRONT)
			return "Front Aux";
		return "Aux";
	case AC_JACK_MIC_IN:
		if (pinctl &&
		    (node->pin_caps &
		     (AC_PINCAP_VREF_80 << AC_PINCAP_VREF_SHIFT)))
			*pinctl |= AC_PINCTL_VREF_80;
		if ((location & 0x0f) == AC_JACK_LOC_FRONT)
			return "Front Mic";
		return "Mic";
	case AC_JACK_SPDIF_IN:
		return "SPDIF";
	case AC_JACK_DIG_OTHER_IN:
		return "Digital";
	}
	return NULL;
}

/*
 * parse the nodes recursively until reach to the input PIN
 *
 * returns 0 if not found, 1 if found, or a negative error code.
 */
static int parse_adc_sub_nodes(struct hda_gnode *node)
{
	int i, err;
	unsigned int pinctl;
	char *label;
	const char *type;

	if (node->checked)
		return 0;

	node->checked = 1;
	if (node->type != AC_WID_PIN) {
		for (i = 0; i < node->nconns; i++) {
			struct hda_gnode *child;
			child = hda_get_node(node->conn_list[i]);
			if (! child)
				continue;
			err = parse_adc_sub_nodes(child);
			if (err < 0)
				return err;
			if (err > 0) {
				/* found one,
				 * select the path, unmute both input and output
				 */
				if (node->nconns > 1)
					select_input_connection(node, i);
				unmute_input(node, i);
				unmute_output(node);
				return err;
			}
		}
		return 0;
	}

	/* input capable? */
	if (!(node->pin_caps & AC_PINCAP_IN))
		return 0;

	if (defcfg_port_conn(node) == AC_JACK_PORT_NONE)
		return 0; /* unconnected */

	if (node->wid_caps & AC_WCAP_DIGITAL)
		return 0; /* skip SPDIF */

	if (input_mux.num_items >= HDA_MAX_NUM_INPUTS) {
		log_error("generic: Too many items for capture\n");
		return -EINVAL;
	}

	pinctl = AC_PINCTL_IN_EN;
	/* create a proper capture source label */
	type = get_input_type(node, &pinctl);
	if (!type) {
		/* input as default? */
		if (!(node->pin_ctl & AC_PINCTL_IN_EN))
			return 0;
		type = "Input";
	}
	label = cap_labels[input_mux.num_items];
	strcpy(label, type);
	input_mux.items[input_mux.num_items].label = label;

	/* unmute the PIN external input */
	unmute_input(node, 0); /* index = 0? */
	/* set PIN-In enable */
	hda_codec_write(node->nid, AC_VERB_SET_PIN_WIDGET_CONTROL, pinctl);
	hda_append_verb(&auto_init,
			node->nid, AC_VERB_SET_PIN_WIDGET_CONTROL, pinctl);
	return 1; /* found */
}

/* add a capture source element */
static void add_cap_src(int idx)
{
	struct hda_input_mux_item *csrc;
	char *buf;
	int num, ocap;

	num = input_mux.num_items;
	csrc = &input_mux.items[num];
	buf = cap_labels[num];
	for (ocap = 0; ocap < num; ocap++) {
		if (!strcmp(buf, cap_labels[ocap])) {
			/* same label already exists,
			 * put the index number to be unique
			 */
			sprintf(buf, "%s %d", cap_labels[ocap], num);
			break;
		}
	}
	csrc->index = idx;
	input_mux.num_items++;
}

/*
 * parse input
 */
static int parse_input_path(struct hda_gnode *adc_node)
{
	struct hda_gnode *node;
	int i, err;

	log_verbose("AUD_IN = %x\n", adc_node->nid);
	clear_check_flags();

	// awk added - fixed no recording due to muted widget
	unmute_input(adc_node, 0);
	
	/*
	 * check each connection of the ADC
	 * if it reaches to a proper input PIN, add the path as the
	 * input path.
	 */
	/* first, check the direct connections to PIN widgets */
	for (i = 0; i < adc_node->nconns; i++) {
		node = hda_get_node(adc_node->conn_list[i]);
		if (node && node->type == AC_WID_PIN) {
			err = parse_adc_sub_nodes(node);
			if (err < 0)
				return err;
			else if (err > 0)
				add_cap_src(i);
		}
	}
	/* ... then check the rests, more complicated connections */
	for (i = 0; i < adc_node->nconns; i++) {
		node = hda_get_node(adc_node->conn_list[i]);
		if (node && node->type != AC_WID_PIN) {
			err = parse_adc_sub_nodes(node);
			if (err < 0)
				return err;
			else if (err > 0)
				add_cap_src(i);
		}
	}

	if (!input_mux.num_items)
		return 0; /* no input path found... */

	log_verbose("[Capture Source] NID=0x%x, #SRC=%d\n",
		    adc_node->nid, input_mux.num_items);
	for (i = 0; i < input_mux.num_items; i++)
		log_verbose("  [%s] IDX=0x%x\n", input_mux.items[i].label,
			    input_mux.items[i].index);

	adc_node = adc_node;
	return 1;
}

/*
 * parse input
 */
static int parse_input(void)
{
	struct hda_gnode *node;
	int err;

	/*
	 * At first we look for an audio input widget.
	 * If it reaches to certain input PINs, we take it as the
	 * input path.
	 */
	for (node = nid_list; node; node = node->next) {
		if (node->wid_caps & AC_WCAP_DIGITAL)
			continue; /* skip SPDIF */
		if (node->type == AC_WID_AUD_IN) {
			err = parse_input_path(node);
			if (err < 0)
				return err;
			else if (err > 0)
				return 0;
		}
	}
	log_info("generic: no proper input path found\n");
	return 0;
}

/*
 * create mixer controls if possible
 */
static int create_mixer(struct hda_gnode *node, unsigned int index,
			const char *type, const char *dir_sfx)
{
	struct hda_std_mixer knew[2];
	char name[32];
	int err;
	int created = 0;

	if (type)
		sprintf(name, "%s %s Switch", type, dir_sfx);
	else
		sprintf(name, "%s Switch", dir_sfx);
	memset(knew, 0, sizeof(knew));
	if ((node->wid_caps & AC_WCAP_IN_AMP) &&
	    (node->amp_in_caps & AC_AMPCAP_MUTE)) {
		knew[0] = (struct hda_std_mixer)
			HDA_CODEC_MUTE("", node->nid, index, HDA_INPUT);
		strcpy(knew[0].head.name, name);
		log_verbose("[%s] NID=0x%x, DIR=IN, IDX=0x%x\n",
			    name, node->nid, index);
		err = hda_register_mixers(knew);
		if (err < 0)
			return err;
		created = 1;
	} else if ((node->wid_caps & AC_WCAP_OUT_AMP) &&
		   (node->amp_out_caps & AC_AMPCAP_MUTE)) {
		knew[0] = (struct hda_std_mixer)
			HDA_CODEC_MUTE("", node->nid, 0, HDA_OUTPUT);
		strcpy(knew[0].head.name, name);
		log_verbose("[%s] NID=0x%x, DIR=OUT\n", name, node->nid);
		err = hda_register_mixers(knew);
		if (err < 0)
			return err;
		created = 1;
	}

	if (type)
		sprintf(name, "%s %s Volume", type, dir_sfx);
	else
		sprintf(name, "%s Volume", dir_sfx);
	if ((node->wid_caps & AC_WCAP_IN_AMP) &&
	    (node->amp_in_caps & AC_AMPCAP_NUM_STEPS)) {
		knew[0] = (struct hda_std_mixer)
			HDA_CODEC_VOLUME("", node->nid, index, HDA_INPUT);
		strcpy(knew[0].head.name, name);
		log_verbose("[%s] NID=0x%x, DIR=IN, IDX=0x%x\n",
			    name, node->nid, index);
		err = hda_register_mixers(knew);
		if (err < 0)
			return err;
		created = 1;
	} else if ((node->wid_caps & AC_WCAP_OUT_AMP) &&
		   (node->amp_out_caps & AC_AMPCAP_NUM_STEPS)) {
		knew[0] = (struct hda_std_mixer)
			HDA_CODEC_VOLUME("", node->nid, 0, HDA_OUTPUT);
		strcpy(knew[0].head.name, name);
		log_verbose("[%s] NID=0x%x, DIR=OUT\n", name, node->nid);
		err = hda_register_mixers(knew);
		if (err < 0)
			return err;
		created = 1;
	}

	return created;
}

/*
 * check whether the controls with the given name and direction suffix
 * already exist
 */
static int check_existing_control(const char *type, const char *dir)
{
	char name[64];

	sprintf(name, "%s %s Volume", type, dir);
	if (hda_look_for_ctl_id(name))
		return 1;
	sprintf(name, "%s %s Switch", type, dir);
	if (hda_look_for_ctl_id(name))
		return 1;
	return 0;
}

/*
 * build output mixer controls
 */
static int create_output_mixers(const char **names)
{
	int i, err;

	for (i = 0; i < pcm_vol_nodes; i++) {
		err = create_mixer(pcm_vol[i].node,
				   pcm_vol[i].index,
				   names[i], "Playback");
		if (err < 0)
			return err;
	}
	return 0;
}

static int build_output_controls(void)
{
	static const char *types_speaker[] = { "Speaker", "Headphone" };
	static const char *types_line[] = { "Front", "Headphone" };

	switch (pcm_vol_nodes) {
	case 1:
		return create_mixer(pcm_vol[0].node,
				    pcm_vol[0].index,
				    "Master", "Playback");
	case 2:
		if (defcfg_type(out_pin_node[0]) == AC_JACK_SPEAKER)
			return create_output_mixers(types_speaker);
		else
			return create_output_mixers(types_line);
	}
	return 0;
}

/* create capture volume/switch */
static int build_input_controls(void)
{
	int i, err;

	if (!adc_node || !input_mux.num_items)
		return 0; /* not found */

	select_input_connection(adc_node, input_mux.items[0].index);

	/* create capture volume and switch controls if the ADC has an amp */
	/* do we have only a single item? */
	if (input_mux.num_items == 1) {
		err = create_mixer(adc_node, input_mux.items[0].index,
				   NULL, "Capture");
		if (err < 0)
			return err;
		return 0;
	}

	/* create input MUX if multiple sources are available */
	err = hda_register_input_mux_mixer("Capture Source",
					   adc_node->nid, &input_mux);
	if (err < 0)
		return err;

	/* no volume control? */
	if (!(adc_node->wid_caps & AC_WCAP_IN_AMP) ||
	    !(adc_node->amp_in_caps & AC_AMPCAP_NUM_STEPS))
		return 0;

	for (i = 0; i < input_mux.num_items; i++) {
		struct hda_std_mixer capvol_mixers[2] = {
			HDA_CODEC_VOLUME("", adc_node->nid,
					 input_mux.items[i].index,
					 HDA_INPUT),
			{}
		};
		sprintf(capvol_mixers[0].head.name, "%s Capture Volume",
			input_mux.items[i].label);

		err = hda_register_mixers(capvol_mixers);
		if (err < 0)
			return err;
	}

	return 0;
}


/*
 * parse the nodes recursively until reach to the output PIN.
 *
 * returns 0 - if not found,
 *         1 - if found, but no mixer is created
 *         2 - if found and mixer was already created, (just skip)
 *         a negative error code
 */
static int parse_loopback_path(struct hda_gnode *node,
			       struct hda_gnode *dest_node,
			       const char *type)
{
	int i, err;

	if (node->checked)
		return 0;

	node->checked = 1;
	if (node == dest_node) {
		/* loopback connection found */
		return 1;
	}

	for (i = 0; i < node->nconns; i++) {
		struct hda_gnode *child = hda_get_node(node->conn_list[i]);
		if (!child)
			continue;
		err = parse_loopback_path(child, dest_node, type);
		if (err < 0)
			return err;
		else if (err >= 1) {
			if (err == 1) {
				err = create_mixer(node, i, type, "Playback");
				if (err < 0)
					return err;
				if (err > 0)
					return 2; /* ok, created */
				/* not created, maybe in the lower path */
				err = 1;
			}
			/* connect and unmute */
			if (node->nconns > 1)
				select_input_connection(node, i);
			unmute_input(node, i);
			unmute_output(node);
			return err;
		}
	}
	return 0;
}

/*
 * parse the tree and build the loopback controls
 */
static int build_loopback_controls(void)
{
	struct hda_gnode *node;
	int err;
	const char *type;

	if (!out_pin_node[0])
		return 0;

	for (node = nid_list; node; node = node->next) {
		if (node->type != AC_WID_PIN)
			continue;
		/* input capable? */
		if (!(node->pin_caps & AC_PINCAP_IN))
			return 0;
		type = get_input_type(node, NULL);
		if (type) {
			if (check_existing_control(type, "Playback"))
				continue;
			clear_check_flags();
			err = parse_loopback_path(out_pin_node[0],
						  node, type);
			if (err < 0)
				return err;
			if (!err)
				continue;
		}
	}
	return 0;
}

/*
 * build mixer controls
 */
static int build_generic_controls(void)
{
	int err;

	err = build_input_controls();
	if (err < 0)
		return err;
	err = build_output_controls();
	if (err < 0)
		return err;
	err = build_loopback_controls();
	if (err < 0)
		return err;

	return 0;
}

/*
 * PCM
 */
static struct hda_usr_pcm_info generic_pcm_playback = {
	.type = HDA_PCM_TYPE_ANALOG_MULTI_OUT,
	.substreams = 1,
	.channels_min = 2,
	.channels_max = 2,
	.name = "Generic PCM",
	.device = 0
};

static struct hda_usr_pcm_info generic_pcm_capture = {
	.type = HDA_PCM_TYPE_ANALOG_IN,
	.substreams = 1,
	.channels_min = 2,
	.channels_max = 2,
	.name = "Generic PCM",
	.device = 0
};

static int build_generic_pcms(void)
{
	int err;

	if (!dac_node[0] && !adc_node) {
		log_verbose("hda_generic: no PCM found\n");
		return 0;
	}

	if (dac_node[0]) {
		generic_pcm_playback.nid = dac_node[0]->nid;
		generic_pcm_playback.assoc_nids[0] = dac_node[0]->nid;
		if (dac_node[1])
			generic_pcm_playback.extra_nids[0] = dac_node[1]->nid;
		err = hda_register_pcm(&generic_pcm_playback);
		if (err < 0)
			return err;
	}
	if (adc_node) {
		generic_pcm_capture.nid = adc_node->nid;
		err = hda_register_pcm(&generic_pcm_capture);
		if (err < 0)
			return err;
	}

	return 0;
}


/*
 * the generic parser
 */
static int patch_generic(const struct hda_codec_table *tbl,
			 int config, const char **args)
{
	int err;

	if (!codec_info.afg) {
		log_error("No AFG is available\n");
		return -ENODEV;
	}

	err = build_afg_tree();
	if (err < 0)
		goto error;

	err = parse_input();
	if (err < 0)
		goto error;
	err = parse_output();
	if (err < 0)
		goto error;
	if (auto_init.num) {
		err = hda_register_init_verbs(auto_init.verbs);
		if (err < 0)
			goto error;
	}

	err = build_generic_controls();
	if (err < 0)
		goto error;
	err = build_generic_pcms();
	if (err < 0)
		goto error;

	return 0;

 error:
	patch_generic_free();
	return err;
}

static struct hda_codec_table generic_presets[] = {
	{ .id = -1, .patch = patch_generic },
	{ }
};

const struct hda_codec_table *patch_descriptor(void)
{
	return generic_presets;
}
