/*
 * ALC880 BIOS auto-configuration module
 *
 * Copyright (C) 2007 Takashi Iwai
 * 
 * Derived from patch_realtek.c:
 *
 * Copyright (c) 2004 Kailang Yang <kailang@realtek.com.tw>
 *                    PeiSen Hou <pshou@realtek.com.tw>
 *                    Takashi Iwai <tiwai@suse.de>
 *                    Jonathan Woithe <jwoithe@physics.adelaide.edu.au>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 */

#include "config.h"
#include <stdio.h>
#include <string.h>
#include "hda/codecs-helper.h"
#include "hda/pincfg.h"


#define ALC880_FRONT_NID	0x02
#define ALC880_ADC_NID		0x08	/* ADC2 (ADC1:0x07, ADC3:0x09) */
#define ALC880_DIGOUT_NID	0x06
#define ALC880_DIGIN_NID	0x0a


/*
 * generic initialization of ADC, input mixers and output mixers
 */
static struct hda_verb alc880_auto_init_verbs[] = {
	/*
	 * Unmute ADC0-2 and set the default input to mic-in
	 */
	{0x07, AC_VERB_SET_CONNECT_SEL, 0x00},
	{0x07, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_UNMUTE(0)},
	{0x08, AC_VERB_SET_CONNECT_SEL, 0x00},
	{0x08, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_UNMUTE(0)},
	{0x09, AC_VERB_SET_CONNECT_SEL, 0x00},
	{0x09, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_UNMUTE(0)},

	/* Unmute input amps (CD, Line In, Mic 1 & Mic 2) of the analog-loopback
	 * mixer widget
	 * Note: PASD motherboards uses the Line In 2 as the input for front
	 * panel mic (mic 2)
	 */
	/* Amp Indices: Mic1 = 0, Mic2 = 1, Line1 = 2, Line2 = 3, CD = 4 */
	{0x0b, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_UNMUTE(0)},
	{0x0b, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_UNMUTE(1)},
	{0x0b, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_UNMUTE(2)},
	{0x0b, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_UNMUTE(3)},
	{0x0b, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_UNMUTE(4)},

	/*
	 * Set up output mixers (0x0c - 0x0f)
	 */
	/* set vol=0 to output mixers */
	{0x0c, AC_VERB_SET_AMP_GAIN_MUTE, AMP_OUT_ZERO},
	{0x0d, AC_VERB_SET_AMP_GAIN_MUTE, AMP_OUT_ZERO},
	{0x0e, AC_VERB_SET_AMP_GAIN_MUTE, AMP_OUT_ZERO},
	{0x0f, AC_VERB_SET_AMP_GAIN_MUTE, AMP_OUT_ZERO},
	/* set up input amps for analog loopback */
	/* Amp Indices: DAC = 0, mixer = 1 */
	{0x0c, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_MUTE(0)},
	{0x0c, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_MUTE(1)},
	{0x0d, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_MUTE(0)},
	{0x0d, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_MUTE(1)},
	{0x0e, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_MUTE(0)},
	{0x0e, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_MUTE(1)},
	{0x0f, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_MUTE(0)},
	{0x0f, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_MUTE(1)},

	{ }
};

/* Enable GPIO mask and set output */
static struct hda_verb alc_gpio1_init_verbs[] = {
	{0x01, AC_VERB_SET_GPIO_MASK, 0x01},
	{0x01, AC_VERB_SET_GPIO_DIRECTION, 0x01},
	{0x01, AC_VERB_SET_GPIO_DATA, 0x01},
	{ }
};

static struct hda_verb alc_gpio2_init_verbs[] = {
	{0x01, AC_VERB_SET_GPIO_MASK, 0x02},
	{0x01, AC_VERB_SET_GPIO_DIRECTION, 0x02},
	{0x01, AC_VERB_SET_GPIO_DATA, 0x02},
	{ }
};

static struct hda_verb alc_gpio3_init_verbs[] = {
	{0x01, AC_VERB_SET_GPIO_MASK, 0x03},
	{0x01, AC_VERB_SET_GPIO_DIRECTION, 0x03},
	{0x01, AC_VERB_SET_GPIO_DATA, 0x03},
	{ }
};

/*
 */
#define alc880_is_fixed_pin(nid)	((nid) >= 0x14 && (nid) <= 0x17)
#define alc880_fixed_pin_idx(nid)	((nid) - 0x14)
#define alc880_is_multi_pin(nid)	((nid) >= 0x18)
#define alc880_multi_pin_idx(nid)	((nid) - 0x18)
#define alc880_is_input_pin(nid)	((nid) >= 0x18)
#define alc880_input_pin_idx(nid)	((nid) - 0x18)
#define alc880_idx_to_dac(nid)		((nid) + 0x02)
#define alc880_dac_to_idx(nid)		((nid) - 0x02)
#define alc880_idx_to_mixer(nid)	((nid) + 0x0c)
#define alc880_idx_to_selector(nid)	((nid) + 0x10)
#define ALC880_PIN_CD_NID		0x1c

/* fill in the dac_nids table from the parsed pin configuration */
static int alc880_auto_fill_dac_nids(const struct hda_auto_pin_cfg *cfg,
				     hda_nid_t *dac_nids)
{
	hda_nid_t nid;
	int assigned[4];
	int i, j;

	memset(assigned, 0, sizeof(assigned));
	/* check the pins hardwired to audio widget */
	for (i = 0; i < cfg->line_outs; i++) {
		nid = cfg->line_out_pins[i];
		if (alc880_is_fixed_pin(nid)) {
			int idx = alc880_fixed_pin_idx(nid);
			dac_nids[i] = alc880_idx_to_dac(idx);
			assigned[idx] = 1;
		}
	}
	/* left pins can be connect to any audio widget */
	for (i = 0; i < cfg->line_outs; i++) {
		nid = cfg->line_out_pins[i];
		if (alc880_is_fixed_pin(nid))
			continue;
		/* search for an empty channel */
		for (j = 0; j < cfg->line_outs; j++) {
			if (!assigned[j]) {
				dac_nids[i] = alc880_idx_to_dac(j);
				assigned[j] = 1;
				break;
			}
		}
	}
	return 0;
}

/* Add a channel control */
static int add_mixer_channel(hda_nid_t nid, const char *pfx, u32 *binds,
			     struct hda_mixer_array *mix)
{
	char name[32];
	int err;

	sprintf(name, "%s Playback Volume", pfx);
	err = hda_append_mixer(mix, HDA_MIXER_TYPE_VOLUME, name,
			       HDA_COMPOSE_AMP_VAL(nid, 3, 0,
						   HDA_OUTPUT));
	if (err < 0)
		return err;
	sprintf(name, "%s Playback Switch", pfx);
	err = hda_append_mixer(mix, HDA_MIXER_TYPE_BIND_SWITCH, name,
			       (long) binds);
	if (err < 0)
		return err;
	return 0;
}

/* Add Center/LFE control */
static int add_mixer_clfe(hda_nid_t nid, struct hda_mixer_array *mix)
{
	int err;
	static u32 center_bind_sws[3];
	static u32 lfe_bind_sws[3];

	err = hda_append_mixer(mix, HDA_MIXER_TYPE_VOLUME,
			       "Center Playback Volume",
			       HDA_COMPOSE_AMP_VAL(nid, 1, 0,
						   HDA_OUTPUT));
	if (err < 0)
		return err;
	err = hda_append_mixer(mix, HDA_MIXER_TYPE_VOLUME,
			       "LFE Playback Volume",
			       HDA_COMPOSE_AMP_VAL(nid, 2, 0,
						   HDA_OUTPUT));
	if (err < 0)
		return err;
	center_bind_sws[0] = HDA_COMPOSE_AMP_VAL(nid, 1, 0, HDA_INPUT);
	center_bind_sws[1] = HDA_COMPOSE_AMP_VAL(nid, 1, 1, HDA_INPUT);
	center_bind_sws[2] = 0;

	err = hda_append_mixer(mix, HDA_MIXER_TYPE_BIND_SWITCH,
			       "Center Playback Switch",
			       (long) center_bind_sws);
	if (err < 0)
		return err;

	lfe_bind_sws[0] = HDA_COMPOSE_AMP_VAL(nid, 2, 0, HDA_INPUT);
	lfe_bind_sws[1] = HDA_COMPOSE_AMP_VAL(nid, 2, 1, HDA_INPUT);
	lfe_bind_sws[2] = 0;
	err = hda_append_mixer(mix, HDA_MIXER_TYPE_BIND_SWITCH,
			       "LFE Playback Switch",
			       (long) lfe_bind_sws);
	if (err < 0)
		return err;
	return 0;
}

/* add playback controls from the parsed DAC table */
static int alc880_auto_create_multi_out_ctls(const struct hda_auto_pin_cfg *cfg,
					     hda_nid_t *dac_nids,
					     struct hda_mixer_array *mix)
{
	static const char *chname[4] = {
		"Front", "Surround", NULL /*CLFE*/, "Side"
	};
	static u32 binds[4][3];
	hda_nid_t nid;
	int i, err;

	for (i = 0; i < cfg->line_outs; i++) {
		if (!dac_nids[i])
			continue;
		nid = alc880_idx_to_mixer(alc880_dac_to_idx(dac_nids[i]));
		if (i == 2)
			err = add_mixer_clfe(nid, mix);
		else
			err = add_mixer_channel(nid, chname[i],
						&binds[i][0], mix);
		if (err < 0)
			return err;
	}
	return 0;
}

/* add playback controls for speaker and HP outputs */
static int alc880_auto_create_extra_out(hda_nid_t pin, const char *pfx,
					hda_nid_t *extra_nids,
					struct hda_mixer_array *mix)
{
	hda_nid_t nid;
	int err;
	static u32 binds[2][3];

	if (!pin)
		return 0;

	if (alc880_is_fixed_pin(pin)) {
		int eid;
		nid = alc880_idx_to_dac(alc880_fixed_pin_idx(pin));
		/* specify the DAC as the extra output */
		if (!extra_nids[0])
			eid = 0;
		else
			eid = 1;
		extra_nids[eid] = nid;
		/* control HP volume/switch on the output mixer amp */
		nid = alc880_idx_to_mixer(alc880_fixed_pin_idx(pin));
		err = add_mixer_channel(nid, pfx, &binds[eid][0], mix);
		if (err < 0)
			return err;
	} else if (alc880_is_multi_pin(pin)) {
		/* set manual connection */
		/* we have only a switch on HP-out PIN */
		char name[32];
		sprintf(name, "%s Playback Switch", pfx);
		err = hda_append_mixer(mix, HDA_MIXER_TYPE_SWITCH, name,
				       HDA_COMPOSE_AMP_VAL(pin, 3, 0, HDA_OUTPUT));
		if (err < 0)
			return err;
	}
	return 0;
}

/* create input playback/capture controls for the given pin */
static int new_analog_input(hda_nid_t pin, const char *ctlname,
			    int idx, hda_nid_t mix_nid,
			    struct hda_mixer_array *mixers)
{
	char name[32];
	int err;

	sprintf(name, "%s Playback Volume", ctlname);
	err = hda_append_mixer(mixers, HDA_MIXER_TYPE_VOLUME, name,
			       HDA_COMPOSE_AMP_VAL(mix_nid, 3, idx, HDA_INPUT));
	if (err < 0)
		return err;
	sprintf(name, "%s Playback Switch", ctlname);
	err = hda_append_mixer(mixers, HDA_MIXER_TYPE_SWITCH, name,
			       HDA_COMPOSE_AMP_VAL(mix_nid, 3, idx, HDA_INPUT));
	if (err < 0)
		return err;
	return 0;
}

/* capture mixer elements */
static struct hda_std_mixer alc880_capture_mixer[] = {
	HDA_CODEC_VOLUME("Capture Volume", 0x08, 0x0, HDA_INPUT),
	HDA_CODEC_MUTE("Capture Switch", 0x08, 0x0, HDA_INPUT),
	{ } /* end */
};

/* create playback/capture controls for input pins */
static int alc880_auto_create_analog_input_ctls(const struct hda_auto_pin_cfg *cfg,
						struct hda_input_mux *imux,
						struct hda_mixer_array *mixers)
{
	int i, err, idx;

	err = hda_append_mixers(mixers, alc880_capture_mixer);
	if (err < 0)
		return err;

	for (i = 0; i < AUTO_PIN_LAST; i++) {
		if (alc880_is_input_pin(cfg->input_pins[i])) {
			idx = alc880_input_pin_idx(cfg->input_pins[i]);
			err = new_analog_input(cfg->input_pins[i],
					       hda_auto_pin_cfg_labels[i],
					       idx, 0x0b, mixers);
			if (err < 0)
				return err;
			imux->items[imux->num_items].label =
				hda_auto_pin_cfg_labels[i];
			imux->items[imux->num_items].index =
				alc880_input_pin_idx(cfg->input_pins[i]);
			imux->num_items++;
		}
	}
	return 0;
}

static int alc880_auto_set_output_and_unmute(struct hda_verb_array *init,
					     hda_nid_t nid, int pin_type,
					     int dac_idx, hda_nid_t *dac_nids)
{
	/* set as output */
	struct hda_verb verbs[] = {
		{ nid, AC_VERB_SET_PIN_WIDGET_CONTROL, pin_type },
		{ nid, AC_VERB_SET_AMP_GAIN_MUTE, AMP_OUT_UNMUTE },
		{ }
	};
	int err;

	err = hda_append_verbs(init, verbs);
	if (err < 0)
		return err;
	/* need the manual connection? */
	if (alc880_is_multi_pin(nid)) {
		int idx = alc880_multi_pin_idx(nid);
		return hda_append_verb(init, alc880_idx_to_selector(idx),
				       AC_VERB_SET_CONNECT_SEL,
				       alc880_dac_to_idx(dac_nids[dac_idx]));
	}
	return 0;
}

static int alc_subsystem_id(struct hda_verb_array *init,
			    unsigned int porta, unsigned int porte,
			    unsigned int portd)
{
	unsigned int ass, tmp;
	int err;

	ass = codec_info.subsystem_id;
	if (!(ass & 1))
		return 0;

	/* Override */
	tmp = (ass & 0x38) >> 3;	/* external Amp control */
	switch (tmp) {
	case 1:
		err = hda_append_verbs(init, alc_gpio1_init_verbs);
		if (err < 0)
			return err;
		break;
	case 3:
		err = hda_append_verbs(init, alc_gpio2_init_verbs);
		if (err < 0)
			return err;
		break;
	case 7:
		err = hda_append_verbs(init, alc_gpio3_init_verbs);
		if (err < 0)
			return err;
		break;
	case 5:
	case 6:
		if (ass & 4) {	/* bit 2 : 0 = Desktop, 1 = Laptop */
			hda_nid_t port = 0;
			tmp = (ass & 0x1800) >> 11;
			switch (tmp) {
			case 0: port = porta; break;
			case 1: port = porte; break;
			case 2: port = portd; break;
			}
			if (port) {
				err = hda_append_verb(init, port,
						      AC_VERB_SET_EAPD_BTLENABLE,
						      2);
				if (err < 0)
					return err;
			}
		}
		err = hda_append_verb(init, 0x1a, AC_VERB_SET_COEF_INDEX, 7);
		if (err < 0)
			return err;
		err = hda_append_verb(init, 0x1a, AC_VERB_SET_PROC_COEF,
				      (tmp == 5 ? 0x3040 : 0x3050));
		if (err < 0)
			return err;
		break;
	}
	return 0;
}

static int get_pin_type(int line_out_type)
{
	if (line_out_type == AUTO_PIN_HP_OUT)
		return PIN_HP;
	else
		return PIN_OUT;
}

static int alc880_auto_init_multi_out(struct hda_auto_pin_cfg *cfg,
				      hda_nid_t *dac_nids,
				      struct hda_verb_array *init)
{
	int i, err;
	
	err = alc_subsystem_id(init, 0x15, 0x1b, 0x14);
	if (err < 0)
		return err;
	for (i = 0; i < cfg->line_outs; i++) {
		hda_nid_t nid = cfg->line_out_pins[i];
		int pin_type = get_pin_type(cfg->line_out_type);
		err = alc880_auto_set_output_and_unmute(init, nid, pin_type, i,
							dac_nids);
		if (err < 0)
			return err;
	}
	return 0;
}

static int alc880_auto_init_extra_out(struct hda_auto_pin_cfg *cfg,
				      hda_nid_t *dac_nids,
				      struct hda_verb_array *init)
{
	hda_nid_t pin;
	int err;

	pin = cfg->speaker_pins[0];
	if (pin) {
		/* connect to front */
		err = alc880_auto_set_output_and_unmute(init, pin, PIN_OUT, 0,
							dac_nids);
		if (err < 0)
			return err;
	}
	pin = cfg->hp_pins[0];
	if (pin) {
		/* connect to front */
		err = alc880_auto_set_output_and_unmute(init, pin, PIN_HP, 0,
							dac_nids);
		if (err < 0)
			return err;
	}
	return 0;
}

static int alc880_auto_init_analog_input(struct hda_auto_pin_cfg *cfg,
					 struct hda_verb_array *init)
{
	int i, err;

	for (i = 0; i < AUTO_PIN_LAST; i++) {
		hda_nid_t nid = cfg->input_pins[i];
		if (alc880_is_input_pin(nid)) {
			err = hda_append_verb(init, nid, 
					      AC_VERB_SET_PIN_WIDGET_CONTROL,
					      i <= AUTO_PIN_FRONT_MIC ?
					      PIN_VREF80 : PIN_IN);
			if (err < 0)
				return err;
			if (nid == ALC880_PIN_CD_NID)
				continue;
			err = hda_append_verb(init, nid,
					      AC_VERB_SET_AMP_GAIN_MUTE,
					      AMP_OUT_MUTE);
			if (err < 0)
				return err;
		}
	}
	return 0;
}


/*
 */

static struct hda_verb_array auto_init;
static struct hda_mixer_array auto_mix;
static struct hda_auto_pin_cfg autocfg;
static hda_nid_t dac_nids[5];
static hda_nid_t extra_nids[5];
static struct hda_input_mux auto_input_mux;

static struct codec_config_preset preset = {
	.init_verbs = { alc880_auto_init_verbs },
	.adc_nid = ALC880_ADC_NID,
	.input_mux = &auto_input_mux,
	.input_mux_nid = ALC880_ADC_NID,
};

static int patch_alc880_auto(const struct hda_codec_table *tbl,
			     int board_config, const char **args)
{
	int err;
	static hda_nid_t alc880_ignore[] = { 0x1d, 0 };

	err = hda_parse_pin_def_config(&autocfg,
				       alc880_ignore);
	if (err < 0)
		return err;
	if (!autocfg.line_outs)
		return -ENODEV; /* can't find valid BIOS pin config */

	err = alc880_auto_fill_dac_nids(&autocfg, dac_nids);
	if (err < 0)
		goto error;
	err = alc880_auto_create_multi_out_ctls(&autocfg, dac_nids,
						&auto_mix);
	if (err < 0)
		goto error;
	err = alc880_auto_create_extra_out(autocfg.speaker_pins[0],
					   "Speaker", extra_nids,
					   &auto_mix);
	if (err < 0)
		goto error;
	err = alc880_auto_create_extra_out(autocfg.hp_pins[0],
					   "Headphone", extra_nids,
					   &auto_mix);
	if (err < 0)
		goto error;
	err = alc880_auto_create_analog_input_ctls(&autocfg,
						   &auto_input_mux,
						   &auto_mix);
	if (err < 0)
		goto error;

	err = alc880_auto_init_multi_out(&autocfg, dac_nids, &auto_init);
	if (err < 0)
		goto error;
	err = alc880_auto_init_extra_out(&autocfg, dac_nids, &auto_init);
	if (err < 0)
		goto error;
	err = alc880_auto_init_analog_input(&autocfg, &auto_init);
	if (err < 0)
		goto error;

	preset.num_dacs = autocfg.line_outs;
	preset.dac_nids = dac_nids;
	preset.extra_nids = extra_nids;
	if (autocfg.dig_out_pin)
		preset.dig_out_nid = ALC880_DIGOUT_NID;
	if (autocfg.dig_in_pin)
		preset.dig_in_nid = ALC880_DIGIN_NID;

	err = codec_preset_add_init_verb(&preset, auto_init.verbs);
	if (err < 0)
		goto error;
	err = codec_preset_add_mixer(&preset, auto_mix.mixers);
	if (err < 0)
		goto error;

	err = codec_build_preset(tbl, &preset);
	
	error:
	hda_clear_verb_array(&auto_init);
	hda_clear_mixer_array(&auto_mix);
	return err;
}

/*
 */

static struct hda_codec_table alc880_auto_table[] = {
	{ .id = 0x10ec0880, .vendor_name = "Realtek", .name = "ALC880",
	  .patch = patch_alc880_auto },
};

const struct hda_codec_table *patch_descriptor(void)
{
	return alc880_auto_table;
}
