/*
 * ALC880 BIOS auto-configuration module
 *
 * Copyright (C) 2007 Takashi Iwai
 * 
 * Derived from patch_realtek.c:
 *
 * Copyright (c) 2004 Kailang Yang <kailang@realtek.com.tw>
 *                    PeiSen Hou <pshou@realtek.com.tw>
 *                    Takashi Iwai <tiwai@suse.de>
 *                    Jonathan Woithe <jwoithe@physics.adelaide.edu.au>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 */

#include "config.h"
#include <stdio.h>
#include <string.h>
#include "hda/codecs-helper.h"
#include "hda/pincfg.h"


#define ALC260_DAC_NID		0x02
#define ALC260_ADC_NID		0x05	/* ADC1, ADC0 = 0x04 */
#define ALC260_DIGOUT_NID	0x03
#define ALC260_DIGIN_NID	0x06

/* Enable GPIO mask and set output */
static struct hda_verb alc_gpio1_init_verbs[] = {
	{0x01, AC_VERB_SET_GPIO_MASK, 0x01},
	{0x01, AC_VERB_SET_GPIO_DIRECTION, 0x01},
	{0x01, AC_VERB_SET_GPIO_DATA, 0x01},
	{ }
};

static struct hda_verb alc_gpio2_init_verbs[] = {
	{0x01, AC_VERB_SET_GPIO_MASK, 0x02},
	{0x01, AC_VERB_SET_GPIO_DIRECTION, 0x02},
	{0x01, AC_VERB_SET_GPIO_DATA, 0x02},
	{ }
};

static struct hda_verb alc_gpio3_init_verbs[] = {
	{0x01, AC_VERB_SET_GPIO_MASK, 0x03},
	{0x01, AC_VERB_SET_GPIO_DIRECTION, 0x03},
	{0x01, AC_VERB_SET_GPIO_DATA, 0x03},
	{ }
};

static int alc_subsystem_id(struct hda_verb_array *init,
			    unsigned int porta, unsigned int porte,
			    unsigned int portd)
{
	unsigned int ass, tmp;
	int err;

	ass = codec_info.subsystem_id;
	if (!(ass & 1))
		return 0;

	/* Override */
	tmp = (ass & 0x38) >> 3;	/* external Amp control */
	switch (tmp) {
	case 1:
		err = hda_append_verbs(init, alc_gpio1_init_verbs);
		if (err < 0)
			return err;
		break;
	case 3:
		err = hda_append_verbs(init, alc_gpio2_init_verbs);
		if (err < 0)
			return err;
		break;
	case 7:
		err = hda_append_verbs(init, alc_gpio3_init_verbs);
		if (err < 0)
			return err;
		break;
	case 5:
	case 6:
		if (ass & 4) {	/* bit 2 : 0 = Desktop, 1 = Laptop */
			hda_nid_t port = 0;
			tmp = (ass & 0x1800) >> 11;
			switch (tmp) {
			case 0: port = porta; break;
			case 1: port = porte; break;
			case 2: port = portd; break;
			}
			if (port) {
				err = hda_append_verb(init, port,
						      AC_VERB_SET_EAPD_BTLENABLE,
						      2);
				if (err < 0)
					return err;
			}
		}
		err = hda_append_verb(init, 0x1a, AC_VERB_SET_COEF_INDEX, 7);
		if (err < 0)
			return err;
		err = hda_append_verb(init, 0x1a, AC_VERB_SET_PROC_COEF,
				      (tmp == 5 ? 0x3040 : 0x3050));
		if (err < 0)
			return err;
		break;
	}
	return 0;
}


static int alc260_add_playback_controls(hda_nid_t nid, const char *pfx,
					struct hda_mixer_array *mix)
{
	hda_nid_t nid_vol;
	u32 vol_val, sw_val;
	char name[32];
	int err;

	if (nid >= 0x0f && nid < 0x11) {
		nid_vol = nid - 0x7;
		vol_val = HDA_COMPOSE_AMP_VAL(nid_vol, 3, 0, HDA_OUTPUT);
		sw_val = HDA_COMPOSE_AMP_VAL(nid, 3, 0, HDA_OUTPUT);
	} else if (nid == 0x11) {
		nid_vol = nid - 0x7;
		vol_val = HDA_COMPOSE_AMP_VAL(nid_vol, 2, 0, HDA_OUTPUT);
		sw_val = HDA_COMPOSE_AMP_VAL(nid, 2, 0, HDA_OUTPUT);
	} else if (nid >= 0x12 && nid <= 0x15) {
		nid_vol = 0x08;
		vol_val = HDA_COMPOSE_AMP_VAL(nid_vol, 3, 0, HDA_OUTPUT);
		sw_val = HDA_COMPOSE_AMP_VAL(nid, 3, 0, HDA_OUTPUT);
	} else
		return 0; /* N/A */
	
	snprintf(name, sizeof(name), "%s Playback Volume", pfx);
	err = hda_append_mixer(mix, HDA_MIXER_TYPE_VOLUME, name, vol_val);
	if (err < 0)
		return err;
	snprintf(name, sizeof(name), "%s Playback Switch", pfx);
	err = hda_append_mixer(mix, HDA_MIXER_TYPE_SWITCH, name, sw_val);
	if (err < 0)
		return err;
	return 0;
}

/* add playback controls from the parsed DAC table */
static int alc260_auto_create_multi_out_ctls(const struct hda_auto_pin_cfg *cfg,
					     struct hda_mixer_array *mix)
{
	hda_nid_t nid;
	int err;

	nid = cfg->line_out_pins[0];
	if (nid) {
		err = alc260_add_playback_controls(nid, "Front", mix);
		if (err < 0)
			return err;
	}

	nid = cfg->speaker_pins[0];
	if (nid) {
		err = alc260_add_playback_controls(nid, "Speaker", mix);
		if (err < 0)
			return err;
	}

	nid = cfg->hp_pins[0];
	if (nid) {
		err = alc260_add_playback_controls(nid, "Headphone", mix);
		if (err < 0)
			return err;
	}
	return 0;
}

/* create input playback/capture controls for the given pin */
static int new_analog_input(hda_nid_t pin, const char *ctlname,
			    int idx, hda_nid_t mix_nid,
			    struct hda_mixer_array *mixers)
{
	char name[32];
	int err;

	sprintf(name, "%s Playback Volume", ctlname);
	err = hda_append_mixer(mixers, HDA_MIXER_TYPE_VOLUME, name,
			       HDA_COMPOSE_AMP_VAL(mix_nid, 3, idx, HDA_INPUT));
	if (err < 0)
		return err;
	sprintf(name, "%s Playback Switch", ctlname);
	err = hda_append_mixer(mixers, HDA_MIXER_TYPE_SWITCH, name,
			       HDA_COMPOSE_AMP_VAL(mix_nid, 3, idx, HDA_INPUT));
	if (err < 0)
		return err;
	return 0;
}

static struct hda_std_mixer alc260_capture_mixer[] = {
	HDA_CODEC_VOLUME("Capture Volume", 0x05, 0x0, HDA_INPUT),
	HDA_CODEC_MUTE("Capture Switch", 0x05, 0x0, HDA_INPUT),
	{ } /* end */
};

/* create playback/capture controls for input pins */
static int alc260_auto_create_analog_input_ctls(const struct hda_auto_pin_cfg *cfg,
						struct hda_input_mux *imux,
						struct hda_mixer_array *mix)
{
	int i, err, idx;

	err = hda_append_mixers(mix, alc260_capture_mixer);
	if (err < 0)
		return err;

	for (i = 0; i < AUTO_PIN_LAST; i++) {
		if (cfg->input_pins[i] >= 0x12) {
			idx = cfg->input_pins[i] - 0x12;
			err = new_analog_input(cfg->input_pins[i],
					       hda_auto_pin_cfg_labels[i], idx,
					       0x07, mix);
			if (err < 0)
				return err;
			imux->items[imux->num_items].label =
				hda_auto_pin_cfg_labels[i];
			imux->items[imux->num_items].index = idx;
			imux->num_items++;
		}
		if (cfg->input_pins[i] >= 0x0f && cfg->input_pins[i] <= 0x10){
			idx = cfg->input_pins[i] - 0x09;
			err = new_analog_input(cfg->input_pins[i],
					       hda_auto_pin_cfg_labels[i], idx,
					       0x07, mix);
			if (err < 0)
				return err;
			imux->items[imux->num_items].label =
				hda_auto_pin_cfg_labels[i];
			imux->items[imux->num_items].index = idx;
			imux->num_items++;
		}
	}
	return 0;
}

static int alc260_auto_set_output_and_unmute(struct hda_verb_array *init,
					     hda_nid_t nid, int pin_type,
					     int sel_idx)
{
	int err;

	/* set as output */
	err = hda_append_verb(init,
			      nid, AC_VERB_SET_PIN_WIDGET_CONTROL, pin_type);
	if (err < 0)
		return err;
	err = hda_append_verb(init, nid, AC_VERB_SET_AMP_GAIN_MUTE,
			      AMP_OUT_UNMUTE);
	if (err < 0)
		return err;
	/* need the manual connection? */
	if (nid >= 0x12) {
		int idx = nid - 0x12;
		err = hda_append_verb(init, idx + 0x0b,
				      AC_VERB_SET_CONNECT_SEL, sel_idx);
		if (err < 0)
			return err;
	}
	return 0;
}

static int get_pin_type(int line_out_type)
{
	if (line_out_type == AUTO_PIN_HP_OUT)
		return PIN_HP;
	else
		return PIN_OUT;
}

static int alc260_auto_init_multi_out(const struct hda_auto_pin_cfg *cfg,
				      struct hda_verb_array *init)
{
	hda_nid_t nid;
	int err;

	alc_subsystem_id(init, 0x10, 0x15, 0x0f);
	nid = cfg->line_out_pins[0];
	if (nid) {
		int pin_type = get_pin_type(cfg->line_out_type);
		err = alc260_auto_set_output_and_unmute(init, nid, pin_type, 0);
		if (err < 0)
			return err;
	}
	
	nid = cfg->speaker_pins[0];
	if (nid) {
		err = alc260_auto_set_output_and_unmute(init, nid, PIN_OUT, 0);
		if (err < 0)
			return err;
	}

	nid = cfg->hp_pins[0];
	if (nid) {
		err = alc260_auto_set_output_and_unmute(init, nid, PIN_HP, 0);
		if (err < 0)
			return err;
	}
	return 0;
}

#define ALC260_PIN_CD_NID		0x16
static int alc260_auto_init_analog_input(const struct hda_auto_pin_cfg *cfg,
					 struct hda_verb_array *init)
{
	int i, err;

	for (i = 0; i < AUTO_PIN_LAST; i++) {
		hda_nid_t nid = cfg->input_pins[i];
		if (nid >= 0x12) {
			err = hda_append_verb(init, nid,
					      AC_VERB_SET_PIN_WIDGET_CONTROL,
					      i <= AUTO_PIN_FRONT_MIC ?
					      PIN_VREF80 : PIN_IN);
			if (err < 0)
				return err;
			if (nid == ALC260_PIN_CD_NID)
				continue;
			err = hda_append_verb(init, nid,
					      AC_VERB_SET_AMP_GAIN_MUTE,
					      AMP_OUT_MUTE);
			if (err < 0)
				return err;
		}
	}
	return 0;
}

/*
 * generic initialization of ADC, input mixers and output mixers
 */
static struct hda_verb alc260_volume_init_verbs[] = {
	/*
	 * Unmute ADC0-1 and set the default input to mic-in
	 */
	{0x04, AC_VERB_SET_CONNECT_SEL, 0x00},
	{0x04, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_UNMUTE(0)},
	{0x05, AC_VERB_SET_CONNECT_SEL, 0x00},
	{0x05, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_UNMUTE(0)},
	
	/* Unmute input amps (CD, Line In, Mic 1 & Mic 2) of the analog-loopback
	 * mixer widget
	 * Note: PASD motherboards uses the Line In 2 as the input for
	 * front panel mic (mic 2)
	 */
	/* Amp Indices: Mic1 = 0, Mic2 = 1, Line1 = 2, Line2 = 3, CD = 4 */
	{0x07, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_UNMUTE(0)},
	{0x07, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_UNMUTE(1)},
	{0x07, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_UNMUTE(2)},
	{0x07, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_UNMUTE(3)},
	{0x07, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_UNMUTE(4)},

	/*
	 * Set up output mixers (0x08 - 0x0a)
	 */
	/* set vol=0 to output mixers */
	{0x08, AC_VERB_SET_AMP_GAIN_MUTE, AMP_OUT_ZERO},
	{0x09, AC_VERB_SET_AMP_GAIN_MUTE, AMP_OUT_ZERO},
	{0x0a, AC_VERB_SET_AMP_GAIN_MUTE, AMP_OUT_ZERO},
	/* set up input amps for analog loopback */
	/* Amp Indices: DAC = 0, mixer = 1 */
	{0x08, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_UNMUTE(0)},
	{0x08, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_UNMUTE(1)},
	{0x09, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_UNMUTE(0)},
	{0x09, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_UNMUTE(1)},
	{0x0a, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_UNMUTE(0)},
	{0x0a, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_UNMUTE(1)},
	
	{ }
};

/*
 */

static struct hda_verb_array auto_init;
static struct hda_mixer_array auto_mix;
static struct hda_auto_pin_cfg autocfg;
static struct hda_input_mux auto_input_mux;

static hda_nid_t alc260_dacs[1] = { ALC260_DAC_NID };

static struct codec_config_preset preset = {
	.init_verbs = { alc260_volume_init_verbs },
	.num_dacs = 1,
	.dac_nids = alc260_dacs,
	.adc_nid = ALC260_ADC_NID,
	.input_mux = &auto_input_mux,
	.input_mux_nid = ALC260_ADC_NID,
};

static int patch_alc260_auto(const struct hda_codec_table *tbl,
			     int board_config, const char **args)
{
	int err;
	static hda_nid_t alc260_ignore[] = { 0x17, 0 };

	err = hda_parse_pin_def_config(&autocfg, alc260_ignore);
	if (err < 0)
		return err;

	err = alc260_auto_create_multi_out_ctls(&autocfg, &auto_mix);
	if (err < 0)
		return err;
	err = alc260_auto_create_analog_input_ctls(&autocfg, &auto_input_mux,
						   &auto_mix);
	if (err < 0)
		return err;

	err = alc260_auto_init_multi_out(&autocfg, &auto_init);
	if (err < 0)
		return err;
	err = alc260_auto_init_analog_input(&autocfg, &auto_init);
	if (err < 0)
		return err;

	if (autocfg.dig_out_pin)
		preset.dig_out_nid = ALC260_DIGOUT_NID;

	err = codec_preset_add_init_verb(&preset, auto_init.verbs);
	if (err < 0)
		goto error;
	err = codec_preset_add_mixer(&preset, auto_mix.mixers);
	if (err < 0)
		goto error;

	err = codec_build_preset(tbl, &preset);
	
	error:
	hda_clear_verb_array(&auto_init);
	hda_clear_mixer_array(&auto_mix);
	return err;
}

/*
 */

static struct hda_codec_table alc260_auto_table[] = {
	{ .id = 0x10ec0260, .vendor_name = "Realtek", .name = "ALC260",
	  .patch = patch_alc260_auto },
};

const struct hda_codec_table *patch_descriptor(void)
{
	return alc260_auto_table;
}
