/*
 * HD-audio user-space configurator
 *
 * Copyright (C) 2007 Takashi Iwai
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 */

#include "config.h"
#include <stdio.h>
#include <getopt.h>
#include "local.h"

static struct option opts[] = {
	{ "force", 0, NULL, 'f' },
	{ "module", 1, NULL, 'm' },
	{ "verbose", 0, NULL, 'v' },
	{ "reset", 0, NULL, 'r' },
	{ "usage", 0, NULL, 'h' },
	{ 0, 0, 0, 0 },
};

#define PROGNAME	"hda-config"

static void usage(void)
{
	fprintf(stderr, "%s - HD-audio user-space configuration tool\n",
		PROGNAME);
	fprintf(stderr, "version %s\n", VERSION);
	fprintf(stderr, "usage: %s [-h] [-v] [-f] [-r] [-m module] "
		"hwdep-device [arguments...]\n", PROGNAME);
}

int main(int argc, char **argv)
{
	int c;
	const char *devfile;
	const char **args = NULL;
	const char *module = NULL;
	int force = 0;
	int reset = 0;

	progname = PROGNAME;
	while ((c = getopt_long(argc, argv, "fhm:rv", opts, NULL)) != -1) {
		switch (c) {
		case 'f':
			force = 1;
			break;
		case 'm':
			module = optarg;
			break;
		case 'v':
			verbose_mode++;
			break;
		case 'r':
			reset = 1;
			break;
		case 'h':
			usage();
			return 0;
		default:
			usage();
			return 1;
		}
	}

	if (optind >= argc) {
		fprintf(stderr, "%s: no device file is specified\n", progname);
		return 1;
	}
	devfile = argv[optind];
	optind++;
	if (optind < argc)
		args = (const char **)(argv + optind);

	if (!hda_open_devfile(devfile))
		return 1;

	if (hda_is_running()) {
		if (!force && !reset) {
			log_error("Device %s already set up\n", devfile);
			return 1;
		}
	}

	hda_reset();
	if (reset)
		return 0;

	if (hda_parse_configs(module, args) < 0) {
		log_error("Cannot set up device %s\n", devfile);
		return 1;
	}

	c = hda_save_env_vars(devfile, args);
	hda_close_devfile();

	if (!c)
		hda_post_install();
	return 0;
}
