/*
 * Parse the codec configuration files and load the appropriate module
 *
 * Copyright (C) 2007 Takashi Iwai
 * 
 *  This library is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License version
 *  2.1 as published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 */

#include "config.h"
#include <stdio.h>
#include <string.h>
#include <dirent.h>
#include <errno.h>
#include <dlfcn.h>
#include <fnmatch.h>
#include <ctype.h>
#include <stdlib.h>
#include "local.h"
#include "hda/patch.h"

enum {
	VAR_VENDOR,		/* codec vendor id */
	VAR_DEVICE,		/* codec device id */
	VAR_SUBVENDOR,		/* codec subsystem vendor id */
	VAR_SUBDEVICE,		/* codec subsystem device id */
	VAR_REVISION,		/* codec revision id */
	VAR_PCI_SUBVENDOR,	/* PCI subsystem vendor id */
	VAR_PCI_SUBDEVICE,	/* PCI subsystem device id */
	VAR_NUMS
};

#define HDA_CONFIG_DIR	HDA_CODEC_DIR "/config"
#define HDA_MODULE_DIR	HDA_CODEC_DIR "/modules"

#define ARRAY_SIZE(ary)		(sizeof(ary) / sizeof((ary)[0]))

/*
 * skip space chars
 * returns NULL if it reachs EOL or a comment
 */
static char *skip_space(char *line)
{
	if (!line)
		return NULL;
	for (; *line && isspace(*line); line++)
		;
	/* skip comments */
	if (!*line || *line == '#')
		return NULL;
	return line;
}

static int get_tokens(char *line, int max_args, char **args)
{
	int num_args = 0;

	while (num_args < max_args) {
		line = skip_space(line);
		if (!line)
			break;
		args[num_args] = strsep(&line, " \t\n");
		if (!args[num_args] || !*args[num_args])
			break;
		num_args++;
	}
	return num_args;
}

static int check_cond(unsigned int var, unsigned int mask, unsigned int val)
{
	if (var == -1)
		return 0;
	val &= mask;
	var &= mask;
	return (val != var);
}

/*
 * check whether the current codec info satifies the requirement of
 * the given config item
 */
static int cfg_match(const unsigned int *vars, const unsigned int *masks)
{
	if (check_cond(vars[VAR_VENDOR], masks[VAR_VENDOR] & 0xffff,
		       codec_info.vendor_id >> 16))
		return 0;
	if (check_cond(vars[VAR_DEVICE], masks[VAR_DEVICE] & 0xffff,
		       codec_info.vendor_id))
		return 0;
	if (check_cond(vars[VAR_SUBVENDOR], masks[VAR_SUBVENDOR] & 0xffff,
		       codec_info.subsystem_id >> 16))
		return 0;
	if (check_cond(vars[VAR_SUBDEVICE], masks[VAR_SUBDEVICE] & 0xffff,
		       codec_info.subsystem_id))
		return 0;
	if (check_cond(vars[VAR_REVISION], masks[VAR_REVISION],
		       codec_info.revision_id))
		return 0;
	if (check_cond(vars[VAR_PCI_SUBVENDOR], masks[VAR_PCI_SUBVENDOR] & 0xffff,
		       codec_info.pci_subvendor))
		return 0;
	if (check_cond(vars[VAR_PCI_SUBDEVICE], masks[VAR_PCI_SUBDEVICE] & 0xffff,
		       codec_info.pci_subdevice))
		return 0;
	return 1;	/* OK, let's use this item */
}

/*
 * retrieve a preset model name from the arguments
 * returns the preset number if found, or -1 if not found
 */
static int check_preset(const struct hda_codec_table *tbl, const char **arg)
{
	unsigned int i;
	const char *preset;

	if (!tbl->num_presets || !tbl->presets)
		return -1;
	if (!arg)
		return -1;
	preset = NULL;
	for (; *arg; arg++) {
		if (!strncmp(*arg, "preset=", 7)) {
			preset = *arg + 7;
			break;
		}
	}
	if (!preset)
		return -1;
	for (i = 0; i < tbl->num_presets; i++) {
		if (!strcmp(preset, tbl->presets[i])) {
			log_verbose("found preset '%s'\n", preset);
			return i;
		}
	}
	return -1;
}

/*
 * Set up a pin config default bytes 0-3
 */
static int setup_pin(char *nids, char *vals)
{
	int i, nid;
	unsigned int val;

	nid = strtol(nids, NULL, 0);
	if (nid < 0 || nid > 0xff)
		return -1;
	val = strtol(vals, NULL, 0);
	for (i = 0; i < 4; i++)
		hda_codec_write(nid, AC_VERB_SET_CONFIG_DEFAULT_BYTES_0 + i,
				(val >> (i * 8)) & 0xff);
	return 0;
}

/*
 * Write a HDA verb
 */
static int setup_verb(char *nids, char *verbs, char *vals)
{
	int nid, verb, val;

	nid = strtol(nids, NULL, 0);
	if (nid < 0 || nid > 0xff)
		return -1;
	verb = strtol(verbs, NULL, 0);
	if (nid < 0 || nid > 0xfff)
		return -1;
	val = strtol(vals, NULL, 0);
	hda_codec_write(nid, verb, val);
	return 0;
}

/*
 * load an init file and execute the contents
 *
 * An init file is named as *.init, and it *  should contain the
 * following lines:
 *
 *   # COMMENTS
 *   CMD ARGS...
 *
 * where CMD is either pin or verb, which takes integer arguments:
 *
 *   pin NID VAL
 *   verb NID VERB VAL
 *
 * in both cases, the first argument is the widget NID.
 * pin takes the 32bit value to override the default pin config.
 * verb takes two more values, one for a HDA verb and one for a value 
 * to write.
 */
static int load_init(const char *file)
{
	char path[256];
	FILE *fp;
	char buf[256], *args[5];
	int lno = 0, nargs;
	
	log_verbose("Loading init file '%s'\n", file);
	snprintf(path, sizeof(path), "%s/%s.init", HDA_CONFIG_DIR, file);
	fp = fopen(path, "r");
	if (!fp) {
		log_error("Cannot load a init file '%s'\n", file);
		return -ENOENT;
	}

	for (; fgets(buf, sizeof(buf), fp); lno++) {
		nargs = get_tokens(buf, ARRAY_SIZE(args), args);
		if (!nargs)
			continue;
		if (nargs < 2) {
			log_error("Invalid line found in %s.init:%d\n",
				  file, lno);
			continue;
		}
		if (!strcmp(args[0], "pin")) {
			if (nargs < 3)
				log_error("No pin argument in %s.init:%d\n",
					  file, lno);
			else {
				if (setup_pin(args[1], args[2]) < 0)
					log_error("Invalid pin cfg at %s.init:%d\n",
						  file, lno);
			}
		} else if (!strcmp(args[0], "verb")) {
			if (nargs < 4)
				log_error("No verb argument in %s.init:%d\n",
					  file, lno);
			else {
				if (setup_verb(args[1], args[2], args[3]) < 0)
					log_error("Invalid verb cfg at %s.init:%d\n",
						  file, lno);
			}
		}
	}
	fclose(fp);
	return 0;
}

/*
 * retrieve a initialize command and execute it (if found)
 */
static int check_init(const struct hda_codec_table *tbl, const char **arg)
{
	if (!arg)
		return 0;
	for (; *arg; arg++) {
		if (!strncmp(*arg, "init=", 5))
			load_init(*arg + 5);
	}
	return 0;
}

/*
 * load a module and execute
 */
static int load_config(const char *module, const char **args)
{
	void *handle;
	hda_patch_desc_t patch_desc;
	const struct hda_codec_table *tbl;
	int err = -ENODEV;;
	char fname[128];

	snprintf(fname, sizeof(fname), "%s/%s.so",
		 HDA_MODULE_DIR, module);
	handle = dlopen(fname, RTLD_LAZY);
	if (!handle) {
		log_error("cannot open a module '%s'\n", fname);
		return -ENODEV;
	}
	patch_desc = (hda_patch_desc_t) dlsym(handle, HDA_PATCH_DESC);
	if (!patch_desc) {
		log_error("cannot find a patch_desc in module '%s'\n", module);
		goto error;
	}
	tbl = patch_desc();
	if (!tbl) {
		log_error("no matching patchd in module '%s'\n", module);
		goto error;
	}
	for (; tbl->id; tbl++) {
		if ((tbl->id == -1 || tbl->id == codec_info.vendor_id) &&
		    (!tbl->revs || tbl->revs == -1 ||
		     tbl->revs == codec_info.revision_id)) {
			int config = check_preset(tbl, args);
			check_init(tbl, args);
			err = tbl->patch(tbl, config, args);
			if (err == -ENODEV)
				hda_reset();
			break;
		}
	}
 error:
	dlclose(handle);
	if (err >= 0)
		hda_set_status(HDA_STATUS_RUNNING);
	return err;
}


/*
 * read a config file and parses and stores each entry to cfg_list_head
 *
 * a config file should contain the following lines:
 *
 *   # COMMENTS
 *   VID:DID:SVID:SDID:REV:PVID:PSID MODULE [ARGS...]
 *   VID:DID:SVID:SDID:REV:PVID:PSID MODULE [ARGS...]
 *   ...
 *
 * in each VID, SID, etc field, '*' can be given as a wildcard to match
 * everything.  The bitmask can be specified with slash, such as
 *    1230/fff0
 * where the MSB 12bits are compared (e.g. 1234 matches)
 */
static int parse_cfg_rules(const char *file, const char **cfg_args)
{
	FILE *fp;
	char path[256];
	char buf[256], *args[64], *cond;
	const char **passed_args;
	int lno = 0, i, nargs, err;
	unsigned int vars[VAR_NUMS];
	unsigned int masks[VAR_NUMS];

	snprintf(path, sizeof(path), "%s/%s", HDA_CONFIG_DIR, file);
	fp = fopen(path, "r");
	if (!fp)
		return 0; /* ignore errnous files */
	for (; fgets(buf, sizeof(buf), fp); lno++) {
		nargs = get_tokens(buf, ARRAY_SIZE(args) - 1, args);
		if (!nargs)
			continue;
		if (nargs < 2) {
			log_error("Invalid line found in %s.conf:%d\n",
				  file, lno);
			continue;
		}
		if (*args[1] == '/') {
			log_error("Invalid module name found in %s.conf:%d\n",
				  file, lno);
			continue;
		}

		/* check the condition */
		for (i = 0; i < VAR_NUMS; i++)
			vars[i] = -1;
		for (i = 0; i < VAR_NUMS; i++)
			masks[i] = -1;
		cond = args[0];
		for (i = 0; cond && i < VAR_NUMS; i++) {
			char *token = strsep(&cond, ": \t");
			char *mask;
			if (!*token || *token == '*')
				continue;
			mask = strchr(token, '/');
			if (mask) {
				*mask = 0;
				masks[i] = strtol(mask + 1, NULL, 16);
			}
			vars[i] = strtol(token, NULL, 16);
		}

		if (!cfg_match(vars, masks))
			continue;

		log_verbose("Loading module '%s'\n", args[1]);
		if (cfg_args)
			passed_args = cfg_args;
		else if (nargs > 2) {
			args[ARRAY_SIZE(args)-1] = NULL; /* terminate */
			passed_args = (const char **)(args + 2);
		} else
			passed_args = NULL;
		err = load_config(args[1], passed_args);
		if (err >= 0) {
			log_verbose("Module '%s' successful\n", args[1]);
			return 0;
		}
		if (err != -ENODEV) {
			/* fatal error */
			return err;
		}
		log_verbose("No match for module '%s'\n", args[1]);
	}
	fclose(fp);
	return -ENODEV;
}

static int config_filter(const struct dirent *de)
{
	return !fnmatch("*.conf", de->d_name,  0);
}

/*
 * read the all config files (matching "*.conf" name), parses the lines
 * and tries to load the module if matched
 */
static int read_and_load_configs(const char **args)
{
	struct dirent **namelist;
	int i, names, err = -ENODEV;

	names = scandir(HDA_CONFIG_DIR, &namelist,
			config_filter, alphasort);
	if (names < 0) {
		log_error("Cannot read config dir %s\n", HDA_CONFIG_DIR);
		return -ENOENT;
	}

	for (i = 0; i < names; i++) {
		int err = parse_cfg_rules(namelist[i]->d_name, args);
		if (!err) {
			/* matched */
			free(namelist);
			return 0;
		}
		if (err < 0 && err != -ENODEV)
			/* fatal error */
			break;
	}
	free(namelist);
	return err;
}

/*
 * read and parse config files, run appropriate modules until found
 */
int hda_parse_configs(const char *module, const char **args)
{
	int err;

	if (module) {
		log_verbose("Loading forced module '%s'\n", module);
		err = load_config(module, args);
		if (err >= 0)
			log_verbose("Module '%s' successful\n", module);
		return err;
	}

	return read_and_load_configs(args);
}
