/*
 * STAC92xx codec module
 *
 * Copyright (c) 2007 Takashi Iwai <tiwai@suse.de>
 *
 * Derived from patch_sigmatel.c:
 *
 * Copyright (c) 2005 Embedded Alley Solutions, Inc.
 * Matt Porter <mporter@embeddedalley.com>
 *
 * Based on patch_cmedia.c and patch_realtek.c
 * Copyright (c) 2004 Takashi Iwai <tiwai@suse.de>
 *
 *  This driver is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This driver is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

#include "config.h"
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include "hda/cmds.h"
#include "hda/io.h"
#include "hda/mixer.h"
#include "hda/log.h"
#include "hda/pincfg.h"
#include "hda/patch.h"

#define STAC_HP_EVENT		0x37

#define ARRAY_SIZE(ary)		(sizeof(ary) / sizeof((ary)[0]))

struct stac_spec {
	unsigned int surr_switch: 1;
	unsigned int line_switch: 1;
	unsigned int mic_switch: 1;
	unsigned int hp_detect: 1;
	unsigned int gpio_mute: 1;

	/* playback */
	hda_nid_t *dac_nids;
	unsigned int num_dacs;
	hda_nid_t hp_nid;
	hda_nid_t dig_out_nid;
	hda_nid_t priv_dac_nids[5];

	/* capture */
	hda_nid_t *adc_nids;
	unsigned int num_adcs;
	hda_nid_t *mux_nids;
	unsigned int num_muxes;
	hda_nid_t *dmic_nids;
	unsigned int num_dmics;
	hda_nid_t dmux_nid;
	hda_nid_t dig_in_nid;

	/* pin widgets */
	hda_nid_t *pin_nids;
	unsigned int num_pins;
	unsigned int pin_configs[16];

	/* codec specific stuff */
	struct hda_verb_array auto_init;

	/* capture sources */
	struct hda_input_mux dig_input_mux;
	struct hda_input_mux input_mux;

	/* dynamic controls and input_mux */
	struct hda_auto_pin_cfg autocfg;
};

static hda_nid_t stac9200_adc_nids[1] = {
        0x03,
};

static hda_nid_t stac9200_mux_nids[1] = {
        0x0c,
};

static hda_nid_t stac9200_dac_nids[1] = {
        0x02,
};

static hda_nid_t stac925x_adc_nids[1] = {
        0x03,
};

static hda_nid_t stac925x_mux_nids[1] = {
        0x0f,
};

static hda_nid_t stac925x_dac_nids[1] = {
        0x02,
};

static hda_nid_t stac925x_dmic_nids[1] = {
	0x15, 
};

static hda_nid_t stac922x_adc_nids[2] = {
        0x06, 0x07,
};

static hda_nid_t stac922x_mux_nids[2] = {
        0x12, 0x13,
};

static hda_nid_t stac927x_adc_nids[3] = {
        0x07, 0x08, 0x09
};

static hda_nid_t stac927x_mux_nids[3] = {
        0x15, 0x16, 0x17
};

static hda_nid_t stac9205_adc_nids[2] = {
        0x12, 0x13
};

static hda_nid_t stac9205_mux_nids[2] = {
        0x19, 0x1a
};

static hda_nid_t stac9205_dmic_nids[2] = {
        0x17, 0x18,
};

static hda_nid_t stac9200_pin_nids[8] = {
	0x08, 0x09, 0x0d, 0x0e, 
	0x0f, 0x10, 0x11, 0x12,
};

static hda_nid_t stac925x_pin_nids[8] = {
	0x07, 0x08, 0x0a, 0x0b, 
	0x0c, 0x0d, 0x10, 0x11,
};

static hda_nid_t stac922x_pin_nids[10] = {
	0x0a, 0x0b, 0x0c, 0x0d, 0x0e,
	0x0f, 0x10, 0x11, 0x15, 0x1b,
};

static hda_nid_t stac927x_pin_nids[14] = {
	0x0a, 0x0b, 0x0c, 0x0d, 0x0e,
	0x0f, 0x10, 0x11, 0x12, 0x13,
	0x14, 0x21, 0x22, 0x23,
};

static hda_nid_t stac9205_pin_nids[12] = {
	0x0a, 0x0b, 0x0c, 0x0d, 0x0e,
	0x0f, 0x14, 0x16, 0x17, 0x18,
	0x21, 0x22,
	
};

static struct hda_verb stac9200_core_init[] = {
	/* set dac0mux for dac converter */
	{ 0x07, AC_VERB_SET_CONNECT_SEL, 0x00},
	{}
};

static struct hda_verb stac925x_core_init[] = {
	/* set dac0mux for dac converter */
	{ 0x06, AC_VERB_SET_CONNECT_SEL, 0x00},
	{}
};

static struct hda_verb stac922x_core_init[] = {
	/* set master volume and direct control */	
	{ 0x16, AC_VERB_SET_VOLUME_KNOB_CONTROL, 0xff},
	{}
};

static struct hda_verb d965_core_init[] = {
	/* set master volume and direct control */	
	{ 0x24, AC_VERB_SET_VOLUME_KNOB_CONTROL, 0xff},
	/* unmute node 0x1b */
	{ 0x1b, AC_VERB_SET_AMP_GAIN_MUTE, 0xb000},
	/* select node 0x03 as DAC */	
	{ 0x0b, AC_VERB_SET_CONNECT_SEL, 0x01},
	{}
};

static struct hda_verb stac927x_core_init[] = {
	/* set master volume and direct control */	
	{ 0x24, AC_VERB_SET_VOLUME_KNOB_CONTROL, 0xff},
	{}
};

static struct hda_verb stac9205_core_init[] = {
	/* set master volume and direct control */	
	{ 0x24, AC_VERB_SET_VOLUME_KNOB_CONTROL, 0xff},
	{}
};

static struct hda_std_mixer stac9200_mixer[] = {
	HDA_CODEC_VOLUME("Master Playback Volume", 0xb, 0, HDA_OUTPUT),
	HDA_CODEC_MUTE("Master Playback Switch", 0xb, 0, HDA_OUTPUT),
	HDA_CODEC_VOLUME("Capture Volume", 0x0a, 0, HDA_OUTPUT),
	HDA_CODEC_MUTE("Capture Switch", 0x0a, 0, HDA_OUTPUT),
	HDA_CODEC_VOLUME("Capture Mux Volume", 0x0c, 0, HDA_OUTPUT),
	{ } /* end */
};

static struct hda_std_mixer stac925x_mixer[] = {
	HDA_CODEC_VOLUME("Master Playback Volume", 0xe, 0, HDA_OUTPUT),
	HDA_CODEC_MUTE("Master Playback Switch", 0xe, 0, HDA_OUTPUT),
	HDA_CODEC_VOLUME("Capture Volume", 0x09, 0, HDA_OUTPUT),
	HDA_CODEC_MUTE("Capture Switch", 0x09, 0, HDA_OUTPUT),
	HDA_CODEC_VOLUME("Capture Mux Volume", 0x0f, 0, HDA_OUTPUT),
	{ } /* end */
};

/* This needs to be generated dynamically based on sequence */
static struct hda_std_mixer stac922x_mixer[] = {
	HDA_CODEC_VOLUME("Capture Volume", 0x17, 0x0, HDA_INPUT),
	HDA_CODEC_MUTE("Capture Switch", 0x17, 0x0, HDA_INPUT),
	HDA_CODEC_VOLUME("Mux Capture Volume", 0x12, 0x0, HDA_OUTPUT),
	{ } /* end */
};

/* This needs to be generated dynamically based on sequence */
static struct hda_std_mixer stac9227_mixer[] = {
	HDA_CODEC_VOLUME("Capture Volume", 0x15, 0x0, HDA_OUTPUT),
	HDA_CODEC_MUTE("Capture Switch", 0x1b, 0x0, HDA_OUTPUT),
	{ } /* end */
};

static struct hda_std_mixer stac927x_mixer[] = {
	HDA_CODEC_VOLUME("InMux Capture Volume", 0x15, 0x0, HDA_OUTPUT),
	HDA_CODEC_VOLUME("InVol Capture Volume", 0x18, 0x0, HDA_INPUT),
	HDA_CODEC_MUTE("ADCMux Capture Switch", 0x1b, 0x0, HDA_OUTPUT),
	{ } /* end */
};

static struct hda_std_mixer stac9205_mixer[] = {
	HDA_CODEC_VOLUME("InMux Capture Volume", 0x19, 0x0, HDA_OUTPUT),
	HDA_CODEC_VOLUME("InVol Capture Volume", 0x1b, 0x0, HDA_INPUT),
	HDA_CODEC_MUTE("ADCMux Capture Switch", 0x1d, 0x0, HDA_OUTPUT),
	{ } /* end */
};

static int stac92xx_build_controls(struct stac_spec *spec)
{
	int err;

	if (spec->input_mux.num_items) {
		err = hda_register_input_mux_mixer("Capture Source",
						   spec->mux_nids[0],
						   &spec->input_mux);
		if (err < 0)
			return err;
	}

	if (spec->dig_input_mux.num_items) {
		err = hda_register_input_mux_mixer("Digital Input Source",
						   spec->dmux_nid,
						   &spec->dig_input_mux);
		if (err < 0)
			return err;
	}

	return 0;	
}

static int stac92xx_save_pin_config_regs(struct stac_spec *spec)
{
	int i, j, n;
	struct hda_verb verbs[spec->num_pins * 4 + 1];

	for (i = 0; i < spec->num_pins; i++) {
		hda_nid_t nid = spec->pin_nids[i];
		unsigned int pin_cfg;
		
		pin_cfg = hda_codec_read(nid, AC_VERB_GET_CONFIG_DEFAULT, 0x00);
		log_verbose("stac92xx: pin nid %2.2x bios pin config %8.8x\n",
			    nid, pin_cfg);
		spec->pin_configs[i] = pin_cfg;
	}

	/* verbs for resume */
	n = 0;
	for (i = 0; i < spec->num_pins; i++) {
		for (j = 0; j < 4; j++) {
			verbs[n].nid = spec->pin_nids[i];
			verbs[n].verb = AC_VERB_SET_CONFIG_DEFAULT_BYTES_0 + j;
			verbs[n].param = 
				(spec->pin_configs[i] >> (j * 8)) & 0xff;
			n++;
		}
	}
	memset(&verbs[n], 0, sizeof(verbs[n]));
	
	return hda_register_resume_verbs(verbs);
}

/*
 * PCM
 */
static struct hda_usr_pcm_info stac92xx_pcm_digital_playback = {
	.name = "STAC92xx Digital",
	.type = HDA_PCM_TYPE_SPDIF_OUT,
	.substreams = 1,
	.channels_min = 2,
	.channels_max = 2,
	/* NID is set in stac92xx_build_pcms */
	.device = 1,
};

static struct hda_usr_pcm_info stac92xx_pcm_digital_capture = {
	.name = "STAC92xx Digital",
	.type = HDA_PCM_TYPE_SPDIF_IN,
	.substreams = 1,
	.channels_min = 2,
	.channels_max = 2,
	/* NID is set in stac92xx_build_pcms */
	.device = 1,
};

static struct hda_usr_pcm_info stac92xx_pcm_analog_playback = {
	.name = "STAC92xx Analog",
	.type = HDA_PCM_TYPE_ANALOG_MULTI_OUT,
	.substreams = 1,
	.channels_min = 2,
	.channels_max = 8,
	.nid = 0x02, /* NID to query formats and rates */
	.device = 0,
};

static struct hda_usr_pcm_info stac92xx_pcm_analog_capture = {
	.name = "STAC92xx Analog",
	.type = HDA_PCM_TYPE_ANALOG_IN,
	.substreams = 1,
	.channels_min = 2,
	.channels_max = 2,
	/* NID is set in stac92xx_build_pcms */
	.device = 0,
};

static int stac92xx_build_pcms(struct stac_spec *spec)
{
	int err;

	stac92xx_pcm_analog_playback.channels_max = spec->num_dacs * 2;
	memcpy(stac92xx_pcm_analog_playback.assoc_nids,
	       spec->dac_nids, spec->num_dacs * sizeof(hda_nid_t));
	stac92xx_pcm_analog_playback.extra_nids[0] = spec->hp_nid;
	err = hda_register_pcm(&stac92xx_pcm_analog_playback);
	if (err < 0)
		return err;

	stac92xx_pcm_analog_capture.assoc_nids[0] = spec->adc_nids[0];
	err = hda_register_pcm(&stac92xx_pcm_analog_playback);
	if (err < 0)
		return err;

	if (spec->dig_out_nid) {
		stac92xx_pcm_digital_playback.nid = spec->dig_out_nid;
		stac92xx_pcm_digital_playback.assoc_nids[0] = spec->dig_out_nid;
		err = hda_register_pcm(&stac92xx_pcm_digital_playback);
		if (err < 0)
			return err;
	}
	if (spec->dig_in_nid) {
		stac92xx_pcm_digital_capture.nid = spec->dig_in_nid;
		stac92xx_pcm_digital_capture.assoc_nids[0] = spec->dig_in_nid;
		err = hda_register_pcm(&stac92xx_pcm_digital_capture);
		if (err < 0)
			return err;
	}

	return 0;
}

static unsigned int stac92xx_get_vref(hda_nid_t nid)
{
	unsigned int pincap = hda_param_read(nid, AC_PAR_PIN_CAP);
	pincap = (pincap & AC_PINCAP_VREF) >> AC_PINCAP_VREF_SHIFT;
	if (pincap & AC_PINCAP_VREF_100)
		return AC_PINCTL_VREF_100;
	if (pincap & AC_PINCAP_VREF_80)
		return AC_PINCTL_VREF_80;
	if (pincap & AC_PINCAP_VREF_50)
		return AC_PINCTL_VREF_50;
	if (pincap & AC_PINCAP_VREF_GRD)
		return AC_PINCTL_VREF_GRD;
	return 0;
}

/* add dynamic controls */
static int stac92xx_add_io_switch(const char *name,
				  hda_nid_t nid, int is_mic)
{
	struct hda_verb verb_on[2] = {
		{ nid, AC_VERB_SET_PIN_WIDGET_CONTROL, AC_PINCTL_OUT_EN },
		{ }
	};
	struct hda_verb verb_off[2] = {
		{ nid, AC_VERB_SET_PIN_WIDGET_CONTROL, AC_PINCTL_IN_EN },
		{ }
	};
	struct hda_cmds io_switch_cmds[3] = {
		STD_VERBS(verb_on),
		STD_VERBS(verb_off),
		STD_NULL
	};

	if (is_mic)
		verb_off[0].param |= stac92xx_get_vref(nid);
	return hda_register_mixer(HDA_MIXER_TYPE_BOOLEAN, name,
				  (long) io_switch_cmds);
}

/* flag inputs as additional dynamic lineouts */
static int stac92xx_add_dyn_out_pins(struct stac_spec *spec,
				     struct hda_auto_pin_cfg *cfg)
{
	unsigned int wcaps, wtype;
	hda_nid_t nid;
	int num_dacs = 0;
	
	/* use the wcaps cache to count all DACs available for line-outs */
	for (nid = codec_info.start_nid;
	     nid < codec_info.start_nid + codec_info.num_nodes; nid++) {
		wcaps = hda_get_wcaps(nid);
		wtype = (wcaps & AC_WCAP_TYPE) >> AC_WCAP_TYPE_SHIFT;
		if (wtype == AC_WID_AUD_OUT && !(wcaps & AC_WCAP_DIGITAL))
			num_dacs++;
	}

	log_verbose("stac92xx: total dac count=%d\n", num_dacs);
	
	switch (cfg->line_outs) {
	case 3:
		/* add line-in as side */
		if (cfg->input_pins[AUTO_PIN_LINE] && num_dacs > 3) {
			cfg->line_out_pins[3] = cfg->input_pins[AUTO_PIN_LINE];
			spec->line_switch = 1;
			cfg->line_outs++;
		}
		break;
	case 2:
		/* add line-in as clfe and mic as side */
		if (cfg->input_pins[AUTO_PIN_LINE] && num_dacs > 2) {
			cfg->line_out_pins[2] = cfg->input_pins[AUTO_PIN_LINE];
			spec->line_switch = 1;
			cfg->line_outs++;
		}
		if (cfg->input_pins[AUTO_PIN_MIC] && num_dacs > 3) {
			cfg->line_out_pins[3] = cfg->input_pins[AUTO_PIN_MIC];
			spec->mic_switch = 1;
			cfg->line_outs++;
		}
		break;
	case 1:
		/* add line-in as surr and mic as clfe */
		if (cfg->input_pins[AUTO_PIN_LINE] && num_dacs > 1) {
			cfg->line_out_pins[1] = cfg->input_pins[AUTO_PIN_LINE];
			spec->line_switch = 1;
			cfg->line_outs++;
		}
		if (cfg->input_pins[AUTO_PIN_MIC] && num_dacs > 2) {
			cfg->line_out_pins[2] = cfg->input_pins[AUTO_PIN_MIC];
			spec->mic_switch = 1;
			cfg->line_outs++;
		}
		break;
	}

	return 0;
}


static int is_in_dac_nids(struct stac_spec *spec, hda_nid_t nid)
{
	int i;
	
	for (i = 0; i < spec->num_dacs; i++) {
		if (spec->dac_nids[i] == nid)
			return 1;
	}

	return 0;
}

/*
 * Fill in the dac_nids table from the parsed pin configuration
 * This function only works when every pin in line_out_pins[]
 * contains atleast one DAC in its connection list. Some 92xx
 * codecs are not connected directly to a DAC, such as the 9200
 * and 9202/925x. For those, dac_nids[] must be hard-coded.
 */
static int stac92xx_auto_fill_dac_nids(struct stac_spec *spec,
				       const struct hda_auto_pin_cfg *cfg)
{
	int err, i, j, conn_len = 0; 
	hda_nid_t nid, conn[HDA_MAX_CONNECTIONS];
	unsigned int wcaps, wtype;
	
	spec->dac_nids = spec->priv_dac_nids;

	for (i = 0; i < cfg->line_outs; i++) {
		nid = cfg->line_out_pins[i];
		conn_len = hda_get_connections(nid, conn,
					       HDA_MAX_CONNECTIONS);
		for (j = 0; j < conn_len; j++) {
			wcaps = hda_param_read(conn[j],
					       AC_PAR_AUDIO_WIDGET_CAP);
			wtype = (wcaps & AC_WCAP_TYPE) >> AC_WCAP_TYPE_SHIFT;

			if (wtype != AC_WID_AUD_OUT ||
			    (wcaps & AC_WCAP_DIGITAL))
				continue;
			/* conn[j] is a DAC routed to this line-out */
			if (!is_in_dac_nids(spec, conn[j]))
				break;
		}

		if (j == conn_len) {
			/* error out, no available DAC found */
			log_error("stac92xx: No available DAC for pin 0x%x\n",
				  nid);
			return -ENODEV;
		}

		spec->dac_nids[i] = conn[j];
		spec->num_dacs++;
		if (conn_len > 1) {
			/* select this DAC in the pin's input mux */
			hda_codec_write(nid, AC_VERB_SET_CONNECT_SEL, j);
			err = hda_append_verb(&spec->auto_init,
					      nid, AC_VERB_SET_CONNECT_SEL, j);
			if (err < 0)
				return err;
		}
	}

	log_verbose("stac92xx: dac_nids=%d (0x%x/0x%x/0x%x/0x%x/0x%x)\n",
		    spec->num_dacs,
		    spec->dac_nids[0],
		    spec->dac_nids[1],
		    spec->dac_nids[2],
		    spec->dac_nids[3],
		    spec->dac_nids[4]);
	return 0;
}

/* create volume control/switch for the given prefx type */
static int create_controls(const char *pfx, hda_nid_t nid, int chs)
{
	int err;
	char name[32];

	sprintf(name, "%s Playback Volume", pfx);
	err = hda_register_mixer(HDA_MIXER_TYPE_VOLUME, name,
				 HDA_COMPOSE_AMP_VAL(nid, chs, 0, HDA_OUTPUT));
	if (err < 0)
		return err;
	sprintf(name, "%s Playback Switch", pfx);
	err = hda_register_mixer(HDA_MIXER_TYPE_SWITCH, name,
				 HDA_COMPOSE_AMP_VAL(nid, chs, 0, HDA_OUTPUT));
	if (err < 0)
		return err;
	return 0;
}

/* add playback controls from the parsed DAC table */
static int stac92xx_auto_create_multi_out_ctls(struct stac_spec *spec,
					       const struct hda_auto_pin_cfg *cfg)
{
	static const char *chname[4] = {
		"Front", "Surround", NULL /*CLFE*/, "Side"
	};
	hda_nid_t nid;
	int i, err;

	for (i = 0; i < cfg->line_outs; i++) {
		if (!spec->dac_nids[i])
			continue;

		nid = spec->dac_nids[i];

		if (i == 2) {
			/* Center/LFE */
			err = create_controls("Center", nid, 1);
			if (err < 0)
				return err;
			err = create_controls("LFE", nid, 2);
			if (err < 0)
				return err;
		} else {
			err = create_controls(chname[i], nid, 3);
			if (err < 0)
				return err;
		}
	}

	if (spec->line_switch) {
		err = stac92xx_add_io_switch("Line In as Output Switch",
					     cfg->input_pins[AUTO_PIN_LINE], 0);
		if (err < 0)
			return err;
	}

	if (spec->mic_switch) {
		err = stac92xx_add_io_switch("Mic as Output Switch",
					     cfg->input_pins[AUTO_PIN_MIC], 1);
		if (err < 0)
			return err;
	}

	return 0;
}

static int check_in_dac_nids(struct stac_spec *spec, hda_nid_t nid)
{
	if (is_in_dac_nids(spec, nid))
		return 1;
	if (spec->hp_nid == nid)
		return 1;
	return 0;
}

static int add_spec_dacs(struct stac_spec *spec, hda_nid_t nid)
{
	if (!spec->hp_nid)
		spec->hp_nid = nid;
	else if (spec->num_dacs > 4) {
		log_info("stac92xx: No space for DAC 0x%x\n", nid);
		return 1;
	} else {
		spec->dac_nids[spec->num_dacs] = nid;
		spec->num_dacs++;
	}
	return 0;
}

/* add playback controls for Speaker and HP outputs */
static int stac92xx_auto_create_hp_ctls(struct stac_spec *spec,
					struct hda_auto_pin_cfg *cfg)
{
	hda_nid_t nid;
	int i, old_num_dacs, err;

	old_num_dacs = spec->num_dacs;
	for (i = 0; i < cfg->hp_outs; i++) {
		unsigned int wid_caps = hda_get_wcaps(cfg->hp_pins[i]);
		if (wid_caps & AC_WCAP_UNSOL_CAP)
			spec->hp_detect = 1;
		nid = hda_codec_read(cfg->hp_pins[i],
				     AC_VERB_GET_CONNECT_LIST, 0) & 0xff;
		if (check_in_dac_nids(spec, nid))
			nid = 0;
		if (!nid)
			continue;
		add_spec_dacs(spec, nid);
	}
	for (i = 0; i < cfg->speaker_outs; i++) {
		nid = hda_codec_read(cfg->speaker_pins[i],
				     AC_VERB_GET_CONNECT_LIST, 0) & 0xff;
		if (check_in_dac_nids(spec, nid))
			nid = 0;
		if (!nid)
			continue;
		add_spec_dacs(spec, nid);
	}

	for (i = old_num_dacs; i < spec->num_dacs; i++) {
		static const char *pfxs[] = {
			"Speaker", "External Speaker", "Speaker2",
		};
		err = create_controls(pfxs[i - old_num_dacs],
				      spec->dac_nids[i], 3);
		if (err < 0)
			return err;
	}
	if (spec->hp_nid) {
		const char *pfx;
		if (old_num_dacs == spec->num_dacs)
			pfx = "Master";
		else
			pfx = "Headphone";
		err = create_controls(pfx, spec->hp_nid, 3);
		if (err < 0)
			return err;
	}

	return 0;
}

/* labels for dmic mux inputs */
static const char *stac92xx_dmic_labels[5] = {
	"Analog Inputs", "Digital Mic 1", "Digital Mic 2",
	"Digital Mic 3", "Digital Mic 4"
};

/* create playback/capture controls for input pins on dmic capable codecs */
static int stac92xx_auto_create_dmic_input_ctls(struct stac_spec *spec,
						const struct hda_auto_pin_cfg *cfg)
{
	struct hda_input_mux *dimux = &spec->dig_input_mux;
	hda_nid_t con_lst[HDA_MAX_NUM_INPUTS];
	int i, j;

	dimux->items[dimux->num_items].label = stac92xx_dmic_labels[0];
	dimux->items[dimux->num_items].index = 0;
	dimux->num_items++;

	for (i = 0; i < spec->num_dmics; i++) {
		int index;
		int num_cons;

		if (!hda_pin_available(spec->dmic_nids[i]))
			continue;

		num_cons = hda_get_connections(spec->dmux_nid,
					       con_lst,
					       HDA_MAX_NUM_INPUTS);
		for (j = 0; j < num_cons; j++)
			if (con_lst[j] == spec->dmic_nids[i]) {
				index = j;
				goto found;
			}
		continue;
found:
		dimux->items[dimux->num_items].label =
			stac92xx_dmic_labels[dimux->num_items];
		dimux->items[dimux->num_items].index = index;
		dimux->num_items++;
	}

	return 0;
}

/* create playback/capture controls for input pins */
static int
stac92xx_auto_create_analog_input_ctls(struct stac_spec *spec,
				       const struct hda_auto_pin_cfg *cfg)
{
	struct hda_input_mux *imux = &spec->input_mux;
	hda_nid_t con_lst[HDA_MAX_NUM_INPUTS];
	int i, j, k, err;

	for (i = 0; i < AUTO_PIN_LAST; i++) {
		int index;

		if (!cfg->input_pins[i])
			continue;
		index = -1;
		for (j = 0; j < spec->num_muxes; j++) {
			int num_cons;
			num_cons = hda_get_connections(spec->mux_nids[j],
						       con_lst,
						       HDA_MAX_NUM_INPUTS);
			for (k = 0; k < num_cons; k++)
				if (con_lst[k] == cfg->input_pins[i]) {
					index = k;
					goto found;
				}
		}
		continue;
	found:
		imux->items[imux->num_items].label = hda_auto_pin_cfg_labels[i];
		imux->items[imux->num_items].index = index;
		imux->num_items++;
	}

	if (imux->num_items) {
		/*
		 * Set the current input for the muxes.
		 * The STAC9221 has two input muxes with identical source
		 * NID lists.  Hopefully this won't get confused.
		 */
		for (i = 0; i < spec->num_muxes; i++) {
			hda_codec_write(spec->mux_nids[i],
					AC_VERB_SET_CONNECT_SEL,
					imux->items[0].index);
			err = hda_append_verb(&spec->auto_init,
					      spec->mux_nids[i],
					      AC_VERB_SET_CONNECT_SEL,
					      imux->items[0].index);
			if (err < 0)
				return err;
		}
	}

	return 0;
}

static int auto_set_pinctl(struct stac_spec *spec, hda_nid_t nid, int ctl)
{
	return hda_append_verb(&spec->auto_init, nid,
			       AC_VERB_SET_PIN_WIDGET_CONTROL, ctl);
}

static int stac92xx_auto_init_multi_out(struct stac_spec *spec,
					struct hda_auto_pin_cfg *cfg)
{
	int i, err;

	for (i = 0; i < cfg->line_outs; i++) {
		hda_nid_t nid = cfg->line_out_pins[i];
		err = auto_set_pinctl(spec, nid, AC_PINCTL_OUT_EN);
		if (err < 0)
			return err;
	}
	return 0;
}

static int stac92xx_auto_init_hp_out(struct stac_spec *spec,
				     struct hda_auto_pin_cfg *cfg)
{
	int i, err;

	for (i = 0; i < cfg->hp_outs; i++) {
		hda_nid_t pin;
		pin = cfg->hp_pins[i];
		if (pin) {
			/* connect to front */
			err = auto_set_pinctl(spec, pin,
					      AC_PINCTL_OUT_EN |
					      AC_PINCTL_HP_EN);
			if (err < 0)
				return err;
		}
	}
	for (i = 0; i < cfg->speaker_outs; i++) {
		hda_nid_t pin;
		pin = cfg->speaker_pins[i];
		if (pin) {
			/* connect to front */
			err = auto_set_pinctl(spec, pin, AC_PINCTL_OUT_EN);
			if (err < 0)
				return err;
		}
	}
	return 0;
}

static int stac92xx_parse_auto_config(struct stac_spec *spec,
				      hda_nid_t dig_out, hda_nid_t dig_in)
{
	int err;

	err = hda_parse_pin_def_config(&spec->autocfg, spec->dmic_nids);
	if (err < 0)
		return err;
	if (!spec->autocfg.line_outs)
		return -ENODEV; /* can't find valid pin config */

	err = stac92xx_add_dyn_out_pins(spec, &spec->autocfg);
	if (err < 0)
		return err;
	if (!spec->num_dacs) {
		err = stac92xx_auto_fill_dac_nids(spec, &spec->autocfg);
		if (err < 0)
			return err;
	}

	err = stac92xx_auto_create_multi_out_ctls(spec, &spec->autocfg);
	if (err < 0)
		return err;
	err = stac92xx_auto_create_hp_ctls(spec, &spec->autocfg);
	if (err < 0)
		return err;
	err = stac92xx_auto_create_analog_input_ctls(spec, &spec->autocfg);
	if (err < 0)
		return err;

	if (spec->num_dmics > 0) {
		err = stac92xx_auto_create_dmic_input_ctls(spec,
							   &spec->autocfg);
		if (err < 0)
			return err;
	}

	if (spec->num_dacs > 1)
		spec->surr_switch = 1;

	if (spec->autocfg.dig_out_pin)
		spec->dig_out_nid = dig_out;
	if (spec->autocfg.dig_in_pin)
		spec->dig_in_nid = dig_in;

	return 0;
}

/* add playback controls for HP output */
static int stac9200_auto_create_hp_ctls(struct stac_spec *spec,
					const struct hda_auto_pin_cfg *cfg)
{
	hda_nid_t pin = cfg->hp_pins[0];
	unsigned int wid_caps;

	if (!pin)
		return 0;

	wid_caps = hda_get_wcaps(pin);
	if (wid_caps & AC_WCAP_UNSOL_CAP)
		spec->hp_detect = 1;

	return 0;
}

/* add playback controls for LFE output */
static int stac9200_auto_create_lfe_ctls(struct stac_spec *spec,
					 const struct hda_auto_pin_cfg *cfg)
{
	int err;
	hda_nid_t lfe_pin = 0x0;
	int i;

	/*
	 * search speaker outs and line outs for a mono speaker pin
	 * with an amp.  If one is found, add LFE controls
	 * for it.
	 */
	for (i = 0; i < cfg->speaker_outs && lfe_pin == 0x0; i++) {
		hda_nid_t pin = cfg->speaker_pins[i];
		unsigned int wcaps = hda_get_wcaps(pin);
		wcaps &= (AC_WCAP_STEREO | AC_WCAP_OUT_AMP);
		if (wcaps == AC_WCAP_OUT_AMP)
			/* found a mono speaker with an amp, must be lfe */
			lfe_pin = pin;
	}

	/* if speaker_outs is 0, then speakers may be in line_outs */
	if (lfe_pin == 0 && cfg->speaker_outs == 0) {
		for (i = 0; i < cfg->line_outs && lfe_pin == 0x0; i++) {
			hda_nid_t pin = cfg->line_out_pins[i];
			unsigned long cfg;
			cfg = hda_codec_read(pin, AC_VERB_GET_CONFIG_DEFAULT,
					     0x00);
			if (get_defcfg_device(cfg) == AC_JACK_SPEAKER) {
				unsigned int wcaps = hda_get_wcaps(pin);
				wcaps &= (AC_WCAP_STEREO | AC_WCAP_OUT_AMP);
				if (wcaps == AC_WCAP_OUT_AMP)
					/* found a mono speaker with an amp,
					   must be lfe */
					lfe_pin = pin;
			}
		}
	}

	if (lfe_pin) {
		err = create_controls("LFE", lfe_pin, 1);
		if (err < 0)
			return err;
	}

	return 0;
}

static int stac9200_parse_auto_config(struct stac_spec *spec)
{
	int err;

	err = hda_parse_pin_def_config(&spec->autocfg, NULL);
	if (err < 0)
		return err;

	err = stac92xx_auto_create_analog_input_ctls(spec, &spec->autocfg);
	if (err < 0)
		return err;

	err = stac9200_auto_create_hp_ctls(spec, &spec->autocfg);
	if (err < 0)
		return err;

	err = stac9200_auto_create_lfe_ctls(spec, &spec->autocfg);
	if (err < 0)
		return err;

	if (spec->autocfg.dig_out_pin)
		spec->dig_out_nid = 0x05;
	if (spec->autocfg.dig_in_pin)
		spec->dig_in_nid = 0x04;

	return 0;
}

static int create_hp_detect_cmds(struct hda_auto_pin_cfg *cfg)
{
	struct hda_cmds cmds;
	struct hda_cmds jack_cmds[cfg->hp_outs + 1];
	struct hda_cmds present[cfg->line_outs + cfg->speaker_outs + 1];
	struct hda_cmds not_present[cfg->line_outs + cfg->speaker_outs + 1];
	int i, n, err;

	n = 0;
	for (i = 0; i < cfg->line_outs; i++) {
		/* disable lineouts, enable hp */
		present[n] = (struct hda_cmds)
			STD_VERB_UPDATE(cfg->line_out_pins[i],
					AC_VERB_SET_PIN_WIDGET_CONTROL,
					AC_PINCTL_OUT_EN,
					STD_INT(0));
		/* enable lineouts, disable hp */
		not_present[n] = (struct hda_cmds)
			STD_VERB_UPDATE(cfg->line_out_pins[i],
					AC_VERB_SET_PIN_WIDGET_CONTROL,
					AC_PINCTL_IN_EN|AC_PINCTL_OUT_EN,
					STD_INT(AC_PINCTL_OUT_EN));
		n++;
	}
	for (i = 0; i < cfg->speaker_outs; i++) {
		/* disable speaker, enable hp */
		present[n] = (struct hda_cmds)
			STD_VERB_UPDATE(cfg->speaker_pins[i],
					AC_VERB_SET_PIN_WIDGET_CONTROL,
					AC_PINCTL_OUT_EN,
					STD_INT(0));
		/* enable speaker, disable hp */
		not_present[n] = (struct hda_cmds)
			STD_VERB_UPDATE(cfg->speaker_pins[i],
					AC_VERB_SET_PIN_WIDGET_CONTROL,
					AC_PINCTL_IN_EN|AC_PINCTL_OUT_EN,
					STD_INT(AC_PINCTL_OUT_EN));
		n++;
	}
	present[n] = (struct hda_cmds) STD_NULL;
	not_present[n] = (struct hda_cmds) STD_NULL;

	/* check all HP pins */
	for (i = 0; i < cfg->hp_outs; i++)
		jack_cmds[i] = (struct hda_cmds)
			STD_AND(STD_VERB(cfg->hp_pins[i],
					 AC_VERB_GET_PIN_SENSE, 0),
				STD_INT(1U << 31));
	jack_cmds[i] = (struct hda_cmds) STD_NULL;

	cmds = (struct hda_cmds)
		STD_IF_ELSE(STD_LOR(jack_cmds),
			    STD_CMDS(present), STD_CMDS(not_present));
	err = hda_register_unsol_cmds(cfg->hp_pins[0], STAC_HP_EVENT, 26,
				      &cmds);
	if (err < 0)
		return err;

	/* execute a pin detection once at initialization */
	return hda_register_init_cmds(&cmds);
}

/*
 * Early 2006 Intel Macintoshes with STAC9220X5 codecs seem to have a
 * funky external mute control using GPIO pins.
 */

static int stac922x_gpio_mute(struct stac_spec *spec, int pin, int muted)
{
	struct hda_verb verbs[4];
	struct hda_cmds delay[] = {
		STD_DELAY(1),
		STD_NULL
	};
	unsigned int gpiostate, gpiomask, gpiodir;
	int err;

	gpiostate = hda_codec_read(codec_info.afg, AC_VERB_GET_GPIO_DATA, 0);
	if (!muted)
		gpiostate |= (1 << pin);
	else
		gpiostate &= ~(1 << pin);

	gpiomask = hda_codec_read(codec_info.afg, AC_VERB_GET_GPIO_MASK, 0);
	gpiomask |= (1 << pin);

	gpiodir = hda_codec_read(codec_info.afg, AC_VERB_GET_GPIO_DIRECTION, 0);
	gpiodir |= (1 << pin);

	memset(verbs, 0, sizeof(verbs));
	/* AppleHDA seems to do this -- WTF is this verb?? */
	verbs[0].nid = codec_info.afg;
	verbs[0].verb = 0x7e7;
	verbs[0].param = 0;

	verbs[1].nid = codec_info.afg;
	verbs[1].verb = AC_VERB_SET_GPIO_MASK;
	verbs[1].param = gpiomask;
	verbs[2].nid = codec_info.afg;
	verbs[2].verb = AC_VERB_SET_GPIO_DIRECTION;
	verbs[2].param = gpiodir;
	err = hda_register_init_verbs(verbs);
	if (err < 0)
		return err;

	err = hda_register_init_cmds(delay);
	if (err < 0)
		return err;

	memset(verbs, 0, sizeof(verbs));
	verbs[0].nid = codec_info.afg;
	verbs[0].verb = AC_VERB_SET_GPIO_DATA;
	verbs[0].param = gpiostate;
	err = hda_register_init_verbs(verbs);
	if (err < 0)
		return err;

	return 0;
}

static int enable_pin_detect(struct stac_spec *spec, hda_nid_t nid,
			      unsigned int event)
{
	if (hda_get_wcaps(nid) & AC_WCAP_UNSOL_CAP)
		return hda_append_verb(&spec->auto_init, nid,
				       AC_VERB_SET_UNSOLICITED_ENABLE,
				       (AC_USRSP_EN | event));
	return 0;
}

static int stac92xx_init(struct stac_spec *spec,
			 struct hda_auto_pin_cfg *cfg)
{
	int i, err;

	err = hda_register_init_verbs(spec->auto_init.verbs);
	if (err < 0)
		return err;
	hda_clear_verb_array(&spec->auto_init);

	/* set up pins */
	for (i = 0; i < AUTO_PIN_LAST; i++) {
		hda_nid_t nid = cfg->input_pins[i];
		if (nid) {
			unsigned int pinctl = AC_PINCTL_IN_EN;
			if (i == AUTO_PIN_MIC || i == AUTO_PIN_FRONT_MIC)
				pinctl |= stac92xx_get_vref(nid);
			err = auto_set_pinctl(spec, nid, pinctl);
			if (err < 0)
				return err;
		}
	}
	if (spec->num_dmics > 0)
		for (i = 0; i < spec->num_dmics; i++) {
			err = auto_set_pinctl(spec, spec->dmic_nids[i],
					      AC_PINCTL_IN_EN);
			if (err < 0)
				return err;
		}

	if (cfg->dig_out_pin) {
		err  = auto_set_pinctl(spec, cfg->dig_out_pin,
				       AC_PINCTL_OUT_EN);
		if (err < 0)
			return err;
	}
	if (cfg->dig_in_pin) {
		err = auto_set_pinctl(spec, cfg->dig_in_pin,
				      AC_PINCTL_IN_EN);
		if (err < 0)
			return err;
	}

	if (spec->gpio_mute) {
		err = stac922x_gpio_mute(spec, 0, 0);
		if (err < 0)
			return err;
		err = stac922x_gpio_mute(spec, 1, 0);
		if (err < 0)
			return err;
	}


	if (spec->hp_detect) {
		/* Enable unsolicited responses on the HP widget */
		for (i = 0; i < cfg->hp_outs; i++) {
			err = enable_pin_detect(spec, cfg->hp_pins[i],
						STAC_HP_EVENT);
			if (err < 0)
				return err;
		}
		/* force to enable the first line-out; the others are set up
		 * in unsol_event
		 */
		err = auto_set_pinctl(spec, cfg->line_out_pins[0],
				      AC_PINCTL_OUT_EN);
		if (err < 0)
			return err;
	} else {
		err = stac92xx_auto_init_multi_out(spec, cfg);
		if (err < 0)
			return err;
	}
	err = stac92xx_auto_init_hp_out(spec, cfg);
	if (err < 0)
		return err;

	err = hda_register_init_verbs(spec->auto_init.verbs);
	if (err < 0)
		return err;
	hda_clear_verb_array(&spec->auto_init);

	if (spec->hp_detect) {
		err = create_hp_detect_cmds(cfg);
		if (err < 0)
			return err;
	}

	return 0;
}

static void stac92xx_free(struct stac_spec *spec)
{
	if (!spec)
		return;

	hda_clear_verb_array(&spec->auto_init);
	free(spec);
}

static int build_stac92xx(const struct hda_codec_table *tbl,
			  struct stac_spec *spec)
{
	int err;

	err = stac92xx_init(spec, &spec->autocfg);
	if (err < 0)
		return err;
	err = stac92xx_build_controls(spec);
	if (err < 0)
		return err;
	err = stac92xx_build_pcms(spec);
	if (err < 0)
		return err;

	sprintf(codec_info.name, "%s %s", tbl->vendor_name, tbl->name);
	hda_update_codec_info();

	stac92xx_free(spec);
	return 0;
}

static int patch_stac9200(const struct hda_codec_table *tbl,
			  int board_config, const char **args)
{
	struct stac_spec *spec;
	int err;

	spec = calloc(1, sizeof(*spec));
	if (spec == NULL)
		return -ENOMEM;

	spec->num_pins = 8;
	spec->pin_nids = stac9200_pin_nids;
	err = stac92xx_save_pin_config_regs(spec);
	if (err < 0)
		goto error;

	spec->num_dacs = 1;
	spec->dac_nids = stac9200_dac_nids;
	spec->adc_nids = stac9200_adc_nids;
	spec->mux_nids = stac9200_mux_nids;
	spec->num_muxes = 1;
	spec->num_dmics = 0;

	err = hda_register_init_verbs(stac9200_core_init);
	if (err < 0)
		goto error;
	err = hda_register_mixers(stac9200_mixer);
	if (err < 0)
		goto error;

	err = stac9200_parse_auto_config(spec);
	if (err < 0)
		goto error;

	return build_stac92xx(tbl, spec);

 error:
	stac92xx_free(spec);
	return err;
}

static int patch_stac925x(const struct hda_codec_table *tbl,
			  int board_config, const char **args)
{
	struct stac_spec *spec;
	int err;

	spec = calloc(1, sizeof(*spec));
	if (spec == NULL)
		return -ENOMEM;

	spec->num_pins = 8;
	spec->pin_nids = stac925x_pin_nids;
	err = stac92xx_save_pin_config_regs(spec);
	if (err < 0)
		goto error;

	spec->num_dacs = 1;
	spec->dac_nids = stac925x_dac_nids;
	spec->adc_nids = stac925x_adc_nids;
	spec->mux_nids = stac925x_mux_nids;
	spec->num_muxes = 1;

	switch (codec_info.vendor_id) {
	case 0x83847632: /* STAC9202  */
	case 0x83847633: /* STAC9202D */
	case 0x83847636: /* STAC9251  */
	case 0x83847637: /* STAC9251D */
		spec->num_dmics = 1;
		spec->dmic_nids = stac925x_dmic_nids;
		break;
	default:
		spec->num_dmics = 0;
		break;
	}

	err = hda_register_init_verbs(stac925x_core_init);
	if (err < 0)
		goto error;
	err = hda_register_mixers(stac925x_mixer);
	if (err < 0)
		goto error;

	err = stac92xx_parse_auto_config(spec, 0x8, 0x7);
	if (err < 0)
		goto error;

	return build_stac92xx(tbl, spec);

 error:
	stac92xx_free(spec);
	return err;
}

static int patch_stac922x(const struct hda_codec_table *tbl,
			  int board_config, const char **args)
{
	struct stac_spec *spec;
	int err;

	spec = calloc(1, sizeof(*spec));
	if (spec == NULL)
		return -ENOMEM;

	spec->num_pins = 10;
	spec->pin_nids = stac922x_pin_nids;
	err = stac92xx_save_pin_config_regs(spec);
	if (err < 0)
		goto error;

	spec->adc_nids = stac922x_adc_nids;
	spec->mux_nids = stac922x_mux_nids;
	spec->num_muxes = ARRAY_SIZE(stac922x_mux_nids);
	spec->num_dmics = 0;

	err = hda_register_init_verbs(stac922x_core_init);
	if (err < 0)
		goto error;
	err = hda_register_mixers(stac922x_mixer);
	if (err < 0)
		goto error;

	/* Intel Mac needs a GPIO hack */
	if (codec_info.pci_subvendor == 0x8384 &&
	    codec_info.pci_subdevice == 0x7680)
		spec->gpio_mute = 1;

	err = stac92xx_parse_auto_config(spec, 0x08, 0x09);
	if (err < 0)
		goto error;

	return build_stac92xx(tbl, spec);

 error:
	stac92xx_free(spec);
	return err;
}

static int patch_stac927x(const struct hda_codec_table *tbl,
			  int board_config, const char **args)
{
	struct stac_spec *spec;
	int err;
	struct hda_verb *init;
	struct hda_std_mixer *mixer;

	spec = calloc(1, sizeof(*spec));
	if (spec == NULL)
		return -ENOMEM;

	spec->num_pins = 14;
	spec->pin_nids = stac927x_pin_nids;
	err = stac92xx_save_pin_config_regs(spec);
	if (err < 0)
		goto error;

	spec->adc_nids = stac927x_adc_nids;
	spec->mux_nids = stac927x_mux_nids;
	spec->num_muxes = ARRAY_SIZE(stac927x_mux_nids);
	spec->num_dmics = 0;

	init = stac927x_core_init;
	mixer = stac927x_mixer;
	for (; args && *args; args++) {
		if (!strcmp(*args, "preset=d965")) {
			init = d965_core_init;
			mixer = stac9227_mixer;
			break;
		}
	}

	err = hda_register_init_verbs(init);
	if (err < 0)
		goto error;
	err = hda_register_mixers(mixer);
	if (err < 0)
		goto error;

	err = stac92xx_parse_auto_config(spec, 0x1e, 0x20);
	if (err < 0)
		goto error;

	return build_stac92xx(tbl, spec);

 error:
	stac92xx_free(spec);
	return err;
}

static int patch_stac9205(const struct hda_codec_table *tbl,
			  int board_config, const char **args)
{
	struct stac_spec *spec;
	int err;

	spec = calloc(1, sizeof(*spec));
	if (spec == NULL)
		return -ENOMEM;

	spec->num_pins = 14;
	spec->pin_nids = stac9205_pin_nids;

	err = stac92xx_save_pin_config_regs(spec);
	if (err < 0)
		goto error;

	spec->adc_nids = stac9205_adc_nids;
	spec->mux_nids = stac9205_mux_nids;
	spec->num_muxes = ARRAY_SIZE(stac9205_mux_nids);
	spec->dmic_nids = stac9205_dmic_nids;
	spec->num_dmics = ARRAY_SIZE(stac9205_dmic_nids);
	spec->dmux_nid = 0x1d;

	err = hda_register_init_verbs(stac9205_core_init);
	if (err < 0)
		goto error;
	err = hda_register_mixers(stac9205_mixer);
	if (err < 0)
		goto error;

	/* Configure GPIO0 as EAPD output */
	err = hda_append_verb(&spec->auto_init, codec_info.afg,
			      AC_VERB_SET_GPIO_DIRECTION, 0x00000001);
	if (err < 0)
		goto error;
	/* Configure GPIO0 as CMOS */
	err = hda_append_verb(&spec->auto_init, codec_info.afg,
			      0x7e7, 0x00000000);
	if (err < 0)
		goto error;
	/* Assert GPIO0 high */
	err = hda_append_verb(&spec->auto_init, codec_info.afg,
			      AC_VERB_SET_GPIO_DATA, 0x00000001);
	if (err < 0)
		goto error;
	/* Enable GPIO0 */
	err = hda_append_verb(&spec->auto_init, codec_info.afg,
			      AC_VERB_SET_GPIO_MASK, 0x00000001);
	if (err < 0)
		goto error;
	err = hda_register_init_verbs(spec->auto_init.verbs);
	if (err < 0)
		goto error;
	hda_clear_verb_array(&spec->auto_init);

	err = stac92xx_parse_auto_config(spec, 0x1f, 0x20);
	if (err < 0)
		goto error;

	return build_stac92xx(tbl, spec);

 error:
	stac92xx_free(spec);
	return err;
}

/*
 * patch entries
 */
static struct hda_codec_table stac_presets[] = {
	{ .id = 0x83847690, .vendor_name = "IDT", .name = "STAC9200",
	  .patch = patch_stac9200 },
	{ .id = 0x83847882, .vendor_name = "IDT", .name = "STAC9220 A1",
	  .patch = patch_stac922x },
	{ .id = 0x83847680, .vendor_name = "IDT", .name = "STAC9221 A1",
	  .patch = patch_stac922x },
	{ .id = 0x83847880, .vendor_name = "IDT", .name = "STAC9220 A2",
	  .patch = patch_stac922x },
	{ .id = 0x83847681, .vendor_name = "IDT", .name = "STAC9220D/9223D A2",
	  .patch = patch_stac922x },
	{ .id = 0x83847682, .vendor_name = "IDT", .name = "STAC9221 A2",
	  .patch = patch_stac922x },
	{ .id = 0x83847683, .vendor_name = "IDT", .name = "STAC9221D A2",
	  .patch = patch_stac922x },
	{ .id = 0x83847618, .vendor_name = "IDT", .name = "STAC9227",
	  .patch = patch_stac927x },
	{ .id = 0x83847619, .vendor_name = "IDT", .name = "STAC9227",
	  .patch = patch_stac927x },
	{ .id = 0x83847616, .vendor_name = "IDT", .name = "STAC9228",
	  .patch = patch_stac927x },
	{ .id = 0x83847617, .vendor_name = "IDT", .name = "STAC9228",
	  .patch = patch_stac927x },
	{ .id = 0x83847614, .vendor_name = "IDT", .name = "STAC9229",
	  .patch = patch_stac927x },
	{ .id = 0x83847615, .vendor_name = "IDT", .name = "STAC9229",
	  .patch = patch_stac927x },
	{ .id = 0x83847620, .vendor_name = "IDT", .name = "STAC9274",
	  .patch = patch_stac927x },
	{ .id = 0x83847621, .vendor_name = "IDT", .name = "STAC9274D",
	  .patch = patch_stac927x },
	{ .id = 0x83847622, .vendor_name = "IDT", .name = "STAC9273X",
	  .patch = patch_stac927x },
	{ .id = 0x83847623, .vendor_name = "IDT", .name = "STAC9273D",
	  .patch = patch_stac927x },
	{ .id = 0x83847624, .vendor_name = "IDT", .name = "STAC9272X",
	  .patch = patch_stac927x },
	{ .id = 0x83847625, .vendor_name = "IDT", .name = "STAC9272D",
	  .patch = patch_stac927x },
	{ .id = 0x83847626, .vendor_name = "IDT", .name = "STAC9271X",
	  .patch = patch_stac927x },
	{ .id = 0x83847627, .vendor_name = "IDT", .name = "STAC9271D",
	  .patch = patch_stac927x },
	{ .id = 0x83847628, .vendor_name = "IDT", .name = "STAC9274X5NH",
	  .patch = patch_stac927x },
	{ .id = 0x83847629, .vendor_name = "IDT", .name = "STAC9274D5NH",
	  .patch = patch_stac927x },
	{ .id = 0x83847632, .vendor_name = "IDT", .name = "STAC9202",
	  .patch = patch_stac925x },
	{ .id = 0x83847633, .vendor_name = "IDT", .name = "STAC9202D",
	  .patch = patch_stac925x },
	{ .id = 0x83847634, .vendor_name = "IDT", .name = "STAC9250",
	  .patch = patch_stac925x },
	{ .id = 0x83847635, .vendor_name = "IDT", .name = "STAC9250D",
	  .patch = patch_stac925x },
	{ .id = 0x83847636, .vendor_name = "IDT", .name = "STAC9251",
	  .patch = patch_stac925x },
	{ .id = 0x83847637, .vendor_name = "IDT", .name = "STAC9250D",
	  .patch = patch_stac925x },
	{ .id = 0x838476a0, .vendor_name = "IDT", .name = "STAC9205",
	  .patch = patch_stac9205 },
	{ .id = 0x838476a1, .vendor_name = "IDT", .name = "STAC9205D",
	  .patch = patch_stac9205 },
	{ .id = 0x838476a2, .vendor_name = "IDT", .name = "STAC9204",
	  .patch = patch_stac9205 },
	{ .id = 0x838476a3, .vendor_name = "IDT", .name = "STAC9204D",
	  .patch = patch_stac9205 },
	{ .id = 0x838476a4, .vendor_name = "IDT", .name = "STAC9255",
	  .patch = patch_stac9205 },
	{ .id = 0x838476a5, .vendor_name = "IDT", .name = "STAC9255D",
	  .patch = patch_stac9205 },
	{ .id = 0x838476a6, .vendor_name = "IDT", .name = "STAC9254",
	  .patch = patch_stac9205 },
	{ .id = 0x838476a7, .vendor_name = "IDT", .name = "STAC9254D",
	  .patch = patch_stac9205 },
	{} /* terminator */
};

const struct hda_codec_table *patch_descriptor(void)
{
	return stac_presets;
}
