/*
 * Helper functions for codec modules
 *
 * Copyright (C) 2007 Takashi Iwai
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 */

#include "config.h"
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include "local.h"
#include "hda/codecs-helper.h"

/*
 * PCM
 */
static struct hda_usr_pcm_info pcm_analog_out = {
	.device = 0,
	.type = HDA_PCM_TYPE_ANALOG_MULTI_OUT,
	.channels_max = 2,
	.channels_min = 2,
	.substreams = 1,
};

static struct hda_usr_pcm_info pcm_analog_in = {
	.device = 0,
	.type = HDA_PCM_TYPE_ANALOG_IN,
	.channels_max = 2,
	.channels_min = 2,
	.substreams = 1,
};

static struct hda_usr_pcm_info pcm_spdif_out = {
	.device = 1,
	.type = HDA_PCM_TYPE_SPDIF_OUT,
	.channels_max = 2,
	.channels_min = 2,
	.substreams = 1,
};

static struct hda_usr_pcm_info pcm_spdif_in = {
	.device = 1,
	.type = HDA_PCM_TYPE_SPDIF_IN,
	.channels_max = 2,
	.channels_min = 2,
	.substreams = 1,
};

int codec_build_preset(const struct hda_codec_table *tbl,
		       struct codec_config_preset *preset)
{
	struct hda_std_mixer **mix;
	struct hda_verb **verb;
	struct codec_config_unsol **unsol;
	struct hda_usr_pcm_info **pcm;
	int i, err;

	for (verb = preset->init_verbs; *verb; verb++) {
		err = hda_register_init_verbs(*verb);
		if (err < 0)
			return err;
	}
	for (mix = preset->mixers; *mix; mix++) {
		err = hda_register_mixers(*mix);
		if (err < 0)
			return err;
	}

	if (preset->input_mux) {
		hda_nid_t nid = preset->input_mux_nid ?
			preset->input_mux_nid : preset->adc_nid;
		if (preset->input_mux_pins) {
			struct hda_input_mux *nmux;
			nmux = hda_fixup_input_mux(nid,
						   preset->input_mux,
						   preset->input_mux_pins);
			if (nmux) {
				err = hda_register_input_mux_mixer("Capture Source",
								   nid, nmux);
				hda_input_mux_free(nmux);
				if (err < 0)
					return err;
			}
		} else {
			err = hda_register_input_mux_mixer("Capture Source",
							   nid,
							   preset->input_mux);
			if (err < 0)
				return err;
		}
	}

	if (preset->num_dacs) {
		sprintf(pcm_analog_out.name, "%s Analog", tbl->name);
		pcm_analog_out.channels_max = preset->num_dacs * 2;
		pcm_analog_out.nid = preset->dac_nids[0];
		memcpy(pcm_analog_out.assoc_nids, preset->dac_nids,
		       preset->num_dacs * sizeof(hda_nid_t));
		if (preset->extra_nids) {
			for (i = 0; preset->extra_nids[i]; i++)
				pcm_analog_out.extra_nids[i] =
					preset->extra_nids[i];
		}
		err = hda_register_pcm(&pcm_analog_out);
		if (err < 0)
			return err;
	}

	if (preset->adc_nid) {
		sprintf(pcm_analog_in.name, "%s Analog", tbl->name);
		pcm_analog_in.nid = preset->adc_nid;
		pcm_analog_in.assoc_nids[0] = preset->adc_nid;
		err = hda_register_pcm(&pcm_analog_in);
		if (err < 0)
			return err;
	}

	if (preset->dig_out_nid) {
		sprintf(pcm_spdif_out.name, "%s Digital", tbl->name);
		pcm_spdif_out.nid = preset->dig_out_nid;
		pcm_spdif_out.assoc_nids[0] = preset->dig_out_nid;
		err = hda_register_pcm(&pcm_spdif_out);
		if (err < 0)
			return err;
	}
	if (preset->dig_in_nid) {
		sprintf(pcm_spdif_in.name, "%s Digital", tbl->name);
		pcm_spdif_in.nid = preset->dig_in_nid;
		pcm_spdif_in.assoc_nids[0] = preset->dig_in_nid;
		err = hda_register_pcm(&pcm_spdif_in);
		if (err < 0)
			return err;
	}

	for (pcm = preset->extra_pcms; *pcm; pcm++) {
		err = hda_register_pcm(*pcm);
		if (err < 0)
			return err;
	}

	for (unsol = preset->unsols; *unsol; unsol++) {
		err = hda_register_unsol_cmds((*unsol)->nid,
					      (*unsol)->tag,
					      (*unsol)->shift ? (*unsol)->shift : 26,
					      &(*unsol)->cmds);
		if (err < 0)
			return err;
	}

	if (tbl->name) {
		if (tbl->vendor_name)
			snprintf(codec_info.name, sizeof(codec_info.name),
				 "%s %s", tbl->vendor_name, tbl->name);
		else
			snprintf(codec_info.name, sizeof(codec_info.name),
				 "%s", tbl->name);
		hda_update_codec_info();
	}

	return 0;
}

int codec_preset_add_init_verb(struct codec_config_preset *preset,
			       struct hda_verb *verb)
{
	int nverbs = 0;
	struct hda_verb **v;

	if (!verb || !verb->nid)
		return 0;
	for (v = preset->init_verbs; *v; v++, nverbs++) {
		if (nverbs >= ARRAY_SIZE(preset->init_verbs) - 1)
			log_error("Too many init verbs to add\n");
			return -ENOMEM;
	}
	*v = verb;
	return 0;
}

int codec_preset_add_mixer(struct codec_config_preset *preset,
			   struct hda_std_mixer *mix)
{
	int nmix = 0;
	struct hda_std_mixer **m;

	if (!mix || !mix->head.name)
		return 0;
	for (m = preset->mixers; *m; m++, nmix++) {
		if (nmix >= ARRAY_SIZE(preset->mixers) - 1)
			log_error("Too many mixers to add\n");
			return -ENOMEM;
	}
	*m = mix;
	return 0;
}
