/*
 * Lowlevel TLV interface
 *
 * Copyright (C) 2007 Takashi Iwai
 * 
 *  This library is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License version
 *  2.1 as published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 */

#include "config.h"
#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include "local.h"

/*
 * allocate a TLV record with the given length
 *
 * the header is initialized using the given values
 */
struct hda_tlv *hda_alloc_tlv(int type, int len)
{
	struct hda_tlv *tlv;

	tlv = malloc(len + 4);
	if (!tlv)
		return NULL;
	tlv->type = type;
	tlv->len = len;
	return tlv;
}

/*
 * make a TLV list record containing the given entries
 */
struct hda_tlv *hda_make_tlv_args(int type, int nargs, struct hda_tlv *args[])
{
	struct hda_tlv *tlv;
	int i, size;
	char *data;

	size = 0;
	for (i = 0; i < nargs; i++)
		size += hda_tlv_len(args[i]);
	tlv = hda_alloc_tlv(type, size);
	if (!tlv)
		return NULL;
	data = (char *)tlv->val;
	for (i = 0; i < nargs; i++) {
		memcpy(data, args[i], hda_tlv_len(args[i]));
		data += hda_tlv_len(args[i]);
	}
	return tlv;
}

/*
 * make a TLV list record containing the given vararg entries
 */
struct hda_tlv *hda_make_tlv(int type, struct hda_tlv *arg, ...)
{
	va_list ap;
	struct hda_tlv *args[MAX_TLV_ARGS];
	struct hda_tlv *tlv;
	int nargs = 0;

	va_start(ap, arg);
	while (arg && nargs < MAX_TLV_ARGS) {
		args[nargs++] = arg;
		arg = va_arg(ap, struct hda_tlv *);
	}
	tlv = hda_make_tlv_args(type, nargs, args);
	va_end(ap);
	return tlv;
}

/*
 * create a 32bit integer TLV entry
 */
struct hda_tlv *hda_make_int_tlv(int type, int val)
{
	struct hda_tlv *tlv;

	tlv = hda_alloc_tlv(type, sizeof(val));
	if (!tlv)
		return NULL;
	memcpy(tlv->val, &val, sizeof(val));
	return tlv;
}

/*
 * create a string TLV entry
 */
struct hda_tlv *hda_make_str_tlv(int type, const char *str)
{
	struct hda_tlv *tlv;
	int len;

	len = strlen(str) + 1;
	len = ((len + 3) / 4) * 4; /* round up to 4 bytes */
	tlv = hda_alloc_tlv(type, len);
	if (!tlv)
		return NULL;
	strcpy(tlv->val, str);
	return tlv;
}
