/*
 * invoking periodical RTC interrupts and report the stack pointers
 * 
 * for i386 only!
 */

#include <linux/config.h>
#include <linux/interrupt.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/kernel.h>
#include <linux/rtc.h>
#include <linux/init.h>
#include <linux/spinlock.h>
#include <linux/time.h>
#include <linux/sched.h>
#include <linux/ioctl.h>
#include <linux/fs.h>
#include <linux/kallsyms.h>
#include <asm/uaccess.h>
#include "latencytest.h"


static spinlock_t my_lock = SPIN_LOCK_UNLOCKED;

static int rtc_freq = 1024;
static rtc_task_t rtc_task;
static int count, irq_count;

static wait_queue_head_t my_sleep;

static struct latency_test_info irq_info;
static struct latency_test_info saved_info;

static int opened;
static int rtc_running;

#ifdef __i386__
static inline void copy_stack(int pcnt)
{
	unsigned long stack;
	int i;
	unsigned long *dump = &irq_info.stacks[pcnt][0];
	unsigned short idx;
	unsigned short *posp = &irq_info.stack_pos[pcnt][0];
	unsigned long *stackp = &stack + 1;
	i = 0;
	idx = 0;
	while (! kstack_end(stackp)) {
		unsigned long addr, check;
		addr = *stackp++;
		idx++;
#if 0
		/* if i could use kernel_text_address()... */
		if (! kernel_text_address(addr))
			continue;
#endif
		check = addr & 0xf0000000;
		if (check != 0xc0000000 && check != 0xd0000000)
			continue;
		*dump++ = addr;
		*posp++ = idx;
		if (++i >= MAX_STACK)
			break;
	}
	memcpy(&irq_info.comm[pcnt][0], current->comm, 16);
}
#elif defined(__x86_64__)
static inline void copy_stack(int pcnt)
{
	unsigned long stack;
	int i;
	unsigned long *dump = &irq_info.stacks[pcnt][0];
	unsigned short idx;
	unsigned short *posp = &irq_info.stack_pos[pcnt][0];
	unsigned long *stackp = &stack + 1;
	i = 0;
	idx = 0;
	while (((long) stackp & (THREAD_SIZE-1)) != 0) {
		unsigned long addr;
		addr = *stackp++;
		idx++;
		if (! kernel_text_address(addr))
			continue;
		*dump++ = addr;
		*posp++ = idx;
		if (++i >= MAX_STACK)
			break;
	}
	memcpy(&irq_info.comm[pcnt][0], current->comm, 16);
}
#elif defined(__ppc__)
static inline void copy_stack(int pcnt)
{
	unsigned long sp, stack_top, prev_sp, ret;
	int i;
	unsigned long *dump = &irq_info.stacks[pcnt][0];
	unsigned short idx;
	unsigned short *posp = &irq_info.stack_pos[pcnt][0];
	i = 0;
	idx = 0;

	asm("mr %0,1" : "=r" (sp));

	prev_sp = (unsigned long) (current->thread_info + 1);
	stack_top = (unsigned long) current->thread_info + THREAD_SIZE;
	sp = *(unsigned long *)sp;
	while (sp > prev_sp && sp < stack_top && (sp & 3) == 0) {
		unsigned long addr = *(unsigned long *)(sp + 4);
		idx++;
#if 0
		if (! kernel_text_address(addr))
			continue;
#endif
		*dump++ = addr;
		*posp++ = idx;
		if (++i >= MAX_STACK)
			break;
		sp = *(unsigned long *)sp;
	}
	memcpy(&irq_info.comm[pcnt][0], current->comm, 16);
}
#else
#error undefined architectures!
#endif

static void my_interrupt(void *private_data)
{
	unsigned int pcnt;

	spin_lock(&my_lock);
	count++;
	if (count < irq_count)
		return;
	count = 0;
	pcnt = irq_info.processed;
	irq_info.processed++;
	if (pcnt < MAX_PROC_CNTS)
		copy_stack(pcnt);
	wake_up(&my_sleep);
	spin_unlock(&my_lock);
}


static int my_start(void)
{
	if (irq_count) {
		rtc_control(&rtc_task, RTC_IRQP_SET, rtc_freq);
		rtc_control(&rtc_task, RTC_PIE_ON, 0);
		rtc_running = 1;
		return 0;
	}
	return -EINVAL;
}

static void my_stop(void)
{
	if (rtc_running)
		rtc_control(&rtc_task, RTC_PIE_OFF, 0);
	rtc_running = 0;
}

static int my_open(struct inode *inode, struct file *file)
{
	int err;

	if (opened)
		return -EBUSY;

	irq_count = 0;
	count = 0;
	memset(&irq_info, 0, sizeof(irq_info));
	init_waitqueue_head(&my_sleep);
	rtc_task.func = my_interrupt;
	err = rtc_register(&rtc_task);
	if (err < 0)
		return err;

	opened = 1;
	return 0;
}

static int my_release(struct inode *inode, struct file *file)
{
	if (opened) {
		my_stop();
		rtc_unregister(&rtc_task);
		opened = 0;
	}
	return 0;
}

static int my_read(unsigned long arg)
{
	wait_queue_t wait;

	init_waitqueue_entry(&wait, current);
	spin_lock_irq(&my_lock);
	add_wait_queue(&my_sleep, &wait);
	while (! irq_info.processed) {
		if (signal_pending(current)) {
			remove_wait_queue(&my_sleep, &wait);
			spin_unlock_irq(&my_lock);
			return -EINTR;
		}
		set_current_state(TASK_INTERRUPTIBLE);
		spin_unlock_irq(&my_lock);
		schedule();
		spin_lock_irq(&my_lock);
	}
	remove_wait_queue(&my_sleep, &wait);
	memcpy(&saved_info, &irq_info, sizeof(irq_info));
	memset(&irq_info, 0, sizeof(irq_info));
	spin_unlock_irq(&my_lock);

	if (copy_to_user((void*)arg, &saved_info, sizeof(saved_info)))
		return -EFAULT;
	return 0;
}

static int my_ioctl(struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg)
{
	int i;
	switch (cmd) {
	case LAT_TEST_FREQ:
		for (i = 0; arg > (1 << i); i++)
			;
		if (arg != (1 << i))
			return -EINVAL;
		rtc_freq = arg;
		return 0;
	case LAT_TEST_COUNT:
		irq_count = arg;
		return 0;
	case LAT_TEST_START:
		return my_start();
	case LAT_TEST_STOP:
		my_stop();
		return 0;
	case LAT_TEST_READ:
		return my_read(arg);
	}
	printk(KERN_ERR "irqtest: unknown ioctl cmd=0x%x\n", cmd);
	return -EINVAL;
}

static struct file_operations my_fops = {
	.owner		= THIS_MODULE,
	.ioctl		= my_ioctl,
	.open		= my_open,
	.release	= my_release,
};

static int major = 35;

module_param(major, int, 0444);
MODULE_PARM_DESC(major, "Major number of the device file to access");

static int __init irq_test_init(void)
{
	if (register_chrdev(major, "irqtest", &my_fops)) {
		printk(KERN_ERR "cannot register device\n");
		return -ENODEV;
	}
	printk(KERN_ERR "irq-test registered\n");
	return 0;
}

static void __exit irq_test_exit(void)
{
	unregister_chrdev(major, "irqtest");
	printk(KERN_ERR "irq-test de-registered\n");
}

module_init(irq_test_init);
module_exit(irq_test_exit);

MODULE_LICENSE("GPL");
