/*
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include "measure.h"

struct kernel_entry {
	unsigned long addr;
	char *func;
};

static int num_ents;
static struct kernel_entry *entries;
static unsigned long addr_valid_range = 0x10000;

static int show_hex = 0;
static int with_proc = 0;

static int compare(const void *a, const void *b)
{
	const struct kernel_entry *ap, *bp;
	ap = a;
	bp = b;
	if ((long)ap->addr < (long)bp->addr)
		return -1;
	else
		return 1;
}

static void load_table(char *ksymfile)
{
	FILE *f;
	char tmp[256];
	int i;

	if ((f = fopen(ksymfile, "r")) == NULL) {
		fprintf(stderr, "can't open ksym file %s\n", ksymfile);
		exit(1);
	}

	num_ents = 0;
	while (fgets(tmp, sizeof(tmp), f))
		num_ents++;

	entries = malloc(sizeof(*entries) * num_ents);
	if (entries == NULL) {
		fprintf(stderr, "can't malloc %d entries\n", num_ents);
		exit(1);
	}

	rewind(f);
	for (i = 0; i < num_ents; i++) {
		unsigned int addr;
		unsigned char type;
		char func[256];
		if (! fgets(tmp, sizeof(tmp), f)) {
			num_ents = i;
			break;
		}
		sscanf(tmp, "%x %c %s", &addr, &type, func);
		entries[i].addr = addr;
		entries[i].func = strdup(func);
		if (! entries[i].func) {
			fprintf(stderr, "can't malloc name\n");
			exit(1);
		}
	}
	qsort(entries, num_ents, sizeof(*entries), compare);
	fclose(f);
}


/* grrr, dumb searching... */
static struct kernel_entry *lookup(unsigned long addr)
{
	int i;
	struct kernel_entry *e = entries;
	if (addr < e[0].addr)
		return NULL;
	for (i = 0; i < num_ents - 1; i++) {
		if (e[1].addr > addr) {
			if (addr - e->addr >= addr_valid_range)
				return NULL;
			return e;
		}
		e++;
	}
	return NULL;
}

static void show_trace(struct latency_test_info *info)
{
	int i, j;

	for (i = 0; i < info->processed && i < MAX_PROC_CNTS; i++) {
		if (i > 0)
			printf("#%d\n", i);
		if (with_proc) {
			char comm[17];
			strncpy(comm, &info->comm[i][0], 16);
			comm[16] = 0;
			printf("  comm=%s\n", comm);
		}
		for (j = 0; j < MAX_STACK; j++) {
			struct kernel_entry *e;
			if (! info->stacks[i][j])
				break;
			if (show_hex)
				printf("  <%08lx>", info->stacks[i][j]);
			e = lookup(info->stacks[i][j]);
			if (e)
				printf(" %s (+%lx)\n", e->func, info->stacks[i][j] - e->addr);
			else if (show_hex)
				printf("\n");
		}
	}
}

static void check_dump(char *file, float threshold)
{
	FILE *fp;
	struct latency_info info;
	struct latency_test_info stack;

	if ((fp = fopen(file, "r")) == NULL) {
		fprintf(stderr, "can't open %s\n", file);
		exit(1);
	}

	while (! feof(fp)) {
		int show = 0;
		fread(&info, sizeof(info), 1, fp);
		if (info.with_stack)
			fread(&stack, sizeof(stack), 1, fp);
		if (threshold == 0) {
			if (info.with_stack)
				show = 1;
		} else if (info.sched_diff > threshold)
			show = 1;
		if (show) {
			printf("T=%g diff=%g\n", info.elapsed,
			       info.sched_diff * 1000.0);
			if (info.with_stack)
				show_trace(&stack);
			printf("\n");
		}
	}

	fclose(fp);
}

static void usage(void)
{
	fprintf(stderr, "usage: showtrace [-opts] data-file [threshold(ms)]\n");
}

int main(int argc, char **argv)
{
	int c;
	float threshold = 0;
	static char *ksymfile = "/proc/kallsyms";

	while ((c = getopt(argc, argv, "k:hpr:)) != -1) {
		switch (c) {
		case 'k':
			ksymfile = optarg;
			break;
		case 'h':
			show_hex = 1;
			break;
		case 'p':
			with_proc = 1;
			break;
		case 'r':
			addr_valid_range = strtol(optarg, NULL, 0);
			break;
		default:
			usage();
			return 1;
		}
	}

	if (argc - optind < 1) {
		usage();
		return 1;
	}

	if (argc > optind + 1)
		threshold = atof(argv[optind + 1]) / 1000.0;

	load_table(ksymfile);
	check_dump(argv[optind], threshold);
	return 0;
}
