/*
   iecset - change IEC958 status bits on ALSA
   Copyright (C) 2003 by Takashi Iwai <tiwai@suse.de>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <stdio.h>
#include <alsa/asoundlib.h>

extern void dump_iec958(snd_aes_iec958_t *iec);

static void usage(void)
{
	printf("Usage: iecdump [-D device] [cmds]\n");
	printf(" common:\n");
	printf("   -p bool  professional (1) / consumer mode (0)\n");
	printf("   -r rate  sample rate\n");
	printf("   -a bool  non-audio (1) / audio (0)\n");
	printf("   -e type  none (0) / 50/15us (1) / CCITT (2)\n");
	printf(" professional mode only:\n");
	printf("   -l bool  unlocked (1) / locked (0)\n");
	printf("   -s type  20bit (2) / 24bit (4) / udef (6)\n");
	printf("   -w type  no (0) / 22-18bit (2) / 23-19bit (4) / 24-20bit (5) / 20-16bit (6)\n");
	printf(" consumer mode only:\n");
	printf("   -C type  category code (0-0x7f)\n");
	printf("   -c bool  non-copyright (1) / copyright (0)\n");
	printf("   -o bool  original (1) / 1st-gen (0)\n");
}

static int get_bool(const char *str)
{
	if (strcmp(str, "yes") == 0 ||
	    strcmp(str, "YES") == 0 ||
	    strcmp(str, "on") == 0 ||
	    strcmp(str, "ON") == 0 ||
	    *str == '1')
		return 1;
	return 0;
}

static int get_int(const char *str)
{
	return (int)strtol(str, NULL, 0);
}

int main(int argc, char **argv)
{
	const char *dev = "default";
	const char *spdif_str = "IEC958 Playback Default";
	snd_ctl_t *ctl;
	snd_ctl_elem_value_t *cval;
	snd_aes_iec958_t iec958;
	int c;
	int prof = -1, rate = -1, nonaudio = -1, not_copy = -1, orig = -1;
	int emphasis = -1, unlocked = -1, sbits = -1, wlen = -1, category= -1;

	while ((c = getopt(argc, argv, "D:p:r:a:e:l:s:w:C:c:o:h")) != -1) {
		switch (c) {
		case 'D':
			dev = optarg;
			break;
		case 'p':
			prof = get_bool(optarg);
			break;
		case 'r':
			rate = get_int(optarg);
			break;
		case 'a':
			nonaudio = get_bool(optarg);
			break;
		case 'e':
			emphasis = get_int(optarg);
			break;
		case 'l':
			unlocked = get_bool(optarg);
			break;
		case 's':
			sbits = get_int(optarg);
			break;
		case 'w':
			wlen = get_int(optarg);
			break;
		case 'C':
			category = get_int(optarg);
			break;
		case 'c':
			not_copy = get_bool(optarg);
			break;
		case 'o':
			orig = get_bool(optarg);
			break;
		default:
			usage();
			return 1;
		}
	}

	if (snd_ctl_open(&ctl, dev, 0) < 0) {
		perror("snd_ctl_open");
		return 1;
	}

	snd_ctl_elem_value_alloca(&cval);
	snd_ctl_elem_value_set_interface(cval, SND_CTL_ELEM_IFACE_MIXER);
	snd_ctl_elem_value_set_name(cval, spdif_str);
	if (snd_ctl_elem_read(ctl, cval) < 0) {
		snd_ctl_elem_value_set_interface(cval, SND_CTL_ELEM_IFACE_PCM);
		if (snd_ctl_elem_read(ctl, cval) < 0) {
			perror("snd_ctl_elem_read");
			return 1;
		}
	}

	snd_ctl_elem_value_get_iec958(cval, &iec958);

	if (prof >= 0) {
		if (prof)
			iec958.status[0] |= IEC958_AES0_PROFESSIONAL;
		else
			iec958.status[0] &= ~IEC958_AES0_PROFESSIONAL;
	}
	if (nonaudio >= 0) {
		if (nonaudio)
			iec958.status[0] |= IEC958_AES0_NONAUDIO;
		else
			iec958.status[0] &= ~IEC958_AES0_NONAUDIO;
	}
	if (rate >= 0) {
		if (iec958.status[0] & IEC958_AES0_PROFESSIONAL) {
			iec958.status[0] &= ~IEC958_AES0_PRO_FS;
			switch (rate) {
			case 44100:
				iec958.status[0] |= IEC958_AES0_PRO_FS_44100;
				break;
			case 48000:
				iec958.status[0] |= IEC958_AES0_PRO_FS_48000;
				break;
			case 3200:
				iec958.status[0] |= IEC958_AES0_PRO_FS_32000;
				break;
			}
		} else {
			iec958.status[3] &= ~IEC958_AES3_CON_FS;
			switch (rate) {
			case 44100:
				iec958.status[3] |= IEC958_AES3_CON_FS_44100;
				break;
			case 48000:
				iec958.status[3] |= IEC958_AES3_CON_FS_48000;
				break;
			case 3200:
				iec958.status[3] |= IEC958_AES3_CON_FS_32000;
				break;
			}
		}
	}
	if (not_copy >= 0) {
		if (! (iec958.status[0] & IEC958_AES0_PROFESSIONAL)) {
			if (not_copy)
				iec958.status[0] |= IEC958_AES0_CON_NOT_COPYRIGHT;
			else
				iec958.status[0] &= ~IEC958_AES0_CON_NOT_COPYRIGHT;
		}
	}
	if (orig >= 0) {
		if (! (iec958.status[0] & IEC958_AES0_PROFESSIONAL)) {
			if (orig)
				iec958.status[1] |= IEC958_AES1_CON_ORIGINAL;
			else
				iec958.status[1] &= ~IEC958_AES1_CON_ORIGINAL;
		}
	}
	if (emphasis >= 0) {
		if (iec958.status[0] & IEC958_AES0_PROFESSIONAL) {
			iec958.status[0] &= ~IEC958_AES0_PRO_EMPHASIS;
			switch (emphasis) {
			case 0:
				iec958.status[0] |= IEC958_AES0_PRO_EMPHASIS_NONE;
				break;
			case 1:
				iec958.status[0] |= IEC958_AES0_PRO_EMPHASIS_5015;
				break;
			case 2:
				iec958.status[0] |= IEC958_AES0_PRO_EMPHASIS_CCITT;
				break;
			}
		} else {
			if (emphasis)
				iec958.status[0] |= IEC958_AES0_CON_EMPHASIS_5015;
			else
				iec958.status[0] &= ~IEC958_AES0_CON_EMPHASIS_5015;
		}
	}
	if (unlocked >= 0) {
		if (iec958.status[0] & IEC958_AES0_PROFESSIONAL) {
			if (unlocked)
				iec958.status[0] |= IEC958_AES0_PRO_FREQ_UNLOCKED;
			else
				iec958.status[0] &= ~IEC958_AES0_PRO_FREQ_UNLOCKED;
		}
	}
	if (sbits >= 0) {
		if (iec958.status[0] & IEC958_AES0_PROFESSIONAL) {
			iec958.status[2] &= ~IEC958_AES2_PRO_SBITS;
			iec958.status[2] |= sbits & 7;
		}
	}
	if (wlen >= 0) {
		if (iec958.status[0] & IEC958_AES0_PROFESSIONAL) {
			iec958.status[2] &= ~IEC958_AES2_PRO_WORDLEN;
			iec958.status[2] |= (wlen & 7) << 3;
		}
	}
	if (category >= 0) {
		if (! (iec958.status[0] & IEC958_AES0_PROFESSIONAL)) {
			iec958.status[1] &= ~IEC958_AES1_CON_CATEGORY;
			iec958.status[1] |= category & 0x7f;
		}
	}
		
	snd_ctl_elem_value_set_iec958(cval, &iec958);
	if (snd_ctl_elem_write(ctl, cval) < 0) {
		perror("snd_ctl_elem_write");
		return 1;
	}

	if (snd_ctl_elem_read(ctl, cval) < 0) {
		perror("snd_ctl_elem_write");
		return 1;
	}

	snd_ctl_elem_value_get_iec958(cval, &iec958);
	dump_iec958(&iec958);

	snd_ctl_close(ctl);
	return 0;
}
