/*
 * brass - Braille and speech server
 *
 * Copyright (C) 2001-2003 by Roger Butenuth, All rights reserved.
 *
 * This is free software, placed under the terms of the
 * GNU General Public License, as published by the Free Software
 * Foundation.  Please see the file COPYING for details.
 *
 * Synthesizer module for the Infovox 700 synthesizer.
 * connected to a serial port.
 *
 * $Id: infovox.c,v 1.7 2003/02/24 20:57:04 butenuth Exp $
 * ======================================================================
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <termios.h>

#include "synthesizer.h"

static int s_close(synth_t *s);
static int s_synth(synth_t *s, unsigned char *buffer);
static int s_flush(synth_t *s);
static int s_clear(synth_t *s);
static int s_index_set(struct synth_struct *s);
static int s_index_wait(struct synth_struct *s, int id, int timeout);
static int s_get_param(struct synth_struct *s, synth_par_t par, int *value);
static int s_set_param(struct synth_struct *s, synth_par_t par, int value);

typedef struct synth_state {
    int  param[S_MAX];
    int  initialized;
    int  channel;
} synth_state_t;

static synth_state_t private_state[2];

static FILE   *sy_fp;
static struct termios old_tio;
static int    sy_fd;
static int    sy_ref = 0;
static int    current_language = -1;

#define INDEX_MOD 1000
static int    index_mark = 0;	/* last index mark sent */
static int    r_index_mark = 0;	/* last index mark received */

static synth_t state[] = {
    {
        &private_state[0],
        &languages[LANG_ENGLISH],
        "Infovox/British English",
        NULL,			/* lib_handle */
        s_close,
        s_synth,
        s_flush,
        s_clear,
        s_index_set,
        s_index_wait,
        s_get_param,
        s_set_param
    }, {
        &private_state[1],
        &languages[LANG_GERMAN],
        "Infovox/German",
        NULL,			/* lib_handle */
        s_close,
        s_synth,
        s_flush,
        s_clear,
        s_index_set,
        s_index_wait,
        s_get_param,
        s_set_param
    }
};


/*
 * Infovox knows speeds from 0 to 9, this corrospondends to the followin
 * value (1000 = normal).
 */
static int speed_table[10] = {
     107,			/* 0 */
     168,			/* 1 */
     262,			/* 2 */
     410,			/* 3 */
     640,			/* 4 */
    1000,			/* 5 (default speed) */
    1563,			/* 6 */
    2441,			/* 7 */
    3815,			/* 8 */
    5960,			/* 9 */
};

/*
 * Pitch of voice. Absolute values guessed.
 * (1000 = default pitch).
 */
static int pitch_table[10] = {
     107,			/* 0 */
     168,			/* 1 */
     262,			/* 2 */
     410,			/* 3 */
     640,			/* 4 */
    1000,			/* 5 (default pitch) */
    1563,			/* 6 */
    2441,			/* 7 */
    3815,			/* 8 */
    5960,			/* 9 */
};


/*
 * Volume of voice. 9 is normal, -1 corresponds to 3 dB.
 * So this table is not very good -> improve.
 * (1000 = default volume).
 */
static int volume_table[10] = {
     107,			/* 0 */
     168,			/* 1 */
     262,			/* 2 */
     410,			/* 3 */
     640,			/* 4 */
    1000,			/* 5 */
    1563,			/* 6 */
    2441,			/* 7 */
    3815,			/* 8 */
    5960,			/* 9 */
};

/*
 * charset conversion table from iso latin-1 == iso 8859-1 to cp437 == ibmpc
 * for chars >=128. 
 */
static unsigned char latin2cp437[128] = {
    199, 252, 233, 226, 228, 224, 229, 231,
    234, 235, 232, 239, 238, 236, 196, 197,
    201, 181, 198, 244, 247, 242, 251, 249,
    223, 214, 220, 243, 183, 209, 158, 159,
    255, 173, 155, 156, 177, 157, 188, 21,
    191, 169, 166, 174, 170, 237, 189, 187,
    248, 241, 253, 179, 180, 230, 20, 250,
    184, 185, 167, 175, 172, 171, 190, 168,
    192, 193, 194, 195, 142, 143, 146, 128,
    200, 144, 202, 203, 204, 205, 206, 207,
    208, 165, 210, 211, 212, 213, 153, 215,
    216, 217, 218, 219, 154, 221, 222, 225,
    133, 160, 131, 227, 132, 134, 145, 135,
    138, 130, 136, 137, 141, 161, 140, 139,
    240, 164, 149, 162, 147, 245, 148, 246,
    176, 151, 163, 150, 129, 178, 254, 152
};


/*
 * ----------------------------------------------------------------------
 * Called before library is loaded.
 * ----------------------------------------------------------------------
 */
void _init(void)
{
}


/*
 * ----------------------------------------------------------------------
 * Called before library is unloaded.
 * ----------------------------------------------------------------------
 */
void _fini(void)
{
}


/*
 * ----------------------------------------------------------------------
 * General open function for german and english synthesizer.
 * Second open increments refcount.
 * Return 0 on succes, 1 on error.
 * Be aware: The file pointer (sy_fp) is only used for writes, 
 * reads use sy_fd directly.
 * Return 0 on succes, or an error number.
 * ----------------------------------------------------------------------
 */
synth_t *synth_open(void *context, lookup_string_t lookup)
{
    synth_t *s = NULL;
    int     r = 0;
    struct  termios new_tio;
    char    *language = (*lookup)(context, "language");
    char    *device = (*lookup)(context, "device");
    int     langi;
    
    /*
     * Configuration parameter checking.
     */
    if (language == NULL) {
        fprintf(stderr, "variable \"language\" not defined\n");
        return NULL;
    }
    if (!strcasecmp(language, "english")) {
        langi = 0;
        s = &state[langi];
        s->state->channel = 2;
        s->state->initialized = 1;
    } else if (!strcasecmp(language, "german")) {
        langi = 1;
        s = &state[langi];
        s->state->channel = 0;
        s->state->initialized = 1;
    } else {
        fprintf(stderr, "\"language\" must be english or german!\n");
        return NULL;
    }
    if (device == NULL) {
        fprintf(stderr, "variable \"device\" not defined\n");
        return NULL;
    }
    
    if (sy_ref == 0) {
	sy_fd = open(device, O_RDWR);
	if (sy_fd < 0) {
	    perror("open speech device");
	    return NULL;
	}
	tcgetattr(sy_fd, &old_tio); /* save current port settings */

        bzero(&new_tio, sizeof(new_tio));
        new_tio.c_cflag = B9600 | CRTSCTS | CS8 | CLOCAL | CREAD;
        new_tio.c_iflag = IGNPAR;
        new_tio.c_oflag = 0;
        /*
	 * set input mode (non-canonical, no echo,...)
	 */
        new_tio.c_lflag = 0;
	/*
	 * Read will be satisfied if a single character is read, or TIME
	 * is exceeded.
	 */
        new_tio.c_cc[VTIME] = 1;
        new_tio.c_cc[VMIN]  = 0;

        tcflush(sy_fd, TCIFLUSH);
        tcflush(sy_fd, TCOFLUSH);
        tcsetattr(sy_fd,TCSANOW, &new_tio);

	sy_fp = fdopen(sy_fd, "w");
	if (sy_fp == NULL)
	    return NULL;
	current_language = -1;
	fprintf(sy_fp, "\033Z\n"); /* reset */
	fprintf(sy_fp, "\033N\n"); /* line mode */
	fprintf(sy_fp, "\033%%\001\n"); /* index character is control-a */
    }
    if (r == 0)
	sy_ref++;

    return s;
}


/*
 * ----------------------------------------------------------------------
 * Decrement refcount, do real close when count reaches zero.
 * ----------------------------------------------------------------------
 */
static int s_close(synth_t *s)
{
    assert(s->state->initialized);
    s->state->initialized = 0;

    assert(sy_ref > 0);
    sy_ref--;
    if (sy_ref == 0) {
	tcsetattr(sy_fd, TCSANOW, &old_tio);
	fclose(sy_fp);
	sy_fp = NULL;
    }

    return 0;
}


/*
 * ----------------------------------------------------------------------
 * Verify that the synthesizer is set to the correct language.
 * Switch if necessary and reset parameters.
 * A language switch without complete reset seems not to work.
 * ----------------------------------------------------------------------
 */
static void verify_language(struct synth_struct *s)
{
    int p;

    if (s->lang->lang == LANG_GERMAN &&	current_language != LANG_GERMAN) {

	fprintf(sy_fp, "\033Z\n"); /* reset */
	fprintf(sy_fp, "\033N\n"); /* line mode */
	fprintf(sy_fp, "\033%%\001\n"); /* index character is control-a */

	fprintf(sy_fp, "\033x%d", s->state->channel);
	fflush(sy_fp);
	usleep(100000);	/* wait 0.1s */
	current_language = LANG_GERMAN;
	for (p = 0; p < S_MAX; p++) {
	    s_set_param(s, p, s->state->param[p]);
	}
    }
    if (s->lang->lang == LANG_ENGLISH &&
	current_language != LANG_ENGLISH) {

	fprintf(sy_fp, "\033Z\n"); /* reset */
	fprintf(sy_fp, "\033N\n"); /* line mode */
	fprintf(sy_fp, "\033%%\001\n"); /* index character is control-a */

	fprintf(sy_fp, "\033x%d", s->state->channel);
	fflush(sy_fp);
	usleep(100000);	/* wait 0.1s */
	current_language = LANG_ENGLISH;
	for (p = 0; p < S_MAX; p++) {
	    s_set_param(s, p, s->state->param[p]);
	}
    }
}


/*
 * ----------------------------------------------------------------------
 * Milliseconds from t1 (start) to t2 (end).
 * ----------------------------------------------------------------------
 */
static int elapsed_msec(struct timeval *t1, struct timeval *t2)
{
    unsigned diff;

    diff = ((t2->tv_sec - t1->tv_sec) * 1000L +
            (t2->tv_usec - t1->tv_usec) / 1000L);

    return diff;
}


/*
 * ----------------------------------------------------------------------
 * 
 * ----------------------------------------------------------------------
 */
static int s_synth(struct synth_struct *s, unsigned char *buffer)
{
    int i, len;
    int c;

    assert(s->state->initialized);

    verify_language(s);
    /*
     * Write string to synthesizer device, convert character when necessary
     * and filter control characters.
     */
    len = strlen(buffer);
    for (i = 0; i < len; i++) {
	c = buffer[i];
	if (c < ' ' || c == '@')
	    ;			/* ignore */
	else if (c < 0x80)
	    fputc(c, sy_fp);	/* normal character, simply write */
	else /* 0x80 <= c <= 0xff */
	    fputc(latin2cp437[c - 0x80], sy_fp);
    }
    fflush(sy_fp);
    return 0;
}


/*
 * ----------------------------------------------------------------------
 * 
 * ----------------------------------------------------------------------
 */
static int s_flush(synth_t *s)
{
    assert(s->state->initialized);

    fprintf(sy_fp, "\r\n");
    fflush(sy_fp);

    return 0;
}


/*
 * ----------------------------------------------------------------------
 * 
 * ----------------------------------------------------------------------
 */
static int s_clear(synth_t *s)
{
    assert(s->state->initialized);

    fprintf(sy_fp, "\033z");
    fflush(sy_fp);
    index_mark = 0;
    r_index_mark = 0;

    return 0;
}

/*
 * ----------------------------------------------------------------------
 * Increment the index mark number and set a mark.
 * ----------------------------------------------------------------------
 */
static int s_index_set(synth_t *s)
{
    index_mark = (index_mark + 1) % INDEX_MOD;
    fprintf(sy_fp, "\001%03d", index_mark); fflush(sy_fp);
    
    return index_mark;
}


/*
 * ----------------------------------------------------------------------
 * Indexing, return value is:
 *  0 sync point not reached (timeout)
 *  1 sync point reached
 * (See chapter 5.x)
 * ----------------------------------------------------------------------
 */
static int s_index_wait(synth_t *s, int id, int timeout)
{
    int            res = -1;
    struct timeval start;
    struct timeval cur;
    int            gone;
    int            value, diff;
    int            state;
    int            c;
    char           ch;

    assert(s->state->initialized);
    verify_language(s);
    gettimeofday(&start, NULL);
    /*
     * Did we already receive the mark we wait for?
     */
    diff = (INDEX_MOD + r_index_mark - id) % INDEX_MOD;
    if (diff < INDEX_MOD / 2)
	return 1;
	
    state = 0;
    value = 0;
    while (res == -1) {
	if (state == 0) {	/* 0: wait for control-a */
	    gettimeofday(&cur, NULL);
	    gone = elapsed_msec(&start, &cur);
	    if (gone >= timeout) {
		res = 0;
	    } else {
		c = read(sy_fd, &ch, 1);
		if (c == 1 && ch == '\001')
		    state = 1;
		value = 0;
	    }
	} else if (state < 4) { /* 1, 2, 3: collect digits */
	    c = read(sy_fd, &ch, 1);
	    if (c == 1) {
		value = 10 * value + ch - '0';
		state++;
	    }
	} else {		/* 4: evaluate result */
	    diff = (INDEX_MOD + value - id) % INDEX_MOD;
	    if (diff < INDEX_MOD / 2) {
		res = 1;
	    } else {
		state = 0;	/* new game, new luck... */
		value = 0;
	    }
	}
    }

    return res;
}

/*
 * ----------------------------------------------------------------------
 *
 * ----------------------------------------------------------------------
 */
static int s_get_param(struct synth_struct *s, synth_par_t par, int *value)
{
    if (par >= 0 && par < S_MAX) {
	*value = s->state->param[par];
	return 0;
    } else
	return 1;
}


/*
 * ----------------------------------------------------------------------
 * Change a parameter. Only when the current synthesizer is the same
 * as the selected, the change is done imediately, otherwise only
 * the request is stored and executed later.
 * ----------------------------------------------------------------------
 */
static int s_set_param(struct synth_struct *s, synth_par_t par, int value)
{
    int r = 0;
    int i;
    switch (par) {
    case S_SPEED:		/* 4.2.12 */
	if (value < 0) value = 0;
	if (s->lang->lang == current_language) {
	    for (i = 0; i < 9; i++)
		if (value <= speed_table[i])
		    break;
	    fprintf(sy_fp, "\033T%d", i);
	    fflush(sy_fp);
	    usleep(100000);	/* wait 0.1s */
	}
	s->state->param[par] = value;
	break;
    case S_PITCH:		/* 4.3.2 */
	/* smaller adjustments possible with <esc>P+ or <esc>P- */
	if (value < 0) value = 0;
	if (s->lang->lang == current_language) {
	    for (i = 0; i < 9; i++)
		if (value <= pitch_table[i])
		    break;
	    fprintf(sy_fp, "\033P%d", i);
	    fflush(sy_fp);
	    usleep(100000);	/* wait 0.1s */
	}
	s->state->param[par] = value;
	break;
    case S_VOLUME:		/* 4.2.11 */
	if (value < 0) value = 0;
	if (s->lang->lang == current_language) {
	    for (i = 0; i < 9; i++)
		if (value <= volume_table[i])
		    break;
	    fprintf(sy_fp, "\033G%d", i);
	    fflush(sy_fp);
	    usleep(100000);	/* wait 0.1s */
	}
	s->state->param[par] = value;
	break;
    default:
	r = 1;
    }
    return r;
}
