/*
 * Copyright (c) 2004 Michael Schroeder (mls@suse.de)
 *
 * This program is licensed under the BSD license, read LICENSE.BSD
 * for further information
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/wait.h>

#include <bzlib.h>
#include <zlib.h>

#include "util.h"
#include "md5.h"
#include "rpmhead.h"
#include "cpio.h"
#include "cfile.h"

#define BLKSHIFT 13
#define BLKSIZE  (1 << BLKSHIFT)
#define BLKMASK  ((1 << BLKSHIFT) - 1)

#define SEQCHECK_MD5   (1<<0)
#define SEQCHECK_SIZE  (1<<1)

#ifndef RPMDUMPHEADER
# define RPMDUMPHEADER "rpmdumpheader"
#endif

/*****************************************************************
 * utility functions
 */

unsigned int bzread4(struct cfile *bfp)
{
  unsigned char d[4];
  if (bfp->read(bfp, d, 4) != 4)
    {
      perror("BZ2_bzread");
      exit(1);
    }
  return d[0] << 24 | d[1] << 16 | d[2] << 8 | d[3];
}

/*****************************************************************
 * fileblock handling, maintain everything we want to know about the
 * filelist
 * 
 */

struct fileblock
{
  struct rpmhead *h;
  int cnt;
  char **filenames;
  unsigned int *filemodes;
  unsigned int *filesizes;
  unsigned int *filerdevs;
  char **filelinktos;
  char **filemd5s;
};


int
headtofb(struct rpmhead *h, struct fileblock *fb)
{
  fb->h = h;
  fb->filelinktos = fb->filemd5s = 0;
  fb->filemodes = fb->filesizes = 0;
  fb->filenames = headexpandfilelist(h, &fb->cnt);
  if (!fb->filenames)
    {
      fb->cnt = 0;
      return 0;
    }
  fb->filemodes = headint16(h, TAG_FILEMODES, (int *)0);
  fb->filesizes = headint32(h, TAG_FILESIZES, (int *)0);
  fb->filerdevs = headint32(h, TAG_FILERDEVS, (int *)0);
  fb->filelinktos = headstringarray(h, TAG_FILELINKTOS, (int *)0);
  fb->filemd5s = headstringarray(h, TAG_FILEMD5S, (int *)0);
  return 0;
}

/*****************************************************************
 * sequence handling, uncompress the sequence string, check if
 * it matches the installed rpm header, check files if requested.
 */

struct openfile;

struct seqdescr {
  int i;
  int cpiolen;
  int datalen;
  int off;
  struct openfile *f;
};

int
checkfilemd5(char *name, unsigned char *hmd5, int size)
{
  int fd, l;
  char buf[4096];
  MD5_CTX ctx;
  unsigned char md5[16];

  if ((fd = open(name, O_RDONLY)) < 0)
    {
      perror(name);
      return -1;
    }
  rpmMD5Init(&ctx);
  while (size && (l = read(fd, buf, sizeof(buf))) > 0)
    {
      if (l > size)
	l = size;
      rpmMD5Update(&ctx, buf, l);
      size -= l;
    }
  close(fd);
  rpmMD5Final(md5, &ctx);
  if (memcmp(md5, hmd5, 16))
    {
      fprintf(stderr, "%s: contents have been changed\n", name);
      return -1;
    }
  return 0;
}

int
checkfilesize(char *name, int size)
{
  struct stat stb;
  if (stat(name, &stb) == -1)
    {
      perror(name);
      return -1;
    }
  if (stb.st_size != size)
    {
      fprintf(stderr, "%s: contents have been changed\n", name);
      return -1;
    }
  return 0;
}

struct seqdescr *
expandseq(unsigned char *seq, int seql, int *nump, struct fileblock *fb, int flags)
{
  unsigned char *s, *fn;
  int *res;
  int i, n, n2, num, nib, shi, tog, jump, pos;
  unsigned int rdev, lsize;
  MD5_CTX seqmd5;
  unsigned char seqmd5res[16];
  struct seqdescr *sd;
  unsigned int off;
  unsigned char fmd5[16];
  int error = 0;

  n = num = nib = shi = jump = pos = 0;
  tog = 1;

  res = xmalloc(fb->cnt * sizeof(unsigned int));
  seql -= 16;
  for (i = 0, s = seq + 16; i < seql; )
    {
      if (!nib)
        n2 = (*s >> 4);
      else
	{
          n2 = (*s & 0x0f);
	  s++;
	  i++;
	}
      nib ^= 1;
      if (n2 & 8)
	{
	  n2 ^= 8;
	  if (shi)
	    n2 <<= shi;
	  n |= n2;
	  shi += 3;
	  continue;
	}
      if (shi)
	n2 <<= shi;
      shi = 0;
      n2 |= n;
      n = 0;
      if (jump)
	{
	  jump = 0;
	  pos = n2;
	  tog = 1;
          continue;
	}
      if (n2 == 0)
	{
	  jump = 1;
	  continue;
	}
      if (!tog)
	{
	  pos += n2;
	  tog = 1;
	  continue;
	}
      for (; n2 > 0; n2--)
	{
	  if (num >= fb->cnt || pos >= fb->cnt)
	    {
	      fprintf(stderr, "corrupt delta: bad sequence\n");
	      exit(1);
	    }
	  res[num++] = pos++;
	}
      tog = 0;
    }
  if (shi)
    {
      fprintf(stderr, "corrupt delta: bad sequence\n");
      exit(1);
    }
  res = xrealloc(res, num * sizeof(unsigned int));
  sd = xmalloc((num + 1) * sizeof(*sd));
  *nump = num + 1;
  rpmMD5Init(&seqmd5);
  off = 0;
  for (n = 0; n < num; n++)
    {
      i = sd[n].i = res[n];
      lsize = rdev = 0;
      if (S_ISREG(fb->filemodes[i]))
	lsize = fb->filesizes[i];
      else if (S_ISLNK(fb->filemodes[i]))
	lsize = strlen(fb->filelinktos[i]);
      if (S_ISBLK(fb->filemodes[i]) || S_ISCHR(fb->filemodes[i]))
	rdev = fb->filerdevs[i];
      fn = fb->filenames[i];
      if (*fn == '/')
	fn++;
      rpmMD5Update(&seqmd5, fn, strlen(fn) + 1);
      rpmMD5Update32(&seqmd5, fb->filemodes[i]);
      rpmMD5Update32(&seqmd5, lsize);
      rpmMD5Update32(&seqmd5, rdev);
      sd[n].cpiolen = 110 + 2 + strlen(fn) + 1;
      if (sd[n].cpiolen & 3)
	sd[n].cpiolen += 4 - (sd[n].cpiolen & 3);
      sd[n].datalen = lsize;
      if (sd[n].datalen & 3)
	sd[n].datalen += 4 - (sd[n].datalen & 3);
      if (S_ISLNK(fb->filemodes[i]))
	rpmMD5Update(&seqmd5, fb->filelinktos[i], strlen(fb->filelinktos[i]) + 1);
      else if (S_ISREG(fb->filemodes[i]) && lsize)
	{
	  parsemd5(fb->filemd5s[i], fmd5);
	  if ((flags & SEQCHECK_MD5) != 0)
	    if (checkfilemd5(fb->filenames[i], fmd5, lsize))
	      error = 1;
	  if ((flags & SEQCHECK_SIZE) != 0)
	    if (checkfilesize(fb->filenames[i], lsize))
	      error = 1;
	  rpmMD5Update(&seqmd5, fmd5, 16);
	}
      sd[n].off = off;
      off += sd[n].cpiolen + sd[n].datalen;
      sd[n].f = 0;
    }
  sd[n].i = -1;
  sd[n].cpiolen = 124;
  sd[n].datalen = 0;
  sd[n].off = off;
  sd[n].f = 0;
  rpmMD5Final(seqmd5res, &seqmd5);
  free(res);
  if (memcmp(seqmd5res, seq, 16) || error)
    {
      fprintf(stderr, "delta does not match installed data\n");
      exit(1);
    }
  return sd;
}

/*****************************************************************
 * openfile, maintain a set of opened files, close descriptors if
 * limit is reached.
 */

struct openfile {
  struct openfile *prev;
  struct openfile *next;
  char *name;
  int fd;
  unsigned int off;
  struct seqdescr *sd;
};

struct openfile *openfiles;
struct openfile *openfilestail;
int nopenfile;
int maxopenfile = 50;



struct openfile *
newopen(struct seqdescr *sd, char *name)
{
  struct openfile *of;
  if (nopenfile < maxopenfile)
    {
      of = xmalloc(sizeof(*of));
      nopenfile++;
    }
  else
    {
      of = openfiles;
      openfiles = of->next;
      if (openfiles)
	openfiles->prev = 0;
      else
	openfilestail = 0;
      of->sd->f = 0;
      // printf("closing %s\n", of->name);
      close(of->fd);
    }
  // printf("opening %s\n", name);
  if ((of->fd = open(name, O_RDONLY)) == -1)
    {
      perror(name);
      fprintf(stderr, "cannot reconstruct rpm from disk files\n");
      exit(1);
    }
  of->name = name;
  of->off = 0;
  of->sd = sd;
  of->prev = of->next = 0;
  if (openfilestail)
    {
      openfilestail->next = of;
      of->prev = openfilestail;
      openfilestail = of;
    }
  else
    openfiles = openfilestail = of;
  sd->f = of;
  return of;
}

/*****************************************************************
 * blk stuff, block contents creation and paging
 */

#define BLK_FREE     0
#define BLK_CORE_REC 1
#define BLK_CORE_ONE 2
#define BLK_PAGE     3

struct blk {
  struct blk *next;
  int type;
  int id;
  union {
    unsigned int off;
    unsigned char *buf;
  } e;
};

struct blk *coreblks;
struct blk *freecoreblks;
struct blk *pageblks;
int ncoreblk = 0;
int npageblk = 0;
int ndropblk = 0;

int maxcoreblk = 5000;

unsigned int *maxblockuse;	/* last time the block will be used */
struct blk **vmem;

unsigned char *cpiodata;
int csdesc = -1;
char *symdata;

char *fromrpm;
struct cfile *outfp;
unsigned int outfpleft;
int outfpid;


int pagefd = -1;

void
pageoutblock(struct blk *cb, int idx)
{
  struct blk *b;

  // printf("pageoutblock %d\n", cb->id);
  for (b = pageblks; b; b = b->next)
    if (b->id == cb->id)
      {
        vmem[b->id] = b;
        return;
      }
  for (b = pageblks; b; b = b->next)
    if (maxblockuse[b->id] < idx)
      break;
  if (!b)
    {
      b = xmalloc(sizeof(*b));
      b->next = pageblks;
      b->type = BLK_PAGE;
      b->e.off = npageblk * BLKSIZE;
      pageblks = b;
      npageblk++;
      if (pagefd < 0)
	{
	  char tmpname[80];
	  sprintf(tmpname, "/tmp/deltarpmpageXXXXXX");
	  pagefd = mkstemp(tmpname);
	  if (pagefd < 0)
	    {
	      fprintf(stderr, "could not create page area\n");
	      exit(1);
	    }
	  unlink(tmpname);
	}
    }
  b->id = cb->id;
  if (lseek(pagefd, (off_t)b->e.off, SEEK_SET) == (off_t)-1)
    {
      perror("page area seek");
      exit(1);
    }
  if (write(pagefd, cb->e.buf, BLKSIZE) != BLKSIZE)
    {
      perror("page area write");
      exit(1);
    }
  vmem[b->id] = b;
}

void
pageinblock(struct blk *cb, struct blk *b)
{
  if (b->type != BLK_PAGE)
    abort();
  // printf("pageinblock %d\n", b->id);
  if (lseek(pagefd, (off_t)b->e.off, SEEK_SET) == (off_t)-1)
    {
      perror("page area seek");
      exit(1);
    }
  if (read(pagefd, cb->e.buf, BLKSIZE) != BLKSIZE)
    {
      perror("page area read");
      exit(1);
    }
  cb->id = b->id;
  cb->type = BLK_CORE_ONE;
  vmem[cb->id] = cb;
}

struct blk *
newcoreblk(void)
{
  struct blk *b;
  b = xmalloc(sizeof(*b) + BLKSIZE);
  b->next = coreblks;
  b->type = BLK_FREE;
  b->e.buf = (unsigned char *)(b + 1);
  coreblks = b;
  ncoreblk++;
  // printf("created new coreblk, have now %d\n", ncoreblk);
  return b;
}

void
pushblock(struct blk *nb, int idx)
{
  struct blk *b;

  b = freecoreblks;
  if (b)
    {
      freecoreblks = b->next;
      b->next = coreblks;
      coreblks = b;
    }
  if (!b && ncoreblk < maxcoreblk)
    b = newcoreblk();
  if (!b)
    {
      /* could not find in-core place */
      if (nb->type == BLK_CORE_ONE)
        pageoutblock(nb, idx);
      else
	vmem[nb->id] = 0;
      return;
    }
  b->type = nb->type;
  b->id = nb->id;
  memcpy(b->e.buf, nb->e.buf, BLKSIZE);
  vmem[b->id] = b;
}

void
createcpiohead(struct seqdescr *sd, struct fileblock *fb)
{
  int i = sd->i;
  unsigned int lsize, rdev;
  char *np;

  if (i == -1)
    {
      sprintf(cpiodata, "%s%c%c%c%c", "07070100000000000000000000000000000000000000010000000000000000000000000000000000000000000000000000000b00000000TRAILER!!!", 0, 0, 0, 0);
      return;
    }
  lsize = rdev = 0;
  np = fb->filenames[i];
  if (*np == '/')
    np++;
  if (S_ISREG(fb->filemodes[i]))
    lsize = fb->filesizes[i];
  else if (S_ISLNK(fb->filemodes[i]))
    {
      symdata = fb->filelinktos[i];
      lsize = strlen(fb->filelinktos[i]);
    }
  if (S_ISBLK(fb->filemodes[i]) || S_ISCHR(fb->filemodes[i]))
    rdev = fb->filerdevs[i];
  sprintf(cpiodata, "07070100000000%08x00000000000000000000000100000000%08x0000000000000000%08x%08x%08x00000000./%s%c%c%c%c", fb->filemodes[i], lsize, devmajor(rdev), devminor(rdev), (int)strlen(np) + 3, np, 0, 0, 0, 0);
}

void
fillblock_disk(struct blk *b, int id, struct seqdescr *sdesc, int nsdesc, struct fileblock *fb, int idx)
{
  unsigned int off, u;
  struct seqdescr *sd;
  int l, l2, i;
  unsigned char *bp;

  l = BLKSIZE;
  bp = b->e.buf;
  off = id << BLKSHIFT;
  i = csdesc >= 0 ? csdesc : 0;
  sd = sdesc + i;
  for (sd = sdesc + i; i > 0 && sd->off > off; i--, sd--)
    ;
  for (; i < nsdesc; i++, sd++)
    if (sd->off <= off && sd->off + sd->cpiolen + sd->datalen > off)
      break;
  if (i == nsdesc)
    {
      fprintf(stderr, "fillblock_disk: block %d out of range\n", id);
      exit(1);
    }
  if (i != csdesc)
    {
      csdesc = i;
      createcpiohead(sd, fb);
    }
  i = sd->i;
  while (l > 0)
    {
      if (off < sd->off + sd->cpiolen)
	{
	  u = off - sd->off;
	  l2 = sd->cpiolen - u;
	  if (l2 > l)
	    l2 = l;
	  memcpy(bp, cpiodata + u, l2);
	  bp += l2;
	  off += l2;
	  l -= l2;
	  continue;
	}
      if (i == -1)
	{
	  memset(bp, 0, l);
	  l = 0;
	  break;
	}
      if (off < sd->off + sd->cpiolen + sd->datalen)
	{
	  u = off - (sd->off + sd->cpiolen);
	  if (S_ISLNK(fb->filemodes[i]))
	    {
	      l2 = sd->datalen - u;
	      if (l2 > l)
		l2 = l;
	      if (u > strlen(symdata))
		memset(bp, 0, l2);
	      else
		strncpy(bp, symdata + u, l2);
	    }
	  else if (u < fb->filesizes[i])
	    {
	      struct openfile *of;
	      l2 = fb->filesizes[i] - u;
	      if (l2 > l)
		l2 = l;
	      if (!(of = sd->f))
		of = newopen(sd, fb->filenames[i]);
	      if (of->next)
		{
		  of->next->prev = of->prev;
		  if (of->prev)
		    of->prev->next = of->next;
		  else
		    openfiles = of->next;
		  of->next = 0;
		  of->prev = openfilestail;
		  openfilestail->next = of;
		  openfilestail = of;
		}
	      if (of->off != u)
		{
		  if (lseek(of->fd, (off_t)u, SEEK_SET) == (off_t)-1)
		    {
		      fprintf(stderr, "%s: seek error\n", of->name);
		      fprintf(stderr, "cannot reconstruct rpm from disk files\n");
		      exit(1);
		    }
		}
	      if (read(of->fd, bp, l2) != l2)
		{
		  fprintf(stderr, "%s: read error\n", of->name);
		  fprintf(stderr, "(tried to read %d bytes from offset %d)\n", l2, u);
		  fprintf(stderr, "cannot reconstruct rpm from disk files\n");
		  exit(1);
		}
	      of->off = u + l2;
	    }
	  else
	    {
	      l2 = sd->datalen - u;
	      if (l2 > l)
		l2 = l;
	      memset(bp, 0, l2);
	    }
	  bp += l2;
	  off += l2;
	  l -= l2;
	  continue;
        }
      csdesc++;
      sd++;
      createcpiohead(sd, fb);
      i = sd->i;
    }
  b->id = id;
  b->type = BLK_CORE_REC;
}

void
fillblock_rpm(struct blk *b, int id, struct seqdescr *sdesc, int nsdesc, struct fileblock *fb, int idx)
{
  unsigned int size, nsize;
  unsigned char *bp, *np;
  int i, l, l2, u;
  struct seqdescr *sd;
  struct cpiophys cph;
  static char *namebuf;
  static int namebufl;
  char skipbuf[4096];

  l = BLKSIZE;
  bp = b->e.buf;
  for (;;)
    {
      if (outfpleft)
	{
	  sd = sdesc + csdesc;
	  if (outfpleft > sd->datalen)
	    {
	      u = sd->cpiolen + sd->datalen - outfpleft;
	      l2 = sd->cpiolen - u;
	      if (l2 > l)
		l2 = l;
	      memcpy(bp, cpiodata + u, l2);
	      bp += l2;
	      outfpleft -= l2;
	      l -= l2;
	    }
	  if (l && outfpleft)
	    {
	      l2 = outfpleft;
	      if (l2 > l)
		l2 = l;
	      if (S_ISLNK(fb->filemodes[sd->i]))
		{
		  strncpy(bp, symdata, l2);
		  if (strlen(symdata) < l2)
		    symdata += strlen(symdata);
		  else
		    symdata += l2;
		}
	      else
		{
		  if (outfp->read(outfp, bp, l2) != l2)
		    {
		      fprintf(stderr, "read error");
		      exit(1);
		    }
		}
	      bp += l2;
	      outfpleft -= l2;
	      l -= l2;
	    }
	}
      if (l && csdesc >= 0 && sdesc[csdesc].i == -1)
	{
	  memset(bp, 0, l); /* blocks are empty after trailer */
	  l = 0;
	}
      if (l == 0)
	{
	  b->type = BLK_CORE_ONE;
	  b->id = outfpid++;
	  if (b->id == id)
	     return;
	  if (b->id > id)
	    {
	      fprintf(stderr, "internal error, cannot rewind blocks (%d %d)\n", b->id, id);
	      exit(1);
	    }
	  if (maxblockuse[b->id] >= idx)
	    pushblock(b, idx);
	  l = BLKSIZE;
	  bp = b->e.buf;
	  continue;
	}
      csdesc++;
      sd = sdesc + csdesc;
      i = sd->i;
      if (i == -1)
	{
	  createcpiohead(sd, fb);
	  outfpleft = sd->cpiolen + sd->datalen;
	  continue;
	}
      for (;;)
	{
	  if (outfp->read(outfp, &cph, sizeof(cph)) != sizeof(cph))
	    {
	      fprintf(stderr, "read error");
	      exit(1);
	    }
	  if (memcmp(cph.magic, "070701", 6))
	    {
	      fprintf(stderr, "read error: bad cpio archive\n");
	      exit(1);
	    }
	  size = cpion(cph.filesize);
	  nsize = cpion(cph.namesize);
	  nsize += (4 - ((nsize + 2) & 3)) & 3;
	  if (nsize > namebufl)
	    {
	      namebuf = xrealloc(namebuf, nsize);
	      namebufl = nsize;
	    }
	  if (outfp->read(outfp, namebuf, nsize) != nsize)
	    {
	      fprintf(stderr, "read failed (name)\n");
	      exit(1);
	    }
	  namebuf[nsize - 1] = 0;
	  if (!strcmp(namebuf, "TRAILER!!!"))
	    {
	      fprintf(stderr, "cpio end reached, bad rpm\n");
	      exit(1);
	    }
	  np = namebuf;
	  if (*np == '.' && np[1] == '/')
	    np += 2;
	  if (!strcmp(fb->filenames[i][0] == '/' ? fb->filenames[i] + 1 : fb->filenames[i], np))
	    break;
	  if (size & 3)
	    size += 4 - (size & 3);
	  while (size > 0)
	    {
	      l2 = size > sizeof(skipbuf) ? sizeof(skipbuf) : size;
	      if (outfp->read(outfp, skipbuf, l2) != l2)
		{
		  fprintf(stderr, "read failed (name)\n");
		  exit(1);
		}
	      size -= l2;
	    }
	}
      createcpiohead(sd, fb);
      if (size & 3)
	size += 4 - (size & 3);
      if (!S_ISREG(fb->filemodes[i]))
	{
	  while (size > 0)
	    {
	      l2 = size > sizeof(skipbuf) ? sizeof(skipbuf) : size;
	      if (outfp->read(outfp, skipbuf, l2) != l2)
		{
		  fprintf(stderr, "read failed (data skip)\n");
		  exit(1);
		}
	      size -= l2;
	    }
	}
      else if (size != sd->datalen)
	{
	  fprintf(stderr, "cpio data size mismatch, bad rpm\n");
	  exit(1);
	}
      outfpleft = sd->cpiolen + sd->datalen;
    }
}

struct blk *
getblock(int id, struct seqdescr *sdesc, int nsdesc, struct fileblock *fb, int idx)
{
  struct blk *b, **bb;
  struct blk *pb;
  static int cleanup_cnt;

  b = vmem[id];
  if (b && (b->type == BLK_CORE_REC || b->type == BLK_CORE_ONE))
    return b;

  b = freecoreblks;
  if (b)
    {
      freecoreblks = b->next;
      b->next = coreblks;
      coreblks = b;
    }

  if (!b && ncoreblk < maxcoreblk && (++cleanup_cnt & 7) != 0)
    b = newcoreblk();

  if (!b)
    {
      for (bb = &coreblks; (b = *bb) != 0; bb = &b->next)
	{
	  if (maxblockuse[b->id] < idx)
	    {
	      *bb = b->next;
	      vmem[b->id] = 0;
	      b->type = BLK_FREE;
	      b->next = freecoreblks;
	      freecoreblks = b;
	    }
	  else
	    bb = &b->next;
	}
      b = freecoreblks;
      if (b)
	{
	  freecoreblks = b->next;
	  b->next = coreblks;
	  coreblks = b;
	}
    }

  if (!b && ncoreblk < maxcoreblk)
    b = newcoreblk();
  if (!b)
    {
      /* use first created block */
      for (bb = &coreblks; (b = *bb); bb = &b->next)
	if (b->next == 0)
	  break;
      *bb = 0;
      b->next = coreblks;
      coreblks = b;
      if (b->type == BLK_CORE_ONE)
	pageoutblock(b, idx);
      else
	{
	  vmem[b->id] = 0;
	  ndropblk++;
	}
      b->type = BLK_FREE;
    }

  /* got destination block, now fill it with data */
  pb = vmem[id];
  if (pb && pb->type == BLK_PAGE)
    {
      pageinblock(b, pb);
      return b;
    }
  if (fromrpm)
    fillblock_rpm(b, id, sdesc, nsdesc, fb, idx);
  else
    fillblock_disk(b, id, sdesc, nsdesc, fb, idx);
  vmem[id] = b;
  return b;
}

char *comp2str(int comp)
{
  if (comp == CFILE_COMP_BZ)
    return "bzip2";
  if (comp == CFILE_COMP_GZ)
    return "gzip";
  if (comp == CFILE_COMP_UN)
    return "uncompressed";
  return "???";
}

/*****************************************************************
 * main program
 */

int
main(int argc, char **argv)
{
  int c, i;
  char *deltarpm;
  struct rpmhead *h, *dh;
  int dfd, fd;
  struct cfile *bfp = 0;
  struct cfile *obfp;
  char *nevr, *fnevr;
  unsigned int nevrl;
  unsigned char rpmlead[96];
  unsigned int seql;
  unsigned char *seq;
  unsigned char targetmd5[16];
  unsigned int targetsize = 0;
  unsigned char *lead;
  unsigned int leadl;
  unsigned int inn;
  unsigned int *in;
  unsigned int outn;
  unsigned int *out;
  unsigned int inlen = 0;
  unsigned int outlen = 0;
  unsigned int paylen, paywritten;
  struct fileblock fb;
  struct seqdescr *sdesc;
  int nsdesc;
  struct blk *lastblk = 0;
  int idx;
  int on;
  unsigned int off, len, ioff, l;
  int bs, be;
  unsigned char buf[4096];
  MD5_CTX wrmd5;
  unsigned char wrmd5res[16];
  FILE *ofp;
  int numblks;
  int percent = 0;
  int curpercent;
  int lastpercent = -1;
  int verbose = 0;
  int seqcheck = 0;
  int check = 0;
  int checkflags = 0;
  int info = 0;
  unsigned int payformatoff;
  unsigned char *addbz2 = 0;
  unsigned int addbz2len = 0;
  bz_stream addbz2strm;
  unsigned char *addbz2blk = 0;
  unsigned char *b;
  int targetcomp = CFILE_COMP_XX;
  unsigned int targetcompparalen = 0;
  unsigned char *targetcomppara = 0;
  int version = 0;

  while ((c = getopt(argc, argv, "cCisvpr:")) != -1)
    {
      switch(c)
	{
	case 'v':
          verbose++;
	  break;
	case 'p':
          percent++;
	  break;
	case 'r':
	  fromrpm = optarg;
	  break;
	case 's':
	  check = 1;
	  seqcheck = 1;
	  break;
	case 'i':
	  info = 1;
	  break;
	case 'c':
	  checkflags = SEQCHECK_MD5;
	  check = 1;
	  break;
	case 'C':
	  checkflags = SEQCHECK_SIZE;
	  check = 1;
	  break;
	default:
	  fprintf(stderr, "usage: applydeltarpm [-r <rpm>] deltarpm rpm\n");
          exit(1);
	}
    }

  if (optind + (check || info ? 1 : 2) != argc)
    {
      fprintf(stderr, "usage: applydeltarpm [-r <rpm>] deltarpm rpm\n");
      exit(1);
    }
  if (checkflags && fromrpm)
    {
      fprintf(stderr, "on-disk checking does not work with the -r option.\n");
      exit(1);
    }

  deltarpm = argv[optind];

  if (seqcheck)
    {
      char *hex;
      if (info)
	{
	  fprintf(stderr, "need real delta-rpm for info\n");
	  exit(1);
	}
      seql = strlen(deltarpm);
      if (seql < 34 || (hex = strrchr(deltarpm, '-')) == 0)
	{
	  fprintf(stderr, "%s: bad sequence\n", deltarpm);
	  exit(1);
	}
      nevr = deltarpm;
      nevrl = hex - deltarpm;
      seql -= nevrl + 1;
      *hex++ = 0;
      seql = (seql + 1) / 2;
      seq = xmalloc(seql);
      if (parsehex(hex, seq, seql) != seql)
	{
	  fprintf(stderr, "bad sequence\n");
	  exit(1);
	}
      dh = 0;
      bfp = 0;
      lead = 0;
      leadl = paylen = 0;
      inn = outn = 0;
      in = out = 0;
    }
  else
    {
      if ((dfd = open(deltarpm, O_RDONLY)) < 0)
	{
	  perror(deltarpm);
	  exit(1);
	}
      if (read(dfd, rpmlead, 96) != 96 || rpmlead[0] != 0xed || rpmlead[1] != 0xab || rpmlead[2] != 0xee || rpmlead[3] != 0xdb)
	{
	  fprintf(stderr, "%s: not a delta rpm\n", deltarpm);
	  exit(1);
	}
      if (rpmlead[4] != 0x03 || rpmlead[0x4e] != 0 || rpmlead[0x4f] != 5)
	{
	  fprintf(stderr, "%s: not a v3 rpm or not new header styles\n", deltarpm);
	  exit(1);
	}
      dh = readhead(dfd, 1);
      if (!dh)
	{
	  fprintf(stderr, "%s: could not read signature header\n", deltarpm);
	  exit(1);
	}
      free(dh);
      dh = readhead(dfd, 0);
      if (!dh)
	{
	  fprintf(stderr, "%s: could not read header\n", deltarpm);
	  exit(1);
	}
      if ((bfp = cfile_open(CFILE_OPEN_RD, dfd, 0, CFILE_COMP_XX, CFILE_LEN_UNLIMITED, 0, 0)) == 0)
	{
	  fprintf(stderr, "%s: payload open failed\n", deltarpm);
	  exit(1);
	}
      version = bzread4(bfp);
      if (version != 0x444c5431 && version != 0x444c5432)
	{
	  fprintf(stderr, "%s: not a delta rpm\n", deltarpm);
	  exit(1);
	}
      nevrl = bzread4(bfp);
      nevr = xmalloc(nevrl + 1);
      nevr[nevrl] = 0;
      if (bfp->read(bfp, nevr, nevrl) != nevrl)
	{
	  fprintf(stderr, "%s: read error nevr\n", deltarpm);
	  exit(1);
	}
      seql = bzread4(bfp);
      if (seql < 16)
	{
	  fprintf(stderr, "%s: corrupt delta\n", deltarpm);
	  exit(1);
	}
      seq = xmalloc(seql);
      if (bfp->read(bfp, seq, seql) != seql)
	{
	  fprintf(stderr, "%s: read error seq\n", deltarpm);
	  exit(1);
	}
      if (bfp->read(bfp, targetmd5, 16) != 16)
	{
	  fprintf(stderr, "%s: read error md5\n", deltarpm);
	  exit(1);
	}
      if (version != 0x444c5431)
	{
          targetsize = bzread4(bfp);
	  targetcomp = bzread4(bfp);
	  targetcompparalen = bzread4(bfp);
	  if (targetcompparalen)
	    {
	      targetcomppara = xmalloc(targetcompparalen);
	      if (bfp->read(bfp, targetcomppara, targetcompparalen) != targetcompparalen)
		{
		  fprintf(stderr, "%s: read error md5\n", deltarpm);
		  exit(1);
		}
	    }
	}
      else
	{
	  char *compressor = headstring(dh, TAG_PAYLOADCOMPRESSOR);
	  if (compressor && !strcmp(compressor, "bzip2"))
	    targetcomp = CFILE_COMP_BZ;
	  else
	    targetcomp = CFILE_COMP_GZ;
	}
      leadl = bzread4(bfp);
      if (leadl < 96 + 16)
	{
	  fprintf(stderr, "%s: corrupt delta\n", deltarpm);
	  exit(1);
	}
      lead = xmalloc(leadl);
      if (bfp->read(bfp, lead, leadl) != leadl)
	{
	  fprintf(stderr, "%s: read error lead\n", deltarpm);
	  exit(1);
	}
      payformatoff = bzread4(bfp);
      if (payformatoff > dh->dcnt - 4)
	{
	  fprintf(stderr, "%s: bad payformat offset\n", deltarpm);
	  exit(1);
	}
      strncpy(dh->dp + payformatoff, "cpio", 4);
      inn = bzread4(bfp);
      outn = bzread4(bfp);
      in = xmalloc(2 * inn * sizeof(unsigned int));
      out = xmalloc(2 * outn * sizeof(unsigned int));
      paylen = 0;
      for (i = 0; i < inn; i++)
	in[2 * i] = bzread4(bfp);
      for (i = 0; i < inn; i++)
	{
	  in[2 * i + 1] = bzread4(bfp);
	  paylen += in[2 * i + 1];
	}
      for (i = 0; i < outn; i++)
	out[2 * i] = bzread4(bfp);
      for (i = 0; i < outn; i++)
	{
	  out[2 * i + 1] = bzread4(bfp);
	  paylen += out[2 * i + 1];
	}

      outlen = bzread4(bfp);
      addbz2len = bzread4(bfp);
      if (addbz2len)
	{
	  addbz2 = xmalloc(addbz2len);
	  if (bfp->read(bfp, addbz2, addbz2len) != addbz2len)
	    {
	      fprintf(stderr, "%s: read error addbz2\n", deltarpm);
	      exit(1);
	    }
	}
      inlen = bzread4(bfp);
      numblks = (outlen + BLKSIZE - 1) >> BLKSHIFT;

      maxblockuse = xcalloc(numblks, sizeof(unsigned int));
      vmem = xcalloc(numblks, sizeof(struct blk *));

      if (verbose)
	{
	  printf("%u bytes source size\n", outlen);
	  printf("%u bytes target size\n", paylen);
	  printf("%u bytes internal data block size\n", inlen);
	  printf("%u bytes add data block size\n", addbz2len);
	  printf("%d blocks\n", numblks);
	  printf("%d copy instructions\n", inn + outn);
	}

      off = 0;
      for (i = 0; i < inn; i++)
	{
	  off += in[2 * i + 1];
	  if (off > inlen)
	    {
	      fprintf(stderr, "corrupt delta instructions (indata off + len %d > %d)\n", off, inlen);
	      exit(1);
	    }
	}
      off = 0;
      for (i = 0; i < outn; i++)
	{
	  if (out[2 * i] & 0x80000000)
	    off -= out[2 * i] ^ 0x80000000;
	  else
	    off += out[2 * i];
	  if (off > outlen)
	    {
	      fprintf(stderr, "corrupt delta instructions (outdata off %d > %d)\n", off, outlen);
	      exit(1);
	    }
	  out[2 * i] = off;
	  bs = off >> BLKSHIFT;
	  off += out[2 * i + 1];
	  if (off < 1 || off > outlen)
	    {
	      fprintf(stderr, "corrupt delta instructions (outdata off + len %d > %d)\n", off, outlen);
	      exit(1);
	    }
	  be = (off - 1) >> BLKSHIFT;
	  for (; bs <= be; bs++)
	    maxblockuse[bs] = i;
	}
    }

  if (info)
    {
      struct rpmhead *dsigh;
      unsigned int *size;
      char *dhnevr;

      dhnevr = headtonevr(dh);
      if (version)
	printf("deltarpm version: %c\n", version & 0xff);
      printf("sequence: %s-", nevr);
      for (i = 0; i < seql; i++)
	printf("%02x", seq[i]);
      putchar('\n');
      printf("source rpm: %s\n", nevr);
      printf("souce payload size: %u\n", outlen);
      printf("target rpm: %s\n", dhnevr);
      printf("target payload size: %u\n", paylen);
      if (targetcomp != CFILE_COMP_XX)
        printf("target payload compression: %s\n", comp2str(targetcomp));
      if (targetsize == 0)
	{
	  dsigh = xmalloc(sizeof(*dsigh) + leadl - 96 - 16);
	  memcpy(dsigh->data, lead + 96 + 16, leadl - 96 - 16);
	  memcpy(dsigh->intro, lead + 96, 16);
	  dsigh->cnt = lead[96 + 8] << 24 | lead[96 + 9] << 16 | lead[96 + 10] << 8 | lead[96 + 11];
	  dsigh->dcnt = lead[96 + 12] << 24 | lead[96 + 13] << 16 | lead[96 + 14] << 8 | lead[96 + 15];
	  dsigh->dp = dsigh->data + dsigh->cnt * 16;
	  size = headint32(dsigh, 1000, (int *)0);
	  if (size)
	    {
	      targetsize = leadl + *size;
	      free(size);
	    }
	  free(dsigh);
	}
      if (targetsize)
        printf("target size: %u\n", targetsize);
      printf("target md5: ");
      for (i = 0; i < 16; i++)
	printf("%02x", targetmd5[i]);
      putchar('\n');
      printf("internal data block size: %d\n", inlen);
      printf("compressed add data block size: %d\n", addbz2len);
      printf("instructions: %d\n", inn + outn);
      free(dhnevr);
      if (bfp)
        bfp->close(bfp);
      exit(0);
    }

  if (targetcompparalen)
    {
      fprintf(stderr, "deltarpm contains unknown compression parameters\n");
      exit(1);
    }
  
  if (!fromrpm)
    {
      pid_t pid;
      int pi[2];

      if (dh && !headstring(dh, TAG_SOURCERPM))
	{
	  fprintf(stderr, "cannot reconstruct source rpms from filesystem\n");
	  exit(1);
	}
      if (pipe(pi))
	{
	  perror("pipe");
	  exit(1);
	}
      if ((pid = fork()) == (pid_t)-1)
	{
	  perror("fork");
	  exit(1);
	}
      if (pid == 0)
	{
	  close(pi[0]);
	  if (pi[1] != 1)
	    {
	      dup2(pi[1], 1);
	      close(pi[1]);
	    }
	  execlp(RPMDUMPHEADER, RPMDUMPHEADER, nevr, (char *)0);
	  perror(RPMDUMPHEADER);
	  _exit(1);
	}
      close(pi[1]);
      fd = pi[0];
    }
  else
    {
      if ((fd = open(fromrpm, O_RDONLY)) < 0)
	{
	  perror(fromrpm);
	  exit(1);
	}

      if (read(fd, rpmlead, 96) != 96 || rpmlead[0] != 0xed || rpmlead[1] != 0xab || rpmlead[2] != 0xee || rpmlead[3] != 0xdb)
	{
	  fprintf(stderr, "%s: not a rpm\n", fromrpm);
	  exit(1);
	}
      if (rpmlead[4] != 0x03 || rpmlead[0x4e] != 0 || rpmlead[0x4f] != 5)
	{
	  fprintf(stderr, "%s: not a v3 rpm or not new header styles\n", fromrpm);
	  exit(1);
	}
      h = readhead(fd, 1);
      if (!h)
	{
	  fprintf(stderr, "could not read signature header\n");
	  exit(1);
	}
      free(h);
    }
  h = readhead(fd, 0);
  if (!h)
    {
      if (fromrpm)
        fprintf(stderr, "could not read header\n");
      exit(1);
    }
  fnevr = headtonevr(h);
  if (strcmp(fnevr, nevr) != 0)
    {
      fprintf(stderr, "delta rpm made for %s, not %s\n", nevr, fnevr);
      exit(1);
    }
  if (headtofb(h, &fb))
    {
      fprintf(stderr, "bad header\n");
      exit(1);
    }
  sdesc = expandseq(seq, seql, &nsdesc, &fb, checkflags);
  if (!sdesc)
    {
      fprintf(stderr, "could not expand sequence data\n");
      exit(1);
    }
  if (check)
    exit(0);

  l = 0;
  for (i = 0; i < nsdesc; i++)
    if (sdesc[i].cpiolen > l)
      l = sdesc[i].cpiolen;
  if (l < 124)
    l = 124;			/* room for tailer */
  cpiodata = xmalloc(l + 4);	/* extra room for padding */

  if (fromrpm)
    {
      if ((outfp = cfile_open(CFILE_OPEN_RD, fd, 0, CFILE_COMP_XX, CFILE_LEN_UNLIMITED, 0, 0)) == 0)
	{
	  fprintf(stderr, "%s: payload open failed\n", deltarpm);
	  exit(1);
	}
    }
  else
    {
      int status;
      close(fd);
      wait(&status);
    }

  rpmMD5Init(&wrmd5);
  if ((ofp = fopen(argv[optind + 1], "w")) == 0)
    {
      perror(argv[optind + 1]);
      exit(1);
    }
  if (fwrite(lead, leadl, 1, ofp) != 1)
    {
      fprintf(stderr, "write error\n");
      exit(1);
    }
  rpmMD5Update(&wrmd5, lead, leadl);
  if (fwrite(dh->intro, 16, 1, ofp) != 1)
    {
      fprintf(stderr, "write error\n");
      exit(1);
    }
  rpmMD5Update(&wrmd5, dh->intro, 16);
  if (fwrite(dh->data, 16 * dh->cnt + dh->dcnt, 1, ofp) != 1)
    {
      fprintf(stderr, "write error\n");
      exit(1);
    }
  rpmMD5Update(&wrmd5, dh->data, 16 * dh->cnt + dh->dcnt);

  if (addbz2len)
    {
      addbz2strm.bzalloc = NULL;
      addbz2strm.bzfree = NULL;
      addbz2strm.opaque = NULL;
      if (BZ2_bzDecompressInit(&addbz2strm, 0, 0) != BZ_OK)
	{
	  fprintf(stderr, "addbz2: BZ2_bzDecompressInit error\n");
	  exit(1);
	}
      addbz2strm.next_in = addbz2;
      addbz2strm.avail_in = addbz2len;
      addbz2blk = xmalloc(BLKSIZE);
    }

  obfp = cfile_open(CFILE_OPEN_WR, CFILE_IO_FILE, ofp, targetcomp, CFILE_LEN_UNLIMITED, (cfile_ctxup)rpmMD5Update, &wrmd5);
  if (!obfp)
    {
      fprintf(stderr, "payload write error\n");
      exit(1);
    }
  idx = 0;
  ioff = 0;
  paywritten = 0;
  while (inn > 0)
    {
      on = *in++;
      if (on > outn)
	{
	  fprintf(stderr, "corrupt delta instructions\n");
	  exit(1);
	}
      while (on > 0)
	{
	  off = *out++;
	  len = *out++;
	  paywritten += len;
	  outn--;
	  on--;
	  bs = off >> BLKSHIFT;
	  while (len > 0)
	    {
	      if (!lastblk || bs != lastblk->id)
		{
		  lastblk = vmem[bs];
		  if (!lastblk || lastblk->type == BLK_PAGE)
		    lastblk = getblock(bs, sdesc, nsdesc, &fb, idx);
		}
	      l = off & BLKMASK;
	      if (l + len > BLKSIZE)
		l = BLKSIZE - l;
	      else
		l = len;
	      b = lastblk->e.buf + (off & BLKMASK);
	      if (addbz2len)
	        {
		  addbz2strm.next_out = addbz2blk;
		  addbz2strm.avail_out = l;
		  BZ2_bzDecompress(&addbz2strm);
		  if (addbz2strm.avail_out != 0)
		    {
		      fprintf(stderr, "addbz2: BZ2_bzDecompress error\n");
		      exit(1);
		    }
		  for (i = 0; i < l; i++)
		    addbz2blk[i] += b[i];
		  b = addbz2blk;
	        }
	      if (obfp->write(obfp, b, l) != l)
		{
		  fprintf(stderr, "write error\n");
		  exit(1);
		}
	      len -= l;
	      off += l;
	      bs++;
	    }
	  idx++;
          if (percent)
	    {
	      curpercent = paywritten * 100. / (paylen ? paylen : 1);
	      if (curpercent != lastpercent)
		{
		  if (percent > 1)
		    printf("%d percent finished.\n", curpercent);
		  else
		    {
		      printf("\r%d percent finished.", curpercent);
		      fflush(stdout);
		    }
		  lastpercent = curpercent;
		}
	    }
	}
      len = *in++;
      ioff += len;
      paywritten += len;
      while (len > 0)
	{
	  l = len > sizeof(buf) ? sizeof(buf) : len;
	  if (bfp->read(bfp, buf, l) != l)
	    {
	      fprintf(stderr, "%s: read error data area\n", deltarpm);
	      exit(1);
	    }
	  if (obfp->write(obfp, buf, l) != l)
	    {
	      fprintf(stderr, "write error\n");
	      exit(1);
	    }
	  len -= l;
	}
      inn--;
      if (percent)
	{
	  curpercent = paywritten * 100. / (paylen ? paylen : 1);
	  if (curpercent != lastpercent)
	    {
	      if (percent > 1)
		printf("%d percent finished.\n", curpercent);
	      else
		printf("\r%d percent finished.", curpercent);
	      fflush(stdout);
	      lastpercent = curpercent;
	    }
	}
    }
  if (percent > 1)
    printf("100 percent finished.\n");
  else if (percent)
    printf("\r100 percent finished.\n");
  if (obfp->close(obfp) == -1)
    {
      fprintf(stderr, "write error\n");
      exit(1);
    }
  if (outfp)
    outfp->close(outfp);
  if (addbz2len)
    BZ2_bzDecompressEnd(&addbz2strm);
  if (verbose)
    {
      printf("used %d core pages\n", ncoreblk);
      printf("used %d swap pages\n", npageblk);
      printf("had to recreate %d core pages\n", ndropblk);
    }
  rpmMD5Final(wrmd5res, &wrmd5);
  if (memcmp(wrmd5res, targetmd5, 16) != 0)
    {
      fprintf(stderr, "%s: md5 mismatch of result\n", deltarpm);
      exit(1);
    }
  exit(0);
}
