#!/usr/bin/env python

"""
Copy store information into another store.

Copyright (C) 2014, 2015, 2016, 2017 Paul Boddie <paul@boddie.org.uk>

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 3 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
details.

You should have received a copy of the GNU General Public License along with
this program.  If not, see <http://www.gnu.org/licenses/>.
"""

from os.path import abspath, split
import sys

# Find the modules.

try:
    import imiptools
except ImportError:
    parent = abspath(split(split(__file__)[0])[0])
    if split(parent)[1] == "imip-agent":
        sys.path.append(parent)

from imiptools.config import settings
from imiptools.data import Object
from imiptools.stores import get_store, get_publisher, get_journal

def copy_store(from_store, from_journal, to_store, to_journal):

    """
    Copy stored information from the specified 'from_store' and 'from_journal'
    to the specified 'to_store' and 'to_journal' respectively.
    """

    # For each user...

    for user in from_store.get_users():

        # Copy requests.

        requests = from_store.get_requests(user)
        if requests:
            to_store.set_requests(user, requests)

        # Copy events, both active and cancellations.

        for dirname in (None, "cancellations"):

            # Get event, recurrence information.

            for uid, recurrenceid in from_store.get_all_events(user, dirname=dirname):
                d = from_store.get_event(user, uid, recurrenceid, dirname=dirname)
                if d:
                    to_store.set_event(user, uid, recurrenceid, Object(d).to_node())
                    if dirname == "cancellations":
                        to_store.cancel_event(user, uid, recurrenceid)
                else:
                    print >>sys.stderr, "Event for %s with UID %s and RECURRENCE-ID %s not found in %s" % (
                        (user, uid, recurrenceid or "null", dirname or "active events"))

                # Copy counter-proposals.

                if dirname is None:
                    for other in from_store.get_counters(user, uid, recurrenceid):
                        d = from_store.get_counter(user, other, uid, recurrenceid)
                        if d:
                            to_store.set_counter(user, other, Object(d).to_node(), uid, recurrenceid)
                        else:
                            print >>sys.stderr, "Counter-proposal for %s with UID %s and RECURRENCE-ID %s not found in %s" % (
                                (user, uid, recurrenceid or "null", dirname or "active events"))

        # Copy free/busy information for the user.

        freebusy = from_store.get_freebusy(user)
        if freebusy:
            to_freebusy = to_store.get_freebusy_for_update(user)
            for period in freebusy:
                to_freebusy.insert_period(period)
            to_store.set_freebusy(user, to_freebusy)

        # Copy free/busy information for other users.

        for other in from_store.get_freebusy_others(user):
            freebusy = from_store.get_freebusy_for_other(user, other)
            if freebusy:
                to_freebusy = to_store.get_freebusy_for_other_for_update(user, other)
                for period in freebusy:
                    to_freebusy.insert_period(period)
                to_store.set_freebusy_for_other(user, to_freebusy, other)

        # Copy free/busy offers.

        offers = from_store.get_freebusy_offers(user)
        if offers:
            to_offers = to_store.get_freebusy_offers_for_update(user)
            for period in offers:
                to_offers.insert_period(period)
            to_store.set_freebusy_offers(user, to_offers)

    # For each quota group...

    if from_journal and to_journal:
        for quota in from_journal.get_quotas():

            # Copy quota limits.

            to_journal.set_limits(quota, from_journal.get_limits(quota))

            # Copy group mappings.

            to_journal.set_groups(quota, from_journal.get_groups(quota))

            # Copy delegates.

            to_journal.set_delegates(quota, from_journal.get_delegates(quota))

            # Copy journal details.

            for group in from_journal.get_quota_users(quota):
                to_journal.set_entries(quota, group, from_journal.get_entries(quota, group))

            # Copy individual free/busy details.

            for other in from_journal.get_freebusy_others(quota):
                to_journal.set_freebusy_for_other(quota, other, from_journal.get_freebusy_for_other(quota, other))

# Main program.

if __name__ == "__main__":

    # Interpret the command line arguments.

    from_store_args = []
    to_store_args = []
    l = ignored = []

    for arg in sys.argv[1:]:
        if arg in ("-t", "--to"):
            l = to_store_args
        elif arg in ("-f", "--from"):
            l = from_store_args
        else:
            l.append(arg)

    if len(from_store_args) not in (0, 2, 3) or len(to_store_args) not in (2, 3):
        print >>sys.stderr, """\
Usage: %s \\
       [ ( -f | --from ) <store type> <store directory> [ <journal directory> ] ] \\
         ( -t | --to ) <store type> <store directory> [ <journal directory> ]

Need details of a destination store indicated by the -t or --to option.
In addition, details of a source store may be indicated by the -f or --from
option; otherwise, the currently-configured store is used.
""" % split(sys.argv[0])[1]
        sys.exit(1)

    # Override defaults if indicated.

    getvalue = lambda value, pos=0, default=None: value and len(value) > pos and value[pos] or default

    from_store_type = getvalue(from_store_args, 0, settings["STORE_TYPE"])
    from_store_dir = getvalue(from_store_args, 1)
    from_journal_dir = getvalue(from_store_args, 2)

    to_store_type = getvalue(to_store_args, 0)
    to_store_dir = getvalue(to_store_args, 1)
    to_journal_dir = getvalue(to_store_args, 2)

    # Obtain store-related objects.

    from_store = get_store(from_store_type, from_store_dir)
    from_journal = from_journal_dir and get_journal(from_store_type, from_journal_dir)

    to_store = get_store(to_store_type, to_store_dir)
    to_journal = to_journal_dir and get_journal(to_store_type, to_journal_dir)

    # Process the store.

    copy_store(from_store, from_journal, to_store, to_journal)

# vim: tabstop=4 expandtab shiftwidth=4
