/*
* Copyright (C) 2022 Apple Inc. All rights reserved.
*
* Redistribution and use in source and binary forms, with or without
* modification, are permitted provided that the following conditions
* are met:
* 1. Redistributions of source code must retain the above copyright
*    notice, this list of conditions and the following disclaimer.
* 2. Redistributions in binary form must reproduce the above copyright
*    notice, this list of conditions and the following disclaimer in the
*    documentation and/or other materials provided with the distribution.
*
* THIS SOFTWARE IS PROVIDED BY APPLE INC. ``AS IS'' AND ANY
* EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
* IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
* PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE INC. OR
* CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
* EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
* PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
* PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
* OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
* OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#include "config.h"
#include "AllowedFonts.h"

#include <pal/crypto/CryptoDigest.h>
#include <wtf/NeverDestroyed.h>
#include <wtf/RobinHoodHashSet.h>
#include <wtf/text/AtomString.h>
#include <wtf/text/AtomStringHash.h>
#include <wtf/text/Base64.h>

namespace WebCore {

static const MemoryCompactLookupOnlyRobinHoodHashSet<AtomString>& allowedFontHashesInLockdownMode()
{
    // Set of base64 strings of the sha256 representation of each allowed font binary.
    static NeverDestroyed<MemoryCompactLookupOnlyRobinHoodHashSet<AtomString>> allowedFontHashes(std::initializer_list<AtomString> {
        // custom font Ahem-10000A.ttf for testing (Tools/TestWebKitAPI/Tests/WebKitCocoa/Ahem-10000A.ttf)
        "OKCJZv9fAWEfrno3PhsS4LJwGfHZ7TqRI2lXGxNUVLk="_s,
        // fonts from app.slack.com
        "7E9jikb2q8/UabqKh32bUXDsUc3kCdzoQui01VIUgVQ="_s,
        "9z6CRGSFgnuS13R0Z1SyQj8ko99zovsa4jBd9eabBEY="_s,
        "nRMe4a1bWOwjXUQpKVQ1RLz3GPo5+Y5n4h+RjmcJCWg="_s,
        "Q1I4D5LOf5pKSiMwa5kr7RAFXb//6QmHzHIIPlg/woA="_s,
        "rIE6rzOQKSw1HBaZT1/2p/oCaWynDdZfj55UuSq14uc="_s,
        "t5eB7+3jeQO+IS/N9jlV5ByGSeZ4trg634JEWdJAoYg="_s,
        "dUI1r5Ss5o7EB829vKq0X0uvho8y/rPsC8pXrbwMkZM="_s,
        "Xi1l8V4mighrjQ2IEZ5V1aVUqnyPg1pI/4OTnfqD0Hs="_s,
        "GqPt1TOUDZTB5BcUNxPlqvIsLSabCiBdYRt3DGuwksY="_s,
        "WX/tMhRDjksLufxWguujkjyF7sWU3FnXNkn2Vi29kDA="_s,
        "V5QVA3QK3y4ORLW+oOK1BXo020y6AxrL1zT2bCsrmew="_s,
        "gmknjdk6dJf4R4wwsQlGA7oa2Nmjrq/Z81PjgZrILko="_s,
        "JSSmSpZb87ziiel+FjGVTHVhlRwKJeG7d/kJvbrU8Rw="_s,
        "3wtC6B29NIcVMZQl8Vie6TuwXaD8QBeneRO3uX4tKQ4="_s,
        "TwR46iKaBl2v/akGo4vrC655vA6zbWA37r0Lxzzxo6s="_s,
        "biH7UQLyk4MYR+JdcEpll0C3/ym9REBT27P6gWQ+nbM="_s,
        "tbjKwdiwKbRxy39tXizHE6O7z+qIjmqF1Zs+ydR47NM="_s,
        "SUs157eujHRqlh6RjjNXQjsAuJq3oaCv8ZTJSeDKUwA="_s,
        // fonts for apple.com
        "/Yc6VsFwolk3H1svy5QaId2D82s2P697rYVWhINXPac="_s /* sf-pro-text_light.woff2 */,
        "1CwpIP7pi2opVpeuQhfuEhXqAcN+LziIffQr2yoaypU="_s /* sf-pro-display_regular.woff2 */,
        "5JwO3xvFFPisB1jprJwfwRuP082ac9snYfUPX98Wya0="_s /* sf-pro-icons_medium.woff2 */,
        "6Ax3KPoEMjoQnZ2pP5bIHcQk6K0Y6feUn4uGZ/huPrY="_s /* sf-pro-icons_regular.woff2 */,
        "7SOWv1LYij7yrQExSKlXd10KZuw71VElPkKORodocqY="_s,
        "AWuYeqFHfsJj1VSHNOT6YeTMP9SMhSukzXO/m/Q0AZA="_s,
        "Axp6ZLMqSxkV8v4C3QBqpA15gn4i2Mbw9tApQWmWubY="_s,
        "Bom/qRvs+XOwx+XGlvmtGy8yv/OMXYhfZ/A/x87uY0U="_s,
        "CX9zKuEqR80srCnsiverjJdj8EiP5ZuT4t6mYRJRfSo="_s /* sf-pro-icons_heavy.woff2 */,
        "CvbFz+fVIb5A3sjyBEVB8DCWVQ2MekusqtfVqAWsGuo="_s,
        "F0JP5c86T0xMdimJ3ZxNQOuLFKwtgMFtcIec1/QGezk="_s,
        "GVQNI+ZO5CljwVWfXQqGKKGU7wxnkrh5dK1noQnUQB8="_s,
        "L47ZjFCKuT3j2P4XmiWG7asOJAWSe39Ms+FeZEnC0AI="_s /* sf-pro-icons_light.woff2 */,
        "MjG3fuJ3Xa26p23oX5V2OXav8QkfY6Z/pVPXJ6btuTM="_s /* sf-pro-text_regular.woff2 */,
        "OQWwrqKryiAWa4i2IUlriw2Sg7DCST/sE6Pn3k1Sbow="_s /* sf-pro-display_bold.woff2 */,
        "OYWx5HkSqSu8wluuS2Bd/HZr+3wy7ltErhkbGBa2yD8="_s /* sf-pro-icons_ultralight.woff2 */,
        "OeI4MbaJldvmAqYPqiSL6Z5Stxcwly3bUzeOv8QKOk8="_s /* sf-pro-display_semibold.woff2 */,
        "P1W9iQxdAYydhgAzhpYtX2T+xzd3wB3kO56hL7eHXv0="_s,
        "ViVhrJRLi7lM+RatKCnRmFUVz3ji9cjcpwpI4JK+6v4="_s /* sf-pro-text_bold.woff2 */,
        "YJRBzgp2ROdUKMIWu5HpA9XyICSeOtsCRIm7ZDdzMNo="_s /* sf-pro-icons_semibold.woff2 */,
        "YisKg1FjW/sDd40Rr6ZaOZCnciKUsbU6n+sl1f501bg="_s,
        "YrJ94xZrMh845nRK4fiBbtajAoOLjt15GDfzJ8EZ+0c="_s,
        "bfcBHlgBQm8KvGrOG+YLAHZGaekPX75sVGtXJNL8UJs="_s,
        "ewBBxIumcIf89fbo7I0k6V2wbK6b94xFtFQphLy3IIw="_s /* sf-pro-text_semibold.woff2 */,
        "iYiF+MKBEMgiOq8NYSIYx4Bkk/6mxAM78phHjbkAR60="_s,
        "jYzorjIWLe8J7c/KJ40RrXWqCeGpLrEreCRdei5rVY0="_s /* sf-pro-icons_black.woff2 */,
        "mMEv7VyqEhcDr23jfi/lhLN/iKLHCqrU5QAT5Jap4nk="_s,
        "maGVvgErnoGoeLIfCtrkhguAojs4Nq5BlcWkcsOEjcI="_s /* sf-pro-icons_bold.woff2 */,
        "qgMI1cxvE2E6XXsfmOZSpPzhgfXjmN/Zvi+t7T4gCYc="_s,
        "r3z/Bd8zcC0mLH2x7nA+O3s+svpAfoRx9yyc2DV6rsM="_s,
        "sC8ehvkP8iFYR3TtsKh9NcneR/+xdpnWq/MyLsMBrfI="_s /* sf-pro-icons_thin.woff2 */,
        "t+Gy+JmRfIqc/kRLffbjHd9wU7sVigrVqyjXIwNLWA8="_s,
        "vLs8XM2hGN29c5rFw9bTtbiEuRp3mEK1kbse4T0Bya8="_s,
        "xYWI08pk1jRI9N74aihsx6tQnu9MqyOtXOd1uvPbxb0="_s,
        "yfLvl8Bo1pobijmu6FSqH/TZJmChFdS6l7nh8702rCg="_s,
        // fontawesome-free-6.2.1-web to fontawesome-free-5.0.1
        "LOs4y1l72hHgXOKRg3ZgcFWx8lqvpXllkj6FWs3JQEM="_s,
        "van2FSpLPMi5Vh8WrydoNkiPUa9EogXpuMyG4UWCMGA="_s,
        "eVabv5jgRnQ0J2c8L1mpZJ7oM/KpCJsuZJfUNbX+Gwk="_s,
        "nwM0w5XEt6YRC2f4FwxLhKddjQ0xTW5E650ir3cBJv8="_s,
        "KBjfMy0IO69dLTmg09m4/mW7zZmHZxeX35I63LskIC8="_s,
        "49zSX9nudhoRpiNy6Kj8LfFFr9MI2/Qm2RZK9d96QbU="_s,
        "+QcCEOSZVHSusHEDRrwmlf15PUY2rMOLkJcot+8WSzw="_s,
        "YFM7xbdQzfNqtOuoRESAuPwVInf/hte2XpdCjdHTrTU="_s,
        "5V2kzpKpta8B69qxSq0B3zCGel/HvVqTD5liA6EO5/4="_s,
        "aPn93z/S5B1A4GsTDFAbiJ4bkt/iFhIItTRPRP5F7fE="_s,
        "ZYz0PbJOnUxXiQ6ViqdGVqExOXVN4k8Z5wbwo1Unnk0="_s,
        "7XUUtsOl/cOGv/TczK7l4Mcug88x+Q/1rE+3DjP7aFc="_s,
        "cro+icY4I9BVFla+sBOgnfKbLHpXsQxb/VpQ4Frv7To="_s,
        "8WwIU6XMovMS6GmF9LdMMcGW11MnsVj3nUI40WMabFk="_s,
        "UOqtkko4NP+b0QVFuz07A8NlB+6/M3rRpZdEtELg7M0="_s,
        "WNaXUHptBfL8J9NVX/cGTDJv2bAuT/mlbY5ewiIoG9I="_s,
        "VbwpiDGCgXpMMsFbDSzgIoTJkQVB0syyhsprL96vuZ8="_s,
        "rCTlMDiBIT6S7zXlXBds4qbUOqe8F4SQDIfYmT1ccDI="_s,
        "lxaKgY2o+HCxzxW/5nGdjlXM5SmaZnCScGq8nHIomHI="_s,
        "N0fun/ZGkelrSjAlpgldWiZd8HEOFK6UWOqv8g0+Ap0="_s,
        "OnvQGZGS9IpfZ5Apf3DmW5S/VpOKnWay52PakW67rGA="_s,
        "P4EVt2nNzydhlQjLgoMZdGfW9ahs3t11b+HK4vSCnqA="_s,
        "tJN6qBYAacUduntSczVgz5ENKqYRIV3KwudqjIdCk8E="_s,
        "+JQc9osZmRvKvPEOV69z4JWb7pIEZl4c8lORuTBH95I="_s,
        "vtdUYk3HCJdfIYqKeVjEnkxViW9VR4P1O56PtpNgfBQ="_s,
        "nmvVstdbukhdIzfQIHUHRJg6NSHsaXrf4hsp7k8U9qk="_s,
        "S1uYnzdKY3weV6sn4qq2tbmGhSSfeCAe5gFzoa+lISQ="_s,
        "85Mau+DxueGCLa9kljKO8f1kByMrHQ4VuKDHnn3K+Zw="_s,
        "iWdd/sS/UuEA8deWH7XdtVAn8kG2uXbi1OgjbrWEJI8="_s,
        "Ee5MrZa4FY3djhWGorxTVOVrNaI/AKeuxOzIYhduAy8="_s,
        "BkJaGvkBfQLWt8qmMsjy6RnT52n+vZTJniPt5bi31vc="_s,
        "VKABXhLnKvmaFt9+w+XkCMmjs7CefB37HjqU9EMmh3M="_s,
        "0s208u1LWF13IGjfdcAdCjYNRdJylBiPzPXOtCVf3IM="_s,
        "UA72YZpkWgtUpuqhH3enH2faen4Mm3Pw5D5zN2cNBLg="_s,
        "sXXNy1ZIINye6UFdrOGJkSGJDBcWfBnEQYqj35Q7yeI="_s,
        "HK0xku9IRT6esqHPh50F87XNqAkhm0GiOTQVoe6mhvE="_s,
        "CtWTnOPCH5as0xO4d7lRpLAECua/j2xfeYWeYcIeqGc="_s,
        "skUC2WbaOsAhpQObouFXVNpRVm6HxymjWeTGYEbFtDM="_s,
        "PkOPiOWUBEcaauaNRhN4csQawu3nNnsPaXv8tnF/fRQ="_s,
        "Hj5AcG+q0g/ENmmYT+15ifWxhZj6J9RylmeZxFdAlBk="_s,
        "y3qmsGqlqO6jZwZixLDDcQQEHBRXX8Fw3EhnegUGozo="_s,
        "6FYghzF7NMSyrGDijicrezPjdSOqzV8q26ek8QjkFcY="_s,
        "P2yMwxzikaV392QcqPWzgw2aZlNLraj+3bir0/BL+tk="_s,
        "lI7Pl1hPyOvz5JcCm/RjuNhkJa1xfpO0ogyRAeDG9GQ="_s,
        "bEcHZsLD4R+v/q193m8Nn0vU5+wXhDMuqFLNCLfXV9A="_s,
        "lHqsFqptKzsKT6IOILje+kYC6bQvj9n6KxlTi0xajqk="_s,
        "uY/+LSaM4z8yet/ziLfkg7pXOtSpQrqqbFJe4VwiSLg="_s,
        "lAs5CL+fwmP/epZA/XGaGj7MqeEiTpzkdYBT+gHtvMA="_s,
        "2/V54ByKZCtdggupCCOCYWQ4Rrm7hsGLy+RZL01cTTw="_s,
        "JwiEaieZN2CiZa/+QQdN2GzRMbUbE3nkd0Z7sil6F3g="_s,
        "2HwNWVafAPqA17I55QppTIcmOKFtzakrFT+GRrBg7UE="_s,
        "6Bx5IlNsr6ofuYH0AoGs8qglH3C9OoD/FxgiHu1UsKE="_s,
        "2rFnLLQOXmN8ICOzFCGaF/O9pHDF54mwyXEMhrNbl8Y="_s,
        "U4gpGx9z3F/ftbAvjFfsCE43PPj7jPnxRhQvCN9YqUU="_s,
        "dHHOyKGia7DwFCbUgQeGkTt3cHnS+WaoU/X5dvIusqo="_s,
        "T57j2PbmIWQpeeao9+dcV8udo0kYzAijir/heNuuHdI="_s,
        "1fRxR24WNuI7AJka6KhdNwOtpVvG1hYkcqKKqU+mTU4="_s,
        "ZDOiBYJy1eJyh/vjtrNqMzx7qvwlAQ5Va0nWIuluOzo="_s,
        "NSPzVRWd/5tqRoE4vIyPzdWSGMAXP+6lG+TYw0iNyhA="_s,
        "OLfDdPIi7MSSuvFEMUwpKmpT5tILyiW7Jt0q2nbE2j4="_s,
        "uXra1+zpeSr0EirogQPK2Pq1RVilKcS6ebUiTCsC6+k="_s,
        "WdxAgoPFekfSeyooUnJvQ9syhcq349Ypfo2nyQGFf3s="_s,
        "krp7+qQ6NclDU+loYNmTdjE+6bX85hJNTmQGcoD5qEE="_s,
        "aQvmH+WAIFEVWM5LBDdReIcmKfbHN+0bMyrsZYVf2Tc="_s,
        "eq28Ssf4BBPeAAUmlMGkTgsNvw4tf3OcYouIsou3K/o="_s,
        "8QG49w42vi6QBCIoden9RbJj9ydpxo4l48dSXThxcos="_s,
        "kVjuLviOpkhES6HJ/OoxC6oeXk5f6G4zjc0p/LCBz8c="_s,
        "Ivavd//usmfFJw+h5QaBrgGpoZ+Onvl2eY4BlslBreY="_s,
        "j7GWM/ElykZqRdqeTPKR2LlRnddMIU+Z3eg174Pzq5Y="_s,
        "7LVxnrZAg1nZH9JTBkYGSj8lGu1pH9SCdyicyNbVPfo="_s,
        "d5gWXuWjxoCTENgmHcvnyNDBLXlbewmnGvPrhuyPM/I="_s,
        "BdvFFlS5ZZDRdsJ++87yz0rASXSZqfKLcxtz7qOZBww="_s,
        "ykQcB5kqcWDk0mIuePiXDJuS8XZpTI3AdZIyZJaoAdY="_s,
        "29CNAlYCt5znGRCypoV3lZy/LvAvxmTiQ79QJ3dU/fw="_s,
        "FpQWiHydPDEL8CoS6vgiadTPd7Fq8ukXRbHBUrugBh0="_s,
        "bLmp99u25m8IsWFWEtXbFm4IYh3YS5Rda9VUpg+BTBc="_s,
        "l2oa1BXNRTKhJSgZYooooSPSE4jxeOxvO1xkk6dXcNU="_s,
        "Fmm+w28Sw1oAotI89xtgYchbhDX1wmRFqrM4+I3+Zik="_s,
        "tBXcsMxPjqxlSk88++LTb81AHdLO3K0Gov4qvwtzryM="_s,
        "4btu0MuAEjTX6UNh+P4iKkL5lHNeAt1q37bxHgHWpKs="_s,
        "fp8IHFasybHxtMfq4yrMUSQUb4ACuomrZARteAab4fI="_s,
        "7r0S7ACi2967pMtV4Vy29aX8VwNnGDiMAWeBXNSm2yc="_s,
        "uCAjTue6DxuklOFCSntN9w3ogvAo27UNS209HDIyWdc="_s,
        "QUqnWihXKumuKoT01fBVWT7LQlxIkkoYP5vfCu1Pm/w="_s,
        "QA1z0ao6Z3EX6pdJuEw4d9RHPLA0gAqdqnsj1bbdvhU="_s,
        "qDB5EkNz2SStFAL7wI0uJNAEMjTUwmVl8cNodF9V9dk="_s,
        "YrXnrp4u1g3NfLLggj3QiEV18hdq/2KfLfHpEt+uIOE="_s,
        "yCw6pMV/jujOdQ4Ygiy6DyZaZv2mehYg5Cx0na3va3Q="_s,
        "VOMVj15VugKyFljn0vxmH+w8KG0ng02Z5WOW7WaHBYA="_s,
        "ODbqX/sgmeUI8upFG/BIQVQ890Yam0jCJU9QXIKNhS4="_s,
        "F4IVa/ogVNtCkne0LxQymKHOA0N8BeJ3nKeHURKxnTw="_s,
        "+nn8TyVZES/z3SfPW+IW2bX4gGbO73KCZPWjoaGV/B4="_s,
        "/elBwJZPq5hPavBpez/skuffOE7smUG6v9Pky71iEcI="_s,
        "cJA0gF0QQp0khirWhdVhgIMoMp3r28pPcEh7UIMgo/8="_s,
        "s6v8zOJ1U2vqrrqZDpAKDAjtL++6v0qLy7Li/+PyQx8="_s,
        "GwmfiMBu0IaYclYcFX8OycvhM6CTnZ7OTuHh9UvUaD0="_s,
        "KbxEaUw5SSHR8AJxEoouTNgpNRYhbiTqwHpz+oIfwfU="_s,
        "nhIoYbxffWGQ1YCyKsxbHD+M26n0lVk8zNO0fSy/ov0="_s,
        "RSb4pmyix1W0SHkM/qpjRTgJX/vbTBIdOMqX6t9Nhko="_s,
        "FwGnIeJCgwQXHMn/w4+n07SEwOcZHveD9R0mfVWN75Q="_s,
        "WOq6+s89OPmVVMfRVuOJty5oIznOXt9b98a4qgf0QtM="_s,
        "zVwcBhSxXhtN79ZxEvqyOl0NLt/dynwjns/CTVze0uo="_s,
        "ePFLQN7Ex90KkIeRA2k/y2uexrnoHqIYIQyKyJffepY="_s,
        "AJrFCaIOYzIrhofJMPh60/o4p0r4QV8AFfd85EXNZt8="_s,
        "aCX7XoRFfRUzokQ/957Bf592ILkLy/1y7CA5dnEAgX0="_s,
        "fnG1Kl+3AIvHc1Esz6up64FgFcQdcBDPd9Ecf9nyGKo="_s,
        "OXvK67Jvmslcz6Tn5rz5gTgT8+Kx6UhtMB3xH1USlyo="_s,
        "De11IcrSNfiE7NFhpCpIi/Qw3O4b8sSfguq2MybuqWY="_s,
        "u6POMaBo86d6VFvw72JbSCNzmtpzPXhIcraO9yK8t5U="_s,
        "N1OivcFwzk62MekN9ItFKqxNMptA+V5FrHHpT+YH3jU="_s,
        "W09K7lXlWC9X+HNV+4LwUdXvp2z5FDFEIpoxfaAQP08="_s,
        "aM6ct0iUr45BePApyoV9pP/1lOAxAtZHaxSjQ0rNxhE="_s,
        "DLeBUpzGGmtTxa1LMOCHHktKo9auvkVHlrUccYJhonI="_s,
        "LPF4cu3KpIL0ZEPCFp77RZrtn2truvb+O46HAFrXoQM="_s,
        "Osr71exEJ++ZV+Eehvh7iPucwyAASnyVtYotjt992n4="_s,
        "D+//oVd3snnOYaBpMuBbrej8tyndm+4E6T/N0h6PRVI="_s,
        "tcuECan4Aq9MvntscxL63j49gWvn7iwYtq2FKr9oaZE="_s,
        "ofqNQt5Rfqh3F7xcrtBJMXdBzfCciCO3IPRoZTJnrRs="_s,
        "jnK3BrWtw1guuP1nXWFKxbVpHLnzeNSMTUn/Z7E1GJk="_s,
        "6WKzmUPoW3D6HqLxwAxnFpOo01t0WQVNkk0Ss617Xb0="_s,
        "uCAjTue6DxuklOFCSntN9w3ogvAo27UNS209HDIyWdc="_s,
        "8RlV55tOY2doMpBheU0tt+zCPVjCMkkImkimoSuBIYY="_s,
        "QtZcuhQ8sGRn6X8dhdYnE8bxAuFpCst+YET8O2zfxQ0="_s,
        "uCrY+8+b+ERyb2SO8mi3T4wvZo9W6v2YsFcD4Ib/HVs="_s,
        "OloZeUciO6vNng51npKEIC1wzjO5+Nfm/9P1vOX+xkk="_s,
        "16493a1ekO4vR6pN68wx425GIPMTEIlECCBbYwOmZ7w="_s,
        "ktCJGkli5k3GdI9a1O8IS2QnQBSr7Cm0EsrwlrganK8="_s,
        "4uIxZEe1NoriupQxVkNrMtaRsUmrvDyYAFgY28Yk45Y="_s,
        "B3erbrnUcU8/Dw3dcYQ2/iIhqayo0EN1oVC1evRaAhU="_s,
        "a9V73W43c5Wig4TWIVghEazOVi8Aiz9+eWsBh6421cM="_s,
        "SIEPLKXBqcsoUXekk7GKPKIsF3r+ubJCpZJ4jq/GBvI="_s,
        "TnyLgbszqQ2b7mGtbQ3GiFvXlQ555sBsnmGZCxPyuck="_s,
        "RY0WPK6GGVDAlOuJKZV36atzGvkghlQNpS0d0HR26BM="_s,
        "9H6VtajpydqM1Q9iaIjMIELW2pnCw0DPPcE32i5U9bI="_s,
        "p9hMFHTxqmlcI8Yhmgm5EghVzmcCVj/LSRHMfnKkVMY="_s,
        "9r2PUdDvOaGVlD7QhMYh7ACTF8JxSQDD7D/idLp7QG0="_s,
        "oYj4uEcxxZFDdw7zkcmtD6JTTTFoYtXLOEYjKFyVwuA="_s,
        "E8IiY5q9YR0GZhnBZcWPWWs63Q1mSxsyvrfH6Cya8hE="_s,
        "yj6hZ2G31EPGTP2Z3Rz4qoR5CiW7RwlYKTWVb+cdAU0="_s,
        "l0lW8be4LOzYroigtoXw1d/lyFNMJ4Tlmr7qcZ6tu8Q="_s,
        "4eMLtuVO6GlNd85j/5tx4RdMKt43kfCUxrm7O6KS/aY="_s,
        "VmDKN8uFHpEU+IgABQpNa1v4cdSL/n5ta6QzLYIgJV8="_s,
        "wqfXZzhxH2nqLb/QSq/pDz9/re8Fsyms+cFshtjmp3o="_s,
        "5Pan6dGnwLSWjZy51CLo2c3KIp9/866sf5bkuqJcdoA="_s,
        "XaMTsEZ/fBsY2YFnKyNGGt0x461BmIwwEBvaq7B05EY="_s,
        "Lds7QRSpy1ZeSbCvIcw6yIzQQ760m+MouPV6Y0mJ4Kw="_s,
        "17Gf4gEQGwirE0T/1HLVnOZSrTw/tfGUw+NaopCOZ0o="_s,
        "9GcVydMAR/jepOsl7YafeR6wacaw9hAjgsMqVq7q37s="_s,
        "jwZUD9d/Hv/h4tqOoQzsSjgt2pzG7wXYFuHW3kRAcvI="_s,
        "xhKHwvqYY7X7WETGg6ForGUgyU2CK7Q9Xq41w6KoIWY="_s,
        "sdHBsBLhp2IcyE/R5cHbq5IBL0DfdCPoeasvpJNSD+4="_s,
        "3JPXFHjIevnqecEvEzG8KPmF+n0kfY+tGpyWysQ0G6Y="_s,
        "u1BXplOWBUBxPwofrYkHTPq3D0knyhIyDeXTDfNyCvA="_s,
        "CVWLHz4PSmKTIyogDiZ9sO78JiP6e6g80eIOM0Chbro="_s,
        "BlJWo8jMPkgg0EmqxoEnWYkCJcXhgjS2pZNUPk6AH7k="_s,
        "hGX3KeuoXiOAf3fqpcXk1Z+9FLNmYL8A3Hq9mLIrhIs="_s,
        "xsDugQO2rKa2cOTiFYVmK1hrCH/DH9J8Z42XBw40+1w="_s,
        "JKzEx+XAmS2V49bBxCh0jPFEPAElLTdF/hT+DMoP26I="_s,
        "ydOS5U2FvFm87ZCVb3BK8vemvqsdxOu6kJxNCTz8yLk="_s,
        "k4BXQzEq/n2MTvSR3qNVI4PmbAuQ6mZzRdfu6adDsug="_s,
        "YxAXBhdCPEcYjKwqVFiop9qeAgIOedcYozmb9+pF7FU="_s,
        "Ns60mhutLp1fda9sULPGnyDWbYa/AQBHULlavFwf6X4="_s,
        "37AnS980iO/fO7FVTezUx0AkIdMkAVZUSCat8xO9k4k="_s,
        "RePLBmFRlJB+0CUvG3MbMaqV7fYbkE8COtOldOoTVCI="_s,
        "fUSfYPzQsbxwfvZ/Zmh4Brols2Yd7cfkPa5jrx7KTgM="_s,
        "UNKr2XCK/cNE6w3IkwT8ug4NoP6SBmDTKk9qrMdKqc4="_s,
        "d2ddBUJ85x9x9jF9kgJ9ohwLNEfgHYWML4IeoNOr0js="_s,
        "6uBek+Epi4qyYvsrlHcotIfeqT2zLRNBaEFqy+/lb+I="_s,
        "jXmXBLCjjtt4FHL9NR+2rbSii6wKdGAdghkwpjMPR28="_s,
        "aYqAC2yeX1ugmZUsF6/ASI6/ag/F5H+KlZhiCt0WWgM="_s,
        "emUJ4woZQuVOjeYSiCwk/mtU+59Ft9xFhjbZSc1Q5XQ="_s,
        "9iB13X5uepulCdOSrzURrETJEgZ8CUN7PlsbN4V0oBo="_s,
        "JPkv0BiM6Tiv6IpgG7NbbIsDuM8ePkFjUjBXA18ThzM="_s,
        "4QOywfo7iRCF3r4dYMmbvmz6APjEl/XCbEHS3KbWbIc="_s,
        "kbIXZxX4gjLlCQaNPRFnoIdht/r52FwjCcOz/uXPm0w="_s,
        "KiRmYdFigZMtiaVQ2RjRU5Ds6fyerMQy3hwqyloaiUs="_s,
        "DBOYZwofq9Fs4G2Ofk9METpea7Hon8urXNje2M25X40="_s,
        "5RJL6dM9BYurY4ikBWn3RqR9JWl1Lcwoxo6XnRWhXng="_s,
        "UH65GqP0gkXRRCljiwZouzDJulVMrGxLW7XsNlmENjk="_s,
        "G3FOvfyA6WLMe3+Xf2xSkWUq7MIcJ23y+0VwriFMKEc="_s,
        "YqJ0tJCZgHFAr9vEn+5mjyAI68fbtBYaAT+lRccKqkc="_s,
        "bVPHBvO3F4zPyCyU0/x5aIKNrNNRcX8nx1emXP2FMUc="_s,
        "/mnZSEFGLTl/rv8lPuCabceUG+kx+UKlXmud7487BI0="_s,
        "TXPygFlwFK6JInDO4XdROtMplV3pGtnSpFofV2CjL/o="_s,
        "/cH3UzAhEX2HaXcV6sJro9Tu+SJcfu27+xsNDP/wq4w="_s,
        "fRws5fNOGzVnU26NMDnGw0+CV3QpgsX/WltHqC5aZ9c="_s,
        "0nvHUhBcB5+KUW6RQkBqn8Esu0Cfm/hoHy3f4DYLUqY="_s,
        "P+iQ0Ijs8MybwbkGkgHlKXLbrWI3hlUkCQ4VmC0N5xg="_s,
        "x64301uO3TIvXIW49U+Mt29ehEZYGuINpefjqb3ukAs="_s,
        "LpES23dkA0a/k7xgMaLi7eeTMkUzJz3gLGE0XTm3DSY="_s,
        "cCjSV/EB7ewUcuhz/DzGM16EOhusQRecKvVQQWO5qtc="_s,
        "Ds3GGIpLLsSOLr+EoqZYTnhHPxIW1xGYMrXcEJvsdJI="_s,
        "gLUVxd3gnXsz913DgrTJG52yD9samXi2ceQlyY5Css0="_s,
        "4rNlzTz/svwp0yAuu//XAWjlnANntQiAB2c6YjCm/RE="_s,
        "n5gsWTgvCxoPn1POICShUEubUeTDcdK/zKS1CQ6wIf4="_s,
        "++WIuxDF77TxgDMIWTmookgE5g8+q+sL7cjXc7IRGxk="_s,
        "V6iqfT06yAKRRDl7cm+ScpAQ2DpsywMZN9kRoaY+X7o="_s,
        "4Uv4UTxoSPmD7SSFLpcA3slVibdI4mus1rLKfcm7Ku4="_s,
        "j3vJOwybz+ZT6cowkSZIEiGnTxX/4Zvm98/0CgEcdp0="_s,
        "9BmtekR382znPHSiPc54QVDKOPpQdajgYQlwnLtxaQM="_s,
        "7tR0pJvb90XBnkY/Bw5nl3wasng1YD63SdnlwknPgfg="_s,
        "6wmP8XJciBnqZ7VNhXtldNIMz7ssxFHB+rhftNlopi0="_s,
        "tS5Bkj39wlDk95FK+rM5OQtJ9oNOFY4ClM3fccj2JIs="_s,
        "xt8f+cvgLqZ49TE7Nb1+blKXJ3YJyiwyRUthhBjOfOo="_s,
        "SJJmnB2Aqeg3vp7epNtxHkImDszGkz4W7kSbyqr9fng="_s,
        "a9V73W43c5Wig4TWIVghEazOVi8Aiz9+eWsBh6421cM="_s,
        "TWTxQs8i1ANDu/VO4ODsSLeyvGVLm+eFF9VjQgcoZSQ="_s,
        "0zjpIOR3FROXbiupLSYOS2FSKzuiFIYjLd32PvAqe9c="_s,
        "8AuEdCeWbTmAluACP2rvWrWsasboSStsvRvLZBIQZR4="_s,
        "KKUiSeipZtDrwWi8mDwTjzDxwauDkm00KgXaNvYfLy8="_s,
        "XWjLX8UNDTZvCkmZd4opBTA6Wx9Tux6f9nnGU31WFWw="_s,
        "W2/rm3ljORR2O0Zlp4Zt6/mnpr+fP3PCte3/brfGFjk="_s,
        "t+aX68AeqNlrzaJW9v5jSvr1uPG/mFIsPXJjOCulv8c="_s,
        "JahUlzqKZmEvn4DEYAuQbYLd5lMMKggDQvxOi7+LiUs="_s,
        "pVh6nco2c7YEqKDhRNJo89yxgKrDN+Ky4WNwS8H8UIo="_s,
        "g4Y1bh1RL1cKrBtIg7ko/kv9ZJwdVKTQ9hg1c1nNO3Q="_s,
        "8h3dfp65Az1TCmJ2b9BSabXwwrHC0AW4ct7NAMcepas="_s,
        "O4s1mo3kmVCd/445R4/gT78+viRCVLEI5mgW8EjQ7Pg="_s,
        "RBZa+4sZhO6Tq4bzWMoMAM35X1dzHoXF7bagu47Gc9c="_s,
        "NTUZYNCxiyMdtpc96Pbu/Erz7/acSG8ALN9TjW8oT7s="_s,
        "WaAyMn2T0xruBWY7CfQKERRL+ZL8H4wiKvcs8YAVzrQ="_s,
        "gV+7OdLAdYEifB2eq3+wMb5U4yNGq3+aUn/kBg3blu4="_s,
        "SN5C5FfxphsDlQOLVgKf/ZY48SUNo1qHDorKH4ZaYtA="_s,
        "Xs+tjF85DW7HGVksVzYu46kvJ+hvLwv9RgcjDLeIspw="_s,
        "byoZ7slgYSDWEl54fu88ZKoMO6p+jjchyzsgJKpdpfo="_s,
        "nLWyaZfpZklx1ArqJKNprtayWXNi+Wo9SQGBhOdkJE4="_s,
        "9OFNhW366jCczh++1WyELeAjK+vFWz3Ushh1TaS8JBI="_s,
        "ktdtp0XOWk68aqGCGcclUZqMrFh7JXfgjSZCjQK+MVo="_s,
        "YGQQnpAeAvcJFidXEH0qb/07ThUli3YxgpZeNGtBLzk="_s,
        "7QwOe6RG3AvlyFgMsguBX0TrW5Z9sdIz3jF+MABZoTA="_s,
        "A9mrJbB7Y3yFCN146Ql5KBcReCkg0axiGFHleh0pVNs="_s,
        "JRJJMP+FOrL4cmEHWDp3FnzPJtzTnwa3XaHscQpv2lA="_s,
        "XvopiysKaqBWsXLwvxp+1gZQYYhDfTZn4vNR+9CdRDg="_s,
        "z4/mpNsU/0s1OmWWXjIzDTk+QQKFCNyk+scdY/l+oAs="_s,
        "WHAVjT6CPZEbrJlfFP5SUt0QWLesXhwHWdfy9r9bmzo="_s,
        "a9V73W43c5Wig4TWIVghEazOVi8Aiz9+eWsBh6421cM="_s,
        "MVFAAmmkjaLum+xhsX6mLnKJzWx8od+SjExh8fwXXD4="_s,
        "kCD5pLiH37m6QzEOhENmrQxnYkV7MeQNmorqNJeYlRc="_s,
        "JUAPCdmn78TsaG6yNhhqkNF1tR8Jx64PurdSIy0owJo="_s,
        "Kf23YTnquhCFIPAGrOsJ17j3Wx0AMNNdskntYWeQapc="_s,
        "zA3driX0hPOqNTo4qVkwxIB9v+iuWK4jm4iWbZ2T2U4="_s,
        "Gtki4lWfACbkloK6N5lnTjL3fwLNUgQt8F3nSwFLTDc="_s,
        "rrDxwf6PtpfSxzOlCM1z1Np3i2AAA79tcm6J7sCmDPY="_s,
        "2F2zK65VPisHLzYzAm8kqqiR0x6jsFIDxQdO9YJrDnc="_s,
        "rBplU2ewJkj+ghfuEdGycnhmBbeJif9hTLC+q19vVHw="_s,
        "Z8oavRB8HFh0iaBq3EHtMiGht3BIvkSaB2pek8k9K5g="_s,
        "mX9zfZQviw/DFZ4PVWZGTWFCLStp/IQSsTARGSpEwxw="_s,
        "R/wZmwFgq6CK7vBtaTy3ASniB82wAN3LeMY6JpzFlTU="_s,
        "3XQYuU4uohJa8llifZkATxil045zWfp3yhyJ51ILBQ8="_s,
        "zvXdZnvomqC50b0KLOt8AXx7GD+B0CE/j5m9OA8EJXc="_s,
        "qEmqSriiBRLC4HPtZs8UbFYVIgx02oXBCQnrLDPTMXo="_s,
        "5cHZ573q8zct7nJNF10lrKh57VKumv0Bj1A+nXTgm1A="_s,
        "wUGvMjBY8S+LC8dgFi+ZKPakFfoElAtIb9tAhihObs8="_s,
        "9q5iJoRkIt9Kv2S2WBTTzNNXCvXgNQmD/bAfvA3QVdY="_s,
        "kGSDmzUwjANOzyDwM16jczEyjZYrr/jlieadAfWlb/4="_s,
        "xWOtutxer7Zwi2ECaPvTk9Wa5B4iCq5arJnKLUWm4VE="_s,
        "etnKq/ZO410RNhL3uxDlMfsXq0MaUeCwiOk/ieGRVUk="_s,
        "FnXhZOXsiOMPuMkSz5NDwGwY/kGWY7QN4KO/kjcbHQc="_s,
        "QYVxAUySUMYZf8OsTo5dlteklxGl1jCMoNNrbzBFxhE="_s,
        "CKo6XuaKIdV3GnCyBJW22hwPmWxGmCzRsER60ttzDRE="_s,
        "yIXp8gF6LtcHXbnodtQKBKoyCBFEQ4A72xIKNK/TsdY="_s,
        "ynNMeSokwg/5Hsh4/RRrM1TUhZ9yj0gcEImj4v+i0w8="_s,
        "Oz+8WfKH8zo4cBdbGeuU18u1Rq8v26sp35G6g0NS0+o="_s,
        "bnBSW7QpBBxeyEqBz0czMDzukJZoCe0lV0H6UOEjrkc="_s,
        "WAjKjyfsPC/8qvN9mO4WJnUpi3qclRhSK57e1FuAd/U="_s,
        "UuxNgzz6UC8Qn8wZe0F3NhZfU9PQtOc6KAHI1QtkGAU="_s,
        "WMy98oM4TSJxHAmQpb554pCekIsRPUqAfggZBfy7xEM="_s,
        "4tRPyNw9lELX1Y7rUUU6/8fqkk4Jf4JxupcbQceAHyY="_s,
        "uPAIHwc2xoYMrU6Bzn+kxhDpmvUuCa/7nb0oGoECLaA="_s,
        "UrvZFpVrTti51x0XhOQAiyB4FOxQYgMyb7NgUvNFGts="_s,
        "lnOU0f0Qs4jZm+yd8qPKtUakD2lfXHBkHa8LUa9WBMY="_s,
        "LNTPiZW6yMKN01RkQfrm740ZoCzono9nut3iyhtYfqw="_s,
        "UvbXegBXJ/S5IFERnnbplWt+5xvywiOFgZr7GobSiqQ="_s,
        "v6afg9SlE7uwG2Nro3fswUIOJjrjK63TfrMwGeL2vzE="_s,
        "Ds2fW7SSvnHjqsyMpPFwyPmJAkKiXixuyaQvE1YbAo8="_s,
        "Jjx1qMq4vda2QvKd2/O7sLg69KZuRWBMk8u/lwmWtbM="_s,
        "YJD36R6Z7N2vxD+YYn0padVj6fR2oEal0r2d0WgwDfg="_s,
        "zR85+wuXzQMbnGbPPBHok7/7M3HcuBWFOhus1FW1Y9Q="_s,
        "kUp7oUt9TrT/O9NNSOJYg0vLlnZNSecQZU9fgJTp+fk="_s,
        "eRFCSteZMfSV2/yXeb9SN3Jh7SUKD/mgcX/vfPcNg7Y="_s,
        "SHRWKaJS+06NJ1BSfA1JNBwsF9X+W8ajfsgrBiroTJw="_s,
        "rlr07d3JsGDC71CcsJ6+GkaD5oljOBFDRoDfMZ/qUQc="_s,
        "XH35nfIyWGERkXCDqFqjG4LuKeSMopkOE/rgwGY6kj8="_s,
        "TQEw0xTxZpyepakR1AHWJQ+WOGpSsMOPez+0PNzRBYk="_s,
        "UtAtcw5l90oI55yjn62wYcFOWd4v+oMLzwXAMPz3AOk="_s,
        "NlFjtwbIzvX8bGmgoN38QLXHJtWpVfwqLOxQ3qzXeSg="_s,
        "Rzqa3CsAqiN6ZrggcG0cAlXvsP5oZHYAQv1Xa+l/2a4="_s,
        "adsnqZUiv+zKuNxIqJyXCY6f0r7j9Lju/c+bBiSi2+4="_s,
        "RSH34sH4EyVojdjm71us0f5P1oan/8h+xaE98MbaK5M="_s,
        "KJRMh3ppVtbp7pVydE6UXaiFXAsT0cisRfU8btaA6w0="_s,
        "zPKhPueef9TL11vIOPBw6cwJiX8RvCi3XDnKLy4gb2M="_s,
        "gsSotlvRZPog46HsoxLhWqV00qtONePhtjrlDEG9scw="_s,
        "KlPHOWjH9FPXmE/UEAc866NAL+W3xehLTnafQeJ16fE="_s,
        "zMEQXnERHLUiYHl614jEZDbqbr/OJ/hVJsZvIRgDNRg="_s,
        "8EFasLHXLycaYKvkplaTVxgITtilnYrZ3ajRk+HMbGg="_s,
        "A/KYbAgkxova8eFpRzUvX+QX0DdBgGCX+p6nEx/joBk="_s,
        "aidOdinA1x3PjKsedzNofr/jLixTtMqfrQULTx1UcfM="_s,
        "hqaHz1reaR4ZqgNyD/p8G6md0TfzbMYZgO7ER/IWBic="_s,
        "Tpan4Jjjb9STGUxk6XO6UjfaZXYMe3RiF+Y8gwMmqaw="_s,
        "DbMb77SDfFa/F26HmnFbXN9FdVP8foh3+XS0xu91sbE="_s,
        "12+06EF0ij9rxj76IxVuAmMcKDv0H4Tvy9rzOeo+G3M="_s,
        "QEx0bI9+P5t2EajyPZCMGjKlyXIja52Ju2iwXZv0uQU="_s,
        "5SP0nBKxOhdgFK8aRq5aBUWxNr9W1JZCaqiBAm0MMYo="_s,
        "ttt3U2PJzXRct4nWfDuz5HL1Xu56tG329HmObJy/SfQ="_s,
        "sVs0Zok2vraLZNvg6Brdmw26bKPt2m3EzBVu6ockVyA="_s,
        "TAYaMC06rYDl0Ep2CPIKJstsqZXz42v7ZVAOF1Ut6+s="_s,
        "XyO3L9HapBktzJv3Q+6bMPuXL+Z5eQRN3yFstvYBmQo="_s,
        "o3jiOhnEfP8+zqY/rTqeXs+DOQE0eMKPhpXL7LGbNVQ="_s,
        "pnjCQrNbAUxusmqCSloWAsAw2G2HH5b1k/MYrDhI1Sw="_s,
        "cLarDietbBEMvIkcyjqnwg1mtaGPeFXdTnIDGHloYpE="_s,
        "yikykcrgLowv54EcvOOX5VWCLDfZCl23PK9YuejHQL4="_s,
        "ALymqScbXhy7OWWnT0jBzgtyvL8IeQqiyrlfjcU2IVM="_s,
        "BtY2aXH5EVdDM11Pbv852UJDvwcdRKqZbEWuWDiL43Y="_s,
        "gP6Qy1WVOBWLwjX05TnZvK4gPhn6t8aXCq03sBVDSP8="_s,
        "l1cUxstwuhBb+ofSQV3y/d3kpGwdOrnQz0VGXlbLqX0="_s,
        "StiOajLbUaQc/xdBlwypWz5DP7+4viaccviBpC8riMY="_s,
        "f8iCldu4M3/PhMDU/sgV4I9ooi8ZNGuToEVc5EeWswc="_s,
        "UEvATZ9ImUT1gT/vRKnzbowVp/shGzJUQEbq79+Tv3s="_s,
        "A4mwYdsI1AZwTJu4gZ4Jw1WKyVYoez6dqOZkWnnVKOo="_s,
        "erijn0xIt0O9asu/XqTHVH78tirZZum+8+QZlq/ET9Q="_s,
        "huSWtTaya6YM22jfndkUOxmmO2XjDjc7AyGDOqsSldY="_s,
        "8p7vZus60ONXTY7bS55yoEaSoLK5LZnrfEuXdhHd1P4="_s,
        "NrUfvmuHWHVBuRc815wWxrNZrSKhcrkiIE+aqHQRrNk="_s,
        "hkyPcCpeYxmLt27wJAtZnMBl1JBMKvxdpsiim78NbGQ="_s,
        "6f35R8OfBvG15jxY7qLy90hQQhtOMgR9rLnHt13UKhY="_s,
        "6vDz0Mra0X7tEEXPaMMNt6v/tMjvTunLX0Am/e/Jmlk="_s,
        "O2DHfgyBwcnNya25at5tusfvK5QCoxYYWFXecSLlF9s="_s,
        "ajNdHOFS8rWwK/gsxEWwLRq8zX9AiocRNCS1+Py7+t4="_s,
        "eH12rW3qtnzPi6wbWEJgIF4RT1CPxVQrYS4/ddSaNOQ="_s,
        "jkVgwWx5cO+kdoBFCyzyOdSkgsBW0wis6hK7kCKQbIs="_s,
        "GvgW25pob6p2v72ned+VnmIT3nFKlLh6pzdKFR9vSQA="_s,
        "azR/8B5YiippCczX9/WGaydIQ5FUfp30UaubTCeSDHE="_s,
        "wUZaa4dDYi91mwim1TNuV+tuq9uhtjk/um0w1FOC89I="_s,
        "8sn6PPAuRd4FAG7TL1vLqreBuVJfRPOv0K7zcFFrsFQ="_s,
        "L8jK1/aHQWFLm/BvKnqHhsuDzjBqWOv1s7kD68NZisU="_s,
        "MlMrsQaPf8QzoyPMcy+ZW56REWyR3/oxaWYjUQ34JU4="_s,
        "HqN27llVSowvbbrt4o7ys6dhYSkcwF3kKMpg1+QIvvM="_s,
        "R765phNLsKkTyf34GwtxOZqzz/+ohi0/Ut6RlD3YBns="_s,
        "EFm9Y8gGiNdjEHvDc46nEaeQ7K8QJFP2KnENr/EOJW0="_s,
        "GmymOOWiCJ3GKY/2uhDud9JYL7aAsZWqgRbaj5Sadqg="_s,
        "GeYEENSY49KenhLMxgB9DuPa3q/1+04BSUslZ2+wu1k="_s,
        "oTLlCIWCSTF50SkKTVAAZRAFx/cpaWz9TgeFacjyq38="_s,
        "zBgJWbdOSZeDWyYed9h1eVE9o0D+lZUutrRNTZvyuPE="_s,
        "fTSfngilAza285hVToF+aSHdOQ752MrPMHSiTUN5vRA="_s,
        "HLq4NXjfMxGXKVyhXiC2g9Aki6bOJsiXPUIbMp6c5W8="_s,
        "hopwo+scpo74++snKaswtQZpnXmLwYmmsDEa73WBPMM="_s,
        "J4tvnwEb6b11k9UhvXnzgg6K1Wv7NlpOLJqyhgdfvr4="_s,
        "mGNduWD5/sXbAkJvQrtEEeU9P1UfeV9CcTfZkizQ4Js="_s,
        "HtbiGSuTO/FDT5Xrrio0a2+Ay9dEF2ZXRJ3/xu+sX00="_s,
        "PtVGZrK8ytWgRMFmpzzdHXvKtgVb63XdLbj5nHZVcQc="_s,
        "CT94ImDXD5FOflkeuhUC0+mS6femmhfR32wyvN3W8Go="_s,
        "aQPebwA/WibX0MlKJPcxebK1AiJnbcs5bbyFk+4+1Ss="_s,
        "FgM72aPMXOXUcGCtZbjNm4mcziSoVz6NlaRhY10fz/Y="_s,
        "UypIf0hi3ap95vbh95sD/yWsXyrYq2g6nt+gq4X4Fik="_s,
        "xRmTYk8tza+Ne5EiHln4bivv2g3z0TkRwMgEZ/O9KTw="_s,
        "qRJ9VqGCpQMz0XrHnjVnU4qpL7aYN0gb8T935QwkP1c="_s,
        "vkOfPr0fzQZqIBTbKNdAeGQbP3HfGOccG4QrpU7Ygyc="_s,
        "QsclX5JGAmk7M86hkNBkrkrj7Ipe67b/0XPOdH/yrlc="_s,
        "LDCXI31g9C6ADr5ACcmvFEuxnlWB4cBQHHslnu5+IQw="_s,
        "CJYwJEYA8zIwAQ9eBMZ0GexkLFIoVArbQuP+ksYx5r8="_s,
        "BFsY0Q8ZJ1XnlLZKvQoSHteOH+8IG6Bb8BJ+hp9ubOk="_s,
        "hVxo47n4XpWE3SRCgbPxM35es6bAT8szuMsrITbSWKE="_s,
        "3MaQh/u4EdLxl/VIT+eh1D1ce6ZKlkYYvx9m/giMRZU="_s,
        "SiuFTNuaBkYrvT7+pj6cvX8eR7foSmBtkG9GoffcDYg="_s,
        "DVVcArLzx8aUafsxf2YxfwM3h7MkueJSUPdFytlSI0o="_s,
        "0E0dxtguDzGBWwsEW/OWvEebnRZp5m2HL2MsyFJzISA="_s,
        "1JWpp4+ub0kburmYg8uI2gLQDo9yO2FhrnTp5ai6GAM="_s,
        "dfw0Ip/LdWGtMoB0gHlM2jN71pHTI1A6Ix4pkbEOX68="_s,
        "dLqwyvLEdhkXvuBI7iWFbBKoiw4Db9DHhCf5pmilCfI="_s,
        "1TK1XkhhPIhC9ucM3JGthJiYFd7HIoExxuS3dqbGRAQ="_s,
        "LtZrsVjQuDXnenGF9J2FIY+RYodBNpPXib8KyLD1pHI="_s,
        "BiHTYzBgrrQefW4GZYM82sf07Iv140cNXmDd5oa1EKM="_s,
        "Gz4oYFQJeIu+l56SUnSh8Zcm+EpzdS2cWjWrAHdYBQc="_s,
        "jYb65SZdl/uZ5AEIEod27xN6DwXkFHiVggrdc8JsBbE="_s,
        "u4R4SJDQ371vCcDbK/EXJeTHBS5B98UJQKyIf4R0e4M="_s,
        "U53QFWVQSvJ097Wb5ZM3zdYCtkJDCds/1UEq/Nj5ymE="_s,
        "ao6c1FNw5HMgc2PV8oMZzFRm1z4BJtrWRYyyS1DZhBE="_s,
        "GSIIn/pTRaVbbfcJTB0Jl3B3NbUiySpyp14R2C5/W/k="_s,
        "O+0zy9w9AVQKzUwKrNobU5Q7XZSgnGs0GqZDOVoZMF8="_s,
        "1d03LtF05+YOQFeNJsRVBSehYfWYAU9gCawDOGM4zMY="_s,
        "G0yXooCc21MVMTlUTh9ds05JF8jwHS3ZTLlRniThqzw="_s,
        "YweJ7JrbaNcS/gZvyQF//iOpQ25ML74MZ7y6PxDEZU4="_s,
        "NAIvuhIYsiHQoTzltFDVU4pBQP9d5H/49hNATygznlI="_s,
        "cT4S6xeQzatIuKzd5R5753rq4dURxK+8XrJt8YyeF/0="_s,
        "ufQIKCsbo30k65vOjrXsuQqmgCq/KoQlsJUsZdOr7HU="_s,
        "4yyEb59206KknzMPZn6u5eQ7crcXLm8MDnzzeIUn7Ac="_s,
        "sdFI/iE3rkci3BYWqkm2xKXJoSpVyunjdmnM9k5pNYo="_s,
        "GIiqn7bH5onB1A61OFecp+O/1t1lRBsJiy/e88KMptI="_s,
        "y7yn2YiLSp6rfUeXVtKST5sGf9ONqzdnlwKd90H5buQ="_s,
        "Hd07e2ipbaApefly5Omotq9jtaF8ddfH4OOQHZ86cpw="_s,
        "0P8OR7FcErBbkG84ks01X8Ycr/eOzV5jZ0BBnHQ1gx0="_s,
        "WGTnYXYJxhc7C/ImWN/NnWnvdVJs0dM/4Qtb8FvoH50="_s,
        "G4d6In2zX70zPUVUho6avrgukjf6iKB4LDNLcxVmdG0="_s,
        "qo0+neZDHX66D7jP657cermM26tvR7iA7G+CuA0iwOI="_s,
        "WZFzBkyhh5K4KQ9pk1m9r8FGAE0c6Dpn1DFrx5pj2vM="_s,
        "okYMDhIsPUXJ7bB3MMgGdPMX3bo2TDevN3Wxm8efsiM="_s,
        "8zyrK32qBKEpWbweztwwU0b14Y7gxsV7rC8CUDHROJE="_s,
        "BtGwoUA8KqjHoVx5HYuRa26WjobHjWcx3DH5GJTxRRU="_s,
        "OF54cqRyIlQKcb0yZGKlU5vGfoNETE0WV/92+l5kt0E="_s,
        "qef7peQKT2aYCEEJYm8cx/sbvEfDIrE9piXmSev46dw="_s,
        "NNnr6WLPzcMvvmugZ+DCTIsIeWy+F0PoY+PFkFxnVG0="_s,
        "d3XrPLqCA1mvHgPgX/8nBurS1m4byTiYDRdluVI6Lok="_s,
        "fuanwXR+qyK2FGlF+dJ/ysZm+bhnsmjA2rDNPHR6ep0="_s,
        "40jXckgPjA5fpUazxTGjhwCuFrXa1d77Xmet5/bTMug="_s,
        "/oVkavIiUAqGb9Y77ttq4AV2xK+rTg0osV2dbZLLfaU="_s,
        "jFmS66FaXVgkYGXKqKkWyQaZFnOXt3VxMCT4oHbZo9M="_s,
        "1cI9uAshJqhR/4CPsnKnvVTbPG29jDwrFfbQN4Abdwg="_s,
        "6Z+NhF64yImHLEpFi8hUAZjqhmwvQ/GXMW0PzzPM7Fc="_s,
        "r4SZUuiBuSz3IGVAqRxf7v4j/1kQr4MJfLsur/YUuiA="_s,
        "QNo8rzBIxO/4oaV4DLuJxd7D55coULqdw98L6wchugU="_s,
        "JFgYsi0exIkvy3IkN+MoiOl/Y6Axa9Iqr59EzeAfTJE="_s,
        "XET52GjFWRPKWH4Dmfxt8Y8VfFRKCzZq5USoPTLwV9U="_s,
        "BPBzGq2T7J7Fv6moVf7hWBIgfkvjCrHANcwhhkeK02k="_s,
        "L6Nthp214NinUr1KOx2zXSVlXheaMd5roz5FaAF+Zvk="_s,
        "ODsnYVFywVDzy4WpfUgl1F6qi0QreB6eg1Hn6DS4cG0="_s,
        "9jUkWbQwy9NW0b9edvo8sOADSIY97n/vs40QWMqumvs="_s,
        "xwTZ5ne3KFnSnd1zoqY+OxW1pNuQPP9ks8KaBZ4YCHE="_s,
        "peFYCswdP70tIEWyqFvbgpHEdZOenLSAA6/FAkmbPy4="_s,
        "z+O3OC5HcFnaEb4gmZFLlPDipPCCQMYFQsN2lXuNllg="_s,
        "HIfSsm3n1VxmA3kWu7TLpseR2g4q36N4MyZ4/xPhLZ0="_s,
        "4BAp8yePxicM0xpf51PL2mlPu3d+4w6PahOQkIcXiYg="_s,
        "B1BGgPiGZZsPDYU5rlwBMkuI8wOHYJaHwvKgBQkovd8="_s,
        "xPcnFCIQCPW9/MkIIczYNczpUkx+yZrV2UdIiTD+FiM="_s,
        "mDv623RRSczrnZAWSWslCpzlJk9b4Kq2T4+juU3uU0A="_s,
        "hWuKM3c3FKrEKqcKihXL3tCVc9Fd19FWKusFIFf73zY="_s,
        "paUSqhxWNQFudAQv0AXoF7K4ew3n/ErKP11GC2F8/kA="_s,
        "zn3bkPF2ShB/bwELfo7dLk0VEOuBbyRwSv/iA2mfey8="_s,
        "J3Il3FpeemdGMwW8B6TiRrf4z+k3IE/jUIYZKMWzd3Y="_s,
        "YRRQDjCKlMnZKlYOFISdPZ8Cu3hAnpcAoXhzdX43EjU="_s,
        "dHj/R6Oe2NkCdM0ydA2PIFqKj+Ykwtlbc1B4dbm1Qc8="_s,
        "u+eqvv4TbHFVtU83l2SM49CPFSF2Zs+EsFft+HlI7tw="_s,
        "F9Y1Caw6tKY30syrhf5HEfwe739ZgJ9oXJP6p02seo4="_s,
        "5UR/zuEdXOjOPrbCzL3IsMbSDOjHyzniuf8t5BXBZac="_s,
        "niS3wd2bXvVGGivvTkuRkmJSyS6zCbmc6YKDk0oIOh0="_s,
        "kNWf226QHYsQDXsatFlt9BNPbNLmKT+o3eq298chBUQ="_s,
        "NnhuvZEyYys4qENvrm+DbMVbqFwzL3Q3qWZPbBjvpRA="_s,
        "zHCSPAfZPpG83Owx6D8yO7YR9xr92ZAbBevFmSaTcRg="_s,
        "eOXrMUaWC1QR0fTcIV1jGjC2cGbUvB/at/NJXxB/7kE="_s,
        "muLjvBVF7l/R9y4PTQZW+AVNHq0uUcKPjwcw+F9oi+0="_s,
        "QuRL4c/Xv6swMVRuH2VkGOnINSMKL2m3o0WUgV0OCoA="_s,
        "o21Pg63SF2o8YkO9V6MqOHoZBiJ+KGaJUw/4eN4ZlOs="_s,
        "9dAL/URXwDYB8o4gDtXbblv1izMhZKHmMPpqr8+ra8s="_s,
        "mBx7OJu9neFW9rkrtPZfTmbTKbnTUDJ31P5u7xAI8/Q="_s,
        "R3OtuwgMUYnVLjHYNli22XQ733M35Tu4pHBt6NwRb/0="_s,
        "dajSHUBfwCkiv05ZEMAtqMaeFCLFTqzgFpAvjRPyTMM="_s,
        "sPPs4yDkD2xVsZN44SPxmKPvXy6PvZq3kGzgQZh3D98="_s,
        "b0P/ny+5jMZeGPc+4WlRus+wVfduaOBvfZGYn9dw+nE="_s,
        "hyGlI4SPXVg8VUVS1trqqs78jrkjm6DJmKCUkmZ/XHU="_s,
        "YlVCd9B7IMa/rnxiZ7MZi0hG9gSjfUCFv59Uw5IhC1Y="_s,
        "FVlj4/sG8TARcFxlvFLtMfmYm4cwnJRZSKdrNfjn2mc="_s,
        "DZMX1VWdQJFRbB4kBolYkSKtmxAfAw73rujAGp/Fp4w="_s,
        "NF3vluu8AdeeZ3hIQgqUygZTlhwDswl3cU+mI5VVNWk="_s,
        "5z1z9nsndWirAdVjItGgHWZAmo+UdzXdc4/i3La7DFg="_s,
        "+0BJO47vFkZGSarGG5ZdU9lVMEGOieroO/NCCEL4t8I="_s,
        "6s5lxk+lctuw+ChaAZDNQHUsi2xgX12OkCmOcwchfNA="_s,
        "nai+Ll3vY8d/U+1mAwYJjeviHWIseXVhgKSmJrohxq8="_s,
        "4FXZyE3LarhoTyfnrGtICCtia3xTY5UC/QW/Rc1Dr+M="_s,
        "P3Oj0gvwowOOa7L2Cih1+wKIf1rq47fuNf8ad2K9AFc="_s,
        "clE0MBPcVROB3jK0CYwLOEo/LkglA7LBYcyunA/dVT0="_s,
        "beUn+y/EhpSamUn+Qg9OBQ6dNxPPIeG7B7bG7JreJ4E="_s,
        "PlZEZzOPsduFlv5QL90dTamDv11+GFrIz14PzmeemsA="_s,
        "HbQsfGREm8IX3JawKyCgcoA4sUbU+N1h2rsn7UgPkxI="_s,
        "2jXQ7Y5SChG9CyPTBM256GbvvL/tB/+dk3jXrfnXTw8="_s,
        "D+akNXUFyw08qLoGca1X32t0EMoCy4Bl7tWOLAOB5kA="_s,
        "V8lv1ClGF/sL84QtH3fsI2X/DQ0AtoF1CLYZLfDowWk="_s,
        "VpS/lk+KpkwG9Z5f479hfvnMGVmfPJAbZ/BiYURuG7Y="_s,
        "Xt9GxZHUcBonIo+P7Y3Rx12J+jQpkxmkKTHpmRcBQd0="_s,
        "k5yg5m6eHClmr6p8z6XqLYKUBDzTJCTthKO/si07AHE="_s,
        "y3+B9UL1xBiju7mtP5++eEFR0TsEzsUOzt7GATMko9o="_s,
        "ro/QKnu4dHn+RqVB4upa9+NCOT6O+I01HNrxDOcc5gE="_s,
        "TSiDRDsk5CRSf2oKeqKJez33HyOdtANzxP92DkgUeAE="_s,
        "i04bhH4iIz1PRn00+u/nvL/rzm+pu77lYMRc2JSGh1E="_s,
        "ZRmhWwgpS+OV7ZFMQCNrpgQhTLEWO1MRqYdfHfXu2G0="_s,
        "S8NZwnBXs+0LrKZtbFNJpdSmBQlvvHh1Bhui2Ro9bQU="_s,
        "0CPFX94iBJXxOoe3KZXg35jI0ok1dymdGZ2lu/W0/gI="_s,
        "KCpk+2AM2sb/eQvQH5LdZx0QqYJ4zM9S+yJHmhiraTQ="_s,
        "lbakhA+HEeyrQnvCNuuGCY235ceCuvsTnIwwgFql/+E="_s,
        "Zxchcur85FY3JeZhBD1duFS4Z2ip6L05dY2+gcR5aTU="_s,
        "Q0RmtZVFqKHKxt2zgZfNxrNZlamMPzgS+4jWGxwwDdM="_s,
        "59TVNAu+V6Adj3mSFC4nY9Q41Xg4kMdnSDBu6/oFamk="_s,
        "n6/Ww+e/wUXbQqkG5fpo/mpvlyIfAT0+gz/DIxuZLEQ="_s,
        "QuNp3Iw3KXO6d6gmti0seiBwUyMT34MqIyiN23aXQVg="_s,
        "RTf6BjQO5iwmTp16TYqryrPee9osXMM0AQ9GtoMQb8s="_s,
        "TS1ee8CAPpd+JNpovgMX+ZxPJqjHjQ9LpHeiJ+f6bnk="_s,
        "UuNiTmAxbeZOgIrZYHs56e+xrCNGNHjw0w6mTtw2m2k="_s,
        "G1la8i7/RFObPcRpVVV+jEfG6lNcYqaRBvOrmzCfqz4="_s,
        "57fnrlMumWr3aUeTBPTb9uXfPTOWamzHRLDXqo9Yh/I="_s,
        "RR18dKhp57RNx5y+BIye0K3IZfriUKbLPwsQ5oCqR0w="_s,
        "jajA2DDgLkDq0XmaHHdJaX6dRkkcrQuZKmWCM8GPdHM="_s,
        "qh8/oNEETkLQ2SUqQ1oCxC2v8h+tFkDN+VPp+tm6zqs="_s,
        "FQdMPV52GrCbvPEGtzojkZB5qWsuzqEfV38p5bESBFU="_s,
        "NLVgzFlR1BLjtmuJscGMLrpp9o1jG3iPuAqzq/A9xkg="_s,
        "k2ooAROk6Ya0t6oY6rsQh31z3U3xlGujAdf98DzAMWk="_s,
        "sETBJpsnqqmepnE8OP5Ne7kQ2CcSj968VagkV/0GIWQ="_s,
        "a4GaBsPbKl5He19Tza6v7LWSPOyVb2aQXMAPu4d5azg="_s,
        "+nCbh6KDHG7SRxBqn3JMT234YmiAcBX7AvUVspkjBjU="_s,
        "v0XQPC+o7hMKfhf1udInNRloJe7BKFNpsLA9dWgDjCk="_s,
        "LgVhCi7HaunpDaM9GztF/ymLv/cKqOfMsPhaR996tPE="_s,
        "fPJbHUWsRDa1dehtyv3Tfvxfrqs72tMUSprzt+rktI8="_s,
        "deZKlSYT5/bNx0Yo+L+FIG25p1zBkm/8y48cqt4uTbc="_s,
        "srtwjXj5znSQJR9nb4+Pba3H6Nfks9GrVgpMETCwxGA="_s,
        "MbtKoAtSk0dMPXTmejFkQqjeBKUzHts+nhE0m3NV/Ow="_s,
        "Ff6ASJv5XpS6Ir5AhKEdz1/zxjH6je0gUcD5nwiJX1I="_s,
        "5NVrpvzXuZ5+g+4Tqizsv2ORvJll0Y9DoTvsKVfs2FE="_s,
        "ucNF9PoTSa6hF+11NJJ4OmMlE/LdddyGlCz7xjDITQU="_s,
        "VJl9rNGdza1L/92ogMmqkmjvUigas5+w0Xt7p2GKSaY="_s,
        "4gq03okT0eVpwwFsXFrs/WWJ8r/yEcOH8tBCLZEjri8="_s,
        "wl0e5xX2ymkgJAQWWqCUJwn/FNG9mfnfWxJFItQgd+8="_s,
        "CPeHT4M2tH5J2XGcOM6hbN6mNili9QAds/LQu0czI1c="_s,
        "muIybDid28k6JjaxIUVjMxUpMVSb1b0Wpc0u4k5gHBY="_s,
        "olR5vaxhjGtPMqP2C+3ua0/nWdsVAMeQ6rX5T961n5I="_s,
        "jFB4hilJkkKamOk4LdFVfvHn1KRdanu/+pdx6DkZDCM="_s,
        "M6hlegmvA4UAczva+WnYkotIR18mAgzDXKcZOlP8vtw="_s,
        "elj3Qf9TmvlHmP9WHJGOWEHX5hZOkMvle+/exKFvak4="_s,
        "k0Uw2GS28D+tjoe2ba+CnRfw56+9dLrq0ypqQVfVhMI="_s,
        "zPTbHutoyW4F50+Ov6dcxgw6D+2GLa5rCthdThtbTk8="_s,
        "4VPv4xfnMeWeuNbRpWioEbj33V3zgjwycQO+jkoAR+Q="_s,
        "4oHyolWYv4tAp388Uii1GQB4giTpHa3j9KD4w4NQX0M="_s,
        "KDW2P/yJLQGqwsw0aWm1AchF2RhPc4WJd57BdpN+HzM="_s,
        "vU4h2OJaiJ6XmsI2d3DDaGNEO44Pmf7q7gmmJiW5yxM="_s,
        "1bISUQ1ytmvbf+sixhFFZiHD3RouiMxCmD19TJq3lWM="_s,
        "IEZK67/1TMF3dkl85BEsM3SlSzj3ul9Y7sEhdBSdZ0I="_s,
        "ESSR3Mype/UOx63/jUMNy6bQtx4KCyA4b4FPO/W7lLU="_s,
        "8YxIaoAXXPAv7g4FwrSs2GwEzbrs7GHB75H5IFCbXv4="_s,
        "3GTXGS+ESXysrVwQrvaCViwkqmEkJw+F/iR+IjYH8+0="_s,
        "F86LnWEol9H+/VzSCW29g7gtBd1dH2BCGsoVxM4kRaw="_s,
        "cHJvmODZj1Q9X94Fo6dLsTHFxn9fv/z01Bt/aLyitWs="_s,
        "SqRKB7cfRwZ4csjxd/Mbq8DGCa8rOEURNTyJscqARFg="_s,
        "7vWI4V8izhG8FerdZ6iyDX6KcSJ0MzY+ZIJ6tTlnmFc="_s,
        "Alj2JeOopci6dGJyaFiX8iXOYaQRXtqRmqj1qvOBeH0="_s,
        "Zjpop2IhUNssl9fcBT463zRstKgK8NkOs2WFHxpdEx8="_s,
        "8go99U+HI7FGWa8teiL0HriJS63oWsNPN3QgvB8jj9c="_s,
        "mKuBXN+DrMR5r/oGhGYiQUaUXuv+GElin0WzSuDqvqs="_s,
        "/0hDnJ0dJvmpwWRaCD0bX6l1PJ5/3FGYGO6TFsaZzDU="_s,
        "d3U1+YZKvzwLQ7CNJoEhjHUX/mGzeUqAjZzV01NUVCk="_s,
        "EviwYjipb+ibSJsIMKdxWj4bCYEzUOGU4DqlQ//Bnqo="_s,
        "TBI62f1oNkAlRcvFIGmKhz2dGN25k114bYZvXxpcHII="_s,
        "ms8kSteIo7hQJEKBkwJ7cHY3IM/GWi2PB7BU2+UcyEs="_s,
        "HtjLXGyiw7fW884fJ9xXtj7svR1xPT6Mnqb4lZYWvZY="_s,
        "Cag/WW6bhMyr7skju936D3WI/1aDgMxxbf5IevPmz+Q="_s,
        "gq8EeqLdY+fS8Fso9d7Rvg5IUjQ5D62Rs8pRAcYyx4M="_s,
        "BtMkgM+ckL/HUWmkJbxS5RosK8IMx19sbwmxheQ8coo="_s,
        "1LV2o/GsTZP8XFhRUHvwoqnVM0lKzYVa+dCh3oc19ss="_s,
        "YsR3jEhsLzTmgeSt9Pm3xXrizTvNE5ECsKJZzbpZbFU="_s,
        "xHxWN9L3DnWPPb0OgWJWa2JTQgFIoBO5QRjHSVqHdkw="_s,
        "EkCumJ6K85vHBJailvTVu42bsiKB+hnA2FblwDhFm+k="_s,
        "corJjAl4wCWSMvfocPiV916vk/1B6x7KNPqtsZJHhW0="_s,
        "C7OrAUv6o2keRXg4pXpjW0ONCbAaj2HIg829kHUlx9M="_s,
        "NjcqSjUg9FKSd91NS/2Sw1QbokoUr5kO3kD4m6aIoUw="_s,
        "HimJvi/Jsx1TLKLF4VwRYHM+3d10SIXb7Gnp6bxeY+Y="_s,
        "Qe2vKKzqmHYPdHFMepYQblLe4cw/SUmMTHzuzmK99tI="_s,
        "Pj8K62X461DHdxVvOo+7nbAfulVos4JUcB5FsvbvHDM="_s,
        "G8HX052aibA5MCIv0TOb9EUvnoiZZoqNdm+3kicsS+Y="_s,
        "C2bRfJ6SzspJBreUSwzhQQ+V+4wFaqKSYbR4UEYIsyk="_s,
        "+92tUGwU1QVE3WjkgOgf0fVBE+KxD4dun4XyeiuHKV8="_s,
        "KAO60Mi6Rd9116PeBTM93dGDCbG9GMhtjwFAss2hY00="_s,
        "CFMvBef0gC2Z1Hoo7KcyC/q4IEr/UcG++PPUf29hivM="_s,
        "yTOKG+QrI42GYYMoEZC3I3FhxviYn/zXbl70sE9NJaI="_s,
        "r2OXUD/O+9YTl2whrVweNymMGLvgfQltsDzNOvbgW6g="_s,
        "vgoISWLYBmiE9/6b0n7BblH1qTtypQLJLFok3IfrLrw="_s,
        "5CqIRERIrD1gVJzHwf8sipyschA0wHPYChSkTnlzDMo="_s,
        "NzwE/SQY9cd+6knVFHMQWPGQepT/O05dfD5XZ+i1PYs="_s,
        "o7mBd4AhTK8B6K7CC83CMFof80oV+ugezQkj35zVzQo="_s,
        "y56eaTGSQTzeKx8hwdwdRLb+eyfMK0WOizWdGPn/j04="_s,
        "5CmUZOewEpaO7WOsLbHJUJ9WvKQJ759x8pJqjDyAsqk="_s,
        "lnTrG9VQRxeQODcJOmdmjqiPLtAG2RNn0NS3qh+SEfw="_s,
        "P200iM9lN09vZ2wxU0CwrCvoMr1VJAyAlEjjbvm5YyY="_s,
        "6HEbu4ca/Y6d6mDhbTDwDH5IN7vJgHBlAXR1uEn6IxM="_s,
        "mDS4KtJuKjdYPSJnahLdLrD+fIA1aiEU0NsaqLOJlTc="_s,
        "jqh5F1SRWomKMQDmPjKXim0XY75t+Oc6OdOpDWkc3u8="_s,
        "+SF/ZodLDAHNjBC2opXbxPYJrLb1rcQcN9pGZBtX6wI="_s,
        "zaWdbv/6aFgw/ZW1X2SunLUSec00skELafhMfsMBV9k="_s,
        "edCIBkvrOCYFT7iBZUFiNYl6hWypUvyhSYscWbFqqkg="_s,
        "Ix1ozvvGhG+3jK/KhGdAGk83HiJkGGYuiQDj1SrcAKs="_s,
        "hl/SPeE7GSAoDhgjWVeeCe/3oJ6yRnbs7sdVKDhVn1U="_s,
        "aoyOnh5/aSwhrxlW3hY/PQJneOZEn+k6CaZxhHyhrmU="_s,
        "MlAXJ7sj/HdhWx7Ha18pjsIhmMDz1qPn1v5Kw88xXbk="_s,
        "AjQLoXidCbzFB1Y8icSoR7pTe7gTIIJ3elBRnPdLy1s="_s,
        "ji2WGYcRtjdS5fL1mHuP/FNt9hVqBlvb4oy3IxyteT0="_s,
        "gR4VdxhAZhCM7KhtWmJKkBIwPfUgbFywsn1eP+3LpaM="_s,
        "52hTmAh1f37sIPCcnumzaw5sVjbpDJ7ml5lcu8buBVU="_s,
        "NS+OrSgEGd2VPSvHhDOGu5Qw6BUsLb0sLSSU9rlWfuc="_s,
        "8yuUopBg3+wMeXod/hETV+sdaC8aykrEuVzw9MPiv6k="_s,
        "f00/0KcF2/hAMpiq2R1d5pcua11TYGjrqLJJVKWgqMc="_s,
        "XZGQKSrN1Iug/DUID350SPPN8NeRmaTSPw9JtTQf3yk="_s,
        "6DbF1d/BPhgXDFvd77T9rHxlrArUOktDGs3EqYhlomA="_s,
        "oMFFOCrP5Zdn0Uh9ZQXzSWZbaFwohc2RtiHRQaKSsPg="_s,
        "AfRBb121ni3WtvvZ3DIzbZnbGPfrYjpJ9YTQSv0nlHM="_s,
        "IGVtGo8upE42wrg1TRXE2yGQnuUUCwIk90+SR34Imfc="_s,
        "lTyCzPVu4aKS5A2KcE4ZLl2fQfGqXPN9D+RqQoGrl38="_s,
        "D8CiLl5nyV0Cw4mhRUrMZ99T4vakavc58+rH41JkR1E="_s,
        "XvTHvpV33eAEBIYHzIciId4A24k/KbqoCcN4sBNwo3A="_s,
        "7Vv76kI3jFiglalqQX8GeAi091OJK7LUSaMb1LMIhM4="_s,
        "Q6iBFhWF2xYXn3DlMkCidPIJr/A6r7zDS8MuF/tNlcY="_s,
        "YPstKNLx29K8Gno7RHAf7By39jC+5DnZJsAkwyEsGpw="_s,
        "hsfYnQ8NKdhcZoTbLY2pqsUUyBvKv0HqCvcm4p3iCkc="_s,
        "fdX8329LMwv4KWWIfva5GWuNJ4Ve3e6ZrAT7Y94ONR4="_s,
        "kTqUop2DhxLP7ZNwKKxKsU6sld3HhNUgfk1FBKtC+hc="_s,
        "PRCAYl0wMOiDV7OsmqN33Owj8bUpxK0D96mkNcyuBL4="_s,
        "08rxJZHRlHEvrNELyhTwqSTttZwkRHo/2ZSkgobbiEM="_s,
        "I9mkWFkE3uyTu+I7kR2X9A/iW832Exc38Xsfh8S2g2c="_s,
        "ynhbOg0PTBvQy74pipia8or/MIa2Uiwur598EQ8ICHQ="_s,
        "sRXT/+79DDonb6lk46iuuPsE94L2kMR2wQQrBupGXNE="_s,
        "nsNWsg9YMOD89lutgbZsOMF68HgN4Uoi+WuJQY9RAtg="_s,
        "Kt/yIY7MrDfTceYHHzIrnfKDl55iT9yzdJsMYJRecBc="_s,
        "3l+0DgaJ8VSQIhNSd4HC+4PN3L+PYioThPHip+dpOyE="_s,
        "hiUh/eQfCrSbn8RLa3rvy3wzdZ85Wf6a2+1OO02bWLg="_s,
        "uHRaalMIQe+7r6mlxbFhSHeLrb8QkZXFhNQFj2LyBu0="_s,
        "9gKdkkMlXhrlgBpoouD6H22eBFxPl+BCSHJ7ZcFi7Dw="_s,
        "fxAxAfBrbvlpWXdVC6mHAybczWd1YXsWdCZJSN+/NeI="_s,
        "fO+qlfIzgKxv+ftEIwk9DSlU6LJXgsQoZOmSgciCw5U="_s,
        "CjNrYcv7uasJ84DkK+W0Qnhyfrw18V9kDpT8eoYlaVQ="_s,
        "V0DAzSTV2qST/mOhFp8RyYRdZwnFlun+SrkuANSTLF0="_s,
        "C/G42KwbTvDK6g24y+G2o1+KhKL1//okIZNswRoakfw="_s,
        "6Yrj/5NrRyPNiiN30sVJpmfOh+gSAexJlcwBvTdMEog="_s,
        "40W3yS2agdU5ww3CtbXaiYCE56wv9jNkp++9PRZ4pbo="_s,
        "Xkg7MwKhOaGPfctgB+5HIt2aVK2VV/WshoL3QTcr+HE="_s,
        "ZoQGswgTWuLoZ9L0vRqTSp5dcvU2GsPwYkaCfsO5qx8="_s,
        "MO91GkmqVfdpdnWdjCq0up4oobfPciZySCf9dd0g0zw="_s,
        "RFGJ3iJInAalSbdcj46VzFZjnUEozXLnaJa00qfEDOI="_s,
        "VkjAOAeaL65c61yzhF7hle4p94BlOy8Qcqzd2LXAKDU="_s,
        "V965zN5tSVZKkWzFinmdjr15PHr/aafzzOSMv7DEh3c="_s,
        "7Dchd7jo3znXVeFlUd+73cxTk4ylJ2X9cw0JJYhclk4="_s,
        "ZcTXUl92RD+hsftyhEotDLh8KD5y6tqM6oWcO8T8CfY="_s,
        "PQavHzHNg6znomWgFLj7Xe4Vdw7KyPelVVUZDmJ+A8I="_s,
        "IyaEm2w9+iF6Dp7K7JVTyRCk4ph8hzFchQc9K5WkhPM="_s,
        "ivxuXoQrqrFgEMLOb89I7E3tjhV5o3wfG8An4SDQSVE="_s,
        "dadhFZriZsUzKk8mbgelVDcS/7du4CYLB3ghlcBNw2Q="_s,
        "Z5taUha735E8wi1q5Ed4we+XXuOHpsTF3ofnXRmiIjI="_s,
        "FMnbT/h/3gj2ewpp3VlLq22HF0gSoNvTTFmDO/7YzA4="_s,
        "QHqXI/xxfJTih0lggNdz4Y4pw8rEniYwFyNDxlwIZKg="_s,
        "+kmPsllqUjXFyG17aKf7526YVcAa9LDlq0HEEEfGSOA="_s,
        "dO3Bi2fEh+MvGBcZ/bNH4udwIHRGUfRG6azXvWgh4uc="_s,
        "xlG4pn0xkyBvYiw8Ow+8pKLycnEIxCErUsHiouhMmzE="_s,
        "0LQlar7XJIFYVmKXEmLqvuNFwZ+DevANfOJCOdO0Du8="_s,
        "Q8BywWye5tZ6zfpsbWaF/x5060I3t8w8E0irHBCLJq8="_s,
        "r/duXJhvKV1LxvgUKnjioxiIsQHC0CXbifecdfZP2Qs="_s,
        "BvTQCSPqJGl99d8LkphBdZkdi9JXdqAtUxu0AeOT7EI="_s,
        "z4P/uM8AI71Dnf3V0C8ZVLbuAn6FiX1s/F+Qu8qewdI="_s,
        "kxPecwS+9FL/Wsk+D5ykjDlx0vu1AsP6Hiwuofka2FM="_s,
        "4U3gQEvL4orEE71MmS9VKvJQ2jCCPmNauadlblDne8g="_s,
        "ahbAQim8K02iJuuX5o2U9JumQ3t7XhbBShAbIaKThOk="_s,
        "1TQq5nnXBk1HVAPOX+BdgPGgq+WIH3Ke4OMnKpw1Xzs="_s,
        "he7hFWLKgzDdbj57cmMJq5URDgYtcsdZq8g/iXuO0HQ="_s,
        "aEFX5A9+4kKXB6I9W3fSxxxdI/JqnYKDknX/mE0TBTs="_s,
        "kx0XM5C5AbhLEzeWouQuVLgqTWD8fwuqI+BXQq+Xk+g="_s,
        "TuGxcP9Rt/2hGs71UdHvSSwOvBr/jlcS0+MPy7g1d20="_s,
        "wUUM3f2YZo5uWT4tJvEbTEP6rMtfAzsLUOpvHOe/S1g="_s,
        "Ti/B04KLk+fpQ8gBelQrCtAPYzPSh0hli7BpfeI/1S4="_s,
        "zTmL4akYFxJs7xAiRzjmJDWO328IBDq61+YMGq7MyNA="_s,
        "H0m4cGVHaC4sXtZkKi8ty9KH2kWDFLlnxg13SqfttHM="_s,
        "73I08enTA9LkKiCcVHwhe5CvpJ1gUC0E8bFMkG78AtE="_s,
        "13RtKWbjAyMmap9+okvoCUD3oCB6C5Hd4WafOzvezTM="_s,
        "2gXl7nx1yWXvsVGm6Hq1WJuAI/JG9pjB0UlGQUvDHJA="_s,
        "gwJ7EEFiFqIKDXUiweNnoWlI9FDArGtMDVr3sFq58gc="_s,
        "FlEBVC2LgznkUVVsGZ1RfNyIgzU3/R7+zOf1RpvOP+Y="_s,
        "s7jCHt/mxeQC/cYHNm/Y0VlJplkU9YE0cz3GiSK8jWE="_s,
        "f6Co0v+c9WNnrz2pJhF+jGSddB7CYBzgFimJhHcalUw="_s,
        "hZSsfqI1BeLFCWZZxvZCxhDS7xJti6cYxCBnzMYzjDI="_s,
        "WcGIwDIN+eM4qvcMy6OtTq5XFP9qSlsyP8rSk33hZN8="_s,
        "mGDW8FOGoVDVNWr8JJpKZ/UWE3L+4sgMm4Ht3RMllwY="_s,
        "ontWH6zp9U7py424W69njjB8G+Pmd0/KYG6X0xag0EA="_s,
        "VLJGQZMntGDzevTy36pKycuQH2LqUXq5XAauhgXPUeY="_s,
        "ljU6kYUWFmB88uf59hUZZ10WEZ3UO01Rvkkwso2iAzA="_s,
        "bJFmac+SO08bLbXFEHyDtsogXnrQ3NhAslHmPwyNKKI="_s,
        "Up0KezlEkpIiFVvKMnK6Goeswvqgmy7SanE4crf/h5Q="_s,
        "jXmsIWS/PWBYPgxEcH951z+N47VeNl+YOckcy2tOXg8="_s,
        "3uA7VZ7xUIL0hkAB1+zFYrL/TdLBp2EEfI/VKRmp7gQ="_s,
        "14WAR75SQBRizwijKnP7jtwvCCT6tLw8ko+X1cEwFVA="_s,
        "HCEME0w0ChEzmRpgWwqsmQzoUFxI/ON8kVBjQClk6ZQ="_s,
        "iG/R5s6ZLt5RR0OoJvpx/v21dwR6KomcqV9vKPMwXQk="_s,
        "p3/E2UhQ5amzlk0I9eLS+tXGY1umjTiyp+F89MVgvzc="_s,
        "5HYWGFSBsOzWNtw1/EZzpKMysayWWF/B+c8NBVPawqA="_s,
        "TcCPFfYc4WuXwMRYC9IzvgyepSFlMz/kYK9UBApcQlw="_s,
        "lOG7wcKkHrxz+lJT/VYyVsADW01pGB5I+a755HShElE="_s,
        "M6JS1jk8vW3r4KxRcinHqiWKDuaPwCU/i+anzui2Xuk="_s,
        "Uj3HLKpCaHyUblqNfmtDiICiaIGWRGbPFCXOURTklwE="_s,
        "aCNSdD2SKrdRVImxgEGyoD7vvpOQXM80C2vc3PXfY74="_s,
        "VEAakg/RytnOL+sACpp0U0HdrrSAuCQ1YS+3k74gBv4="_s,
        "wGkc3TsmPdNN85rSAsMeHvDd3HdtKPlpNrGzgWASFHc="_s,
        "7Z2hTxTl3dXEcK1AO9v9Vef2KaUJjyDmCvKNOHqvFp0="_s,
        "Tt+csKqcOW0oftpNrPEFlEmavyM5aR58vnRD2Gr65Jc="_s,
        "pdjG8bPwgun7Pt99VzGoEDmvJ/5EYGDu2HRQ3jVdfd8="_s,
        "i3ADysxd2P7zRlk8NNruecfR8JZO7Vftd9Ni6Xq3gvY="_s,
        "IOSUZaQHdfd/sMO3lasmnJwc/mZefeU0J8k/Emo9zxQ="_s,
        "9xL/UKwJf4Lk5VyH0ypUe4w61OH20/QTgLrClB3s2L4="_s,
        "LK3tJCwEE5dhdC/gzaf2WS3xtmhoV1MsinwuJTa5duQ="_s,
        "1Cpk3DSamAdei+Elh5Q/K9UgZai7GJYNfcc5C1NRF+A="_s,
        "ziDtijIxF8inGP8d3G2ruZc3O1dajolvK/ArhGwILJ0="_s,
        "4OPEryg0jXIfivYDWVwV0nOlbysDOS+aQTJV/lY19TY="_s,
        "4nScskp3IIq92f2jXw0U8JGUjETSH5d8gEijsT5L7Ms="_s,
        "Z5nJmeQicQ9A9wpgphOPw4EGImxE170bECP1u2W+/vk="_s,
        "YSjdRP7ToEb/jYNdZ34Kg3xwxk3ByUS37f3gTrjYuHk="_s,
        "GkbngM5b62UH1ir4sgqSszyPBC6HxhL0u/gzC/w1NBk="_s,
        "qrlxreFjOrg2IiB0zq4KrYoILZAJCPJ0kbIh1ug5mMo="_s,
        "XoEfCzLUiLmhg7d8/HrB70Sz6nqu0BToOXXf5ZfSIfY="_s,
        "a1VZIONY+KJaQimItEhhXDO8zLT5MugzHOv8jipzf8c="_s,
        "cbPOcmgPQYPSjbhrGEVCBR/VM7sRRpMyM+T2ogz5jLo="_s,
        "oDdcBUoAQb1Y4qC/f6PffDkEv8T3kP0k4y/z7nD9Du8="_s,
        "QE1ggxk+VpvFwox7G8DhPs6AxuD1pQrY6WM/SPPAkVU="_s,
        "+YU60zfVI8CzX+esMGJopwNc4P92JHEO2LOcK4iyCjM="_s,
        // font-awesome 4.7.0
        "qljzPyOaD7AvXHpsRcBD16msmgkzNYBmlOzW1O3A1qg="_s,
        "Kt78vAQefRj88tQXh53FoJmXqmTWdbejxLbOM9oT8/4="_s,
        "RE3UNmYV/8ShbQErL6kBNwZdPMtBD6b9Xk3de15P/NU="_s,
        "ugxZ3rVFD1y0Gz+TYJ7i0NmVQVh33foiPoqKdTNHTwc="_s,
        "e/yrbbmdXPvxcFygU23ceFhUMsxfpBu9etDwCQM7KXk="_s,
        "rWFXkmwWIrpOHQPUePFUE2hSS/xG9R5C/g2UX37zI+Q="_s,
        // font-awesome 3.2.1
        "GOa1/1EbkO3wmOYqxF7Z1mc6Pu4QFl0N5BZNTQKjp38="_s,
        "YckPeDM0c8QPlRJ4lwjNzYIG3SaUARA/M9Q4zYOLsfA="_s,
        "qVleW/O227wHaQK5q96jVgU9aaL+ZlBnBt6bs5oSa48="_s,
        "FNy48Cs7gcK6qyMGEMNJqUKCu9l2aTdZ6bSoaU1IZFE="_s,
        "hE8tLlXUXqIk12XzvzO0z9QWsMC9ayZ+FvjH+/bd4sE="_s,
        // font-awesome 2.0.0
        "+AOWr19Kg+TClydZbWr6XGkJMldfDP+gTsxQuEI2j/E="_s,
        "qIyr/3Pxuda3SAYznoLyut+MR33D4ciYbVBCNjztqFw="_s,
        "WjelroPmneBGeHFs8oeDdIxV3ig/PNu+viyXwV+AIeA="_s,
        "tNJIRHp7jTHskJs6WfAucrVRJpqjCuXPJu8afrq5qME="_s,
        // font-awesome 1.0.0
        "aJmXHqcFloA3EYjbk8dvbFIWqF7D4SnUblDcuHJRYzg="_s,
        "vQ47JwN2wtUCrB4QocyfnFX52oyW35K8KdREcYGq2u8="_s,
        "V9tA0sas96RD7azKpRswnWdOs/eTf7tshAo1U7gIBIU="_s,
        "dF+idgSeiO6zxDQWPk2vXzzWiidIaXARyz8Y6JCWXyU="_s,
        "2q/PWx1LWzvU8MbfMEXcypiltHdu51Z+gD27SLtlax8="_s,
        // internal build website fonts/appleicons_ultralight.woff
        "0vR/heBF2lpWdK/on1m+u85WzVNGnaDe6lilZrVGYgw="_s,
        // internal build website fonts/appleicons_text.woff
        "tIMv6b3XcFfkujgiwBrBz8x/PLespHnG6XOfQDsl5Po="_s,
        // internal build website fonts/appleicons_thin.woff
        "BeSxgqYI30lPA9+4OhNV5gZdbPSOmr1Uc3Pfh+zxgwY="_s,
        // confluence
        "2W4TXu8CrouqU+yAxbOXQv7+cmCwBxTA8Qu4s3FiPao="_s /* adgs-icons.woff */ });

    return allowedFontHashes;
}

static AtomString hashForFontData(const void* data, size_t size)
{
    auto cryptoDigest = PAL::CryptoDigest::create(PAL::CryptoDigest::Algorithm::SHA_256);
    cryptoDigest->addBytes(data, size);
    auto digest = cryptoDigest->computeHash();
    return makeAtomString(base64Encoded(digest.data(), digest.size()));
}

bool isFontBinaryAllowed(const void* data, size_t size, DownloadableBinaryFontAllowedTypes allowedType)
{
    switch (allowedType) {
    case DownloadableBinaryFontAllowedTypes::Any:
        return true;
    case DownloadableBinaryFontAllowedTypes::None:
        return false;
    case DownloadableBinaryFontAllowedTypes::Restricted: {
        auto sha = hashForFontData(data, size);
        return allowedFontHashesInLockdownMode().contains(sha);
    }
    }
    ASSERT_NOT_REACHED();
    return false;
}

bool isFontBinaryAllowed(std::span<const uint8_t> data, DownloadableBinaryFontAllowedTypes allowedType)
{
    return isFontBinaryAllowed(data.data(), data.size(), allowedType);
}

}; // namespace WebCore
