/*
 * Copyright (C) 2007-2013 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "glue.h"

#include "sig_boolean.h"

static void
sig_boolean_flush(struct sig_boolean *b)
{
	unsigned int nr;
	unsigned int i;

	for (i = 0; i < b->in_count; i++) {
		void (*func)(void *, unsigned int);
		unsigned int new_val;

		new_val = 0;

		for (nr = 0; nr < b->out_count; nr++) {
			if (b->out[nr].s == b->in[i].s) continue;

			new_val |= b->out[nr].out;
		}

		func = b->in[i].f->set_ext;
		if (func
		 && b->in[i].in != new_val) {
			b->in[i].in = new_val;
			func(b->in[i].s, new_val);
		}

		for (nr = 0; nr < b->out_count; nr++) {
			if (b->out[nr].s != b->in[i].s) continue;

			new_val |= b->out[nr].out;
		}

		func = b->in[i].f->set;
		if (func
		 && b->in[i].in != new_val) {
			b->in[i].in = new_val;
			func(b->in[i].s, new_val);
		}
	}
}

void
sig_boolean_set(struct sig_boolean *b, void *s, unsigned int val)
{
	unsigned int nr;

	assert(b);
	assert(b->type == SIG_GEN_BOOLEAN);

again:	;
	for (nr = 0; ; nr++) {
		if (nr == b->out_count) {
			fprintf(stderr, "WARNING: late registration of boolean driver of %p!\n", s);
			sig_boolean_connect_out(b, s, 0);
			goto again;
		}
		assert(nr < b->out_count);
		if (b->out[nr].s == s) {
			b->out[nr].out = val;
			break;
		}
	}

	sig_boolean_flush(b);
}

void
sig_boolean_connect_in(
	struct sig_boolean *b,
	void *s,
	const struct sig_boolean_funcs *f
)
{
	assert(b);
	assert(b->type == SIG_GEN_BOOLEAN);
	assert(b->in_count < sizeof(b->in) / sizeof(b->in[0]));
	assert(s);
	assert(f);
	assert(f->set || f->set_ext);
	assert(! f->set || ! f->set_ext);

	b->in[b->in_count].s = s;
	b->in[b->in_count].f = f;
	b->in[b->in_count].in = 0;
	b->in_count++;

	sig_boolean_flush(b);
}

void
sig_boolean_connect_out(
	struct sig_boolean *b,
	void *s,
	unsigned int val
)
{
	assert(b);
	assert(b->type == SIG_GEN_BOOLEAN);
	assert(b->out_count < sizeof(b->out) / sizeof(b->out[0]));
	// assert(s);
	assert(val == 0 || val == 1);

	if (! s) {
		fprintf(stderr, "WARNING: %s: s is NULL pointer!\n",
				__FUNCTION__);
	}

	b->out[b->out_count].s = s;
	b->out[b->out_count].out = val;
	b->out_count++;

	sig_boolean_flush(b);
}

static void
sig_boolean_s0_set(void *_f, unsigned int val)
{
	struct sig_boolean_merge *f = (struct sig_boolean_merge *) _f;

	sig_boolean_set(f->s1, f, val);
}

static void
sig_boolean_s1_set(void *_f, unsigned int val)
{
	struct sig_boolean_merge *f = (struct sig_boolean_merge *) _f;

	sig_boolean_set(f->s0, f, val);
}

struct sig_boolean_merge *
sig_boolean_merge(struct sig_boolean *s0, struct sig_boolean *s1)
{
	static struct sig_boolean_funcs s0_funcs = {
		.set_ext = sig_boolean_s0_set,
	};
	static struct sig_boolean_funcs s1_funcs = {
		.set_ext = sig_boolean_s1_set,
	};
	struct sig_boolean_merge *m;

	m = shm_alloc(sizeof(*m));
	assert(m);

	/* Out */
	m->s0 = s0;
	sig_boolean_connect_out(s0, m, 0);
	m->s1 = s1;
	sig_boolean_connect_out(s1, m, 0);

	/* In */
	sig_boolean_connect_in(s0, m, &s0_funcs);
	sig_boolean_connect_in(s1, m, &s1_funcs);

	return m;
}

void
sig_boolean_split(struct sig_boolean_merge *m)
{
	fixme();
}

struct sig_boolean *
sig_boolean_create(const char *name)
{
	struct sig_boolean *b;

	b = shm_alloc(sizeof(*b));
	assert(b);
	memset(b, 0, sizeof(*b));

	b->type = SIG_GEN_BOOLEAN;
	b->in_count = 0;
	b->out_count = 0;

	return b;
}

void
sig_boolean_destroy(struct sig_boolean *b)
{
	assert(b);
	assert(b->type == SIG_GEN_BOOLEAN);

	shm_free(b);
}

void
sig_boolean_suspend(struct sig_boolean *b, FILE *fSig)
{
	size_t size = sizeof(*b);
	
	generic_suspend(b, size, fSig);
}

void
sig_boolean_resume(struct sig_boolean *b, FILE *fSig)
{
	size_t size = sizeof(*b);
	
	generic_resume(b, size, fSig);
}
