# `faust2api` Documentation

**NOTE:** this documentation was taken from the `faust2api` README.

## Overview

Thanks to its architectures system, [Faust](http://faust.grame.fr) can generate a wide range of objects compatible with different platforms (Linux, Windows, OSX, RPI, Android, iOS, ROS, Bela and Web/JavaScript) and tools (Max/MSP, SuperCollider, PD, VST, AU, LV2, etc.). The mechanism behind this was designed to be as modular and reusable as possible. Additionally, the Faust compiler can be embedded in any C++ program using the [LLVM](http://llvm.org/) technology making it very portable.

The goal of the `faust2api` project is to provide a tool to easily generate custom APIs based on one or several Faust objects. On one hand, [Faust DSP libraries](http://faust.grame.fr/libraries.html) implement hundreds of open source DSP algorithms that can be turned into C++, C, JAVA, JavaScript and LLVM bit code and embedded in your applications. On the other hand, Faust C++ libraries can carry out a wide range of tasks going from connecting Faust DSP objects to a specific audio engine (CoreAudio, OpenSL/ES, Alsa, JACK, etc.) or adding MIDI and polyphony support, sensor data handling, etc. to the same object.

Most major Faust targets are supported:

* iOS
* Android
* OSX CoreAudio
* ALSA
* JACK
* PortAudio
* RTAudio
* openFrameworks
* JUCE

**NOTE:** This documentation only provides high level information on how to use `faust2api`. For detailed tutorials on this topic, visit [this page](https://ccrma.stanford.edu/~rmichon/faustTutorials/#adding-faust-real-time-audio-support-to-android-apps).

## Using `faust2api`

Elements generated by `faust2api` greatly vary from one platform to another. Thus, a custom documentation is generated and integrated to packages generated by `faust2api` in function of the options provided to it (if you only want to generate the package without the documentation, which can save some compilation time, you can use the `-mdoc` option). As a result, this page doesn't provide information on how to use the generated APIs, but rather high level instructions on how to configure `faust2api` to carry out specific tasks.

The various options of `faust2api` can be displayed at any point by running:

	faust2api -help

`faust2api` is part of the [Faust distribution](https://github.com/grame-cncm/faust). To use it, Faust must be properly installed on your system (read the Faust README in the previous link to get more information on how to do that).


Common options for all platforms:

- `-opt native|generic`: to activate the best compilation options for the native or generic CPU.
- `-nvoices <num>` : creates a polyphonic object with <num> voices.
- `-effect <effect.dsp>`: adds an effect to the polyphonic synth (this option is ignored if -nvoices is not specified).
- `-effect auto`: adds an effect (extracted automatically from the dsp file) to the polyphonic synth (this option is ignored if -nvoices is not specified).
- `-nodoc` : prevents documentation from being generated.
- `-nozip` : prevents generated files to be put in a zip file.


### Android Support

To turn a Faust code into an Android API, just run the following command:

```
faust2api -android yourFaustCode.dsp
```

The JAVA package name of the API generated by `faust2api` is `com.DspFaust`. It can easily be changed by using the `-package` option. For example, to change the package to `com.you.DspFaust`, just run the following command:

```
faust2api -android -package com.you yourFaustCode.dsp
```

### iOS Support

To turn a Faust code into an iOS API, just run the following command:

```
faust2api -ios yourFaustCode.dsp
```

The `-midi` option can be used to add RtMidi support to the API.
The `-osc` option can be used to add OSC support to the API.

### OSX CoreAudio Support

To turn a Faust code into an OSX CoreAudio API, just run the following command:

```
faust2api -coreaudio yourFaustCode.dsp
```

The `-midi` option can be used to add RtMidi support to the API.
The `-osc` option can be used to add OSC support to the API.

### ALSA Support

To turn a Faust code into an ALSA API, just run the following command:

```
faust2api -alsa yourFaustCode.dsp
```

The `-midi` option can be used to add RtMidi support to the API.
The `-osc` option can be used to add OSC support to the API.

### JACK Support

To turn a Faust code into a JACK API, just run the following command:

```
faust2api -jack yourFaustCode.dsp
```

The `-midi` option can be used to add JACK MIDI support to the API.
The `-osc` option can be used to add OSC support to the API.

### PortAudio Support

To turn a Faust code into a PortAudio API, just run the following command:

```
faust2api -portaudio yourFaustCode.dsp
```

The `-midi` option can be used to add RtMidi support to the API.
The `-osc` option can be used to add OSC support to the API.

### RTAudio Support

To turn a Faust code into an RTAudio API, just run the following command:

```
faust2api -rtaudio yourFaustCode.dsp
```

The `-midi` option can be used to add RtMidi support to the API.
The `-osc` option can be used to add OSC support to the API.

### OpenFrameworks Support

To turn a Faust code into an OpenFrameworks API, just run the following command:

```
faust2api -of yourFaustCode.dsp
```

### JUCE Support

To turn a Faust code into a JUCE API, just run the following command:

```
faust2api -juce yourFaustCode.dsp
```

The `-midi` option can be used to add JUCE MIDI support to the API.
The `-osc` option can be used to add JUCE OSC support to the API.

### Customizing the API

#### Polyphonic Object

To create a polyphonic object, set the maximum number of voices of this object
using the `-nvoices` option e.g.:

```
faust2api -android -nvoices 12 yourSynth.dsp
```

In this case, we're creating an object with a maximum number of 12 polyphonic voices. Voices are only instantiated and computed when they are used, so this is just a safe guard...

#### MIDI Enabled Polyphonic Object

To create a polyphonic object controllable by a MIDI keyboard, run the previous command but make sure that the `freq`, `gain` and `gate` parameters are declared in your Faust code. E.g:

```
f = hslider("freq",300,50,2000,0.01);
bend = hslider("bend[midi:pitchwheel]",1,0,10,0.01) : si.polySmooth(gate,0.999,1);
gain = hslider("gain",1,0,1,0.01);
s = hslider("sustain[midi:ctrl 64]",0,0,1,1); // for sustain pedal
t = button("gate");

cutoff = hslider("cutoff[midi:ctrl 1]",50,50,4000,0.001) : si.polySmooth(gate,0.999,2);
gate = t+s : min(1);
freq = f*bend;
envelope = gate*gain : si.smoo;

process = os.sawtooth(freq)*envelope : fi.lowpass(3,cutoff) <: _,_;
```

`freq` (which should be a frequency) will be automatically mapped to MIDI note numbers, `gain` (which should be a value between 0 and 1) to velocity and `gate` to *note-on* / *note-off* events. Thus, `gate` can be used as a trigger signal for any envelope generator, etc.

Continuous pitch control is carried out through `bend` which is used to modulate
the pitch set by `freq`.

Note that we're using the `si.polySmooth` function here to smooth the value of some parameters. This function is very useful when creating polyphonic objects as it only starts smoothing after the note was created, preventing "ugly" sweeps, etc.

0.999 is the pole of the lowpass used for smoothing and 2 is the number of samples to wait before smoothing begins after the voice was created.

#### Adding an Audio Effect to a Polyphonic Object

If creating a polyphonic synthesizer, it is very likely that you will need to plug its output to an audio effect like a reverb. Putting that effect in your main Faust code is not a good idea since it will be instanciated for each voice which would be very inefficient. Fortunately, `faust2api` can take a second Faust file as one of its arguments, containing an audio effect (could be a chain of audio effects, of course). The only rule is that the number of outputs of the synth must the same as the number of inputs of the effect. For example, your effect Faust file could look like this:

	import("stdfaust.lib");
	process = dm.zita_rev1;

`zita_rev1` is a stereo reverb with a built-in UI that can be connected to the synthesizer presented in the previous section. This code must be saved in a separate Faust file (e.g., `effect.dsp`).

To generate the corresponding API on Android, run:

	faust2api -android -nvoices 12 -effect effect.dsp yourSynth.dsp

Keep in mind that the package generated by `faust2api` contains a README file that you should really read at this point! Also, for more information, check our tutorials on [Using `faust2api` to Add Faust Audio Support to Android Apps](https://ccrma.stanford.edu/~rmichon/faustTutorials/#adding-faust-real-time-audio-support-to-android-apps) and [Using `faust2api` to Add Faust Audio Support to iOS Apps](https://ccrma.stanford.edu/~rmichon/faustTutorials/#adding-faust-real-time-audio-support-to-ios-apps).

## Additional Resources

* [What Is Faust?](https://ccrma.stanford.edu/~rmichon/faustTutorials/#what-is-faust)
* [Faust Hero in 2 Hours](https://ccrma.stanford.edu/~rmichon/faustTutorials/#faust-hero-in-2-hours)
* [Using `faust2api` to Add Faust Audio Support to Android Apps](https://ccrma.stanford.edu/~rmichon/faustTutorials/#adding-faust-real-time-audio-support-to-android-apps)
* [Using `faust2api` to Add Faust Audio Support to iOS Apps](https://ccrma.stanford.edu/~rmichon/faustTutorials/#adding-faust-real-time-audio-support-to-ios-apps)
