
#pragma once

#include <array>
#include <vector>
#include <functional>

#include <glbinding/glbinding_api.h>

#include <glbinding/ContextHandle.h>
#include <glbinding/Function.h>

#include <glbinding/gl/types.h>


namespace glbinding
{


/**
 * @brief
 *   The main interface to handle additional features to OpenGL functions besides regular function calls.
 *
 * Additional features include binding initialization (even for multi-threaded environments), additional function registration,
 * context switches (for multi-context environments) and basic reflection in form of accessors to the full list of functions.
 */
class GLBINDING_API Binding
{
public:
    using array_t = std::array<AbstractFunction *, 2852>; ///< The type of the build-in functions collection.
    using ContextSwitchCallback = std::function<void(ContextHandle)>; ///< The signature of the context switch callback.

    /**
     * @brief
     *   Deleted Constructor as all functions are static.
     */
    Binding() = delete;

    /**
     * @brief
     *   Initializes the binding for the current active OpenGL context.
     *
     * @param[in] resolveFunctions (optional)
     *   Whether to resolve function pointers lazy (resolveFunctions = false) or immediately.
     *
     * After this call, the initialized context is already set active for the current thread.
     */
    static void initialize(bool resolveFunctions = true);
    
    /**
     * @brief
     *   Initializes the binding for the current active OpenGL context.
     *
     * @param[in] context
     *   The context handle of the context to initialize.
     * @param[in] useContext
     *   Whether to set the context active (useContext = true) after the initialization.
     * @param[in] resolveFunctions (optional)
     *   Whether to resolve function pointers lazy (resolveFunctions = false) or immediately.
     */
    static void initialize(ContextHandle context, bool useContext = true, bool resolveFunctions = true);
    
    /**
     * @brief
     *   Registers an additional function for the additional features.
     *
     * @param[in] function
     *   The function to register.
     *
     * The additional features are callbacks, and use in multi-context environments.
     */
    static void registerAdditionalFunction(AbstractFunction * function);

    /**
     * @brief
     *   Resolves the funtion pointers of all registered OpenGL functions immediately for the current context.
     */
    static void resolveFunctions();
    
    /**
     * @brief
     *   Update the current context state in glbinding.
     *   This function queries the driver for the current OpenGL context.
     */
    static void useCurrentContext();
    
    /**
     * @brief
     *   Update the current context state in glbinding.
     *
     * @param[in] context
     *   The context handle of the context to set current.
     */
    static void useContext(ContextHandle context);

    /**
     * @brief
     *   Removes the current context from the state of glbinding.
     *   This function queries the driver for the current OpenGL context.
     */
    static void releaseCurrentContext();
    
    /**
     * @brief
     *   Removes the current context from the state of glbinding.
     *
     * @param[in] context
     *   The context handle of the context to remove.
     */
    static void releaseContext(ContextHandle context);
    
    /**
     * @brief
     *   Registers an additional callback that gets called each time the context is switched using the useContext method.
     *
     * There may be multiple context switch callbacks registered at once.
     */
    static void addContextSwitchCallback(ContextSwitchCallback callback);

    /**
     * @brief
     *   The accessor for the number of all registered functions (build-in and additional).
     *
     * @return
     *   The number of all registered functions.
     */
    static size_t size();

    /**
     * @brief
     *   The accessor for all build-in functions.
     *
     * @return
     *   The list of all build-in functions.
     */
    static const array_t & functions();
    
    /**
     * @brief
     *   The accessor for all additional functions.
     *
     * @return
     *   The list of all additional functions.
     */
    static const std::vector<AbstractFunction *> & additionalFunctions();

public:
    static Function<void, gl::GLenum, gl::GLfloat> Accum;
    static Function<void, gl::GLenum, gl::GLfixed> AccumxOES;
    static Function<void, gl::GLuint> ActiveProgramEXT;
    static Function<void, gl::GLuint, gl::GLuint> ActiveShaderProgram;
    static Function<void, gl::GLenum> ActiveStencilFaceEXT;
    static Function<void, gl::GLenum> ActiveTexture;
    static Function<void, gl::GLenum> ActiveTextureARB;
    static Function<void, gl::GLuint, const gl::GLchar *> ActiveVaryingNV;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint> AlphaFragmentOp1ATI;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint> AlphaFragmentOp2ATI;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint> AlphaFragmentOp3ATI;
    static Function<void, gl::GLenum, gl::GLfloat> AlphaFunc;
    static Function<void, gl::GLenum, gl::GLfixed> AlphaFuncxOES;
    static Function<void> ApplyFramebufferAttachmentCMAAINTEL;
    static Function<void, gl::GLenum> ApplyTextureEXT;
    static Function<gl::GLboolean, gl::GLsizei, const gl::GLuint *, gl::GLboolean *> AreProgramsResidentNV;
    static Function<gl::GLboolean, gl::GLsizei, const gl::GLuint *, gl::GLboolean *> AreTexturesResident;
    static Function<gl::GLboolean, gl::GLsizei, const gl::GLuint *, gl::GLboolean *> AreTexturesResidentEXT;
    static Function<void, gl::GLint> ArrayElement;
    static Function<void, gl::GLint> ArrayElementEXT;
    static Function<void, gl::GLenum, gl::GLint, gl::GLenum, gl::GLsizei, gl::GLuint, gl::GLuint> ArrayObjectATI;
    static Function<void, gl::GLuint> AsyncMarkerSGIX;
    static Function<void, gl::GLhandleARB, gl::GLhandleARB> AttachObjectARB;
    static Function<void, gl::GLuint, gl::GLuint> AttachShader;
    static Function<void, gl::GLenum> Begin;
    static Function<void, gl::GLuint, gl::GLenum> BeginConditionalRender;
    static Function<void, gl::GLuint, gl::GLenum> BeginConditionalRenderNV;
    static Function<void, gl::GLuint> BeginConditionalRenderNVX;
    static Function<void> BeginFragmentShaderATI;
    static Function<void, gl::GLuint> BeginOcclusionQueryNV;
    static Function<void, gl::GLuint> BeginPerfMonitorAMD;
    static Function<void, gl::GLuint> BeginPerfQueryINTEL;
    static Function<void, gl::GLenum, gl::GLuint> BeginQuery;
    static Function<void, gl::GLenum, gl::GLuint> BeginQueryARB;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLuint> BeginQueryIndexed;
    static Function<void, gl::GLenum> BeginTransformFeedback;
    static Function<void, gl::GLenum> BeginTransformFeedbackEXT;
    static Function<void, gl::GLenum> BeginTransformFeedbackNV;
    static Function<void> BeginVertexShaderEXT;
    static Function<void, gl::GLuint> BeginVideoCaptureNV;
    static Function<void, gl::GLuint, gl::GLuint, const gl::GLchar *> BindAttribLocation;
    static Function<void, gl::GLhandleARB, gl::GLuint, const gl::GLcharARB *> BindAttribLocationARB;
    static Function<void, gl::GLenum, gl::GLuint> BindBuffer;
    static Function<void, gl::GLenum, gl::GLuint> BindBufferARB;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLuint> BindBufferBase;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLuint> BindBufferBaseEXT;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLuint> BindBufferBaseNV;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLuint, gl::GLintptr> BindBufferOffsetEXT;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLuint, gl::GLintptr> BindBufferOffsetNV;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLuint, gl::GLintptr, gl::GLsizeiptr> BindBufferRange;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLuint, gl::GLintptr, gl::GLsizeiptr> BindBufferRangeEXT;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLuint, gl::GLintptr, gl::GLsizeiptr> BindBufferRangeNV;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLsizei, const gl::GLuint *> BindBuffersBase;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLsizei, const gl::GLuint *, const gl::GLintptr *, const gl::GLsizeiptr *> BindBuffersRange;
    static Function<void, gl::GLuint, gl::GLuint, const gl::GLchar *> BindFragDataLocation;
    static Function<void, gl::GLuint, gl::GLuint, const gl::GLchar *> BindFragDataLocationEXT;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLuint, const gl::GLchar *> BindFragDataLocationIndexed;
    static Function<void, gl::GLuint> BindFragmentShaderATI;
    static Function<void, gl::GLenum, gl::GLuint> BindFramebuffer;
    static Function<void, gl::GLenum, gl::GLuint> BindFramebufferEXT;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLint, gl::GLboolean, gl::GLint, gl::GLenum, gl::GLenum> BindImageTexture;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLint, gl::GLboolean, gl::GLint, gl::GLenum, gl::GLint> BindImageTextureEXT;
    static Function<void, gl::GLuint, gl::GLsizei, const gl::GLuint *> BindImageTextures;
    static Function<gl::GLuint, gl::GLenum, gl::GLenum> BindLightParameterEXT;
    static Function<gl::GLuint, gl::GLenum, gl::GLenum> BindMaterialParameterEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLuint> BindMultiTextureEXT;
    static Function<gl::GLuint, gl::GLenum> BindParameterEXT;
    static Function<void, gl::GLenum, gl::GLuint> BindProgramARB;
    static Function<void, gl::GLenum, gl::GLuint> BindProgramNV;
    static Function<void, gl::GLuint> BindProgramPipeline;
    static Function<void, gl::GLenum, gl::GLuint> BindRenderbuffer;
    static Function<void, gl::GLenum, gl::GLuint> BindRenderbufferEXT;
    static Function<void, gl::GLuint, gl::GLuint> BindSampler;
    static Function<void, gl::GLuint, gl::GLsizei, const gl::GLuint *> BindSamplers;
    static Function<gl::GLuint, gl::GLenum, gl::GLenum, gl::GLenum> BindTexGenParameterEXT;
    static Function<void, gl::GLenum, gl::GLuint> BindTexture;
    static Function<void, gl::GLenum, gl::GLuint> BindTextureEXT;
    static Function<void, gl::GLuint, gl::GLuint> BindTextureUnit;
    static Function<gl::GLuint, gl::GLenum, gl::GLenum> BindTextureUnitParameterEXT;
    static Function<void, gl::GLuint, gl::GLsizei, const gl::GLuint *> BindTextures;
    static Function<void, gl::GLenum, gl::GLuint> BindTransformFeedback;
    static Function<void, gl::GLenum, gl::GLuint> BindTransformFeedbackNV;
    static Function<void, gl::GLuint> BindVertexArray;
    static Function<void, gl::GLuint> BindVertexArrayAPPLE;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLintptr, gl::GLsizei> BindVertexBuffer;
    static Function<void, gl::GLuint, gl::GLsizei, const gl::GLuint *, const gl::GLintptr *, const gl::GLsizei *> BindVertexBuffers;
    static Function<void, gl::GLuint> BindVertexShaderEXT;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLenum, gl::GLintptrARB> BindVideoCaptureStreamBufferNV;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLenum, gl::GLenum, gl::GLuint> BindVideoCaptureStreamTextureNV;
    static Function<void, gl::GLbyte, gl::GLbyte, gl::GLbyte> Binormal3bEXT;
    static Function<void, const gl::GLbyte *> Binormal3bvEXT;
    static Function<void, gl::GLdouble, gl::GLdouble, gl::GLdouble> Binormal3dEXT;
    static Function<void, const gl::GLdouble *> Binormal3dvEXT;
    static Function<void, gl::GLfloat, gl::GLfloat, gl::GLfloat> Binormal3fEXT;
    static Function<void, const gl::GLfloat *> Binormal3fvEXT;
    static Function<void, gl::GLint, gl::GLint, gl::GLint> Binormal3iEXT;
    static Function<void, const gl::GLint *> Binormal3ivEXT;
    static Function<void, gl::GLshort, gl::GLshort, gl::GLshort> Binormal3sEXT;
    static Function<void, const gl::GLshort *> Binormal3svEXT;
    static Function<void, gl::GLenum, gl::GLsizei, const void *> BinormalPointerEXT;
    static Function<void, gl::GLsizei, gl::GLsizei, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, const gl::GLubyte *> Bitmap;
    static Function<void, gl::GLsizei, gl::GLsizei, gl::GLfixed, gl::GLfixed, gl::GLfixed, gl::GLfixed, const gl::GLubyte *> BitmapxOES;
    static Function<void> BlendBarrierKHR;
    static Function<void> BlendBarrierNV;
    static Function<void, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat> BlendColor;
    static Function<void, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat> BlendColorEXT;
    static Function<void, gl::GLfixed, gl::GLfixed, gl::GLfixed, gl::GLfixed> BlendColorxOES;
    static Function<void, gl::GLenum> BlendEquation;
    static Function<void, gl::GLenum> BlendEquationEXT;
    static Function<void, gl::GLuint, gl::GLenum> BlendEquationIndexedAMD;
    static Function<void, gl::GLenum, gl::GLenum> BlendEquationSeparate;
    static Function<void, gl::GLenum, gl::GLenum> BlendEquationSeparateEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLenum> BlendEquationSeparateIndexedAMD;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLenum> BlendEquationSeparatei;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLenum> BlendEquationSeparateiARB;
    static Function<void, gl::GLuint, gl::GLenum> BlendEquationi;
    static Function<void, gl::GLuint, gl::GLenum> BlendEquationiARB;
    static Function<void, gl::GLenum, gl::GLenum> BlendFunc;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLenum> BlendFuncIndexedAMD;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLenum> BlendFuncSeparate;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLenum> BlendFuncSeparateEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLenum> BlendFuncSeparateINGR;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLenum> BlendFuncSeparateIndexedAMD;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLenum> BlendFuncSeparatei;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLenum> BlendFuncSeparateiARB;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLenum> BlendFunci;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLenum> BlendFunciARB;
    static Function<void, gl::GLenum, gl::GLint> BlendParameteriNV;
    static Function<void, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::ClearBufferMask, gl::GLenum> BlitFramebuffer;
    static Function<void, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::ClearBufferMask, gl::GLenum> BlitFramebufferEXT;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::ClearBufferMask, gl::GLenum> BlitNamedFramebuffer;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLuint64EXT, gl::GLsizeiptr> BufferAddressRangeNV;
    static Function<void, gl::GLenum, gl::GLsizeiptr, const void *, gl::GLenum> BufferData;
    static Function<void, gl::GLenum, gl::GLsizeiptrARB, const void *, gl::GLenum> BufferDataARB;
    static Function<void, gl::GLenum, gl::GLintptr, gl::GLsizeiptr, gl::GLboolean> BufferPageCommitmentARB;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint> BufferParameteriAPPLE;
    static Function<void, gl::GLenum, gl::GLsizeiptr, const void *, gl::BufferStorageMask> BufferStorage;
    static Function<void, gl::GLenum, gl::GLintptr, gl::GLsizeiptr, const void *> BufferSubData;
    static Function<void, gl::GLenum, gl::GLintptrARB, gl::GLsizeiptrARB, const void *> BufferSubDataARB;
    static Function<void, gl::GLuint> CallCommandListNV;
    static Function<void, gl::GLuint> CallList;
    static Function<void, gl::GLsizei, gl::GLenum, const void *> CallLists;
    static Function<gl::GLenum, gl::GLenum> CheckFramebufferStatus;
    static Function<gl::GLenum, gl::GLenum> CheckFramebufferStatusEXT;
    static Function<gl::GLenum, gl::GLuint, gl::GLenum> CheckNamedFramebufferStatus;
    static Function<gl::GLenum, gl::GLuint, gl::GLenum> CheckNamedFramebufferStatusEXT;
    static Function<void, gl::GLenum, gl::GLenum> ClampColor;
    static Function<void, gl::GLenum, gl::GLenum> ClampColorARB;
    static Function<void, gl::ClearBufferMask> Clear;
    static Function<void, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat> ClearAccum;
    static Function<void, gl::GLfixed, gl::GLfixed, gl::GLfixed, gl::GLfixed> ClearAccumxOES;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLenum, const void *> ClearBufferData;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLintptr, gl::GLsizeiptr, gl::GLenum, gl::GLenum, const void *> ClearBufferSubData;
    static Function<void, gl::GLenum, gl::GLint, gl::GLfloat, gl::GLint> ClearBufferfi;
    static Function<void, gl::GLenum, gl::GLint, const gl::GLfloat *> ClearBufferfv;
    static Function<void, gl::GLenum, gl::GLint, const gl::GLint *> ClearBufferiv;
    static Function<void, gl::GLenum, gl::GLint, const gl::GLuint *> ClearBufferuiv;
    static Function<void, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat> ClearColor;
    static Function<void, gl::GLint, gl::GLint, gl::GLint, gl::GLint> ClearColorIiEXT;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint> ClearColorIuiEXT;
    static Function<void, gl::GLfixed, gl::GLfixed, gl::GLfixed, gl::GLfixed> ClearColorxOES;
    static Function<void, gl::GLdouble> ClearDepth;
    static Function<void, gl::GLdouble> ClearDepthdNV;
    static Function<void, gl::GLfloat> ClearDepthf;
    static Function<void, gl::GLclampf> ClearDepthfOES;
    static Function<void, gl::GLfixed> ClearDepthxOES;
    static Function<void, gl::GLfloat> ClearIndex;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLenum, gl::GLenum, const void *> ClearNamedBufferData;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLenum, gl::GLenum, const void *> ClearNamedBufferDataEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLintptr, gl::GLsizeiptr, gl::GLenum, gl::GLenum, const void *> ClearNamedBufferSubData;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLsizeiptr, gl::GLsizeiptr, gl::GLenum, gl::GLenum, const void *> ClearNamedBufferSubDataEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint, gl::GLfloat, gl::GLint> ClearNamedFramebufferfi;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint, const gl::GLfloat *> ClearNamedFramebufferfv;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint, const gl::GLint *> ClearNamedFramebufferiv;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint, const gl::GLuint *> ClearNamedFramebufferuiv;
    static Function<void, gl::GLint> ClearStencil;
    static Function<void, gl::GLuint, gl::GLint, gl::GLenum, gl::GLenum, const void *> ClearTexImage;
    static Function<void, gl::GLuint, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLsizei, gl::GLenum, gl::GLenum, const void *> ClearTexSubImage;
    static Function<void, gl::GLenum> ClientActiveTexture;
    static Function<void, gl::GLenum> ClientActiveTextureARB;
    static Function<void, gl::GLenum> ClientActiveVertexStreamATI;
    static Function<void, gl::ClientAttribMask> ClientAttribDefaultEXT;
    static Function<gl::GLenum, gl::GLsync, gl::SyncObjectMask, gl::GLuint64> ClientWaitSync;
    static Function<void, gl::GLenum, gl::GLenum> ClipControl;
    static Function<void, gl::GLenum, const gl::GLdouble *> ClipPlane;
    static Function<void, gl::GLenum, const gl::GLfloat *> ClipPlanefOES;
    static Function<void, gl::GLenum, const gl::GLfixed *> ClipPlanexOES;
    static Function<void, gl::GLbyte, gl::GLbyte, gl::GLbyte> Color3b;
    static Function<void, const gl::GLbyte *> Color3bv;
    static Function<void, gl::GLdouble, gl::GLdouble, gl::GLdouble> Color3d;
    static Function<void, const gl::GLdouble *> Color3dv;
    static Function<void, gl::GLfloat, gl::GLfloat, gl::GLfloat> Color3f;
    static Function<void, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat> Color3fVertex3fSUN;
    static Function<void, const gl::GLfloat *, const gl::GLfloat *> Color3fVertex3fvSUN;
    static Function<void, const gl::GLfloat *> Color3fv;
    static Function<void, gl::GLhalfNV, gl::GLhalfNV, gl::GLhalfNV> Color3hNV;
    static Function<void, const gl::GLhalfNV *> Color3hvNV;
    static Function<void, gl::GLint, gl::GLint, gl::GLint> Color3i;
    static Function<void, const gl::GLint *> Color3iv;
    static Function<void, gl::GLshort, gl::GLshort, gl::GLshort> Color3s;
    static Function<void, const gl::GLshort *> Color3sv;
    static Function<void, gl::GLubyte, gl::GLubyte, gl::GLubyte> Color3ub;
    static Function<void, const gl::GLubyte *> Color3ubv;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLuint> Color3ui;
    static Function<void, const gl::GLuint *> Color3uiv;
    static Function<void, gl::GLushort, gl::GLushort, gl::GLushort> Color3us;
    static Function<void, const gl::GLushort *> Color3usv;
    static Function<void, gl::GLfixed, gl::GLfixed, gl::GLfixed> Color3xOES;
    static Function<void, const gl::GLfixed *> Color3xvOES;
    static Function<void, gl::GLbyte, gl::GLbyte, gl::GLbyte, gl::GLbyte> Color4b;
    static Function<void, const gl::GLbyte *> Color4bv;
    static Function<void, gl::GLdouble, gl::GLdouble, gl::GLdouble, gl::GLdouble> Color4d;
    static Function<void, const gl::GLdouble *> Color4dv;
    static Function<void, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat> Color4f;
    static Function<void, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat> Color4fNormal3fVertex3fSUN;
    static Function<void, const gl::GLfloat *, const gl::GLfloat *, const gl::GLfloat *> Color4fNormal3fVertex3fvSUN;
    static Function<void, const gl::GLfloat *> Color4fv;
    static Function<void, gl::GLhalfNV, gl::GLhalfNV, gl::GLhalfNV, gl::GLhalfNV> Color4hNV;
    static Function<void, const gl::GLhalfNV *> Color4hvNV;
    static Function<void, gl::GLint, gl::GLint, gl::GLint, gl::GLint> Color4i;
    static Function<void, const gl::GLint *> Color4iv;
    static Function<void, gl::GLshort, gl::GLshort, gl::GLshort, gl::GLshort> Color4s;
    static Function<void, const gl::GLshort *> Color4sv;
    static Function<void, gl::GLubyte, gl::GLubyte, gl::GLubyte, gl::GLubyte> Color4ub;
    static Function<void, gl::GLubyte, gl::GLubyte, gl::GLubyte, gl::GLubyte, gl::GLfloat, gl::GLfloat> Color4ubVertex2fSUN;
    static Function<void, const gl::GLubyte *, const gl::GLfloat *> Color4ubVertex2fvSUN;
    static Function<void, gl::GLubyte, gl::GLubyte, gl::GLubyte, gl::GLubyte, gl::GLfloat, gl::GLfloat, gl::GLfloat> Color4ubVertex3fSUN;
    static Function<void, const gl::GLubyte *, const gl::GLfloat *> Color4ubVertex3fvSUN;
    static Function<void, const gl::GLubyte *> Color4ubv;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint> Color4ui;
    static Function<void, const gl::GLuint *> Color4uiv;
    static Function<void, gl::GLushort, gl::GLushort, gl::GLushort, gl::GLushort> Color4us;
    static Function<void, const gl::GLushort *> Color4usv;
    static Function<void, gl::GLfixed, gl::GLfixed, gl::GLfixed, gl::GLfixed> Color4xOES;
    static Function<void, const gl::GLfixed *> Color4xvOES;
    static Function<void, gl::GLint, gl::GLenum, gl::GLsizei> ColorFormatNV;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint> ColorFragmentOp1ATI;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint> ColorFragmentOp2ATI;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint> ColorFragmentOp3ATI;
    static Function<void, gl::GLboolean, gl::GLboolean, gl::GLboolean, gl::GLboolean> ColorMask;
    static Function<void, gl::GLuint, gl::GLboolean, gl::GLboolean, gl::GLboolean, gl::GLboolean> ColorMaskIndexedEXT;
    static Function<void, gl::GLuint, gl::GLboolean, gl::GLboolean, gl::GLboolean, gl::GLboolean> ColorMaski;
    static Function<void, gl::GLenum, gl::GLenum> ColorMaterial;
    static Function<void, gl::GLenum, gl::GLuint> ColorP3ui;
    static Function<void, gl::GLenum, const gl::GLuint *> ColorP3uiv;
    static Function<void, gl::GLenum, gl::GLuint> ColorP4ui;
    static Function<void, gl::GLenum, const gl::GLuint *> ColorP4uiv;
    static Function<void, gl::GLint, gl::GLenum, gl::GLsizei, const void *> ColorPointer;
    static Function<void, gl::GLint, gl::GLenum, gl::GLsizei, gl::GLsizei, const void *> ColorPointerEXT;
    static Function<void, gl::GLint, gl::GLenum, gl::GLint, const void **, gl::GLint> ColorPointerListIBM;
    static Function<void, gl::GLint, gl::GLenum, const void **> ColorPointervINTEL;
    static Function<void, gl::GLenum, gl::GLsizei, gl::GLsizei, gl::GLenum, gl::GLenum, const void *> ColorSubTable;
    static Function<void, gl::GLenum, gl::GLsizei, gl::GLsizei, gl::GLenum, gl::GLenum, const void *> ColorSubTableEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLsizei, gl::GLenum, gl::GLenum, const void *> ColorTable;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLsizei, gl::GLenum, gl::GLenum, const void *> ColorTableEXT;
    static Function<void, gl::GLenum, gl::GLenum, const gl::GLfloat *> ColorTableParameterfv;
    static Function<void, gl::GLenum, gl::GLenum, const gl::GLfloat *> ColorTableParameterfvSGI;
    static Function<void, gl::GLenum, gl::GLenum, const gl::GLint *> ColorTableParameteriv;
    static Function<void, gl::GLenum, gl::GLenum, const gl::GLint *> ColorTableParameterivSGI;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLsizei, gl::GLenum, gl::GLenum, const void *> ColorTableSGI;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLenum> CombinerInputNV;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLboolean, gl::GLboolean, gl::GLboolean> CombinerOutputNV;
    static Function<void, gl::GLenum, gl::GLfloat> CombinerParameterfNV;
    static Function<void, gl::GLenum, const gl::GLfloat *> CombinerParameterfvNV;
    static Function<void, gl::GLenum, gl::GLint> CombinerParameteriNV;
    static Function<void, gl::GLenum, const gl::GLint *> CombinerParameterivNV;
    static Function<void, gl::GLenum, gl::GLenum, const gl::GLfloat *> CombinerStageParameterfvNV;
    static Function<void, gl::GLuint, gl::GLuint> CommandListSegmentsNV;
    static Function<void, gl::GLuint> CompileCommandListNV;
    static Function<void, gl::GLuint> CompileShader;
    static Function<void, gl::GLhandleARB> CompileShaderARB;
    static Function<void, gl::GLuint, gl::GLsizei, const gl::GLchar *const*, const gl::GLint *> CompileShaderIncludeARB;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint, gl::GLenum, gl::GLsizei, gl::GLint, gl::GLsizei, const void *> CompressedMultiTexImage1DEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint, gl::GLenum, gl::GLsizei, gl::GLsizei, gl::GLint, gl::GLsizei, const void *> CompressedMultiTexImage2DEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint, gl::GLenum, gl::GLsizei, gl::GLsizei, gl::GLsizei, gl::GLint, gl::GLsizei, const void *> CompressedMultiTexImage3DEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint, gl::GLint, gl::GLsizei, gl::GLenum, gl::GLsizei, const void *> CompressedMultiTexSubImage1DEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLenum, gl::GLsizei, const void *> CompressedMultiTexSubImage2DEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLsizei, gl::GLenum, gl::GLsizei, const void *> CompressedMultiTexSubImage3DEXT;
    static Function<void, gl::GLenum, gl::GLint, gl::GLenum, gl::GLsizei, gl::GLint, gl::GLsizei, const void *> CompressedTexImage1D;
    static Function<void, gl::GLenum, gl::GLint, gl::GLenum, gl::GLsizei, gl::GLint, gl::GLsizei, const void *> CompressedTexImage1DARB;
    static Function<void, gl::GLenum, gl::GLint, gl::GLenum, gl::GLsizei, gl::GLsizei, gl::GLint, gl::GLsizei, const void *> CompressedTexImage2D;
    static Function<void, gl::GLenum, gl::GLint, gl::GLenum, gl::GLsizei, gl::GLsizei, gl::GLint, gl::GLsizei, const void *> CompressedTexImage2DARB;
    static Function<void, gl::GLenum, gl::GLint, gl::GLenum, gl::GLsizei, gl::GLsizei, gl::GLsizei, gl::GLint, gl::GLsizei, const void *> CompressedTexImage3D;
    static Function<void, gl::GLenum, gl::GLint, gl::GLenum, gl::GLsizei, gl::GLsizei, gl::GLsizei, gl::GLint, gl::GLsizei, const void *> CompressedTexImage3DARB;
    static Function<void, gl::GLenum, gl::GLint, gl::GLint, gl::GLsizei, gl::GLenum, gl::GLsizei, const void *> CompressedTexSubImage1D;
    static Function<void, gl::GLenum, gl::GLint, gl::GLint, gl::GLsizei, gl::GLenum, gl::GLsizei, const void *> CompressedTexSubImage1DARB;
    static Function<void, gl::GLenum, gl::GLint, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLenum, gl::GLsizei, const void *> CompressedTexSubImage2D;
    static Function<void, gl::GLenum, gl::GLint, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLenum, gl::GLsizei, const void *> CompressedTexSubImage2DARB;
    static Function<void, gl::GLenum, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLsizei, gl::GLenum, gl::GLsizei, const void *> CompressedTexSubImage3D;
    static Function<void, gl::GLenum, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLsizei, gl::GLenum, gl::GLsizei, const void *> CompressedTexSubImage3DARB;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint, gl::GLenum, gl::GLsizei, gl::GLint, gl::GLsizei, const void *> CompressedTextureImage1DEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint, gl::GLenum, gl::GLsizei, gl::GLsizei, gl::GLint, gl::GLsizei, const void *> CompressedTextureImage2DEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint, gl::GLenum, gl::GLsizei, gl::GLsizei, gl::GLsizei, gl::GLint, gl::GLsizei, const void *> CompressedTextureImage3DEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLint, gl::GLsizei, gl::GLenum, gl::GLsizei, const void *> CompressedTextureSubImage1D;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint, gl::GLint, gl::GLsizei, gl::GLenum, gl::GLsizei, const void *> CompressedTextureSubImage1DEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLenum, gl::GLsizei, const void *> CompressedTextureSubImage2D;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLenum, gl::GLsizei, const void *> CompressedTextureSubImage2DEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLsizei, gl::GLenum, gl::GLsizei, const void *> CompressedTextureSubImage3D;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLsizei, gl::GLenum, gl::GLsizei, const void *> CompressedTextureSubImage3DEXT;
    static Function<void, gl::GLenum, gl::GLfloat> ConservativeRasterParameterfNV;
    static Function<void, gl::GLenum, gl::GLint> ConservativeRasterParameteriNV;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLsizei, gl::GLenum, gl::GLenum, const void *> ConvolutionFilter1D;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLsizei, gl::GLenum, gl::GLenum, const void *> ConvolutionFilter1DEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLsizei, gl::GLsizei, gl::GLenum, gl::GLenum, const void *> ConvolutionFilter2D;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLsizei, gl::GLsizei, gl::GLenum, gl::GLenum, const void *> ConvolutionFilter2DEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfloat> ConvolutionParameterf;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfloat> ConvolutionParameterfEXT;
    static Function<void, gl::GLenum, gl::GLenum, const gl::GLfloat *> ConvolutionParameterfv;
    static Function<void, gl::GLenum, gl::GLenum, const gl::GLfloat *> ConvolutionParameterfvEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint> ConvolutionParameteri;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint> ConvolutionParameteriEXT;
    static Function<void, gl::GLenum, gl::GLenum, const gl::GLint *> ConvolutionParameteriv;
    static Function<void, gl::GLenum, gl::GLenum, const gl::GLint *> ConvolutionParameterivEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfixed> ConvolutionParameterxOES;
    static Function<void, gl::GLenum, gl::GLenum, const gl::GLfixed *> ConvolutionParameterxvOES;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLintptr, gl::GLintptr, gl::GLsizeiptr> CopyBufferSubData;
    static Function<void, gl::GLenum, gl::GLsizei, gl::GLint, gl::GLint, gl::GLsizei> CopyColorSubTable;
    static Function<void, gl::GLenum, gl::GLsizei, gl::GLint, gl::GLint, gl::GLsizei> CopyColorSubTableEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint, gl::GLint, gl::GLsizei> CopyColorTable;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint, gl::GLint, gl::GLsizei> CopyColorTableSGI;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint, gl::GLint, gl::GLsizei> CopyConvolutionFilter1D;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint, gl::GLint, gl::GLsizei> CopyConvolutionFilter1DEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei> CopyConvolutionFilter2D;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei> CopyConvolutionFilter2DEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLuint, gl::GLenum, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLsizei> CopyImageSubData;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLuint, gl::GLenum, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLsizei> CopyImageSubDataNV;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint, gl::GLenum, gl::GLint, gl::GLint, gl::GLsizei, gl::GLint> CopyMultiTexImage1DEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint, gl::GLenum, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLint> CopyMultiTexImage2DEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLsizei> CopyMultiTexSubImage1DEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei> CopyMultiTexSubImage2DEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei> CopyMultiTexSubImage3DEXT;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLintptr, gl::GLintptr, gl::GLsizeiptr> CopyNamedBufferSubData;
    static Function<void, gl::GLuint, gl::GLuint> CopyPathNV;
    static Function<void, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLenum> CopyPixels;
    static Function<void, gl::GLenum, gl::GLint, gl::GLenum, gl::GLint, gl::GLint, gl::GLsizei, gl::GLint> CopyTexImage1D;
    static Function<void, gl::GLenum, gl::GLint, gl::GLenum, gl::GLint, gl::GLint, gl::GLsizei, gl::GLint> CopyTexImage1DEXT;
    static Function<void, gl::GLenum, gl::GLint, gl::GLenum, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLint> CopyTexImage2D;
    static Function<void, gl::GLenum, gl::GLint, gl::GLenum, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLint> CopyTexImage2DEXT;
    static Function<void, gl::GLenum, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLsizei> CopyTexSubImage1D;
    static Function<void, gl::GLenum, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLsizei> CopyTexSubImage1DEXT;
    static Function<void, gl::GLenum, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei> CopyTexSubImage2D;
    static Function<void, gl::GLenum, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei> CopyTexSubImage2DEXT;
    static Function<void, gl::GLenum, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei> CopyTexSubImage3D;
    static Function<void, gl::GLenum, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei> CopyTexSubImage3DEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint, gl::GLenum, gl::GLint, gl::GLint, gl::GLsizei, gl::GLint> CopyTextureImage1DEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint, gl::GLenum, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLint> CopyTextureImage2DEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLsizei> CopyTextureSubImage1D;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLsizei> CopyTextureSubImage1DEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei> CopyTextureSubImage2D;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei> CopyTextureSubImage2DEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei> CopyTextureSubImage3D;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei> CopyTextureSubImage3DEXT;
    static Function<void, gl::GLsizei, gl::GLenum, const void *, gl::GLuint, gl::GLenum, gl::GLenum, const gl::GLfloat *> CoverFillPathInstancedNV;
    static Function<void, gl::GLuint, gl::GLenum> CoverFillPathNV;
    static Function<void, gl::GLsizei, gl::GLenum, const void *, gl::GLuint, gl::GLenum, gl::GLenum, const gl::GLfloat *> CoverStrokePathInstancedNV;
    static Function<void, gl::GLuint, gl::GLenum> CoverStrokePathNV;
    static Function<void, gl::GLenum> CoverageModulationNV;
    static Function<void, gl::GLsizei, const gl::GLfloat *> CoverageModulationTableNV;
    static Function<void, gl::GLsizei, gl::GLuint *> CreateBuffers;
    static Function<void, gl::GLsizei, gl::GLuint *> CreateCommandListsNV;
    static Function<void, gl::GLsizei, gl::GLuint *> CreateFramebuffers;
    static Function<void, gl::GLuint, gl::GLuint *> CreatePerfQueryINTEL;
    static Function<gl::GLuint> CreateProgram;
    static Function<gl::GLhandleARB> CreateProgramObjectARB;
    static Function<void, gl::GLsizei, gl::GLuint *> CreateProgramPipelines;
    static Function<void, gl::GLenum, gl::GLsizei, gl::GLuint *> CreateQueries;
    static Function<void, gl::GLsizei, gl::GLuint *> CreateRenderbuffers;
    static Function<void, gl::GLsizei, gl::GLuint *> CreateSamplers;
    static Function<gl::GLuint, gl::GLenum> CreateShader;
    static Function<gl::GLhandleARB, gl::GLenum> CreateShaderObjectARB;
    static Function<gl::GLuint, gl::GLenum, const gl::GLchar *> CreateShaderProgramEXT;
    static Function<gl::GLuint, gl::GLenum, gl::GLsizei, const gl::GLchar *const*> CreateShaderProgramv;
    static Function<void, gl::GLsizei, gl::GLuint *> CreateStatesNV;
    static Function<gl::GLsync, gl::_cl_context *, gl::_cl_event *, gl::UnusedMask> CreateSyncFromCLeventARB;
    static Function<void, gl::GLenum, gl::GLsizei, gl::GLuint *> CreateTextures;
    static Function<void, gl::GLsizei, gl::GLuint *> CreateTransformFeedbacks;
    static Function<void, gl::GLsizei, gl::GLuint *> CreateVertexArrays;
    static Function<void, gl::GLenum> CullFace;
    static Function<void, gl::GLenum, gl::GLdouble *> CullParameterdvEXT;
    static Function<void, gl::GLenum, gl::GLfloat *> CullParameterfvEXT;
    static Function<void, gl::GLint> CurrentPaletteMatrixARB;
    static Function<void, gl::GLDEBUGPROC, const void *> DebugMessageCallback;
    static Function<void, gl::GLDEBUGPROCAMD, void *> DebugMessageCallbackAMD;
    static Function<void, gl::GLDEBUGPROCARB, const void *> DebugMessageCallbackARB;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLsizei, const gl::GLuint *, gl::GLboolean> DebugMessageControl;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLsizei, const gl::GLuint *, gl::GLboolean> DebugMessageControlARB;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLsizei, const gl::GLuint *, gl::GLboolean> DebugMessageEnableAMD;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLuint, gl::GLenum, gl::GLsizei, const gl::GLchar *> DebugMessageInsert;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLuint, gl::GLsizei, const gl::GLchar *> DebugMessageInsertAMD;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLuint, gl::GLenum, gl::GLsizei, const gl::GLchar *> DebugMessageInsertARB;
    static Function<void, gl::FfdMaskSGIX> DeformSGIX;
    static Function<void, gl::GLenum, gl::GLdouble, gl::GLdouble, gl::GLint, gl::GLint, gl::GLdouble, gl::GLdouble, gl::GLint, gl::GLint, gl::GLdouble, gl::GLdouble, gl::GLint, gl::GLint, const gl::GLdouble *> DeformationMap3dSGIX;
    static Function<void, gl::GLenum, gl::GLfloat, gl::GLfloat, gl::GLint, gl::GLint, gl::GLfloat, gl::GLfloat, gl::GLint, gl::GLint, gl::GLfloat, gl::GLfloat, gl::GLint, gl::GLint, const gl::GLfloat *> DeformationMap3fSGIX;
    static Function<void, gl::GLuint, gl::GLsizei> DeleteAsyncMarkersSGIX;
    static Function<void, gl::GLsizei, const gl::GLuint *> DeleteBuffers;
    static Function<void, gl::GLsizei, const gl::GLuint *> DeleteBuffersARB;
    static Function<void, gl::GLsizei, const gl::GLuint *> DeleteCommandListsNV;
    static Function<void, gl::GLsizei, const gl::GLuint *> DeleteFencesAPPLE;
    static Function<void, gl::GLsizei, const gl::GLuint *> DeleteFencesNV;
    static Function<void, gl::GLuint> DeleteFragmentShaderATI;
    static Function<void, gl::GLsizei, const gl::GLuint *> DeleteFramebuffers;
    static Function<void, gl::GLsizei, const gl::GLuint *> DeleteFramebuffersEXT;
    static Function<void, gl::GLuint, gl::GLsizei> DeleteLists;
    static Function<void, gl::GLint, const gl::GLchar *> DeleteNamedStringARB;
    static Function<void, gl::GLenum, gl::GLuint, const gl::GLuint *> DeleteNamesAMD;
    static Function<void, gl::GLhandleARB> DeleteObjectARB;
    static Function<void, gl::GLsizei, const gl::GLuint *> DeleteOcclusionQueriesNV;
    static Function<void, gl::GLuint, gl::GLsizei> DeletePathsNV;
    static Function<void, gl::GLsizei, gl::GLuint *> DeletePerfMonitorsAMD;
    static Function<void, gl::GLuint> DeletePerfQueryINTEL;
    static Function<void, gl::GLuint> DeleteProgram;
    static Function<void, gl::GLsizei, const gl::GLuint *> DeleteProgramPipelines;
    static Function<void, gl::GLsizei, const gl::GLuint *> DeleteProgramsARB;
    static Function<void, gl::GLsizei, const gl::GLuint *> DeleteProgramsNV;
    static Function<void, gl::GLsizei, const gl::GLuint *> DeleteQueries;
    static Function<void, gl::GLsizei, const gl::GLuint *> DeleteQueriesARB;
    static Function<void, gl::GLsizei, const gl::GLuint *> DeleteRenderbuffers;
    static Function<void, gl::GLsizei, const gl::GLuint *> DeleteRenderbuffersEXT;
    static Function<void, gl::GLsizei, const gl::GLuint *> DeleteSamplers;
    static Function<void, gl::GLuint> DeleteShader;
    static Function<void, gl::GLsizei, const gl::GLuint *> DeleteStatesNV;
    static Function<void, gl::GLsync> DeleteSync;
    static Function<void, gl::GLsizei, const gl::GLuint *> DeleteTextures;
    static Function<void, gl::GLsizei, const gl::GLuint *> DeleteTexturesEXT;
    static Function<void, gl::GLsizei, const gl::GLuint *> DeleteTransformFeedbacks;
    static Function<void, gl::GLsizei, const gl::GLuint *> DeleteTransformFeedbacksNV;
    static Function<void, gl::GLsizei, const gl::GLuint *> DeleteVertexArrays;
    static Function<void, gl::GLsizei, const gl::GLuint *> DeleteVertexArraysAPPLE;
    static Function<void, gl::GLuint> DeleteVertexShaderEXT;
    static Function<void, gl::GLclampd, gl::GLclampd> DepthBoundsEXT;
    static Function<void, gl::GLdouble, gl::GLdouble> DepthBoundsdNV;
    static Function<void, gl::GLenum> DepthFunc;
    static Function<void, gl::GLboolean> DepthMask;
    static Function<void, gl::GLdouble, gl::GLdouble> DepthRange;
    static Function<void, gl::GLuint, gl::GLsizei, const gl::GLdouble *> DepthRangeArrayv;
    static Function<void, gl::GLuint, gl::GLdouble, gl::GLdouble> DepthRangeIndexed;
    static Function<void, gl::GLdouble, gl::GLdouble> DepthRangedNV;
    static Function<void, gl::GLfloat, gl::GLfloat> DepthRangef;
    static Function<void, gl::GLclampf, gl::GLclampf> DepthRangefOES;
    static Function<void, gl::GLfixed, gl::GLfixed> DepthRangexOES;
    static Function<void, gl::GLhandleARB, gl::GLhandleARB> DetachObjectARB;
    static Function<void, gl::GLuint, gl::GLuint> DetachShader;
    static Function<void, gl::GLenum, gl::GLsizei, const gl::GLfloat *> DetailTexFuncSGIS;
    static Function<void, gl::GLenum> Disable;
    static Function<void, gl::GLenum> DisableClientState;
    static Function<void, gl::GLenum, gl::GLuint> DisableClientStateIndexedEXT;
    static Function<void, gl::GLenum, gl::GLuint> DisableClientStateiEXT;
    static Function<void, gl::GLenum, gl::GLuint> DisableIndexedEXT;
    static Function<void, gl::GLuint> DisableVariantClientStateEXT;
    static Function<void, gl::GLuint, gl::GLuint> DisableVertexArrayAttrib;
    static Function<void, gl::GLuint, gl::GLuint> DisableVertexArrayAttribEXT;
    static Function<void, gl::GLuint, gl::GLenum> DisableVertexArrayEXT;
    static Function<void, gl::GLuint, gl::GLenum> DisableVertexAttribAPPLE;
    static Function<void, gl::GLuint> DisableVertexAttribArray;
    static Function<void, gl::GLuint> DisableVertexAttribArrayARB;
    static Function<void, gl::GLenum, gl::GLuint> Disablei;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLuint> DispatchCompute;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint> DispatchComputeGroupSizeARB;
    static Function<void, gl::GLintptr> DispatchComputeIndirect;
    static Function<void, gl::GLenum, gl::GLint, gl::GLsizei> DrawArrays;
    static Function<void, gl::GLenum, gl::GLint, gl::GLsizei> DrawArraysEXT;
    static Function<void, gl::GLenum, const void *> DrawArraysIndirect;
    static Function<void, gl::GLenum, gl::GLint, gl::GLsizei, gl::GLsizei> DrawArraysInstanced;
    static Function<void, gl::GLenum, gl::GLint, gl::GLsizei, gl::GLsizei> DrawArraysInstancedARB;
    static Function<void, gl::GLenum, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLuint> DrawArraysInstancedBaseInstance;
    static Function<void, gl::GLenum, gl::GLint, gl::GLsizei, gl::GLsizei> DrawArraysInstancedEXT;
    static Function<void, gl::GLenum> DrawBuffer;
    static Function<void, gl::GLsizei, const gl::GLenum *> DrawBuffers;
    static Function<void, gl::GLsizei, const gl::GLenum *> DrawBuffersARB;
    static Function<void, gl::GLsizei, const gl::GLenum *> DrawBuffersATI;
    static Function<void, gl::GLenum, const gl::GLuint64 *, const gl::GLsizei *, gl::GLuint> DrawCommandsAddressNV;
    static Function<void, gl::GLenum, gl::GLuint, const gl::GLintptr *, const gl::GLsizei *, gl::GLuint> DrawCommandsNV;
    static Function<void, const gl::GLuint64 *, const gl::GLsizei *, const gl::GLuint *, const gl::GLuint *, gl::GLuint> DrawCommandsStatesAddressNV;
    static Function<void, gl::GLuint, const gl::GLintptr *, const gl::GLsizei *, const gl::GLuint *, const gl::GLuint *, gl::GLuint> DrawCommandsStatesNV;
    static Function<void, gl::GLenum, gl::GLint, gl::GLsizei> DrawElementArrayAPPLE;
    static Function<void, gl::GLenum, gl::GLsizei> DrawElementArrayATI;
    static Function<void, gl::GLenum, gl::GLsizei, gl::GLenum, const void *> DrawElements;
    static Function<void, gl::GLenum, gl::GLsizei, gl::GLenum, const void *, gl::GLint> DrawElementsBaseVertex;
    static Function<void, gl::GLenum, gl::GLenum, const void *> DrawElementsIndirect;
    static Function<void, gl::GLenum, gl::GLsizei, gl::GLenum, const void *, gl::GLsizei> DrawElementsInstanced;
    static Function<void, gl::GLenum, gl::GLsizei, gl::GLenum, const void *, gl::GLsizei> DrawElementsInstancedARB;
    static Function<void, gl::GLenum, gl::GLsizei, gl::GLenum, const void *, gl::GLsizei, gl::GLuint> DrawElementsInstancedBaseInstance;
    static Function<void, gl::GLenum, gl::GLsizei, gl::GLenum, const void *, gl::GLsizei, gl::GLint> DrawElementsInstancedBaseVertex;
    static Function<void, gl::GLenum, gl::GLsizei, gl::GLenum, const void *, gl::GLsizei, gl::GLint, gl::GLuint> DrawElementsInstancedBaseVertexBaseInstance;
    static Function<void, gl::GLenum, gl::GLsizei, gl::GLenum, const void *, gl::GLsizei> DrawElementsInstancedEXT;
    static Function<void, gl::GLenum, gl::GLint, gl::GLsizei, gl::GLsizei> DrawMeshArraysSUN;
    static Function<void, gl::GLsizei, gl::GLsizei, gl::GLenum, gl::GLenum, const void *> DrawPixels;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLuint, gl::GLint, gl::GLsizei> DrawRangeElementArrayAPPLE;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLuint, gl::GLsizei> DrawRangeElementArrayATI;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLuint, gl::GLsizei, gl::GLenum, const void *> DrawRangeElements;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLuint, gl::GLsizei, gl::GLenum, const void *, gl::GLint> DrawRangeElementsBaseVertex;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLuint, gl::GLsizei, gl::GLenum, const void *> DrawRangeElementsEXT;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat> DrawTextureNV;
    static Function<void, gl::GLenum, gl::GLuint> DrawTransformFeedback;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLsizei> DrawTransformFeedbackInstanced;
    static Function<void, gl::GLenum, gl::GLuint> DrawTransformFeedbackNV;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLuint> DrawTransformFeedbackStream;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLuint, gl::GLsizei> DrawTransformFeedbackStreamInstanced;
    static Function<void, gl::GLboolean> EdgeFlag;
    static Function<void, gl::GLsizei> EdgeFlagFormatNV;
    static Function<void, gl::GLsizei, const void *> EdgeFlagPointer;
    static Function<void, gl::GLsizei, gl::GLsizei, const gl::GLboolean *> EdgeFlagPointerEXT;
    static Function<void, gl::GLint, const gl::GLboolean **, gl::GLint> EdgeFlagPointerListIBM;
    static Function<void, const gl::GLboolean *> EdgeFlagv;
    static Function<void, gl::GLenum, const void *> ElementPointerAPPLE;
    static Function<void, gl::GLenum, const void *> ElementPointerATI;
    static Function<void, gl::GLenum> Enable;
    static Function<void, gl::GLenum> EnableClientState;
    static Function<void, gl::GLenum, gl::GLuint> EnableClientStateIndexedEXT;
    static Function<void, gl::GLenum, gl::GLuint> EnableClientStateiEXT;
    static Function<void, gl::GLenum, gl::GLuint> EnableIndexedEXT;
    static Function<void, gl::GLuint> EnableVariantClientStateEXT;
    static Function<void, gl::GLuint, gl::GLuint> EnableVertexArrayAttrib;
    static Function<void, gl::GLuint, gl::GLuint> EnableVertexArrayAttribEXT;
    static Function<void, gl::GLuint, gl::GLenum> EnableVertexArrayEXT;
    static Function<void, gl::GLuint, gl::GLenum> EnableVertexAttribAPPLE;
    static Function<void, gl::GLuint> EnableVertexAttribArray;
    static Function<void, gl::GLuint> EnableVertexAttribArrayARB;
    static Function<void, gl::GLenum, gl::GLuint> Enablei;
    static Function<void> End;
    static Function<void> EndConditionalRender;
    static Function<void> EndConditionalRenderNV;
    static Function<void> EndConditionalRenderNVX;
    static Function<void> EndFragmentShaderATI;
    static Function<void> EndList;
    static Function<void> EndOcclusionQueryNV;
    static Function<void, gl::GLuint> EndPerfMonitorAMD;
    static Function<void, gl::GLuint> EndPerfQueryINTEL;
    static Function<void, gl::GLenum> EndQuery;
    static Function<void, gl::GLenum> EndQueryARB;
    static Function<void, gl::GLenum, gl::GLuint> EndQueryIndexed;
    static Function<void> EndTransformFeedback;
    static Function<void> EndTransformFeedbackEXT;
    static Function<void> EndTransformFeedbackNV;
    static Function<void> EndVertexShaderEXT;
    static Function<void, gl::GLuint> EndVideoCaptureNV;
    static Function<void, gl::GLdouble> EvalCoord1d;
    static Function<void, const gl::GLdouble *> EvalCoord1dv;
    static Function<void, gl::GLfloat> EvalCoord1f;
    static Function<void, const gl::GLfloat *> EvalCoord1fv;
    static Function<void, gl::GLfixed> EvalCoord1xOES;
    static Function<void, const gl::GLfixed *> EvalCoord1xvOES;
    static Function<void, gl::GLdouble, gl::GLdouble> EvalCoord2d;
    static Function<void, const gl::GLdouble *> EvalCoord2dv;
    static Function<void, gl::GLfloat, gl::GLfloat> EvalCoord2f;
    static Function<void, const gl::GLfloat *> EvalCoord2fv;
    static Function<void, gl::GLfixed, gl::GLfixed> EvalCoord2xOES;
    static Function<void, const gl::GLfixed *> EvalCoord2xvOES;
    static Function<void, gl::GLenum, gl::GLenum> EvalMapsNV;
    static Function<void, gl::GLenum, gl::GLint, gl::GLint> EvalMesh1;
    static Function<void, gl::GLenum, gl::GLint, gl::GLint, gl::GLint, gl::GLint> EvalMesh2;
    static Function<void, gl::GLint> EvalPoint1;
    static Function<void, gl::GLint, gl::GLint> EvalPoint2;
    static Function<void> EvaluateDepthValuesARB;
    static Function<void, gl::GLenum, gl::GLuint, const gl::GLfloat *> ExecuteProgramNV;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLuint> ExtractComponentEXT;
    static Function<void, gl::GLsizei, gl::GLenum, gl::GLfloat *> FeedbackBuffer;
    static Function<void, gl::GLsizei, gl::GLenum, const gl::GLfixed *> FeedbackBufferxOES;
    static Function<gl::GLsync, gl::GLenum, gl::UnusedMask> FenceSync;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLenum> FinalCombinerInputNV;
    static Function<void> Finish;
    static Function<gl::GLint, gl::GLuint *> FinishAsyncSGIX;
    static Function<void, gl::GLuint> FinishFenceAPPLE;
    static Function<void, gl::GLuint> FinishFenceNV;
    static Function<void, gl::GLenum, gl::GLint> FinishObjectAPPLE;
    static Function<void> FinishTextureSUNX;
    static Function<void> Flush;
    static Function<void, gl::GLenum, gl::GLintptr, gl::GLsizeiptr> FlushMappedBufferRange;
    static Function<void, gl::GLenum, gl::GLintptr, gl::GLsizeiptr> FlushMappedBufferRangeAPPLE;
    static Function<void, gl::GLuint, gl::GLintptr, gl::GLsizeiptr> FlushMappedNamedBufferRange;
    static Function<void, gl::GLuint, gl::GLintptr, gl::GLsizeiptr> FlushMappedNamedBufferRangeEXT;
    static Function<void, gl::GLenum> FlushPixelDataRangeNV;
    static Function<void> FlushRasterSGIX;
    static Function<void, gl::GLenum> FlushStaticDataIBM;
    static Function<void, gl::GLsizei, void *> FlushVertexArrayRangeAPPLE;
    static Function<void> FlushVertexArrayRangeNV;
    static Function<void, gl::GLenum, gl::GLsizei> FogCoordFormatNV;
    static Function<void, gl::GLenum, gl::GLsizei, const void *> FogCoordPointer;
    static Function<void, gl::GLenum, gl::GLsizei, const void *> FogCoordPointerEXT;
    static Function<void, gl::GLenum, gl::GLint, const void **, gl::GLint> FogCoordPointerListIBM;
    static Function<void, gl::GLdouble> FogCoordd;
    static Function<void, gl::GLdouble> FogCoorddEXT;
    static Function<void, const gl::GLdouble *> FogCoorddv;
    static Function<void, const gl::GLdouble *> FogCoorddvEXT;
    static Function<void, gl::GLfloat> FogCoordf;
    static Function<void, gl::GLfloat> FogCoordfEXT;
    static Function<void, const gl::GLfloat *> FogCoordfv;
    static Function<void, const gl::GLfloat *> FogCoordfvEXT;
    static Function<void, gl::GLhalfNV> FogCoordhNV;
    static Function<void, const gl::GLhalfNV *> FogCoordhvNV;
    static Function<void, gl::GLsizei, const gl::GLfloat *> FogFuncSGIS;
    static Function<void, gl::GLenum, gl::GLfloat> Fogf;
    static Function<void, gl::GLenum, const gl::GLfloat *> Fogfv;
    static Function<void, gl::GLenum, gl::GLint> Fogi;
    static Function<void, gl::GLenum, const gl::GLint *> Fogiv;
    static Function<void, gl::GLenum, gl::GLfixed> FogxOES;
    static Function<void, gl::GLenum, const gl::GLfixed *> FogxvOES;
    static Function<void, gl::GLenum, gl::GLenum> FragmentColorMaterialSGIX;
    static Function<void, gl::GLuint> FragmentCoverageColorNV;
    static Function<void, gl::GLenum, gl::GLfloat> FragmentLightModelfSGIX;
    static Function<void, gl::GLenum, const gl::GLfloat *> FragmentLightModelfvSGIX;
    static Function<void, gl::GLenum, gl::GLint> FragmentLightModeliSGIX;
    static Function<void, gl::GLenum, const gl::GLint *> FragmentLightModelivSGIX;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfloat> FragmentLightfSGIX;
    static Function<void, gl::GLenum, gl::GLenum, const gl::GLfloat *> FragmentLightfvSGIX;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint> FragmentLightiSGIX;
    static Function<void, gl::GLenum, gl::GLenum, const gl::GLint *> FragmentLightivSGIX;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfloat> FragmentMaterialfSGIX;
    static Function<void, gl::GLenum, gl::GLenum, const gl::GLfloat *> FragmentMaterialfvSGIX;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint> FragmentMaterialiSGIX;
    static Function<void, gl::GLenum, gl::GLenum, const gl::GLint *> FragmentMaterialivSGIX;
    static Function<void> FrameTerminatorGREMEDY;
    static Function<void, gl::GLint> FrameZoomSGIX;
    static Function<void, gl::GLuint, gl::GLenum> FramebufferDrawBufferEXT;
    static Function<void, gl::GLuint, gl::GLsizei, const gl::GLenum *> FramebufferDrawBuffersEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint> FramebufferParameteri;
    static Function<void, gl::GLuint, gl::GLenum> FramebufferReadBufferEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLuint> FramebufferRenderbuffer;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLuint> FramebufferRenderbufferEXT;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLsizei, const gl::GLfloat *> FramebufferSampleLocationsfvARB;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLsizei, const gl::GLfloat *> FramebufferSampleLocationsfvNV;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLuint, gl::GLint> FramebufferTexture;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLuint, gl::GLint> FramebufferTexture1D;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLuint, gl::GLint> FramebufferTexture1DEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLuint, gl::GLint> FramebufferTexture2D;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLuint, gl::GLint> FramebufferTexture2DEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLuint, gl::GLint, gl::GLint> FramebufferTexture3D;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLuint, gl::GLint, gl::GLint> FramebufferTexture3DEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLuint, gl::GLint> FramebufferTextureARB;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLuint, gl::GLint> FramebufferTextureEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLuint, gl::GLint, gl::GLenum> FramebufferTextureFaceARB;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLuint, gl::GLint, gl::GLenum> FramebufferTextureFaceEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLuint, gl::GLint, gl::GLint> FramebufferTextureLayer;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLuint, gl::GLint, gl::GLint> FramebufferTextureLayerARB;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLuint, gl::GLint, gl::GLint> FramebufferTextureLayerEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLuint, gl::GLint, gl::GLint, gl::GLsizei> FramebufferTextureMultiviewOVR;
    static Function<void, gl::GLuint> FreeObjectBufferATI;
    static Function<void, gl::GLenum> FrontFace;
    static Function<void, gl::GLdouble, gl::GLdouble, gl::GLdouble, gl::GLdouble, gl::GLdouble, gl::GLdouble> Frustum;
    static Function<void, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat> FrustumfOES;
    static Function<void, gl::GLfixed, gl::GLfixed, gl::GLfixed, gl::GLfixed, gl::GLfixed, gl::GLfixed> FrustumxOES;
    static Function<gl::GLuint, gl::GLsizei> GenAsyncMarkersSGIX;
    static Function<void, gl::GLsizei, gl::GLuint *> GenBuffers;
    static Function<void, gl::GLsizei, gl::GLuint *> GenBuffersARB;
    static Function<void, gl::GLsizei, gl::GLuint *> GenFencesAPPLE;
    static Function<void, gl::GLsizei, gl::GLuint *> GenFencesNV;
    static Function<gl::GLuint, gl::GLuint> GenFragmentShadersATI;
    static Function<void, gl::GLsizei, gl::GLuint *> GenFramebuffers;
    static Function<void, gl::GLsizei, gl::GLuint *> GenFramebuffersEXT;
    static Function<gl::GLuint, gl::GLsizei> GenLists;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLuint *> GenNamesAMD;
    static Function<void, gl::GLsizei, gl::GLuint *> GenOcclusionQueriesNV;
    static Function<gl::GLuint, gl::GLsizei> GenPathsNV;
    static Function<void, gl::GLsizei, gl::GLuint *> GenPerfMonitorsAMD;
    static Function<void, gl::GLsizei, gl::GLuint *> GenProgramPipelines;
    static Function<void, gl::GLsizei, gl::GLuint *> GenProgramsARB;
    static Function<void, gl::GLsizei, gl::GLuint *> GenProgramsNV;
    static Function<void, gl::GLsizei, gl::GLuint *> GenQueries;
    static Function<void, gl::GLsizei, gl::GLuint *> GenQueriesARB;
    static Function<void, gl::GLsizei, gl::GLuint *> GenRenderbuffers;
    static Function<void, gl::GLsizei, gl::GLuint *> GenRenderbuffersEXT;
    static Function<void, gl::GLsizei, gl::GLuint *> GenSamplers;
    static Function<gl::GLuint, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLuint> GenSymbolsEXT;
    static Function<void, gl::GLsizei, gl::GLuint *> GenTextures;
    static Function<void, gl::GLsizei, gl::GLuint *> GenTexturesEXT;
    static Function<void, gl::GLsizei, gl::GLuint *> GenTransformFeedbacks;
    static Function<void, gl::GLsizei, gl::GLuint *> GenTransformFeedbacksNV;
    static Function<void, gl::GLsizei, gl::GLuint *> GenVertexArrays;
    static Function<void, gl::GLsizei, gl::GLuint *> GenVertexArraysAPPLE;
    static Function<gl::GLuint, gl::GLuint> GenVertexShadersEXT;
    static Function<void, gl::GLenum> GenerateMipmap;
    static Function<void, gl::GLenum> GenerateMipmapEXT;
    static Function<void, gl::GLenum, gl::GLenum> GenerateMultiTexMipmapEXT;
    static Function<void, gl::GLuint> GenerateTextureMipmap;
    static Function<void, gl::GLuint, gl::GLenum> GenerateTextureMipmapEXT;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLenum, gl::GLint *> GetActiveAtomicCounterBufferiv;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLsizei, gl::GLsizei *, gl::GLint *, gl::GLenum *, gl::GLchar *> GetActiveAttrib;
    static Function<void, gl::GLhandleARB, gl::GLuint, gl::GLsizei, gl::GLsizei *, gl::GLint *, gl::GLenum *, gl::GLcharARB *> GetActiveAttribARB;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLuint, gl::GLsizei, gl::GLsizei *, gl::GLchar *> GetActiveSubroutineName;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLuint, gl::GLsizei, gl::GLsizei *, gl::GLchar *> GetActiveSubroutineUniformName;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLuint, gl::GLenum, gl::GLint *> GetActiveSubroutineUniformiv;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLsizei, gl::GLsizei *, gl::GLint *, gl::GLenum *, gl::GLchar *> GetActiveUniform;
    static Function<void, gl::GLhandleARB, gl::GLuint, gl::GLsizei, gl::GLsizei *, gl::GLint *, gl::GLenum *, gl::GLcharARB *> GetActiveUniformARB;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLsizei, gl::GLsizei *, gl::GLchar *> GetActiveUniformBlockName;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLenum, gl::GLint *> GetActiveUniformBlockiv;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLsizei, gl::GLsizei *, gl::GLchar *> GetActiveUniformName;
    static Function<void, gl::GLuint, gl::GLsizei, const gl::GLuint *, gl::GLenum, gl::GLint *> GetActiveUniformsiv;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLsizei, gl::GLsizei *, gl::GLsizei *, gl::GLenum *, gl::GLchar *> GetActiveVaryingNV;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfloat *> GetArrayObjectfvATI;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint *> GetArrayObjectivATI;
    static Function<void, gl::GLhandleARB, gl::GLsizei, gl::GLsizei *, gl::GLhandleARB *> GetAttachedObjectsARB;
    static Function<void, gl::GLuint, gl::GLsizei, gl::GLsizei *, gl::GLuint *> GetAttachedShaders;
    static Function<gl::GLint, gl::GLuint, const gl::GLchar *> GetAttribLocation;
    static Function<gl::GLint, gl::GLhandleARB, const gl::GLcharARB *> GetAttribLocationARB;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLboolean *> GetBooleanIndexedvEXT;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLboolean *> GetBooleani_v;
    static Function<void, gl::GLenum, gl::GLboolean *> GetBooleanv;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint64 *> GetBufferParameteri64v;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint *> GetBufferParameteriv;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint *> GetBufferParameterivARB;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLuint64EXT *> GetBufferParameterui64vNV;
    static Function<void, gl::GLenum, gl::GLenum, void **> GetBufferPointerv;
    static Function<void, gl::GLenum, gl::GLenum, void **> GetBufferPointervARB;
    static Function<void, gl::GLenum, gl::GLintptr, gl::GLsizeiptr, void *> GetBufferSubData;
    static Function<void, gl::GLenum, gl::GLintptrARB, gl::GLsizeiptrARB, void *> GetBufferSubDataARB;
    static Function<void, gl::GLenum, gl::GLdouble *> GetClipPlane;
    static Function<void, gl::GLenum, gl::GLfloat *> GetClipPlanefOES;
    static Function<void, gl::GLenum, gl::GLfixed *> GetClipPlanexOES;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, void *> GetColorTable;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, void *> GetColorTableEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfloat *> GetColorTableParameterfv;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfloat *> GetColorTableParameterfvEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfloat *> GetColorTableParameterfvSGI;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint *> GetColorTableParameteriv;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint *> GetColorTableParameterivEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint *> GetColorTableParameterivSGI;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, void *> GetColorTableSGI;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLfloat *> GetCombinerInputParameterfvNV;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLint *> GetCombinerInputParameterivNV;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLfloat *> GetCombinerOutputParameterfvNV;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLint *> GetCombinerOutputParameterivNV;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfloat *> GetCombinerStageParameterfvNV;
    static Function<gl::GLuint, gl::GLenum, gl::GLuint> GetCommandHeaderNV;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint, void *> GetCompressedMultiTexImageEXT;
    static Function<void, gl::GLenum, gl::GLint, void *> GetCompressedTexImage;
    static Function<void, gl::GLenum, gl::GLint, void *> GetCompressedTexImageARB;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, void *> GetCompressedTextureImage;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint, void *> GetCompressedTextureImageEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLsizei, gl::GLsizei, void *> GetCompressedTextureSubImage;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, void *> GetConvolutionFilter;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, void *> GetConvolutionFilterEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfloat *> GetConvolutionParameterfv;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfloat *> GetConvolutionParameterfvEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint *> GetConvolutionParameteriv;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint *> GetConvolutionParameterivEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfixed *> GetConvolutionParameterxvOES;
    static Function<void, gl::GLsizei, gl::GLfloat *> GetCoverageModulationTableNV;
    static Function<gl::GLuint, gl::GLuint, gl::GLsizei, gl::GLenum *, gl::GLenum *, gl::GLuint *, gl::GLenum *, gl::GLsizei *, gl::GLchar *> GetDebugMessageLog;
    static Function<gl::GLuint, gl::GLuint, gl::GLsizei, gl::GLenum *, gl::GLuint *, gl::GLuint *, gl::GLsizei *, gl::GLchar *> GetDebugMessageLogAMD;
    static Function<gl::GLuint, gl::GLuint, gl::GLsizei, gl::GLenum *, gl::GLenum *, gl::GLuint *, gl::GLenum *, gl::GLsizei *, gl::GLchar *> GetDebugMessageLogARB;
    static Function<void, gl::GLenum, gl::GLfloat *> GetDetailTexFuncSGIS;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLdouble *> GetDoubleIndexedvEXT;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLdouble *> GetDoublei_v;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLdouble *> GetDoublei_vEXT;
    static Function<void, gl::GLenum, gl::GLdouble *> GetDoublev;
    static Function<gl::GLenum> GetError;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint *> GetFenceivNV;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfloat *> GetFinalCombinerInputParameterfvNV;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint *> GetFinalCombinerInputParameterivNV;
    static Function<void, gl::GLuint *> GetFirstPerfQueryIdINTEL;
    static Function<void, gl::GLenum, gl::GLfixed *> GetFixedvOES;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLfloat *> GetFloatIndexedvEXT;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLfloat *> GetFloati_v;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLfloat *> GetFloati_vEXT;
    static Function<void, gl::GLenum, gl::GLfloat *> GetFloatv;
    static Function<void, gl::GLfloat *> GetFogFuncSGIS;
    static Function<gl::GLint, gl::GLuint, const gl::GLchar *> GetFragDataIndex;
    static Function<gl::GLint, gl::GLuint, const gl::GLchar *> GetFragDataLocation;
    static Function<gl::GLint, gl::GLuint, const gl::GLchar *> GetFragDataLocationEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfloat *> GetFragmentLightfvSGIX;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint *> GetFragmentLightivSGIX;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfloat *> GetFragmentMaterialfvSGIX;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint *> GetFragmentMaterialivSGIX;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLint *> GetFramebufferAttachmentParameteriv;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLint *> GetFramebufferAttachmentParameterivEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint *> GetFramebufferParameteriv;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint *> GetFramebufferParameterivEXT;
    static Function<gl::GLenum> GetGraphicsResetStatus;
    static Function<gl::GLenum> GetGraphicsResetStatusARB;
    static Function<gl::GLhandleARB, gl::GLenum> GetHandleARB;
    static Function<void, gl::GLenum, gl::GLboolean, gl::GLenum, gl::GLenum, void *> GetHistogram;
    static Function<void, gl::GLenum, gl::GLboolean, gl::GLenum, gl::GLenum, void *> GetHistogramEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfloat *> GetHistogramParameterfv;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfloat *> GetHistogramParameterfvEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint *> GetHistogramParameteriv;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint *> GetHistogramParameterivEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfixed *> GetHistogramParameterxvOES;
    static Function<gl::GLuint64, gl::GLuint, gl::GLint, gl::GLboolean, gl::GLint, gl::GLenum> GetImageHandleARB;
    static Function<gl::GLuint64, gl::GLuint, gl::GLint, gl::GLboolean, gl::GLint, gl::GLenum> GetImageHandleNV;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfloat *> GetImageTransformParameterfvHP;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint *> GetImageTransformParameterivHP;
    static Function<void, gl::GLhandleARB, gl::GLsizei, gl::GLsizei *, gl::GLcharARB *> GetInfoLogARB;
    static Function<gl::GLint> GetInstrumentsSGIX;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLint64 *> GetInteger64i_v;
    static Function<void, gl::GLenum, gl::GLint64 *> GetInteger64v;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLint *> GetIntegerIndexedvEXT;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLint *> GetIntegeri_v;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLuint64EXT *> GetIntegerui64i_vNV;
    static Function<void, gl::GLenum, gl::GLuint64EXT *> GetIntegerui64vNV;
    static Function<void, gl::GLenum, gl::GLint *> GetIntegerv;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLsizei, gl::GLenum, gl::GLsizei, gl::GLint *> GetInternalformatSampleivNV;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLsizei, gl::GLint64 *> GetInternalformati64v;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLsizei, gl::GLint *> GetInternalformativ;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLboolean *> GetInvariantBooleanvEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLfloat *> GetInvariantFloatvEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint *> GetInvariantIntegervEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfloat *> GetLightfv;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint *> GetLightiv;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfixed *> GetLightxOES;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLfloat *> GetListParameterfvSGIX;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint *> GetListParameterivSGIX;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLboolean *> GetLocalConstantBooleanvEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLfloat *> GetLocalConstantFloatvEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint *> GetLocalConstantIntegervEXT;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLenum, gl::GLfloat *> GetMapAttribParameterfvNV;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLenum, gl::GLint *> GetMapAttribParameterivNV;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLenum, gl::GLsizei, gl::GLsizei, gl::GLboolean, void *> GetMapControlPointsNV;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfloat *> GetMapParameterfvNV;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint *> GetMapParameterivNV;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLdouble *> GetMapdv;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfloat *> GetMapfv;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint *> GetMapiv;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfixed *> GetMapxvOES;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfloat *> GetMaterialfv;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint *> GetMaterialiv;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfixed> GetMaterialxOES;
    static Function<void, gl::GLenum, gl::GLboolean, gl::GLenum, gl::GLenum, void *> GetMinmax;
    static Function<void, gl::GLenum, gl::GLboolean, gl::GLenum, gl::GLenum, void *> GetMinmaxEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfloat *> GetMinmaxParameterfv;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfloat *> GetMinmaxParameterfvEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint *> GetMinmaxParameteriv;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint *> GetMinmaxParameterivEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLfloat *> GetMultiTexEnvfvEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLint *> GetMultiTexEnvivEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLdouble *> GetMultiTexGendvEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLfloat *> GetMultiTexGenfvEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLint *> GetMultiTexGenivEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint, gl::GLenum, gl::GLenum, void *> GetMultiTexImageEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint, gl::GLenum, gl::GLfloat *> GetMultiTexLevelParameterfvEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint, gl::GLenum, gl::GLint *> GetMultiTexLevelParameterivEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLint *> GetMultiTexParameterIivEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLuint *> GetMultiTexParameterIuivEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLfloat *> GetMultiTexParameterfvEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLint *> GetMultiTexParameterivEXT;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLfloat *> GetMultisamplefv;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLfloat *> GetMultisamplefvNV;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint64 *> GetNamedBufferParameteri64v;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint *> GetNamedBufferParameteriv;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint *> GetNamedBufferParameterivEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLuint64EXT *> GetNamedBufferParameterui64vNV;
    static Function<void, gl::GLuint, gl::GLenum, void **> GetNamedBufferPointerv;
    static Function<void, gl::GLuint, gl::GLenum, void **> GetNamedBufferPointervEXT;
    static Function<void, gl::GLuint, gl::GLintptr, gl::GLsizeiptr, void *> GetNamedBufferSubData;
    static Function<void, gl::GLuint, gl::GLintptr, gl::GLsizeiptr, void *> GetNamedBufferSubDataEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLenum, gl::GLint *> GetNamedFramebufferAttachmentParameteriv;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLenum, gl::GLint *> GetNamedFramebufferAttachmentParameterivEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint *> GetNamedFramebufferParameteriv;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint *> GetNamedFramebufferParameterivEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLuint, gl::GLint *> GetNamedProgramLocalParameterIivEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLuint, gl::GLuint *> GetNamedProgramLocalParameterIuivEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLuint, gl::GLdouble *> GetNamedProgramLocalParameterdvEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLuint, gl::GLfloat *> GetNamedProgramLocalParameterfvEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLenum, void *> GetNamedProgramStringEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLenum, gl::GLint *> GetNamedProgramivEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint *> GetNamedRenderbufferParameteriv;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint *> GetNamedRenderbufferParameterivEXT;
    static Function<void, gl::GLint, const gl::GLchar *, gl::GLsizei, gl::GLint *, gl::GLchar *> GetNamedStringARB;
    static Function<void, gl::GLint, const gl::GLchar *, gl::GLenum, gl::GLint *> GetNamedStringivARB;
    static Function<void, gl::GLuint, gl::GLuint *> GetNextPerfQueryIdINTEL;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLfloat *> GetObjectBufferfvATI;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint *> GetObjectBufferivATI;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLsizei, gl::GLsizei *, gl::GLchar *> GetObjectLabel;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLsizei, gl::GLsizei *, gl::GLchar *> GetObjectLabelEXT;
    static Function<void, gl::GLhandleARB, gl::GLenum, gl::GLfloat *> GetObjectParameterfvARB;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLenum, gl::GLint *> GetObjectParameterivAPPLE;
    static Function<void, gl::GLhandleARB, gl::GLenum, gl::GLint *> GetObjectParameterivARB;
    static Function<void, const void *, gl::GLsizei, gl::GLsizei *, gl::GLchar *> GetObjectPtrLabel;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint *> GetOcclusionQueryivNV;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLuint *> GetOcclusionQueryuivNV;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfloat *> GetPathColorGenfvNV;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint *> GetPathColorGenivNV;
    static Function<void, gl::GLuint, gl::GLubyte *> GetPathCommandsNV;
    static Function<void, gl::GLuint, gl::GLfloat *> GetPathCoordsNV;
    static Function<void, gl::GLuint, gl::GLfloat *> GetPathDashArrayNV;
    static Function<gl::GLfloat, gl::GLuint, gl::GLsizei, gl::GLsizei> GetPathLengthNV;
    static Function<void, gl::PathRenderingMaskNV, gl::GLuint, gl::GLsizei, gl::GLsizei, gl::GLfloat *> GetPathMetricRangeNV;
    static Function<void, gl::PathRenderingMaskNV, gl::GLsizei, gl::GLenum, const void *, gl::GLuint, gl::GLsizei, gl::GLfloat *> GetPathMetricsNV;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLfloat *> GetPathParameterfvNV;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint *> GetPathParameterivNV;
    static Function<void, gl::GLenum, gl::GLsizei, gl::GLenum, const void *, gl::GLuint, gl::GLfloat, gl::GLfloat, gl::GLenum, gl::GLfloat *> GetPathSpacingNV;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfloat *> GetPathTexGenfvNV;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint *> GetPathTexGenivNV;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLchar *, gl::GLuint, gl::GLchar *, gl::GLuint *, gl::GLuint *, gl::GLuint *, gl::GLuint *, gl::GLuint64 *> GetPerfCounterInfoINTEL;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLsizei, gl::GLuint *, gl::GLint *> GetPerfMonitorCounterDataAMD;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLenum, void *> GetPerfMonitorCounterInfoAMD;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLsizei, gl::GLsizei *, gl::GLchar *> GetPerfMonitorCounterStringAMD;
    static Function<void, gl::GLuint, gl::GLint *, gl::GLint *, gl::GLsizei, gl::GLuint *> GetPerfMonitorCountersAMD;
    static Function<void, gl::GLuint, gl::GLsizei, gl::GLsizei *, gl::GLchar *> GetPerfMonitorGroupStringAMD;
    static Function<void, gl::GLint *, gl::GLsizei, gl::GLuint *> GetPerfMonitorGroupsAMD;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLsizei, gl::GLvoid *, gl::GLuint *> GetPerfQueryDataINTEL;
    static Function<void, gl::GLchar *, gl::GLuint *> GetPerfQueryIdByNameINTEL;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLchar *, gl::GLuint *, gl::GLuint *, gl::GLuint *, gl::GLuint *> GetPerfQueryInfoINTEL;
    static Function<void, gl::GLenum, gl::GLfloat *> GetPixelMapfv;
    static Function<void, gl::GLenum, gl::GLuint *> GetPixelMapuiv;
    static Function<void, gl::GLenum, gl::GLushort *> GetPixelMapusv;
    static Function<void, gl::GLenum, gl::GLint, gl::GLfixed *> GetPixelMapxv;
    static Function<void, gl::GLenum, gl::GLfloat *> GetPixelTexGenParameterfvSGIS;
    static Function<void, gl::GLenum, gl::GLint *> GetPixelTexGenParameterivSGIS;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfloat *> GetPixelTransformParameterfvEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint *> GetPixelTransformParameterivEXT;
    static Function<void, gl::GLenum, gl::GLuint, void **> GetPointerIndexedvEXT;
    static Function<void, gl::GLenum, gl::GLuint, void **> GetPointeri_vEXT;
    static Function<void, gl::GLenum, void **> GetPointerv;
    static Function<void, gl::GLenum, void **> GetPointervEXT;
    static Function<void, gl::GLubyte *> GetPolygonStipple;
    static Function<void, gl::GLuint, gl::GLsizei, gl::GLsizei *, gl::GLenum *, void *> GetProgramBinary;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLint *> GetProgramEnvParameterIivNV;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLuint *> GetProgramEnvParameterIuivNV;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLdouble *> GetProgramEnvParameterdvARB;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLfloat *> GetProgramEnvParameterfvARB;
    static Function<void, gl::GLuint, gl::GLsizei, gl::GLsizei *, gl::GLchar *> GetProgramInfoLog;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLenum, gl::GLint *> GetProgramInterfaceiv;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLint *> GetProgramLocalParameterIivNV;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLuint *> GetProgramLocalParameterIuivNV;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLdouble *> GetProgramLocalParameterdvARB;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLfloat *> GetProgramLocalParameterfvARB;
    static Function<void, gl::GLuint, gl::GLsizei, const gl::GLubyte *, gl::GLdouble *> GetProgramNamedParameterdvNV;
    static Function<void, gl::GLuint, gl::GLsizei, const gl::GLubyte *, gl::GLfloat *> GetProgramNamedParameterfvNV;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLenum, gl::GLdouble *> GetProgramParameterdvNV;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLenum, gl::GLfloat *> GetProgramParameterfvNV;
    static Function<void, gl::GLuint, gl::GLsizei, gl::GLsizei *, gl::GLchar *> GetProgramPipelineInfoLog;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint *> GetProgramPipelineiv;
    static Function<gl::GLuint, gl::GLuint, gl::GLenum, const gl::GLchar *> GetProgramResourceIndex;
    static Function<gl::GLint, gl::GLuint, gl::GLenum, const gl::GLchar *> GetProgramResourceLocation;
    static Function<gl::GLint, gl::GLuint, gl::GLenum, const gl::GLchar *> GetProgramResourceLocationIndex;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLuint, gl::GLsizei, gl::GLsizei *, gl::GLchar *> GetProgramResourceName;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLuint, gl::GLsizei, const gl::GLenum *, gl::GLsizei, gl::GLsizei *, gl::GLfloat *> GetProgramResourcefvNV;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLuint, gl::GLsizei, const gl::GLenum *, gl::GLsizei, gl::GLsizei *, gl::GLint *> GetProgramResourceiv;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLenum, gl::GLint *> GetProgramStageiv;
    static Function<void, gl::GLenum, gl::GLenum, void *> GetProgramStringARB;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLubyte *> GetProgramStringNV;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLuint *> GetProgramSubroutineParameteruivNV;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint *> GetProgramiv;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint *> GetProgramivARB;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint *> GetProgramivNV;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLenum, gl::GLintptr> GetQueryBufferObjecti64v;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLenum, gl::GLintptr> GetQueryBufferObjectiv;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLenum, gl::GLintptr> GetQueryBufferObjectui64v;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLenum, gl::GLintptr> GetQueryBufferObjectuiv;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLenum, gl::GLint *> GetQueryIndexediv;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint64 *> GetQueryObjecti64v;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint64 *> GetQueryObjecti64vEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint *> GetQueryObjectiv;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint *> GetQueryObjectivARB;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLuint64 *> GetQueryObjectui64v;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLuint64 *> GetQueryObjectui64vEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLuint *> GetQueryObjectuiv;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLuint *> GetQueryObjectuivARB;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint *> GetQueryiv;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint *> GetQueryivARB;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint *> GetRenderbufferParameteriv;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint *> GetRenderbufferParameterivEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint *> GetSamplerParameterIiv;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLuint *> GetSamplerParameterIuiv;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLfloat *> GetSamplerParameterfv;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint *> GetSamplerParameteriv;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, void *, void *, void *> GetSeparableFilter;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, void *, void *, void *> GetSeparableFilterEXT;
    static Function<void, gl::GLuint, gl::GLsizei, gl::GLsizei *, gl::GLchar *> GetShaderInfoLog;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint *, gl::GLint *> GetShaderPrecisionFormat;
    static Function<void, gl::GLuint, gl::GLsizei, gl::GLsizei *, gl::GLchar *> GetShaderSource;
    static Function<void, gl::GLhandleARB, gl::GLsizei, gl::GLsizei *, gl::GLcharARB *> GetShaderSourceARB;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint *> GetShaderiv;
    static Function<void, gl::GLenum, gl::GLfloat *> GetSharpenTexFuncSGIS;
    static Function<gl::GLushort, gl::GLenum> GetStageIndexNV;
    static Function<const gl::GLubyte *, gl::GLenum> GetString;
    static Function<const gl::GLubyte *, gl::GLenum, gl::GLuint> GetStringi;
    static Function<gl::GLuint, gl::GLuint, gl::GLenum, const gl::GLchar *> GetSubroutineIndex;
    static Function<gl::GLint, gl::GLuint, gl::GLenum, const gl::GLchar *> GetSubroutineUniformLocation;
    static Function<void, gl::GLsync, gl::GLenum, gl::GLsizei, gl::GLsizei *, gl::GLint *> GetSynciv;
    static Function<void, gl::GLenum, gl::GLfloat *> GetTexBumpParameterfvATI;
    static Function<void, gl::GLenum, gl::GLint *> GetTexBumpParameterivATI;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfloat *> GetTexEnvfv;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint *> GetTexEnviv;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfixed *> GetTexEnvxvOES;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfloat *> GetTexFilterFuncSGIS;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLdouble *> GetTexGendv;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfloat *> GetTexGenfv;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint *> GetTexGeniv;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfixed *> GetTexGenxvOES;
    static Function<void, gl::GLenum, gl::GLint, gl::GLenum, gl::GLenum, void *> GetTexImage;
    static Function<void, gl::GLenum, gl::GLint, gl::GLenum, gl::GLfloat *> GetTexLevelParameterfv;
    static Function<void, gl::GLenum, gl::GLint, gl::GLenum, gl::GLint *> GetTexLevelParameteriv;
    static Function<void, gl::GLenum, gl::GLint, gl::GLenum, gl::GLfixed *> GetTexLevelParameterxvOES;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint *> GetTexParameterIiv;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint *> GetTexParameterIivEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLuint *> GetTexParameterIuiv;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLuint *> GetTexParameterIuivEXT;
    static Function<void, gl::GLenum, gl::GLenum, void **> GetTexParameterPointervAPPLE;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfloat *> GetTexParameterfv;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint *> GetTexParameteriv;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfixed *> GetTexParameterxvOES;
    static Function<gl::GLuint64, gl::GLuint> GetTextureHandleARB;
    static Function<gl::GLuint64, gl::GLuint> GetTextureHandleNV;
    static Function<void, gl::GLuint, gl::GLint, gl::GLenum, gl::GLenum, gl::GLsizei, void *> GetTextureImage;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint, gl::GLenum, gl::GLenum, void *> GetTextureImageEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLenum, gl::GLfloat *> GetTextureLevelParameterfv;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint, gl::GLenum, gl::GLfloat *> GetTextureLevelParameterfvEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLenum, gl::GLint *> GetTextureLevelParameteriv;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint, gl::GLenum, gl::GLint *> GetTextureLevelParameterivEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint *> GetTextureParameterIiv;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLenum, gl::GLint *> GetTextureParameterIivEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLuint *> GetTextureParameterIuiv;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLenum, gl::GLuint *> GetTextureParameterIuivEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLfloat *> GetTextureParameterfv;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLenum, gl::GLfloat *> GetTextureParameterfvEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint *> GetTextureParameteriv;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLenum, gl::GLint *> GetTextureParameterivEXT;
    static Function<gl::GLuint64, gl::GLuint, gl::GLuint> GetTextureSamplerHandleARB;
    static Function<gl::GLuint64, gl::GLuint, gl::GLuint> GetTextureSamplerHandleNV;
    static Function<void, gl::GLuint, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLsizei, gl::GLenum, gl::GLenum, gl::GLsizei, void *> GetTextureSubImage;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLenum, gl::GLint *> GetTrackMatrixivNV;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLsizei, gl::GLsizei *, gl::GLsizei *, gl::GLenum *, gl::GLchar *> GetTransformFeedbackVarying;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLsizei, gl::GLsizei *, gl::GLsizei *, gl::GLenum *, gl::GLchar *> GetTransformFeedbackVaryingEXT;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLint *> GetTransformFeedbackVaryingNV;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLuint, gl::GLint64 *> GetTransformFeedbacki64_v;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLuint, gl::GLint *> GetTransformFeedbacki_v;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint *> GetTransformFeedbackiv;
    static Function<gl::GLuint, gl::GLuint, const gl::GLchar *> GetUniformBlockIndex;
    static Function<gl::GLint, gl::GLuint, gl::GLint> GetUniformBufferSizeEXT;
    static Function<void, gl::GLuint, gl::GLsizei, const gl::GLchar *const*, gl::GLuint *> GetUniformIndices;
    static Function<gl::GLint, gl::GLuint, const gl::GLchar *> GetUniformLocation;
    static Function<gl::GLint, gl::GLhandleARB, const gl::GLcharARB *> GetUniformLocationARB;
    static Function<gl::GLintptr, gl::GLuint, gl::GLint> GetUniformOffsetEXT;
    static Function<void, gl::GLenum, gl::GLint, gl::GLuint *> GetUniformSubroutineuiv;
    static Function<void, gl::GLuint, gl::GLint, gl::GLdouble *> GetUniformdv;
    static Function<void, gl::GLuint, gl::GLint, gl::GLfloat *> GetUniformfv;
    static Function<void, gl::GLhandleARB, gl::GLint, gl::GLfloat *> GetUniformfvARB;
    static Function<void, gl::GLuint, gl::GLint, gl::GLint64 *> GetUniformi64vARB;
    static Function<void, gl::GLuint, gl::GLint, gl::GLint64EXT *> GetUniformi64vNV;
    static Function<void, gl::GLuint, gl::GLint, gl::GLint *> GetUniformiv;
    static Function<void, gl::GLhandleARB, gl::GLint, gl::GLint *> GetUniformivARB;
    static Function<void, gl::GLuint, gl::GLint, gl::GLuint64 *> GetUniformui64vARB;
    static Function<void, gl::GLuint, gl::GLint, gl::GLuint64EXT *> GetUniformui64vNV;
    static Function<void, gl::GLuint, gl::GLint, gl::GLuint *> GetUniformuiv;
    static Function<void, gl::GLuint, gl::GLint, gl::GLuint *> GetUniformuivEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLfloat *> GetVariantArrayObjectfvATI;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint *> GetVariantArrayObjectivATI;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLboolean *> GetVariantBooleanvEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLfloat *> GetVariantFloatvEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint *> GetVariantIntegervEXT;
    static Function<void, gl::GLuint, gl::GLenum, void **> GetVariantPointervEXT;
    static Function<gl::GLint, gl::GLuint, const gl::GLchar *> GetVaryingLocationNV;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLenum, gl::GLint64 *> GetVertexArrayIndexed64iv;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLenum, gl::GLint *> GetVertexArrayIndexediv;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLenum, gl::GLint *> GetVertexArrayIntegeri_vEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint *> GetVertexArrayIntegervEXT;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLenum, void **> GetVertexArrayPointeri_vEXT;
    static Function<void, gl::GLuint, gl::GLenum, void **> GetVertexArrayPointervEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint *> GetVertexArrayiv;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLfloat *> GetVertexAttribArrayObjectfvATI;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint *> GetVertexAttribArrayObjectivATI;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint *> GetVertexAttribIiv;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint *> GetVertexAttribIivEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLuint *> GetVertexAttribIuiv;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLuint *> GetVertexAttribIuivEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLdouble *> GetVertexAttribLdv;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLdouble *> GetVertexAttribLdvEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint64EXT *> GetVertexAttribLi64vNV;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLuint64EXT *> GetVertexAttribLui64vARB;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLuint64EXT *> GetVertexAttribLui64vNV;
    static Function<void, gl::GLuint, gl::GLenum, void **> GetVertexAttribPointerv;
    static Function<void, gl::GLuint, gl::GLenum, void **> GetVertexAttribPointervARB;
    static Function<void, gl::GLuint, gl::GLenum, void **> GetVertexAttribPointervNV;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLdouble *> GetVertexAttribdv;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLdouble *> GetVertexAttribdvARB;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLdouble *> GetVertexAttribdvNV;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLfloat *> GetVertexAttribfv;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLfloat *> GetVertexAttribfvARB;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLfloat *> GetVertexAttribfvNV;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint *> GetVertexAttribiv;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint *> GetVertexAttribivARB;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint *> GetVertexAttribivNV;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLenum, gl::GLdouble *> GetVideoCaptureStreamdvNV;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLenum, gl::GLfloat *> GetVideoCaptureStreamfvNV;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLenum, gl::GLint *> GetVideoCaptureStreamivNV;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint *> GetVideoCaptureivNV;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint64EXT *> GetVideoi64vNV;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint *> GetVideoivNV;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLuint64EXT *> GetVideoui64vNV;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLuint *> GetVideouivNV;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLsizei, void *> GetnColorTable;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLsizei, void *> GetnColorTableARB;
    static Function<void, gl::GLenum, gl::GLint, gl::GLsizei, void *> GetnCompressedTexImage;
    static Function<void, gl::GLenum, gl::GLint, gl::GLsizei, void *> GetnCompressedTexImageARB;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLsizei, void *> GetnConvolutionFilter;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLsizei, void *> GetnConvolutionFilterARB;
    static Function<void, gl::GLenum, gl::GLboolean, gl::GLenum, gl::GLenum, gl::GLsizei, void *> GetnHistogram;
    static Function<void, gl::GLenum, gl::GLboolean, gl::GLenum, gl::GLenum, gl::GLsizei, void *> GetnHistogramARB;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLsizei, gl::GLdouble *> GetnMapdv;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLsizei, gl::GLdouble *> GetnMapdvARB;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLsizei, gl::GLfloat *> GetnMapfv;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLsizei, gl::GLfloat *> GetnMapfvARB;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLsizei, gl::GLint *> GetnMapiv;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLsizei, gl::GLint *> GetnMapivARB;
    static Function<void, gl::GLenum, gl::GLboolean, gl::GLenum, gl::GLenum, gl::GLsizei, void *> GetnMinmax;
    static Function<void, gl::GLenum, gl::GLboolean, gl::GLenum, gl::GLenum, gl::GLsizei, void *> GetnMinmaxARB;
    static Function<void, gl::GLenum, gl::GLsizei, gl::GLfloat *> GetnPixelMapfv;
    static Function<void, gl::GLenum, gl::GLsizei, gl::GLfloat *> GetnPixelMapfvARB;
    static Function<void, gl::GLenum, gl::GLsizei, gl::GLuint *> GetnPixelMapuiv;
    static Function<void, gl::GLenum, gl::GLsizei, gl::GLuint *> GetnPixelMapuivARB;
    static Function<void, gl::GLenum, gl::GLsizei, gl::GLushort *> GetnPixelMapusv;
    static Function<void, gl::GLenum, gl::GLsizei, gl::GLushort *> GetnPixelMapusvARB;
    static Function<void, gl::GLsizei, gl::GLubyte *> GetnPolygonStipple;
    static Function<void, gl::GLsizei, gl::GLubyte *> GetnPolygonStippleARB;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLsizei, void *, gl::GLsizei, void *, void *> GetnSeparableFilter;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLsizei, void *, gl::GLsizei, void *, void *> GetnSeparableFilterARB;
    static Function<void, gl::GLenum, gl::GLint, gl::GLenum, gl::GLenum, gl::GLsizei, void *> GetnTexImage;
    static Function<void, gl::GLenum, gl::GLint, gl::GLenum, gl::GLenum, gl::GLsizei, void *> GetnTexImageARB;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, gl::GLdouble *> GetnUniformdv;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, gl::GLdouble *> GetnUniformdvARB;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, gl::GLfloat *> GetnUniformfv;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, gl::GLfloat *> GetnUniformfvARB;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, gl::GLint64 *> GetnUniformi64vARB;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, gl::GLint *> GetnUniformiv;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, gl::GLint *> GetnUniformivARB;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, gl::GLuint64 *> GetnUniformui64vARB;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, gl::GLuint *> GetnUniformuiv;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, gl::GLuint *> GetnUniformuivARB;
    static Function<void, gl::GLbyte> GlobalAlphaFactorbSUN;
    static Function<void, gl::GLdouble> GlobalAlphaFactordSUN;
    static Function<void, gl::GLfloat> GlobalAlphaFactorfSUN;
    static Function<void, gl::GLint> GlobalAlphaFactoriSUN;
    static Function<void, gl::GLshort> GlobalAlphaFactorsSUN;
    static Function<void, gl::GLubyte> GlobalAlphaFactorubSUN;
    static Function<void, gl::GLuint> GlobalAlphaFactoruiSUN;
    static Function<void, gl::GLushort> GlobalAlphaFactorusSUN;
    static Function<void, gl::GLenum, gl::GLenum> Hint;
    static Function<void, gl::GLenum, gl::GLint> HintPGI;
    static Function<void, gl::GLenum, gl::GLsizei, gl::GLenum, gl::GLboolean> Histogram;
    static Function<void, gl::GLenum, gl::GLsizei, gl::GLenum, gl::GLboolean> HistogramEXT;
    static Function<void, gl::GLenum, const void *> IglooInterfaceSGIX;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfloat> ImageTransformParameterfHP;
    static Function<void, gl::GLenum, gl::GLenum, const gl::GLfloat *> ImageTransformParameterfvHP;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint> ImageTransformParameteriHP;
    static Function<void, gl::GLenum, gl::GLenum, const gl::GLint *> ImageTransformParameterivHP;
    static Function<gl::GLsync, gl::GLenum, gl::GLintptr, gl::UnusedMask> ImportSyncEXT;
    static Function<void, gl::GLenum, gl::GLsizei> IndexFormatNV;
    static Function<void, gl::GLenum, gl::GLclampf> IndexFuncEXT;
    static Function<void, gl::GLuint> IndexMask;
    static Function<void, gl::GLenum, gl::GLenum> IndexMaterialEXT;
    static Function<void, gl::GLenum, gl::GLsizei, const void *> IndexPointer;
    static Function<void, gl::GLenum, gl::GLsizei, gl::GLsizei, const void *> IndexPointerEXT;
    static Function<void, gl::GLenum, gl::GLint, const void **, gl::GLint> IndexPointerListIBM;
    static Function<void, gl::GLdouble> Indexd;
    static Function<void, const gl::GLdouble *> Indexdv;
    static Function<void, gl::GLfloat> Indexf;
    static Function<void, const gl::GLfloat *> Indexfv;
    static Function<void, gl::GLint> Indexi;
    static Function<void, const gl::GLint *> Indexiv;
    static Function<void, gl::GLshort> Indexs;
    static Function<void, const gl::GLshort *> Indexsv;
    static Function<void, gl::GLubyte> Indexub;
    static Function<void, const gl::GLubyte *> Indexubv;
    static Function<void, gl::GLfixed> IndexxOES;
    static Function<void, const gl::GLfixed *> IndexxvOES;
    static Function<void> InitNames;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLuint> InsertComponentEXT;
    static Function<void, gl::GLsizei, const gl::GLchar *> InsertEventMarkerEXT;
    static Function<void, gl::GLsizei, gl::GLint *> InstrumentsBufferSGIX;
    static Function<void, gl::GLenum, gl::GLsizei, const void *> InterleavedArrays;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLfloat> InterpolatePathsNV;
    static Function<void, gl::GLuint> InvalidateBufferData;
    static Function<void, gl::GLuint, gl::GLintptr, gl::GLsizeiptr> InvalidateBufferSubData;
    static Function<void, gl::GLenum, gl::GLsizei, const gl::GLenum *> InvalidateFramebuffer;
    static Function<void, gl::GLuint, gl::GLsizei, const gl::GLenum *> InvalidateNamedFramebufferData;
    static Function<void, gl::GLuint, gl::GLsizei, const gl::GLenum *, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei> InvalidateNamedFramebufferSubData;
    static Function<void, gl::GLenum, gl::GLsizei, const gl::GLenum *, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei> InvalidateSubFramebuffer;
    static Function<void, gl::GLuint, gl::GLint> InvalidateTexImage;
    static Function<void, gl::GLuint, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLsizei> InvalidateTexSubImage;
    static Function<gl::GLboolean, gl::GLuint> IsAsyncMarkerSGIX;
    static Function<gl::GLboolean, gl::GLuint> IsBuffer;
    static Function<gl::GLboolean, gl::GLuint> IsBufferARB;
    static Function<gl::GLboolean, gl::GLenum> IsBufferResidentNV;
    static Function<gl::GLboolean, gl::GLuint> IsCommandListNV;
    static Function<gl::GLboolean, gl::GLenum> IsEnabled;
    static Function<gl::GLboolean, gl::GLenum, gl::GLuint> IsEnabledIndexedEXT;
    static Function<gl::GLboolean, gl::GLenum, gl::GLuint> IsEnabledi;
    static Function<gl::GLboolean, gl::GLuint> IsFenceAPPLE;
    static Function<gl::GLboolean, gl::GLuint> IsFenceNV;
    static Function<gl::GLboolean, gl::GLuint> IsFramebuffer;
    static Function<gl::GLboolean, gl::GLuint> IsFramebufferEXT;
    static Function<gl::GLboolean, gl::GLuint64> IsImageHandleResidentARB;
    static Function<gl::GLboolean, gl::GLuint64> IsImageHandleResidentNV;
    static Function<gl::GLboolean, gl::GLuint> IsList;
    static Function<gl::GLboolean, gl::GLenum, gl::GLuint> IsNameAMD;
    static Function<gl::GLboolean, gl::GLuint> IsNamedBufferResidentNV;
    static Function<gl::GLboolean, gl::GLint, const gl::GLchar *> IsNamedStringARB;
    static Function<gl::GLboolean, gl::GLuint> IsObjectBufferATI;
    static Function<gl::GLboolean, gl::GLuint> IsOcclusionQueryNV;
    static Function<gl::GLboolean, gl::GLuint> IsPathNV;
    static Function<gl::GLboolean, gl::GLuint, gl::GLuint, gl::GLfloat, gl::GLfloat> IsPointInFillPathNV;
    static Function<gl::GLboolean, gl::GLuint, gl::GLfloat, gl::GLfloat> IsPointInStrokePathNV;
    static Function<gl::GLboolean, gl::GLuint> IsProgram;
    static Function<gl::GLboolean, gl::GLuint> IsProgramARB;
    static Function<gl::GLboolean, gl::GLuint> IsProgramNV;
    static Function<gl::GLboolean, gl::GLuint> IsProgramPipeline;
    static Function<gl::GLboolean, gl::GLuint> IsQuery;
    static Function<gl::GLboolean, gl::GLuint> IsQueryARB;
    static Function<gl::GLboolean, gl::GLuint> IsRenderbuffer;
    static Function<gl::GLboolean, gl::GLuint> IsRenderbufferEXT;
    static Function<gl::GLboolean, gl::GLuint> IsSampler;
    static Function<gl::GLboolean, gl::GLuint> IsShader;
    static Function<gl::GLboolean, gl::GLuint> IsStateNV;
    static Function<gl::GLboolean, gl::GLsync> IsSync;
    static Function<gl::GLboolean, gl::GLuint> IsTexture;
    static Function<gl::GLboolean, gl::GLuint> IsTextureEXT;
    static Function<gl::GLboolean, gl::GLuint64> IsTextureHandleResidentARB;
    static Function<gl::GLboolean, gl::GLuint64> IsTextureHandleResidentNV;
    static Function<gl::GLboolean, gl::GLuint> IsTransformFeedback;
    static Function<gl::GLboolean, gl::GLuint> IsTransformFeedbackNV;
    static Function<gl::GLboolean, gl::GLuint, gl::GLenum> IsVariantEnabledEXT;
    static Function<gl::GLboolean, gl::GLuint> IsVertexArray;
    static Function<gl::GLboolean, gl::GLuint> IsVertexArrayAPPLE;
    static Function<gl::GLboolean, gl::GLuint, gl::GLenum> IsVertexAttribEnabledAPPLE;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLsizei, const gl::GLchar *> LabelObjectEXT;
    static Function<void, gl::GLenum, gl::GLint> LightEnviSGIX;
    static Function<void, gl::GLenum, gl::GLfloat> LightModelf;
    static Function<void, gl::GLenum, const gl::GLfloat *> LightModelfv;
    static Function<void, gl::GLenum, gl::GLint> LightModeli;
    static Function<void, gl::GLenum, const gl::GLint *> LightModeliv;
    static Function<void, gl::GLenum, gl::GLfixed> LightModelxOES;
    static Function<void, gl::GLenum, const gl::GLfixed *> LightModelxvOES;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfloat> Lightf;
    static Function<void, gl::GLenum, gl::GLenum, const gl::GLfloat *> Lightfv;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint> Lighti;
    static Function<void, gl::GLenum, gl::GLenum, const gl::GLint *> Lightiv;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfixed> LightxOES;
    static Function<void, gl::GLenum, gl::GLenum, const gl::GLfixed *> LightxvOES;
    static Function<void, gl::GLint, gl::GLushort> LineStipple;
    static Function<void, gl::GLfloat> LineWidth;
    static Function<void, gl::GLfixed> LineWidthxOES;
    static Function<void, gl::GLuint> LinkProgram;
    static Function<void, gl::GLhandleARB> LinkProgramARB;
    static Function<void, gl::GLuint> ListBase;
    static Function<void, gl::GLuint, gl::GLuint, const void **, const gl::GLsizei *, const gl::GLuint *, const gl::GLuint *, gl::GLuint> ListDrawCommandsStatesClientNV;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLfloat> ListParameterfSGIX;
    static Function<void, gl::GLuint, gl::GLenum, const gl::GLfloat *> ListParameterfvSGIX;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint> ListParameteriSGIX;
    static Function<void, gl::GLuint, gl::GLenum, const gl::GLint *> ListParameterivSGIX;
    static Function<void> LoadIdentity;
    static Function<void, gl::FfdMaskSGIX> LoadIdentityDeformationMapSGIX;
    static Function<void, const gl::GLdouble *> LoadMatrixd;
    static Function<void, const gl::GLfloat *> LoadMatrixf;
    static Function<void, const gl::GLfixed *> LoadMatrixxOES;
    static Function<void, gl::GLuint> LoadName;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLsizei, const gl::GLubyte *> LoadProgramNV;
    static Function<void, const gl::GLdouble *> LoadTransposeMatrixd;
    static Function<void, const gl::GLdouble *> LoadTransposeMatrixdARB;
    static Function<void, const gl::GLfloat *> LoadTransposeMatrixf;
    static Function<void, const gl::GLfloat *> LoadTransposeMatrixfARB;
    static Function<void, const gl::GLfixed *> LoadTransposeMatrixxOES;
    static Function<void, gl::GLint, gl::GLsizei> LockArraysEXT;
    static Function<void, gl::GLenum> LogicOp;
    static Function<void, gl::GLenum> MakeBufferNonResidentNV;
    static Function<void, gl::GLenum, gl::GLenum> MakeBufferResidentNV;
    static Function<void, gl::GLuint64> MakeImageHandleNonResidentARB;
    static Function<void, gl::GLuint64> MakeImageHandleNonResidentNV;
    static Function<void, gl::GLuint64, gl::GLenum> MakeImageHandleResidentARB;
    static Function<void, gl::GLuint64, gl::GLenum> MakeImageHandleResidentNV;
    static Function<void, gl::GLuint> MakeNamedBufferNonResidentNV;
    static Function<void, gl::GLuint, gl::GLenum> MakeNamedBufferResidentNV;
    static Function<void, gl::GLuint64> MakeTextureHandleNonResidentARB;
    static Function<void, gl::GLuint64> MakeTextureHandleNonResidentNV;
    static Function<void, gl::GLuint64> MakeTextureHandleResidentARB;
    static Function<void, gl::GLuint64> MakeTextureHandleResidentNV;
    static Function<void, gl::GLenum, gl::GLdouble, gl::GLdouble, gl::GLint, gl::GLint, const gl::GLdouble *> Map1d;
    static Function<void, gl::GLenum, gl::GLfloat, gl::GLfloat, gl::GLint, gl::GLint, const gl::GLfloat *> Map1f;
    static Function<void, gl::GLenum, gl::GLfixed, gl::GLfixed, gl::GLint, gl::GLint, gl::GLfixed> Map1xOES;
    static Function<void, gl::GLenum, gl::GLdouble, gl::GLdouble, gl::GLint, gl::GLint, gl::GLdouble, gl::GLdouble, gl::GLint, gl::GLint, const gl::GLdouble *> Map2d;
    static Function<void, gl::GLenum, gl::GLfloat, gl::GLfloat, gl::GLint, gl::GLint, gl::GLfloat, gl::GLfloat, gl::GLint, gl::GLint, const gl::GLfloat *> Map2f;
    static Function<void, gl::GLenum, gl::GLfixed, gl::GLfixed, gl::GLint, gl::GLint, gl::GLfixed, gl::GLfixed, gl::GLint, gl::GLint, gl::GLfixed> Map2xOES;
    static Function<void *, gl::GLenum, gl::GLenum> MapBuffer;
    static Function<void *, gl::GLenum, gl::GLenum> MapBufferARB;
    static Function<void *, gl::GLenum, gl::GLintptr, gl::GLsizeiptr, gl::BufferAccessMask> MapBufferRange;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLenum, gl::GLsizei, gl::GLsizei, gl::GLint, gl::GLint, gl::GLboolean, const void *> MapControlPointsNV;
    static Function<void, gl::GLint, gl::GLdouble, gl::GLdouble> MapGrid1d;
    static Function<void, gl::GLint, gl::GLfloat, gl::GLfloat> MapGrid1f;
    static Function<void, gl::GLint, gl::GLfixed, gl::GLfixed> MapGrid1xOES;
    static Function<void, gl::GLint, gl::GLdouble, gl::GLdouble, gl::GLint, gl::GLdouble, gl::GLdouble> MapGrid2d;
    static Function<void, gl::GLint, gl::GLfloat, gl::GLfloat, gl::GLint, gl::GLfloat, gl::GLfloat> MapGrid2f;
    static Function<void, gl::GLint, gl::GLfixed, gl::GLfixed, gl::GLfixed, gl::GLfixed> MapGrid2xOES;
    static Function<void *, gl::GLuint, gl::GLenum> MapNamedBuffer;
    static Function<void *, gl::GLuint, gl::GLenum> MapNamedBufferEXT;
    static Function<void *, gl::GLuint, gl::GLintptr, gl::GLsizeiptr, gl::BufferAccessMask> MapNamedBufferRange;
    static Function<void *, gl::GLuint, gl::GLintptr, gl::GLsizeiptr, gl::BufferAccessMask> MapNamedBufferRangeEXT;
    static Function<void *, gl::GLuint> MapObjectBufferATI;
    static Function<void, gl::GLenum, gl::GLenum, const gl::GLfloat *> MapParameterfvNV;
    static Function<void, gl::GLenum, gl::GLenum, const gl::GLint *> MapParameterivNV;
    static Function<void *, gl::GLuint, gl::GLint, gl::MapBufferUsageMask, gl::GLint *, gl::GLenum *> MapTexture2DINTEL;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLdouble, gl::GLdouble, gl::GLint, gl::GLint, const gl::GLdouble *> MapVertexAttrib1dAPPLE;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLfloat, gl::GLfloat, gl::GLint, gl::GLint, const gl::GLfloat *> MapVertexAttrib1fAPPLE;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLdouble, gl::GLdouble, gl::GLint, gl::GLint, gl::GLdouble, gl::GLdouble, gl::GLint, gl::GLint, const gl::GLdouble *> MapVertexAttrib2dAPPLE;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLfloat, gl::GLfloat, gl::GLint, gl::GLint, gl::GLfloat, gl::GLfloat, gl::GLint, gl::GLint, const gl::GLfloat *> MapVertexAttrib2fAPPLE;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfloat> Materialf;
    static Function<void, gl::GLenum, gl::GLenum, const gl::GLfloat *> Materialfv;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint> Materiali;
    static Function<void, gl::GLenum, gl::GLenum, const gl::GLint *> Materialiv;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfixed> MaterialxOES;
    static Function<void, gl::GLenum, gl::GLenum, const gl::GLfixed *> MaterialxvOES;
    static Function<void, gl::GLenum, gl::GLdouble, gl::GLdouble, gl::GLdouble, gl::GLdouble, gl::GLdouble, gl::GLdouble> MatrixFrustumEXT;
    static Function<void, gl::GLint, gl::GLenum, gl::GLsizei, const void *> MatrixIndexPointerARB;
    static Function<void, gl::GLint, const gl::GLubyte *> MatrixIndexubvARB;
    static Function<void, gl::GLint, const gl::GLuint *> MatrixIndexuivARB;
    static Function<void, gl::GLint, const gl::GLushort *> MatrixIndexusvARB;
    static Function<void, gl::GLenum, const gl::GLfloat *> MatrixLoad3x2fNV;
    static Function<void, gl::GLenum, const gl::GLfloat *> MatrixLoad3x3fNV;
    static Function<void, gl::GLenum> MatrixLoadIdentityEXT;
    static Function<void, gl::GLenum, const gl::GLfloat *> MatrixLoadTranspose3x3fNV;
    static Function<void, gl::GLenum, const gl::GLdouble *> MatrixLoadTransposedEXT;
    static Function<void, gl::GLenum, const gl::GLfloat *> MatrixLoadTransposefEXT;
    static Function<void, gl::GLenum, const gl::GLdouble *> MatrixLoaddEXT;
    static Function<void, gl::GLenum, const gl::GLfloat *> MatrixLoadfEXT;
    static Function<void, gl::GLenum> MatrixMode;
    static Function<void, gl::GLenum, const gl::GLfloat *> MatrixMult3x2fNV;
    static Function<void, gl::GLenum, const gl::GLfloat *> MatrixMult3x3fNV;
    static Function<void, gl::GLenum, const gl::GLfloat *> MatrixMultTranspose3x3fNV;
    static Function<void, gl::GLenum, const gl::GLdouble *> MatrixMultTransposedEXT;
    static Function<void, gl::GLenum, const gl::GLfloat *> MatrixMultTransposefEXT;
    static Function<void, gl::GLenum, const gl::GLdouble *> MatrixMultdEXT;
    static Function<void, gl::GLenum, const gl::GLfloat *> MatrixMultfEXT;
    static Function<void, gl::GLenum, gl::GLdouble, gl::GLdouble, gl::GLdouble, gl::GLdouble, gl::GLdouble, gl::GLdouble> MatrixOrthoEXT;
    static Function<void, gl::GLenum> MatrixPopEXT;
    static Function<void, gl::GLenum> MatrixPushEXT;
    static Function<void, gl::GLenum, gl::GLdouble, gl::GLdouble, gl::GLdouble, gl::GLdouble> MatrixRotatedEXT;
    static Function<void, gl::GLenum, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat> MatrixRotatefEXT;
    static Function<void, gl::GLenum, gl::GLdouble, gl::GLdouble, gl::GLdouble> MatrixScaledEXT;
    static Function<void, gl::GLenum, gl::GLfloat, gl::GLfloat, gl::GLfloat> MatrixScalefEXT;
    static Function<void, gl::GLenum, gl::GLdouble, gl::GLdouble, gl::GLdouble> MatrixTranslatedEXT;
    static Function<void, gl::GLenum, gl::GLfloat, gl::GLfloat, gl::GLfloat> MatrixTranslatefEXT;
    static Function<void, gl::GLuint> MaxShaderCompilerThreadsARB;
    static Function<void, gl::MemoryBarrierMask> MemoryBarrier;
    static Function<void, gl::MemoryBarrierMask> MemoryBarrierByRegion;
    static Function<void, gl::MemoryBarrierMask> MemoryBarrierEXT;
    static Function<void, gl::GLfloat> MinSampleShading;
    static Function<void, gl::GLfloat> MinSampleShadingARB;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLboolean> Minmax;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLboolean> MinmaxEXT;
    static Function<void, const gl::GLdouble *> MultMatrixd;
    static Function<void, const gl::GLfloat *> MultMatrixf;
    static Function<void, const gl::GLfixed *> MultMatrixxOES;
    static Function<void, const gl::GLdouble *> MultTransposeMatrixd;
    static Function<void, const gl::GLdouble *> MultTransposeMatrixdARB;
    static Function<void, const gl::GLfloat *> MultTransposeMatrixf;
    static Function<void, const gl::GLfloat *> MultTransposeMatrixfARB;
    static Function<void, const gl::GLfixed *> MultTransposeMatrixxOES;
    static Function<void, gl::GLenum, const gl::GLint *, const gl::GLsizei *, gl::GLsizei> MultiDrawArrays;
    static Function<void, gl::GLenum, const gl::GLint *, const gl::GLsizei *, gl::GLsizei> MultiDrawArraysEXT;
    static Function<void, gl::GLenum, const void *, gl::GLsizei, gl::GLsizei> MultiDrawArraysIndirect;
    static Function<void, gl::GLenum, const void *, gl::GLsizei, gl::GLsizei> MultiDrawArraysIndirectAMD;
    static Function<void, gl::GLenum, const void *, gl::GLsizei, gl::GLsizei, gl::GLsizei, gl::GLint> MultiDrawArraysIndirectBindlessCountNV;
    static Function<void, gl::GLenum, const void *, gl::GLsizei, gl::GLsizei, gl::GLint> MultiDrawArraysIndirectBindlessNV;
    static Function<void, gl::GLenum, gl::GLintptr, gl::GLintptr, gl::GLsizei, gl::GLsizei> MultiDrawArraysIndirectCountARB;
    static Function<void, gl::GLenum, const gl::GLint *, const gl::GLsizei *, gl::GLsizei> MultiDrawElementArrayAPPLE;
    static Function<void, gl::GLenum, const gl::GLsizei *, gl::GLenum, const void *const*, gl::GLsizei> MultiDrawElements;
    static Function<void, gl::GLenum, const gl::GLsizei *, gl::GLenum, const void *const*, gl::GLsizei, const gl::GLint *> MultiDrawElementsBaseVertex;
    static Function<void, gl::GLenum, const gl::GLsizei *, gl::GLenum, const void *const*, gl::GLsizei> MultiDrawElementsEXT;
    static Function<void, gl::GLenum, gl::GLenum, const void *, gl::GLsizei, gl::GLsizei> MultiDrawElementsIndirect;
    static Function<void, gl::GLenum, gl::GLenum, const void *, gl::GLsizei, gl::GLsizei> MultiDrawElementsIndirectAMD;
    static Function<void, gl::GLenum, gl::GLenum, const void *, gl::GLsizei, gl::GLsizei, gl::GLsizei, gl::GLint> MultiDrawElementsIndirectBindlessCountNV;
    static Function<void, gl::GLenum, gl::GLenum, const void *, gl::GLsizei, gl::GLsizei, gl::GLint> MultiDrawElementsIndirectBindlessNV;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLintptr, gl::GLintptr, gl::GLsizei, gl::GLsizei> MultiDrawElementsIndirectCountARB;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLuint, const gl::GLint *, const gl::GLsizei *, gl::GLsizei> MultiDrawRangeElementArrayAPPLE;
    static Function<void, const gl::GLenum *, const gl::GLint *, const gl::GLsizei *, gl::GLsizei, gl::GLint> MultiModeDrawArraysIBM;
    static Function<void, const gl::GLenum *, const gl::GLsizei *, gl::GLenum, const void *const*, gl::GLsizei, gl::GLint> MultiModeDrawElementsIBM;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLuint> MultiTexBufferEXT;
    static Function<void, gl::GLenum, gl::GLbyte> MultiTexCoord1bOES;
    static Function<void, gl::GLenum, const gl::GLbyte *> MultiTexCoord1bvOES;
    static Function<void, gl::GLenum, gl::GLdouble> MultiTexCoord1d;
    static Function<void, gl::GLenum, gl::GLdouble> MultiTexCoord1dARB;
    static Function<void, gl::GLenum, const gl::GLdouble *> MultiTexCoord1dv;
    static Function<void, gl::GLenum, const gl::GLdouble *> MultiTexCoord1dvARB;
    static Function<void, gl::GLenum, gl::GLfloat> MultiTexCoord1f;
    static Function<void, gl::GLenum, gl::GLfloat> MultiTexCoord1fARB;
    static Function<void, gl::GLenum, const gl::GLfloat *> MultiTexCoord1fv;
    static Function<void, gl::GLenum, const gl::GLfloat *> MultiTexCoord1fvARB;
    static Function<void, gl::GLenum, gl::GLhalfNV> MultiTexCoord1hNV;
    static Function<void, gl::GLenum, const gl::GLhalfNV *> MultiTexCoord1hvNV;
    static Function<void, gl::GLenum, gl::GLint> MultiTexCoord1i;
    static Function<void, gl::GLenum, gl::GLint> MultiTexCoord1iARB;
    static Function<void, gl::GLenum, const gl::GLint *> MultiTexCoord1iv;
    static Function<void, gl::GLenum, const gl::GLint *> MultiTexCoord1ivARB;
    static Function<void, gl::GLenum, gl::GLshort> MultiTexCoord1s;
    static Function<void, gl::GLenum, gl::GLshort> MultiTexCoord1sARB;
    static Function<void, gl::GLenum, const gl::GLshort *> MultiTexCoord1sv;
    static Function<void, gl::GLenum, const gl::GLshort *> MultiTexCoord1svARB;
    static Function<void, gl::GLenum, gl::GLfixed> MultiTexCoord1xOES;
    static Function<void, gl::GLenum, const gl::GLfixed *> MultiTexCoord1xvOES;
    static Function<void, gl::GLenum, gl::GLbyte, gl::GLbyte> MultiTexCoord2bOES;
    static Function<void, gl::GLenum, const gl::GLbyte *> MultiTexCoord2bvOES;
    static Function<void, gl::GLenum, gl::GLdouble, gl::GLdouble> MultiTexCoord2d;
    static Function<void, gl::GLenum, gl::GLdouble, gl::GLdouble> MultiTexCoord2dARB;
    static Function<void, gl::GLenum, const gl::GLdouble *> MultiTexCoord2dv;
    static Function<void, gl::GLenum, const gl::GLdouble *> MultiTexCoord2dvARB;
    static Function<void, gl::GLenum, gl::GLfloat, gl::GLfloat> MultiTexCoord2f;
    static Function<void, gl::GLenum, gl::GLfloat, gl::GLfloat> MultiTexCoord2fARB;
    static Function<void, gl::GLenum, const gl::GLfloat *> MultiTexCoord2fv;
    static Function<void, gl::GLenum, const gl::GLfloat *> MultiTexCoord2fvARB;
    static Function<void, gl::GLenum, gl::GLhalfNV, gl::GLhalfNV> MultiTexCoord2hNV;
    static Function<void, gl::GLenum, const gl::GLhalfNV *> MultiTexCoord2hvNV;
    static Function<void, gl::GLenum, gl::GLint, gl::GLint> MultiTexCoord2i;
    static Function<void, gl::GLenum, gl::GLint, gl::GLint> MultiTexCoord2iARB;
    static Function<void, gl::GLenum, const gl::GLint *> MultiTexCoord2iv;
    static Function<void, gl::GLenum, const gl::GLint *> MultiTexCoord2ivARB;
    static Function<void, gl::GLenum, gl::GLshort, gl::GLshort> MultiTexCoord2s;
    static Function<void, gl::GLenum, gl::GLshort, gl::GLshort> MultiTexCoord2sARB;
    static Function<void, gl::GLenum, const gl::GLshort *> MultiTexCoord2sv;
    static Function<void, gl::GLenum, const gl::GLshort *> MultiTexCoord2svARB;
    static Function<void, gl::GLenum, gl::GLfixed, gl::GLfixed> MultiTexCoord2xOES;
    static Function<void, gl::GLenum, const gl::GLfixed *> MultiTexCoord2xvOES;
    static Function<void, gl::GLenum, gl::GLbyte, gl::GLbyte, gl::GLbyte> MultiTexCoord3bOES;
    static Function<void, gl::GLenum, const gl::GLbyte *> MultiTexCoord3bvOES;
    static Function<void, gl::GLenum, gl::GLdouble, gl::GLdouble, gl::GLdouble> MultiTexCoord3d;
    static Function<void, gl::GLenum, gl::GLdouble, gl::GLdouble, gl::GLdouble> MultiTexCoord3dARB;
    static Function<void, gl::GLenum, const gl::GLdouble *> MultiTexCoord3dv;
    static Function<void, gl::GLenum, const gl::GLdouble *> MultiTexCoord3dvARB;
    static Function<void, gl::GLenum, gl::GLfloat, gl::GLfloat, gl::GLfloat> MultiTexCoord3f;
    static Function<void, gl::GLenum, gl::GLfloat, gl::GLfloat, gl::GLfloat> MultiTexCoord3fARB;
    static Function<void, gl::GLenum, const gl::GLfloat *> MultiTexCoord3fv;
    static Function<void, gl::GLenum, const gl::GLfloat *> MultiTexCoord3fvARB;
    static Function<void, gl::GLenum, gl::GLhalfNV, gl::GLhalfNV, gl::GLhalfNV> MultiTexCoord3hNV;
    static Function<void, gl::GLenum, const gl::GLhalfNV *> MultiTexCoord3hvNV;
    static Function<void, gl::GLenum, gl::GLint, gl::GLint, gl::GLint> MultiTexCoord3i;
    static Function<void, gl::GLenum, gl::GLint, gl::GLint, gl::GLint> MultiTexCoord3iARB;
    static Function<void, gl::GLenum, const gl::GLint *> MultiTexCoord3iv;
    static Function<void, gl::GLenum, const gl::GLint *> MultiTexCoord3ivARB;
    static Function<void, gl::GLenum, gl::GLshort, gl::GLshort, gl::GLshort> MultiTexCoord3s;
    static Function<void, gl::GLenum, gl::GLshort, gl::GLshort, gl::GLshort> MultiTexCoord3sARB;
    static Function<void, gl::GLenum, const gl::GLshort *> MultiTexCoord3sv;
    static Function<void, gl::GLenum, const gl::GLshort *> MultiTexCoord3svARB;
    static Function<void, gl::GLenum, gl::GLfixed, gl::GLfixed, gl::GLfixed> MultiTexCoord3xOES;
    static Function<void, gl::GLenum, const gl::GLfixed *> MultiTexCoord3xvOES;
    static Function<void, gl::GLenum, gl::GLbyte, gl::GLbyte, gl::GLbyte, gl::GLbyte> MultiTexCoord4bOES;
    static Function<void, gl::GLenum, const gl::GLbyte *> MultiTexCoord4bvOES;
    static Function<void, gl::GLenum, gl::GLdouble, gl::GLdouble, gl::GLdouble, gl::GLdouble> MultiTexCoord4d;
    static Function<void, gl::GLenum, gl::GLdouble, gl::GLdouble, gl::GLdouble, gl::GLdouble> MultiTexCoord4dARB;
    static Function<void, gl::GLenum, const gl::GLdouble *> MultiTexCoord4dv;
    static Function<void, gl::GLenum, const gl::GLdouble *> MultiTexCoord4dvARB;
    static Function<void, gl::GLenum, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat> MultiTexCoord4f;
    static Function<void, gl::GLenum, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat> MultiTexCoord4fARB;
    static Function<void, gl::GLenum, const gl::GLfloat *> MultiTexCoord4fv;
    static Function<void, gl::GLenum, const gl::GLfloat *> MultiTexCoord4fvARB;
    static Function<void, gl::GLenum, gl::GLhalfNV, gl::GLhalfNV, gl::GLhalfNV, gl::GLhalfNV> MultiTexCoord4hNV;
    static Function<void, gl::GLenum, const gl::GLhalfNV *> MultiTexCoord4hvNV;
    static Function<void, gl::GLenum, gl::GLint, gl::GLint, gl::GLint, gl::GLint> MultiTexCoord4i;
    static Function<void, gl::GLenum, gl::GLint, gl::GLint, gl::GLint, gl::GLint> MultiTexCoord4iARB;
    static Function<void, gl::GLenum, const gl::GLint *> MultiTexCoord4iv;
    static Function<void, gl::GLenum, const gl::GLint *> MultiTexCoord4ivARB;
    static Function<void, gl::GLenum, gl::GLshort, gl::GLshort, gl::GLshort, gl::GLshort> MultiTexCoord4s;
    static Function<void, gl::GLenum, gl::GLshort, gl::GLshort, gl::GLshort, gl::GLshort> MultiTexCoord4sARB;
    static Function<void, gl::GLenum, const gl::GLshort *> MultiTexCoord4sv;
    static Function<void, gl::GLenum, const gl::GLshort *> MultiTexCoord4svARB;
    static Function<void, gl::GLenum, gl::GLfixed, gl::GLfixed, gl::GLfixed, gl::GLfixed> MultiTexCoord4xOES;
    static Function<void, gl::GLenum, const gl::GLfixed *> MultiTexCoord4xvOES;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLuint> MultiTexCoordP1ui;
    static Function<void, gl::GLenum, gl::GLenum, const gl::GLuint *> MultiTexCoordP1uiv;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLuint> MultiTexCoordP2ui;
    static Function<void, gl::GLenum, gl::GLenum, const gl::GLuint *> MultiTexCoordP2uiv;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLuint> MultiTexCoordP3ui;
    static Function<void, gl::GLenum, gl::GLenum, const gl::GLuint *> MultiTexCoordP3uiv;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLuint> MultiTexCoordP4ui;
    static Function<void, gl::GLenum, gl::GLenum, const gl::GLuint *> MultiTexCoordP4uiv;
    static Function<void, gl::GLenum, gl::GLint, gl::GLenum, gl::GLsizei, const void *> MultiTexCoordPointerEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLfloat> MultiTexEnvfEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, const gl::GLfloat *> MultiTexEnvfvEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLint> MultiTexEnviEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, const gl::GLint *> MultiTexEnvivEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLdouble> MultiTexGendEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, const gl::GLdouble *> MultiTexGendvEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLfloat> MultiTexGenfEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, const gl::GLfloat *> MultiTexGenfvEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLint> MultiTexGeniEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, const gl::GLint *> MultiTexGenivEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint, gl::GLint, gl::GLsizei, gl::GLint, gl::GLenum, gl::GLenum, const void *> MultiTexImage1DEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLint, gl::GLenum, gl::GLenum, const void *> MultiTexImage2DEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLsizei, gl::GLint, gl::GLenum, gl::GLenum, const void *> MultiTexImage3DEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, const gl::GLint *> MultiTexParameterIivEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, const gl::GLuint *> MultiTexParameterIuivEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLfloat> MultiTexParameterfEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, const gl::GLfloat *> MultiTexParameterfvEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLint> MultiTexParameteriEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, const gl::GLint *> MultiTexParameterivEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLuint> MultiTexRenderbufferEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint, gl::GLint, gl::GLsizei, gl::GLenum, gl::GLenum, const void *> MultiTexSubImage1DEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLenum, gl::GLenum, const void *> MultiTexSubImage2DEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLsizei, gl::GLenum, gl::GLenum, const void *> MultiTexSubImage3DEXT;
    static Function<void, gl::GLuint, gl::GLsizeiptr, const void *, gl::GLenum> NamedBufferData;
    static Function<void, gl::GLuint, gl::GLsizeiptr, const void *, gl::GLenum> NamedBufferDataEXT;
    static Function<void, gl::GLuint, gl::GLintptr, gl::GLsizeiptr, gl::GLboolean> NamedBufferPageCommitmentARB;
    static Function<void, gl::GLuint, gl::GLintptr, gl::GLsizeiptr, gl::GLboolean> NamedBufferPageCommitmentEXT;
    static Function<void, gl::GLuint, gl::GLsizeiptr, const void *, gl::BufferStorageMask> NamedBufferStorage;
    static Function<void, gl::GLuint, gl::GLsizeiptr, const void *, gl::BufferStorageMask> NamedBufferStorageEXT;
    static Function<void, gl::GLuint, gl::GLintptr, gl::GLsizeiptr, const void *> NamedBufferSubData;
    static Function<void, gl::GLuint, gl::GLintptr, gl::GLsizeiptr, const void *> NamedBufferSubDataEXT;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLintptr, gl::GLintptr, gl::GLsizeiptr> NamedCopyBufferSubDataEXT;
    static Function<void, gl::GLuint, gl::GLenum> NamedFramebufferDrawBuffer;
    static Function<void, gl::GLuint, gl::GLsizei, const gl::GLenum *> NamedFramebufferDrawBuffers;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint> NamedFramebufferParameteri;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint> NamedFramebufferParameteriEXT;
    static Function<void, gl::GLuint, gl::GLenum> NamedFramebufferReadBuffer;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLenum, gl::GLuint> NamedFramebufferRenderbuffer;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLenum, gl::GLuint> NamedFramebufferRenderbufferEXT;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLsizei, const gl::GLfloat *> NamedFramebufferSampleLocationsfvARB;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLsizei, const gl::GLfloat *> NamedFramebufferSampleLocationsfvNV;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLuint, gl::GLint> NamedFramebufferTexture;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLenum, gl::GLuint, gl::GLint> NamedFramebufferTexture1DEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLenum, gl::GLuint, gl::GLint> NamedFramebufferTexture2DEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLenum, gl::GLuint, gl::GLint, gl::GLint> NamedFramebufferTexture3DEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLuint, gl::GLint> NamedFramebufferTextureEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLuint, gl::GLint, gl::GLenum> NamedFramebufferTextureFaceEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLuint, gl::GLint, gl::GLint> NamedFramebufferTextureLayer;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLuint, gl::GLint, gl::GLint> NamedFramebufferTextureLayerEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLuint, gl::GLdouble, gl::GLdouble, gl::GLdouble, gl::GLdouble> NamedProgramLocalParameter4dEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLuint, const gl::GLdouble *> NamedProgramLocalParameter4dvEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLuint, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat> NamedProgramLocalParameter4fEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLuint, const gl::GLfloat *> NamedProgramLocalParameter4fvEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLuint, gl::GLint, gl::GLint, gl::GLint, gl::GLint> NamedProgramLocalParameterI4iEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLuint, const gl::GLint *> NamedProgramLocalParameterI4ivEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint> NamedProgramLocalParameterI4uiEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLuint, const gl::GLuint *> NamedProgramLocalParameterI4uivEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLuint, gl::GLsizei, const gl::GLfloat *> NamedProgramLocalParameters4fvEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLuint, gl::GLsizei, const gl::GLint *> NamedProgramLocalParametersI4ivEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLuint, gl::GLsizei, const gl::GLuint *> NamedProgramLocalParametersI4uivEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLenum, gl::GLsizei, const void *> NamedProgramStringEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLsizei, gl::GLsizei> NamedRenderbufferStorage;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLsizei, gl::GLsizei> NamedRenderbufferStorageEXT;
    static Function<void, gl::GLuint, gl::GLsizei, gl::GLenum, gl::GLsizei, gl::GLsizei> NamedRenderbufferStorageMultisample;
    static Function<void, gl::GLuint, gl::GLsizei, gl::GLsizei, gl::GLenum, gl::GLsizei, gl::GLsizei> NamedRenderbufferStorageMultisampleCoverageEXT;
    static Function<void, gl::GLuint, gl::GLsizei, gl::GLenum, gl::GLsizei, gl::GLsizei> NamedRenderbufferStorageMultisampleEXT;
    static Function<void, gl::GLenum, gl::GLint, const gl::GLchar *, gl::GLint, const gl::GLchar *> NamedStringARB;
    static Function<void, gl::GLuint, gl::GLenum> NewList;
    static Function<gl::GLuint, gl::GLsizei, const void *, gl::GLenum> NewObjectBufferATI;
    static Function<void, gl::GLbyte, gl::GLbyte, gl::GLbyte> Normal3b;
    static Function<void, const gl::GLbyte *> Normal3bv;
    static Function<void, gl::GLdouble, gl::GLdouble, gl::GLdouble> Normal3d;
    static Function<void, const gl::GLdouble *> Normal3dv;
    static Function<void, gl::GLfloat, gl::GLfloat, gl::GLfloat> Normal3f;
    static Function<void, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat> Normal3fVertex3fSUN;
    static Function<void, const gl::GLfloat *, const gl::GLfloat *> Normal3fVertex3fvSUN;
    static Function<void, const gl::GLfloat *> Normal3fv;
    static Function<void, gl::GLhalfNV, gl::GLhalfNV, gl::GLhalfNV> Normal3hNV;
    static Function<void, const gl::GLhalfNV *> Normal3hvNV;
    static Function<void, gl::GLint, gl::GLint, gl::GLint> Normal3i;
    static Function<void, const gl::GLint *> Normal3iv;
    static Function<void, gl::GLshort, gl::GLshort, gl::GLshort> Normal3s;
    static Function<void, const gl::GLshort *> Normal3sv;
    static Function<void, gl::GLfixed, gl::GLfixed, gl::GLfixed> Normal3xOES;
    static Function<void, const gl::GLfixed *> Normal3xvOES;
    static Function<void, gl::GLenum, gl::GLsizei> NormalFormatNV;
    static Function<void, gl::GLenum, gl::GLuint> NormalP3ui;
    static Function<void, gl::GLenum, const gl::GLuint *> NormalP3uiv;
    static Function<void, gl::GLenum, gl::GLsizei, const void *> NormalPointer;
    static Function<void, gl::GLenum, gl::GLsizei, gl::GLsizei, const void *> NormalPointerEXT;
    static Function<void, gl::GLenum, gl::GLint, const void **, gl::GLint> NormalPointerListIBM;
    static Function<void, gl::GLenum, const void **> NormalPointervINTEL;
    static Function<void, gl::GLenum, gl::GLbyte, gl::GLbyte, gl::GLbyte> NormalStream3bATI;
    static Function<void, gl::GLenum, const gl::GLbyte *> NormalStream3bvATI;
    static Function<void, gl::GLenum, gl::GLdouble, gl::GLdouble, gl::GLdouble> NormalStream3dATI;
    static Function<void, gl::GLenum, const gl::GLdouble *> NormalStream3dvATI;
    static Function<void, gl::GLenum, gl::GLfloat, gl::GLfloat, gl::GLfloat> NormalStream3fATI;
    static Function<void, gl::GLenum, const gl::GLfloat *> NormalStream3fvATI;
    static Function<void, gl::GLenum, gl::GLint, gl::GLint, gl::GLint> NormalStream3iATI;
    static Function<void, gl::GLenum, const gl::GLint *> NormalStream3ivATI;
    static Function<void, gl::GLenum, gl::GLshort, gl::GLshort, gl::GLshort> NormalStream3sATI;
    static Function<void, gl::GLenum, const gl::GLshort *> NormalStream3svATI;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLsizei, const gl::GLchar *> ObjectLabel;
    static Function<void, const void *, gl::GLsizei, const gl::GLchar *> ObjectPtrLabel;
    static Function<gl::GLenum, gl::GLenum, gl::GLuint, gl::GLenum> ObjectPurgeableAPPLE;
    static Function<gl::GLenum, gl::GLenum, gl::GLuint, gl::GLenum> ObjectUnpurgeableAPPLE;
    static Function<void, gl::GLdouble, gl::GLdouble, gl::GLdouble, gl::GLdouble, gl::GLdouble, gl::GLdouble> Ortho;
    static Function<void, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat> OrthofOES;
    static Function<void, gl::GLfixed, gl::GLfixed, gl::GLfixed, gl::GLfixed, gl::GLfixed, gl::GLfixed> OrthoxOES;
    static Function<void, gl::GLenum, gl::GLfloat> PNTrianglesfATI;
    static Function<void, gl::GLenum, gl::GLint> PNTrianglesiATI;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLenum> PassTexCoordATI;
    static Function<void, gl::GLfloat> PassThrough;
    static Function<void, gl::GLfixed> PassThroughxOES;
    static Function<void, gl::GLenum, const gl::GLfloat *> PatchParameterfv;
    static Function<void, gl::GLenum, gl::GLint> PatchParameteri;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, const gl::GLfloat *> PathColorGenNV;
    static Function<void, gl::GLuint, gl::GLsizei, const gl::GLubyte *, gl::GLsizei, gl::GLenum, const void *> PathCommandsNV;
    static Function<void, gl::GLuint, gl::GLsizei, gl::GLenum, const void *> PathCoordsNV;
    static Function<void, gl::GLenum> PathCoverDepthFuncNV;
    static Function<void, gl::GLuint, gl::GLsizei, const gl::GLfloat *> PathDashArrayNV;
    static Function<void, gl::GLenum> PathFogGenNV;
    static Function<gl::GLenum, gl::GLuint, gl::GLenum, const void *, gl::PathFontStyle, gl::GLuint, gl::GLsizei, gl::GLuint, gl::GLfloat> PathGlyphIndexArrayNV;
    static Function<gl::GLenum, gl::GLenum, const void *, gl::PathFontStyle, gl::GLuint, gl::GLfloat, gl::GLuint_array_2> PathGlyphIndexRangeNV;
    static Function<void, gl::GLuint, gl::GLenum, const void *, gl::PathFontStyle, gl::GLuint, gl::GLsizei, gl::GLenum, gl::GLuint, gl::GLfloat> PathGlyphRangeNV;
    static Function<void, gl::GLuint, gl::GLenum, const void *, gl::PathFontStyle, gl::GLsizei, gl::GLenum, const void *, gl::GLenum, gl::GLuint, gl::GLfloat> PathGlyphsNV;
    static Function<gl::GLenum, gl::GLuint, gl::GLenum, gl::GLsizeiptr, const void *, gl::GLsizei, gl::GLuint, gl::GLsizei, gl::GLuint, gl::GLfloat> PathMemoryGlyphIndexArrayNV;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLfloat> PathParameterfNV;
    static Function<void, gl::GLuint, gl::GLenum, const gl::GLfloat *> PathParameterfvNV;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint> PathParameteriNV;
    static Function<void, gl::GLuint, gl::GLenum, const gl::GLint *> PathParameterivNV;
    static Function<void, gl::GLfloat, gl::GLfloat> PathStencilDepthOffsetNV;
    static Function<void, gl::GLenum, gl::GLint, gl::GLuint> PathStencilFuncNV;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLsizei, const void *> PathStringNV;
    static Function<void, gl::GLuint, gl::GLsizei, gl::GLsizei, gl::GLsizei, const gl::GLubyte *, gl::GLsizei, gl::GLenum, const void *> PathSubCommandsNV;
    static Function<void, gl::GLuint, gl::GLsizei, gl::GLsizei, gl::GLenum, const void *> PathSubCoordsNV;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint, const gl::GLfloat *> PathTexGenNV;
    static Function<void> PauseTransformFeedback;
    static Function<void> PauseTransformFeedbackNV;
    static Function<void, gl::GLenum, gl::GLsizei, const void *> PixelDataRangeNV;
    static Function<void, gl::GLenum, gl::GLsizei, const gl::GLfloat *> PixelMapfv;
    static Function<void, gl::GLenum, gl::GLsizei, const gl::GLuint *> PixelMapuiv;
    static Function<void, gl::GLenum, gl::GLsizei, const gl::GLushort *> PixelMapusv;
    static Function<void, gl::GLenum, gl::GLint, const gl::GLfixed *> PixelMapx;
    static Function<void, gl::GLenum, gl::GLfloat> PixelStoref;
    static Function<void, gl::GLenum, gl::GLint> PixelStorei;
    static Function<void, gl::GLenum, gl::GLfixed> PixelStorex;
    static Function<void, gl::GLenum, gl::GLfloat> PixelTexGenParameterfSGIS;
    static Function<void, gl::GLenum, const gl::GLfloat *> PixelTexGenParameterfvSGIS;
    static Function<void, gl::GLenum, gl::GLint> PixelTexGenParameteriSGIS;
    static Function<void, gl::GLenum, const gl::GLint *> PixelTexGenParameterivSGIS;
    static Function<void, gl::GLenum> PixelTexGenSGIX;
    static Function<void, gl::GLenum, gl::GLfloat> PixelTransferf;
    static Function<void, gl::GLenum, gl::GLint> PixelTransferi;
    static Function<void, gl::GLenum, gl::GLfixed> PixelTransferxOES;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfloat> PixelTransformParameterfEXT;
    static Function<void, gl::GLenum, gl::GLenum, const gl::GLfloat *> PixelTransformParameterfvEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint> PixelTransformParameteriEXT;
    static Function<void, gl::GLenum, gl::GLenum, const gl::GLint *> PixelTransformParameterivEXT;
    static Function<void, gl::GLfloat, gl::GLfloat> PixelZoom;
    static Function<void, gl::GLfixed, gl::GLfixed> PixelZoomxOES;
    static Function<gl::GLboolean, gl::GLuint, gl::GLsizei, gl::GLsizei, gl::GLfloat, gl::GLfloat *, gl::GLfloat *, gl::GLfloat *, gl::GLfloat *> PointAlongPathNV;
    static Function<void, gl::GLenum, gl::GLfloat> PointParameterf;
    static Function<void, gl::GLenum, gl::GLfloat> PointParameterfARB;
    static Function<void, gl::GLenum, gl::GLfloat> PointParameterfEXT;
    static Function<void, gl::GLenum, gl::GLfloat> PointParameterfSGIS;
    static Function<void, gl::GLenum, const gl::GLfloat *> PointParameterfv;
    static Function<void, gl::GLenum, const gl::GLfloat *> PointParameterfvARB;
    static Function<void, gl::GLenum, const gl::GLfloat *> PointParameterfvEXT;
    static Function<void, gl::GLenum, const gl::GLfloat *> PointParameterfvSGIS;
    static Function<void, gl::GLenum, gl::GLint> PointParameteri;
    static Function<void, gl::GLenum, gl::GLint> PointParameteriNV;
    static Function<void, gl::GLenum, const gl::GLint *> PointParameteriv;
    static Function<void, gl::GLenum, const gl::GLint *> PointParameterivNV;
    static Function<void, gl::GLenum, const gl::GLfixed *> PointParameterxvOES;
    static Function<void, gl::GLfloat> PointSize;
    static Function<void, gl::GLfixed> PointSizexOES;
    static Function<gl::GLint, gl::GLuint *> PollAsyncSGIX;
    static Function<gl::GLint, gl::GLint *> PollInstrumentsSGIX;
    static Function<void, gl::GLenum, gl::GLenum> PolygonMode;
    static Function<void, gl::GLfloat, gl::GLfloat> PolygonOffset;
    static Function<void, gl::GLfloat, gl::GLfloat, gl::GLfloat> PolygonOffsetClampEXT;
    static Function<void, gl::GLfloat, gl::GLfloat> PolygonOffsetEXT;
    static Function<void, gl::GLfixed, gl::GLfixed> PolygonOffsetxOES;
    static Function<void, const gl::GLubyte *> PolygonStipple;
    static Function<void> PopAttrib;
    static Function<void> PopClientAttrib;
    static Function<void> PopDebugGroup;
    static Function<void> PopGroupMarkerEXT;
    static Function<void> PopMatrix;
    static Function<void> PopName;
    static Function<void, gl::GLuint, gl::GLuint64EXT, gl::GLuint, gl::GLuint, gl::GLenum, gl::GLenum, gl::GLuint, gl::GLenum, gl::GLuint, gl::GLenum, gl::GLuint, gl::GLenum, gl::GLuint> PresentFrameDualFillNV;
    static Function<void, gl::GLuint, gl::GLuint64EXT, gl::GLuint, gl::GLuint, gl::GLenum, gl::GLenum, gl::GLuint, gl::GLuint, gl::GLenum, gl::GLuint, gl::GLuint> PresentFrameKeyedNV;
    static Function<void, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat> PrimitiveBoundingBoxARB;
    static Function<void, gl::GLuint> PrimitiveRestartIndex;
    static Function<void, gl::GLuint> PrimitiveRestartIndexNV;
    static Function<void> PrimitiveRestartNV;
    static Function<void, gl::GLsizei, const gl::GLuint *, const gl::GLfloat *> PrioritizeTextures;
    static Function<void, gl::GLsizei, const gl::GLuint *, const gl::GLclampf *> PrioritizeTexturesEXT;
    static Function<void, gl::GLsizei, const gl::GLuint *, const gl::GLfixed *> PrioritizeTexturesxOES;
    static Function<void, gl::GLuint, gl::GLenum, const void *, gl::GLsizei> ProgramBinary;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLuint, gl::GLsizei, const gl::GLint *> ProgramBufferParametersIivNV;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLuint, gl::GLsizei, const gl::GLuint *> ProgramBufferParametersIuivNV;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLuint, gl::GLsizei, const gl::GLfloat *> ProgramBufferParametersfvNV;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLdouble, gl::GLdouble, gl::GLdouble, gl::GLdouble> ProgramEnvParameter4dARB;
    static Function<void, gl::GLenum, gl::GLuint, const gl::GLdouble *> ProgramEnvParameter4dvARB;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat> ProgramEnvParameter4fARB;
    static Function<void, gl::GLenum, gl::GLuint, const gl::GLfloat *> ProgramEnvParameter4fvARB;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLint, gl::GLint, gl::GLint, gl::GLint> ProgramEnvParameterI4iNV;
    static Function<void, gl::GLenum, gl::GLuint, const gl::GLint *> ProgramEnvParameterI4ivNV;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint> ProgramEnvParameterI4uiNV;
    static Function<void, gl::GLenum, gl::GLuint, const gl::GLuint *> ProgramEnvParameterI4uivNV;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLsizei, const gl::GLfloat *> ProgramEnvParameters4fvEXT;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLsizei, const gl::GLint *> ProgramEnvParametersI4ivNV;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLsizei, const gl::GLuint *> ProgramEnvParametersI4uivNV;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLdouble, gl::GLdouble, gl::GLdouble, gl::GLdouble> ProgramLocalParameter4dARB;
    static Function<void, gl::GLenum, gl::GLuint, const gl::GLdouble *> ProgramLocalParameter4dvARB;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat> ProgramLocalParameter4fARB;
    static Function<void, gl::GLenum, gl::GLuint, const gl::GLfloat *> ProgramLocalParameter4fvARB;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLint, gl::GLint, gl::GLint, gl::GLint> ProgramLocalParameterI4iNV;
    static Function<void, gl::GLenum, gl::GLuint, const gl::GLint *> ProgramLocalParameterI4ivNV;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint> ProgramLocalParameterI4uiNV;
    static Function<void, gl::GLenum, gl::GLuint, const gl::GLuint *> ProgramLocalParameterI4uivNV;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLsizei, const gl::GLfloat *> ProgramLocalParameters4fvEXT;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLsizei, const gl::GLint *> ProgramLocalParametersI4ivNV;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLsizei, const gl::GLuint *> ProgramLocalParametersI4uivNV;
    static Function<void, gl::GLuint, gl::GLsizei, const gl::GLubyte *, gl::GLdouble, gl::GLdouble, gl::GLdouble, gl::GLdouble> ProgramNamedParameter4dNV;
    static Function<void, gl::GLuint, gl::GLsizei, const gl::GLubyte *, const gl::GLdouble *> ProgramNamedParameter4dvNV;
    static Function<void, gl::GLuint, gl::GLsizei, const gl::GLubyte *, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat> ProgramNamedParameter4fNV;
    static Function<void, gl::GLuint, gl::GLsizei, const gl::GLubyte *, const gl::GLfloat *> ProgramNamedParameter4fvNV;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLdouble, gl::GLdouble, gl::GLdouble, gl::GLdouble> ProgramParameter4dNV;
    static Function<void, gl::GLenum, gl::GLuint, const gl::GLdouble *> ProgramParameter4dvNV;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat> ProgramParameter4fNV;
    static Function<void, gl::GLenum, gl::GLuint, const gl::GLfloat *> ProgramParameter4fvNV;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint> ProgramParameteri;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint> ProgramParameteriARB;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint> ProgramParameteriEXT;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLsizei, const gl::GLdouble *> ProgramParameters4dvNV;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLsizei, const gl::GLfloat *> ProgramParameters4fvNV;
    static Function<void, gl::GLuint, gl::GLint, gl::GLenum, gl::GLint, const gl::GLfloat *> ProgramPathFragmentInputGenNV;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLsizei, const void *> ProgramStringARB;
    static Function<void, gl::GLenum, gl::GLsizei, const gl::GLuint *> ProgramSubroutineParametersuivNV;
    static Function<void, gl::GLuint, gl::GLint, gl::GLdouble> ProgramUniform1d;
    static Function<void, gl::GLuint, gl::GLint, gl::GLdouble> ProgramUniform1dEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLdouble *> ProgramUniform1dv;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLdouble *> ProgramUniform1dvEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLfloat> ProgramUniform1f;
    static Function<void, gl::GLuint, gl::GLint, gl::GLfloat> ProgramUniform1fEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLfloat *> ProgramUniform1fv;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLfloat *> ProgramUniform1fvEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLint> ProgramUniform1i;
    static Function<void, gl::GLuint, gl::GLint, gl::GLint64> ProgramUniform1i64ARB;
    static Function<void, gl::GLuint, gl::GLint, gl::GLint64EXT> ProgramUniform1i64NV;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLint64 *> ProgramUniform1i64vARB;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLint64EXT *> ProgramUniform1i64vNV;
    static Function<void, gl::GLuint, gl::GLint, gl::GLint> ProgramUniform1iEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLint *> ProgramUniform1iv;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLint *> ProgramUniform1ivEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLuint> ProgramUniform1ui;
    static Function<void, gl::GLuint, gl::GLint, gl::GLuint64> ProgramUniform1ui64ARB;
    static Function<void, gl::GLuint, gl::GLint, gl::GLuint64EXT> ProgramUniform1ui64NV;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLuint64 *> ProgramUniform1ui64vARB;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLuint64EXT *> ProgramUniform1ui64vNV;
    static Function<void, gl::GLuint, gl::GLint, gl::GLuint> ProgramUniform1uiEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLuint *> ProgramUniform1uiv;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLuint *> ProgramUniform1uivEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLdouble, gl::GLdouble> ProgramUniform2d;
    static Function<void, gl::GLuint, gl::GLint, gl::GLdouble, gl::GLdouble> ProgramUniform2dEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLdouble *> ProgramUniform2dv;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLdouble *> ProgramUniform2dvEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLfloat, gl::GLfloat> ProgramUniform2f;
    static Function<void, gl::GLuint, gl::GLint, gl::GLfloat, gl::GLfloat> ProgramUniform2fEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLfloat *> ProgramUniform2fv;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLfloat *> ProgramUniform2fvEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLint, gl::GLint> ProgramUniform2i;
    static Function<void, gl::GLuint, gl::GLint, gl::GLint64, gl::GLint64> ProgramUniform2i64ARB;
    static Function<void, gl::GLuint, gl::GLint, gl::GLint64EXT, gl::GLint64EXT> ProgramUniform2i64NV;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLint64 *> ProgramUniform2i64vARB;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLint64EXT *> ProgramUniform2i64vNV;
    static Function<void, gl::GLuint, gl::GLint, gl::GLint, gl::GLint> ProgramUniform2iEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLint *> ProgramUniform2iv;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLint *> ProgramUniform2ivEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLuint, gl::GLuint> ProgramUniform2ui;
    static Function<void, gl::GLuint, gl::GLint, gl::GLuint64, gl::GLuint64> ProgramUniform2ui64ARB;
    static Function<void, gl::GLuint, gl::GLint, gl::GLuint64EXT, gl::GLuint64EXT> ProgramUniform2ui64NV;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLuint64 *> ProgramUniform2ui64vARB;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLuint64EXT *> ProgramUniform2ui64vNV;
    static Function<void, gl::GLuint, gl::GLint, gl::GLuint, gl::GLuint> ProgramUniform2uiEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLuint *> ProgramUniform2uiv;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLuint *> ProgramUniform2uivEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLdouble, gl::GLdouble, gl::GLdouble> ProgramUniform3d;
    static Function<void, gl::GLuint, gl::GLint, gl::GLdouble, gl::GLdouble, gl::GLdouble> ProgramUniform3dEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLdouble *> ProgramUniform3dv;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLdouble *> ProgramUniform3dvEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLfloat, gl::GLfloat, gl::GLfloat> ProgramUniform3f;
    static Function<void, gl::GLuint, gl::GLint, gl::GLfloat, gl::GLfloat, gl::GLfloat> ProgramUniform3fEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLfloat *> ProgramUniform3fv;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLfloat *> ProgramUniform3fvEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLint, gl::GLint, gl::GLint> ProgramUniform3i;
    static Function<void, gl::GLuint, gl::GLint, gl::GLint64, gl::GLint64, gl::GLint64> ProgramUniform3i64ARB;
    static Function<void, gl::GLuint, gl::GLint, gl::GLint64EXT, gl::GLint64EXT, gl::GLint64EXT> ProgramUniform3i64NV;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLint64 *> ProgramUniform3i64vARB;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLint64EXT *> ProgramUniform3i64vNV;
    static Function<void, gl::GLuint, gl::GLint, gl::GLint, gl::GLint, gl::GLint> ProgramUniform3iEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLint *> ProgramUniform3iv;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLint *> ProgramUniform3ivEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLuint, gl::GLuint, gl::GLuint> ProgramUniform3ui;
    static Function<void, gl::GLuint, gl::GLint, gl::GLuint64, gl::GLuint64, gl::GLuint64> ProgramUniform3ui64ARB;
    static Function<void, gl::GLuint, gl::GLint, gl::GLuint64EXT, gl::GLuint64EXT, gl::GLuint64EXT> ProgramUniform3ui64NV;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLuint64 *> ProgramUniform3ui64vARB;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLuint64EXT *> ProgramUniform3ui64vNV;
    static Function<void, gl::GLuint, gl::GLint, gl::GLuint, gl::GLuint, gl::GLuint> ProgramUniform3uiEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLuint *> ProgramUniform3uiv;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLuint *> ProgramUniform3uivEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLdouble, gl::GLdouble, gl::GLdouble, gl::GLdouble> ProgramUniform4d;
    static Function<void, gl::GLuint, gl::GLint, gl::GLdouble, gl::GLdouble, gl::GLdouble, gl::GLdouble> ProgramUniform4dEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLdouble *> ProgramUniform4dv;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLdouble *> ProgramUniform4dvEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat> ProgramUniform4f;
    static Function<void, gl::GLuint, gl::GLint, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat> ProgramUniform4fEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLfloat *> ProgramUniform4fv;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLfloat *> ProgramUniform4fvEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLint> ProgramUniform4i;
    static Function<void, gl::GLuint, gl::GLint, gl::GLint64, gl::GLint64, gl::GLint64, gl::GLint64> ProgramUniform4i64ARB;
    static Function<void, gl::GLuint, gl::GLint, gl::GLint64EXT, gl::GLint64EXT, gl::GLint64EXT, gl::GLint64EXT> ProgramUniform4i64NV;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLint64 *> ProgramUniform4i64vARB;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLint64EXT *> ProgramUniform4i64vNV;
    static Function<void, gl::GLuint, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLint> ProgramUniform4iEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLint *> ProgramUniform4iv;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLint *> ProgramUniform4ivEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint> ProgramUniform4ui;
    static Function<void, gl::GLuint, gl::GLint, gl::GLuint64, gl::GLuint64, gl::GLuint64, gl::GLuint64> ProgramUniform4ui64ARB;
    static Function<void, gl::GLuint, gl::GLint, gl::GLuint64EXT, gl::GLuint64EXT, gl::GLuint64EXT, gl::GLuint64EXT> ProgramUniform4ui64NV;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLuint64 *> ProgramUniform4ui64vARB;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLuint64EXT *> ProgramUniform4ui64vNV;
    static Function<void, gl::GLuint, gl::GLint, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint> ProgramUniform4uiEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLuint *> ProgramUniform4uiv;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLuint *> ProgramUniform4uivEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLuint64> ProgramUniformHandleui64ARB;
    static Function<void, gl::GLuint, gl::GLint, gl::GLuint64> ProgramUniformHandleui64NV;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLuint64 *> ProgramUniformHandleui64vARB;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLuint64 *> ProgramUniformHandleui64vNV;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLdouble *> ProgramUniformMatrix2dv;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLdouble *> ProgramUniformMatrix2dvEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLfloat *> ProgramUniformMatrix2fv;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLfloat *> ProgramUniformMatrix2fvEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLdouble *> ProgramUniformMatrix2x3dv;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLdouble *> ProgramUniformMatrix2x3dvEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLfloat *> ProgramUniformMatrix2x3fv;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLfloat *> ProgramUniformMatrix2x3fvEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLdouble *> ProgramUniformMatrix2x4dv;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLdouble *> ProgramUniformMatrix2x4dvEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLfloat *> ProgramUniformMatrix2x4fv;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLfloat *> ProgramUniformMatrix2x4fvEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLdouble *> ProgramUniformMatrix3dv;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLdouble *> ProgramUniformMatrix3dvEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLfloat *> ProgramUniformMatrix3fv;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLfloat *> ProgramUniformMatrix3fvEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLdouble *> ProgramUniformMatrix3x2dv;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLdouble *> ProgramUniformMatrix3x2dvEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLfloat *> ProgramUniformMatrix3x2fv;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLfloat *> ProgramUniformMatrix3x2fvEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLdouble *> ProgramUniformMatrix3x4dv;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLdouble *> ProgramUniformMatrix3x4dvEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLfloat *> ProgramUniformMatrix3x4fv;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLfloat *> ProgramUniformMatrix3x4fvEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLdouble *> ProgramUniformMatrix4dv;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLdouble *> ProgramUniformMatrix4dvEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLfloat *> ProgramUniformMatrix4fv;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLfloat *> ProgramUniformMatrix4fvEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLdouble *> ProgramUniformMatrix4x2dv;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLdouble *> ProgramUniformMatrix4x2dvEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLfloat *> ProgramUniformMatrix4x2fv;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLfloat *> ProgramUniformMatrix4x2fvEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLdouble *> ProgramUniformMatrix4x3dv;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLdouble *> ProgramUniformMatrix4x3dvEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLfloat *> ProgramUniformMatrix4x3fv;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLfloat *> ProgramUniformMatrix4x3fvEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLuint64EXT> ProgramUniformui64NV;
    static Function<void, gl::GLuint, gl::GLint, gl::GLsizei, const gl::GLuint64EXT *> ProgramUniformui64vNV;
    static Function<void, gl::GLenum, gl::GLint> ProgramVertexLimitNV;
    static Function<void, gl::GLenum> ProvokingVertex;
    static Function<void, gl::GLenum> ProvokingVertexEXT;
    static Function<void, gl::AttribMask> PushAttrib;
    static Function<void, gl::ClientAttribMask> PushClientAttrib;
    static Function<void, gl::ClientAttribMask> PushClientAttribDefaultEXT;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLsizei, const gl::GLchar *> PushDebugGroup;
    static Function<void, gl::GLsizei, const gl::GLchar *> PushGroupMarkerEXT;
    static Function<void> PushMatrix;
    static Function<void, gl::GLuint> PushName;
    static Function<void, gl::GLuint, gl::GLenum> QueryCounter;
    static Function<gl::GLbitfield, gl::GLfixed *, gl::GLint *> QueryMatrixxOES;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLenum, gl::GLuint> QueryObjectParameteruiAMD;
    static Function<void, gl::GLdouble, gl::GLdouble> RasterPos2d;
    static Function<void, const gl::GLdouble *> RasterPos2dv;
    static Function<void, gl::GLfloat, gl::GLfloat> RasterPos2f;
    static Function<void, const gl::GLfloat *> RasterPos2fv;
    static Function<void, gl::GLint, gl::GLint> RasterPos2i;
    static Function<void, const gl::GLint *> RasterPos2iv;
    static Function<void, gl::GLshort, gl::GLshort> RasterPos2s;
    static Function<void, const gl::GLshort *> RasterPos2sv;
    static Function<void, gl::GLfixed, gl::GLfixed> RasterPos2xOES;
    static Function<void, const gl::GLfixed *> RasterPos2xvOES;
    static Function<void, gl::GLdouble, gl::GLdouble, gl::GLdouble> RasterPos3d;
    static Function<void, const gl::GLdouble *> RasterPos3dv;
    static Function<void, gl::GLfloat, gl::GLfloat, gl::GLfloat> RasterPos3f;
    static Function<void, const gl::GLfloat *> RasterPos3fv;
    static Function<void, gl::GLint, gl::GLint, gl::GLint> RasterPos3i;
    static Function<void, const gl::GLint *> RasterPos3iv;
    static Function<void, gl::GLshort, gl::GLshort, gl::GLshort> RasterPos3s;
    static Function<void, const gl::GLshort *> RasterPos3sv;
    static Function<void, gl::GLfixed, gl::GLfixed, gl::GLfixed> RasterPos3xOES;
    static Function<void, const gl::GLfixed *> RasterPos3xvOES;
    static Function<void, gl::GLdouble, gl::GLdouble, gl::GLdouble, gl::GLdouble> RasterPos4d;
    static Function<void, const gl::GLdouble *> RasterPos4dv;
    static Function<void, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat> RasterPos4f;
    static Function<void, const gl::GLfloat *> RasterPos4fv;
    static Function<void, gl::GLint, gl::GLint, gl::GLint, gl::GLint> RasterPos4i;
    static Function<void, const gl::GLint *> RasterPos4iv;
    static Function<void, gl::GLshort, gl::GLshort, gl::GLshort, gl::GLshort> RasterPos4s;
    static Function<void, const gl::GLshort *> RasterPos4sv;
    static Function<void, gl::GLfixed, gl::GLfixed, gl::GLfixed, gl::GLfixed> RasterPos4xOES;
    static Function<void, const gl::GLfixed *> RasterPos4xvOES;
    static Function<void, gl::GLuint, gl::GLboolean> RasterSamplesEXT;
    static Function<void, gl::GLenum> ReadBuffer;
    static Function<void, gl::GLint> ReadInstrumentsSGIX;
    static Function<void, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLenum, gl::GLenum, void *> ReadPixels;
    static Function<void, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLenum, gl::GLenum, gl::GLsizei, void *> ReadnPixels;
    static Function<void, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLenum, gl::GLenum, gl::GLsizei, void *> ReadnPixelsARB;
    static Function<void, gl::GLdouble, gl::GLdouble, gl::GLdouble, gl::GLdouble> Rectd;
    static Function<void, const gl::GLdouble *, const gl::GLdouble *> Rectdv;
    static Function<void, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat> Rectf;
    static Function<void, const gl::GLfloat *, const gl::GLfloat *> Rectfv;
    static Function<void, gl::GLint, gl::GLint, gl::GLint, gl::GLint> Recti;
    static Function<void, const gl::GLint *, const gl::GLint *> Rectiv;
    static Function<void, gl::GLshort, gl::GLshort, gl::GLshort, gl::GLshort> Rects;
    static Function<void, const gl::GLshort *, const gl::GLshort *> Rectsv;
    static Function<void, gl::GLfixed, gl::GLfixed, gl::GLfixed, gl::GLfixed> RectxOES;
    static Function<void, const gl::GLfixed *, const gl::GLfixed *> RectxvOES;
    static Function<void, const gl::GLdouble *> ReferencePlaneSGIX;
    static Function<void> ReleaseShaderCompiler;
    static Function<gl::GLint, gl::GLenum> RenderMode;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLsizei, gl::GLsizei> RenderbufferStorage;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLsizei, gl::GLsizei> RenderbufferStorageEXT;
    static Function<void, gl::GLenum, gl::GLsizei, gl::GLenum, gl::GLsizei, gl::GLsizei> RenderbufferStorageMultisample;
    static Function<void, gl::GLenum, gl::GLsizei, gl::GLsizei, gl::GLenum, gl::GLsizei, gl::GLsizei> RenderbufferStorageMultisampleCoverageNV;
    static Function<void, gl::GLenum, gl::GLsizei, gl::GLenum, gl::GLsizei, gl::GLsizei> RenderbufferStorageMultisampleEXT;
    static Function<void, gl::GLenum, gl::GLsizei, const void **> ReplacementCodePointerSUN;
    static Function<void, gl::GLubyte> ReplacementCodeubSUN;
    static Function<void, const gl::GLubyte *> ReplacementCodeubvSUN;
    static Function<void, gl::GLuint, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat> ReplacementCodeuiColor3fVertex3fSUN;
    static Function<void, const gl::GLuint *, const gl::GLfloat *, const gl::GLfloat *> ReplacementCodeuiColor3fVertex3fvSUN;
    static Function<void, gl::GLuint, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat> ReplacementCodeuiColor4fNormal3fVertex3fSUN;
    static Function<void, const gl::GLuint *, const gl::GLfloat *, const gl::GLfloat *, const gl::GLfloat *> ReplacementCodeuiColor4fNormal3fVertex3fvSUN;
    static Function<void, gl::GLuint, gl::GLubyte, gl::GLubyte, gl::GLubyte, gl::GLubyte, gl::GLfloat, gl::GLfloat, gl::GLfloat> ReplacementCodeuiColor4ubVertex3fSUN;
    static Function<void, const gl::GLuint *, const gl::GLubyte *, const gl::GLfloat *> ReplacementCodeuiColor4ubVertex3fvSUN;
    static Function<void, gl::GLuint, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat> ReplacementCodeuiNormal3fVertex3fSUN;
    static Function<void, const gl::GLuint *, const gl::GLfloat *, const gl::GLfloat *> ReplacementCodeuiNormal3fVertex3fvSUN;
    static Function<void, gl::GLuint> ReplacementCodeuiSUN;
    static Function<void, gl::GLuint, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat> ReplacementCodeuiTexCoord2fColor4fNormal3fVertex3fSUN;
    static Function<void, const gl::GLuint *, const gl::GLfloat *, const gl::GLfloat *, const gl::GLfloat *, const gl::GLfloat *> ReplacementCodeuiTexCoord2fColor4fNormal3fVertex3fvSUN;
    static Function<void, gl::GLuint, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat> ReplacementCodeuiTexCoord2fNormal3fVertex3fSUN;
    static Function<void, const gl::GLuint *, const gl::GLfloat *, const gl::GLfloat *, const gl::GLfloat *> ReplacementCodeuiTexCoord2fNormal3fVertex3fvSUN;
    static Function<void, gl::GLuint, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat> ReplacementCodeuiTexCoord2fVertex3fSUN;
    static Function<void, const gl::GLuint *, const gl::GLfloat *, const gl::GLfloat *> ReplacementCodeuiTexCoord2fVertex3fvSUN;
    static Function<void, gl::GLuint, gl::GLfloat, gl::GLfloat, gl::GLfloat> ReplacementCodeuiVertex3fSUN;
    static Function<void, const gl::GLuint *, const gl::GLfloat *> ReplacementCodeuiVertex3fvSUN;
    static Function<void, const gl::GLuint *> ReplacementCodeuivSUN;
    static Function<void, gl::GLushort> ReplacementCodeusSUN;
    static Function<void, const gl::GLushort *> ReplacementCodeusvSUN;
    static Function<void, gl::GLsizei, const gl::GLuint *> RequestResidentProgramsNV;
    static Function<void, gl::GLenum> ResetHistogram;
    static Function<void, gl::GLenum> ResetHistogramEXT;
    static Function<void, gl::GLenum> ResetMinmax;
    static Function<void, gl::GLenum> ResetMinmaxEXT;
    static Function<void> ResizeBuffersMESA;
    static Function<void> ResolveDepthValuesNV;
    static Function<void> ResumeTransformFeedback;
    static Function<void> ResumeTransformFeedbackNV;
    static Function<void, gl::GLdouble, gl::GLdouble, gl::GLdouble, gl::GLdouble> Rotated;
    static Function<void, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat> Rotatef;
    static Function<void, gl::GLfixed, gl::GLfixed, gl::GLfixed, gl::GLfixed> RotatexOES;
    static Function<void, gl::GLfloat, gl::GLboolean> SampleCoverage;
    static Function<void, gl::GLfloat, gl::GLboolean> SampleCoverageARB;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLenum> SampleMapATI;
    static Function<void, gl::GLclampf, gl::GLboolean> SampleMaskEXT;
    static Function<void, gl::GLuint, gl::GLbitfield> SampleMaskIndexedNV;
    static Function<void, gl::GLclampf, gl::GLboolean> SampleMaskSGIS;
    static Function<void, gl::GLuint, gl::GLbitfield> SampleMaski;
    static Function<void, gl::GLenum> SamplePatternEXT;
    static Function<void, gl::GLenum> SamplePatternSGIS;
    static Function<void, gl::GLuint, gl::GLenum, const gl::GLint *> SamplerParameterIiv;
    static Function<void, gl::GLuint, gl::GLenum, const gl::GLuint *> SamplerParameterIuiv;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLfloat> SamplerParameterf;
    static Function<void, gl::GLuint, gl::GLenum, const gl::GLfloat *> SamplerParameterfv;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint> SamplerParameteri;
    static Function<void, gl::GLuint, gl::GLenum, const gl::GLint *> SamplerParameteriv;
    static Function<void, gl::GLdouble, gl::GLdouble, gl::GLdouble> Scaled;
    static Function<void, gl::GLfloat, gl::GLfloat, gl::GLfloat> Scalef;
    static Function<void, gl::GLfixed, gl::GLfixed, gl::GLfixed> ScalexOES;
    static Function<void, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei> Scissor;
    static Function<void, gl::GLuint, gl::GLsizei, const gl::GLint *> ScissorArrayv;
    static Function<void, gl::GLuint, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei> ScissorIndexed;
    static Function<void, gl::GLuint, const gl::GLint *> ScissorIndexedv;
    static Function<void, gl::GLbyte, gl::GLbyte, gl::GLbyte> SecondaryColor3b;
    static Function<void, gl::GLbyte, gl::GLbyte, gl::GLbyte> SecondaryColor3bEXT;
    static Function<void, const gl::GLbyte *> SecondaryColor3bv;
    static Function<void, const gl::GLbyte *> SecondaryColor3bvEXT;
    static Function<void, gl::GLdouble, gl::GLdouble, gl::GLdouble> SecondaryColor3d;
    static Function<void, gl::GLdouble, gl::GLdouble, gl::GLdouble> SecondaryColor3dEXT;
    static Function<void, const gl::GLdouble *> SecondaryColor3dv;
    static Function<void, const gl::GLdouble *> SecondaryColor3dvEXT;
    static Function<void, gl::GLfloat, gl::GLfloat, gl::GLfloat> SecondaryColor3f;
    static Function<void, gl::GLfloat, gl::GLfloat, gl::GLfloat> SecondaryColor3fEXT;
    static Function<void, const gl::GLfloat *> SecondaryColor3fv;
    static Function<void, const gl::GLfloat *> SecondaryColor3fvEXT;
    static Function<void, gl::GLhalfNV, gl::GLhalfNV, gl::GLhalfNV> SecondaryColor3hNV;
    static Function<void, const gl::GLhalfNV *> SecondaryColor3hvNV;
    static Function<void, gl::GLint, gl::GLint, gl::GLint> SecondaryColor3i;
    static Function<void, gl::GLint, gl::GLint, gl::GLint> SecondaryColor3iEXT;
    static Function<void, const gl::GLint *> SecondaryColor3iv;
    static Function<void, const gl::GLint *> SecondaryColor3ivEXT;
    static Function<void, gl::GLshort, gl::GLshort, gl::GLshort> SecondaryColor3s;
    static Function<void, gl::GLshort, gl::GLshort, gl::GLshort> SecondaryColor3sEXT;
    static Function<void, const gl::GLshort *> SecondaryColor3sv;
    static Function<void, const gl::GLshort *> SecondaryColor3svEXT;
    static Function<void, gl::GLubyte, gl::GLubyte, gl::GLubyte> SecondaryColor3ub;
    static Function<void, gl::GLubyte, gl::GLubyte, gl::GLubyte> SecondaryColor3ubEXT;
    static Function<void, const gl::GLubyte *> SecondaryColor3ubv;
    static Function<void, const gl::GLubyte *> SecondaryColor3ubvEXT;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLuint> SecondaryColor3ui;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLuint> SecondaryColor3uiEXT;
    static Function<void, const gl::GLuint *> SecondaryColor3uiv;
    static Function<void, const gl::GLuint *> SecondaryColor3uivEXT;
    static Function<void, gl::GLushort, gl::GLushort, gl::GLushort> SecondaryColor3us;
    static Function<void, gl::GLushort, gl::GLushort, gl::GLushort> SecondaryColor3usEXT;
    static Function<void, const gl::GLushort *> SecondaryColor3usv;
    static Function<void, const gl::GLushort *> SecondaryColor3usvEXT;
    static Function<void, gl::GLint, gl::GLenum, gl::GLsizei> SecondaryColorFormatNV;
    static Function<void, gl::GLenum, gl::GLuint> SecondaryColorP3ui;
    static Function<void, gl::GLenum, const gl::GLuint *> SecondaryColorP3uiv;
    static Function<void, gl::GLint, gl::GLenum, gl::GLsizei, const void *> SecondaryColorPointer;
    static Function<void, gl::GLint, gl::GLenum, gl::GLsizei, const void *> SecondaryColorPointerEXT;
    static Function<void, gl::GLint, gl::GLenum, gl::GLint, const void **, gl::GLint> SecondaryColorPointerListIBM;
    static Function<void, gl::GLsizei, gl::GLuint *> SelectBuffer;
    static Function<void, gl::GLuint, gl::GLboolean, gl::GLuint, gl::GLint, gl::GLuint *> SelectPerfMonitorCountersAMD;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLsizei, gl::GLsizei, gl::GLenum, gl::GLenum, const void *, const void *> SeparableFilter2D;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLsizei, gl::GLsizei, gl::GLenum, gl::GLenum, const void *, const void *> SeparableFilter2DEXT;
    static Function<void, gl::GLuint> SetFenceAPPLE;
    static Function<void, gl::GLuint, gl::GLenum> SetFenceNV;
    static Function<void, gl::GLuint, const gl::GLfloat *> SetFragmentShaderConstantATI;
    static Function<void, gl::GLuint, gl::GLenum, const void *> SetInvariantEXT;
    static Function<void, gl::GLuint, gl::GLenum, const void *> SetLocalConstantEXT;
    static Function<void, gl::GLenum, gl::GLuint, const gl::GLfloat *> SetMultisamplefvAMD;
    static Function<void, gl::GLenum> ShadeModel;
    static Function<void, gl::GLsizei, const gl::GLuint *, gl::GLenum, const void *, gl::GLsizei> ShaderBinary;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLuint> ShaderOp1EXT;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLuint, gl::GLuint> ShaderOp2EXT;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint> ShaderOp3EXT;
    static Function<void, gl::GLuint, gl::GLsizei, const gl::GLchar *const*, const gl::GLint *> ShaderSource;
    static Function<void, gl::GLhandleARB, gl::GLsizei, const gl::GLcharARB **, const gl::GLint *> ShaderSourceARB;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLuint> ShaderStorageBlockBinding;
    static Function<void, gl::GLenum, gl::GLsizei, const gl::GLfloat *> SharpenTexFuncSGIS;
    static Function<void, gl::GLenum, gl::GLfloat> SpriteParameterfSGIX;
    static Function<void, gl::GLenum, const gl::GLfloat *> SpriteParameterfvSGIX;
    static Function<void, gl::GLenum, gl::GLint> SpriteParameteriSGIX;
    static Function<void, gl::GLenum, const gl::GLint *> SpriteParameterivSGIX;
    static Function<void> StartInstrumentsSGIX;
    static Function<void, gl::GLuint, gl::GLenum> StateCaptureNV;
    static Function<void, gl::GLsizei, gl::GLuint> StencilClearTagEXT;
    static Function<void, gl::GLsizei, gl::GLenum, const void *, gl::GLuint, gl::GLenum, gl::GLuint, gl::GLenum, const gl::GLfloat *> StencilFillPathInstancedNV;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLuint> StencilFillPathNV;
    static Function<void, gl::GLenum, gl::GLint, gl::GLuint> StencilFunc;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint, gl::GLuint> StencilFuncSeparate;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint, gl::GLuint> StencilFuncSeparateATI;
    static Function<void, gl::GLuint> StencilMask;
    static Function<void, gl::GLenum, gl::GLuint> StencilMaskSeparate;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum> StencilOp;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLenum> StencilOpSeparate;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLenum> StencilOpSeparateATI;
    static Function<void, gl::GLenum, gl::GLuint> StencilOpValueAMD;
    static Function<void, gl::GLsizei, gl::GLenum, const void *, gl::GLuint, gl::GLint, gl::GLuint, gl::GLenum, const gl::GLfloat *> StencilStrokePathInstancedNV;
    static Function<void, gl::GLuint, gl::GLint, gl::GLuint> StencilStrokePathNV;
    static Function<void, gl::GLsizei, gl::GLenum, const void *, gl::GLuint, gl::GLenum, gl::GLuint, gl::GLenum, gl::GLenum, const gl::GLfloat *> StencilThenCoverFillPathInstancedNV;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLuint, gl::GLenum> StencilThenCoverFillPathNV;
    static Function<void, gl::GLsizei, gl::GLenum, const void *, gl::GLuint, gl::GLint, gl::GLuint, gl::GLenum, gl::GLenum, const gl::GLfloat *> StencilThenCoverStrokePathInstancedNV;
    static Function<void, gl::GLuint, gl::GLint, gl::GLuint, gl::GLenum> StencilThenCoverStrokePathNV;
    static Function<void, gl::GLint> StopInstrumentsSGIX;
    static Function<void, gl::GLsizei, const void *> StringMarkerGREMEDY;
    static Function<void, gl::GLuint, gl::GLuint> SubpixelPrecisionBiasNV;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLenum> SwizzleEXT;
    static Function<void, gl::GLuint> SyncTextureINTEL;
    static Function<void> TagSampleBufferSGIX;
    static Function<void, gl::GLbyte, gl::GLbyte, gl::GLbyte> Tangent3bEXT;
    static Function<void, const gl::GLbyte *> Tangent3bvEXT;
    static Function<void, gl::GLdouble, gl::GLdouble, gl::GLdouble> Tangent3dEXT;
    static Function<void, const gl::GLdouble *> Tangent3dvEXT;
    static Function<void, gl::GLfloat, gl::GLfloat, gl::GLfloat> Tangent3fEXT;
    static Function<void, const gl::GLfloat *> Tangent3fvEXT;
    static Function<void, gl::GLint, gl::GLint, gl::GLint> Tangent3iEXT;
    static Function<void, const gl::GLint *> Tangent3ivEXT;
    static Function<void, gl::GLshort, gl::GLshort, gl::GLshort> Tangent3sEXT;
    static Function<void, const gl::GLshort *> Tangent3svEXT;
    static Function<void, gl::GLenum, gl::GLsizei, const void *> TangentPointerEXT;
    static Function<void, gl::GLuint> TbufferMask3DFX;
    static Function<void, gl::GLfloat> TessellationFactorAMD;
    static Function<void, gl::GLenum> TessellationModeAMD;
    static Function<gl::GLboolean, gl::GLuint> TestFenceAPPLE;
    static Function<gl::GLboolean, gl::GLuint> TestFenceNV;
    static Function<gl::GLboolean, gl::GLenum, gl::GLuint> TestObjectAPPLE;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLuint> TexBuffer;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLuint> TexBufferARB;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLuint> TexBufferEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLuint, gl::GLintptr, gl::GLsizeiptr> TexBufferRange;
    static Function<void, gl::GLenum, const gl::GLfloat *> TexBumpParameterfvATI;
    static Function<void, gl::GLenum, const gl::GLint *> TexBumpParameterivATI;
    static Function<void, gl::GLbyte> TexCoord1bOES;
    static Function<void, const gl::GLbyte *> TexCoord1bvOES;
    static Function<void, gl::GLdouble> TexCoord1d;
    static Function<void, const gl::GLdouble *> TexCoord1dv;
    static Function<void, gl::GLfloat> TexCoord1f;
    static Function<void, const gl::GLfloat *> TexCoord1fv;
    static Function<void, gl::GLhalfNV> TexCoord1hNV;
    static Function<void, const gl::GLhalfNV *> TexCoord1hvNV;
    static Function<void, gl::GLint> TexCoord1i;
    static Function<void, const gl::GLint *> TexCoord1iv;
    static Function<void, gl::GLshort> TexCoord1s;
    static Function<void, const gl::GLshort *> TexCoord1sv;
    static Function<void, gl::GLfixed> TexCoord1xOES;
    static Function<void, const gl::GLfixed *> TexCoord1xvOES;
    static Function<void, gl::GLbyte, gl::GLbyte> TexCoord2bOES;
    static Function<void, const gl::GLbyte *> TexCoord2bvOES;
    static Function<void, gl::GLdouble, gl::GLdouble> TexCoord2d;
    static Function<void, const gl::GLdouble *> TexCoord2dv;
    static Function<void, gl::GLfloat, gl::GLfloat> TexCoord2f;
    static Function<void, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat> TexCoord2fColor3fVertex3fSUN;
    static Function<void, const gl::GLfloat *, const gl::GLfloat *, const gl::GLfloat *> TexCoord2fColor3fVertex3fvSUN;
    static Function<void, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat> TexCoord2fColor4fNormal3fVertex3fSUN;
    static Function<void, const gl::GLfloat *, const gl::GLfloat *, const gl::GLfloat *, const gl::GLfloat *> TexCoord2fColor4fNormal3fVertex3fvSUN;
    static Function<void, gl::GLfloat, gl::GLfloat, gl::GLubyte, gl::GLubyte, gl::GLubyte, gl::GLubyte, gl::GLfloat, gl::GLfloat, gl::GLfloat> TexCoord2fColor4ubVertex3fSUN;
    static Function<void, const gl::GLfloat *, const gl::GLubyte *, const gl::GLfloat *> TexCoord2fColor4ubVertex3fvSUN;
    static Function<void, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat> TexCoord2fNormal3fVertex3fSUN;
    static Function<void, const gl::GLfloat *, const gl::GLfloat *, const gl::GLfloat *> TexCoord2fNormal3fVertex3fvSUN;
    static Function<void, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat> TexCoord2fVertex3fSUN;
    static Function<void, const gl::GLfloat *, const gl::GLfloat *> TexCoord2fVertex3fvSUN;
    static Function<void, const gl::GLfloat *> TexCoord2fv;
    static Function<void, gl::GLhalfNV, gl::GLhalfNV> TexCoord2hNV;
    static Function<void, const gl::GLhalfNV *> TexCoord2hvNV;
    static Function<void, gl::GLint, gl::GLint> TexCoord2i;
    static Function<void, const gl::GLint *> TexCoord2iv;
    static Function<void, gl::GLshort, gl::GLshort> TexCoord2s;
    static Function<void, const gl::GLshort *> TexCoord2sv;
    static Function<void, gl::GLfixed, gl::GLfixed> TexCoord2xOES;
    static Function<void, const gl::GLfixed *> TexCoord2xvOES;
    static Function<void, gl::GLbyte, gl::GLbyte, gl::GLbyte> TexCoord3bOES;
    static Function<void, const gl::GLbyte *> TexCoord3bvOES;
    static Function<void, gl::GLdouble, gl::GLdouble, gl::GLdouble> TexCoord3d;
    static Function<void, const gl::GLdouble *> TexCoord3dv;
    static Function<void, gl::GLfloat, gl::GLfloat, gl::GLfloat> TexCoord3f;
    static Function<void, const gl::GLfloat *> TexCoord3fv;
    static Function<void, gl::GLhalfNV, gl::GLhalfNV, gl::GLhalfNV> TexCoord3hNV;
    static Function<void, const gl::GLhalfNV *> TexCoord3hvNV;
    static Function<void, gl::GLint, gl::GLint, gl::GLint> TexCoord3i;
    static Function<void, const gl::GLint *> TexCoord3iv;
    static Function<void, gl::GLshort, gl::GLshort, gl::GLshort> TexCoord3s;
    static Function<void, const gl::GLshort *> TexCoord3sv;
    static Function<void, gl::GLfixed, gl::GLfixed, gl::GLfixed> TexCoord3xOES;
    static Function<void, const gl::GLfixed *> TexCoord3xvOES;
    static Function<void, gl::GLbyte, gl::GLbyte, gl::GLbyte, gl::GLbyte> TexCoord4bOES;
    static Function<void, const gl::GLbyte *> TexCoord4bvOES;
    static Function<void, gl::GLdouble, gl::GLdouble, gl::GLdouble, gl::GLdouble> TexCoord4d;
    static Function<void, const gl::GLdouble *> TexCoord4dv;
    static Function<void, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat> TexCoord4f;
    static Function<void, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat> TexCoord4fColor4fNormal3fVertex4fSUN;
    static Function<void, const gl::GLfloat *, const gl::GLfloat *, const gl::GLfloat *, const gl::GLfloat *> TexCoord4fColor4fNormal3fVertex4fvSUN;
    static Function<void, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat> TexCoord4fVertex4fSUN;
    static Function<void, const gl::GLfloat *, const gl::GLfloat *> TexCoord4fVertex4fvSUN;
    static Function<void, const gl::GLfloat *> TexCoord4fv;
    static Function<void, gl::GLhalfNV, gl::GLhalfNV, gl::GLhalfNV, gl::GLhalfNV> TexCoord4hNV;
    static Function<void, const gl::GLhalfNV *> TexCoord4hvNV;
    static Function<void, gl::GLint, gl::GLint, gl::GLint, gl::GLint> TexCoord4i;
    static Function<void, const gl::GLint *> TexCoord4iv;
    static Function<void, gl::GLshort, gl::GLshort, gl::GLshort, gl::GLshort> TexCoord4s;
    static Function<void, const gl::GLshort *> TexCoord4sv;
    static Function<void, gl::GLfixed, gl::GLfixed, gl::GLfixed, gl::GLfixed> TexCoord4xOES;
    static Function<void, const gl::GLfixed *> TexCoord4xvOES;
    static Function<void, gl::GLint, gl::GLenum, gl::GLsizei> TexCoordFormatNV;
    static Function<void, gl::GLenum, gl::GLuint> TexCoordP1ui;
    static Function<void, gl::GLenum, const gl::GLuint *> TexCoordP1uiv;
    static Function<void, gl::GLenum, gl::GLuint> TexCoordP2ui;
    static Function<void, gl::GLenum, const gl::GLuint *> TexCoordP2uiv;
    static Function<void, gl::GLenum, gl::GLuint> TexCoordP3ui;
    static Function<void, gl::GLenum, const gl::GLuint *> TexCoordP3uiv;
    static Function<void, gl::GLenum, gl::GLuint> TexCoordP4ui;
    static Function<void, gl::GLenum, const gl::GLuint *> TexCoordP4uiv;
    static Function<void, gl::GLint, gl::GLenum, gl::GLsizei, const void *> TexCoordPointer;
    static Function<void, gl::GLint, gl::GLenum, gl::GLsizei, gl::GLsizei, const void *> TexCoordPointerEXT;
    static Function<void, gl::GLint, gl::GLenum, gl::GLint, const void **, gl::GLint> TexCoordPointerListIBM;
    static Function<void, gl::GLint, gl::GLenum, const void **> TexCoordPointervINTEL;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfloat> TexEnvf;
    static Function<void, gl::GLenum, gl::GLenum, const gl::GLfloat *> TexEnvfv;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint> TexEnvi;
    static Function<void, gl::GLenum, gl::GLenum, const gl::GLint *> TexEnviv;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfixed> TexEnvxOES;
    static Function<void, gl::GLenum, gl::GLenum, const gl::GLfixed *> TexEnvxvOES;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLsizei, const gl::GLfloat *> TexFilterFuncSGIS;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLdouble> TexGend;
    static Function<void, gl::GLenum, gl::GLenum, const gl::GLdouble *> TexGendv;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfloat> TexGenf;
    static Function<void, gl::GLenum, gl::GLenum, const gl::GLfloat *> TexGenfv;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint> TexGeni;
    static Function<void, gl::GLenum, gl::GLenum, const gl::GLint *> TexGeniv;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfixed> TexGenxOES;
    static Function<void, gl::GLenum, gl::GLenum, const gl::GLfixed *> TexGenxvOES;
    static Function<void, gl::GLenum, gl::GLint, gl::GLint, gl::GLsizei, gl::GLint, gl::GLenum, gl::GLenum, const void *> TexImage1D;
    static Function<void, gl::GLenum, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLint, gl::GLenum, gl::GLenum, const void *> TexImage2D;
    static Function<void, gl::GLenum, gl::GLsizei, gl::GLenum, gl::GLsizei, gl::GLsizei, gl::GLboolean> TexImage2DMultisample;
    static Function<void, gl::GLenum, gl::GLsizei, gl::GLsizei, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLboolean> TexImage2DMultisampleCoverageNV;
    static Function<void, gl::GLenum, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLsizei, gl::GLint, gl::GLenum, gl::GLenum, const void *> TexImage3D;
    static Function<void, gl::GLenum, gl::GLint, gl::GLenum, gl::GLsizei, gl::GLsizei, gl::GLsizei, gl::GLint, gl::GLenum, gl::GLenum, const void *> TexImage3DEXT;
    static Function<void, gl::GLenum, gl::GLsizei, gl::GLenum, gl::GLsizei, gl::GLsizei, gl::GLsizei, gl::GLboolean> TexImage3DMultisample;
    static Function<void, gl::GLenum, gl::GLsizei, gl::GLsizei, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLsizei, gl::GLboolean> TexImage3DMultisampleCoverageNV;
    static Function<void, gl::GLenum, gl::GLint, gl::GLenum, gl::GLsizei, gl::GLsizei, gl::GLsizei, gl::GLsizei, gl::GLint, gl::GLenum, gl::GLenum, const void *> TexImage4DSGIS;
    static Function<void, gl::GLenum, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLsizei, gl::GLboolean> TexPageCommitmentARB;
    static Function<void, gl::GLenum, gl::GLenum, const gl::GLint *> TexParameterIiv;
    static Function<void, gl::GLenum, gl::GLenum, const gl::GLint *> TexParameterIivEXT;
    static Function<void, gl::GLenum, gl::GLenum, const gl::GLuint *> TexParameterIuiv;
    static Function<void, gl::GLenum, gl::GLenum, const gl::GLuint *> TexParameterIuivEXT;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfloat> TexParameterf;
    static Function<void, gl::GLenum, gl::GLenum, const gl::GLfloat *> TexParameterfv;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLint> TexParameteri;
    static Function<void, gl::GLenum, gl::GLenum, const gl::GLint *> TexParameteriv;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLfixed> TexParameterxOES;
    static Function<void, gl::GLenum, gl::GLenum, const gl::GLfixed *> TexParameterxvOES;
    static Function<void, gl::GLenum, gl::GLuint> TexRenderbufferNV;
    static Function<void, gl::GLenum, gl::GLsizei, gl::GLenum, gl::GLsizei> TexStorage1D;
    static Function<void, gl::GLenum, gl::GLsizei, gl::GLenum, gl::GLsizei, gl::GLsizei> TexStorage2D;
    static Function<void, gl::GLenum, gl::GLsizei, gl::GLenum, gl::GLsizei, gl::GLsizei, gl::GLboolean> TexStorage2DMultisample;
    static Function<void, gl::GLenum, gl::GLsizei, gl::GLenum, gl::GLsizei, gl::GLsizei, gl::GLsizei> TexStorage3D;
    static Function<void, gl::GLenum, gl::GLsizei, gl::GLenum, gl::GLsizei, gl::GLsizei, gl::GLsizei, gl::GLboolean> TexStorage3DMultisample;
    static Function<void, gl::GLenum, gl::GLenum, gl::GLsizei, gl::GLsizei, gl::GLsizei, gl::GLsizei, gl::TextureStorageMaskAMD> TexStorageSparseAMD;
    static Function<void, gl::GLenum, gl::GLint, gl::GLint, gl::GLsizei, gl::GLenum, gl::GLenum, const void *> TexSubImage1D;
    static Function<void, gl::GLenum, gl::GLint, gl::GLint, gl::GLsizei, gl::GLenum, gl::GLenum, const void *> TexSubImage1DEXT;
    static Function<void, gl::GLenum, gl::GLint, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLenum, gl::GLenum, const void *> TexSubImage2D;
    static Function<void, gl::GLenum, gl::GLint, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLenum, gl::GLenum, const void *> TexSubImage2DEXT;
    static Function<void, gl::GLenum, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLsizei, gl::GLenum, gl::GLenum, const void *> TexSubImage3D;
    static Function<void, gl::GLenum, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLsizei, gl::GLenum, gl::GLenum, const void *> TexSubImage3DEXT;
    static Function<void, gl::GLenum, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLsizei, gl::GLsizei, gl::GLenum, gl::GLenum, const void *> TexSubImage4DSGIS;
    static Function<void> TextureBarrier;
    static Function<void> TextureBarrierNV;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLuint> TextureBuffer;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLenum, gl::GLuint> TextureBufferEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLuint, gl::GLintptr, gl::GLsizeiptr> TextureBufferRange;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLenum, gl::GLuint, gl::GLintptr, gl::GLsizeiptr> TextureBufferRangeEXT;
    static Function<void, gl::GLboolean, gl::GLboolean, gl::GLboolean, gl::GLboolean> TextureColorMaskSGIS;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint, gl::GLint, gl::GLsizei, gl::GLint, gl::GLenum, gl::GLenum, const void *> TextureImage1DEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLint, gl::GLenum, gl::GLenum, const void *> TextureImage2DEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLsizei, gl::GLsizei, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLboolean> TextureImage2DMultisampleCoverageNV;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLsizei, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLboolean> TextureImage2DMultisampleNV;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLsizei, gl::GLint, gl::GLenum, gl::GLenum, const void *> TextureImage3DEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLsizei, gl::GLsizei, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLsizei, gl::GLboolean> TextureImage3DMultisampleCoverageNV;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLsizei, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLsizei, gl::GLboolean> TextureImage3DMultisampleNV;
    static Function<void, gl::GLenum> TextureLightEXT;
    static Function<void, gl::GLenum, gl::GLenum> TextureMaterialEXT;
    static Function<void, gl::GLenum> TextureNormalEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLsizei, gl::GLboolean> TexturePageCommitmentEXT;
    static Function<void, gl::GLuint, gl::GLenum, const gl::GLint *> TextureParameterIiv;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLenum, const gl::GLint *> TextureParameterIivEXT;
    static Function<void, gl::GLuint, gl::GLenum, const gl::GLuint *> TextureParameterIuiv;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLenum, const gl::GLuint *> TextureParameterIuivEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLfloat> TextureParameterf;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLenum, gl::GLfloat> TextureParameterfEXT;
    static Function<void, gl::GLuint, gl::GLenum, const gl::GLfloat *> TextureParameterfv;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLenum, const gl::GLfloat *> TextureParameterfvEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint> TextureParameteri;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLenum, gl::GLint> TextureParameteriEXT;
    static Function<void, gl::GLuint, gl::GLenum, const gl::GLint *> TextureParameteriv;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLenum, const gl::GLint *> TextureParameterivEXT;
    static Function<void, gl::GLenum, gl::GLsizei, const void *> TextureRangeAPPLE;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLuint> TextureRenderbufferEXT;
    static Function<void, gl::GLuint, gl::GLsizei, gl::GLenum, gl::GLsizei> TextureStorage1D;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLsizei, gl::GLenum, gl::GLsizei> TextureStorage1DEXT;
    static Function<void, gl::GLuint, gl::GLsizei, gl::GLenum, gl::GLsizei, gl::GLsizei> TextureStorage2D;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLsizei, gl::GLenum, gl::GLsizei, gl::GLsizei> TextureStorage2DEXT;
    static Function<void, gl::GLuint, gl::GLsizei, gl::GLenum, gl::GLsizei, gl::GLsizei, gl::GLboolean> TextureStorage2DMultisample;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLsizei, gl::GLenum, gl::GLsizei, gl::GLsizei, gl::GLboolean> TextureStorage2DMultisampleEXT;
    static Function<void, gl::GLuint, gl::GLsizei, gl::GLenum, gl::GLsizei, gl::GLsizei, gl::GLsizei> TextureStorage3D;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLsizei, gl::GLenum, gl::GLsizei, gl::GLsizei, gl::GLsizei> TextureStorage3DEXT;
    static Function<void, gl::GLuint, gl::GLsizei, gl::GLenum, gl::GLsizei, gl::GLsizei, gl::GLsizei, gl::GLboolean> TextureStorage3DMultisample;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLsizei, gl::GLenum, gl::GLsizei, gl::GLsizei, gl::GLsizei, gl::GLboolean> TextureStorage3DMultisampleEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLenum, gl::GLsizei, gl::GLsizei, gl::GLsizei, gl::GLsizei, gl::TextureStorageMaskAMD> TextureStorageSparseAMD;
    static Function<void, gl::GLuint, gl::GLint, gl::GLint, gl::GLsizei, gl::GLenum, gl::GLenum, const void *> TextureSubImage1D;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint, gl::GLint, gl::GLsizei, gl::GLenum, gl::GLenum, const void *> TextureSubImage1DEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLenum, gl::GLenum, const void *> TextureSubImage2D;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLenum, gl::GLenum, const void *> TextureSubImage2DEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLsizei, gl::GLenum, gl::GLenum, const void *> TextureSubImage3D;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei, gl::GLsizei, gl::GLenum, gl::GLenum, const void *> TextureSubImage3DEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLuint, gl::GLenum, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint> TextureView;
    static Function<void, gl::GLenum, gl::GLuint, gl::GLenum, gl::GLenum> TrackMatrixNV;
    static Function<void, gl::GLsizei, const gl::GLint *, gl::GLenum> TransformFeedbackAttribsNV;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLuint> TransformFeedbackBufferBase;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLintptr, gl::GLsizeiptr> TransformFeedbackBufferRange;
    static Function<void, gl::GLsizei, const gl::GLint *, gl::GLsizei, const gl::GLint *, gl::GLenum> TransformFeedbackStreamAttribsNV;
    static Function<void, gl::GLuint, gl::GLsizei, const gl::GLchar *const*, gl::GLenum> TransformFeedbackVaryings;
    static Function<void, gl::GLuint, gl::GLsizei, const gl::GLchar *const*, gl::GLenum> TransformFeedbackVaryingsEXT;
    static Function<void, gl::GLuint, gl::GLsizei, const gl::GLint *, gl::GLenum> TransformFeedbackVaryingsNV;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLenum, const gl::GLfloat *> TransformPathNV;
    static Function<void, gl::GLdouble, gl::GLdouble, gl::GLdouble> Translated;
    static Function<void, gl::GLfloat, gl::GLfloat, gl::GLfloat> Translatef;
    static Function<void, gl::GLfixed, gl::GLfixed, gl::GLfixed> TranslatexOES;
    static Function<void, gl::GLint, gl::GLdouble> Uniform1d;
    static Function<void, gl::GLint, gl::GLsizei, const gl::GLdouble *> Uniform1dv;
    static Function<void, gl::GLint, gl::GLfloat> Uniform1f;
    static Function<void, gl::GLint, gl::GLfloat> Uniform1fARB;
    static Function<void, gl::GLint, gl::GLsizei, const gl::GLfloat *> Uniform1fv;
    static Function<void, gl::GLint, gl::GLsizei, const gl::GLfloat *> Uniform1fvARB;
    static Function<void, gl::GLint, gl::GLint> Uniform1i;
    static Function<void, gl::GLint, gl::GLint64> Uniform1i64ARB;
    static Function<void, gl::GLint, gl::GLint64EXT> Uniform1i64NV;
    static Function<void, gl::GLint, gl::GLsizei, const gl::GLint64 *> Uniform1i64vARB;
    static Function<void, gl::GLint, gl::GLsizei, const gl::GLint64EXT *> Uniform1i64vNV;
    static Function<void, gl::GLint, gl::GLint> Uniform1iARB;
    static Function<void, gl::GLint, gl::GLsizei, const gl::GLint *> Uniform1iv;
    static Function<void, gl::GLint, gl::GLsizei, const gl::GLint *> Uniform1ivARB;
    static Function<void, gl::GLint, gl::GLuint> Uniform1ui;
    static Function<void, gl::GLint, gl::GLuint64> Uniform1ui64ARB;
    static Function<void, gl::GLint, gl::GLuint64EXT> Uniform1ui64NV;
    static Function<void, gl::GLint, gl::GLsizei, const gl::GLuint64 *> Uniform1ui64vARB;
    static Function<void, gl::GLint, gl::GLsizei, const gl::GLuint64EXT *> Uniform1ui64vNV;
    static Function<void, gl::GLint, gl::GLuint> Uniform1uiEXT;
    static Function<void, gl::GLint, gl::GLsizei, const gl::GLuint *> Uniform1uiv;
    static Function<void, gl::GLint, gl::GLsizei, const gl::GLuint *> Uniform1uivEXT;
    static Function<void, gl::GLint, gl::GLdouble, gl::GLdouble> Uniform2d;
    static Function<void, gl::GLint, gl::GLsizei, const gl::GLdouble *> Uniform2dv;
    static Function<void, gl::GLint, gl::GLfloat, gl::GLfloat> Uniform2f;
    static Function<void, gl::GLint, gl::GLfloat, gl::GLfloat> Uniform2fARB;
    static Function<void, gl::GLint, gl::GLsizei, const gl::GLfloat *> Uniform2fv;
    static Function<void, gl::GLint, gl::GLsizei, const gl::GLfloat *> Uniform2fvARB;
    static Function<void, gl::GLint, gl::GLint, gl::GLint> Uniform2i;
    static Function<void, gl::GLint, gl::GLint64, gl::GLint64> Uniform2i64ARB;
    static Function<void, gl::GLint, gl::GLint64EXT, gl::GLint64EXT> Uniform2i64NV;
    static Function<void, gl::GLint, gl::GLsizei, const gl::GLint64 *> Uniform2i64vARB;
    static Function<void, gl::GLint, gl::GLsizei, const gl::GLint64EXT *> Uniform2i64vNV;
    static Function<void, gl::GLint, gl::GLint, gl::GLint> Uniform2iARB;
    static Function<void, gl::GLint, gl::GLsizei, const gl::GLint *> Uniform2iv;
    static Function<void, gl::GLint, gl::GLsizei, const gl::GLint *> Uniform2ivARB;
    static Function<void, gl::GLint, gl::GLuint, gl::GLuint> Uniform2ui;
    static Function<void, gl::GLint, gl::GLuint64, gl::GLuint64> Uniform2ui64ARB;
    static Function<void, gl::GLint, gl::GLuint64EXT, gl::GLuint64EXT> Uniform2ui64NV;
    static Function<void, gl::GLint, gl::GLsizei, const gl::GLuint64 *> Uniform2ui64vARB;
    static Function<void, gl::GLint, gl::GLsizei, const gl::GLuint64EXT *> Uniform2ui64vNV;
    static Function<void, gl::GLint, gl::GLuint, gl::GLuint> Uniform2uiEXT;
    static Function<void, gl::GLint, gl::GLsizei, const gl::GLuint *> Uniform2uiv;
    static Function<void, gl::GLint, gl::GLsizei, const gl::GLuint *> Uniform2uivEXT;
    static Function<void, gl::GLint, gl::GLdouble, gl::GLdouble, gl::GLdouble> Uniform3d;
    static Function<void, gl::GLint, gl::GLsizei, const gl::GLdouble *> Uniform3dv;
    static Function<void, gl::GLint, gl::GLfloat, gl::GLfloat, gl::GLfloat> Uniform3f;
    static Function<void, gl::GLint, gl::GLfloat, gl::GLfloat, gl::GLfloat> Uniform3fARB;
    static Function<void, gl::GLint, gl::GLsizei, const gl::GLfloat *> Uniform3fv;
    static Function<void, gl::GLint, gl::GLsizei, const gl::GLfloat *> Uniform3fvARB;
    static Function<void, gl::GLint, gl::GLint, gl::GLint, gl::GLint> Uniform3i;
    static Function<void, gl::GLint, gl::GLint64, gl::GLint64, gl::GLint64> Uniform3i64ARB;
    static Function<void, gl::GLint, gl::GLint64EXT, gl::GLint64EXT, gl::GLint64EXT> Uniform3i64NV;
    static Function<void, gl::GLint, gl::GLsizei, const gl::GLint64 *> Uniform3i64vARB;
    static Function<void, gl::GLint, gl::GLsizei, const gl::GLint64EXT *> Uniform3i64vNV;
    static Function<void, gl::GLint, gl::GLint, gl::GLint, gl::GLint> Uniform3iARB;
    static Function<void, gl::GLint, gl::GLsizei, const gl::GLint *> Uniform3iv;
    static Function<void, gl::GLint, gl::GLsizei, const gl::GLint *> Uniform3ivARB;
    static Function<void, gl::GLint, gl::GLuint, gl::GLuint, gl::GLuint> Uniform3ui;
    static Function<void, gl::GLint, gl::GLuint64, gl::GLuint64, gl::GLuint64> Uniform3ui64ARB;
    static Function<void, gl::GLint, gl::GLuint64EXT, gl::GLuint64EXT, gl::GLuint64EXT> Uniform3ui64NV;
    static Function<void, gl::GLint, gl::GLsizei, const gl::GLuint64 *> Uniform3ui64vARB;
    static Function<void, gl::GLint, gl::GLsizei, const gl::GLuint64EXT *> Uniform3ui64vNV;
    static Function<void, gl::GLint, gl::GLuint, gl::GLuint, gl::GLuint> Uniform3uiEXT;
    static Function<void, gl::GLint, gl::GLsizei, const gl::GLuint *> Uniform3uiv;
    static Function<void, gl::GLint, gl::GLsizei, const gl::GLuint *> Uniform3uivEXT;
    static Function<void, gl::GLint, gl::GLdouble, gl::GLdouble, gl::GLdouble, gl::GLdouble> Uniform4d;
    static Function<void, gl::GLint, gl::GLsizei, const gl::GLdouble *> Uniform4dv;
    static Function<void, gl::GLint, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat> Uniform4f;
    static Function<void, gl::GLint, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat> Uniform4fARB;
    static Function<void, gl::GLint, gl::GLsizei, const gl::GLfloat *> Uniform4fv;
    static Function<void, gl::GLint, gl::GLsizei, const gl::GLfloat *> Uniform4fvARB;
    static Function<void, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLint> Uniform4i;
    static Function<void, gl::GLint, gl::GLint64, gl::GLint64, gl::GLint64, gl::GLint64> Uniform4i64ARB;
    static Function<void, gl::GLint, gl::GLint64EXT, gl::GLint64EXT, gl::GLint64EXT, gl::GLint64EXT> Uniform4i64NV;
    static Function<void, gl::GLint, gl::GLsizei, const gl::GLint64 *> Uniform4i64vARB;
    static Function<void, gl::GLint, gl::GLsizei, const gl::GLint64EXT *> Uniform4i64vNV;
    static Function<void, gl::GLint, gl::GLint, gl::GLint, gl::GLint, gl::GLint> Uniform4iARB;
    static Function<void, gl::GLint, gl::GLsizei, const gl::GLint *> Uniform4iv;
    static Function<void, gl::GLint, gl::GLsizei, const gl::GLint *> Uniform4ivARB;
    static Function<void, gl::GLint, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint> Uniform4ui;
    static Function<void, gl::GLint, gl::GLuint64, gl::GLuint64, gl::GLuint64, gl::GLuint64> Uniform4ui64ARB;
    static Function<void, gl::GLint, gl::GLuint64EXT, gl::GLuint64EXT, gl::GLuint64EXT, gl::GLuint64EXT> Uniform4ui64NV;
    static Function<void, gl::GLint, gl::GLsizei, const gl::GLuint64 *> Uniform4ui64vARB;
    static Function<void, gl::GLint, gl::GLsizei, const gl::GLuint64EXT *> Uniform4ui64vNV;
    static Function<void, gl::GLint, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint> Uniform4uiEXT;
    static Function<void, gl::GLint, gl::GLsizei, const gl::GLuint *> Uniform4uiv;
    static Function<void, gl::GLint, gl::GLsizei, const gl::GLuint *> Uniform4uivEXT;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLuint> UniformBlockBinding;
    static Function<void, gl::GLuint, gl::GLint, gl::GLuint> UniformBufferEXT;
    static Function<void, gl::GLint, gl::GLuint64> UniformHandleui64ARB;
    static Function<void, gl::GLint, gl::GLuint64> UniformHandleui64NV;
    static Function<void, gl::GLint, gl::GLsizei, const gl::GLuint64 *> UniformHandleui64vARB;
    static Function<void, gl::GLint, gl::GLsizei, const gl::GLuint64 *> UniformHandleui64vNV;
    static Function<void, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLdouble *> UniformMatrix2dv;
    static Function<void, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLfloat *> UniformMatrix2fv;
    static Function<void, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLfloat *> UniformMatrix2fvARB;
    static Function<void, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLdouble *> UniformMatrix2x3dv;
    static Function<void, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLfloat *> UniformMatrix2x3fv;
    static Function<void, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLdouble *> UniformMatrix2x4dv;
    static Function<void, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLfloat *> UniformMatrix2x4fv;
    static Function<void, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLdouble *> UniformMatrix3dv;
    static Function<void, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLfloat *> UniformMatrix3fv;
    static Function<void, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLfloat *> UniformMatrix3fvARB;
    static Function<void, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLdouble *> UniformMatrix3x2dv;
    static Function<void, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLfloat *> UniformMatrix3x2fv;
    static Function<void, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLdouble *> UniformMatrix3x4dv;
    static Function<void, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLfloat *> UniformMatrix3x4fv;
    static Function<void, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLdouble *> UniformMatrix4dv;
    static Function<void, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLfloat *> UniformMatrix4fv;
    static Function<void, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLfloat *> UniformMatrix4fvARB;
    static Function<void, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLdouble *> UniformMatrix4x2dv;
    static Function<void, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLfloat *> UniformMatrix4x2fv;
    static Function<void, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLdouble *> UniformMatrix4x3dv;
    static Function<void, gl::GLint, gl::GLsizei, gl::GLboolean, const gl::GLfloat *> UniformMatrix4x3fv;
    static Function<void, gl::GLenum, gl::GLsizei, const gl::GLuint *> UniformSubroutinesuiv;
    static Function<void, gl::GLint, gl::GLuint64EXT> Uniformui64NV;
    static Function<void, gl::GLint, gl::GLsizei, const gl::GLuint64EXT *> Uniformui64vNV;
    static Function<void> UnlockArraysEXT;
    static Function<gl::GLboolean, gl::GLenum> UnmapBuffer;
    static Function<gl::GLboolean, gl::GLenum> UnmapBufferARB;
    static Function<gl::GLboolean, gl::GLuint> UnmapNamedBuffer;
    static Function<gl::GLboolean, gl::GLuint> UnmapNamedBufferEXT;
    static Function<void, gl::GLuint> UnmapObjectBufferATI;
    static Function<void, gl::GLuint, gl::GLint> UnmapTexture2DINTEL;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLsizei, const void *, gl::GLenum> UpdateObjectBufferATI;
    static Function<void, gl::GLuint> UseProgram;
    static Function<void, gl::GLhandleARB> UseProgramObjectARB;
    static Function<void, gl::GLuint, gl::UseProgramStageMask, gl::GLuint> UseProgramStages;
    static Function<void, gl::GLenum, gl::GLuint> UseShaderProgramEXT;
    static Function<void> VDPAUFiniNV;
    static Function<void, gl::GLvdpauSurfaceNV, gl::GLenum, gl::GLsizei, gl::GLsizei *, gl::GLint *> VDPAUGetSurfaceivNV;
    static Function<void, const void *, const void *> VDPAUInitNV;
    static Function<gl::GLboolean, gl::GLvdpauSurfaceNV> VDPAUIsSurfaceNV;
    static Function<void, gl::GLsizei, const gl::GLvdpauSurfaceNV *> VDPAUMapSurfacesNV;
    static Function<gl::GLvdpauSurfaceNV, const void *, gl::GLenum, gl::GLsizei, const gl::GLuint *> VDPAURegisterOutputSurfaceNV;
    static Function<gl::GLvdpauSurfaceNV, const void *, gl::GLenum, gl::GLsizei, const gl::GLuint *> VDPAURegisterVideoSurfaceNV;
    static Function<void, gl::GLvdpauSurfaceNV, gl::GLenum> VDPAUSurfaceAccessNV;
    static Function<void, gl::GLsizei, const gl::GLvdpauSurfaceNV *> VDPAUUnmapSurfacesNV;
    static Function<void, gl::GLvdpauSurfaceNV> VDPAUUnregisterSurfaceNV;
    static Function<void, gl::GLuint> ValidateProgram;
    static Function<void, gl::GLhandleARB> ValidateProgramARB;
    static Function<void, gl::GLuint> ValidateProgramPipeline;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLsizei, gl::GLuint, gl::GLuint> VariantArrayObjectATI;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLuint, const void *> VariantPointerEXT;
    static Function<void, gl::GLuint, const gl::GLbyte *> VariantbvEXT;
    static Function<void, gl::GLuint, const gl::GLdouble *> VariantdvEXT;
    static Function<void, gl::GLuint, const gl::GLfloat *> VariantfvEXT;
    static Function<void, gl::GLuint, const gl::GLint *> VariantivEXT;
    static Function<void, gl::GLuint, const gl::GLshort *> VariantsvEXT;
    static Function<void, gl::GLuint, const gl::GLubyte *> VariantubvEXT;
    static Function<void, gl::GLuint, const gl::GLuint *> VariantuivEXT;
    static Function<void, gl::GLuint, const gl::GLushort *> VariantusvEXT;
    static Function<void, gl::GLbyte, gl::GLbyte> Vertex2bOES;
    static Function<void, const gl::GLbyte *> Vertex2bvOES;
    static Function<void, gl::GLdouble, gl::GLdouble> Vertex2d;
    static Function<void, const gl::GLdouble *> Vertex2dv;
    static Function<void, gl::GLfloat, gl::GLfloat> Vertex2f;
    static Function<void, const gl::GLfloat *> Vertex2fv;
    static Function<void, gl::GLhalfNV, gl::GLhalfNV> Vertex2hNV;
    static Function<void, const gl::GLhalfNV *> Vertex2hvNV;
    static Function<void, gl::GLint, gl::GLint> Vertex2i;
    static Function<void, const gl::GLint *> Vertex2iv;
    static Function<void, gl::GLshort, gl::GLshort> Vertex2s;
    static Function<void, const gl::GLshort *> Vertex2sv;
    static Function<void, gl::GLfixed> Vertex2xOES;
    static Function<void, const gl::GLfixed *> Vertex2xvOES;
    static Function<void, gl::GLbyte, gl::GLbyte, gl::GLbyte> Vertex3bOES;
    static Function<void, const gl::GLbyte *> Vertex3bvOES;
    static Function<void, gl::GLdouble, gl::GLdouble, gl::GLdouble> Vertex3d;
    static Function<void, const gl::GLdouble *> Vertex3dv;
    static Function<void, gl::GLfloat, gl::GLfloat, gl::GLfloat> Vertex3f;
    static Function<void, const gl::GLfloat *> Vertex3fv;
    static Function<void, gl::GLhalfNV, gl::GLhalfNV, gl::GLhalfNV> Vertex3hNV;
    static Function<void, const gl::GLhalfNV *> Vertex3hvNV;
    static Function<void, gl::GLint, gl::GLint, gl::GLint> Vertex3i;
    static Function<void, const gl::GLint *> Vertex3iv;
    static Function<void, gl::GLshort, gl::GLshort, gl::GLshort> Vertex3s;
    static Function<void, const gl::GLshort *> Vertex3sv;
    static Function<void, gl::GLfixed, gl::GLfixed> Vertex3xOES;
    static Function<void, const gl::GLfixed *> Vertex3xvOES;
    static Function<void, gl::GLbyte, gl::GLbyte, gl::GLbyte, gl::GLbyte> Vertex4bOES;
    static Function<void, const gl::GLbyte *> Vertex4bvOES;
    static Function<void, gl::GLdouble, gl::GLdouble, gl::GLdouble, gl::GLdouble> Vertex4d;
    static Function<void, const gl::GLdouble *> Vertex4dv;
    static Function<void, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat> Vertex4f;
    static Function<void, const gl::GLfloat *> Vertex4fv;
    static Function<void, gl::GLhalfNV, gl::GLhalfNV, gl::GLhalfNV, gl::GLhalfNV> Vertex4hNV;
    static Function<void, const gl::GLhalfNV *> Vertex4hvNV;
    static Function<void, gl::GLint, gl::GLint, gl::GLint, gl::GLint> Vertex4i;
    static Function<void, const gl::GLint *> Vertex4iv;
    static Function<void, gl::GLshort, gl::GLshort, gl::GLshort, gl::GLshort> Vertex4s;
    static Function<void, const gl::GLshort *> Vertex4sv;
    static Function<void, gl::GLfixed, gl::GLfixed, gl::GLfixed> Vertex4xOES;
    static Function<void, const gl::GLfixed *> Vertex4xvOES;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLuint> VertexArrayAttribBinding;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLint, gl::GLenum, gl::GLboolean, gl::GLuint> VertexArrayAttribFormat;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLint, gl::GLenum, gl::GLuint> VertexArrayAttribIFormat;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLint, gl::GLenum, gl::GLuint> VertexArrayAttribLFormat;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLintptr, gl::GLsizei> VertexArrayBindVertexBufferEXT;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLuint> VertexArrayBindingDivisor;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLint, gl::GLenum, gl::GLsizei, gl::GLintptr> VertexArrayColorOffsetEXT;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLsizei, gl::GLintptr> VertexArrayEdgeFlagOffsetEXT;
    static Function<void, gl::GLuint, gl::GLuint> VertexArrayElementBuffer;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLenum, gl::GLsizei, gl::GLintptr> VertexArrayFogCoordOffsetEXT;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLenum, gl::GLsizei, gl::GLintptr> VertexArrayIndexOffsetEXT;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLenum, gl::GLint, gl::GLenum, gl::GLsizei, gl::GLintptr> VertexArrayMultiTexCoordOffsetEXT;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLenum, gl::GLsizei, gl::GLintptr> VertexArrayNormalOffsetEXT;
    static Function<void, gl::GLenum, gl::GLint> VertexArrayParameteriAPPLE;
    static Function<void, gl::GLsizei, void *> VertexArrayRangeAPPLE;
    static Function<void, gl::GLsizei, const void *> VertexArrayRangeNV;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLint, gl::GLenum, gl::GLsizei, gl::GLintptr> VertexArraySecondaryColorOffsetEXT;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLint, gl::GLenum, gl::GLsizei, gl::GLintptr> VertexArrayTexCoordOffsetEXT;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLuint> VertexArrayVertexAttribBindingEXT;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLuint> VertexArrayVertexAttribDivisorEXT;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLint, gl::GLenum, gl::GLboolean, gl::GLuint> VertexArrayVertexAttribFormatEXT;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLint, gl::GLenum, gl::GLuint> VertexArrayVertexAttribIFormatEXT;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLint, gl::GLenum, gl::GLsizei, gl::GLintptr> VertexArrayVertexAttribIOffsetEXT;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLint, gl::GLenum, gl::GLuint> VertexArrayVertexAttribLFormatEXT;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLint, gl::GLenum, gl::GLsizei, gl::GLintptr> VertexArrayVertexAttribLOffsetEXT;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLint, gl::GLenum, gl::GLboolean, gl::GLsizei, gl::GLintptr> VertexArrayVertexAttribOffsetEXT;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLuint> VertexArrayVertexBindingDivisorEXT;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLintptr, gl::GLsizei> VertexArrayVertexBuffer;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLsizei, const gl::GLuint *, const gl::GLintptr *, const gl::GLsizei *> VertexArrayVertexBuffers;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLint, gl::GLenum, gl::GLsizei, gl::GLintptr> VertexArrayVertexOffsetEXT;
    static Function<void, gl::GLuint, gl::GLdouble> VertexAttrib1d;
    static Function<void, gl::GLuint, gl::GLdouble> VertexAttrib1dARB;
    static Function<void, gl::GLuint, gl::GLdouble> VertexAttrib1dNV;
    static Function<void, gl::GLuint, const gl::GLdouble *> VertexAttrib1dv;
    static Function<void, gl::GLuint, const gl::GLdouble *> VertexAttrib1dvARB;
    static Function<void, gl::GLuint, const gl::GLdouble *> VertexAttrib1dvNV;
    static Function<void, gl::GLuint, gl::GLfloat> VertexAttrib1f;
    static Function<void, gl::GLuint, gl::GLfloat> VertexAttrib1fARB;
    static Function<void, gl::GLuint, gl::GLfloat> VertexAttrib1fNV;
    static Function<void, gl::GLuint, const gl::GLfloat *> VertexAttrib1fv;
    static Function<void, gl::GLuint, const gl::GLfloat *> VertexAttrib1fvARB;
    static Function<void, gl::GLuint, const gl::GLfloat *> VertexAttrib1fvNV;
    static Function<void, gl::GLuint, gl::GLhalfNV> VertexAttrib1hNV;
    static Function<void, gl::GLuint, const gl::GLhalfNV *> VertexAttrib1hvNV;
    static Function<void, gl::GLuint, gl::GLshort> VertexAttrib1s;
    static Function<void, gl::GLuint, gl::GLshort> VertexAttrib1sARB;
    static Function<void, gl::GLuint, gl::GLshort> VertexAttrib1sNV;
    static Function<void, gl::GLuint, const gl::GLshort *> VertexAttrib1sv;
    static Function<void, gl::GLuint, const gl::GLshort *> VertexAttrib1svARB;
    static Function<void, gl::GLuint, const gl::GLshort *> VertexAttrib1svNV;
    static Function<void, gl::GLuint, gl::GLdouble, gl::GLdouble> VertexAttrib2d;
    static Function<void, gl::GLuint, gl::GLdouble, gl::GLdouble> VertexAttrib2dARB;
    static Function<void, gl::GLuint, gl::GLdouble, gl::GLdouble> VertexAttrib2dNV;
    static Function<void, gl::GLuint, const gl::GLdouble *> VertexAttrib2dv;
    static Function<void, gl::GLuint, const gl::GLdouble *> VertexAttrib2dvARB;
    static Function<void, gl::GLuint, const gl::GLdouble *> VertexAttrib2dvNV;
    static Function<void, gl::GLuint, gl::GLfloat, gl::GLfloat> VertexAttrib2f;
    static Function<void, gl::GLuint, gl::GLfloat, gl::GLfloat> VertexAttrib2fARB;
    static Function<void, gl::GLuint, gl::GLfloat, gl::GLfloat> VertexAttrib2fNV;
    static Function<void, gl::GLuint, const gl::GLfloat *> VertexAttrib2fv;
    static Function<void, gl::GLuint, const gl::GLfloat *> VertexAttrib2fvARB;
    static Function<void, gl::GLuint, const gl::GLfloat *> VertexAttrib2fvNV;
    static Function<void, gl::GLuint, gl::GLhalfNV, gl::GLhalfNV> VertexAttrib2hNV;
    static Function<void, gl::GLuint, const gl::GLhalfNV *> VertexAttrib2hvNV;
    static Function<void, gl::GLuint, gl::GLshort, gl::GLshort> VertexAttrib2s;
    static Function<void, gl::GLuint, gl::GLshort, gl::GLshort> VertexAttrib2sARB;
    static Function<void, gl::GLuint, gl::GLshort, gl::GLshort> VertexAttrib2sNV;
    static Function<void, gl::GLuint, const gl::GLshort *> VertexAttrib2sv;
    static Function<void, gl::GLuint, const gl::GLshort *> VertexAttrib2svARB;
    static Function<void, gl::GLuint, const gl::GLshort *> VertexAttrib2svNV;
    static Function<void, gl::GLuint, gl::GLdouble, gl::GLdouble, gl::GLdouble> VertexAttrib3d;
    static Function<void, gl::GLuint, gl::GLdouble, gl::GLdouble, gl::GLdouble> VertexAttrib3dARB;
    static Function<void, gl::GLuint, gl::GLdouble, gl::GLdouble, gl::GLdouble> VertexAttrib3dNV;
    static Function<void, gl::GLuint, const gl::GLdouble *> VertexAttrib3dv;
    static Function<void, gl::GLuint, const gl::GLdouble *> VertexAttrib3dvARB;
    static Function<void, gl::GLuint, const gl::GLdouble *> VertexAttrib3dvNV;
    static Function<void, gl::GLuint, gl::GLfloat, gl::GLfloat, gl::GLfloat> VertexAttrib3f;
    static Function<void, gl::GLuint, gl::GLfloat, gl::GLfloat, gl::GLfloat> VertexAttrib3fARB;
    static Function<void, gl::GLuint, gl::GLfloat, gl::GLfloat, gl::GLfloat> VertexAttrib3fNV;
    static Function<void, gl::GLuint, const gl::GLfloat *> VertexAttrib3fv;
    static Function<void, gl::GLuint, const gl::GLfloat *> VertexAttrib3fvARB;
    static Function<void, gl::GLuint, const gl::GLfloat *> VertexAttrib3fvNV;
    static Function<void, gl::GLuint, gl::GLhalfNV, gl::GLhalfNV, gl::GLhalfNV> VertexAttrib3hNV;
    static Function<void, gl::GLuint, const gl::GLhalfNV *> VertexAttrib3hvNV;
    static Function<void, gl::GLuint, gl::GLshort, gl::GLshort, gl::GLshort> VertexAttrib3s;
    static Function<void, gl::GLuint, gl::GLshort, gl::GLshort, gl::GLshort> VertexAttrib3sARB;
    static Function<void, gl::GLuint, gl::GLshort, gl::GLshort, gl::GLshort> VertexAttrib3sNV;
    static Function<void, gl::GLuint, const gl::GLshort *> VertexAttrib3sv;
    static Function<void, gl::GLuint, const gl::GLshort *> VertexAttrib3svARB;
    static Function<void, gl::GLuint, const gl::GLshort *> VertexAttrib3svNV;
    static Function<void, gl::GLuint, const gl::GLbyte *> VertexAttrib4Nbv;
    static Function<void, gl::GLuint, const gl::GLbyte *> VertexAttrib4NbvARB;
    static Function<void, gl::GLuint, const gl::GLint *> VertexAttrib4Niv;
    static Function<void, gl::GLuint, const gl::GLint *> VertexAttrib4NivARB;
    static Function<void, gl::GLuint, const gl::GLshort *> VertexAttrib4Nsv;
    static Function<void, gl::GLuint, const gl::GLshort *> VertexAttrib4NsvARB;
    static Function<void, gl::GLuint, gl::GLubyte, gl::GLubyte, gl::GLubyte, gl::GLubyte> VertexAttrib4Nub;
    static Function<void, gl::GLuint, gl::GLubyte, gl::GLubyte, gl::GLubyte, gl::GLubyte> VertexAttrib4NubARB;
    static Function<void, gl::GLuint, const gl::GLubyte *> VertexAttrib4Nubv;
    static Function<void, gl::GLuint, const gl::GLubyte *> VertexAttrib4NubvARB;
    static Function<void, gl::GLuint, const gl::GLuint *> VertexAttrib4Nuiv;
    static Function<void, gl::GLuint, const gl::GLuint *> VertexAttrib4NuivARB;
    static Function<void, gl::GLuint, const gl::GLushort *> VertexAttrib4Nusv;
    static Function<void, gl::GLuint, const gl::GLushort *> VertexAttrib4NusvARB;
    static Function<void, gl::GLuint, const gl::GLbyte *> VertexAttrib4bv;
    static Function<void, gl::GLuint, const gl::GLbyte *> VertexAttrib4bvARB;
    static Function<void, gl::GLuint, gl::GLdouble, gl::GLdouble, gl::GLdouble, gl::GLdouble> VertexAttrib4d;
    static Function<void, gl::GLuint, gl::GLdouble, gl::GLdouble, gl::GLdouble, gl::GLdouble> VertexAttrib4dARB;
    static Function<void, gl::GLuint, gl::GLdouble, gl::GLdouble, gl::GLdouble, gl::GLdouble> VertexAttrib4dNV;
    static Function<void, gl::GLuint, const gl::GLdouble *> VertexAttrib4dv;
    static Function<void, gl::GLuint, const gl::GLdouble *> VertexAttrib4dvARB;
    static Function<void, gl::GLuint, const gl::GLdouble *> VertexAttrib4dvNV;
    static Function<void, gl::GLuint, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat> VertexAttrib4f;
    static Function<void, gl::GLuint, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat> VertexAttrib4fARB;
    static Function<void, gl::GLuint, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat> VertexAttrib4fNV;
    static Function<void, gl::GLuint, const gl::GLfloat *> VertexAttrib4fv;
    static Function<void, gl::GLuint, const gl::GLfloat *> VertexAttrib4fvARB;
    static Function<void, gl::GLuint, const gl::GLfloat *> VertexAttrib4fvNV;
    static Function<void, gl::GLuint, gl::GLhalfNV, gl::GLhalfNV, gl::GLhalfNV, gl::GLhalfNV> VertexAttrib4hNV;
    static Function<void, gl::GLuint, const gl::GLhalfNV *> VertexAttrib4hvNV;
    static Function<void, gl::GLuint, const gl::GLint *> VertexAttrib4iv;
    static Function<void, gl::GLuint, const gl::GLint *> VertexAttrib4ivARB;
    static Function<void, gl::GLuint, gl::GLshort, gl::GLshort, gl::GLshort, gl::GLshort> VertexAttrib4s;
    static Function<void, gl::GLuint, gl::GLshort, gl::GLshort, gl::GLshort, gl::GLshort> VertexAttrib4sARB;
    static Function<void, gl::GLuint, gl::GLshort, gl::GLshort, gl::GLshort, gl::GLshort> VertexAttrib4sNV;
    static Function<void, gl::GLuint, const gl::GLshort *> VertexAttrib4sv;
    static Function<void, gl::GLuint, const gl::GLshort *> VertexAttrib4svARB;
    static Function<void, gl::GLuint, const gl::GLshort *> VertexAttrib4svNV;
    static Function<void, gl::GLuint, gl::GLubyte, gl::GLubyte, gl::GLubyte, gl::GLubyte> VertexAttrib4ubNV;
    static Function<void, gl::GLuint, const gl::GLubyte *> VertexAttrib4ubv;
    static Function<void, gl::GLuint, const gl::GLubyte *> VertexAttrib4ubvARB;
    static Function<void, gl::GLuint, const gl::GLubyte *> VertexAttrib4ubvNV;
    static Function<void, gl::GLuint, const gl::GLuint *> VertexAttrib4uiv;
    static Function<void, gl::GLuint, const gl::GLuint *> VertexAttrib4uivARB;
    static Function<void, gl::GLuint, const gl::GLushort *> VertexAttrib4usv;
    static Function<void, gl::GLuint, const gl::GLushort *> VertexAttrib4usvARB;
    static Function<void, gl::GLuint, gl::GLint, gl::GLenum, gl::GLboolean, gl::GLsizei, gl::GLuint, gl::GLuint> VertexAttribArrayObjectATI;
    static Function<void, gl::GLuint, gl::GLuint> VertexAttribBinding;
    static Function<void, gl::GLuint, gl::GLuint> VertexAttribDivisor;
    static Function<void, gl::GLuint, gl::GLuint> VertexAttribDivisorARB;
    static Function<void, gl::GLuint, gl::GLint, gl::GLenum, gl::GLboolean, gl::GLuint> VertexAttribFormat;
    static Function<void, gl::GLuint, gl::GLint, gl::GLenum, gl::GLboolean, gl::GLsizei> VertexAttribFormatNV;
    static Function<void, gl::GLuint, gl::GLint> VertexAttribI1i;
    static Function<void, gl::GLuint, gl::GLint> VertexAttribI1iEXT;
    static Function<void, gl::GLuint, const gl::GLint *> VertexAttribI1iv;
    static Function<void, gl::GLuint, const gl::GLint *> VertexAttribI1ivEXT;
    static Function<void, gl::GLuint, gl::GLuint> VertexAttribI1ui;
    static Function<void, gl::GLuint, gl::GLuint> VertexAttribI1uiEXT;
    static Function<void, gl::GLuint, const gl::GLuint *> VertexAttribI1uiv;
    static Function<void, gl::GLuint, const gl::GLuint *> VertexAttribI1uivEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLint> VertexAttribI2i;
    static Function<void, gl::GLuint, gl::GLint, gl::GLint> VertexAttribI2iEXT;
    static Function<void, gl::GLuint, const gl::GLint *> VertexAttribI2iv;
    static Function<void, gl::GLuint, const gl::GLint *> VertexAttribI2ivEXT;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLuint> VertexAttribI2ui;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLuint> VertexAttribI2uiEXT;
    static Function<void, gl::GLuint, const gl::GLuint *> VertexAttribI2uiv;
    static Function<void, gl::GLuint, const gl::GLuint *> VertexAttribI2uivEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLint, gl::GLint> VertexAttribI3i;
    static Function<void, gl::GLuint, gl::GLint, gl::GLint, gl::GLint> VertexAttribI3iEXT;
    static Function<void, gl::GLuint, const gl::GLint *> VertexAttribI3iv;
    static Function<void, gl::GLuint, const gl::GLint *> VertexAttribI3ivEXT;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint> VertexAttribI3ui;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint> VertexAttribI3uiEXT;
    static Function<void, gl::GLuint, const gl::GLuint *> VertexAttribI3uiv;
    static Function<void, gl::GLuint, const gl::GLuint *> VertexAttribI3uivEXT;
    static Function<void, gl::GLuint, const gl::GLbyte *> VertexAttribI4bv;
    static Function<void, gl::GLuint, const gl::GLbyte *> VertexAttribI4bvEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLint, gl::GLint, gl::GLint> VertexAttribI4i;
    static Function<void, gl::GLuint, gl::GLint, gl::GLint, gl::GLint, gl::GLint> VertexAttribI4iEXT;
    static Function<void, gl::GLuint, const gl::GLint *> VertexAttribI4iv;
    static Function<void, gl::GLuint, const gl::GLint *> VertexAttribI4ivEXT;
    static Function<void, gl::GLuint, const gl::GLshort *> VertexAttribI4sv;
    static Function<void, gl::GLuint, const gl::GLshort *> VertexAttribI4svEXT;
    static Function<void, gl::GLuint, const gl::GLubyte *> VertexAttribI4ubv;
    static Function<void, gl::GLuint, const gl::GLubyte *> VertexAttribI4ubvEXT;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint> VertexAttribI4ui;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint, gl::GLuint> VertexAttribI4uiEXT;
    static Function<void, gl::GLuint, const gl::GLuint *> VertexAttribI4uiv;
    static Function<void, gl::GLuint, const gl::GLuint *> VertexAttribI4uivEXT;
    static Function<void, gl::GLuint, const gl::GLushort *> VertexAttribI4usv;
    static Function<void, gl::GLuint, const gl::GLushort *> VertexAttribI4usvEXT;
    static Function<void, gl::GLuint, gl::GLint, gl::GLenum, gl::GLuint> VertexAttribIFormat;
    static Function<void, gl::GLuint, gl::GLint, gl::GLenum, gl::GLsizei> VertexAttribIFormatNV;
    static Function<void, gl::GLuint, gl::GLint, gl::GLenum, gl::GLsizei, const void *> VertexAttribIPointer;
    static Function<void, gl::GLuint, gl::GLint, gl::GLenum, gl::GLsizei, const void *> VertexAttribIPointerEXT;
    static Function<void, gl::GLuint, gl::GLdouble> VertexAttribL1d;
    static Function<void, gl::GLuint, gl::GLdouble> VertexAttribL1dEXT;
    static Function<void, gl::GLuint, const gl::GLdouble *> VertexAttribL1dv;
    static Function<void, gl::GLuint, const gl::GLdouble *> VertexAttribL1dvEXT;
    static Function<void, gl::GLuint, gl::GLint64EXT> VertexAttribL1i64NV;
    static Function<void, gl::GLuint, const gl::GLint64EXT *> VertexAttribL1i64vNV;
    static Function<void, gl::GLuint, gl::GLuint64EXT> VertexAttribL1ui64ARB;
    static Function<void, gl::GLuint, gl::GLuint64EXT> VertexAttribL1ui64NV;
    static Function<void, gl::GLuint, const gl::GLuint64EXT *> VertexAttribL1ui64vARB;
    static Function<void, gl::GLuint, const gl::GLuint64EXT *> VertexAttribL1ui64vNV;
    static Function<void, gl::GLuint, gl::GLdouble, gl::GLdouble> VertexAttribL2d;
    static Function<void, gl::GLuint, gl::GLdouble, gl::GLdouble> VertexAttribL2dEXT;
    static Function<void, gl::GLuint, const gl::GLdouble *> VertexAttribL2dv;
    static Function<void, gl::GLuint, const gl::GLdouble *> VertexAttribL2dvEXT;
    static Function<void, gl::GLuint, gl::GLint64EXT, gl::GLint64EXT> VertexAttribL2i64NV;
    static Function<void, gl::GLuint, const gl::GLint64EXT *> VertexAttribL2i64vNV;
    static Function<void, gl::GLuint, gl::GLuint64EXT, gl::GLuint64EXT> VertexAttribL2ui64NV;
    static Function<void, gl::GLuint, const gl::GLuint64EXT *> VertexAttribL2ui64vNV;
    static Function<void, gl::GLuint, gl::GLdouble, gl::GLdouble, gl::GLdouble> VertexAttribL3d;
    static Function<void, gl::GLuint, gl::GLdouble, gl::GLdouble, gl::GLdouble> VertexAttribL3dEXT;
    static Function<void, gl::GLuint, const gl::GLdouble *> VertexAttribL3dv;
    static Function<void, gl::GLuint, const gl::GLdouble *> VertexAttribL3dvEXT;
    static Function<void, gl::GLuint, gl::GLint64EXT, gl::GLint64EXT, gl::GLint64EXT> VertexAttribL3i64NV;
    static Function<void, gl::GLuint, const gl::GLint64EXT *> VertexAttribL3i64vNV;
    static Function<void, gl::GLuint, gl::GLuint64EXT, gl::GLuint64EXT, gl::GLuint64EXT> VertexAttribL3ui64NV;
    static Function<void, gl::GLuint, const gl::GLuint64EXT *> VertexAttribL3ui64vNV;
    static Function<void, gl::GLuint, gl::GLdouble, gl::GLdouble, gl::GLdouble, gl::GLdouble> VertexAttribL4d;
    static Function<void, gl::GLuint, gl::GLdouble, gl::GLdouble, gl::GLdouble, gl::GLdouble> VertexAttribL4dEXT;
    static Function<void, gl::GLuint, const gl::GLdouble *> VertexAttribL4dv;
    static Function<void, gl::GLuint, const gl::GLdouble *> VertexAttribL4dvEXT;
    static Function<void, gl::GLuint, gl::GLint64EXT, gl::GLint64EXT, gl::GLint64EXT, gl::GLint64EXT> VertexAttribL4i64NV;
    static Function<void, gl::GLuint, const gl::GLint64EXT *> VertexAttribL4i64vNV;
    static Function<void, gl::GLuint, gl::GLuint64EXT, gl::GLuint64EXT, gl::GLuint64EXT, gl::GLuint64EXT> VertexAttribL4ui64NV;
    static Function<void, gl::GLuint, const gl::GLuint64EXT *> VertexAttribL4ui64vNV;
    static Function<void, gl::GLuint, gl::GLint, gl::GLenum, gl::GLuint> VertexAttribLFormat;
    static Function<void, gl::GLuint, gl::GLint, gl::GLenum, gl::GLsizei> VertexAttribLFormatNV;
    static Function<void, gl::GLuint, gl::GLint, gl::GLenum, gl::GLsizei, const void *> VertexAttribLPointer;
    static Function<void, gl::GLuint, gl::GLint, gl::GLenum, gl::GLsizei, const void *> VertexAttribLPointerEXT;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLboolean, gl::GLuint> VertexAttribP1ui;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLboolean, const gl::GLuint *> VertexAttribP1uiv;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLboolean, gl::GLuint> VertexAttribP2ui;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLboolean, const gl::GLuint *> VertexAttribP2uiv;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLboolean, gl::GLuint> VertexAttribP3ui;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLboolean, const gl::GLuint *> VertexAttribP3uiv;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLboolean, gl::GLuint> VertexAttribP4ui;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLboolean, const gl::GLuint *> VertexAttribP4uiv;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLint> VertexAttribParameteriAMD;
    static Function<void, gl::GLuint, gl::GLint, gl::GLenum, gl::GLboolean, gl::GLsizei, const void *> VertexAttribPointer;
    static Function<void, gl::GLuint, gl::GLint, gl::GLenum, gl::GLboolean, gl::GLsizei, const void *> VertexAttribPointerARB;
    static Function<void, gl::GLuint, gl::GLint, gl::GLenum, gl::GLsizei, const void *> VertexAttribPointerNV;
    static Function<void, gl::GLuint, gl::GLsizei, const gl::GLdouble *> VertexAttribs1dvNV;
    static Function<void, gl::GLuint, gl::GLsizei, const gl::GLfloat *> VertexAttribs1fvNV;
    static Function<void, gl::GLuint, gl::GLsizei, const gl::GLhalfNV *> VertexAttribs1hvNV;
    static Function<void, gl::GLuint, gl::GLsizei, const gl::GLshort *> VertexAttribs1svNV;
    static Function<void, gl::GLuint, gl::GLsizei, const gl::GLdouble *> VertexAttribs2dvNV;
    static Function<void, gl::GLuint, gl::GLsizei, const gl::GLfloat *> VertexAttribs2fvNV;
    static Function<void, gl::GLuint, gl::GLsizei, const gl::GLhalfNV *> VertexAttribs2hvNV;
    static Function<void, gl::GLuint, gl::GLsizei, const gl::GLshort *> VertexAttribs2svNV;
    static Function<void, gl::GLuint, gl::GLsizei, const gl::GLdouble *> VertexAttribs3dvNV;
    static Function<void, gl::GLuint, gl::GLsizei, const gl::GLfloat *> VertexAttribs3fvNV;
    static Function<void, gl::GLuint, gl::GLsizei, const gl::GLhalfNV *> VertexAttribs3hvNV;
    static Function<void, gl::GLuint, gl::GLsizei, const gl::GLshort *> VertexAttribs3svNV;
    static Function<void, gl::GLuint, gl::GLsizei, const gl::GLdouble *> VertexAttribs4dvNV;
    static Function<void, gl::GLuint, gl::GLsizei, const gl::GLfloat *> VertexAttribs4fvNV;
    static Function<void, gl::GLuint, gl::GLsizei, const gl::GLhalfNV *> VertexAttribs4hvNV;
    static Function<void, gl::GLuint, gl::GLsizei, const gl::GLshort *> VertexAttribs4svNV;
    static Function<void, gl::GLuint, gl::GLsizei, const gl::GLubyte *> VertexAttribs4ubvNV;
    static Function<void, gl::GLuint, gl::GLuint> VertexBindingDivisor;
    static Function<void, gl::GLint> VertexBlendARB;
    static Function<void, gl::GLenum, gl::GLfloat> VertexBlendEnvfATI;
    static Function<void, gl::GLenum, gl::GLint> VertexBlendEnviATI;
    static Function<void, gl::GLint, gl::GLenum, gl::GLsizei> VertexFormatNV;
    static Function<void, gl::GLenum, gl::GLuint> VertexP2ui;
    static Function<void, gl::GLenum, const gl::GLuint *> VertexP2uiv;
    static Function<void, gl::GLenum, gl::GLuint> VertexP3ui;
    static Function<void, gl::GLenum, const gl::GLuint *> VertexP3uiv;
    static Function<void, gl::GLenum, gl::GLuint> VertexP4ui;
    static Function<void, gl::GLenum, const gl::GLuint *> VertexP4uiv;
    static Function<void, gl::GLint, gl::GLenum, gl::GLsizei, const void *> VertexPointer;
    static Function<void, gl::GLint, gl::GLenum, gl::GLsizei, gl::GLsizei, const void *> VertexPointerEXT;
    static Function<void, gl::GLint, gl::GLenum, gl::GLint, const void **, gl::GLint> VertexPointerListIBM;
    static Function<void, gl::GLint, gl::GLenum, const void **> VertexPointervINTEL;
    static Function<void, gl::GLenum, gl::GLdouble> VertexStream1dATI;
    static Function<void, gl::GLenum, const gl::GLdouble *> VertexStream1dvATI;
    static Function<void, gl::GLenum, gl::GLfloat> VertexStream1fATI;
    static Function<void, gl::GLenum, const gl::GLfloat *> VertexStream1fvATI;
    static Function<void, gl::GLenum, gl::GLint> VertexStream1iATI;
    static Function<void, gl::GLenum, const gl::GLint *> VertexStream1ivATI;
    static Function<void, gl::GLenum, gl::GLshort> VertexStream1sATI;
    static Function<void, gl::GLenum, const gl::GLshort *> VertexStream1svATI;
    static Function<void, gl::GLenum, gl::GLdouble, gl::GLdouble> VertexStream2dATI;
    static Function<void, gl::GLenum, const gl::GLdouble *> VertexStream2dvATI;
    static Function<void, gl::GLenum, gl::GLfloat, gl::GLfloat> VertexStream2fATI;
    static Function<void, gl::GLenum, const gl::GLfloat *> VertexStream2fvATI;
    static Function<void, gl::GLenum, gl::GLint, gl::GLint> VertexStream2iATI;
    static Function<void, gl::GLenum, const gl::GLint *> VertexStream2ivATI;
    static Function<void, gl::GLenum, gl::GLshort, gl::GLshort> VertexStream2sATI;
    static Function<void, gl::GLenum, const gl::GLshort *> VertexStream2svATI;
    static Function<void, gl::GLenum, gl::GLdouble, gl::GLdouble, gl::GLdouble> VertexStream3dATI;
    static Function<void, gl::GLenum, const gl::GLdouble *> VertexStream3dvATI;
    static Function<void, gl::GLenum, gl::GLfloat, gl::GLfloat, gl::GLfloat> VertexStream3fATI;
    static Function<void, gl::GLenum, const gl::GLfloat *> VertexStream3fvATI;
    static Function<void, gl::GLenum, gl::GLint, gl::GLint, gl::GLint> VertexStream3iATI;
    static Function<void, gl::GLenum, const gl::GLint *> VertexStream3ivATI;
    static Function<void, gl::GLenum, gl::GLshort, gl::GLshort, gl::GLshort> VertexStream3sATI;
    static Function<void, gl::GLenum, const gl::GLshort *> VertexStream3svATI;
    static Function<void, gl::GLenum, gl::GLdouble, gl::GLdouble, gl::GLdouble, gl::GLdouble> VertexStream4dATI;
    static Function<void, gl::GLenum, const gl::GLdouble *> VertexStream4dvATI;
    static Function<void, gl::GLenum, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat> VertexStream4fATI;
    static Function<void, gl::GLenum, const gl::GLfloat *> VertexStream4fvATI;
    static Function<void, gl::GLenum, gl::GLint, gl::GLint, gl::GLint, gl::GLint> VertexStream4iATI;
    static Function<void, gl::GLenum, const gl::GLint *> VertexStream4ivATI;
    static Function<void, gl::GLenum, gl::GLshort, gl::GLshort, gl::GLshort, gl::GLshort> VertexStream4sATI;
    static Function<void, gl::GLenum, const gl::GLshort *> VertexStream4svATI;
    static Function<void, gl::GLint, gl::GLenum, gl::GLsizei, const void *> VertexWeightPointerEXT;
    static Function<void, gl::GLfloat> VertexWeightfEXT;
    static Function<void, const gl::GLfloat *> VertexWeightfvEXT;
    static Function<void, gl::GLhalfNV> VertexWeighthNV;
    static Function<void, const gl::GLhalfNV *> VertexWeighthvNV;
    static Function<gl::GLenum, gl::GLuint, gl::GLuint *, gl::GLuint64EXT *> VideoCaptureNV;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLenum, const gl::GLdouble *> VideoCaptureStreamParameterdvNV;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLenum, const gl::GLfloat *> VideoCaptureStreamParameterfvNV;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLenum, const gl::GLint *> VideoCaptureStreamParameterivNV;
    static Function<void, gl::GLint, gl::GLint, gl::GLsizei, gl::GLsizei> Viewport;
    static Function<void, gl::GLuint, gl::GLsizei, const gl::GLfloat *> ViewportArrayv;
    static Function<void, gl::GLuint, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat> ViewportIndexedf;
    static Function<void, gl::GLuint, const gl::GLfloat *> ViewportIndexedfv;
    static Function<void, gl::GLuint, gl::GLfloat, gl::GLfloat> ViewportPositionWScaleNV;
    static Function<void, gl::GLuint, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLenum> ViewportSwizzleNV;
    static Function<void, gl::GLsync, gl::UnusedMask, gl::GLuint64> WaitSync;
    static Function<void, gl::GLuint, gl::GLsizei, const gl::GLuint *, const gl::GLfloat *> WeightPathsNV;
    static Function<void, gl::GLint, gl::GLenum, gl::GLsizei, const void *> WeightPointerARB;
    static Function<void, gl::GLint, const gl::GLbyte *> WeightbvARB;
    static Function<void, gl::GLint, const gl::GLdouble *> WeightdvARB;
    static Function<void, gl::GLint, const gl::GLfloat *> WeightfvARB;
    static Function<void, gl::GLint, const gl::GLint *> WeightivARB;
    static Function<void, gl::GLint, const gl::GLshort *> WeightsvARB;
    static Function<void, gl::GLint, const gl::GLubyte *> WeightubvARB;
    static Function<void, gl::GLint, const gl::GLuint *> WeightuivARB;
    static Function<void, gl::GLint, const gl::GLushort *> WeightusvARB;
    static Function<void, gl::GLdouble, gl::GLdouble> WindowPos2d;
    static Function<void, gl::GLdouble, gl::GLdouble> WindowPos2dARB;
    static Function<void, gl::GLdouble, gl::GLdouble> WindowPos2dMESA;
    static Function<void, const gl::GLdouble *> WindowPos2dv;
    static Function<void, const gl::GLdouble *> WindowPos2dvARB;
    static Function<void, const gl::GLdouble *> WindowPos2dvMESA;
    static Function<void, gl::GLfloat, gl::GLfloat> WindowPos2f;
    static Function<void, gl::GLfloat, gl::GLfloat> WindowPos2fARB;
    static Function<void, gl::GLfloat, gl::GLfloat> WindowPos2fMESA;
    static Function<void, const gl::GLfloat *> WindowPos2fv;
    static Function<void, const gl::GLfloat *> WindowPos2fvARB;
    static Function<void, const gl::GLfloat *> WindowPos2fvMESA;
    static Function<void, gl::GLint, gl::GLint> WindowPos2i;
    static Function<void, gl::GLint, gl::GLint> WindowPos2iARB;
    static Function<void, gl::GLint, gl::GLint> WindowPos2iMESA;
    static Function<void, const gl::GLint *> WindowPos2iv;
    static Function<void, const gl::GLint *> WindowPos2ivARB;
    static Function<void, const gl::GLint *> WindowPos2ivMESA;
    static Function<void, gl::GLshort, gl::GLshort> WindowPos2s;
    static Function<void, gl::GLshort, gl::GLshort> WindowPos2sARB;
    static Function<void, gl::GLshort, gl::GLshort> WindowPos2sMESA;
    static Function<void, const gl::GLshort *> WindowPos2sv;
    static Function<void, const gl::GLshort *> WindowPos2svARB;
    static Function<void, const gl::GLshort *> WindowPos2svMESA;
    static Function<void, gl::GLdouble, gl::GLdouble, gl::GLdouble> WindowPos3d;
    static Function<void, gl::GLdouble, gl::GLdouble, gl::GLdouble> WindowPos3dARB;
    static Function<void, gl::GLdouble, gl::GLdouble, gl::GLdouble> WindowPos3dMESA;
    static Function<void, const gl::GLdouble *> WindowPos3dv;
    static Function<void, const gl::GLdouble *> WindowPos3dvARB;
    static Function<void, const gl::GLdouble *> WindowPos3dvMESA;
    static Function<void, gl::GLfloat, gl::GLfloat, gl::GLfloat> WindowPos3f;
    static Function<void, gl::GLfloat, gl::GLfloat, gl::GLfloat> WindowPos3fARB;
    static Function<void, gl::GLfloat, gl::GLfloat, gl::GLfloat> WindowPos3fMESA;
    static Function<void, const gl::GLfloat *> WindowPos3fv;
    static Function<void, const gl::GLfloat *> WindowPos3fvARB;
    static Function<void, const gl::GLfloat *> WindowPos3fvMESA;
    static Function<void, gl::GLint, gl::GLint, gl::GLint> WindowPos3i;
    static Function<void, gl::GLint, gl::GLint, gl::GLint> WindowPos3iARB;
    static Function<void, gl::GLint, gl::GLint, gl::GLint> WindowPos3iMESA;
    static Function<void, const gl::GLint *> WindowPos3iv;
    static Function<void, const gl::GLint *> WindowPos3ivARB;
    static Function<void, const gl::GLint *> WindowPos3ivMESA;
    static Function<void, gl::GLshort, gl::GLshort, gl::GLshort> WindowPos3s;
    static Function<void, gl::GLshort, gl::GLshort, gl::GLshort> WindowPos3sARB;
    static Function<void, gl::GLshort, gl::GLshort, gl::GLshort> WindowPos3sMESA;
    static Function<void, const gl::GLshort *> WindowPos3sv;
    static Function<void, const gl::GLshort *> WindowPos3svARB;
    static Function<void, const gl::GLshort *> WindowPos3svMESA;
    static Function<void, gl::GLdouble, gl::GLdouble, gl::GLdouble, gl::GLdouble> WindowPos4dMESA;
    static Function<void, const gl::GLdouble *> WindowPos4dvMESA;
    static Function<void, gl::GLfloat, gl::GLfloat, gl::GLfloat, gl::GLfloat> WindowPos4fMESA;
    static Function<void, const gl::GLfloat *> WindowPos4fvMESA;
    static Function<void, gl::GLint, gl::GLint, gl::GLint, gl::GLint> WindowPos4iMESA;
    static Function<void, const gl::GLint *> WindowPos4ivMESA;
    static Function<void, gl::GLshort, gl::GLshort, gl::GLshort, gl::GLshort> WindowPos4sMESA;
    static Function<void, const gl::GLshort *> WindowPos4svMESA;
    static Function<void, gl::GLenum, gl::GLsizei, const gl::GLint *> WindowRectanglesEXT;
    static Function<void, gl::GLuint, gl::GLuint, gl::GLenum, gl::GLenum, gl::GLenum, gl::GLenum> WriteMaskEXT;

protected:
	static const array_t s_functions; ///< The list of all build-in functions.
	static std::vector<AbstractFunction *> s_additionalFunctions; ///< The list of all additional functions;
	static std::vector<ContextSwitchCallback> s_callbacks; ///< The list of all currently registered context switch callbacks.
};


} // namespace glbinding
