/*
 * Python bindings module for libfwevt (pyfwevt)
 *
 * Copyright (C) 2011-2021, Joachim Metz <joachim.metz@gmail.com>
 *
 * Refer to AUTHORS for acknowledgements.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

#include <common.h>
#include <narrow_string.h>
#include <types.h>

#if defined( HAVE_STDLIB_H ) || defined( HAVE_WINAPI )
#include <stdlib.h>
#endif

#include "pyfwevt.h"
#include "pyfwevt_error.h"
#include "pyfwevt_libcerror.h"
#include "pyfwevt_libfwevt.h"
#include "pyfwevt_manifest.h"
#include "pyfwevt_provider.h"
#include "pyfwevt_providers.h"
#include "pyfwevt_python.h"
#include "pyfwevt_unused.h"

/* The pyfwevt module methods
 */
PyMethodDef pyfwevt_module_methods[] = {
	{ "get_version",
	  (PyCFunction) pyfwevt_get_version,
	  METH_NOARGS,
	  "get_version() -> String\n"
	  "\n"
	  "Retrieves the version." },

	/* Sentinel */
	{ NULL, NULL, 0, NULL }
};

/* Retrieves the pyfwevt/libfwevt version
 * Returns a Python object if successful or NULL on error
 */
PyObject *pyfwevt_get_version(
           PyObject *self PYFWEVT_ATTRIBUTE_UNUSED,
           PyObject *arguments PYFWEVT_ATTRIBUTE_UNUSED )
{
	const char *errors           = NULL;
	const char *version_string   = NULL;
	size_t version_string_length = 0;

	PYFWEVT_UNREFERENCED_PARAMETER( self )
	PYFWEVT_UNREFERENCED_PARAMETER( arguments )

	Py_BEGIN_ALLOW_THREADS

	version_string = libfwevt_get_version();

	Py_END_ALLOW_THREADS

	version_string_length = narrow_string_length(
	                         version_string );

	/* Pass the string length to PyUnicode_DecodeUTF8
	 * otherwise it makes the end of string character is part
	 * of the string
	 */
	return( PyUnicode_DecodeUTF8(
	         version_string,
	         (Py_ssize_t) version_string_length,
	         errors ) );
}

#if PY_MAJOR_VERSION >= 3

/* The pyfwevt module definition
 */
PyModuleDef pyfwevt_module_definition = {
	PyModuleDef_HEAD_INIT,

	/* m_name */
	"pyfwevt",
	/* m_doc */
	"Python libfwevt module (pyfwevt).",
	/* m_size */
	-1,
	/* m_methods */
	pyfwevt_module_methods,
	/* m_reload */
	NULL,
	/* m_traverse */
	NULL,
	/* m_clear */
	NULL,
	/* m_free */
	NULL,
};

#endif /* PY_MAJOR_VERSION >= 3 */

/* Initializes the pyfwevt module
 */
#if PY_MAJOR_VERSION >= 3
PyMODINIT_FUNC PyInit_pyfwevt(
                void )
#else
PyMODINIT_FUNC initpyfwevt(
                void )
#endif
{
	PyObject *module           = NULL;
	PyGILState_STATE gil_state = 0;

#if defined( HAVE_DEBUG_OUTPUT )
	libfwevt_notify_set_stream(
	 stderr,
	 NULL );
	libfwevt_notify_set_verbose(
	 1 );
#endif

	/* Create the module
	 * This function must be called before grabbing the GIL
	 * otherwise the module will segfault on a version mismatch
	 */
#if PY_MAJOR_VERSION >= 3
	module = PyModule_Create(
	          &pyfwevt_module_definition );
#else
	module = Py_InitModule3(
	          "pyfwevt",
	          pyfwevt_module_methods,
	          "Python libfwevt module (pyfwevt)." );
#endif
	if( module == NULL )
	{
#if PY_MAJOR_VERSION >= 3
		return( NULL );
#else
		return;
#endif
	}
#if PY_VERSION_HEX < 0x03070000
	PyEval_InitThreads();
#endif
	gil_state = PyGILState_Ensure();

	/* Setup the manifest type object
	 */
	pyfwevt_manifest_type_object.tp_new = PyType_GenericNew;

	if( PyType_Ready(
	     &pyfwevt_manifest_type_object ) < 0 )
	{
		goto on_error;
	}
	Py_IncRef(
	 (PyObject *) &pyfwevt_manifest_type_object );

	PyModule_AddObject(
	 module,
	 "manifest",
	 (PyObject *) &pyfwevt_manifest_type_object );

	/* Setup the provider type object
	 */
	pyfwevt_provider_type_object.tp_new = PyType_GenericNew;

	if( PyType_Ready(
	     &pyfwevt_provider_type_object ) < 0 )
	{
		goto on_error;
	}
	Py_IncRef(
	 (PyObject *) &pyfwevt_provider_type_object );

	PyModule_AddObject(
	 module,
	 "provider",
	 (PyObject *) &pyfwevt_provider_type_object );

	/* Setup the providers type object
	 */
	pyfwevt_providers_type_object.tp_new = PyType_GenericNew;

	if( PyType_Ready(
	     &pyfwevt_providers_type_object ) < 0 )
	{
		goto on_error;
	}
	Py_IncRef(
	 (PyObject *) &pyfwevt_providers_type_object );

	PyModule_AddObject(
	 module,
	 "providers",
	 (PyObject *) &pyfwevt_providers_type_object );

	PyGILState_Release(
	 gil_state );

#if PY_MAJOR_VERSION >= 3
	return( module );
#else
	return;
#endif

on_error:
	PyGILState_Release(
	 gil_state );

#if PY_MAJOR_VERSION >= 3
	return( NULL );
#else
	return;
#endif
}

