/* Time-stamp: <2008-03-14 18:09:53 poser> */
/*
 * Copyright (C) 1993-2008 William J. Poser.
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of version 3 of the GNU General Public License
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */


#include "config.h"
#include "compdefs.h"
#include <stdlib.h>
#ifdef HAVE_STDINT_H
#include <stdint.h>
#endif
#include <wchar.h>
#include <stdio.h>
#ifdef HAVE_UNINUM_UNICODE_H
#include <uninum/unicode.h>
#else
#include "unicode.h"
#endif
#include "dstr.h"
#include "exitcode.h"

void StripDiacritics (wchar_t *s) {
  wchar_t c;
  wchar_t *src;
  wchar_t *tgt;

  tgt = src = s;
  while ((c = *src) != L'\0'){
    src++;
    switch (c)
      {
      case 0x00C0:
      case 0x00C1:
      case 0x00C2:
      case 0x00C3:
      case 0x00C4:
      case 0x00C5:
      case 0x0100:
      case 0x0102:
      case 0x0104:
      case 0x01CD:
      case 0x01DE:
      case 0x01E0:
      case 0x01E2:
      case 0x01FA:
      case 0x0200:
      case 0x0202:
      case 0x0226:
      case 0x1E00:
      case 0x1EA0:
      case 0x1EA2:
      case 0x1EA4:
      case 0x1EA6:
      case 0x1EA8:
      case 0x1EAA:
      case 0x1EAC:
      case 0x1EAE:
      case 0x1EB0:
      case 0x1EB2:
      case 0x1EB4:
      case 0x1EB6:
	*tgt++ = 0x0041;		/* A */
	break;
      case 0x0181:
      case 0x0182:
      case 0x0299:
      case 0x1E02:
      case 0x1E04:
      case 0x1E06:
	*tgt++ = 0x0042;		/* B */
	break;
      case 0x00C7:
      case 0x0106:
      case 0x0108:
      case 0x010A:
      case 0x010C:
      case 0x0187:
      case 0x023B:
      case 0x1E08:
	*tgt++ = 0x0043;		/* C */
	break;
      case 0x010E:
      case 0x0110:
      case 0x018A:
      case 0x018B:
      case 0x1E0A:
      case 0x1E0C:
      case 0x1E0E:
      case 0x1E10:
      case 0x1E12:
	*tgt++ = 0x0044;		/* D */
	break;
      case 0x00C8:
      case 0x00C9:
      case 0x00CA:
      case 0x00CB:
      case 0x0112:
      case 0x0114:
      case 0x0116:
      case 0x0118:
      case 0x011A:
      case 0x0204:
      case 0x0206:
      case 0x0228:
      case 0x1E14:
      case 0x1E16:
      case 0x1E18:
      case 0x1E1A:
      case 0x1E1C:
      case 0x1EB8:
      case 0x1EBA:
      case 0x1EBC:
      case 0x1EBE:
      case 0x1EC0:
      case 0x1EC2:
      case 0x1EC4:
      case 0x1EC6:
	*tgt++ = 0x0045;		/* E */
	break;
      case 0x0191:
      case 0x1E1E:
	*tgt++ = 0x0046;		/* F */
	break;
      case 0x011C:
      case 0x011E:
      case 0x0120:
      case 0x0122:
      case 0x0193:
      case 0x01E4:
      case 0x01E6:
      case 0x01F4:
      case 0x1E20:
	*tgt++ = 0x0047;		/* G */
	break;
      case 0x0124:
      case 0x0126:
      case 0x021E:
	*tgt++ = 0x0048;		/* H */
	break;
      case 0x00CC:
      case 0x00CD:
      case 0x00CE:
      case 0x00CF:
      case 0x0128:
      case 0x012A:
      case 0x012C:
      case 0x012E:
      case 0x0130:
      case 0x0197:
      case 0x01CF:
      case 0x0208:
      case 0x020A:
      case 0x026A:
      case 0x1D7B:
      case 0x1E2C:
      case 0x1E2E:
      case 0x1EC8:
      case 0x1ECA:
	*tgt++ = 0x0049;		/* I */
	break;
      case 0x0134:
	*tgt++ = 0x004A;		/* J */
	break;
      case 0x0198:
      case 0x01E8:
      case 0x1E30:
      case 0x1E32:
      case 0x1E34:
	*tgt++ = 0x004B;		/* K */
	break;
      case 0x023D:
      case 0x1E36:
      case 0x1E38:
      case 0x1E3A:
      case 0x1E3C:
	*tgt++ = 0x004C;		/* L */
	break;
      case 0x1E3E:
      case 0x1E40:
      case 0x1E42:
	*tgt++ = 0x004D;		/* M */
	break;
      case 0x00D1:
      case 0x019D:
      case 0x01F8:
      case 0x0220:
      case 0x1E44:
      case 0x1E46:
      case 0x1E48:
      case 0x1E4A:
	*tgt++ = 0x004E;		/* N */
	break;
      case 0x00D2:
      case 0x00D3:
      case 0x00D4:
      case 0x00D5:
      case 0x00D6:
      case 0x00D8:
      case 0x014C:
      case 0x014E:
      case 0x0150:
      case 0x019F:
      case 0x01A0:
      case 0x01D1:
      case 0x01EA:
      case 0x01EC:
      case 0x01FE:
      case 0x020C:
      case 0x020E:
      case 0x022A:
      case 0x022C:
      case 0x022E:
      case 0x0230:
      case 0x1D0F:
      case 0x1E4C:
      case 0x1E4E:
      case 0x1E50:
      case 0x1E52:
      case 0x1ECC:
      case 0x1ECE:
      case 0x1ED0:
      case 0x1ED2:
      case 0x1ED4:
      case 0x1ED6:
      case 0x1ED8:
      case 0x1EDA:
      case 0x1EDC:
      case 0x1EDE:
      case 0x1EE0:
      case 0x1EE2:
	*tgt++ = 0x004F;		/* O */
	break;
      case 0x01A4:
      case 0x1E54:
      case 0x1E56:
	*tgt++ = 0x0050;		/* P */
	break;
      case 0x0210:
      case 0x0212:
      case 0x1E58:
      case 0x1E5A:
      case 0x1E5C:
      case 0x1E5E:
	*tgt++ = 0x0052;		/* R */
	break;
      case 0x0218:
      case 0x1E60:
      case 0x1E62:
      case 0x1E64:
      case 0x1E66:
      case 0x1E68:
	*tgt++ = 0x0053;		/* S */
	break;
      case 0x01AC:
      case 0x01AE:
      case 0x021A:
      case 0x023E:
      case 0x1E6A:
      case 0x1E6C:
      case 0x1E6E:
      case 0x1E70:
	*tgt++ = 0x0054;		/* T */
	break;
      case 0x00D9:
      case 0x00DA:
      case 0x00DB:
      case 0x00DC:
      case 0x0168:
      case 0x016A:
      case 0x016C:
      case 0x016E:
      case 0x0170:
      case 0x0172:
      case 0x01AF:
      case 0x01D3:
      case 0x01D5:
      case 0x01D7:
      case 0x01D9:
      case 0x01DB:
      case 0x0214:
      case 0x0216:
      case 0x1D1C:
      case 0x1D7E:
      case 0x1E72:
      case 0x1E74:
      case 0x1E76:
      case 0x1E78:
      case 0x1E7A:
      case 0x1EE4:
      case 0x1EE6:
      case 0x1EE8:
      case 0x1EEA:
      case 0x1EEC:
      case 0x1EEE:
      case 0x1EF0:
	*tgt++ = 0x0055;		/* U */
	break;
      case 0x01B2:
      case 0x1E7C:
      case 0x1E7E:
	*tgt++ = 0x0056;		/* V */
	break;
      case 0x0174:
      case 0x1E80:
      case 0x1E82:
      case 0x1E84:
      case 0x1E86:
      case 0x1E88:
	*tgt++ = 0x0057;		/* W */
	break;
      case 0x1E8A:
      case 0x1E8C:
	*tgt++ = 0x0058;		/* X */
	break;
      case 0x00DD:
      case 0x0176:
      case 0x0178:
      case 0x01B3:
      case 0x0232:
      case 0x028F:
      case 0x1E8E:
      case 0x1EF2:
      case 0x1EF4:
      case 0x1EF6:
      case 0x1EF8:
	*tgt++ = 0x0059;		/* Y */
	break;
      case 0x0179:
      case 0x017B:
      case 0x017D:
      case 0x01B5:
      case 0x0224:
      case 0x1D22:
      case 0x1E90:
      case 0x1E92:
      case 0x1E94:
	*tgt++ = 0x005A;		/* Z */
	break;
      case 0x00E0: 		/* a */
      case 0x00E1:
      case 0x00E2:
      case 0x00E3:
      case 0x00E4:
      case 0x00E5:
      case 0x0101:
      case 0x0103:
      case 0x0105:
      case 0x01CE:
      case 0x01DF:
      case 0x01E1:
      case 0x01FB:
      case 0x0201:
      case 0x0203:
      case 0x0227:
      case 0x1D00:
      case 0x1E01:
      case 0x1E9A:
      case 0x1EA1:
      case 0x1EA3:
      case 0x1EA5:
      case 0x1EA7:
      case 0x1EA9:
      case 0x1EB1:
      case 0x1EB3:
      case 0x1EB5:
      case 0x1EB7:
	*tgt++ = 0x0061;
	break;
      case 0x0062:		/* b */
      case 0x0180:
      case 0x0183:
      case 0x0253:
      case 0x1D03:
      case 0x1E03:
      case 0x1E05:
      case 0x1E07:
	*tgt++ = 0x0062;
	break;
      case 0x00E7:
      case 0x0107:
      case 0x0109:
      case 0x010B:
      case 0x010D:
      case 0x0188:
      case 0x0255:
      case 0x1E09:
	*tgt++ = 0x0063;		/* c */
	break;
      case 0x010F:
      case 0x0111:
      case 0x018C:
      case 0x0221:
      case 0x0256:
      case 0x0257:
      case 0x1D6D:
      case 0x1D81:
      case 0x1D91:
      case 0x1E0B:
      case 0x1E0D:
      case 0x1E0F:
      case 0x1E11:
      case 0x1E13:
	*tgt++ = 0x0064;		/* d */
	break;
      case 0x00E8:
      case 0x00E9:
      case 0x00EA:
      case 0x00EB:
      case 0x0113:
      case 0x0115:
      case 0x0117:
      case 0x0119:
      case 0x011B:
      case 0x0205:
      case 0x0207:
      case 0x0229:
      case 0x1D07:
      case 0x1E15:
      case 0x1E17:
      case 0x1E19:
      case 0x1E1B:
      case 0x1E1D:
      case 0x1EB9:
      case 0x1EBB:
      case 0x1EBD:
      case 0x1EBF:
      case 0x1EC1:
      case 0x1EC3:
      case 0x1EC5:
      case 0x1EC7:
	*tgt++ = 0x0065;		/* e */
	break;
      case 0x0192:
      case 0x1D6E:
      case 0x1D82:
      case 0x1E1F:
	*tgt++ = 0x0066;		/* f */
	break;
      case 0x011D:
      case 0x011F:
      case 0x0121:
      case 0x0123:
      case 0x01E5:
      case 0x01E7:
      case 0x01F5:
      case 0x0260:
      case 0x1E21:
	*tgt++ = 0x0067;		/* g */
	break;
      case 0x0125:
      case 0x0127:
      case 0x021F:
      case 0x0266:
      case 0x1E23:
      case 0x1E25:
      case 0x1E27:
      case 0x1E29:
      case 0x1E2B:
      case 0x1E96:
	*tgt++ = 0x0068;		/* h */
	break;
      case 0x00EC:
      case 0x00ED:
      case 0x00EE:
      case 0x00EF:
      case 0x0129:
      case 0x012B:
      case 0x012D:
      case 0x012F:
      case 0x01D0:
      case 0x0209:
      case 0x020B:
      case 0x0268:
      case 0x1D96:
      case 0x1E2D:
      case 0x1E2F:
      case 0x1EC9:
      case 0x1ECB:
      case 0x2071:
	*tgt++ = 0x0069;		/* i */
	break;
      case 0x0135:
      case 0x01F0:
      case 0x029D:
	*tgt++ = 0x006A;		/* j */
	break;
      case 0x0199:
      case 0x01E9:
      case 0x1D84:
      case 0x1E31:
      case 0x1E33:
      case 0x1E35:
	*tgt++ = 0x006B;		/* k */
	break;
      case 0x019A:
      case 0x0234:
      case 0x026B:
      case 0x026C:
      case 0x026D:
      case 0x1D85:
      case 0x1E37:
      case 0x1E39:
      case 0x1E3B:
      case 0x1E3D:
	*tgt++ = 0x006C;		/* l */
	break;
      case 0x0271:
      case 0x1D6F:
      case 0x1D86:
      case 0x1E3F:
      case 0x1E41:
      case 0x1E43:
	*tgt++ = 0x006D;		/* m */
	break;
      case 0x00F1:
      case 0x019E:
      case 0x01F9:
      case 0x0272:
      case 0x0273:
      case 0x1D70:
      case 0x1D87:
      case 0x1E45:
      case 0x1E47:
      case 0x1E49:
      case 0x1E4B:
	*tgt++ = 0x006E;		/* n */
	break;
      case 0x00F2:
      case 0x00F3:
      case 0x00F4:
      case 0x00F5:
      case 0x00F6:
      case 0x00F8:
      case 0x01A1:
      case 0x01D2:
      case 0x01EB:
      case 0x01ED:
      case 0x01FF:
      case 0x020D:
      case 0x020F:
      case 0x022B:
      case 0x022D:
      case 0x022F:
      case 0x0231:
      case 0x1E4D:
      case 0x1E4F:
      case 0x1E51:
      case 0x1E53:
      case 0x1ECD:
      case 0x1ECF:
      case 0x1ED1:
      case 0x1ED3:
      case 0x1ED5:
      case 0x1ED7:
      case 0x1ED9:
      case 0x1EDB:
      case 0x1EDD:
      case 0x1EDF:
      case 0x1EE1:
      case 0x1EE3:
	*tgt++ = 0x006F;		/* o */
	break;
      case 0x01A5:
      case 0x1D71:
      case 0x1D7D:
      case 0x1D88:
      case 0x1E55:
      case 0x1E57:
	*tgt++ = 0x0070;		/* p */
	break;
      case 0x02A0:
	*tgt++ = 0x0071;		/* q */
	break;
      case 0x0211:
      case 0x0213:
      case 0x027C:
      case 0x027D:
      case 0x027E:
      case 0x1D72:
      case 0x1D73:
      case 0x1D89:
      case 0x1E59:
      case 0x1E5B:
      case 0x1E5D:
      case 0x1E5F:
	*tgt++ = 0x0072;		/* r */
	break;
      case 0x0219:
      case 0x023F:
      case 0x0282:
      case 0x1D74:
      case 0x1D8A:
      case 0x1E61:
      case 0x1E63:
      case 0x1E65:
      case 0x1E67:
      case 0x1E69:
	*tgt++ = 0x0073;		/* s */
	break;
      case 0x01AB:
      case 0x01AD:
      case 0x021B:
      case 0x0288:
      case 0x1D75:
      case 0x1E6B:
      case 0x1E6D:
      case 0x1E6F:
      case 0x1E71:
      case 0x1E97:
	*tgt++ = 0x0074;		/* t */
	break;
      case 0x00F9:
      case 0x00FA:
      case 0x00FB:
      case 0x00FC:
      case 0x0171:
      case 0x0173:
      case 0x01B0:
      case 0x01D4:
      case 0x01D6:
      case 0x01D8:
      case 0x01DA:
      case 0x01DC:
      case 0x0215:
      case 0x0217:
      case 0x0289:
      case 0x1D99:
      case 0x1E73:
      case 0x1E75:
      case 0x1E77:
      case 0x1E79:
      case 0x1E7B:
      case 0x1EE5:
      case 0x1EE7:
      case 0x1EE9:
      case 0x1EEB:
      case 0x1EED:
      case 0x1EEF:
      case 0x1EF1:
	*tgt++ = 0x0075;		/* u */
	break;
      case 0x028B:
      case 0x1D8C:
      case 0x1E7D:
      case 0x1E7F:
	*tgt++ = 0x0076;		/* v */
	break;
      case 0x0175:
      case 0x1E81:
      case 0x1E83:
      case 0x1E85:
      case 0x1E87:
      case 0x1E89:
      case 0x1E98:
	*tgt++ = 0x0077;		/* w */
	break;
      case 0x1D8D:
      case 0x1E8B:
      case 0x1E8D:
	*tgt++ = 0x0078;		/* x */
	break;
      case 0x00FD:
      case 0x00FE:
      case 0x00FF:
      case 0x0177:
      case 0x01B4:
      case 0x0233:
      case 0x1E8F:
      case 0x1E99:
      case 0x1EF3:
      case 0x1EF5:
      case 0x1EF7:
      case 0x1EF9:
	*tgt++ = 0x0079;		/* y */
	break;
      case 0x017A:
      case 0x017C:
      case 0x017E:
      case 0x01B6:
      case 0x0225:
      case 0x0290:
      case 0x0291:
      case 0x1E91:
      case 0x1E93:
      case 0x1E95:
	*tgt++ = 0x007A;		/* z */
	break;
      case 0x0300:		/* Combining diacrtics - just skip */
      case 0x0301:
      case 0x0302:
      case 0x0303:
      case 0x0304:
      case 0x0305:
      case 0x0306:
      case 0x0307:
      case 0x0308:
      case 0x0309:
      case 0x030A:
      case 0x030B:
      case 0x030C:
      case 0x030D:
      case 0x030E:
      case 0x030F:
      case 0x0310:
      case 0x0311:
      case 0x0312:
      case 0x0313:
      case 0x0314:
      case 0x0315:
      case 0x0316:
      case 0x0317:
      case 0x0318:
      case 0x0319:
      case 0x031A:
      case 0x031B:
      case 0x031C:
      case 0x031D:
      case 0x031E:
      case 0x031F:
      case 0x0320:
      case 0x0321:
      case 0x0322:
      case 0x0323:
      case 0x0324:
      case 0x0325:
      case 0x0326:
      case 0x0327:
      case 0x0328:
      case 0x0329:
      case 0x032A:
      case 0x032B:
      case 0x032C:
      case 0x032D:
      case 0x032E:
      case 0x032F:
      case 0x0330:
      case 0x0331:
      case 0x0332:
      case 0x0333:
      case 0x0334:
      case 0x0335:
      case 0x0336:
      case 0x0337:
      case 0x0338:
      case 0x0339:
      case 0x033A:
      case 0x033B:
      case 0x033C:
      case 0x033D:
      case 0x033E:
      case 0x033F:
      case 0x0340:
      case 0x0341:
      case 0x0342:
      case 0x0343:
      case 0x0344:
      case 0x0345:
      case 0x0346:
      case 0x0347:
      case 0x0348:
      case 0x0349:
      case 0x034A:
      case 0x034B:
      case 0x034C:
      case 0x034D:
      case 0x034E:
      case 0x034F:
      case 0x0350:
      case 0x0351:
      case 0x0352:
      case 0x0353:
      case 0x0354:
      case 0x0355:
      case 0x0356:
      case 0x0357:
      case 0x0358:
      case 0x0359:
      case 0x035A:
      case 0x035B:
      case 0x035C:
      case 0x035D:
      case 0x035E:
      case 0x035F:
      case 0x0360:
      case 0x0361:
      case 0x0362:
	continue;
      default:
	*tgt++ = c;
	break;
      }	/* End of switch */
  } /* End of while */
  *tgt = L'\0';
}

void ConvertEnclosed (wchar_t *s) {
  wchar_t c;
  wchar_t *src;
  wchar_t *tgt;

  tgt = src = s;
  while ((c = *src) != L'\0'){
    src++;
    switch (c) {
      /* Parenthesized letters */
    case 0x249C:
    case 0x24D0:
      *tgt++ = 0x61;
      break;
    case 0x249D:
    case 0x24D1:
      *tgt++ = 0x62;
      break;
    case 0x249E:
    case 0x24D2:
      *tgt++ = 0x63;
      break;
    case 0x249F:
    case 0x24D3:
      *tgt++ = 0x64;
      break;
    case 0x24A0:
    case 0x24D4:
      *tgt++ = 0x65;
      break;
    case 0x24A1:
    case 0x24D5:
      *tgt++ = 0x66;
      break;
    case 0x24A2:
    case 0x24D6:
      *tgt++ = 0x67;
      break;
    case 0x24A3:
    case 0x24D7:
      *tgt++ = 0x68;
      break;
    case 0x24A4:
    case 0x24D8:
      *tgt++ = 0x69;
      break;
    case 0x24A5:
    case 0x24D9:
      *tgt++ = 0x6A;
      break;
    case 0x24A6:
    case 0x24DA:
      *tgt++ = 0x6B;
      break;
    case 0x24A7:
    case 0x24DB:
      *tgt++ = 0x6C;
      break;
    case 0x24A8:
    case 0x24DC:
      *tgt++ = 0x6D;
      break;
    case 0x24A9:
    case 0x24DD:
      *tgt++ = 0x6E;
      break;
    case 0x24AA:
    case 0x24DE:
      *tgt++ = 0x6F;
      break;
    case 0x24AB:
    case 0x24DF:
      *tgt++ = 0x70;
      break;
    case 0x24AC:
    case 0x24E0:
      *tgt++ = 0x71;
      break;
    case 0x24AD:
    case 0x24E1:
      *tgt++ = 0x72;
      break;
    case 0x24AE:
    case 0x24E2:
      *tgt++ = 0x73;
      break;
    case 0x24AF:
    case 0x24E3:
      *tgt++ = 0x74;
      break;
    case 0x24B0:
    case 0x24E4:
      *tgt++ = 0x75;
      break;
    case 0x24B1:
    case 0x24E5:
      *tgt++ = 0x76;
      break;
    case 0x24B2:
    case 0x24E6:
      *tgt++ = 0x77;
      break;
    case 0x24B3:
    case 0x24E7:
      *tgt++ = 0x78;
      break;
    case 0x24B4:
    case 0x24E8:
      *tgt++ = 0x79;
      break;
    case 0x24B5:
    case 0x24E9:
      *tgt++ = 0x7A;
      break;
      /* Circled capital letters */
    case 0x24B6:
      *tgt++ = 0x41;
      break;
    case 0x24B7:
      *tgt++ = 0x42;
      break;
    case 0x24B8:
      *tgt++ = 0x43;
      break;
    case 0x24B9:
      *tgt++ = 0x44;
      break;
    case 0x24BA:
      *tgt++ = 0x45;
      break;
    case 0x24BB:
      *tgt++ = 0x46;
      break;
    case 0x24BC:
      *tgt++ = 0x47;
      break;
    case 0x24BD:
      *tgt++ = 0x48;
      break;
    case 0x24BE:
      *tgt++ = 0x49;
      break;
    case 0x24BF:
      *tgt++ = 0x4A;
      break;
    case 0x24C0:
      *tgt++ = 0x4B;
      break;
    case 0x24C1:
      *tgt++ = 0x4C;
      break;
    case 0x24C2:
      *tgt++ = 0x4D;
      break;
    case 0x24C3:
      *tgt++ = 0x4E;
      break;
    case 0x24C4:
      *tgt++ = 0x4F;
      break;
    case 0x24C5:
      *tgt++ = 0x50;
      break;
    case 0x24C6:
      *tgt++ = 0x51;
      break;
    case 0x24C7:
      *tgt++ = 0x52;
      break;
    case 0x24C8:
      *tgt++ = 0x53;
      break;
    case 0x24C9:
      *tgt++ = 0x54;
      break;
    case 0x24CA:
      *tgt++ = 0x55;
      break;
    case 0x24CB:
      *tgt++ = 0x56;
      break;
    case 0x24CC:
      *tgt++ = 0x57;
      break;
    case 0x24CD:
      *tgt++ = 0x58;
      break;
    case 0x24CE:
      *tgt++ = 0x59;
      break;
    case 0x24CF:
      *tgt++ = 0x5A;
      break;
    default:
	*tgt++ = c;
	break;
      }	/* End of switch */
  } /* End of while */
  *tgt = L'\0';
}

void ConvertStylistic (struct dstr *ptr) {
  wchar_t c;
  wchar_t *src;
  wchar_t *tgt;
  int ExtraCnt = 0;
  int CharCnt = 0;
  int CharsNeeded;
  wchar_t *new;

  /*If necessary, increase storage for Hebrew presentation forms */
  src = ptr->s;
  while ((c = *src++) != L'\0'){
    CharCnt++;
    if((c >= 0xFB1D) && (c <= 0xFB4F)) {
      ExtraCnt++;
    }
  }
  if(ExtraCnt) {
    CharsNeeded = CharCnt + ExtraCnt;
    if(CharsNeeded > ptr->c) {
      new = (wchar_t *) malloc((CharsNeeded +1) * sizeof(wchar_t));
      if(new == NULL){
	fprintf(stderr,"Out of memory.\n");
	exit(OUTOFMEMORY);
      }
      ptr->l = ptr->c = CharsNeeded;
    }
    src = ptr->s;
    tgt = new;
  }
  else {tgt = src = ptr->s;}

  while ((c = *src) != L'\0'){
    src++;
    switch (c)
      {
      case 0xFE57:
      case 0xFF01:		/* Exclamation mark */
	*tgt++ = 0x21;
	break;
      case 0xFF02:		/* Quotation mark */
	*tgt++ = 0x22;
	break;
      case 0xFE5F:
      case 0xFF03:		/* Number sign */
	*tgt++ = 0x23;
	break;
      case 0xFE69:
      case 0xFF04:		/* Dollar sign */
	*tgt++ = 0x24;
	break;
      case 0xFE6A:
      case 0xFF05:		/* Percent sign */
	*tgt++ = 0x25;
	break;
      case 0xFE60:
      case 0xFF06:		/* Ampersand  */
	*tgt++ = 0x26;
	break;
      case 0xFF07:		/* Apostrophe */
	*tgt++ = 0x27;
	break;
      case 0x207D:
      case 0x208D:
      case 0xFE59:
      case 0xFF08:		/* Left parenthesis */
	*tgt++ = 0x28;
	break;
      case 0x207E:
      case 0x208E:
      case 0xFE5A:
      case 0xFF09:		/* Right parenthesis */
	*tgt++ = 0x29;
	break;
      case 0xFE61:
      case 0xFF0A:		/* Asterisk */
	*tgt++ = 0x2A;
	break;
      case 0x207A:
      case 0x208A:
      case 0xFE62:
      case 0xFF0B:		/* Plus sign */
	*tgt++ = 0x2B;
	break;
      case 0xFE50:
      case 0xFF0C:		/* Comma */
	*tgt++ = 0x2C;
	break;
      case 0x207B:
      case 0x208B:
      case 0xFE63:
      case 0xFF0D:		/* Hyphen */
	*tgt++ = 0x2D;
	break;
      case 0xFE52:
      case 0xFF0E:		/* Full stop */
	*tgt++ = 0x2E;
	break;
      case 0xFF0F:		/* Solidus */
	*tgt++ = 0x2F;
	break;
      case 0x2070:
      case 0x2080:
      case 0xFF10:		/* Digit zero */
	*tgt++ = 0x30;
	break;
      case 0x00B9:
      case 0x2081:
      case 0xFF11:
	*tgt++ = 0x31;
	break;
      case 0x00B2:
      case 0x2082:
      case 0xFF12:
	*tgt++ = 0x32;
	break;
      case 0x00B3:
      case 0x2083:
      case 0xFF13:
	*tgt++ = 0x33;
	break;
      case 0x2074:
      case 0x2084:
      case 0xFF14:
	*tgt++ = 0x34;
	break;
      case 0x2075:
      case 0x2085:
      case 0xFF15:
	*tgt++ = 0x35;
	break;
      case 0x2076:
      case 0x2086:
      case 0xFF16:
	*tgt++ = 0x36;
	break;
      case 0x2077:
      case 0x2087:
      case 0xFF17:
	*tgt++ = 0x37;
	break;
      case 0x2078:
      case 0x2088:
      case 0xFF18:
	*tgt++ = 0x38;
	break;
      case 0x2079:
      case 0x2089:
      case 0xFF19:		/* Digit nine */
	*tgt++ = 0x39;
	break;
      case 0xFE55:
      case 0xFF1A:		/* Colon */
	*tgt++ = 0x3A;
	break;
      case 0xFE54:
      case 0xFF1B:		/* Semicolon */
	*tgt++ = 0x3B;
	break;
      case 0xFE64:
      case 0xFF1C:		/* Less than sign */
	*tgt++ = 0x3C;
	break;
      case 0x207C:
      case 0x208C:
      case 0xFE66:
      case 0xFF1D:		
	*tgt++ = 0x3D;		/* Equals sign */
	break;
      case 0xFE65:
      case 0xFF1E:		/* Greater than sign */
	*tgt++ = 0x3E;
	break;
      case 0xFE56:
      case 0xFF1F:		/* Question mark */
	*tgt++ = 0x3F;
	break;
      case 0xFE6B:
      case 0xFF20:		/* At sign */
	*tgt++ = 0x40;
	break;
      case 0x0041:		/* A */
      case 0xFF21:
      case 0x1D400:
      case 0x1D434:
      case 0x1D468:
      case 0x1D49C:
      case 0x1D4D0:
      case 0x1D504:
      case 0x1D538:
      case 0x1D56C:
      case 0x1D5A0:
      case 0x1D5D4:
      case 0x1D63C:
      case 0x1D670:
	*tgt++ = 0x41;
	break;
      case 0x0042:		/* B */
      case 0x212C:
      case 0xFF22:
      case 0x1D401:
      case 0x1D435:
      case 0x1D469:
      case 0x1D4D1:
      case 0x1D505:
      case 0x1D539:
      case 0x1D56D:
      case 0x1D5A1:
      case 0x1D5D5:
      case 0x1D609:
      case 0x1D63D:
      case 0x1D671:
	*tgt++ = 0x42;
	break;
      case 0x2102:
      case 0x212D:
      case 0xFF23:
      case 0x1D402:
      case 0x1D436:
      case 0x1D46A:
      case 0x1D49E:
      case 0x1D4D2:
      case 0x1D56E:
      case 0x1D5A2:
      case 0x1D5D6:
      case 0x1D60A:
      case 0x1D63E:
      case 0x1D672:
	*tgt++ = 0x43;		/* C */
	break;
      case 0x2145:
      case 0xFF24:
      case 0x1D403:
      case 0x1D437:
      case 0x1D46B:
      case 0x1D49F:
      case 0x1D4D3:
      case 0x1D507:
      case 0x1D53B:
      case 0x1D56F:
      case 0x1D5A3:
      case 0x1D5D7:
      case 0x1D60B:
      case 0x1D63F:
      case 0x1D673:
	*tgt++ = 0x44;		/* D */
	break;
      case 0x0045:		/* E*/
      case 0x2130:
      case 0xFF25:
      case 0x1D404:
      case 0x1D438:
      case 0x1D46C:
      case 0x1D4D4:
      case 0x1D508:
      case 0x1D53C:
      case 0x1D570:
      case 0x1D5A4:
      case 0x1D5D8:
      case 0x1D60C:
      case 0x1D640:
      case 0x1D674:
	*tgt++ = 0x45;
	break;
      case 0x2131:
      case 0x213F:
      case 0xFF26:
      case 0x1D405:
      case 0x1D439:
      case 0x1D46D:
      case 0x1D4D5:
      case 0x1D509:
      case 0x1D53D:
      case 0x1D571:
      case 0x1D5A5:
      case 0x1D5D9:
      case 0x1D60D:
      case 0x1D641:
      case 0x1D675:
	*tgt++ = 0x46;		/* F */
	break;
      case 0x0262:
      case 0xFF27:
      case 0x1D406:
      case 0x1D43A:
      case 0x1D46E:
      case 0x1D4A2:
      case 0x1D4D6:
      case 0x1D50A:
      case 0x1D53E:
      case 0x1D572:
      case 0x1D5A6:
      case 0x1D5DA:
      case 0x1D60E:
      case 0x1D642:
      case 0x1D676:
	*tgt++ = 0x47;		/* G */
	break;
      case 0x029C:
      case 0x210B:
      case 0x210C:
      case 0x210D:
      case 0xFF28:
      case 0x1D407:
      case 0x1D43B:
      case 0x1D46F:
      case 0x1D4D7:
      case 0x1D573:
      case 0x1D5A7:
      case 0x1D5DB:
      case 0x1D60F:
      case 0x1D643:
      case 0x1D677:
	*tgt++ = 0x48;		/* H */
	break;
      case 0x0049:
      case 0x2110:
      case 0x2111:
      case 0x2160:
      case 0xFF29:
      case 0x1D408:
      case 0x1D43C:
      case 0x1D470:
      case 0x1D4D8:
      case 0x1D540:
      case 0x1D574:
      case 0x1D5A8:
      case 0x1D5DC:
      case 0x1D610:
      case 0x1D644:
      case 0x1D678:
	*tgt++ = 0x49;		/* I */
	break;
      case 0x026A:
      case 0xFF2A:
      case 0x1D409:
      case 0x1D43D:
      case 0x1D471:
      case 0x1D4D9:
      case 0x1D541:
      case 0x1D575:
      case 0x1D5A9:
      case 0x1D5DD:
      case 0x1D611:
      case 0x1D645:
      case 0x1D679:
	*tgt++ = 0x4A;		/* J */
	break;
      case 0xFF2B:
      case 0x1D40A:
      case 0x1D43E:
      case 0x1D472:
      case 0x1D4A5:
      case 0x1D4DA:
      case 0x1D50D:
      case 0x1D542:
      case 0x1D576:
      case 0x1D5AA:
      case 0x1D5DE:
      case 0x1D612:
      case 0x1D646:
      case 0x1D67A:
	*tgt++ = 0x4B;		/* K */
	break;
      case 0xFF2C:
      case 0x1D40B:
      case 0x1D43F:
      case 0x1D473:
      case 0x1D4A6:
      case 0x1D4DB:
      case 0x1D50E:
      case 0x1D543:
      case 0x1D577:
      case 0x1D5AB:
      case 0x1D5DF:
      case 0x1D613:
      case 0x1D647:
      case 0x1D67B:
	*tgt++ = 0x4C;		/* L */
	break;
      case 0x029F:
      case 0xFF2D:
      case 0x2112:
      case 0x1D40C:
      case 0x1D440:
      case 0x1D474:
      case 0x1D4DC:
      case 0x1D50F:
      case 0x1D544:
      case 0x1D578:
      case 0x1D5AC:
      case 0x1D5E0:
      case 0x1D614:
      case 0x1D648:
      case 0x1D67C:
	*tgt++ = 0x4D;		/* M */
	break;
      case 0x2133:
      case 0xFF2E:
      case 0x1D40D:
      case 0x1D441:
      case 0x1D475:
      case 0x1D4A9:
      case 0x1D4DD:
      case 0x1D511:
      case 0x1D579:
      case 0x1D5AD:
      case 0x1D5E1:
      case 0x1D615:
      case 0x1D649:
      case 0x1D67D:
	*tgt++ = 0x4E;		/* N */
	break;
      case 0x004F:
      case 0x2205:
      case 0xFF2F:
      case 0x1D40E:
      case 0x1D442:
      case 0x1D476:
      case 0x1D4AA:
      case 0x1D4DE:
      case 0x1D512:
      case 0x1D546:
      case 0x1D57A:
      case 0x1D5AE:
      case 0x1D5E2:
      case 0x1D616:
      case 0x1D64A:
      case 0x1D67E:
	*tgt++ = 0x4F;		/* O */
	break;
      case 0x2118:
      case 0x2119:
      case 0xFF30:
      case 0x1D40F:
      case 0x1D443:
      case 0x1D477:
      case 0x1D4AB:
      case 0x1D4DF:
      case 0x1D513:
      case 0x1D57B:
      case 0x1D5AF:
      case 0x1D5E3:
      case 0x1D617:
      case 0x1D64B:
      case 0x1D67F:
	*tgt++ = 0x50;		/* P */
	break;
      case 0x211A:
      case 0xFF31:
      case 0x1D410:
      case 0x1D444:
      case 0x1D478:
      case 0x1D4AC:
      case 0x1D4E0:
      case 0x1D514:
      case 0x1D57C:
      case 0x1D5B0:
      case 0x1D5E4:
      case 0x1D618:
      case 0x1D64C:
      case 0x1D680:
	*tgt++ = 0x51;		/* Q */
	break;
      case 0x0280:
      case 0x211B:
      case 0x211C:
      case 0x211D:
      case 0xFF32:
      case 0x1D411:
      case 0x1D445:
      case 0x1D479:
      case 0x1D4E1:
      case 0x1D57D:
      case 0x1D5B1:
      case 0x1D5E5:
      case 0x1D619:
      case 0x1D64D:
      case 0x1D681:
	*tgt++ = 0x52;		/* R */
	break;
      case 0xFF33:
      case 0x1D412:
      case 0x1D446:
      case 0x1D47A:
      case 0x1D4AE:
      case 0x1D4E2:
      case 0x1D516:
      case 0x1D54A:
      case 0x1D57E:
      case 0x1D5B2:
      case 0x1D5E6:
      case 0x1D61A:
      case 0x1D64E:
      case 0x1D682:
	*tgt++ = 0x53;		/* S */
	break;
      case 0xFF34:
      case 0x1D413:
      case 0x1D447:
      case 0x1D47B:
      case 0x1D4AF:
      case 0x1D4E3:
      case 0x1D517:
      case 0x1D54B:
      case 0x1D57F:
      case 0x1D5B3:
      case 0x1D5E7:
      case 0x1D61B:
      case 0x1D64F:
      case 0x1D683:
	*tgt++ = 0x54;		/* T */
	break;
      case 0x0055:
      case 0xFF35:
      case 0x1D414:
      case 0x1D448:
      case 0x1D47C:
      case 0x1D4B0:
      case 0x1D4E4:
      case 0x1D518:
      case 0x1D54C:
      case 0x1D580:
      case 0x1D5B4:
      case 0x1D5E8:
      case 0x1D61C:
      case 0x1D650:
      case 0x1D684:
	*tgt++ = 0x55;		/* U */
	break;
      case 0xFF36:
      case 0x1D415:
      case 0x1D449:
      case 0x1D47D:
      case 0x1D4B1:
      case 0x1D4E5:
      case 0x1D519:
      case 0x1D54D:
      case 0x1D581:
      case 0x1D5B5:
      case 0x1D5E9:
      case 0x1D61D:
      case 0x1D651:
      case 0x1D685:
	*tgt++ = 0x56;		/* V */
	break;
      case 0xFF37:
      case 0x1D416:
      case 0x1D44A:
      case 0x1D47E:
      case 0x1D4B2:
      case 0x1D4E6:
      case 0x1D51A:
      case 0x1D54E:
      case 0x1D582:
      case 0x1D5B6:
      case 0x1D5EA:
      case 0x1D61E:
      case 0x1D652:
      case 0x1D686:
	*tgt++ = 0x57;		/* W */
	break;
      case 0xFF38:
      case 0x1D417:
      case 0x1D44B:
      case 0x1D47F:
      case 0x1D4B3:
      case 0x1D4E7:
      case 0x1D51B:
      case 0x1D54F:
      case 0x1D583:
      case 0x1D5B7:
      case 0x1D5EB:
      case 0x1D61F:
      case 0x1D653:
      case 0x1D687:
	*tgt++ = 0x58;		/* X */
	break;
      case 0x0059:
      case 0x2144:
      case 0xFF39:
      case 0x1D418:
      case 0x1D44C:
      case 0x1D480:
      case 0x1D4B4:
      case 0x1D4E8:
      case 0x1D51C:
      case 0x1D550:
      case 0x1D584:
      case 0x1D5B8:
      case 0x1D5EC:
      case 0x1D620:
      case 0x1D654:
      case 0x1D688:
	*tgt++ = 0x59;		/* Y */
	break;
      case 0x29F5:
      case 0x29F9:
      case 0xFE68:
	*tgt++ = 0x5C;		/* Backslash */
	break;
      case 0x1D00:
      case 0x2124:
      case 0x2128:
      case 0xFF3A:
      case 0x1D419:
      case 0x1D44D:
      case 0x1D656:
      case 0x1D68A:
	*tgt++ = 0x61;		/* a */
	break;
      case 0xFF42:		/* b */
      case 0x1D41B:
      case 0x1D44F:
      case 0x1D483:
      case 0x1D4B7:
      case 0x1D4EB:
      case 0x1D51F:
      case 0x1D553:
      case 0x1D587:
      case 0x1D5BB:
      case 0x1D5EF:
      case 0x1D623:
      case 0x1D657:
      case 0x1D68B:
	*tgt++ = 0x62;
	break;
      case 0x0063:		/* c */
      case 0x1D04:
      case 0xFF43:
      case 0x1D41C:
      case 0x1D450:
      case 0x1D484:
      case 0x1D4B8:
      case 0x1D4EC:
      case 0x1D520:
      case 0x1D554:
      case 0x1D588:
      case 0x1D5BC:
      case 0x1D5F0:
      case 0x1D624:
      case 0x1D658:
      case 0x1D68C:
	*tgt++ = 0x63;
	break;
      case 0x0064:		/* d */
      case 0x1D05:
      case 0x2146:
      case 0xFF44:
      case 0x1D41D:
      case 0x1D451:
      case 0x1D485:
      case 0x1D4B9:
      case 0x1D4ED:
      case 0x1D521:
      case 0x1D555:
      case 0x1D589:
      case 0x1D5BD:
      case 0x1D5F1:
      case 0x1D625:
      case 0x1D659:
      case 0x1D68D:
	*tgt++ = 0x64;
	break;
      case 0x1D07:
      case 0x212F:
      case 0x2147:
      case 0xFF45:
      case 0x1D41E:
      case 0x1D452:
      case 0x1D486:
      case 0x1D4EE:
      case 0x1D522:
      case 0x1D556:
      case 0x1D5BE:
      case 0x1D5F2:
      case 0x1D626:
      case 0x1D65A:
      case 0x1D68E:
	*tgt++ = 0x65;		/* e */
	break;
      case 0xFF46:
      case 0x1D41F:
      case 0x1D453:
      case 0x1D487:
      case 0x1D4BB:
      case 0x1D4EF:
      case 0x1D523:
      case 0x1D557:
      case 0x1D58B:
      case 0x1D5BF:
      case 0x1D5F3:
      case 0x1D627:
      case 0x1D65B:
      case 0x1D68F:
	*tgt++ = 0x66;
	break;
      case 0x0067:		/* g */
      case 0xFF47:
      case 0x210A:
      case 0x1D420:
      case 0x1D454:
      case 0x1D488:
      case 0x1D4F0:
      case 0x1D524:
      case 0x1D558:
      case 0x1D58C:
      case 0x1D5C0:
      case 0x1D5F4:
      case 0x1D628:
      case 0x1D65C:
      case 0x1D690:
	*tgt++ = 0x67;
	break;
      case 0x0068:		/* h */
      case 0xFF48:
      case 0x1D421:
      case 0x1D489:
      case 0x1D4BD:
      case 0x1D4F1:
      case 0x1D525:
      case 0x1D559:
      case 0x1D58D:
      case 0x1D5C1:
      case 0x1D5F5:
      case 0x1D629:
      case 0x1D65D:
      case 0x1D691:
	*tgt++ = 0x68;
	break;
      case 0x0069:		/* i */
      case 0xFF49:
      case 0x2148:
      case 0x1D422:
      case 0x1D456:
      case 0x1D48A:
      case 0x1D4BE:
      case 0x1D4F2:
      case 0x1D526:
      case 0x1D55A:
      case 0x1D58E:
      case 0x1D5C2:
      case 0x1D5F6:
      case 0x1D62A:
      case 0x1D65E:
      case 0x1D692:
	*tgt++ = 0x69;
	break;
      case 0x006A:		/* j */
      case 0x1D0A:
      case 0x2149:
      case 0xFF4A:
      case 0x1D423:
      case 0x1D457:
      case 0x1D48B:
      case 0x1D4BF:
      case 0x1D4F3:
      case 0x1D527:
      case 0x1D55B:
      case 0x1D58F:
      case 0x1D5C3:
      case 0x1D5F7:
      case 0x1D62B:
      case 0x1D65F:
      case 0x1D693:
	*tgt++ = 0x6A;
	break;
      case 0x006B:		/* k */
      case 0x1D0B:
      case 0xFF4B:
      case 0x1D424:
      case 0x1D458:
      case 0x1D48C:
      case 0x1D4C0:
      case 0x1D4F4:
      case 0x1D528:
      case 0x1D55C:
      case 0x1D590:
      case 0x1D5C4:
      case 0x1D5F8:
      case 0x1D62C:
      case 0x1D660:
      case 0x1D694:
	*tgt++ = 0x6B;
	break;
      case 0x006C:		/* l */
      case 0x2113:
      case 0xFF4C:
      case 0x1D425:
      case 0x1D459:
      case 0x1D48D:
      case 0x1D4C1:
      case 0x1D4F5:
      case 0x1D529:
      case 0x1D55D:
      case 0x1D591:
      case 0x1D5C5:
      case 0x1D5F9:
      case 0x1D62D:
      case 0x1D661:
      case 0x1D695:
	*tgt++ = 0x6C;
	break;
      case 0x006D:		/* m */
      case 0x1D0D:
      case 0xFF4D:
      case 0x1D426:
      case 0x1D45A:
      case 0x1D48E:
      case 0x1D4C2:
      case 0x1D52A:
      case 0x1D55E:
      case 0x1D592:
      case 0x1D5C6:
      case 0x1D5FA:
      case 0x1D62E:
      case 0x1D662:
      case 0x1D696:
	*tgt++ = 0x6D;
	break;
      case 0x207F:
      case 0xFF4E:
      case 0x1D427:
      case 0x1D45B:
      case 0x1D48F:
      case 0x1D4C3:
      case 0x1D4F7:
      case 0x1D52B:
      case 0x1D55F:
      case 0x1D593:
      case 0x1D5C7:
      case 0x1D5FB:
      case 0x1D62F:
      case 0x1D663:
      case 0x1D697:
	*tgt++ = 0x6E;		/* n */
	break;
      case 0x006F:		/* o */
      case 0x1D0F:
      case 0x2134:
      case 0xFF4F:
      case 0x1D428:
      case 0x1D45C:
      case 0x1D490:
      case 0x1D4F8:
      case 0x1D52C:
      case 0x1D560:
      case 0x1D594:
      case 0x1D5C8:
      case 0x1D5FC:
      case 0x1D630:
      case 0x1D664:
      case 0x1D698:
	*tgt++ = 0x6F;
	break;
      case 0x0070:		/* p */
      case 0x1D18:
      case 0xFF50:
      case 0x213C:
      case 0x1D429:
      case 0x1D45D:
      case 0x1D491:
      case 0x1D4C5:
      case 0x1D4F9:
      case 0x1D52D:
      case 0x1D561:
      case 0x1D595:
      case 0x1D5C9:
      case 0x1D5FD:
      case 0x1D631:
      case 0x1D665:
      case 0x1D699:
	*tgt++ = 0x70;
	break;
      case 0x0071:		/* q */
      case 0xFF51:
      case 0x1D42A:
      case 0x1D45E:
      case 0x1D492:
      case 0x1D4C6:
      case 0x1D4FA:
      case 0x1D52E:
      case 0x1D562:
      case 0x1D596:
      case 0x1D5CA:
      case 0x1D5FE:
      case 0x1D632:
      case 0x1D666:
      case 0x1D69A:
	*tgt++ = 0x71;
	break;
      case 0x0072:		/* r */
      case 0xFF52:
      case 0x1D42B:
      case 0x1D45F:
      case 0x1D493:
      case 0x1D4C7:
      case 0x1D4FB:
      case 0x1D52F:
      case 0x1D563:
      case 0x1D597:
      case 0x1D5CB:
      case 0x1D5FF:
      case 0x1D633:
      case 0x1D667:
      case 0x1D69B:
	*tgt++ = 0x72;
	break;
      case 0x0073:		/* s */
      case 0xFF53:
      case 0x1D42C:
      case 0x1D460:
      case 0x1D494:
      case 0x1D4C8:
      case 0x1D4FC:
      case 0x1D530:
      case 0x1D564:
      case 0x1D598:
      case 0x1D5CC:
      case 0x1D600:
      case 0x1D634:
      case 0x1D668:
      case 0x1D69C:
	*tgt++ = 0x73;
	break;
      case 0x0074:		/* t */
      case 0x1D1B:
      case 0xFF54:
      case 0x1D42D:
      case 0x1D461:
      case 0x1D495:
      case 0x1D4C9:
      case 0x1D4FD:
      case 0x1D531:
      case 0x1D565:
      case 0x1D599:
      case 0x1D5CD:
      case 0x1D601:
      case 0x1D635:
      case 0x1D669:
      case 0x1D69D:
	*tgt++ = 0x74;
	break;
      case 0x0075:		/* u */
      case 0x1D1C:
      case 0xFF55:
      case 0x1D42E:
      case 0x1D462:
      case 0x1D496:
      case 0x1D4CA:
      case 0x1D4FE:
      case 0x1D532:
      case 0x1D566:
      case 0x1D59A:
      case 0x1D5CE:
      case 0x1D602:
      case 0x1D636:
      case 0x1D66A:
      case 0x1D69E:
	*tgt++ = 0x75;
	break;
      case 0x0076:		/* v */
      case 0x1D20:
      case 0xFF56:
      case 0x1D42F:
      case 0x1D463:
      case 0x1D497:
      case 0x1D4CB:
      case 0x1D4FF:
      case 0x1D533:
      case 0x1D567:
      case 0x1D59B:
      case 0x1D5CF:
      case 0x1D603:
      case 0x1D637:
      case 0x1D66B:
      case 0x1D69F:
	*tgt++ = 0x76;
	break;
      case 0x0077:		/* w */
      case 0x1D21:
      case 0x24B2:
      case 0x24E6:
      case 0xFF57:
      case 0x1D430:
      case 0x1D464:
      case 0x1D498:
      case 0x1D4CC:
      case 0x1D500:
      case 0x1D534:
      case 0x1D568:
      case 0x1D59C:
      case 0x1D5D0:
      case 0x1D604:
      case 0x1D638:
      case 0x1D66C:
      case 0x1D6A0:
	*tgt++ = 0x77;
	break;
      case 0x0078:		/* x */
      case 0xFF58:
      case 0x1D431:
      case 0x1D465:
      case 0x1D499:
      case 0x1D4CD:
      case 0x1D501:
      case 0x1D535:
      case 0x1D569:
      case 0x1D59D:
      case 0x1D5D1:
      case 0x1D605:
      case 0x1D639:
      case 0x1D66D:
      case 0x1D6A1:
	*tgt++ = 0x78;
	break;
      case 0xFF59:
      case 0x1D432:
      case 0x1D466:
      case 0x1D49A:
      case 0x1D4CE:
      case 0x1D502:
      case 0x1D536:
      case 0x1D56A:
      case 0x1D59E:
      case 0x1D5D2:
      case 0x1D606:
      case 0x1D63A:
      case 0x1D66E:
      case 0x1D6A2:
	*tgt++ = 0x79;		/* y */
	break;
      case 0x007A:		/* z */
      case 0x1D22:
      case 0xFF5A:
      case 0x1D433:
      case 0x1D467:
      case 0x1D49B:
      case 0x1D4CF:
      case 0x1D503:
      case 0x1D537:
      case 0x1D56B:
      case 0x1D59F:
      case 0x1D5D3:
      case 0x1D607:
      case 0x1D63B:
      case 0x1D66F:
      case 0x1D6A3:
	*tgt++ = 0x7A;
	break;
      case 0xFE5B:
      case 0xFF5B:		/* left curly bracket */
	*tgt++ = 0x7B;
	break;
      case 0xFF5C:		/* pipe */
	*tgt++ = 0x7C;
	break;
      case 0xFE5C:
      case 0xFF5D:		/* right curly bracket */
	*tgt++ = 0x7D;
	break;
      case 0xFF5E:		/* tilde */
	*tgt++ = 0x7E;
	break;
				/* Hebrew presentation forms */
      case 0xFB1D:
	break;
	*tgt++ = 0x05D9;
	*tgt++ = 0x05B4;
      case 0xFB1F:
	*tgt++ = 0x05F2;
	*tgt++ = 0x05B7;
      case 0xFB2A:
	*tgt++ = 0x05E9;
	*tgt++ = 0x05C1;
      case 0xFB2B:
	*tgt++ = 0x05E9;
	*tgt++ = 0x05C2;
      case 0xFB2C:
	*tgt++ = 0xFB49;
	*tgt++ = 0x05C1;
      case 0xFB2D:
	*tgt++ = 0xFB49;
	*tgt++ = 0x05C2;
      case 0xFB2E:
	*tgt++ = 0x05D0;
	*tgt++ = 0x05B7;
      case 0xFB2F:
	*tgt++ = 0x05D0;
	*tgt++ = 0x05B8;
      case 0xFB30:
	*tgt++ = 0x05D0;
	*tgt++ = 0x05BC;
      case 0xFB31:
	*tgt++ = 0x05D1;
	*tgt++ = 0x05BC;
      case 0xFB32:
	*tgt++ = 0x05D2;
	*tgt++ = 0x05BC;
      case 0xFB33:
	*tgt++ = 0x05D3;
	*tgt++ = 0x05BC;
      case 0xFB34:
	*tgt++ = 0x05D4;
	*tgt++ = 0x05BC;
      case 0xFB35:
	*tgt++ = 0x05D5;
	*tgt++ = 0x05BC;
      case 0xFB36:
	*tgt++ = 0x05D6;
	*tgt++ = 0x05BC;
      case 0xFB38:
	*tgt++ = 0x05D8;
	*tgt++ = 0x05BC;
      case 0xFB39:
	*tgt++ = 0x05D9;
	*tgt++ = 0x05BC;
      case 0xFB3A:
	*tgt++ = 0x05DA;
	*tgt++ = 0x05BC;
      case 0xFB3B:
	*tgt++ = 0x05DB;
	*tgt++ = 0x05BC;
      case 0xFB3C:
	*tgt++ = 0x05DC;
	*tgt++ = 0x05BC;
      case 0xFB3E:
	*tgt++ = 0x05DE;
	*tgt++ = 0x05BC;
      case 0xFB40:
	*tgt++ = 0x05E0;
	*tgt++ = 0x05BC;
      case 0xFB41:
	*tgt++ = 0x05E1;
	*tgt++ = 0x05BC;
      case 0xFB43:
	*tgt++ = 0x05E3;
	*tgt++ = 0x05BC;
      case 0xFB44:
	*tgt++ = 0x05E4;
	*tgt++ = 0x05BC;
      case 0xFB46:
	*tgt++ = 0x05E6;
	*tgt++ = 0x05BC;
      case 0xFB47:
	*tgt++ = 0x05E7;
	*tgt++ = 0x05BC;
      case 0xFB48:
	*tgt++ = 0x05E8;
	*tgt++ = 0x05BC;
      case 0xFB49:
	*tgt++ = 0x05E9;
	*tgt++ = 0x05BC;
      case 0xFB4A:
	*tgt++ = 0x05EA;
	*tgt++ = 0x05BC;
      case 0xFB4B:
	*tgt++ = 0x05D5;
	*tgt++ = 0x05B9;
      case 0xFB4C:
	*tgt++ = 0x05D1;
	*tgt++ = 0x05BF;
      case 0xFB4D:
	*tgt++ = 0x05DB;
	*tgt++ = 0x05BF;
      case 0xFB4E:
	*tgt++ = 0x05E4;
	*tgt++ = 0x05BF;
      case 0xFB4F:
	*tgt++ = 0x05D0;
	*tgt++ = 0x05DC;
      case 0xFB1E:
	*tgt++ = 0x05BF;
      case 0xFB20:
	*tgt++ = 0x05E2;
      case 0xFB21:
	*tgt++ = 0x05D0;
      case 0xFB22:
	*tgt++ = 0x05D3;
      case 0xFB23:
	*tgt++ = 0x05D4;
      case 0xFB24:
	*tgt++ = 0x05DB;
      case 0xFB25:
	*tgt++ = 0x05DC;
      case 0xFB26:
	*tgt++ = 0x05DD;
      case 0xFB27:
	*tgt++ = 0x05E8;
      case 0xFB28:
	*tgt++ = 0x05EA;
      case 0xFB29:
	*tgt++ = 0x002B;
				/* Arabic presentation forms */
      case 0xFE81:
      case 0xFE82:
	*tgt++ = 0x0622;
	break;
      case 0xFE8D:
      case 0xFE8E:
	*tgt++ = 0x0627;
	break;
      case 0xFE8F:
      case 0xFE91:
      case 0xFE92:
      case 0xFE90:
	*tgt++ = 0x0628;
	break;
      case 0xFB56:
      case 0xFB58:
      case 0xFB59:
      case 0xFB57:
	*tgt++ = 0x067E;
	break;
      case 0xFE95:
      case 0xFE97:
      case 0xFE98:
      case 0xFE96:
	*tgt++ = 0x062A;
	break;
      case 0xFB66:
      case 0xFB68:
      case 0xFB69:
      case 0xFB67:
	*tgt++ = 0x0679;
	break;
      case 0xFE99:
      case 0xFE9B:
      case 0xFE9C:
      case 0xFE9A:
	*tgt++ = 0x062B;
	break;
      case 0xFE9D:
      case 0xFE9F:
      case 0xFEA0:
      case 0xFE9E:
	*tgt++ = 0x062C;
	break;
      case 0xFB7A:
      case 0xFB7C:
      case 0xFB7D:
      case 0xFB7B:
	*tgt++ = 0x0686;
	break;
      case 0xFEA1:
      case 0xFEA3:
      case 0xFEA4:
      case 0xFEA2:
	*tgt++ = 0x062D;
	break;
      case 0xFEA5:
      case 0xFEA7:
      case 0xFEA8:
      case 0xFEA6:
	*tgt++ = 0x062E;
	break;
      case 0xFEA9:
      case 0xFEAA:
	*tgt++ = 0x062F;
	break;
      case 0xFB88:
      case 0xFB89:
	*tgt++ = 0x0688;
	break;
      case 0xFEAB:
      case 0xFEAC:
	*tgt++ = 0x0630;
	break;
      case 0xFEAD:
      case 0xFEAE:
      case 0xFB8C:
      case 0xFB8D:
	*tgt++ = 0x0691;
	break;
      case 0xFB8A:
      case 0xFB8B:
	*tgt++ = 0x0698;
	break;
      case 0xFEAF:
      case 0xFEB0:
	*tgt++ = 0x0632;
	break;
      case 0xFEB1:
      case 0xFEB3:
      case 0xFEB4:
      case 0xFEB2:
	*tgt++ = 0x0633;
	break;
      case 0xFEB5:
      case 0xFEB7:
      case 0xFEB8:
      case 0xFEB6:
	*tgt++ = 0x0634;
	break;
      case 0xFEB9:
      case 0xFEBB:
      case 0xFEBC:
      case 0xFEBA:
	*tgt++ = 0x0635;
	break;
      case 0xFEBD:
      case 0xFEBF:
      case 0xFEC0:
      case 0xFEBE:
	*tgt++ = 0x0636;
	break;
      case 0xFEC1:
      case 0xFEC3:
      case 0xFEC4:
      case 0xFEC2:
	*tgt++ = 0x0637;
	break;
      case 0xFEC5:
      case 0xFEC7:
      case 0xFEC8:
      case 0xFEC6:
	*tgt++ = 0x0638;
	break;
      case 0xFEC9:
      case 0xFECB:
      case 0xFECC:
      case 0xFECA:
	*tgt++ = 0x0639;
      case 0xFECD:
      case 0xFECF:
      case 0xFED0:
      case 0xFECE:
	*tgt++ = 0x063A;
	break;
      case 0xFED1:
      case 0xFED3:
      case 0xFED4:
      case 0xFED2:
	*tgt++ = 0x0641;
	break;
      case 0xFED9:
      case 0xFEDB:
      case 0xFEDC:
      case 0xFEDA:
	*tgt++ = 0x0643;
	break;
      case 0xFB8E:
      case 0xFB90:
      case 0xFB91:
      case 0xFB8F:
	*tgt++ = 0x06A9;
	break;
      case 0xFED5:
      case 0xFED7:
      case 0xFED8:
      case 0xFED6:
	*tgt++ = 0x0642;
	break;
      case 0xFB92:
      case 0xFB94:
      case 0xFB95:
      case 0xFB93:
	*tgt++ = 0x06AF;
	break;
      case 0xFEDD:
      case 0xFEDE:
      case 0xFEDF:
      case 0xFEE0:
	*tgt++ = 0x0644;
	break;
      case 0xFEE1:
      case 0xFEE3:
      case 0xFEE4:
      case 0xFEE2:
	*tgt++ = 0x0645;
	break;
      case 0xFEE5:
      case 0xFEE7:
      case 0xFEE8:
      case 0xFEE6:
	*tgt++ = 0x0646;
	break;
      case 0xFB9E:
      case 0xFB9F:
	*tgt++ = 0x06BA;
	break;
      case 0xFEED:
      case 0xFEEE:
	*tgt++ = 0x0648;
	break;
      case 0xFE85:
      case 0xFE86:
	*tgt++ = 0x0624;
	break;
      case 0xFE80:
	*tgt++ = 0x0621;
	break;
      case 0xFBA6:
      case 0xFBA8:
      case 0xFBA9:
      case 0xFBA7:
	*tgt++ = 0x06C1;
	break;
      case 0xFEE9:
      case 0xFEEB:
      case 0xFEEC:
      case 0xFEEA:
	*tgt++ = 0x0647;
	break;
      case 0xFBAA:
      case 0xFBAC:
      case 0xFBAD:
      case 0xFBAB:
	*tgt++ = 0x06BE;
	break;
      case 0xFE93:
      case 0xFE94:
	*tgt++ = 0x0629;
	break;
      case 0xFBA4:
      case 0xFBA5:
	*tgt++ = 0x06C0;
	break;
      case 0xFBFC:
      case 0xFBFE:
      case 0xFBFF:
      case 0xFBFD:
	*tgt++ = 0x06CC;
	break;
      case 0xFEEF:
      case 0xFEF0:
	*tgt++ = 0x0649;
	break;
      case 0xFEF1:
      case 0xFEF3:
      case 0xFEF4:
      case 0xFEF2:
	*tgt++ = 0x064A;
	break;
      case 0xFBAE:
      case 0xFBAF:
	*tgt++ = 0x06D2;
	break;
      case 0xFBB0:
      case 0xFBB1:
	*tgt++ = 0x06D3;
	break;
      case 0xFE89:
      case 0xFE8B:
      case 0xFE8C:
      case 0xFE8A:
	*tgt++ = 0x0626;
	break;
      case 0xFE83:
      case 0xFE84:
	*tgt++ = 0x0623;
	break;
      case 0xFE87:
      case 0xFE88:
	*tgt++ = 0x0625;
	break;
      case 0xFD7A:
      case 0xFD7B:
	break;
      case 0xFE74:
	*tgt++ = 0x064D;
	break;
      case 0xFE7C:
      case 0xFE7D:
	*tgt++ = 0x0651;
	break;
      case 0xFE72:
	*tgt++ = 0x064C;
	break;
      case 0xFE76:
      case 0xFE77:
	*tgt++ = 0x064E;
	break;
      case 0xFE7E:
      case 0xFE7F:
	*tgt++ = 0x0652;
	break;
      case 0xFE70:
	*tgt++ = 0x064B;
      case 0xFDF7:
	*tgt++ = 0x0611;
	break;
	/* Halfwidth CJK punctuation */
      case 0xFF61:
	*tgt++ = 0x3002;
	break;
      case 0xFF62:
	*tgt++ = 0x300C;
	break;
      case 0xFF63:
	*tgt++ = 0x300D;
	break;
      case 0xFF64:
	*tgt++ = 0x3001;
	break;
	/* Halfwidth kana */
      case 0xFF65:
	*tgt++ = 0x30FB;
	break;
      case 0xFF66:
	*tgt++ = 0x30F2;
	break;
      case 0xFF67:
	*tgt++ = 0x30A1;
	break;
      case 0xFF68:
	*tgt++ = 0x30A3;
	break;
      case 0xFF69:
	*tgt++ = 0x30A5;
	break;
      case 0xFF6A:
	*tgt++ = 0x30A7;
	break;
      case 0xFF6B:
	*tgt++ = 0x30A9;
	break;
      case 0xFF6C:
	*tgt++ = 0x30E3;
	break;
      case 0xFF6D:
	*tgt++ = 0x30E5;
	break;
      case 0xFF6E:
	*tgt++ = 0x30E7;
	break;
      case 0xFF6F:
	*tgt++ = 0x30C3;
	break;
      case 0xFF70:
	*tgt++ = 0x30FC;
	break;
      case 0xFF71:
	*tgt++ = 0x30A2;
	break;
      case 0xFF72:
	*tgt++ = 0x30A4;
	break;
      case 0xFF73:
	*tgt++ = 0x30A6;
	break;
      case 0xFF74:
	*tgt++ = 0x30A8;
	break;
      case 0xFF75:
	*tgt++ = 0x30AA;
	break;
      case 0xFF76:
	*tgt++ = 0x30AB;
	break;
      case 0xFF77:
	*tgt++ = 0x30AD;
	break;
      case 0xFF78:
	*tgt++ = 0x30AF;
	break;
      case 0xFF79:
	*tgt++ = 0x30B1;
	break;
      case 0xFF7A:
	*tgt++ = 0x30B3;
	break;
      case 0xFF7B:
	*tgt++ = 0x30B5;
	break;
      case 0xFF7C:
	*tgt++ = 0x30B7;
	break;
      case 0xFF7D:
	*tgt++ = 0x30B9;
	break;
      case 0xFF7E:
	*tgt++ = 0x30BB;
	break;
      case 0xFF7F:
	*tgt++ = 0x30BD;
	break;
      case 0xFF80:
	*tgt++ = 0x30BF;
	break;
      case 0xFF81:
	*tgt++ = 0x30C1;
	break;
      case 0xFF82:
	*tgt++ = 0x30C4;
	break;
      case 0xFF83:
	*tgt++ = 0x30C6;
	break;
      case 0xFF84:
	*tgt++ = 0x30C8;
	break;
      case 0xFF85:
	*tgt++ = 0x30CA;
	break;
      case 0xFF86:
	*tgt++ = 0x30CB;
	break;
      case 0xFF87:
	*tgt++ = 0x30CC;
	break;
      case 0xFF88:
	*tgt++ = 0x30CD;
	break;
      case 0xFF89:
	*tgt++ = 0x30CE;
	break;
      case 0xFF8A:
	*tgt++ = 0x30CF;
	break;
      case 0xFF8B:
	*tgt++ = 0x30D2;
	break;
      case 0xFF8C:
	*tgt++ = 0x30D5;
	break;
      case 0xFF8D:
	*tgt++ = 0x30D8;
	break;
      case 0xFF8E:
	*tgt++ = 0x30DB;
	break;
      case 0xFF8F:
	*tgt++ = 0x30DE;
	break;
      case 0xFF90:
	*tgt++ = 0x30DF;
	break;
      case 0xFF91:
	*tgt++ = 0x30E0;
	break;
      case 0xFF92:
	*tgt++ = 0x30E1;
	break;
      case 0xFF93:
	*tgt++ = 0x30E2;
	break;
      case 0xFF94:
	*tgt++ = 0x30E4;
	break;
      case 0xFF95:
	*tgt++ = 0x30E6;
	break;
      case 0xFF96:
	*tgt++ = 0x30E8;
	break;
      case 0xFF97:
	*tgt++ = 0x30E9;
	break;
      case 0xFF98:
	*tgt++ = 0x30EA;
	break;
      case 0xFF99:
	*tgt++ = 0x30EB;
	break;
      case 0xFF9A:
	*tgt++ = 0x30EC;
	break;
      case 0xFF9B:
	*tgt++ = 0x30ED;
	break;
      case 0xFF9C:
	*tgt++ = 0x30EF;
	break;
      case 0xFF9D:
	*tgt++ = 0x30F3;
	break;
      case 0xFF9E:
	*tgt++ = 0x3099;
	break;
      case 0xFF9F:
	*tgt++ = 0x309A;
	break;
	/* Halfwidth hangul */
      case 0xFFA0:
	*tgt++ = 0x3164;
	break;
      case 0xFFA1:
	*tgt++ = 0x3131;
	break;
      case 0xFFA2:
	*tgt++ = 0x3132;
	break;
      case 0xFFA3:
	*tgt++ = 0x3133;
	break;
      case 0xFFA4:
	*tgt++ = 0x3134;
	break;
      case 0xFFA5:
	*tgt++ = 0x3135;
	break;
      case 0xFFA6:
	*tgt++ = 0x3136;
	break;
      case 0xFFA7:
	*tgt++ = 0x3137;
	break;
      case 0xFFA8:
	*tgt++ = 0x3138;
	break;
      case 0xFFA9:
	*tgt++ = 0x3139;
	break;
      case 0xFFAA:
	*tgt++ = 0x313A;
	break;
      case 0xFFAB:
	*tgt++ = 0x313B;
	break;
      case 0xFFAC:
	*tgt++ = 0x313C;
	break;
      case 0xFFAD:
	*tgt++ = 0x313D;
	break;
      case 0xFFAE:
	*tgt++ = 0x313E;
	break;
      case 0xFFAF:
	*tgt++ = 0x313F;
	break;
      case 0xFFB0:
	*tgt++ = 0x3140;
	break;
      case 0xFFB1:
	*tgt++ = 0x3141;
	break;
      case 0xFFB2:
	*tgt++ = 0x3142;
	break;
      case 0xFFB3:
	*tgt++ = 0x3143;
	break;
      case 0xFFB4:
	*tgt++ = 0x3144;
	break;
      case 0xFFB5:
	*tgt++ = 0x3145;
	break;
      case 0xFFB6:
	*tgt++ = 0x3146;
	break;
      case 0xFFB7:
	*tgt++ = 0x3147;
	break;
      case 0xFFB8:
	*tgt++ = 0x3148;
	break;
      case 0xFFB9:
	*tgt++ = 0x3149;
	break;
      case 0xFFBA:
	*tgt++ = 0x314A;
	break;
      case 0xFFBB:
	*tgt++ = 0x314B;
	break;
      case 0xFFBC:
	*tgt++ = 0x314C;
	break;
      case 0xFFBD:
	*tgt++ = 0x314D;
	break;
      case 0xFFBE:
	*tgt++ = 0x314E;
	break;
      case 0xFFC2:
	*tgt++ = 0x314F;
	break;
      case 0xFFC3:
	*tgt++ = 0x3150;
	break;
      case 0xFFC4:
	*tgt++ = 0x3151;
	break;
      case 0xFFC5:
	*tgt++ = 0x3152;
	break;
      case 0xFFC6:
	*tgt++ = 0x3153;
	break;
      case 0xFFC7:
	*tgt++ = 0x3154;
	break;
      case 0xFFCA:
	*tgt++ = 0x3155;
	break;
      case 0xFFCB:
	*tgt++ = 0x3156;
	break;
      case 0xFFCC:
	*tgt++ = 0x3157;
	break;
      case 0xFFCD:
	*tgt++ = 0x3158;
	break;
      case 0xFFCE:
	*tgt++ = 0x3159;
	break;
      case 0xFFCF:
	*tgt++ = 0x315A;
	break;
      case 0xFFD2:
	*tgt++ = 0x315B;
	break;
      case 0xFFD3:
	*tgt++ = 0x315C;
	break;
      case 0xFFD4:
	*tgt++ = 0x315D;
	break;
      case 0xFFD5:
	*tgt++ = 0x315E;
	break;
      case 0xFFD6:
	*tgt++ = 0x315F;
	break;
      case 0xFFD7:
	*tgt++ = 0x3160;
	break;
      case 0xFFDA:
	*tgt++ = 0x3161;
	break;
      case 0xFFDB:
	*tgt++ = 0x3162;
	break;
      case 0xFFDC:
	*tgt++ = 0x3163;
	break;
	/* Halfwidth (Latin but non-ASCII) symbols */
      case 0xFFE0:
	*tgt++ = 0x00A2;
	break;
      case 0xFFE1:
	*tgt++ = 0x00A3;
	break;
      case 0xFFE2:
	*tgt++ = 0x00AC;
	break;
      case 0xFFE3:
	*tgt++ = 0x00AF;
	break;
      case 0xFFE4:
	*tgt++ = 0x00A6;
	break;
      case 0xFFE5:
	*tgt++ = 0x00A5;
	break;
      case 0xFFE6:
	*tgt++ = 0x20A9;
	break;
      case 0xFFE8:
	*tgt++ = 0x2502;
	break;
      case 0xFFE9:
	*tgt++ = 0x2190;
	break;
      case 0xFFEA:
	*tgt++ = 0x2191;
	break;
      case 0xFFEB:
	*tgt++ = 0x2192;
	break;
      case 0xFFEC:
	*tgt++ = 0x2193;
	break;
      case 0xFFED:
	*tgt++ = 0x25A0;
	break;
      case 0xFFEE:
	*tgt++ = 0x25CB;
	break;
      default:
	*tgt++ = c;
	break;
      }	/* End of switch */
  } /* End of while */
  *tgt = L'\0';
  if(ExtraCnt) {
    ptr->s = new;
    free ( (void *) ptr->s);
  }
} /* End of ConvertStylistic */

