<!--

@license Apache-2.0

Copyright (c) 2019 The Stdlib Authors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

   http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

-->

# iterFilterMap

> Create an [iterator][mdn-iterator-protocol] which both filters and maps the values of another [iterator][mdn-iterator-protocol].

<!-- Section to include introductory text. Make sure to keep an empty line after the intro `section` element and another before the `/section` close. -->

<section class="intro">

</section>

<!-- /.intro -->

<!-- Package usage documentation. -->

<section class="usage">

## Usage

```javascript
var iterFilterMap = require( '@stdlib/iter/filter-map' );
```

#### iterFilterMap( iterator, fcn\[, thisArg] )

Returns an [iterator][mdn-iterator-protocol] which both filters **and** maps the values of another [`iterator`][mdn-iterator-protocol] according to a callback function.

```javascript
var array2iterator = require( '@stdlib/array/to-iterator' );

function fcn( v ) {
    if ( v > 2 ) {
        return v * 10;
    }
}

var it = iterFilterMap( array2iterator( [ 1, 3, 2, 4 ] ), fcn );
// returns <Object>

var v = it.next().value;
// returns 30

v = it.next().value;
// returns 40

var bool = it.next().done;
// returns true
```

The returned [iterator][mdn-iterator-protocol] protocol-compliant object has the following properties:

-   **next**: function which returns an [iterator][mdn-iterator-protocol] protocol-compliant object containing the next iterated value (if one exists) assigned to a `value` property and a `done` property having a `boolean` value indicating whether the [iterator][mdn-iterator-protocol] is finished.
-   **return**: function which closes an [iterator][mdn-iterator-protocol] and returns a single (optional) argument in an [iterator][mdn-iterator-protocol] protocol-compliant object.

The callback function is provided two arguments:

-   `value`: iterated value
-   `index`: iteration index (zero-based)

```javascript
var array2iterator = require( '@stdlib/array/to-iterator' );

function fcn( v, i ) {
    if ( i >= 2 ) {
        return v * 10;
    }
}

var it = iterFilterMap( array2iterator( [ 3, 4, 1, 2 ] ), fcn );
// returns <Object>

var v = it.next().value;
// returns 10

v = it.next().value;
// returns 20

var bool = it.next().done;
// returns true
```

To set the callback function execution context, provide a `thisArg`.

<!-- eslint-disable no-invalid-this -->

```javascript
var array2iterator = require( '@stdlib/array/to-iterator' );

function fcn( v ) {
    this.count += 1;
    if ( v > 0 ) {
        return v * 10;
    }
}

var ctx = {
    'count': 0
};

var it = iterFilterMap( array2iterator( [ 1, 2, 3, 4 ] ), fcn, ctx );
// returns <Object>

var v = it.next().value;
// returns 10

v = it.next().value;
// returns 20

v = it.next().value;
// returns 30

var count = ctx.count;
// returns 3
```

</section>

<!-- /.usage -->

<!-- Package usage notes. Make sure to keep an empty line after the `section` element and another before the `/section` close. -->

<section class="notes">

## Notes

-   If the callback function returns `undefined`, the returned [iterator][mdn-iterator-protocol] invokes the callback for the next value of the provided [`iterator`][mdn-iterator-protocol]. If the callback function returns a value other than `undefined`, the returned [iterator][mdn-iterator-protocol] returns the callback's return value.
-   If an environment supports `Symbol.iterator` **and** a provided [iterator][mdn-iterator-protocol] is iterable, the returned [iterator][mdn-iterator-protocol] is iterable.

</section>

<!-- /.notes -->

<!-- Package usage examples. -->

<section class="examples">

## Examples

<!-- eslint no-undef: "error" -->

```javascript
var randu = require( '@stdlib/random/iter/randu' );
var iterFilterMap = require( '@stdlib/iter/filter-map' );

function fcn( v ) {
    if ( v > 0.5 ) {
        return v * 10.0;
    }
}

// Create a seeded iterator for generating pseudorandom numbers:
var rand = randu({
    'seed': 1234,
    'iter': 20
});

// Create an iterator which both filters and maps the generated numbers:
var it = iterFilterMap( rand, fcn );

// Perform manual iteration...
var r;
while ( true ) {
    r = it.next();
    if ( r.done ) {
        break;
    }
    console.log( r.value );
}
```

</section>

<!-- /.examples -->

<!-- Section to include cited references. If references are included, add a horizontal rule *before* the section. Make sure to keep an empty line after the `section` element and another before the `/section` close. -->

<section class="references">

</section>

<!-- /.references -->

<!-- Section for all links. Make sure to keep an empty line after the `section` element and another before the `/section` close. -->

<section class="links">

[mdn-iterator-protocol]: https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Iteration_protocols#The_iterator_protocol

</section>

<!-- /.links -->
