#/
# @license Apache-2.0
#
# Copyright (c) 2019 The Stdlib Authors.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#/

# VARIABLES #

# Define the path to the [dtslint][1] executable.
#
# To install dtslint:
#
# ```bash
# $ npm install dtslint
# ```
#
# [1]: https://github.com/Microsoft/dtslint
DTSLINT ?= $(BIN_DIR)/dtslint

# Define the path to the TypeScript compiler configuration file:
DTSLINT_CONF_TSCONFIG ?= $(CONFIG_DIR)/dtslint/tsconfig.json

# Define the path to the TSLint configuration file for linting TypeScript declaration files:
DTSLINT_CONF_TSLINT ?= $(CONFIG_DIR)/dtslint/tslint.json

# Define the command-line options to use when invoking the dtslint executable:
DTSLINT_FLAGS ?=


# RULES #

#/
# Lints TypeScript declaration files using [dtslint][1].
#
# ## Notes
#
# -   This rule is useful when wanting to glob for TypeScript declaration files (e.g., lint all TypeScript declaration files for a particular package).
#
# [1]: https://github.com/Microsoft/dtslint
#
# @private
# @param {string} [TYPESCRIPT_DECLARATIONS_FILTER] - file path pattern (e.g., `.*/math/base/special/abs/.*`)
# @param {*} [FAST_FAIL] - flag indicating whether to stop linting upon encountering a lint error
#
# @example
# make dtslint-declarations
#
# @example
# make dtslint-declarations TYPESCRIPT_DECLARATIONS_FILTER=".*/math/base/special/abs/.*"
#/
dtslint-declarations: $(NODE_MODULES)
ifeq ($(FAIL_FAST), true)
	$(QUIET) $(FIND_TYPESCRIPT_DECLARATIONS_CMD) | grep '^[\/]\|^[a-zA-Z]:[/\]' | while read -r file; do \
		echo ''; \
		echo "Linting file: $$file"; \
		cd $(ROOT_DIR); \
		dir=$$(dirname $$file); \
		$(CP) $(DTSLINT_CONF_TSCONFIG) $(DTSLINT_CONF_TSLINT) "$$dir"; \
		cd "$$dir"; \
		NODE_PATH=$(NODE_PATH) $(DTSLINT) $(DTSLINT_FLAGS) || exit 1; \
		rm "$$(basename $(DTSLINT_CONF_TSCONFIG))" "$$(basename $(DTSLINT_CONF_TSLINT))"; \
	done
	$(QUIET) cd $(ROOT_DIR)
else
	$(QUIET) $(FIND_TYPESCRIPT_DECLARATIONS_CMD) | grep '^[\/]\|^[a-zA-Z]:[/\]' | while read -r file; do \
		echo ''; \
		echo "Linting file: $$file"; \
		cd $(ROOT_DIR); \
		dir=$$(dirname $$file); \
		$(CP) $(DTSLINT_CONF_TSCONFIG) $(DTSLINT_CONF_TSLINT) "$$dir"; \
		cd "$$dir"; \
		NODE_PATH=$(NODE_PATH) $(DTSLINT) $(DTSLINT_FLAGS) || exit 1; \
		rm "$$(basename $(DTSLINT_CONF_TSCONFIG))" "$$(basename $(DTSLINT_CONF_TSLINT))"; \
	done
	$(QUIET) cd $(ROOT_DIR)
endif

.PHONY: dtslint-declarations

#/
# Lints a specified list of TypeScript declaration files using [dtslint][1].
#
# ## Notes
#
# -   This rule is useful when wanting to lint a list of TypeScript declaration files generated by some other command (e.g., a list of changed TypeScript declaration files obtained via `git diff`).
#
# [1]: https://github.com/Microsoft/dtslint
#
# @private
# @param {string} FILES - list of TypeScript declaration file paths
# @param {*} [FAST_FAIL] - flag indicating whether to stop linting upon encountering a lint error
#
# @example
# make dtslint-files FILES='/foo/index.js /bar/index.js'
#/
dtslint-files: $(NODE_MODULES)
ifeq ($(FAIL_FAST), true)
	$(QUIET) for file in $(FILES); do \
		echo ''; \
		echo "Linting file: $$file"; \
		cd $(ROOT_DIR); \
		dir=$$(dirname $$file); \
		$(CP) $(DTSLINT_CONF_TSCONFIG) $(DTSLINT_CONF_TSLINT) "$$dir"; \
		cd "$$dir"; \
		NODE_PATH=$(NODE_PATH) $(DTSLINT) $(DTSLINT_FLAGS) || exit 1; \
		rm "$$(basename $(DTSLINT_CONF_TSCONFIG))" "$$(basename $(DTSLINT_CONF_TSLINT))"; \
	done
	$(QUIET) cd $(ROOT_DIR)
else
	$(QUIET) for file in $(FILES); do \
		echo ''; \
		echo "Linting file: $$file"; \
		cd $(ROOT_DIR); \
		dir=$$(dirname $$file); \
		$(CP) $(DTSLINT_CONF_TSCONFIG) $(DTSLINT_CONF_TSLINT) "$$dir"; \
		cd "$$dir"; \
		NODE_PATH=$(NODE_PATH) $(DTSLINT) $(DTSLINT_FLAGS) || exit 1; \
		rm "$$(basename $(DTSLINT_CONF_TSCONFIG))" "$$(basename $(DTSLINT_CONF_TSLINT))"; \
	done
	$(QUIET) cd $(ROOT_DIR)
endif

.PHONY: dtslint-files
