//--------------------------------------------------------------------------
// Copyright (C) 2014-2025 Cisco and/or its affiliates. All rights reserved.
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License Version 2 as published
// by the Free Software Foundation.  You may not use, modify or distribute
// this program under any other version of the GNU General Public License.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License along
// with this program; if not, write to the Free Software Foundation, Inc.,
// 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
//--------------------------------------------------------------------------
// ips_metadata.cc author Russ Combs <rucombs@cisco.com>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "framework/decode_data.h"
#include "framework/ips_option.h"
#include "framework/module.h"
#include "main/snort_config.h"

using namespace snort;
using namespace std;

#define s_name "metadata"

//-------------------------------------------------------------------------
// module
//-------------------------------------------------------------------------

static const Parameter s_params[] =
{
    { "*", Parameter::PT_STRING, nullptr, nullptr,
      "comma-separated list of arbitrary name value pairs" },

    { nullptr, Parameter::PT_MAX, nullptr, nullptr, nullptr }
};

#define s_help \
    "rule option for conveying arbitrary comma-separated name, value data within the rule text"

class MetadataModule : public Module
{
public:
    MetadataModule() : Module(s_name, s_help, s_params) { }

    bool begin(const char*, int, SnortConfig*) override;
    bool set(const char*, Value&, SnortConfig*) override;

    Usage get_usage() const override
    { return DETECT; }

    bool match = false;
};

bool MetadataModule::begin(const char*, int, SnortConfig*)
{
    match = false;
    return true;
}

bool MetadataModule::set(const char*, Value& v, SnortConfig* sc)
{
    assert(v.is("*"));

    if ( !match and !sc->metadata_filter.empty() )
        match = strstr(v.get_string(), sc->metadata_filter.c_str()) != nullptr;

    return true;
}

//-------------------------------------------------------------------------
// api methods
//-------------------------------------------------------------------------

static Module* mod_ctor()
{
    return new MetadataModule;
}

static void mod_dtor(Module* m)
{
    delete m;
}

static IpsOption* metadata_ctor(Module* p, IpsInfo& info)
{
    MetadataModule* m = (MetadataModule*)p;

    if ( m->match )
        IpsOption::set_metadata_match(info);

    return nullptr;
}

static const IpsApi metadata_api =
{
    {
        PT_IPS_OPTION,
        sizeof(IpsApi),
        IPSAPI_VERSION,
        0,
        API_RESERVED,
        API_OPTIONS,
        s_name,
        s_help,
        mod_ctor,
        mod_dtor
    },
    OPT_TYPE_META,
    0, PROTO_BIT__NONE,
    nullptr,
    nullptr,
    nullptr,
    nullptr,
    metadata_ctor,
    nullptr,
    nullptr
};

#ifdef BUILDING_SO
SO_PUBLIC const BaseApi* snort_plugins[] =
#else
const BaseApi* ips_metadata[] =
#endif
{
    &metadata_api.base,
    nullptr
};

