/*
 * polling.c -- code to emulate scancodes for volume buttons on certain
 * 		models
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * Written by Sos Pter <sp@osb.hu>, 2002-2003
 */


#include <linux/config.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/types.h>
#include <linux/init.h>
#include <linux/pm.h>
#include <linux/proc_fs.h>
#include <linux/version.h>
#include <linux/timer.h>
#include <linux/kbd_ll.h>

#include <asm/system.h>
#include <asm/errno.h>
#include <asm/io.h>
#include <asm/uaccess.h>

#ifdef OMNIBOOK_STANDALONE
#include "omnibook.h"
#else
#include <linux/omnibook.h>
#endif

#include "ec.h"

static struct pm_dev *pm_key_polling;
static pm_callback pm_key_polling_callback = NULL;

static struct proc_dir_entry *proc_key_polling;

static struct timer_list omnibook_key_timer;

int omnibook_key_polling_enabled = 0;

static void omnibook_key_poller(unsigned long data)
{
	u8 q0a;

	switch (omnibook_ectype) {
	case XE3GC:
		omnibook_ec_read(XE3GC_Q0A, &q0a);
		omnibook_ec_write(XE3GC_Q0A, 0);

		if (q0a & XE3GC_SLPB_MASK)
			printk(KERN_INFO "%s: Sleep button pressed.\n", MODULE_NAME);
		if (q0a & XE3GC_F5_MASK)
			printk(KERN_INFO "%s: Fn-F5 - LCD/CRT switch pressed.\n", MODULE_NAME);
		/* Volume button scancode emulaton */
		if (q0a & XE3GC_VOLD_MASK) {
			printk(KERN_INFO "%s: Fn-down arrow or Volume down pressed.\n", MODULE_NAME);
			handle_scancode(0xe0, 1);
			handle_scancode(XE3GC_VOLD_SCAN, 1);
			handle_scancode(0xe0, 0);
			handle_scancode(XE3GC_VOLD_SCAN, 0);
		}
		if (q0a & XE3GC_VOLU_MASK) {
			printk(KERN_INFO "%s: Fn-up arrow or Volume up pressed.\n", MODULE_NAME);
			handle_scancode(0xe0, 1);
			handle_scancode(XE3GC_VOLU_SCAN, 1);
			handle_scancode(0xe0, 0);
			handle_scancode(XE3GC_VOLU_SCAN, 0);
		}
		if (q0a & XE3GC_MUTE_MASK) {
			printk(KERN_INFO "%s: Fn+F7 - Volume mute pressed.\n", MODULE_NAME);
			handle_scancode(0xe0, 1);
			handle_scancode(XE3GC_MUTE_SCAN, 1);
			handle_scancode(0xe0, 0);
			handle_scancode(XE3GC_MUTE_SCAN, 0);
		}
		if (q0a & XE3GC_CNTR_MASK)
			printk(KERN_INFO "%s: Fn+F3/Fn+F4 - Contrast up or down pressed.\n", MODULE_NAME);
		if (q0a & XE3GC_BRGT_MASK)
			printk(KERN_INFO "%s: Fn+F1/Fn+F2 - Brightness up or down pressed.\n", MODULE_NAME);
		break;
	}

	mod_timer(&omnibook_key_timer, jiffies + OMNIBOOK_POLL);
}

int omnibook_key_polling_enable(void)
{
#ifdef CONFIG_VT
	switch (omnibook_ectype) {
	case XE3GC:
		printk(KERN_INFO "%s: Volume buttons do not generate scancodes on this model.\n", MODULE_NAME);
		init_timer(&omnibook_key_timer);
		omnibook_key_timer.data = 0;
		omnibook_key_timer.function = omnibook_key_poller;
		omnibook_key_timer.expires = jiffies + OMNIBOOK_POLL;
		add_timer(&omnibook_key_timer);
		printk(KERN_INFO "%s: Scancode emulation for volume buttons enabled.\n", MODULE_NAME);
		break;
	case OB500:
	case OB510:
		omnibook_key_polling_enabled = 0;
		return -ENODEV;
		break;
	default:
		omnibook_key_polling_enabled = 0;
		return -ENODEV;
	}
#endif
	return 0;
}
	
int omnibook_key_polling_disable(void)
{
#ifdef CONFIG_VT
	switch (omnibook_ectype) {
	case XE3GC:
		if (&omnibook_key_timer) {
			del_timer(&omnibook_key_timer);
			printk(KERN_INFO "%s: Scancode emulation for volume buttons disabled.\n", MODULE_NAME);
		}
		break;
	case OB500:
	case OB510:
		omnibook_key_polling_enabled = 0;
		return -ENODEV;
		break;
	default:
		omnibook_key_polling_enabled = 0;
		return -ENODEV;
	}
#endif
	return 0;
}

/*
 * Power management handler: on resume it reenables key polling if it was enabled previously
 */

static int pm_key_polling_handler(struct pm_dev *dev, pm_request_t rqst, void *data)
{
	switch (rqst) {
	case PM_RESUME:
		if (omnibook_key_polling_enabled)
			return omnibook_key_polling_enable();
		break;
	case PM_SUSPEND:
		if (omnibook_key_polling_enabled)
			return omnibook_key_polling_disable();
		break;
	}
	return 0;
}

static int omnibook_key_polling_register(void)
{
		pm_key_polling_callback = pm_key_polling_handler;
		pm_key_polling = pm_register(PM_SYS_DEV, PM_SYS_KBC, pm_key_polling_callback);
		if (pm_key_polling)
			return 0;
		else
			return -EFAULT;
}

static void omnibook_key_polling_unregister(void)
{
		pm_key_polling_callback = NULL;
		pm_unregister(pm_key_polling);
}

static int omnibook_key_polling_status(char *buffer, char **start, off_t off, int count, int *eof, void *data)
{
	int len;
	char *b = buffer;
	char *str;

	str = (omnibook_key_polling_enabled) ? "enabled" : "disabled";
	
	b += sprintf(b, "Volume buttons handling is %s\n", str);

	len = b - buffer;
	if (len < off + count)
		*eof = 1;
	*start = buffer + off;
	len -= off;
	if (len > count)
		len = count;
	if (len < 0)
		len = 0;
	return len;
}

static int omnibook_key_polling_set(struct file *file, const char *buffer, unsigned long count, void *data)
{
	char status[1] = {'\0'};

	if (copy_from_user(status, buffer, 1))
		return -EFAULT;
	switch (*status) {
	case '0':
		omnibook_key_polling_disable();
		break;
	case '1':
		omnibook_key_polling_enable();
		break;
	default:
		count = -EINVAL;
	}
	return count;
}


int __init omnibook_key_polling_init(void)
{
#ifdef CONFIG_VT
/* FIXME - Scancode emulation does not work on 2.5.x kernels yet */
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0))

	int retval;
	mode_t pmode;

	switch (omnibook_ectype) {
	case XE3GC:
		pmode = S_IFREG | S_IWUSR | S_IRUGO;
		if (omnibook_userset)
			pmode = pmode | S_IWUGO;
		proc_key_polling = create_proc_entry("key_polling", pmode, omnibook_proc_root);
		break;
	case OB500:
	case OB510:
		printk(KERN_INFO "%s: Key polling is unsupported on this machine.\n", MODULE_NAME);
		omnibook_key_polling_enabled = 0;
		return 0;
	default:
		printk(KERN_INFO "%s: Key polling is unnecessary on this machine.\n", MODULE_NAME);
		omnibook_key_polling_enabled = 0;
		return 0;
	}
	if (proc_key_polling) {
		proc_key_polling->read_proc = omnibook_key_polling_status;
		proc_key_polling->write_proc = omnibook_key_polling_set;
	} else {
		printk(KERN_ERR "%s: Unable to create /proc/%s/key_polling.\n", MODULE_NAME, MODULE_NAME);
		return -ENOENT;
	}
	retval = omnibook_key_polling_register();
	if (retval)
		return retval;
	retval = omnibook_key_polling_enable();
	if (retval)
		return retval;

	printk(KERN_INFO "%s: Key polling is enabled.\n", MODULE_NAME);
	return 0;
#else
	printk(KERN_INFO "%s: Key polling is supported only on 2.4.x kernels.\n", MODULE_NAME);
	return -ENODEV;
#endif
#else
	printk(KERN_ERR "%s: Virtual terminal support is not compiled into your kernel.\n", MODULE_NAME);
	return -ENODEV;
#endif
}

void __exit omnibook_key_polling_cleanup(void)
{
#ifdef CONFIG_VT
/* FIXME - Scancode emulation does not work on 2.5.x kernels yet */
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0))
	if (proc_key_polling)
		remove_proc_entry("key_polling", omnibook_proc_root);
	if (pm_key_polling)
		omnibook_key_polling_unregister();
	
	omnibook_key_polling_disable();
#endif
#endif
}

EXPORT_SYMBOL(omnibook_key_polling_enabled);
EXPORT_SYMBOL(omnibook_key_polling_enable);
EXPORT_SYMBOL(omnibook_key_polling_disable);

/* End of file */
