#
# Copyright (C) 2005  Robert Collins  <robertc@squid-cache.org>
# 
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import os
import re
from shutil import rmtree
import sys
import urllib
from config_manager.optparse import (OptionParser, Sequence,
                                     StringArgument, TerminalElement,
                                     ArgumentValueError, Optional)


import config_manager.implementations
#TODO import implementations dynamically.
import config_manager.implementations.fake_vcs
import config_manager.implementations.arch_vcs
fake_updates = []

class UnsupportedScheme(KeyError):
    """A specific URL scheme is not supported."""

def test_suite():
    import config_manager.tests
    result = config_manager.tests.tests.TestSuite()
    result.addTest(config_manager.implementations.test_suite())
    result.addTest(config_manager.tests.test_suite())
    return result

class Config(object):
    """A configurationof recipe."""

    _re_config_line = re.compile("^(#.*)|(([^\s]+)\s+([^\s]+)|\s*)$")
    # comment is 0, path 2, source 3

    def __init__(self, fromStream=None, override_mapper=None):
        self._entries = {}
        self._override_mapper = override_mapper
        if fromStream is not None:
            # parsing might be better in a factory ?
            for line in fromStream.readlines():
                groups = self._re_config_line.match(line).groups()
                if groups[2] and groups[3]:
                    self.add_entry(ConfigEntry(groups[2],
                                               groups[3],
                                               self._override_mapper))

    def add_entry(self, entry):
        assert (entry.path not in self._entries)
        self._entries[entry.path] = entry

    def build(self, dir):
        """Perform a build of this config with base dir dir, which must exist
        already.
        """
        to_process = self._entries.items()
        to_process.sort(key=lambda x:len(x[0]))
        for path, entry in to_process:
            entry.build(dir)

    def get_entries(self):
        """Return a diction of ConfigEntries."""
        # this returns a copy of the internal dict to prevent
        # garbage being put in the internal dict.
        # it might be nice to return a readonly reference to the
        # internal one though ...
        return self._entries.copy()

    def update(self, dir):
        """Perform a update of this config with base dir dir, which must exist
        already.
        """
        to_process = self._entries.items()
        to_process.sort(key=lambda x:len(x[0]))
        for path, entry in to_process:
            entry.update(dir)


class ConfigEntry(object):
    """A single element in a configuration."""

    def __eq__(self, other):
        try:
            return self.path == other.path and self.url == other.url
        except AttributeError:
            return False
        
    def __init__(self, relative_path, url, override_mapper=None):
        """Construct a ConfigEntry for the provided path and url.
        
        If an override_mapper is provided, it should be a URLMapper and is
        used to establish local overrides for url.
        """
        self.path = relative_path
        self.url = url
        if override_mapper is not None:
            self.url = override_mapper.map(url)
            
    def _build_cvs_url(self, path):
        if self.url.startswith("pserver://"):
            scheme=":pserver:"
            url=self.url[10:]
        elif self.url.startswith("ext://"):
            scheme=":ext:"
            url=self.url[6:]
        else:
            raise ValueError, "CVS scheme not recognised: '%s'" % self.url

        module=os.path.basename(url)
        repo=os.path.dirname(url)
        print "cvs -d %s%s checkout -d %s %s" % (scheme, repo, self.path, module)
        os.system("cvs -d %s%s checkout -d %s %s" % (scheme, repo, self.path, module))

    def _build_svn_url(self, path):
        if self.url.startswith("svn://") or self.url.startswith("svn+ssh://"):
            url = self.url
        else:
            url = self.url[4:]
        print "svn checkout %s %s" % (url, os.path.normpath(os.path.join(path, self.path)))
        os.system("svn checkout %s %s" % (url, os.path.normpath(os.path.join(path, self.path))))

    def _build_pybaz_name(self, path):
        try:
            import pybaz
        except ImportError:
            # pybaz not available
            return False
        try:
            # try as registered name
            pybaz.get(self.url, os.path.join(path, self.path))
            return True
        except pybaz.errors.ExecProblem, e:
            rmtree(os.path.join(path, self.path), ignore_errors=True)
        except pybaz.errors.NamespaceError:
            return False

    def _build_pybaz_url(self, path):
        try:
            import pybaz
        except ImportError:
            # pybaz not available
            return False
        try:
            lastslash = self.url.rfind('/')
            url = self.url[:lastslash]
            version = self.url[lastslash:]
            archive = str(pybaz.register_archive(None, url))
            pybaz.get(archive + version, os.path.join(path, self.path))
            return True
        except pybaz.errors.ExecProblem:
            rmtree(os.path.join(path, self.path), ignore_errors=True)

    def _build_bzr_url(self, path):
        try:
            import errno
            from bzrlib.errors import (
                DivergedBranches,
                NotBranchError,
                NoSuchRevision,
                )
            from bzrlib.branch import Branch
            from bzrlib.revisionspec import RevisionSpec
            from bzrlib.errors import (DivergedBranches,
                                       NotBranchError,
                                       NoSuchRevision)

            components = self.url.split(';revno=')
            from_location = components[0]
            if len(components) == 2:
                revision = RevisionSpec.from_string(components[1])
            else:
                revision = None
            to_location = os.path.join(path, self.path)

            #if from_location.startswith("file://"):
            #    from_location = from_location[7:]

            try:
                br_from = Branch.open(from_location)
# we should catch all and report this type as not supported on that url.
            except NotBranchError:
                return False

            if revision is not None:
                revision_id = revision.as_revision_id(br_from)
            else:
                revision_id = None
            os.mkdir(to_location)
            to_dir = br_from.bzrdir.sprout(to_location, revision_id)
            # remove any local modifications
            to_dir.open_workingtree().revert()
            return True
        except Exception, e:
            print "%r" % e, e
# bare except because I havn't tested a bad url yet.
            return False

    def get_implementations(self):
        """Get the implementations represented by the url in this entry."""
        for scheme, implementations in \
            config_manager.implementations.schemes.items():
            if self.url.startswith(scheme):
                return implementations
        raise UnsupportedScheme

    def _update_bzr(self, path):
        from bzrlib.branch import Branch
        from bzrlib.bzrdir import BzrDir
        from bzrlib.revisionspec import RevisionSpec
        from bzrlib.errors import DivergedBranches, NotBranchError
        path = os.path.join(path, self.path)
        try:
            dir_to = BzrDir.open(path)
        except NotBranchError:
            return False
        components = self.url.split(';revno=')
        url = components[0]
        br_from = Branch.open(url)
        if len(components) == 2:
            revision = RevisionSpec.from_string(components[1])
            revision_id = revision.as_revision_id(br_from)
        else:
            revision_id = None
        try:
            dir_to.open_workingtree().pull(br_from, stop_revision=revision_id)
            return True
        except DivergedBranches:
            print "Branch %s has diverged from %s - you need to merge." % (
                path, self.url)
            raise RuntimeError("Failed to update.")
        return False
                    
    def build(self, path):
        """Build this element from root path."""
        ## FIXME: the C++ version uses a ConfigSource to abstract out
        ## the url types, we should do that once we have a couple of
        ## types implemented
        target = os.path.abspath(os.path.join(path, self.path))
        if not os.path.isdir(os.path.dirname(target)):
            raise KeyError("Cannot build to target %s"
                           ", its parent does not exists" % target)
        # protocol specific urls
        if (self.url.startswith("arch://") or 
            self.url.startswith("fake://")):
            implementations = self.get_implementations()
            for implementation in implementations:
                if implementation.checkout(self.url, target):
                    return
        elif self.url.startswith("svn"):
            self._build_svn_url(path)
        elif self.url.startswith("pserver://"):
            self._build_cvs_url(path)
        elif self.url.startswith("ext://"):
            self._build_cvs_url(path)
        else:
            # autodetect urls
            #
            if self._build_pybaz_name(path):
                return
            elif self._build_pybaz_url(path):
                return
            elif self._build_bzr_url(path):
                return
            else:
                raise ValueError("unknown url type '%s'" % self.url)
            
    def _update_pybaz(self, path):
        try:
            import pybaz
        except ImportError:
            # pybaz not available
            return False
        try:
            tree = pybaz.WorkingTree(os.path.join(path, self.path))
            # if not tree.version == self.url[7:] ... wrong version
            tree.update()
        except pybaz.errors.SourceTreeError:
            return False
        return True

    def update(self, path):
        """Update this entry (from root path)."""
        ## FIXME: the C++ version uses a ConfigSource to abstract out
        ## the url types, we should do that once we have a couple of
        ## types implemented
        target = os.path.abspath(os.path.join(path, self.path))
        if not os.path.isdir(os.path.dirname(target)):
            raise KeyError("Cannot build to target %s"
                           ", its parent does not exists" % target)
        if not os.path.exists(os.path.join(path, self.path)):
            return self.build(path)
        # protocol specific urls
        if self.url.startswith("arch://"):
            try:
                import pybaz
            except ImportError:
                # pybaz not available
                raise ValueError("arch:// support library pybaz not present.")
            tree = pybaz.WorkingTree(os.path.join(path, self.path))
            # if not tree.version == self.url[7:] ... wrong version
            tree.update()
        elif self.url.startswith("fake://"):
            fake_updates.append((self.url, os.path.join(path, self.path)))
        elif self.url.startswith("svn"):
            print "svn update %s" % (os.path.normpath(os.path.join(path, self.path)),)
            os.system("svn update %s" % (os.path.normpath(os.path.join(path, self.path)),))
        elif self.url.startswith("pserver://") or self.url.startswith("ext://"):
            # XXX: duplicated from build() above.  Remove when refactored
            # into separate classes.
            if self.url.startswith("pserver://"):
                scheme=":pserver:"
                url=self.url[10:]
            elif self.url.startswith("ext://"):
                scheme=":ext:"
                url=self.url[6:]
            else:
                raise ValueError, "CVS scheme not recognised: '%s'" % self.url

            module=os.path.basename(url)
            repo=os.path.dirname(url)
            print "cd %s && cvs -d %s%s update -Pd" % (self.path, scheme, repo)
            os.system("cd %s && cvs -d %s%s update -Pd" % (self.path, scheme, repo))
        # autodetection
        elif self._update_pybaz(path):
            return
        elif self._update_bzr(path):
            return
        else:
            raise ValueError("unknown url type '%s'" % self.url)


class CommandArgument(StringArgument):

    def rule(self):
        return "build|show|update"


class UrlArgument(StringArgument):

    def rule(self):
        return "url"

def main(argv):
    """The entry point for main()."""
    parser = OptionParser(prog="cm")
    parser.set_arguments(Sequence((CommandArgument(), Optional(UrlArgument()),
                                   TerminalElement())))
    parser.set_usage("cm [options] %s" % parser.arguments.rule())
    try:
        (options, args) = parser.parse_args(argv[1:])
    except ArgumentValueError:
        parser.print_usage()
        return 0
    if len(args) == 1:
        config = Config(fromStream=sys.stdin)
    else:
        stream = urllib.urlopen(args[1])
        config = Config(fromStream=stream)
    if args[0] == "build":
        config.build(os.path.abspath(os.curdir))
        return 0
    if args[0] == "show":
        entries = config.get_entries().items()
        entries.sort(key=lambda x:len(x[0]))
        for path, entry in entries:
            print "%s\t%s" % (path, entry.url)
        return 0
    if args[0] == "update":
        config.update(os.path.abspath(os.curdir))
        return 0
        

class URLMapper(object):
    """A mapping between urls that works with prefixes."""

    def add_map(self, from_url, to_url):
        """Add a mapping rule from from_url to to_url.

        Note that URLs are by definition non-empty ascii strings.
        """
        self._rules[from_url] = to_url

    def __eq__(self, other):
        return self._rules == other._rules

    def __init__(self):
        self._rules = {}

    def map(self, url):
        """Map url to produce a new url.

        If no mapping is defined, url is returned.
        """
        rules = self._rules.items()
        for prefix, replacement in rules:
            if url == prefix:
                return replacement
            if url.startswith(prefix) and url[len(prefix)] == '/':
                # valid prefix
                return replacement + url[len(prefix):]
        return url
