/*
 * Copyright (C) 2007-2017 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

/*
 * Configure options
 */
/* Number of possible simultaneous faults. */
#define NFAULTS		 16

#ifndef DEBUG_START
#define DEBUG_START	0
#endif
#ifndef DEBUG_END
#define DEBUG_END	0
#endif

#include "config.h"

#include <assert.h>
#include <inttypes.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "glue.h"

#include "sig_boolean.h"

#if (DATA_BITS == 64)
/* FIXME */
#define bus_addr_t	uint64_t
#elif (ADDR_BITS <= 32)
#define bus_addr_t	uint32_t
#else
#error "Bad Address Size."
#endif

#if (BS_BITS != 1 && BS_BITS != 2 && BS_BITS != 4 && BS_BITS != 8)
#error "Bad BS Size."
#endif

#if (DATA_BITS == 8)
#include "sig_cs8.h"
#define bus_data_t	uint8_t
#define sig_bus		sig_cs8
#define sig_bus_funcs	sig_cs8_funcs
#define sig_bus_connect	sig_cs8_connect
#define sig_bus_unmap	sig_cs8_unmap
#elif (DATA_BITS == 16)
#include "sig_cs16.h"
#define bus_data_t	uint16_t
#define sig_bus		sig_cs16
#define sig_bus_funcs	sig_cs16_funcs
#define sig_bus_connect	sig_cs16_connect
#define sig_bus_unmap	sig_cs16_unmap
#elif (DATA_BITS == 32)
#include "sig_cs32.h"
#define bus_data_t	uint32_t
#define sig_bus		sig_cs32
#define sig_bus_funcs	sig_cs32_funcs
#define sig_bus_connect	sig_cs32_connect
#define sig_bus_unmap	sig_cs32_unmap
#elif (DATA_BITS == 64)
#include "sig_cs64.h"
#define bus_data_t	uint64_t
#define sig_bus		sig_cs64
#define sig_bus_funcs	sig_cs64_funcs
#define sig_bus_connect	sig_cs64_connect
#define sig_bus_unmap	sig_cs64_unmap
#else
#error "Bad Data Size."
#endif

#define MEM_SIZE	((1 << ADDR_BITS) * (DATA_BITS / 8))


/*
 * Types
 */
struct cpssp {
	uint8_t mem[MEM_SIZE];

	struct sig_bus *port_cs;

	struct fault {
		enum type {
			UNUSED,
			FLIP,
			STUCK_AT_0,
			STUCK_AT_1,
			COUPLING
		} type;
		int active;
		unsigned long addr;
		unsigned int bit;
		unsigned long addr2;
		unsigned int bit2;

		struct sig_boolean *sig;
	} fault[NFAULTS];
};

static void
CHIP_(read)(struct cpssp *cpssp, bus_addr_t addr, bus_data_t *valp)
{
	bus_data_t lval;
	bus_data_t lval2;
	unsigned int bit;
	unsigned int i;

	addr &= MEM_SIZE - 1;

	/*
	 * Read word.
	 */
	lval = *(bus_data_t *) &cpssp->mem[addr];

	/*
	 * Do fault injection.
	 */
	for (i = 0;
	    i < sizeof(cpssp->fault) / sizeof(cpssp->fault[0]);
	    i++) {
		if (cpssp->fault[i].type != UNUSED
		 && cpssp->fault[i].active
		 && cpssp->fault[i].addr == addr) {
			switch (cpssp->fault[i].type) {
			case STUCK_AT_0:
				/* Clear faulty bit. */
				lval &= ~((bus_data_t) 1 << cpssp->fault[i].bit);
				break;
			case STUCK_AT_1:
				/* Set faulty bit. */
				lval |=  ((bus_data_t) 1 << cpssp->fault[i].bit);
				break;
			case COUPLING:
				/* Read faulty bit from coupled cell. */
				if (cpssp->fault[i].addr != cpssp->fault[i].addr2) {
					CHIP_(read)(cpssp, cpssp->fault[i].addr2, &lval2);
				} else {
					lval2 = lval;
				}

				bit = (lval2 >> cpssp->fault[i].bit2) & 1;

				lval &= ~((bus_data_t) 1 << cpssp->fault[i].bit);
				lval |= (bus_data_t) bit << cpssp->fault[i].bit;
				break;
			default:
				assert(0);
			}
		}
	}

	*valp = lval;
}

static void
CHIP_(write)(struct cpssp *cpssp, bus_addr_t addr, bus_data_t val)
{
	bus_data_t val2;
	unsigned int bit;
	unsigned int i;

	addr &= MEM_SIZE - 1;

	/*
	 * Write word.
	 */
	*(bus_data_t *) &cpssp->mem[addr] = val;

	/*
	 * Do fault injection.
	 */
	for (i = 0;
	    i < sizeof(cpssp->fault) / sizeof(cpssp->fault[0]);
	    i++) {
		if (cpssp->fault[i].type != UNUSED
		 && cpssp->fault[i].active
		 && cpssp->fault[i].addr == addr) {
			switch (cpssp->fault[i].type) {
			case STUCK_AT_0:
			case STUCK_AT_1:
				/* Nothing to do. */
				break;
			case COUPLING:
				/* Change bit in coupled cell, too. */
				bit = (val >> cpssp->fault[i].bit) & 1;

				if (cpssp->fault[i].addr != cpssp->fault[i].addr2) {
					CHIP_(read)(cpssp, cpssp->fault[i].addr2, &val2);
				} else {
					val2 = val;
				}
				val2 &= ~((bus_data_t) 1 << cpssp->fault[i].bit2);
				val2 |= (bus_data_t) bit << cpssp->fault[i].bit2;
				if (cpssp->fault[i].addr != cpssp->fault[i].addr2) {
					CHIP_(write)(cpssp, cpssp->fault[i].addr2, val2);
				} else {
					val = val2;
				}
				break;
			default:
				assert(0);
			}
		}
	}
}

static int
CHIP_(port)(
	struct cpssp *cpssp,
	const char *port,
	enum type *typep,
	bus_addr_t *addrp,
	unsigned int *bitp,
	bus_addr_t *addr2p,
	unsigned int *bit2p
)
{
	char *port2;

	/* Get Fault Type */
	if (strncmp(port, "bitflip", strlen("bitflip")) == 0) {
		*typep = FLIP;
		port += strlen("bitflip");
	} else if (strncmp(port, "stuck_at_0", strlen("stuck_at_0")) == 0) {
		*typep = STUCK_AT_0;
		port += strlen("stuck_at_0");
	} else if (strncmp(port, "stuck_at_1", strlen("stuck_at_1")) == 0) {
		*typep = STUCK_AT_1;
		port += strlen("stuck_at_1");
	} else if (strncmp(port, "coupling", strlen("coupling")) == 0) {
		*typep = COUPLING;
		port += strlen("coupling");
	} else {
		return 1;
	}

	/* Skip / */
	if (*port == '/') {
		port++;
	} else {
		return 1;
	}

	/* Get Address */
	*addrp = strtoul(port, &port2, 0);
	port = port2;

	/* Skip / */
	if (*port == '/') {
		port++;
	} else {
		return 1;
	}

	/* Get Bit Number */
	*bitp = strtoul(port, &port2, 0);
	port = port2;

	*bitp += (*addrp & 3) * 8;
	*addrp &= ~3;

	if (*typep == COUPLING) {
		/* Skip / */
		if (*port == '/') {
			port++;
		} else {
			return 1;
		}

		/* Get Address */
		*addr2p = strtoul(port, &port2, 0);
		port = port2;

		/* Skip / */
		if (*port == '/') {
			port++;
		} else {
			return 1;
		}

		/* Get Bit Number */
		*bit2p = strtoul(port, &port2, 0);
		port = port2;

		*bit2p += (*addr2p & 3) * 8;
		*addr2p &= ~3;
	}

	if (*port != '\0') {
		return 1;
	}

	*addrp %= MEM_SIZE;

	return 0;
}

static void
CHIP_(fault_set)(void *_cpssp, unsigned int i, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;
	bus_addr_t addr;
	unsigned int bit;
	bus_addr_t addr2;
	unsigned int bit2;

	assert(cpssp->fault[i].type != UNUSED);

	addr = cpssp->fault[i].addr;
	bit = cpssp->fault[i].bit;
	addr2 = cpssp->fault[i].addr2;
	bit2 = cpssp->fault[i].bit2;

	if (cpssp->fault[i].type == FLIP) {
		if (val) {
			bus_data_t data;

			CHIP_(read)(cpssp, addr, &data);
			data ^= (bus_data_t) 1 << bit;
			CHIP_(write)(cpssp, addr, data);
		}

	} else {
		cpssp->fault[i].active = val;

		sig_bus_unmap(cpssp->port_cs, cpssp, addr, sizeof(bus_data_t));
		if (cpssp->fault[i].type == COUPLING
		 && addr != addr2) {
			sig_bus_unmap(cpssp->port_cs, cpssp, addr2, sizeof(bus_data_t));
		}
	}
}

static void
CHIP_(fault_set0)(void *_cpssp, unsigned int val)
{
	CHIP_(fault_set)(_cpssp, 0, val);
}

static void
CHIP_(fault_set1)(void *_cpssp, unsigned int val)
{
	CHIP_(fault_set)(_cpssp, 1, val);
}

static void
CHIP_(fault_set2)(void *_cpssp, unsigned int val)
{
	CHIP_(fault_set)(_cpssp, 2, val);
}

static void
CHIP_(fault_set3)(void *_cpssp, unsigned int val)
{
	CHIP_(fault_set)(_cpssp, 3, val);
}

static void
CHIP_(fault_set4)(void *_cpssp, unsigned int val)
{
	CHIP_(fault_set)(_cpssp, 4, val);
}

static void
CHIP_(fault_set5)(void *_cpssp, unsigned int val)
{
	CHIP_(fault_set)(_cpssp, 5, val);
}

static void
CHIP_(fault_set6)(void *_cpssp, unsigned int val)
{
	CHIP_(fault_set)(_cpssp, 6, val);
}

static void
CHIP_(fault_set7)(void *_cpssp, unsigned int val)
{
	CHIP_(fault_set)(_cpssp, 7, val);
}


static void
CHIP_(fault_set8)(void *_cpssp, unsigned int val)
{
	CHIP_(fault_set)(_cpssp, 8, val);
}

static void
CHIP_(fault_set9)(void *_cpssp, unsigned int val)
{
	CHIP_(fault_set)(_cpssp, 9, val);
}

static void
CHIP_(fault_set10)(void *_cpssp, unsigned int val)
{
	CHIP_(fault_set)(_cpssp, 10, val);
}

static void
CHIP_(fault_set11)(void *_cpssp, unsigned int val)
{
	CHIP_(fault_set)(_cpssp, 11, val);
}

static void
CHIP_(fault_set12)(void *_cpssp, unsigned int val)
{
	CHIP_(fault_set)(_cpssp, 12, val);
}

static void
CHIP_(fault_set13)(void *_cpssp, unsigned int val)
{
	CHIP_(fault_set)(_cpssp, 13, val);
}

static void
CHIP_(fault_set14)(void *_cpssp, unsigned int val)
{
	CHIP_(fault_set)(_cpssp, 14, val);
}

static void
CHIP_(fault_set15)(void *_cpssp, unsigned int val)
{
	CHIP_(fault_set)(_cpssp, 15, val);
}


static void
CHIP_(connect)(void *_cpssp, const char *port, void *_sig)
{
	static const struct sig_boolean_funcs fault_funcs[] = {
		{ .set = CHIP_(fault_set0), },
		{ .set = CHIP_(fault_set1), },
		{ .set = CHIP_(fault_set2), },
		{ .set = CHIP_(fault_set3), },
		{ .set = CHIP_(fault_set4), },
		{ .set = CHIP_(fault_set5), },
		{ .set = CHIP_(fault_set6), },
		{ .set = CHIP_(fault_set7), },
		{ .set = CHIP_(fault_set8), },
		{ .set = CHIP_(fault_set9), },
		{ .set = CHIP_(fault_set10), },
		{ .set = CHIP_(fault_set11), },
		{ .set = CHIP_(fault_set12), },
		{ .set = CHIP_(fault_set13), },
		{ .set = CHIP_(fault_set14), },
		{ .set = CHIP_(fault_set15), },
	};
	struct cpssp *cpssp = (struct cpssp *) _cpssp;
	struct sig_boolean *sig = (struct sig_boolean *) _sig;
	enum type type;
	bus_addr_t addr;
	unsigned int bit;
	bus_addr_t addr2;
	unsigned int bit2;
	unsigned int i;

	assert(sizeof(fault_funcs) / sizeof(fault_funcs[0])
			== sizeof(cpssp->fault) / sizeof(cpssp->fault[0]));

	if (CHIP_(port)(_cpssp, port, &type, &addr, &bit, &addr2, &bit2)) {
		return;
	}

	/* Lookup Unused Entry */
	for (i = 0; ; i++) {
		assert(i < sizeof(cpssp->fault) / sizeof(cpssp->fault[0]));
		if (cpssp->fault[i].type == UNUSED) {
			break;
		}
	}

	/* Add Entry */
	cpssp->fault[i].type = type;
	cpssp->fault[i].active = 0;
	cpssp->fault[i].addr = addr & ~0x3UL;
	cpssp->fault[i].bit = bit + (addr & 3) * 8;
	cpssp->fault[i].addr2 = addr2 & ~0x3UL;
	cpssp->fault[i].bit2 = bit2 + (addr2 & 3) * 8;

	cpssp->fault[i].sig = sig;

	sig_boolean_connect_in(cpssp->fault[i].sig, cpssp, &fault_funcs[i]);
}


static void
CHIP_(disconnect)(void *_cpssp, const char *port)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;
	enum type type;
	bus_addr_t addr;
	unsigned int bit;
	bus_addr_t addr2;
	unsigned int bit2;
	unsigned int i;

	if (CHIP_(port)(_cpssp, port, &type, &addr, &bit, &addr2, &bit2)) {
		return;
	}

	/* Lookup Entry */
	for (i = 0; ; i++) {
		assert(i < sizeof(cpssp->fault) / sizeof(cpssp->fault[0]));
		if (cpssp->fault[i].type == type
		 && cpssp->fault[i].addr == (addr & ~0x3UL)
		 && cpssp->fault[i].bit == bit + (addr & 3) * 8
		 && cpssp->fault[i].addr2 == (addr2 & ~0x3UL)
		 && cpssp->fault[i].bit2 == bit2 + (addr2 & 3) * 8) {
			/* Entry found. */
			break;
		}
	}

	// sig_boolean_disconnect_in(cpssp->fault[i].sig, cpssp);

	/* Disable Entry */
	cpssp->fault[i].type = UNUSED;
}

static int
CHIP_(mr)(void *_cpssp, bus_addr_t addr, unsigned int bs, bus_data_t *valp)
{
	struct cpssp *cpssp = _cpssp;

	CHIP_(read)(cpssp, addr, valp);

	if (loglevel
	 && DEBUG_START <= addr
	 && addr < DEBUG_END) {
		fprintf(stderr, "%s: addr=0x%08lx bs=0x%x val=0x%08lx\n",
				__FUNCTION__,
				(unsigned long) addr, bs, (unsigned long) *valp);
	}

	return 0;
}

static int
CHIP_(mw)(void *_cpssp, bus_addr_t addr, unsigned int bs, bus_data_t val)
{
	struct cpssp *cpssp = _cpssp;
	bus_data_t lval;

	if (loglevel
	 && DEBUG_START <= addr
	 && addr < DEBUG_END) {
		fprintf(stderr, "%s: addr=0x%08lx bs=0x%x val=0x%08lx\n",
				__FUNCTION__,
				(unsigned long) addr, bs, (unsigned long) val);
	}

#if (BS_BITS == 1)
	lval = val;

#else
	if (bs == (1 << BS_BITS) - 1) {
		/* Short-cut */
		lval = val;

	} else {
		CHIP_(read)(cpssp, addr, &lval);

		if ((bs >> 0) & 1) {
			lval &= ~((bus_data_t) 0xff << 0);
			lval |= val & ((bus_data_t) 0xff << 0);
		}
#if (2 <= BS_BITS)
		if ((bs >> 1) & 1) {
			lval &= ~((bus_data_t) 0xff << 8);
			lval |= val & ((bus_data_t) 0xff << 8);
		}
#endif
#if (3 <= BS_BITS)
		if ((bs >> 2) & 1) {
			lval &= ~((bus_data_t) 0xff << 16);
			lval |= val & ((bus_data_t) 0xff << 16);
		}
#endif
#if (4 <= BS_BITS)
		if ((bs >> 3) & 1) {
			lval &= ~((bus_data_t) 0xff << 24);
			lval |= val & ((bus_data_t) 0xff << 24);
		}
#endif
#if (5 <= BS_BITS)
		if ((bs >> 4) & 1) {
			lval &= ~((bus_data_t) 0xff << 32);
			lval |= val & ((bus_data_t) 0xff << 32);
		}
#endif
#if (6 <= BS_BITS)
		if ((bs >> 5) & 1) {
			lval &= ~((bus_data_t) 0xff << 40);
			lval |= val & ((bus_data_t) 0xff << 40);
		}
#endif
#if (7 <= BS_BITS)
		if ((bs >> 6) & 1) {
			lval &= ~((bus_data_t) 0xff << 48);
			lval |= val & ((bus_data_t) 0xff << 48);
		}
#endif
#if (8 <= BS_BITS)
		if ((bs >> 7) & 1) {
			lval &= ~((bus_data_t) 0xff << 56);
			lval |= val & ((bus_data_t) 0xff << 56);
		}
#endif
	}
#endif

	CHIP_(write)(cpssp, addr, lval);

	return 0;
}

static void
CHIP_(map)(struct cpssp *cpssp, bus_addr_t addr, char **haddrp)
{
	unsigned int i;

	if (DEBUG_START <= addr
	 && addr < DEBUG_END) {
		*haddrp = NULL;
		return;
	}

	for (i = 0; ; i++) {
		if (i == sizeof(cpssp->fault) / sizeof(cpssp->fault[0])) {
			*haddrp = &cpssp->mem[addr];
			return;
		}
		if (cpssp->fault[i].type == UNUSED
		 || ! cpssp->fault[i].active) {
			continue;
		}
		if ((addr & ~0xfff) == (cpssp->fault[i].addr & ~0xfff)) {
			*haddrp = NULL;
			return;
		}
	}
}

static int
CHIP_(map_r)(
	void *_cpssp,
	bus_addr_t addr,
	int (**cfp)(void *, bus_addr_t, unsigned int, bus_data_t *),
	void **csp,
	char **haddrp
)
{
	struct cpssp *cpssp = _cpssp;

	addr &= MEM_SIZE - 1;

	*cfp = CHIP_(mr);
	*csp = cpssp;
	CHIP_(map)(cpssp, addr, haddrp);

	return 0;
}

static int
CHIP_(map_w)(
	void *_cpssp,
	bus_addr_t addr,
	int (**cfp)(void *, bus_addr_t, unsigned int, bus_data_t),
	void **csp,
	char **haddrp
)
{
	struct cpssp *cpssp = _cpssp;

	addr &= MEM_SIZE - 1;

	*cfp = CHIP_(mw);
	*csp = cpssp;
	CHIP_(map)(cpssp, addr, haddrp);

	return 0;
}

static void
CHIP_(power_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	if (val) {
		memset(cpssp->mem, 0, sizeof(cpssp->mem));
	}
}

void *
CHIP_(create)(
	const char *name,
	struct sig_manage *port_manage,
	struct sig_std_logic *port_power,
	struct sig_bus *port_cs,
	struct sig_mem_bus *port_bus
)
{
	static const struct sig_manage_funcs manage_funcs = {
		.connect = CHIP_(connect),
		.disconnect = CHIP_(disconnect),
	};
	static const struct sig_std_logic_funcs power_funcs = {
		.boolean_or_set = CHIP_(power_set),
	};
	static const struct sig_bus_funcs cs_funcs = {
		.mr	= CHIP_(mr),
		.mw	= CHIP_(mw),
		.map_r	= CHIP_(map_r),
		.map_w	= CHIP_(map_w),
	};
	struct cpssp *cpssp;

	cpssp = shm_alloc(sizeof(*cpssp));
	assert(cpssp);

	sig_manage_connect(port_manage, cpssp, &manage_funcs);

	sig_std_logic_connect_in(port_power, cpssp, &power_funcs);

	cpssp->port_cs = port_cs;
	sig_bus_connect(cpssp->port_cs, cpssp, &cs_funcs);

	return cpssp;
}

void
CHIP_(destroy)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	shm_free(cpssp);
}

void
CHIP_(suspend)(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;
	
	generic_suspend(cpssp, sizeof(*cpssp), fp);
}

void
CHIP_(resume)(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;
	
	generic_resume(cpssp, sizeof(*cpssp), fp);
}
