/*--------------------------------------------------------------------------+
$Id: CounterSet.java 26268 2010-02-18 10:44:30Z juergens $
|                                                                          |
| Copyright 2005-2010 Technische Universitaet Muenchen                     |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package edu.tum.cs.commons.collections;

import java.util.LinkedHashMap;
import java.util.Map;

/**
 * This class manages a set of counters (i.e. is a mapping from some key objects
 * to integers). As the implementation is based on hash maps, key objects must
 * provide suitable hash keys.
 * 
 * 
 * @author Florian Deissenboeck
 * @author $Author: juergens $
 * @version $Rev: 26268 $
 * @levd.rating GREEN Hash: FE6470D850FAD6E9A77E4059CD0692C5
 */
public class CounterSet<E> {

	/** The underlying map. */
	protected final Map<E, Integer> map = new LinkedHashMap<E, Integer>();

	/** Stores total value. */
	protected int total = 0;

	/**
	 * Add the given increment to an element. If the element was not present
	 * before, it is interpreted as if it was present with value 0.
	 * 
	 * @param key
	 *            the key of the counter to increment.
	 * @param increment
	 *            the increment.
	 */
	public void inc(E key, int increment) {
		Integer value = map.get(key);
		int newValue;
		if (value == null) {
			newValue = increment;
		} else {
			newValue = value + increment;
		}
		map.put(key, newValue);

		// update total sum
		total += increment;
	}

	/**
	 * Same as <code>inc(key, 1)</code>.
	 * 
	 * @see #inc(Object, int)
	 */
	public void inc(E key) {
		inc(key, 1);
	}

	/**
	 * Checks if an element is stored in the array.
	 */
	public boolean contains(E key) {
		return map.containsKey(key);
	}

	/**
	 * Get the value for an element. If the the element is not stored in the
	 * counter <code>0</code> is returned.
	 */
	public int getValue(E key) {
		Integer value = map.get(key);
		if (value == null) {
			return 0;
		}
		return value;
	}

	/**
	 * Returns the set of all elements used a keys for counters.
	 */
	public UnmodifiableSet<E> getKeys() {
		return CollectionUtils.asUnmodifiable(map.keySet());
	}

	/**
	 * Get total sum of all elements.
	 */
	public int getTotal() {
		return total;
	}

}