/*--------------------------------------------------------------------------+
$Id: PairList.java 28497 2010-06-22 09:27:40Z deissenb $
|                                                                          |
| Copyright 2005-2010 Technische Universitaet Muenchen                     |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package edu.tum.cs.commons.collections;

import java.util.ArrayList;
import java.util.List;

import edu.tum.cs.commons.assertion.CCSMPre;

/**
 * A list for storing pairs in a specific order.
 * 
 * @author hummelb
 * @author $Author: deissenb $
 * @version $Rev: 28497 $
 * @levd.rating GREEN Hash: 5510B017A5CFB62174A38A897F2160AD
 */
public class PairList<S, T> {

	/** The current size. */
	private int size = 0;

	/** The array used for storing the S. */
	private Object[] firstElements;

	/** The array used for storing the T. */
	private Object[] secondElements;

	/** Constructor. */
	public PairList() {
		this(16);
	}

	/** Constructor. */
	public PairList(int initialCapacity) {
		firstElements = new Object[initialCapacity];
		secondElements = new Object[initialCapacity];
	}

	/** Copy constructor. */
	public PairList(PairList<S, T> other) {
		this(other.size);
		addAll(other);
	}

	/** Returns whether the list is empty. */
	public boolean isEmpty() {
		return size == 0;
	}

	/** Returns the size of the list. */
	public int size() {
		return size;
	}

	/** Add the given pair to the list. */
	public void add(S first, T second) {
		ensureSpace(size + 1);
		firstElements[size] = first;
		secondElements[size] = second;
		++size;
	}

	/** Adds all pairs from another list. */
	public void addAll(PairList<S, T> other) {
		// we have to store this in a local var, as other.size may change if
		// other == this
		int otherSize = other.size;

		ensureSpace(size + otherSize);
		for (int i = 0; i < otherSize; ++i) {
			firstElements[size] = other.firstElements[i];
			secondElements[size] = other.secondElements[i];
			++size;
		}
	}

	/** Make sure there is space for at least the given amount of elements. */
	protected void ensureSpace(int space) {
		if (space <= firstElements.length) {
			return;
		}

		Object[] oldFirst = firstElements;
		Object[] oldSecond = secondElements;
		int newSize = firstElements.length * 2;
		while (newSize < space) {
			newSize *= 2;
		}

		firstElements = new Object[newSize];
		secondElements = new Object[newSize];
		System.arraycopy(oldFirst, 0, firstElements, 0, size);
		System.arraycopy(oldSecond, 0, secondElements, 0, size);
	}

	/** Returns the first element at given index. */
	@SuppressWarnings("unchecked")
	public S getFirst(int i) {
		checkWithinBounds(i);
		return (S) firstElements[i];
	}

	/**
	 * Checks whether the given <code>i</code> is within the bounds. Throws an
	 * exception otherwise.
	 */
	private void checkWithinBounds(int i) {
		if (i < 0 || i >= size) {
			throw new IndexOutOfBoundsException("Out of bounds: " + i);
		}
	}

	/** Sets the first element at given index. */
	public void setFirst(int i, S value) {
		checkWithinBounds(i);
		firstElements[i] = value;
	}

	/** Returns the second element at given index. */
	@SuppressWarnings("unchecked")
	public T getSecond(int i) {
		checkWithinBounds(i);
		return (T) secondElements[i];
	}

	/** Sets the first element at given index. */
	public void setSecond(int i, T value) {
		checkWithinBounds(i);
		secondElements[i] = value;
	}

	/** Creates a new list containing all first elements. */
	@SuppressWarnings("unchecked")
	public List<S> extractFirstList() {
		List<S> result = new ArrayList<S>(size + 1);
		for (int i = 0; i < size; ++i) {
			result.add((S) firstElements[i]);
		}
		return result;
	}

	/** Creates a new list containing all second elements. */
	@SuppressWarnings("unchecked")
	public List<T> extractSecondList() {
		List<T> result = new ArrayList<T>(size + 1);
		for (int i = 0; i < size; ++i) {
			result.add((T) secondElements[i]);
		}
		return result;
	}

	/**
	 * Swaps the pairs of this list. Is S and T are different types, this will
	 * be extremely dangerous.
	 */
	public void swapPairs() {
		Object[] temp = firstElements;
		firstElements = secondElements;
		secondElements = temp;
	}

	/** Clears this list. */
	public void clear() {
		size = 0;
	}

	/** Removes the last element of the list. */
	public void removeLast() {
		CCSMPre.isTrue(size > 0, "Size must be positive!");
		size -= 1;
	}
}