/*--------------------------------------------------------------------------+
$Id: MD5Digest.java 26722 2010-03-15 08:00:59Z heineman $
|                                                                          |
| Copyright 2005-2010 Technische Universitaet Muenchen                     |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package edu.tum.cs.commons.digest;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.security.MessageDigest;
import java.util.Arrays;

import edu.tum.cs.commons.assertion.CCSMPre;
import edu.tum.cs.commons.string.StringUtils;

/**
 * An MD5 digest. This is just a thin thin wrapper around a byte array with some
 * convenience methods and {@link #hashCode()}, {@link #equals(Object)} and
 * {@link #compareTo(MD5Digest)} implemented correctly. This class is used
 * instead of plain strings to save both memory and (some) execution time. The
 * class is immutable. Custom (de)serialization is provided to make this
 * efficient to use in storage or RMI senarios.
 * 
 * @author hummelb
 * @author $Author: heineman $
 * @version $Rev: 26722 $
 * @levd.rating GREEN Hash: 1F3799422871C3DC0A88A5AF75FE6BCD
 */
public final class MD5Digest implements Serializable, Comparable<MD5Digest> {

	/** The digest. */
	private byte[] digest;

	/** Number of bytes in an MD5 sum. */
	public static final int MD5_BYTES = 16;

	/**
	 * Constructor. This calls {@link MessageDigest#digest()}, so the digester
	 * will be reset afterwards.
	 */
	public MD5Digest(MessageDigest digester) {
		digest = digester.digest();
		CCSMPre.isTrue(digest.length == MD5_BYTES,
				"Invalid digester used; not MD5");
	}

	/** Constructor. */
	public MD5Digest(byte[] digest) {
		CCSMPre.isTrue(digest.length == MD5_BYTES,
				"Invalid size of MD5 digest!");
		this.digest = digest.clone();
	}

	/**
	 * Inserts the digest data into the given MD. This method is used to rehash
	 * multiple hashes.
	 * <p>
	 * This method is provided instead of a getter, to keep this immutable.
	 */
	public void insertIntoDigester(MessageDigest md) {
		md.update(digest);
	}

	/** {@inheritDoc} */
	@Override
	public int hashCode() {
		return Arrays.hashCode(digest);
	}

	/**
	 * Calculates and returns a hashcode that only depends on the first 3 bytes.
	 */
	public int partialHashCode() {
		return digest[0] | (digest[1] << 8) | (digest[2] << 16);
	}

	/** {@inheritDoc} */
	@Override
	public boolean equals(Object o) {
		if (!(o instanceof MD5Digest)) {
			return false;
		}
		return Arrays.equals(digest, ((MD5Digest) o).digest);
	}

	/** {@inheritDoc} */
	public int compareTo(MD5Digest o) {
		if (o == null) {
			return -1;
		}

		int lengthDelta = digest.length - o.digest.length;
		if (lengthDelta != 0) {
			return lengthDelta;
		}

		for (int i = 0; i < digest.length; ++i) {
			int delta = digest[i] - o.digest[i];
			if (delta != 0) {
				return delta;
			}
		}

		return 0;
	}

	/** Returns a copy of the internal byte representation. */
	public byte[] getBytes() {
		return digest.clone();
	}

	/** {@inheritDoc} */
	@Override
	public String toString() {
		return StringUtils.encodeAsHex(digest);
	}

	/** Custom serialization for MD5 hashes. */
	private void writeObject(ObjectOutputStream out) throws IOException {
		out.writeByte(digest.length);
		out.write(digest);
	}

	/** Custom deserialization for MD5 hashes. */
	private void readObject(ObjectInputStream in) throws IOException {
		int size = in.readByte();
		digest = new byte[size];
		int pos = 0;
		while (pos < size) {
			pos += in.read(digest, pos, size - pos);
		}
	}

	/** Comparator for {@link MD5Digest}. */
	public static class Comparator implements java.util.Comparator<MD5Digest>,
			Serializable {

		/** {@inheritDoc} */
		@Override
		public int compare(MD5Digest o1, MD5Digest o2) {
			return o1.compareTo(o2);
		}
	}
}