/*--------------------------------------------------------------------------+
$Id: ModelReader.java 26268 2010-02-18 10:44:30Z juergens $
|                                                                          |
| Copyright 2005-2010 Technische Universitaet Muenchen                     |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package edu.tum.cs.commons.xml.example;

import java.io.File;
import java.io.IOException;

import org.xml.sax.SAXException;

import edu.tum.cs.commons.error.NeverThrownRuntimeException;
import edu.tum.cs.commons.xml.IXMLElementProcessor;
import edu.tum.cs.commons.xml.XMLReader;

/**
 * Example model reader.
 * 
 * @author Florian Deissenboeck
 * @author $Author: juergens $
 * @version $Rev: 26268 $
 * @levd.rating GREEN Hash: C22FF1000F1D4DB9669A8E6C38AB6C75
 */
public class ModelReader extends
		XMLReader<EXMLElement, EXMLAttribute, ModelPersistenceException> {

	/** The model to populate. */
	private final Model model;

	/**
	 * Create new model reader
	 * 
	 * @param file
	 *            the file to reade
	 * @param model
	 *            the model to populate
	 */
	public ModelReader(File file, Model model) {
		super(file, new ModelResolver());
		this.model = model;
	}

	/** Read model. */
	public void readModel() throws ModelPersistenceException {
		// clear model first
		model.clear();

		parseModel();

		// process all persons
		processChildElements(new PersonProcessor());
	}

	/**
	 * This methods wraps the call to {@link #parseFile()} to create proper
	 * exceptions.
	 * 
	 * @throws ModelPersistenceException
	 */
	private void parseModel() throws ModelPersistenceException {
		try {
			parseFile();
		} catch (SAXException e) {
			throw new ModelPersistenceException("Parsing Error");
		} catch (IOException e) {
			throw new ModelPersistenceException("IO Error");
		}

	}

	/** This processor is responsible for {@link EXMLElement#ADDRESS}s. */
	private class AddressProcessor implements
			IXMLElementProcessor<EXMLElement, ModelPersistenceException> {

		/** Stores parsed address object. */
		private Address address;

		/** This method defines the target element of this processor. */
		public EXMLElement getTargetElement() {
			return EXMLElement.ADDRESS;
		}

		/** Process {@link EXMLElement#ADDRESS} element. */
		public void process() throws NeverThrownRuntimeException {

			// text content of child element STREET
			String street = getChildText(EXMLElement.STREET);

			// text content of child element CITY
			String city = getChildText(EXMLElement.CITY);

			// store address
			address = new Address(street, city);

		}

	}

	/**
	 * This processor is responsible for {@link EXMLElement#ORDER}s. This
	 * processor may create an model persistence exception.
	 */
	private class OrderProcessor implements
			IXMLElementProcessor<EXMLElement, ModelPersistenceException> {

		/** The {@link Person}-object this order belongs to. */
		private final Person person;

		/**
		 * Create new processor.
		 * 
		 * @param person
		 *            The {@link Person}-object this order belongs to.
		 */
		public OrderProcessor(Person person) {
			this.person = person;
		}

		/** This method defines the target element of this processor. */
		public EXMLElement getTargetElement() {
			return EXMLElement.ORDER;
		}

		/** Process {@link EXMLElement#ORDER} element. */
		public void process() throws ModelPersistenceException {
			// if a person has more than two orders, create an exception. This
			// is just an example to demonstrate exception handling within the
			// reader.
			if (person.getOrderCount() >= 2) {
				throw new ModelPersistenceException("Too many orders");
			}

			String orderNumber = getStringAttribute(EXMLAttribute.ARTICLE_NUMBER);

			// get text content
			String description = getText();

			// create order
			Order order = new Order(description, orderNumber);

			// add it to the person
			person.add(order);

		}

	}

	/**
	 * This processor is responsible for {@link EXMLElement#ORDERS}. Note: The
	 * {@link EXMLElement#ORDERS}-element does not have an corresponding model
	 * element it only used to achieve a more readable XML file. Therefore this
	 * processor merely forwards to {@link OrderProcessor}.
	 */
	private class OrdersProcessor implements
			IXMLElementProcessor<EXMLElement, ModelPersistenceException> {

		/** The {@link Person}-object these orders belong to. */
		private final Person person;

		/**
		 * Create new processor.
		 * 
		 * @param person
		 *            The {@link Person}-object these orders belong to.
		 */
		public OrdersProcessor(Person person) {
			this.person = person;
		}

		/** This method defines the target element of this processor. */
		public EXMLElement getTargetElement() {
			return EXMLElement.ORDERS;
		}

		/** Forward processing to {@link OrderProcessor}. */
		public void process() throws ModelPersistenceException {
			processChildElements(new OrderProcessor(person));
		}

	}

	/**
	 * This processor is responsible for {@link EXMLElement#PERSON}s. The
	 * exception type must match the exception type of the reader class.
	 */
	private class PersonProcessor implements
			IXMLElementProcessor<EXMLElement, ModelPersistenceException> {
		/** This method defines the target element of this processor. */
		public EXMLElement getTargetElement() {
			return EXMLElement.PERSON;
		}

		/** Process {@link EXMLElement#PERSON} element. */
		public void process() throws ModelPersistenceException {

			// Read text content of child element NAME
			String name = getChildText(EXMLElement.NAME);

			// Read CDATA section with details
			String details = getChildText(EXMLElement.DETAILS);

			// Forward processing of child element ADDRESS to the address
			// processor
			AddressProcessor addressProcessor = new AddressProcessor();
			processChildElements(addressProcessor);

			// Create new person, the address processor provides the address
			Person person = new Person(name, addressProcessor.address, details);

			// Process the ORDERS child element
			processChildElements(new OrdersProcessor(person));

			// Add person to the model
			model.add(person);
		}

	}

}